@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cfg.kotlin

import com.pulumi.aws.cfg.RecorderStatusArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Manages status (recording / stopped) of an AWS Config Configuration Recorder.
 * > **Note:** Starting Configuration Recorder requires a Delivery Channel to be present. Use of `depends_on` (as shown below) is recommended to avoid race conditions.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const b = new aws.s3.BucketV2("b", {bucket: "awsconfig-example"});
 * const fooDeliveryChannel = new aws.cfg.DeliveryChannel("foo", {
 *     name: "example",
 *     s3BucketName: b.bucket,
 * });
 * const assumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["config&#46;amazonaws&#46;com"],
 *         }],
 *         actions: ["sts:AssumeRole"],
 *     }],
 * });
 * const r = new aws.iam.Role("r", {
 *     name: "example-awsconfig",
 *     assumeRolePolicy: assumeRole.then(assumeRole => assumeRole.json),
 * });
 * const fooRecorder = new aws.cfg.Recorder("foo", {
 *     name: "example",
 *     roleArn: r.arn,
 * });
 * const foo = new aws.cfg.RecorderStatus("foo", {
 *     name: fooRecorder.name,
 *     isEnabled: true,
 * }, {
 *     dependsOn: [fooDeliveryChannel],
 * });
 * const a = new aws.iam.RolePolicyAttachment("a", {
 *     role: r.name,
 *     policyArn: "arn:aws:iam::aws:policy/service-role/AWS_ConfigRole",
 * });
 * const p = aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         effect: "Allow",
 *         actions: ["s3:*"],
 *         resources: [
 *             b.arn,
 *             pulumi.interpolate`${b.arn}/*`,
 *         ],
 *     }],
 * });
 * const pRolePolicy = new aws.iam.RolePolicy("p", {
 *     name: "awsconfig-example",
 *     role: r.id,
 *     policy: p.apply(p => p.json),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * b = aws.s3.BucketV2("b", bucket="awsconfig-example")
 * foo_delivery_channel = aws.cfg.DeliveryChannel("foo",
 *     name="example",
 *     s3_bucket_name=b.bucket)
 * assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["config&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": ["sts:AssumeRole"],
 * }])
 * r = aws.iam.Role("r",
 *     name="example-awsconfig",
 *     assume_role_policy=assume_role.json)
 * foo_recorder = aws.cfg.Recorder("foo",
 *     name="example",
 *     role_arn=r.arn)
 * foo = aws.cfg.RecorderStatus("foo",
 *     name=foo_recorder.name,
 *     is_enabled=True,
 *     opts = pulumi.ResourceOptions(depends_on=[foo_delivery_channel]))
 * a = aws.iam.RolePolicyAttachment("a",
 *     role=r.name,
 *     policy_arn="arn:aws:iam::aws:policy/service-role/AWS_ConfigRole")
 * p = aws.iam.get_policy_document_output(statements=[{
 *     "effect": "Allow",
 *     "actions": ["s3:*"],
 *     "resources": [
 *         b.arn,
 *         b.arn.apply(lambda arn: f"{arn}/*"),
 *     ],
 * }])
 * p_role_policy = aws.iam.RolePolicy("p",
 *     name="awsconfig-example",
 *     role=r.id,
 *     policy=p.json)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var b = new Aws.S3.BucketV2("b", new()
 *     {
 *         Bucket = "awsconfig-example",
 *     });
 *     var fooDeliveryChannel = new Aws.Cfg.DeliveryChannel("foo", new()
 *     {
 *         Name = "example",
 *         S3BucketName = b.Bucket,
 *     });
 *     var assumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "config.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *             },
 *         },
 *     });
 *     var r = new Aws.Iam.Role("r", new()
 *     {
 *         Name = "example-awsconfig",
 *         AssumeRolePolicy = assumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var fooRecorder = new Aws.Cfg.Recorder("foo", new()
 *     {
 *         Name = "example",
 *         RoleArn = r.Arn,
 *     });
 *     var foo = new Aws.Cfg.RecorderStatus("foo", new()
 *     {
 *         Name = fooRecorder.Name,
 *         IsEnabled = true,
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             fooDeliveryChannel,
 *         },
 *     });
 *     var a = new Aws.Iam.RolePolicyAttachment("a", new()
 *     {
 *         Role = r.Name,
 *         PolicyArn = "arn:aws:iam::aws:policy/service-role/AWS_ConfigRole",
 *     });
 *     var p = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Actions = new[]
 *                 {
 *                     "s3:*",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     b.Arn,
 *                     $"{b.Arn}/*",
 *                 },
 *             },
 *         },
 *     });
 *     var pRolePolicy = new Aws.Iam.RolePolicy("p", new()
 *     {
 *         Name = "awsconfig-example",
 *         Role = r.Id,
 *         Policy = p.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cfg"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		b, err := s3.NewBucketV2(ctx, "b", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("awsconfig-example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		fooDeliveryChannel, err := cfg.NewDeliveryChannel(ctx, "foo", &cfg.DeliveryChannelArgs{
 * 			Name:         pulumi.String("example"),
 * 			S3BucketName: b.Bucket,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		assumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"config.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		r, err := iam.NewRole(ctx, "r", &iam.RoleArgs{
 * 			Name:             pulumi.String("example-awsconfig"),
 * 			AssumeRolePolicy: pulumi.String(assumeRole.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		fooRecorder, err := cfg.NewRecorder(ctx, "foo", &cfg.RecorderArgs{
 * 			Name:    pulumi.String("example"),
 * 			RoleArn: r.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cfg.NewRecorderStatus(ctx, "foo", &cfg.RecorderStatusArgs{
 * 			Name:      fooRecorder.Name,
 * 			IsEnabled: pulumi.Bool(true),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			fooDeliveryChannel,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iam.NewRolePolicyAttachment(ctx, "a", &iam.RolePolicyAttachmentArgs{
 * 			Role:      r.Name,
 * 			PolicyArn: pulumi.String("arn:aws:iam::aws:policy/service-role/AWS_ConfigRole"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		p := iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * 			Statements: iam.GetPolicyDocumentStatementArray{
 * 				&iam.GetPolicyDocumentStatementArgs{
 * 					Effect: pulumi.String("Allow"),
 * 					Actions: pulumi.StringArray{
 * 						pulumi.String("s3:*"),
 * 					},
 * 					Resources: pulumi.StringArray{
 * 						b.Arn,
 * 						b.Arn.ApplyT(func(arn string) (string, error) {
 * 							return fmt.Sprintf("%v/*", arn), nil
 * 						}).(pulumi.StringOutput),
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		_, err = iam.NewRolePolicy(ctx, "p", &iam.RolePolicyArgs{
 * 			Name: pulumi.String("awsconfig-example"),
 * 			Role: r.ID(),
 * 			Policy: pulumi.String(p.ApplyT(func(p iam.GetPolicyDocumentResult) (*string, error) {
 * 				return &p.Json, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.cfg.DeliveryChannel;
 * import com.pulumi.aws.cfg.DeliveryChannelArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.cfg.Recorder;
 * import com.pulumi.aws.cfg.RecorderArgs;
 * import com.pulumi.aws.cfg.RecorderStatus;
 * import com.pulumi.aws.cfg.RecorderStatusArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import com.pulumi.aws.iam.RolePolicy;
 * import com.pulumi.aws.iam.RolePolicyArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var b = new BucketV2("b", BucketV2Args.builder()
 *             .bucket("awsconfig-example")
 *             .build());
 *         var fooDeliveryChannel = new DeliveryChannel("fooDeliveryChannel", DeliveryChannelArgs.builder()
 *             .name("example")
 *             .s3BucketName(b.bucket())
 *             .build());
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("config.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var r = new Role("r", RoleArgs.builder()
 *             .name("example-awsconfig")
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var fooRecorder = new Recorder("fooRecorder", RecorderArgs.builder()
 *             .name("example")
 *             .roleArn(r.arn())
 *             .build());
 *         var foo = new RecorderStatus("foo", RecorderStatusArgs.builder()
 *             .name(fooRecorder.name())
 *             .isEnabled(true)
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(fooDeliveryChannel)
 *                 .build());
 *         var a = new RolePolicyAttachment("a", RolePolicyAttachmentArgs.builder()
 *             .role(r.name())
 *             .policyArn("arn:aws:iam::aws:policy/service-role/AWS_ConfigRole")
 *             .build());
 *         final var p = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .actions("s3:*")
 *                 .resources(
 *                     b.arn(),
 *                     b.arn().applyValue(arn -> String.format("%s/*", arn)))
 *                 .build())
 *             .build());
 *         var pRolePolicy = new RolePolicy("pRolePolicy", RolePolicyArgs.builder()
 *             .name("awsconfig-example")
 *             .role(r.id())
 *             .policy(p.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(p -> p.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:cfg:RecorderStatus
 *     properties:
 *       name: ${fooRecorder.name}
 *       isEnabled: true
 *     options:
 *       dependson:
 *         - ${fooDeliveryChannel}
 *   a:
 *     type: aws:iam:RolePolicyAttachment
 *     properties:
 *       role: ${r.name}
 *       policyArn: arn:aws:iam::aws:policy/service-role/AWS_ConfigRole
 *   b:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: awsconfig-example
 *   fooDeliveryChannel:
 *     type: aws:cfg:DeliveryChannel
 *     name: foo
 *     properties:
 *       name: example
 *       s3BucketName: ${b.bucket}
 *   fooRecorder:
 *     type: aws:cfg:Recorder
 *     name: foo
 *     properties:
 *       name: example
 *       roleArn: ${r.arn}
 *   r:
 *     type: aws:iam:Role
 *     properties:
 *       name: example-awsconfig
 *       assumeRolePolicy: ${assumeRole.json}
 *   pRolePolicy:
 *     type: aws:iam:RolePolicy
 *     name: p
 *     properties:
 *       name: awsconfig-example
 *       role: ${r.id}
 *       policy: ${p.json}
 * variables:
 *   assumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - config.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 *   p:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             actions:
 *               - s3:*
 *             resources:
 *               - ${b.arn}
 *               - ${b.arn}/*
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Configuration Recorder Status using the name of the Configuration Recorder. For example:
 * ```sh
 * $ pulumi import aws:cfg/recorderStatus:RecorderStatus foo example
 * ```
 * @property isEnabled Whether the configuration recorder should be enabled or disabled.
 * @property name The name of the recorder
 * */*/*/*/*/*/
 */
public data class RecorderStatusArgs(
    public val isEnabled: Output<Boolean>? = null,
    public val name: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.cfg.RecorderStatusArgs> {
    override fun toJava(): com.pulumi.aws.cfg.RecorderStatusArgs =
        com.pulumi.aws.cfg.RecorderStatusArgs.builder()
            .isEnabled(isEnabled?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [RecorderStatusArgs].
 */
@PulumiTagMarker
public class RecorderStatusArgsBuilder internal constructor() {
    private var isEnabled: Output<Boolean>? = null

    private var name: Output<String>? = null

    /**
     * @param value Whether the configuration recorder should be enabled or disabled.
     */
    @JvmName("dfkhiumlohftttee")
    public suspend fun isEnabled(`value`: Output<Boolean>) {
        this.isEnabled = value
    }

    /**
     * @param value The name of the recorder
     */
    @JvmName("ayvtrwwrbijjimcy")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Whether the configuration recorder should be enabled or disabled.
     */
    @JvmName("ysjnpawakrjhkxdl")
    public suspend fun isEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.isEnabled = mapped
    }

    /**
     * @param value The name of the recorder
     */
    @JvmName("xkpxkqdoxkkoigbt")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    internal fun build(): RecorderStatusArgs = RecorderStatusArgs(
        isEnabled = isEnabled,
        name = name,
    )
}
