@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cfg.kotlin

import com.pulumi.aws.cfg.RuleArgs.builder
import com.pulumi.aws.cfg.kotlin.inputs.RuleEvaluationModeArgs
import com.pulumi.aws.cfg.kotlin.inputs.RuleEvaluationModeArgsBuilder
import com.pulumi.aws.cfg.kotlin.inputs.RuleScopeArgs
import com.pulumi.aws.cfg.kotlin.inputs.RuleScopeArgsBuilder
import com.pulumi.aws.cfg.kotlin.inputs.RuleSourceArgs
import com.pulumi.aws.cfg.kotlin.inputs.RuleSourceArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an AWS Config Rule.
 * > **Note:** Config Rule requires an existing Configuration Recorder to be present. Use of `depends_on` is recommended (as shown below) to avoid race conditions.
 * ## Example Usage
 * ### AWS Managed Rules
 * AWS managed rules can be used by setting the source owner to `AWS` and the source identifier to the name of the managed rule. More information about AWS managed rules can be found in the [AWS Config Developer Guide](https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html).
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const assumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["config&#46;amazonaws&#46;com"],
 *         }],
 *         actions: ["sts:AssumeRole"],
 *     }],
 * });
 * const rRole = new aws.iam.Role("r", {
 *     name: "my-awsconfig-role",
 *     assumeRolePolicy: assumeRole.then(assumeRole => assumeRole.json),
 * });
 * const foo = new aws.cfg.Recorder("foo", {
 *     name: "example",
 *     roleArn: rRole.arn,
 * });
 * const r = new aws.cfg.Rule("r", {
 *     name: "example",
 *     source: {
 *         owner: "AWS",
 *         sourceIdentifier: "S3_BUCKET_VERSIONING_ENABLED",
 *     },
 * }, {
 *     dependsOn: [foo],
 * });
 * const p = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         actions: ["config:Put*"],
 *         resources: ["*"],
 *     }],
 * });
 * const pRolePolicy = new aws.iam.RolePolicy("p", {
 *     name: "my-awsconfig-policy",
 *     role: rRole.id,
 *     policy: p.then(p => p.json),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["config&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": ["sts:AssumeRole"],
 * }])
 * r_role = aws.iam.Role("r",
 *     name="my-awsconfig-role",
 *     assume_role_policy=assume_role.json)
 * foo = aws.cfg.Recorder("foo",
 *     name="example",
 *     role_arn=r_role.arn)
 * r = aws.cfg.Rule("r",
 *     name="example",
 *     source={
 *         "owner": "AWS",
 *         "source_identifier": "S3_BUCKET_VERSIONING_ENABLED",
 *     },
 *     opts = pulumi.ResourceOptions(depends_on=[foo]))
 * p = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "actions": ["config:Put*"],
 *     "resources": ["*"],
 * }])
 * p_role_policy = aws.iam.RolePolicy("p",
 *     name="my-awsconfig-policy",
 *     role=r_role.id,
 *     policy=p.json)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var assumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "config.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *             },
 *         },
 *     });
 *     var rRole = new Aws.Iam.Role("r", new()
 *     {
 *         Name = "my-awsconfig-role",
 *         AssumeRolePolicy = assumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var foo = new Aws.Cfg.Recorder("foo", new()
 *     {
 *         Name = "example",
 *         RoleArn = rRole.Arn,
 *     });
 *     var r = new Aws.Cfg.Rule("r", new()
 *     {
 *         Name = "example",
 *         Source = new Aws.Cfg.Inputs.RuleSourceArgs
 *         {
 *             Owner = "AWS",
 *             SourceIdentifier = "S3_BUCKET_VERSIONING_ENABLED",
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             foo,
 *         },
 *     });
 *     var p = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Actions = new[]
 *                 {
 *                     "config:Put*",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     "*",
 *                 },
 *             },
 *         },
 *     });
 *     var pRolePolicy = new Aws.Iam.RolePolicy("p", new()
 *     {
 *         Name = "my-awsconfig-policy",
 *         Role = rRole.Id,
 *         Policy = p.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cfg"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		assumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"config.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		rRole, err := iam.NewRole(ctx, "r", &iam.RoleArgs{
 * 			Name:             pulumi.String("my-awsconfig-role"),
 * 			AssumeRolePolicy: pulumi.String(assumeRole.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		foo, err := cfg.NewRecorder(ctx, "foo", &cfg.RecorderArgs{
 * 			Name:    pulumi.String("example"),
 * 			RoleArn: rRole.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cfg.NewRule(ctx, "r", &cfg.RuleArgs{
 * 			Name: pulumi.String("example"),
 * 			Source: &cfg.RuleSourceArgs{
 * 				Owner:            pulumi.String("AWS"),
 * 				SourceIdentifier: pulumi.String("S3_BUCKET_VERSIONING_ENABLED"),
 * 			},
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			foo,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		p, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Actions: []string{
 * 						"config:Put*",
 * 					},
 * 					Resources: []string{
 * 						"*",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iam.NewRolePolicy(ctx, "p", &iam.RolePolicyArgs{
 * 			Name:   pulumi.String("my-awsconfig-policy"),
 * 			Role:   rRole.ID(),
 * 			Policy: pulumi.String(p.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.cfg.Recorder;
 * import com.pulumi.aws.cfg.RecorderArgs;
 * import com.pulumi.aws.cfg.Rule;
 * import com.pulumi.aws.cfg.RuleArgs;
 * import com.pulumi.aws.cfg.inputs.RuleSourceArgs;
 * import com.pulumi.aws.iam.RolePolicy;
 * import com.pulumi.aws.iam.RolePolicyArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("config.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var rRole = new Role("rRole", RoleArgs.builder()
 *             .name("my-awsconfig-role")
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var foo = new Recorder("foo", RecorderArgs.builder()
 *             .name("example")
 *             .roleArn(rRole.arn())
 *             .build());
 *         var r = new Rule("r", RuleArgs.builder()
 *             .name("example")
 *             .source(RuleSourceArgs.builder()
 *                 .owner("AWS")
 *                 .sourceIdentifier("S3_BUCKET_VERSIONING_ENABLED")
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(foo)
 *                 .build());
 *         final var p = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .actions("config:Put*")
 *                 .resources("*")
 *                 .build())
 *             .build());
 *         var pRolePolicy = new RolePolicy("pRolePolicy", RolePolicyArgs.builder()
 *             .name("my-awsconfig-policy")
 *             .role(rRole.id())
 *             .policy(p.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   r:
 *     type: aws:cfg:Rule
 *     properties:
 *       name: example
 *       source:
 *         owner: AWS
 *         sourceIdentifier: S3_BUCKET_VERSIONING_ENABLED
 *     options:
 *       dependson:
 *         - ${foo}
 *   foo:
 *     type: aws:cfg:Recorder
 *     properties:
 *       name: example
 *       roleArn: ${rRole.arn}
 *   rRole:
 *     type: aws:iam:Role
 *     name: r
 *     properties:
 *       name: my-awsconfig-role
 *       assumeRolePolicy: ${assumeRole.json}
 *   pRolePolicy:
 *     type: aws:iam:RolePolicy
 *     name: p
 *     properties:
 *       name: my-awsconfig-policy
 *       role: ${rRole.id}
 *       policy: ${p.json}
 * variables:
 *   assumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - config.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 *   p:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             actions:
 *               - config:Put*
 *             resources:
 *               - '*'
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Custom Rules
 * Custom rules can be used by setting the source owner to `CUSTOM_LAMBDA` and the source identifier to the Amazon Resource Name (ARN) of the Lambda Function. The AWS Config service must have permissions to invoke the Lambda Function, e.g., via the `aws.lambda.Permission` resource. More information about custom rules can be found in the [AWS Config Developer Guide](https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules.html).
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cfg.Recorder("example", {});
 * const exampleFunction = new aws.lambda.Function("example", {});
 * const examplePermission = new aws.lambda.Permission("example", {
 *     action: "lambda:InvokeFunction",
 *     "function": exampleFunction.arn,
 *     principal: "config.amazonaws.com",
 *     statementId: "AllowExecutionFromConfig",
 * });
 * const exampleRule = new aws.cfg.Rule("example", {source: {
 *     owner: "CUSTOM_LAMBDA",
 *     sourceIdentifier: exampleFunction.arn,
 * }}, {
 *     dependsOn: [
 *         example,
 *         examplePermission,
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cfg.Recorder("example")
 * example_function = aws.lambda_.Function("example")
 * example_permission = aws.lambda_.Permission("example",
 *     action="lambda:InvokeFunction",
 *     function=example_function.arn,
 *     principal="config.amazonaws.com",
 *     statement_id="AllowExecutionFromConfig")
 * example_rule = aws.cfg.Rule("example", source={
 *     "owner": "CUSTOM_LAMBDA",
 *     "source_identifier": example_function.arn,
 * },
 * opts = pulumi.ResourceOptions(depends_on=[
 *         example,
 *         example_permission,
 *     ]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Cfg.Recorder("example");
 *     var exampleFunction = new Aws.Lambda.Function("example");
 *     var examplePermission = new Aws.Lambda.Permission("example", new()
 *     {
 *         Action = "lambda:InvokeFunction",
 *         Function = exampleFunction.Arn,
 *         Principal = "config.amazonaws.com",
 *         StatementId = "AllowExecutionFromConfig",
 *     });
 *     var exampleRule = new Aws.Cfg.Rule("example", new()
 *     {
 *         Source = new Aws.Cfg.Inputs.RuleSourceArgs
 *         {
 *             Owner = "CUSTOM_LAMBDA",
 *             SourceIdentifier = exampleFunction.Arn,
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             example,
 *             examplePermission,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cfg"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lambda"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := cfg.NewRecorder(ctx, "example", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleFunction, err := lambda.NewFunction(ctx, "example", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		examplePermission, err := lambda.NewPermission(ctx, "example", &lambda.PermissionArgs{
 * 			Action:      pulumi.String("lambda:InvokeFunction"),
 * 			Function:    exampleFunction.Arn,
 * 			Principal:   pulumi.String("config.amazonaws.com"),
 * 			StatementId: pulumi.String("AllowExecutionFromConfig"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cfg.NewRule(ctx, "example", &cfg.RuleArgs{
 * 			Source: &cfg.RuleSourceArgs{
 * 				Owner:            pulumi.String("CUSTOM_LAMBDA"),
 * 				SourceIdentifier: exampleFunction.Arn,
 * 			},
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			example,
 * 			examplePermission,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cfg.Recorder;
 * import com.pulumi.aws.lambda.Function;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import com.pulumi.aws.cfg.Rule;
 * import com.pulumi.aws.cfg.RuleArgs;
 * import com.pulumi.aws.cfg.inputs.RuleSourceArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Recorder("example");
 *         var exampleFunction = new Function("exampleFunction");
 *         var examplePermission = new Permission("examplePermission", PermissionArgs.builder()
 *             .action("lambda:InvokeFunction")
 *             .function(exampleFunction.arn())
 *             .principal("config.amazonaws.com")
 *             .statementId("AllowExecutionFromConfig")
 *             .build());
 *         var exampleRule = new Rule("exampleRule", RuleArgs.builder()
 *             .source(RuleSourceArgs.builder()
 *                 .owner("CUSTOM_LAMBDA")
 *                 .sourceIdentifier(exampleFunction.arn())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(
 *                     example,
 *                     examplePermission)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cfg:Recorder
 *   exampleFunction:
 *     type: aws:lambda:Function
 *     name: example
 *   examplePermission:
 *     type: aws:lambda:Permission
 *     name: example
 *     properties:
 *       action: lambda:InvokeFunction
 *       function: ${exampleFunction.arn}
 *       principal: config.amazonaws.com
 *       statementId: AllowExecutionFromConfig
 *   exampleRule:
 *     type: aws:cfg:Rule
 *     name: example
 *     properties:
 *       source:
 *         owner: CUSTOM_LAMBDA
 *         sourceIdentifier: ${exampleFunction.arn}
 *     options:
 *       dependson:
 *         - ${example}
 *         - ${examplePermission}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Custom Policies
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cfg.Rule("example", {
 *     name: "example",
 *     source: {
 *         owner: "CUSTOM_POLICY",
 *         sourceDetails: [{
 *             messageType: "ConfigurationItemChangeNotification",
 *         }],
 *         customPolicyDetails: {
 *             policyRuntime: "guard-2.x.x",
 *             policyText: `\x09  rule tableisactive when
 * \x09\x09  resourceType == "AWS::DynamoDB::Table" {
 * \x09\x09  configuration.tableStatus == ['ACTIVE']
 * \x09  }
 * \x09
 * \x09  rule checkcompliance when
 * \x09\x09  resourceType == "AWS::DynamoDB::Table"
 * \x09\x09  tableisactive {
 * \x09\x09\x09  supplementaryConfiguration.ContinuousBackupsDescription.pointInTimeRecoveryDescription.pointInTimeRecoveryStatus == "ENABLED"
 * \x09  }
 * `,
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cfg.Rule("example",
 *     name="example",
 *     source={
 *         "owner": "CUSTOM_POLICY",
 *         "source_details": [{
 *             "message_type": "ConfigurationItemChangeNotification",
 *         }],
 *         "custom_policy_details": {
 *             "policy_runtime": "guard-2.x.x",
 *             "policy_text": """\x09  rule tableisactive when
 * \x09\x09  resourceType == "AWS::DynamoDB::Table" {
 * \x09\x09  configuration.tableStatus == ['ACTIVE']
 * \x09  }
 * \x09
 * \x09  rule checkcompliance when
 * \x09\x09  resourceType == "AWS::DynamoDB::Table"
 * \x09\x09  tableisactive {
 * \x09\x09\x09  supplementaryConfiguration.ContinuousBackupsDescription.pointInTimeRecoveryDescription.pointInTimeRecoveryStatus == "ENABLED"
 * \x09  }
 * """,
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Cfg.Rule("example", new()
 *     {
 *         Name = "example",
 *         Source = new Aws.Cfg.Inputs.RuleSourceArgs
 *         {
 *             Owner = "CUSTOM_POLICY",
 *             SourceDetails = new[]
 *             {
 *                 new Aws.Cfg.Inputs.RuleSourceSourceDetailArgs
 *                 {
 *                     MessageType = "ConfigurationItemChangeNotification",
 *                 },
 *             },
 *             CustomPolicyDetails = new Aws.Cfg.Inputs.RuleSourceCustomPolicyDetailsArgs
 *             {
 *                 PolicyRuntime = "guard-2.x.x",
 *                 PolicyText = @"	  rule tableisactive when
 * 		  resourceType == ""AWS::DynamoDB::Table"" {
 * 		  configuration.tableStatus == ['ACTIVE']
 * 	  }
 * 	  rule checkcompliance when
 * 		  resourceType == ""AWS::DynamoDB::Table""
 * 		  tableisactive {
 * 			  supplementaryConfiguration.ContinuousBackupsDescription.pointInTimeRecoveryDescription.pointInTimeRecoveryStatus == ""ENABLED""
 * 	  }
 * ",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cfg"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cfg.NewRule(ctx, "example", &cfg.RuleArgs{
 * 			Name: pulumi.String("example"),
 * 			Source: &cfg.RuleSourceArgs{
 * 				Owner: pulumi.String("CUSTOM_POLICY"),
 * 				SourceDetails: cfg.RuleSourceSourceDetailArray{
 * 					&cfg.RuleSourceSourceDetailArgs{
 * 						MessageType: pulumi.String("ConfigurationItemChangeNotification"),
 * 					},
 * 				},
 * 				CustomPolicyDetails: &cfg.RuleSourceCustomPolicyDetailsArgs{
 * 					PolicyRuntime: pulumi.String("guard-2.x.x"),
 * 					PolicyText: pulumi.String(`	  rule tableisactive when
 * 		  resourceType == "AWS::DynamoDB::Table" {
 * 		  configuration.tableStatus == ['ACTIVE']
 * 	  }
 * 	  rule checkcompliance when
 * 		  resourceType == "AWS::DynamoDB::Table"
 * 		  tableisactive {
 * 			  supplementaryConfiguration.ContinuousBackupsDescription.pointInTimeRecoveryDescription.pointInTimeRecoveryStatus == "ENABLED"
 * 	  }
 * `),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cfg.Rule;
 * import com.pulumi.aws.cfg.RuleArgs;
 * import com.pulumi.aws.cfg.inputs.RuleSourceArgs;
 * import com.pulumi.aws.cfg.inputs.RuleSourceCustomPolicyDetailsArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Rule("example", RuleArgs.builder()
 *             .name("example")
 *             .source(RuleSourceArgs.builder()
 *                 .owner("CUSTOM_POLICY")
 *                 .sourceDetails(RuleSourceSourceDetailArgs.builder()
 *                     .messageType("ConfigurationItemChangeNotification")
 *                     .build())
 *                 .customPolicyDetails(RuleSourceCustomPolicyDetailsArgs.builder()
 *                     .policyRuntime("guard-2.x.x")
 *                     .policyText("""
 * 	  rule tableisactive when
 * 		  resourceType == "AWS::DynamoDB::Table" {
 * 		  configuration.tableStatus == ['ACTIVE']
 * 	  }
 * 	  rule checkcompliance when
 * 		  resourceType == "AWS::DynamoDB::Table"
 * 		  tableisactive {
 * 			  supplementaryConfiguration.ContinuousBackupsDescription.pointInTimeRecoveryDescription.pointInTimeRecoveryStatus == "ENABLED"
 * 	  }
 *                     """)
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cfg:Rule
 *     properties:
 *       name: example
 *       source:
 *         owner: CUSTOM_POLICY
 *         sourceDetails:
 *           - messageType: ConfigurationItemChangeNotification
 *         customPolicyDetails:
 *           policyRuntime: guard-2.x.x
 *           policyText: "\t  rule tableisactive when\n\t\t  resourceType == \"AWS::DynamoDB::Table\" {\n\t\t  configuration.tableStatus == ['ACTIVE']\n\t  }\n\t  \n\t  rule checkcompliance when\n\t\t  resourceType == \"AWS::DynamoDB::Table\"\n\t\t  tableisactive {\n\t\t\t  supplementaryConfiguration.ContinuousBackupsDescription.pointInTimeRecoveryDescription.pointInTimeRecoveryStatus == \"ENABLED\"\n\t  }\n"
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Config Rule using the name. For example:
 * ```sh
 * $ pulumi import aws:cfg/rule:Rule foo example
 * ```
 * @property description Description of the rule
 * @property evaluationModes The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
 * @property inputParameters A string in JSON format that is passed to the AWS Config rule Lambda function.
 * @property maximumExecutionFrequency The maximum frequency with which AWS Config runs evaluations for a rule.
 * @property name The name of the rule
 * @property scope Scope defines which resources can trigger an evaluation for the rule. See Scope Below.
 * @property source Source specifies the rule owner, the rule identifier, and the notifications that cause the function to evaluate your AWS resources. See Source Below.
 * @property tags A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class RuleArgs(
    public val description: Output<String>? = null,
    public val evaluationModes: Output<List<RuleEvaluationModeArgs>>? = null,
    public val inputParameters: Output<String>? = null,
    public val maximumExecutionFrequency: Output<String>? = null,
    public val name: Output<String>? = null,
    public val scope: Output<RuleScopeArgs>? = null,
    public val source: Output<RuleSourceArgs>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.cfg.RuleArgs> {
    override fun toJava(): com.pulumi.aws.cfg.RuleArgs = com.pulumi.aws.cfg.RuleArgs.builder()
        .description(description?.applyValue({ args0 -> args0 }))
        .evaluationModes(
            evaluationModes?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        )
        .inputParameters(inputParameters?.applyValue({ args0 -> args0 }))
        .maximumExecutionFrequency(maximumExecutionFrequency?.applyValue({ args0 -> args0 }))
        .name(name?.applyValue({ args0 -> args0 }))
        .scope(scope?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .source(source?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .tags(
            tags?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }),
        ).build()
}

/**
 * Builder for [RuleArgs].
 */
@PulumiTagMarker
public class RuleArgsBuilder internal constructor() {
    private var description: Output<String>? = null

    private var evaluationModes: Output<List<RuleEvaluationModeArgs>>? = null

    private var inputParameters: Output<String>? = null

    private var maximumExecutionFrequency: Output<String>? = null

    private var name: Output<String>? = null

    private var scope: Output<RuleScopeArgs>? = null

    private var source: Output<RuleSourceArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value Description of the rule
     */
    @JvmName("ukpcpbwlwjgkyvmd")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
     */
    @JvmName("tbsafdehwotljoqc")
    public suspend fun evaluationModes(`value`: Output<List<RuleEvaluationModeArgs>>) {
        this.evaluationModes = value
    }

    @JvmName("lhukximictuvvddb")
    public suspend fun evaluationModes(vararg values: Output<RuleEvaluationModeArgs>) {
        this.evaluationModes = Output.all(values.asList())
    }

    /**
     * @param values The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
     */
    @JvmName("xpphjpxgpfichjxj")
    public suspend fun evaluationModes(values: List<Output<RuleEvaluationModeArgs>>) {
        this.evaluationModes = Output.all(values)
    }

    /**
     * @param value A string in JSON format that is passed to the AWS Config rule Lambda function.
     */
    @JvmName("ernrcjuicafvsweb")
    public suspend fun inputParameters(`value`: Output<String>) {
        this.inputParameters = value
    }

    /**
     * @param value The maximum frequency with which AWS Config runs evaluations for a rule.
     */
    @JvmName("qmderahyngvcjjle")
    public suspend fun maximumExecutionFrequency(`value`: Output<String>) {
        this.maximumExecutionFrequency = value
    }

    /**
     * @param value The name of the rule
     */
    @JvmName("oitkpdfqivatnipa")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Scope defines which resources can trigger an evaluation for the rule. See Scope Below.
     */
    @JvmName("gdjbvwsomnjkgtni")
    public suspend fun scope(`value`: Output<RuleScopeArgs>) {
        this.scope = value
    }

    /**
     * @param value Source specifies the rule owner, the rule identifier, and the notifications that cause the function to evaluate your AWS resources. See Source Below.
     */
    @JvmName("fmlyfbqccliuybul")
    public suspend fun source(`value`: Output<RuleSourceArgs>) {
        this.source = value
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cocfhpmdveakbail")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Description of the rule
     */
    @JvmName("bvwmvfsnnctcxgeb")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
     */
    @JvmName("lqpifemfevypyijy")
    public suspend fun evaluationModes(`value`: List<RuleEvaluationModeArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.evaluationModes = mapped
    }

    /**
     * @param argument The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
     */
    @JvmName("jffjwdtmqdaejora")
    public suspend fun evaluationModes(argument: List<suspend RuleEvaluationModeArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            RuleEvaluationModeArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.evaluationModes = mapped
    }

    /**
     * @param argument The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
     */
    @JvmName("xwjihqrpctycewyj")
    public suspend fun evaluationModes(vararg argument: suspend RuleEvaluationModeArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            RuleEvaluationModeArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.evaluationModes = mapped
    }

    /**
     * @param argument The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
     */
    @JvmName("smgedardvqjeurmt")
    public suspend fun evaluationModes(argument: suspend RuleEvaluationModeArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(RuleEvaluationModeArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.evaluationModes = mapped
    }

    /**
     * @param values The modes the Config rule can be evaluated in. See Evaluation Mode for more details.
     */
    @JvmName("ubjfhdhqmbwodflk")
    public suspend fun evaluationModes(vararg values: RuleEvaluationModeArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.evaluationModes = mapped
    }

    /**
     * @param value A string in JSON format that is passed to the AWS Config rule Lambda function.
     */
    @JvmName("bmiwtnsnsnadisbg")
    public suspend fun inputParameters(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.inputParameters = mapped
    }

    /**
     * @param value The maximum frequency with which AWS Config runs evaluations for a rule.
     */
    @JvmName("kmmkegwrkxgxyqta")
    public suspend fun maximumExecutionFrequency(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maximumExecutionFrequency = mapped
    }

    /**
     * @param value The name of the rule
     */
    @JvmName("eakyvjbdisinbrjj")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Scope defines which resources can trigger an evaluation for the rule. See Scope Below.
     */
    @JvmName("mamuevrapbyahgyh")
    public suspend fun scope(`value`: RuleScopeArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.scope = mapped
    }

    /**
     * @param argument Scope defines which resources can trigger an evaluation for the rule. See Scope Below.
     */
    @JvmName("scldosohtvreseqs")
    public suspend fun scope(argument: suspend RuleScopeArgsBuilder.() -> Unit) {
        val toBeMapped = RuleScopeArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.scope = mapped
    }

    /**
     * @param value Source specifies the rule owner, the rule identifier, and the notifications that cause the function to evaluate your AWS resources. See Source Below.
     */
    @JvmName("nlcskjmvlcjwxlrl")
    public suspend fun source(`value`: RuleSourceArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.source = mapped
    }

    /**
     * @param argument Source specifies the rule owner, the rule identifier, and the notifications that cause the function to evaluate your AWS resources. See Source Below.
     */
    @JvmName("kwmflyskhwflirys")
    public suspend fun source(argument: suspend RuleSourceArgsBuilder.() -> Unit) {
        val toBeMapped = RuleSourceArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.source = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jnogqwgxexdpityd")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ildukidjxcnpwjyp")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): RuleArgs = RuleArgs(
        description = description,
        evaluationModes = evaluationModes,
        inputParameters = inputParameters,
        maximumExecutionFrequency = maximumExecutionFrequency,
        name = name,
        scope = scope,
        source = source,
        tags = tags,
    )
}
