@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cloud9.kotlin

import com.pulumi.aws.cloud9.EnvironmentEC2Args.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a Cloud9 EC2 Development Environment.
 * ## Example Usage
 * Basic usage:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloud9.EnvironmentEC2("example", {
 *     instanceType: "t2.micro",
 *     name: "example-env",
 *     imageId: "amazonlinux-2023-x86_64",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloud9.EnvironmentEC2("example",
 *     instance_type="t2.micro",
 *     name="example-env",
 *     image_id="amazonlinux-2023-x86_64")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Cloud9.EnvironmentEC2("example", new()
 *     {
 *         InstanceType = "t2.micro",
 *         Name = "example-env",
 *         ImageId = "amazonlinux-2023-x86_64",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloud9"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloud9.NewEnvironmentEC2(ctx, "example", &cloud9.EnvironmentEC2Args{
 * 			InstanceType: pulumi.String("t2.micro"),
 * 			Name:         pulumi.String("example-env"),
 * 			ImageId:      pulumi.String("amazonlinux-2023-x86_64"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloud9.EnvironmentEC2;
 * import com.pulumi.aws.cloud9.EnvironmentEC2Args;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new EnvironmentEC2("example", EnvironmentEC2Args.builder()
 *             .instanceType("t2.micro")
 *             .name("example-env")
 *             .imageId("amazonlinux-2023-x86_64")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloud9:EnvironmentEC2
 *     properties:
 *       instanceType: t2.micro
 *       name: example-env
 *       imageId: amazonlinux-2023-x86_64
 * ```
 * <!--End PulumiCodeChooser -->
 * Get the URL of the Cloud9 environment after creation:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloud9.EnvironmentEC2("example", {instanceType: "t2.micro"});
 * const cloud9Instance = aws.ec2.getInstanceOutput({
 *     filters: [{
 *         name: "tag:aws:cloud9:environment",
 *         values: [example&#46;id],
 *     }],
 * });
 * export const cloud9Url = pulumi.interpolate`https://${region}.console.aws.amazon.com/cloud9/ide/${example.id}`;
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloud9.EnvironmentEC2("example", instance_type="t2.micro")
 * cloud9_instance = aws.ec2.get_instance_output(filters=[{
 *     "name": "tag:aws:cloud9:environment",
 *     "values": [example&#46;id],
 * }])
 * pulumi.export("cloud9Url", example.id.apply(lambda id: f"https://{region}.console.aws.amazon.com/cloud9/ide/{id}"))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Cloud9.EnvironmentEC2("example", new()
 *     {
 *         InstanceType = "t2.micro",
 *     });
 *     var cloud9Instance = Aws.Ec2.GetInstance.Invoke(new()
 *     {
 *         Filters = new[]
 *         {
 *             new Aws.Ec2.Inputs.GetInstanceFilterInputArgs
 *             {
 *                 Name = "tag:aws:cloud9:environment",
 *                 Values = new[]
 *                 {
 *                     example.Id,
 *                 },
 *             },
 *         },
 *     });
 *     return new Dictionary<string, object?>
 *     {
 *         ["cloud9Url"] = example.Id.Apply(id => $"https://{region}.console.aws.amazon.com/cloud9/ide/{id}"),
 *     };
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloud9"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := cloud9.NewEnvironmentEC2(ctx, "example", &cloud9.EnvironmentEC2Args{
 * 			InstanceType: pulumi.String("t2.micro"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_ = ec2.LookupInstanceOutput(ctx, ec2.GetInstanceOutputArgs{
 * 			Filters: ec2.GetInstanceFilterArray{
 * 				&ec2.GetInstanceFilterArgs{
 * 					Name: pulumi.String("tag:aws:cloud9:environment"),
 * 					Values: pulumi.StringArray{
 * 						example.ID(),
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		ctx.Export("cloud9Url", example.ID().ApplyT(func(id string) (string, error) {
 * 			return fmt.Sprintf("https://%v.console.aws.amazon.com/cloud9/ide/%v", region, id), nil
 * 		}).(pulumi.StringOutput))
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloud9.EnvironmentEC2;
 * import com.pulumi.aws.cloud9.EnvironmentEC2Args;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetInstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new EnvironmentEC2("example", EnvironmentEC2Args.builder()
 *             .instanceType("t2.micro")
 *             .build());
 *         final var cloud9Instance = Ec2Functions.getInstance(GetInstanceArgs.builder()
 *             .filters(GetInstanceFilterArgs.builder()
 *                 .name("tag:aws:cloud9:environment")
 *                 .values(example.id())
 *                 .build())
 *             .build());
 *         ctx.export("cloud9Url", example.id().applyValue(id -> String.format("https://%s.console.aws.amazon.com/cloud9/ide/%s", region,id)));
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloud9:EnvironmentEC2
 *     properties:
 *       instanceType: t2.micro
 * variables:
 *   cloud9Instance:
 *     fn::invoke:
 *       Function: aws:ec2:getInstance
 *       Arguments:
 *         filters:
 *           - name: tag:aws:cloud9:environment
 *             values:
 *               - ${example.id}
 * outputs:
 *   cloud9Url: https://${region}.console.aws.amazon.com/cloud9/ide/${example.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * Allocate a static IP to the Cloud9 environment:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloud9.EnvironmentEC2("example", {instanceType: "t2.micro"});
 * const cloud9Instance = aws.ec2.getInstanceOutput({
 *     filters: [{
 *         name: "tag:aws:cloud9:environment",
 *         values: [example&#46;id],
 *     }],
 * });
 * const cloud9Eip = new aws.ec2.Eip("cloud9_eip", {
 *     instance: cloud9Instance.apply(cloud9Instance => cloud9Instance.id),
 *     domain: "vpc",
 * });
 * export const cloud9PublicIp = cloud9Eip.publicIp;
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloud9.EnvironmentEC2("example", instance_type="t2.micro")
 * cloud9_instance = aws.ec2.get_instance_output(filters=[{
 *     "name": "tag:aws:cloud9:environment",
 *     "values": [example&#46;id],
 * }])
 * cloud9_eip = aws.ec2.Eip("cloud9_eip",
 *     instance=cloud9_instance.id,
 *     domain="vpc")
 * pulumi.export("cloud9PublicIp", cloud9_eip.public_ip)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Cloud9.EnvironmentEC2("example", new()
 *     {
 *         InstanceType = "t2.micro",
 *     });
 *     var cloud9Instance = Aws.Ec2.GetInstance.Invoke(new()
 *     {
 *         Filters = new[]
 *         {
 *             new Aws.Ec2.Inputs.GetInstanceFilterInputArgs
 *             {
 *                 Name = "tag:aws:cloud9:environment",
 *                 Values = new[]
 *                 {
 *                     example.Id,
 *                 },
 *             },
 *         },
 *     });
 *     var cloud9Eip = new Aws.Ec2.Eip("cloud9_eip", new()
 *     {
 *         Instance = cloud9Instance.Apply(getInstanceResult => getInstanceResult.Id),
 *         Domain = "vpc",
 *     });
 *     return new Dictionary<string, object?>
 *     {
 *         ["cloud9PublicIp"] = cloud9Eip.PublicIp,
 *     };
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloud9"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := cloud9.NewEnvironmentEC2(ctx, "example", &cloud9.EnvironmentEC2Args{
 * 			InstanceType: pulumi.String("t2.micro"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		cloud9Instance := ec2.LookupInstanceOutput(ctx, ec2.GetInstanceOutputArgs{
 * 			Filters: ec2.GetInstanceFilterArray{
 * 				&ec2.GetInstanceFilterArgs{
 * 					Name: pulumi.String("tag:aws:cloud9:environment"),
 * 					Values: pulumi.StringArray{
 * 						example.ID(),
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		cloud9Eip, err := ec2.NewEip(ctx, "cloud9_eip", &ec2.EipArgs{
 * 			Instance: pulumi.String(cloud9Instance.ApplyT(func(cloud9Instance ec2.GetInstanceResult) (*string, error) {
 * 				return &cloud9Instance.Id, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 			Domain: pulumi.String("vpc"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		ctx.Export("cloud9PublicIp", cloud9Eip.PublicIp)
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloud9.EnvironmentEC2;
 * import com.pulumi.aws.cloud9.EnvironmentEC2Args;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetInstanceArgs;
 * import com.pulumi.aws.ec2.Eip;
 * import com.pulumi.aws.ec2.EipArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new EnvironmentEC2("example", EnvironmentEC2Args.builder()
 *             .instanceType("t2.micro")
 *             .build());
 *         final var cloud9Instance = Ec2Functions.getInstance(GetInstanceArgs.builder()
 *             .filters(GetInstanceFilterArgs.builder()
 *                 .name("tag:aws:cloud9:environment")
 *                 .values(example.id())
 *                 .build())
 *             .build());
 *         var cloud9Eip = new Eip("cloud9Eip", EipArgs.builder()
 *             .instance(cloud9Instance.applyValue(getInstanceResult -> getInstanceResult).applyValue(cloud9Instance -> cloud9Instance.applyValue(getInstanceResult -> getInstanceResult.id())))
 *             .domain("vpc")
 *             .build());
 *         ctx.export("cloud9PublicIp", cloud9Eip.publicIp());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloud9:EnvironmentEC2
 *     properties:
 *       instanceType: t2.micro
 *   cloud9Eip:
 *     type: aws:ec2:Eip
 *     name: cloud9_eip
 *     properties:
 *       instance: ${cloud9Instance.id}
 *       domain: vpc
 * variables:
 *   cloud9Instance:
 *     fn::invoke:
 *       Function: aws:ec2:getInstance
 *       Arguments:
 *         filters:
 *           - name: tag:aws:cloud9:environment
 *             values:
 *               - ${example.id}
 * outputs:
 *   cloud9PublicIp: ${cloud9Eip.publicIp}
 * ```
 * <!--End PulumiCodeChooser -->
 * @property automaticStopTimeMinutes The number of minutes until the running instance is shut down after the environment has last been used.
 * @property connectionType The connection type used for connecting to an Amazon EC2 environment. Valid values are `CONNECT_SSH` and `CONNECT_SSM`. For more information please refer [AWS documentation for Cloud9](https://docs.aws.amazon.com/cloud9/latest/user-guide/ec2-ssm.html).
 * @property description The description of the environment.
 * @property imageId The identifier for the Amazon Machine Image (AMI) that's used to create the EC2 instance. Valid values are
 * * `amazonlinux-2-x86_64`
 * * `amazonlinux-2023-x86_64`
 * * `ubuntu-18.04-x86_64`
 * * `ubuntu-22.04-x86_64`
 * * `resolve:ssm:/aws/service/cloud9/amis/amazonlinux-2-x86_64`
 * * `resolve:ssm:/aws/service/cloud9/amis/amazonlinux-2023-x86_64`
 * * `resolve:ssm:/aws/service/cloud9/amis/ubuntu-18.04-x86_64`
 * * `resolve:ssm:/aws/service/cloud9/amis/ubuntu-22.04-x86_64`
 * @property instanceType The type of instance to connect to the environment, e.g., `t2.micro`.
 * @property name The name of the environment.
 * @property ownerArn The ARN of the environment owner. This can be ARN of any AWS IAM principal. Defaults to the environment's creator.
 * @property subnetId The ID of the subnet in Amazon VPC that AWS Cloud9 will use to communicate with the Amazon EC2 instance.
 * @property tags Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class EnvironmentEC2Args(
    public val automaticStopTimeMinutes: Output<Int>? = null,
    public val connectionType: Output<String>? = null,
    public val description: Output<String>? = null,
    public val imageId: Output<String>? = null,
    public val instanceType: Output<String>? = null,
    public val name: Output<String>? = null,
    public val ownerArn: Output<String>? = null,
    public val subnetId: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.cloud9.EnvironmentEC2Args> {
    override fun toJava(): com.pulumi.aws.cloud9.EnvironmentEC2Args =
        com.pulumi.aws.cloud9.EnvironmentEC2Args.builder()
            .automaticStopTimeMinutes(automaticStopTimeMinutes?.applyValue({ args0 -> args0 }))
            .connectionType(connectionType?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .imageId(imageId?.applyValue({ args0 -> args0 }))
            .instanceType(instanceType?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .ownerArn(ownerArn?.applyValue({ args0 -> args0 }))
            .subnetId(subnetId?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [EnvironmentEC2Args].
 */
@PulumiTagMarker
public class EnvironmentEC2ArgsBuilder internal constructor() {
    private var automaticStopTimeMinutes: Output<Int>? = null

    private var connectionType: Output<String>? = null

    private var description: Output<String>? = null

    private var imageId: Output<String>? = null

    private var instanceType: Output<String>? = null

    private var name: Output<String>? = null

    private var ownerArn: Output<String>? = null

    private var subnetId: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value The number of minutes until the running instance is shut down after the environment has last been used.
     */
    @JvmName("htbpvlihwhduqtyr")
    public suspend fun automaticStopTimeMinutes(`value`: Output<Int>) {
        this.automaticStopTimeMinutes = value
    }

    /**
     * @param value The connection type used for connecting to an Amazon EC2 environment. Valid values are `CONNECT_SSH` and `CONNECT_SSM`. For more information please refer [AWS documentation for Cloud9](https://docs.aws.amazon.com/cloud9/latest/user-guide/ec2-ssm.html).
     */
    @JvmName("ocgwcekwegslelxy")
    public suspend fun connectionType(`value`: Output<String>) {
        this.connectionType = value
    }

    /**
     * @param value The description of the environment.
     */
    @JvmName("bgfnjtxxambixmah")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The identifier for the Amazon Machine Image (AMI) that's used to create the EC2 instance. Valid values are
     * * `amazonlinux-2-x86_64`
     * * `amazonlinux-2023-x86_64`
     * * `ubuntu-18.04-x86_64`
     * * `ubuntu-22.04-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/amazonlinux-2-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/amazonlinux-2023-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/ubuntu-18.04-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/ubuntu-22.04-x86_64`
     */
    @JvmName("agtjoriymjnvmkye")
    public suspend fun imageId(`value`: Output<String>) {
        this.imageId = value
    }

    /**
     * @param value The type of instance to connect to the environment, e.g., `t2.micro`.
     */
    @JvmName("mecxtmirmqayakfh")
    public suspend fun instanceType(`value`: Output<String>) {
        this.instanceType = value
    }

    /**
     * @param value The name of the environment.
     */
    @JvmName("vndnsmxoiyfnkxdq")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The ARN of the environment owner. This can be ARN of any AWS IAM principal. Defaults to the environment's creator.
     */
    @JvmName("nsnpirsybbaciche")
    public suspend fun ownerArn(`value`: Output<String>) {
        this.ownerArn = value
    }

    /**
     * @param value The ID of the subnet in Amazon VPC that AWS Cloud9 will use to communicate with the Amazon EC2 instance.
     */
    @JvmName("jgfhounmfnvduhxw")
    public suspend fun subnetId(`value`: Output<String>) {
        this.subnetId = value
    }

    /**
     * @param value Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ggeulnopjcfygoiq")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The number of minutes until the running instance is shut down after the environment has last been used.
     */
    @JvmName("fwaropabritcmnyo")
    public suspend fun automaticStopTimeMinutes(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.automaticStopTimeMinutes = mapped
    }

    /**
     * @param value The connection type used for connecting to an Amazon EC2 environment. Valid values are `CONNECT_SSH` and `CONNECT_SSM`. For more information please refer [AWS documentation for Cloud9](https://docs.aws.amazon.com/cloud9/latest/user-guide/ec2-ssm.html).
     */
    @JvmName("kqibtndtavaaxxmo")
    public suspend fun connectionType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.connectionType = mapped
    }

    /**
     * @param value The description of the environment.
     */
    @JvmName("reilxjvblrmqrpqj")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The identifier for the Amazon Machine Image (AMI) that's used to create the EC2 instance. Valid values are
     * * `amazonlinux-2-x86_64`
     * * `amazonlinux-2023-x86_64`
     * * `ubuntu-18.04-x86_64`
     * * `ubuntu-22.04-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/amazonlinux-2-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/amazonlinux-2023-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/ubuntu-18.04-x86_64`
     * * `resolve:ssm:/aws/service/cloud9/amis/ubuntu-22.04-x86_64`
     */
    @JvmName("sqbgtgtuagihptyt")
    public suspend fun imageId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.imageId = mapped
    }

    /**
     * @param value The type of instance to connect to the environment, e.g., `t2.micro`.
     */
    @JvmName("yqistbfgydelqwwa")
    public suspend fun instanceType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceType = mapped
    }

    /**
     * @param value The name of the environment.
     */
    @JvmName("crgiewyrickapnvw")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The ARN of the environment owner. This can be ARN of any AWS IAM principal. Defaults to the environment's creator.
     */
    @JvmName("mthvmljboiwnejpv")
    public suspend fun ownerArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ownerArn = mapped
    }

    /**
     * @param value The ID of the subnet in Amazon VPC that AWS Cloud9 will use to communicate with the Amazon EC2 instance.
     */
    @JvmName("qkcaeelktvoaajhv")
    public suspend fun subnetId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetId = mapped
    }

    /**
     * @param value Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("urbemxinnpiqeood")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("nmjfqoolgjxckhth")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): EnvironmentEC2Args = EnvironmentEC2Args(
        automaticStopTimeMinutes = automaticStopTimeMinutes,
        connectionType = connectionType,
        description = description,
        imageId = imageId,
        instanceType = instanceType,
        name = name,
        ownerArn = ownerArn,
        subnetId = subnetId,
        tags = tags,
    )
}
