@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cloudformation.kotlin

import com.pulumi.aws.cloudformation.StackSetInstanceArgs.builder
import com.pulumi.aws.cloudformation.kotlin.inputs.StackSetInstanceDeploymentTargetsArgs
import com.pulumi.aws.cloudformation.kotlin.inputs.StackSetInstanceDeploymentTargetsArgsBuilder
import com.pulumi.aws.cloudformation.kotlin.inputs.StackSetInstanceOperationPreferencesArgs
import com.pulumi.aws.cloudformation.kotlin.inputs.StackSetInstanceOperationPreferencesArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages a CloudFormation StackSet Instance. Instances are managed in the account and region of the StackSet after the target account permissions have been configured. Additional information about StackSets can be found in the [AWS CloudFormation User Guide](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html).
 * > **NOTE:** All target accounts must have an IAM Role created that matches the name of the execution role configured in the StackSet (the `execution_role_name` argument in the `aws.cloudformation.StackSet` resource) in a trust relationship with the administrative account or administration IAM Role. The execution role must have appropriate permissions to manage resources defined in the template along with those required for StackSets to operate. See the [AWS CloudFormation User Guide](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html) for more details.
 * > **NOTE:** To retain the Stack during resource destroy, ensure `retain_stack` has been set to `true` in the state first. This must be completed _before_ a deployment that would destroy the resource.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloudformation.StackSetInstance("example", {
 *     accountId: "123456789012",
 *     region: "us-east-1",
 *     stackSetName: exampleAwsCloudformationStackSet.name,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloudformation.StackSetInstance("example",
 *     account_id="123456789012",
 *     region="us-east-1",
 *     stack_set_name=example_aws_cloudformation_stack_set["name"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CloudFormation.StackSetInstance("example", new()
 *     {
 *         AccountId = "123456789012",
 *         Region = "us-east-1",
 *         StackSetName = exampleAwsCloudformationStackSet.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudformation"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudformation.NewStackSetInstance(ctx, "example", &cloudformation.StackSetInstanceArgs{
 * 			AccountId:    pulumi.String("123456789012"),
 * 			Region:       pulumi.String("us-east-1"),
 * 			StackSetName: pulumi.Any(exampleAwsCloudformationStackSet.Name),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudformation.StackSetInstance;
 * import com.pulumi.aws.cloudformation.StackSetInstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new StackSetInstance("example", StackSetInstanceArgs.builder()
 *             .accountId("123456789012")
 *             .region("us-east-1")
 *             .stackSetName(exampleAwsCloudformationStackSet.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloudformation:StackSetInstance
 *     properties:
 *       accountId: '123456789012'
 *       region: us-east-1
 *       stackSetName: ${exampleAwsCloudformationStackSet.name}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Example IAM Setup in Target Account
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy = aws.iam.getPolicyDocument({
 *     statements: [{
 *         actions: ["sts:AssumeRole"],
 *         effect: "Allow",
 *         principals: [{
 *             identifiers: [aWSCloudFormationStackSetAdministrationRole&#46;arn],
 *             type: "AWS",
 *         }],
 *     }],
 * });
 * const aWSCloudFormationStackSetExecutionRole = new aws.iam.Role("AWSCloudFormationStackSetExecutionRole", {
 *     assumeRolePolicy: aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy.then(aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy => aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy.json),
 *     name: "AWSCloudFormationStackSetExecutionRole",
 * });
 * // Documentation: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html
 * // Additional IAM permissions necessary depend on the resources defined in the StackSet template
 * const aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy = aws.iam.getPolicyDocument({
 *     statements: [{
 *         actions: [
 *             "cloudformation:*",
 *             "s3:*",
 *             "sns:*",
 *         ],
 *         effect: "Allow",
 *         resources: ["*"],
 *     }],
 * });
 * const aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicyRolePolicy = new aws.iam.RolePolicy("AWSCloudFormationStackSetExecutionRole_MinimumExecutionPolicy", {
 *     name: "MinimumExecutionPolicy",
 *     policy: aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy.then(aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy => aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy.json),
 *     role: aWSCloudFormationStackSetExecutionRole.name,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * a_ws_cloud_formation_stack_set_execution_role_assume_role_policy = aws.iam.get_policy_document(statements=[{
 *     "actions": ["sts:AssumeRole"],
 *     "effect": "Allow",
 *     "principals": [{
 *         "identifiers": [a_ws_cloud_formation_stack_set_administration_role["arn"]],
 *         "type": "AWS",
 *     }],
 * }])
 * a_ws_cloud_formation_stack_set_execution_role = aws.iam.Role("AWSCloudFormationStackSetExecutionRole",
 *     assume_role_policy=a_ws_cloud_formation_stack_set_execution_role_assume_role_policy.json,
 *     name="AWSCloudFormationStackSetExecutionRole")
 * # Documentation: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html
 * # Additional IAM permissions necessary depend on the resources defined in the StackSet template
 * a_ws_cloud_formation_stack_set_execution_role_minimum_execution_policy = aws.iam.get_policy_document(statements=[{
 *     "actions": [
 *         "cloudformation:*",
 *         "s3:*",
 *         "sns:*",
 *     ],
 *     "effect": "Allow",
 *     "resources": ["*"],
 * }])
 * a_ws_cloud_formation_stack_set_execution_role_minimum_execution_policy_role_policy = aws.iam.RolePolicy("AWSCloudFormationStackSetExecutionRole_MinimumExecutionPolicy",
 *     name="MinimumExecutionPolicy",
 *     policy=a_ws_cloud_formation_stack_set_execution_role_minimum_execution_policy.json,
 *     role=a_ws_cloud_formation_stack_set_execution_role.name)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Identifiers = new[]
 *                         {
 *                             aWSCloudFormationStackSetAdministrationRole.Arn,
 *                         },
 *                         Type = "AWS",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     var aWSCloudFormationStackSetExecutionRole = new Aws.Iam.Role("AWSCloudFormationStackSetExecutionRole", new()
 *     {
 *         AssumeRolePolicy = aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *         Name = "AWSCloudFormationStackSetExecutionRole",
 *     });
 *     // Documentation: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html
 *     // Additional IAM permissions necessary depend on the resources defined in the StackSet template
 *     var aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "cloudformation:*",
 *                     "s3:*",
 *                     "sns:*",
 *                 },
 *                 Effect = "Allow",
 *                 Resources = new[]
 *                 {
 *                     "*",
 *                 },
 *             },
 *         },
 *     });
 *     var aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicyRolePolicy = new Aws.Iam.RolePolicy("AWSCloudFormationStackSetExecutionRole_MinimumExecutionPolicy", new()
 *     {
 *         Name = "MinimumExecutionPolicy",
 *         Policy = aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *         Role = aWSCloudFormationStackSetExecutionRole.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * Statements: []iam.GetPolicyDocumentStatement{
 * {
 * Actions: []string{
 * "sts:AssumeRole",
 * },
 * Effect: pulumi.StringRef("Allow"),
 * Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * {
 * Identifiers: interface{}{
 * aWSCloudFormationStackSetAdministrationRole.Arn,
 * },
 * Type: "AWS",
 * },
 * },
 * },
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * aWSCloudFormationStackSetExecutionRole, err := iam.NewRole(ctx, "AWSCloudFormationStackSetExecutionRole", &iam.RoleArgs{
 * AssumeRolePolicy: pulumi.String(aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy.Json),
 * Name: pulumi.String("AWSCloudFormationStackSetExecutionRole"),
 * })
 * if err != nil {
 * return err
 * }
 * // Documentation: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html
 * // Additional IAM permissions necessary depend on the resources defined in the StackSet template
 * aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * Statements: []iam.GetPolicyDocumentStatement{
 * {
 * Actions: []string{
 * "cloudformation:*",
 * "s3:*",
 * "sns:*",
 * },
 * Effect: pulumi.StringRef("Allow"),
 * Resources: []string{
 * "*",
 * },
 * },
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * _, err = iam.NewRolePolicy(ctx, "AWSCloudFormationStackSetExecutionRole_MinimumExecutionPolicy", &iam.RolePolicyArgs{
 * Name: pulumi.String("MinimumExecutionPolicy"),
 * Policy: pulumi.String(aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy.Json),
 * Role: aWSCloudFormationStackSetExecutionRole.Name,
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.iam.RolePolicy;
 * import com.pulumi.aws.iam.RolePolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .actions("sts:AssumeRole")
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .identifiers(aWSCloudFormationStackSetAdministrationRole.arn())
 *                     .type("AWS")
 *                     .build())
 *                 .build())
 *             .build());
 *         var aWSCloudFormationStackSetExecutionRole = new Role("aWSCloudFormationStackSetExecutionRole", RoleArgs.builder()
 *             .assumeRolePolicy(aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .name("AWSCloudFormationStackSetExecutionRole")
 *             .build());
 *         // Documentation: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html
 *         // Additional IAM permissions necessary depend on the resources defined in the StackSet template
 *         final var aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .actions(
 *                     "cloudformation:*",
 *                     "s3:*",
 *                     "sns:*")
 *                 .effect("Allow")
 *                 .resources("*")
 *                 .build())
 *             .build());
 *         var aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicyRolePolicy = new RolePolicy("aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicyRolePolicy", RolePolicyArgs.builder()
 *             .name("MinimumExecutionPolicy")
 *             .policy(aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .role(aWSCloudFormationStackSetExecutionRole.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   aWSCloudFormationStackSetExecutionRole:
 *     type: aws:iam:Role
 *     name: AWSCloudFormationStackSetExecutionRole
 *     properties:
 *       assumeRolePolicy: ${aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy.json}
 *       name: AWSCloudFormationStackSetExecutionRole
 *   aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicyRolePolicy:
 *     type: aws:iam:RolePolicy
 *     name: AWSCloudFormationStackSetExecutionRole_MinimumExecutionPolicy
 *     properties:
 *       name: MinimumExecutionPolicy
 *       policy: ${aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy.json}
 *       role: ${aWSCloudFormationStackSetExecutionRole.name}
 * variables:
 *   aWSCloudFormationStackSetExecutionRoleAssumeRolePolicy:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - actions:
 *               - sts:AssumeRole
 *             effect: Allow
 *             principals:
 *               - identifiers:
 *                   - ${aWSCloudFormationStackSetAdministrationRole.arn}
 *                 type: AWS
 *   # Documentation: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html
 *   # Additional IAM permissions necessary depend on the resources defined in the StackSet template
 *   aWSCloudFormationStackSetExecutionRoleMinimumExecutionPolicy:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - actions:
 *               - cloudformation:*
 *               - s3:*
 *               - sns:*
 *             effect: Allow
 *             resources:
 *               - '*'
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Example Deployment across Organizations account
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloudformation.StackSetInstance("example", {
 *     deploymentTargets: {
 *         organizationalUnitIds: [exampleAwsOrganizationsOrganization&#46;roots[0]&#46;id],
 *     },
 *     region: "us-east-1",
 *     stackSetName: exampleAwsCloudformationStackSet.name,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloudformation.StackSetInstance("example",
 *     deployment_targets={
 *         "organizational_unit_ids": [example_aws_organizations_organization["roots"][0]["id"]],
 *     },
 *     region="us-east-1",
 *     stack_set_name=example_aws_cloudformation_stack_set["name"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CloudFormation.StackSetInstance("example", new()
 *     {
 *         DeploymentTargets = new Aws.CloudFormation.Inputs.StackSetInstanceDeploymentTargetsArgs
 *         {
 *             OrganizationalUnitIds = new[]
 *             {
 *                 exampleAwsOrganizationsOrganization.Roots[0].Id,
 *             },
 *         },
 *         Region = "us-east-1",
 *         StackSetName = exampleAwsCloudformationStackSet.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudformation"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudformation.NewStackSetInstance(ctx, "example", &cloudformation.StackSetInstanceArgs{
 * 			DeploymentTargets: &cloudformation.StackSetInstanceDeploymentTargetsArgs{
 * 				OrganizationalUnitIds: pulumi.StringArray{
 * 					exampleAwsOrganizationsOrganization.Roots[0].Id,
 * 				},
 * 			},
 * 			Region:       pulumi.String("us-east-1"),
 * 			StackSetName: pulumi.Any(exampleAwsCloudformationStackSet.Name),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudformation.StackSetInstance;
 * import com.pulumi.aws.cloudformation.StackSetInstanceArgs;
 * import com.pulumi.aws.cloudformation.inputs.StackSetInstanceDeploymentTargetsArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new StackSetInstance("example", StackSetInstanceArgs.builder()
 *             .deploymentTargets(StackSetInstanceDeploymentTargetsArgs.builder()
 *                 .organizationalUnitIds(exampleAwsOrganizationsOrganization.roots()[0].id())
 *                 .build())
 *             .region("us-east-1")
 *             .stackSetName(exampleAwsCloudformationStackSet.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloudformation:StackSetInstance
 *     properties:
 *       deploymentTargets:
 *         organizationalUnitIds:
 *           - ${exampleAwsOrganizationsOrganization.roots[0].id}
 *       region: us-east-1
 *       stackSetName: ${exampleAwsCloudformationStackSet.name}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Import CloudFormation StackSet Instances that target AWS Organizational Units using the StackSet name, a slash (`/`) separated list of organizational unit IDs, and target AWS Region separated by commas (`,`). For example:
 * Import CloudFormation StackSet Instances when acting a delegated administrator in a member account using the StackSet name, target AWS account ID or slash (`/`) separated list of organizational unit IDs, target AWS Region and `call_as` value separated by commas (`,`). For example:
 * Using `pulumi import`, import CloudFormation StackSet Instances that target an AWS Account ID using the StackSet name, target AWS account ID, and target AWS Region separated by commas (`,`). For example:
 * ```sh
 * $ pulumi import aws:cloudformation/stackSetInstance:StackSetInstance example example,123456789012,us-east-1
 * ```
 * Using `pulumi import`, import CloudFormation StackSet Instances that target AWS Organizational Units using the StackSet name, a slash (`/`) separated list of organizational unit IDs, and target AWS Region separated by commas (`,`). For example:
 * ```sh
 * $ pulumi import aws:cloudformation/stackSetInstance:StackSetInstance example example,ou-sdas-123123123/ou-sdas-789789789,us-east-1
 * ```
 * Using `pulumi import`, import CloudFormation StackSet Instances when acting a delegated administrator in a member account using the StackSet name, target AWS account ID or slash (`/`) separated list of organizational unit IDs, target AWS Region and `call_as` value separated by commas (`,`). For example:
 * ```sh
 * $ pulumi import aws:cloudformation/stackSetInstance:StackSetInstance example example,ou-sdas-123123123/ou-sdas-789789789,us-east-1,DELEGATED_ADMIN
 * ```
 * @property accountId Target AWS Account ID to create a Stack based on the StackSet. Defaults to current account.
 * @property callAs Specifies whether you are acting as an account administrator in the organization's management account or as a delegated administrator in a member account. Valid values: `SELF` (default), `DELEGATED_ADMIN`.
 * @property deploymentTargets AWS Organizations accounts to which StackSets deploys. StackSets doesn't deploy stack instances to the organization management account, even if the organization management account is in your organization or in an OU in your organization. Drift detection is not possible for this argument. See deployment_targets below.
 * @property operationPreferences Preferences for how AWS CloudFormation performs a stack set operation.
 * @property parameterOverrides Key-value map of input parameters to override from the StackSet for this Instance.
 * @property region Target AWS Region to create a Stack based on the StackSet. Defaults to current region.
 * @property retainStack During resource destroy, remove Instance from StackSet while keeping the Stack and its associated resources. Must be enabled in the state _before_ destroy operation to take effect. You cannot reassociate a retained Stack or add an existing, saved Stack to a new StackSet. Defaults to `false`.
 * @property stackSetName Name of the StackSet.
 */
public data class StackSetInstanceArgs(
    public val accountId: Output<String>? = null,
    public val callAs: Output<String>? = null,
    public val deploymentTargets: Output<StackSetInstanceDeploymentTargetsArgs>? = null,
    public val operationPreferences: Output<StackSetInstanceOperationPreferencesArgs>? = null,
    public val parameterOverrides: Output<Map<String, String>>? = null,
    public val region: Output<String>? = null,
    public val retainStack: Output<Boolean>? = null,
    public val stackSetName: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.cloudformation.StackSetInstanceArgs> {
    override fun toJava(): com.pulumi.aws.cloudformation.StackSetInstanceArgs =
        com.pulumi.aws.cloudformation.StackSetInstanceArgs.builder()
            .accountId(accountId?.applyValue({ args0 -> args0 }))
            .callAs(callAs?.applyValue({ args0 -> args0 }))
            .deploymentTargets(deploymentTargets?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .operationPreferences(
                operationPreferences?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .parameterOverrides(
                parameterOverrides?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .region(region?.applyValue({ args0 -> args0 }))
            .retainStack(retainStack?.applyValue({ args0 -> args0 }))
            .stackSetName(stackSetName?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [StackSetInstanceArgs].
 */
@PulumiTagMarker
public class StackSetInstanceArgsBuilder internal constructor() {
    private var accountId: Output<String>? = null

    private var callAs: Output<String>? = null

    private var deploymentTargets: Output<StackSetInstanceDeploymentTargetsArgs>? = null

    private var operationPreferences: Output<StackSetInstanceOperationPreferencesArgs>? = null

    private var parameterOverrides: Output<Map<String, String>>? = null

    private var region: Output<String>? = null

    private var retainStack: Output<Boolean>? = null

    private var stackSetName: Output<String>? = null

    /**
     * @param value Target AWS Account ID to create a Stack based on the StackSet. Defaults to current account.
     */
    @JvmName("oqnhsdbsxfshejrw")
    public suspend fun accountId(`value`: Output<String>) {
        this.accountId = value
    }

    /**
     * @param value Specifies whether you are acting as an account administrator in the organization's management account or as a delegated administrator in a member account. Valid values: `SELF` (default), `DELEGATED_ADMIN`.
     */
    @JvmName("tnkyqciewfngkjhn")
    public suspend fun callAs(`value`: Output<String>) {
        this.callAs = value
    }

    /**
     * @param value AWS Organizations accounts to which StackSets deploys. StackSets doesn't deploy stack instances to the organization management account, even if the organization management account is in your organization or in an OU in your organization. Drift detection is not possible for this argument. See deployment_targets below.
     */
    @JvmName("xatrfulxhuxmqbql")
    public suspend fun deploymentTargets(`value`: Output<StackSetInstanceDeploymentTargetsArgs>) {
        this.deploymentTargets = value
    }

    /**
     * @param value Preferences for how AWS CloudFormation performs a stack set operation.
     */
    @JvmName("mglapfjicqglkiid")
    public suspend fun operationPreferences(`value`: Output<StackSetInstanceOperationPreferencesArgs>) {
        this.operationPreferences = value
    }

    /**
     * @param value Key-value map of input parameters to override from the StackSet for this Instance.
     */
    @JvmName("qwbiolcycqmldatx")
    public suspend fun parameterOverrides(`value`: Output<Map<String, String>>) {
        this.parameterOverrides = value
    }

    /**
     * @param value Target AWS Region to create a Stack based on the StackSet. Defaults to current region.
     */
    @JvmName("swqkosmdvtemmfoa")
    public suspend fun region(`value`: Output<String>) {
        this.region = value
    }

    /**
     * @param value During resource destroy, remove Instance from StackSet while keeping the Stack and its associated resources. Must be enabled in the state _before_ destroy operation to take effect. You cannot reassociate a retained Stack or add an existing, saved Stack to a new StackSet. Defaults to `false`.
     */
    @JvmName("hiiuubhvkgcvffdf")
    public suspend fun retainStack(`value`: Output<Boolean>) {
        this.retainStack = value
    }

    /**
     * @param value Name of the StackSet.
     */
    @JvmName("gbxyxhymmjtsvcyj")
    public suspend fun stackSetName(`value`: Output<String>) {
        this.stackSetName = value
    }

    /**
     * @param value Target AWS Account ID to create a Stack based on the StackSet. Defaults to current account.
     */
    @JvmName("gdaaflqeuphoydto")
    public suspend fun accountId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accountId = mapped
    }

    /**
     * @param value Specifies whether you are acting as an account administrator in the organization's management account or as a delegated administrator in a member account. Valid values: `SELF` (default), `DELEGATED_ADMIN`.
     */
    @JvmName("nuncnmsumbkgnhnd")
    public suspend fun callAs(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.callAs = mapped
    }

    /**
     * @param value AWS Organizations accounts to which StackSets deploys. StackSets doesn't deploy stack instances to the organization management account, even if the organization management account is in your organization or in an OU in your organization. Drift detection is not possible for this argument. See deployment_targets below.
     */
    @JvmName("aebgibgahqmxjukc")
    public suspend fun deploymentTargets(`value`: StackSetInstanceDeploymentTargetsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deploymentTargets = mapped
    }

    /**
     * @param argument AWS Organizations accounts to which StackSets deploys. StackSets doesn't deploy stack instances to the organization management account, even if the organization management account is in your organization or in an OU in your organization. Drift detection is not possible for this argument. See deployment_targets below.
     */
    @JvmName("afajlxabkrfjasxc")
    public suspend fun deploymentTargets(argument: suspend StackSetInstanceDeploymentTargetsArgsBuilder.() -> Unit) {
        val toBeMapped = StackSetInstanceDeploymentTargetsArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.deploymentTargets = mapped
    }

    /**
     * @param value Preferences for how AWS CloudFormation performs a stack set operation.
     */
    @JvmName("exxbwmdliwihfmcu")
    public suspend fun operationPreferences(`value`: StackSetInstanceOperationPreferencesArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.operationPreferences = mapped
    }

    /**
     * @param argument Preferences for how AWS CloudFormation performs a stack set operation.
     */
    @JvmName("ltrhrywhkivkbwak")
    public suspend fun operationPreferences(argument: suspend StackSetInstanceOperationPreferencesArgsBuilder.() -> Unit) {
        val toBeMapped = StackSetInstanceOperationPreferencesArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.operationPreferences = mapped
    }

    /**
     * @param value Key-value map of input parameters to override from the StackSet for this Instance.
     */
    @JvmName("pfuwohyktyxeponj")
    public suspend fun parameterOverrides(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parameterOverrides = mapped
    }

    /**
     * @param values Key-value map of input parameters to override from the StackSet for this Instance.
     */
    @JvmName("hmpjyrxbejoprdci")
    public fun parameterOverrides(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.parameterOverrides = mapped
    }

    /**
     * @param value Target AWS Region to create a Stack based on the StackSet. Defaults to current region.
     */
    @JvmName("gbqsvhvatmcluwnu")
    public suspend fun region(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.region = mapped
    }

    /**
     * @param value During resource destroy, remove Instance from StackSet while keeping the Stack and its associated resources. Must be enabled in the state _before_ destroy operation to take effect. You cannot reassociate a retained Stack or add an existing, saved Stack to a new StackSet. Defaults to `false`.
     */
    @JvmName("tqvlttljeblemglu")
    public suspend fun retainStack(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.retainStack = mapped
    }

    /**
     * @param value Name of the StackSet.
     */
    @JvmName("sfdqtpdcmdphuboe")
    public suspend fun stackSetName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.stackSetName = mapped
    }

    internal fun build(): StackSetInstanceArgs = StackSetInstanceArgs(
        accountId = accountId,
        callAs = callAs,
        deploymentTargets = deploymentTargets,
        operationPreferences = operationPreferences,
        parameterOverrides = parameterOverrides,
        region = region,
        retainStack = retainStack,
        stackSetName = stackSetName,
    )
}
