@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cloudfront.kotlin

import com.pulumi.aws.cloudfront.DistributionArgs.builder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionCustomErrorResponseArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionCustomErrorResponseArgsBuilder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionDefaultCacheBehaviorArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionDefaultCacheBehaviorArgsBuilder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionLoggingConfigArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionLoggingConfigArgsBuilder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionOrderedCacheBehaviorArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionOrderedCacheBehaviorArgsBuilder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionOriginArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionOriginArgsBuilder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionOriginGroupArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionOriginGroupArgsBuilder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionRestrictionsArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionRestrictionsArgsBuilder
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionViewerCertificateArgs
import com.pulumi.aws.cloudfront.kotlin.inputs.DistributionViewerCertificateArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Creates an Amazon CloudFront web distribution.
 * For information about CloudFront distributions, see the [Amazon CloudFront Developer Guide](http://docs&#46;aws&#46;amazon&#46;com/AmazonCloudFront/latest/DeveloperGuide/Introduction&#46;html)&#46; For specific information about creating CloudFront web distributions, see the [POST Distribution](https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html) page in the Amazon CloudFront API Reference.
 * > **NOTE:** CloudFront distributions take about 15 minutes to reach a deployed state after creation or modification. During this time, deletes to resources will be blocked. If you need to delete a distribution that is enabled and you do not want to wait, you need to use the `retain_on_delete` flag.
 * ## Example Usage
 * ### S3 Origin
 * The example below creates a CloudFront distribution with an S3 origin.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const b = new aws.s3.BucketV2("b", {
 *     bucket: "mybucket",
 *     tags: {
 *         Name: "My bucket",
 *     },
 * });
 * const bAcl = new aws.s3.BucketAclV2("b_acl", {
 *     bucket: b.id,
 *     acl: "private",
 * });
 * const s3OriginId = "myS3Origin";
 * const s3Distribution = new aws.cloudfront.Distribution("s3_distribution", {
 *     origins: [{
 *         domainName: b.bucketRegionalDomainName,
 *         originAccessControlId: _default.id,
 *         originId: s3OriginId,
 *     }],
 *     enabled: true,
 *     isIpv6Enabled: true,
 *     comment: "Some comment",
 *     defaultRootObject: "index.html",
 *     loggingConfig: {
 *         includeCookies: false,
 *         bucket: "mylogs.s3.amazonaws.com",
 *         prefix: "myprefix",
 *     },
 *     aliases: [
 *         "mysite.example.com",
 *         "yoursite.example.com",
 *     ],
 *     defaultCacheBehavior: {
 *         allowedMethods: [
 *             "DELETE",
 *             "GET",
 *             "HEAD",
 *             "OPTIONS",
 *             "PATCH",
 *             "POST",
 *             "PUT",
 *         ],
 *         cachedMethods: [
 *             "GET",
 *             "HEAD",
 *         ],
 *         targetOriginId: s3OriginId,
 *         forwardedValues: {
 *             queryString: false,
 *             cookies: {
 *                 forward: "none",
 *             },
 *         },
 *         viewerProtocolPolicy: "allow-all",
 *         minTtl: 0,
 *         defaultTtl: 3600,
 *         maxTtl: 86400,
 *     },
 *     orderedCacheBehaviors: [
 *         {
 *             pathPattern: "/content/immutable/*",
 *             allowedMethods: [
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *             ],
 *             cachedMethods: [
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *             ],
 *             targetOriginId: s3OriginId,
 *             forwardedValues: {
 *                 queryString: false,
 *                 headers: ["Origin"],
 *                 cookies: {
 *                     forward: "none",
 *                 },
 *             },
 *             minTtl: 0,
 *             defaultTtl: 86400,
 *             maxTtl: 31536000,
 *             compress: true,
 *             viewerProtocolPolicy: "redirect-to-https",
 *         },
 *         {
 *             pathPattern: "/content/*",
 *             allowedMethods: [
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *             ],
 *             cachedMethods: [
 *                 "GET",
 *                 "HEAD",
 *             ],
 *             targetOriginId: s3OriginId,
 *             forwardedValues: {
 *                 queryString: false,
 *                 cookies: {
 *                     forward: "none",
 *                 },
 *             },
 *             minTtl: 0,
 *             defaultTtl: 3600,
 *             maxTtl: 86400,
 *             compress: true,
 *             viewerProtocolPolicy: "redirect-to-https",
 *         },
 *     ],
 *     priceClass: "PriceClass_200",
 *     restrictions: {
 *         geoRestriction: {
 *             restrictionType: "whitelist",
 *             locations: [
 *                 "US",
 *                 "CA",
 *                 "GB",
 *                 "DE",
 *             ],
 *         },
 *     },
 *     tags: {
 *         Environment: "production",
 *     },
 *     viewerCertificate: {
 *         cloudfrontDefaultCertificate: true,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * b = aws.s3.BucketV2("b",
 *     bucket="mybucket",
 *     tags={
 *         "Name": "My bucket",
 *     })
 * b_acl = aws.s3.BucketAclV2("b_acl",
 *     bucket=b.id,
 *     acl="private")
 * s3_origin_id = "myS3Origin"
 * s3_distribution = aws.cloudfront.Distribution("s3_distribution",
 *     origins=[{
 *         "domain_name": b.bucket_regional_domain_name,
 *         "origin_access_control_id": default["id"],
 *         "origin_id": s3_origin_id,
 *     }],
 *     enabled=True,
 *     is_ipv6_enabled=True,
 *     comment="Some comment",
 *     default_root_object="index.html",
 *     logging_config={
 *         "include_cookies": False,
 *         "bucket": "mylogs.s3.amazonaws.com",
 *         "prefix": "myprefix",
 *     },
 *     aliases=[
 *         "mysite.example.com",
 *         "yoursite.example.com",
 *     ],
 *     default_cache_behavior={
 *         "allowed_methods": [
 *             "DELETE",
 *             "GET",
 *             "HEAD",
 *             "OPTIONS",
 *             "PATCH",
 *             "POST",
 *             "PUT",
 *         ],
 *         "cached_methods": [
 *             "GET",
 *             "HEAD",
 *         ],
 *         "target_origin_id": s3_origin_id,
 *         "forwarded_values": {
 *             "query_string": False,
 *             "cookies": {
 *                 "forward": "none",
 *             },
 *         },
 *         "viewer_protocol_policy": "allow-all",
 *         "min_ttl": 0,
 *         "default_ttl": 3600,
 *         "max_ttl": 86400,
 *     },
 *     ordered_cache_behaviors=[
 *         {
 *             "path_pattern": "/content/immutable/*",
 *             "allowed_methods": [
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *             ],
 *             "cached_methods": [
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *             ],
 *             "target_origin_id": s3_origin_id,
 *             "forwarded_values": {
 *                 "query_string": False,
 *                 "headers": ["Origin"],
 *                 "cookies": {
 *                     "forward": "none",
 *                 },
 *             },
 *             "min_ttl": 0,
 *             "default_ttl": 86400,
 *             "max_ttl": 31536000,
 *             "compress": True,
 *             "viewer_protocol_policy": "redirect-to-https",
 *         },
 *         {
 *             "path_pattern": "/content/*",
 *             "allowed_methods": [
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *             ],
 *             "cached_methods": [
 *                 "GET",
 *                 "HEAD",
 *             ],
 *             "target_origin_id": s3_origin_id,
 *             "forwarded_values": {
 *                 "query_string": False,
 *                 "cookies": {
 *                     "forward": "none",
 *                 },
 *             },
 *             "min_ttl": 0,
 *             "default_ttl": 3600,
 *             "max_ttl": 86400,
 *             "compress": True,
 *             "viewer_protocol_policy": "redirect-to-https",
 *         },
 *     ],
 *     price_class="PriceClass_200",
 *     restrictions={
 *         "geo_restriction": {
 *             "restriction_type": "whitelist",
 *             "locations": [
 *                 "US",
 *                 "CA",
 *                 "GB",
 *                 "DE",
 *             ],
 *         },
 *     },
 *     tags={
 *         "Environment": "production",
 *     },
 *     viewer_certificate={
 *         "cloudfront_default_certificate": True,
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var b = new Aws.S3.BucketV2("b", new()
 *     {
 *         Bucket = "mybucket",
 *         Tags =
 *         {
 *             { "Name", "My bucket" },
 *         },
 *     });
 *     var bAcl = new Aws.S3.BucketAclV2("b_acl", new()
 *     {
 *         Bucket = b.Id,
 *         Acl = "private",
 *     });
 *     var s3OriginId = "myS3Origin";
 *     var s3Distribution = new Aws.CloudFront.Distribution("s3_distribution", new()
 *     {
 *         Origins = new[]
 *         {
 *             new Aws.CloudFront.Inputs.DistributionOriginArgs
 *             {
 *                 DomainName = b.BucketRegionalDomainName,
 *                 OriginAccessControlId = @default.Id,
 *                 OriginId = s3OriginId,
 *             },
 *         },
 *         Enabled = true,
 *         IsIpv6Enabled = true,
 *         Comment = "Some comment",
 *         DefaultRootObject = "index.html",
 *         LoggingConfig = new Aws.CloudFront.Inputs.DistributionLoggingConfigArgs
 *         {
 *             IncludeCookies = false,
 *             Bucket = "mylogs.s3.amazonaws.com",
 *             Prefix = "myprefix",
 *         },
 *         Aliases = new[]
 *         {
 *             "mysite.example.com",
 *             "yoursite.example.com",
 *         },
 *         DefaultCacheBehavior = new Aws.CloudFront.Inputs.DistributionDefaultCacheBehaviorArgs
 *         {
 *             AllowedMethods = new[]
 *             {
 *                 "DELETE",
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *                 "PATCH",
 *                 "POST",
 *                 "PUT",
 *             },
 *             CachedMethods = new[]
 *             {
 *                 "GET",
 *                 "HEAD",
 *             },
 *             TargetOriginId = s3OriginId,
 *             ForwardedValues = new Aws.CloudFront.Inputs.DistributionDefaultCacheBehaviorForwardedValuesArgs
 *             {
 *                 QueryString = false,
 *                 Cookies = new Aws.CloudFront.Inputs.DistributionDefaultCacheBehaviorForwardedValuesCookiesArgs
 *                 {
 *                     Forward = "none",
 *                 },
 *             },
 *             ViewerProtocolPolicy = "allow-all",
 *             MinTtl = 0,
 *             DefaultTtl = 3600,
 *             MaxTtl = 86400,
 *         },
 *         OrderedCacheBehaviors = new[]
 *         {
 *             new Aws.CloudFront.Inputs.DistributionOrderedCacheBehaviorArgs
 *             {
 *                 PathPattern = "/content/immutable/*",
 *                 AllowedMethods = new[]
 *                 {
 *                     "GET",
 *                     "HEAD",
 *                     "OPTIONS",
 *                 },
 *                 CachedMethods = new[]
 *                 {
 *                     "GET",
 *                     "HEAD",
 *                     "OPTIONS",
 *                 },
 *                 TargetOriginId = s3OriginId,
 *                 ForwardedValues = new Aws.CloudFront.Inputs.DistributionOrderedCacheBehaviorForwardedValuesArgs
 *                 {
 *                     QueryString = false,
 *                     Headers = new[]
 *                     {
 *                         "Origin",
 *                     },
 *                     Cookies = new Aws.CloudFront.Inputs.DistributionOrderedCacheBehaviorForwardedValuesCookiesArgs
 *                     {
 *                         Forward = "none",
 *                     },
 *                 },
 *                 MinTtl = 0,
 *                 DefaultTtl = 86400,
 *                 MaxTtl = 31536000,
 *                 Compress = true,
 *                 ViewerProtocolPolicy = "redirect-to-https",
 *             },
 *             new Aws.CloudFront.Inputs.DistributionOrderedCacheBehaviorArgs
 *             {
 *                 PathPattern = "/content/*",
 *                 AllowedMethods = new[]
 *                 {
 *                     "GET",
 *                     "HEAD",
 *                     "OPTIONS",
 *                 },
 *                 CachedMethods = new[]
 *                 {
 *                     "GET",
 *                     "HEAD",
 *                 },
 *                 TargetOriginId = s3OriginId,
 *                 ForwardedValues = new Aws.CloudFront.Inputs.DistributionOrderedCacheBehaviorForwardedValuesArgs
 *                 {
 *                     QueryString = false,
 *                     Cookies = new Aws.CloudFront.Inputs.DistributionOrderedCacheBehaviorForwardedValuesCookiesArgs
 *                     {
 *                         Forward = "none",
 *                     },
 *                 },
 *                 MinTtl = 0,
 *                 DefaultTtl = 3600,
 *                 MaxTtl = 86400,
 *                 Compress = true,
 *                 ViewerProtocolPolicy = "redirect-to-https",
 *             },
 *         },
 *         PriceClass = "PriceClass_200",
 *         Restrictions = new Aws.CloudFront.Inputs.DistributionRestrictionsArgs
 *         {
 *             GeoRestriction = new Aws.CloudFront.Inputs.DistributionRestrictionsGeoRestrictionArgs
 *             {
 *                 RestrictionType = "whitelist",
 *                 Locations = new[]
 *                 {
 *                     "US",
 *                     "CA",
 *                     "GB",
 *                     "DE",
 *                 },
 *             },
 *         },
 *         Tags =
 *         {
 *             { "Environment", "production" },
 *         },
 *         ViewerCertificate = new Aws.CloudFront.Inputs.DistributionViewerCertificateArgs
 *         {
 *             CloudfrontDefaultCertificate = true,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudfront"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		b, err := s3.NewBucketV2(ctx, "b", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("mybucket"),
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("My bucket"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketAclV2(ctx, "b_acl", &s3.BucketAclV2Args{
 * 			Bucket: b.ID(),
 * 			Acl:    pulumi.String("private"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		s3OriginId := "myS3Origin"
 * 		_, err = cloudfront.NewDistribution(ctx, "s3_distribution", &cloudfront.DistributionArgs{
 * 			Origins: cloudfront.DistributionOriginArray{
 * 				&cloudfront.DistributionOriginArgs{
 * 					DomainName:            b.BucketRegionalDomainName,
 * 					OriginAccessControlId: pulumi.Any(_default.Id),
 * 					OriginId:              pulumi.String(s3OriginId),
 * 				},
 * 			},
 * 			Enabled:           pulumi.Bool(true),
 * 			IsIpv6Enabled:     pulumi.Bool(true),
 * 			Comment:           pulumi.String("Some comment"),
 * 			DefaultRootObject: pulumi.String("index.html"),
 * 			LoggingConfig: &cloudfront.DistributionLoggingConfigArgs{
 * 				IncludeCookies: pulumi.Bool(false),
 * 				Bucket:         pulumi.String("mylogs.s3.amazonaws.com"),
 * 				Prefix:         pulumi.String("myprefix"),
 * 			},
 * 			Aliases: pulumi.StringArray{
 * 				pulumi.String("mysite.example.com"),
 * 				pulumi.String("yoursite.example.com"),
 * 			},
 * 			DefaultCacheBehavior: &cloudfront.DistributionDefaultCacheBehaviorArgs{
 * 				AllowedMethods: pulumi.StringArray{
 * 					pulumi.String("DELETE"),
 * 					pulumi.String("GET"),
 * 					pulumi.String("HEAD"),
 * 					pulumi.String("OPTIONS"),
 * 					pulumi.String("PATCH"),
 * 					pulumi.String("POST"),
 * 					pulumi.String("PUT"),
 * 				},
 * 				CachedMethods: pulumi.StringArray{
 * 					pulumi.String("GET"),
 * 					pulumi.String("HEAD"),
 * 				},
 * 				TargetOriginId: pulumi.String(s3OriginId),
 * 				ForwardedValues: &cloudfront.DistributionDefaultCacheBehaviorForwardedValuesArgs{
 * 					QueryString: pulumi.Bool(false),
 * 					Cookies: &cloudfront.DistributionDefaultCacheBehaviorForwardedValuesCookiesArgs{
 * 						Forward: pulumi.String("none"),
 * 					},
 * 				},
 * 				ViewerProtocolPolicy: pulumi.String("allow-all"),
 * 				MinTtl:               pulumi.Int(0),
 * 				DefaultTtl:           pulumi.Int(3600),
 * 				MaxTtl:               pulumi.Int(86400),
 * 			},
 * 			OrderedCacheBehaviors: cloudfront.DistributionOrderedCacheBehaviorArray{
 * 				&cloudfront.DistributionOrderedCacheBehaviorArgs{
 * 					PathPattern: pulumi.String("/content/immutable/*"),
 * 					AllowedMethods: pulumi.StringArray{
 * 						pulumi.String("GET"),
 * 						pulumi.String("HEAD"),
 * 						pulumi.String("OPTIONS"),
 * 					},
 * 					CachedMethods: pulumi.StringArray{
 * 						pulumi.String("GET"),
 * 						pulumi.String("HEAD"),
 * 						pulumi.String("OPTIONS"),
 * 					},
 * 					TargetOriginId: pulumi.String(s3OriginId),
 * 					ForwardedValues: &cloudfront.DistributionOrderedCacheBehaviorForwardedValuesArgs{
 * 						QueryString: pulumi.Bool(false),
 * 						Headers: pulumi.StringArray{
 * 							pulumi.String("Origin"),
 * 						},
 * 						Cookies: &cloudfront.DistributionOrderedCacheBehaviorForwardedValuesCookiesArgs{
 * 							Forward: pulumi.String("none"),
 * 						},
 * 					},
 * 					MinTtl:               pulumi.Int(0),
 * 					DefaultTtl:           pulumi.Int(86400),
 * 					MaxTtl:               pulumi.Int(31536000),
 * 					Compress:             pulumi.Bool(true),
 * 					ViewerProtocolPolicy: pulumi.String("redirect-to-https"),
 * 				},
 * 				&cloudfront.DistributionOrderedCacheBehaviorArgs{
 * 					PathPattern: pulumi.String("/content/*"),
 * 					AllowedMethods: pulumi.StringArray{
 * 						pulumi.String("GET"),
 * 						pulumi.String("HEAD"),
 * 						pulumi.String("OPTIONS"),
 * 					},
 * 					CachedMethods: pulumi.StringArray{
 * 						pulumi.String("GET"),
 * 						pulumi.String("HEAD"),
 * 					},
 * 					TargetOriginId: pulumi.String(s3OriginId),
 * 					ForwardedValues: &cloudfront.DistributionOrderedCacheBehaviorForwardedValuesArgs{
 * 						QueryString: pulumi.Bool(false),
 * 						Cookies: &cloudfront.DistributionOrderedCacheBehaviorForwardedValuesCookiesArgs{
 * 							Forward: pulumi.String("none"),
 * 						},
 * 					},
 * 					MinTtl:               pulumi.Int(0),
 * 					DefaultTtl:           pulumi.Int(3600),
 * 					MaxTtl:               pulumi.Int(86400),
 * 					Compress:             pulumi.Bool(true),
 * 					ViewerProtocolPolicy: pulumi.String("redirect-to-https"),
 * 				},
 * 			},
 * 			PriceClass: pulumi.String("PriceClass_200"),
 * 			Restrictions: &cloudfront.DistributionRestrictionsArgs{
 * 				GeoRestriction: &cloudfront.DistributionRestrictionsGeoRestrictionArgs{
 * 					RestrictionType: pulumi.String("whitelist"),
 * 					Locations: pulumi.StringArray{
 * 						pulumi.String("US"),
 * 						pulumi.String("CA"),
 * 						pulumi.String("GB"),
 * 						pulumi.String("DE"),
 * 					},
 * 				},
 * 			},
 * 			Tags: pulumi.StringMap{
 * 				"Environment": pulumi.String("production"),
 * 			},
 * 			ViewerCertificate: &cloudfront.DistributionViewerCertificateArgs{
 * 				CloudfrontDefaultCertificate: pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.s3.BucketAclV2;
 * import com.pulumi.aws.s3.BucketAclV2Args;
 * import com.pulumi.aws.cloudfront.Distribution;
 * import com.pulumi.aws.cloudfront.DistributionArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionLoggingConfigArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionDefaultCacheBehaviorArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionDefaultCacheBehaviorForwardedValuesArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionDefaultCacheBehaviorForwardedValuesCookiesArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOrderedCacheBehaviorArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOrderedCacheBehaviorForwardedValuesArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOrderedCacheBehaviorForwardedValuesCookiesArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionRestrictionsArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionRestrictionsGeoRestrictionArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionViewerCertificateArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var b = new BucketV2("b", BucketV2Args.builder()
 *             .bucket("mybucket")
 *             .tags(Map.of("Name", "My bucket"))
 *             .build());
 *         var bAcl = new BucketAclV2("bAcl", BucketAclV2Args.builder()
 *             .bucket(b.id())
 *             .acl("private")
 *             .build());
 *         final var s3OriginId = "myS3Origin";
 *         var s3Distribution = new Distribution("s3Distribution", DistributionArgs.builder()
 *             .origins(DistributionOriginArgs.builder()
 *                 .domainName(b.bucketRegionalDomainName())
 *                 .originAccessControlId(default_.id())
 *                 .originId(s3OriginId)
 *                 .build())
 *             .enabled(true)
 *             .isIpv6Enabled(true)
 *             .comment("Some comment")
 *             .defaultRootObject("index.html")
 *             .loggingConfig(DistributionLoggingConfigArgs.builder()
 *                 .includeCookies(false)
 *                 .bucket("mylogs.s3.amazonaws.com")
 *                 .prefix("myprefix")
 *                 .build())
 *             .aliases(
 *                 "mysite.example.com",
 *                 "yoursite.example.com")
 *             .defaultCacheBehavior(DistributionDefaultCacheBehaviorArgs.builder()
 *                 .allowedMethods(
 *                     "DELETE",
 *                     "GET",
 *                     "HEAD",
 *                     "OPTIONS",
 *                     "PATCH",
 *                     "POST",
 *                     "PUT")
 *                 .cachedMethods(
 *                     "GET",
 *                     "HEAD")
 *                 .targetOriginId(s3OriginId)
 *                 .forwardedValues(DistributionDefaultCacheBehaviorForwardedValuesArgs.builder()
 *                     .queryString(false)
 *                     .cookies(DistributionDefaultCacheBehaviorForwardedValuesCookiesArgs.builder()
 *                         .forward("none")
 *                         .build())
 *                     .build())
 *                 .viewerProtocolPolicy("allow-all")
 *                 .minTtl(0)
 *                 .defaultTtl(3600)
 *                 .maxTtl(86400)
 *                 .build())
 *             .orderedCacheBehaviors(
 *                 DistributionOrderedCacheBehaviorArgs.builder()
 *                     .pathPattern("/content/immutable/*")
 *                     .allowedMethods(
 *                         "GET",
 *                         "HEAD",
 *                         "OPTIONS")
 *                     .cachedMethods(
 *                         "GET",
 *                         "HEAD",
 *                         "OPTIONS")
 *                     .targetOriginId(s3OriginId)
 *                     .forwardedValues(DistributionOrderedCacheBehaviorForwardedValuesArgs.builder()
 *                         .queryString(false)
 *                         .headers("Origin")
 *                         .cookies(DistributionOrderedCacheBehaviorForwardedValuesCookiesArgs.builder()
 *                             .forward("none")
 *                             .build())
 *                         .build())
 *                     .minTtl(0)
 *                     .defaultTtl(86400)
 *                     .maxTtl(31536000)
 *                     .compress(true)
 *                     .viewerProtocolPolicy("redirect-to-https")
 *                     .build(),
 *                 DistributionOrderedCacheBehaviorArgs.builder()
 *                     .pathPattern("/content/*")
 *                     .allowedMethods(
 *                         "GET",
 *                         "HEAD",
 *                         "OPTIONS")
 *                     .cachedMethods(
 *                         "GET",
 *                         "HEAD")
 *                     .targetOriginId(s3OriginId)
 *                     .forwardedValues(DistributionOrderedCacheBehaviorForwardedValuesArgs.builder()
 *                         .queryString(false)
 *                         .cookies(DistributionOrderedCacheBehaviorForwardedValuesCookiesArgs.builder()
 *                             .forward("none")
 *                             .build())
 *                         .build())
 *                     .minTtl(0)
 *                     .defaultTtl(3600)
 *                     .maxTtl(86400)
 *                     .compress(true)
 *                     .viewerProtocolPolicy("redirect-to-https")
 *                     .build())
 *             .priceClass("PriceClass_200")
 *             .restrictions(DistributionRestrictionsArgs.builder()
 *                 .geoRestriction(DistributionRestrictionsGeoRestrictionArgs.builder()
 *                     .restrictionType("whitelist")
 *                     .locations(
 *                         "US",
 *                         "CA",
 *                         "GB",
 *                         "DE")
 *                     .build())
 *                 .build())
 *             .tags(Map.of("Environment", "production"))
 *             .viewerCertificate(DistributionViewerCertificateArgs.builder()
 *                 .cloudfrontDefaultCertificate(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   b:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: mybucket
 *       tags:
 *         Name: My bucket
 *   bAcl:
 *     type: aws:s3:BucketAclV2
 *     name: b_acl
 *     properties:
 *       bucket: ${b.id}
 *       acl: private
 *   s3Distribution:
 *     type: aws:cloudfront:Distribution
 *     name: s3_distribution
 *     properties:
 *       origins:
 *         - domainName: ${b.bucketRegionalDomainName}
 *           originAccessControlId: ${default.id}
 *           originId: ${s3OriginId}
 *       enabled: true
 *       isIpv6Enabled: true
 *       comment: Some comment
 *       defaultRootObject: index.html
 *       loggingConfig:
 *         includeCookies: false
 *         bucket: mylogs.s3.amazonaws.com
 *         prefix: myprefix
 *       aliases:
 *         - mysite.example.com
 *         - yoursite.example.com
 *       defaultCacheBehavior:
 *         allowedMethods:
 *           - DELETE
 *           - GET
 *           - HEAD
 *           - OPTIONS
 *           - PATCH
 *           - POST
 *           - PUT
 *         cachedMethods:
 *           - GET
 *           - HEAD
 *         targetOriginId: ${s3OriginId}
 *         forwardedValues:
 *           queryString: false
 *           cookies:
 *             forward: none
 *         viewerProtocolPolicy: allow-all
 *         minTtl: 0
 *         defaultTtl: 3600
 *         maxTtl: 86400
 *       orderedCacheBehaviors:
 *         - pathPattern: /content/immutable/*
 *           allowedMethods:
 *             - GET
 *             - HEAD
 *             - OPTIONS
 *           cachedMethods:
 *             - GET
 *             - HEAD
 *             - OPTIONS
 *           targetOriginId: ${s3OriginId}
 *           forwardedValues:
 *             queryString: false
 *             headers:
 *               - Origin
 *             cookies:
 *               forward: none
 *           minTtl: 0
 *           defaultTtl: 86400
 *           maxTtl: 3.1536e+07
 *           compress: true
 *           viewerProtocolPolicy: redirect-to-https
 *         - pathPattern: /content/*
 *           allowedMethods:
 *             - GET
 *             - HEAD
 *             - OPTIONS
 *           cachedMethods:
 *             - GET
 *             - HEAD
 *           targetOriginId: ${s3OriginId}
 *           forwardedValues:
 *             queryString: false
 *             cookies:
 *               forward: none
 *           minTtl: 0
 *           defaultTtl: 3600
 *           maxTtl: 86400
 *           compress: true
 *           viewerProtocolPolicy: redirect-to-https
 *       priceClass: PriceClass_200
 *       restrictions:
 *         geoRestriction:
 *           restrictionType: whitelist
 *           locations:
 *             - US
 *             - CA
 *             - GB
 *             - DE
 *       tags:
 *         Environment: production
 *       viewerCertificate:
 *         cloudfrontDefaultCertificate: true
 * variables:
 *   s3OriginId: myS3Origin
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Failover Routing
 * The example below creates a CloudFront distribution with an origin group for failover routing.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const s3Distribution = new aws.cloudfront.Distribution("s3_distribution", {
 *     originGroups: [{
 *         originId: "groupS3",
 *         failoverCriteria: {
 *             statusCodes: [
 *                 403,
 *                 404,
 *                 500,
 *                 502,
 *             ],
 *         },
 *         members: [
 *             {
 *                 originId: "primaryS3",
 *             },
 *             {
 *                 originId: "failoverS3",
 *             },
 *         ],
 *     }],
 *     origins: [
 *         {
 *             domainName: primary.bucketRegionalDomainName,
 *             originId: "primaryS3",
 *             s3OriginConfig: {
 *                 originAccessIdentity: _default.cloudfrontAccessIdentityPath,
 *             },
 *         },
 *         {
 *             domainName: failover.bucketRegionalDomainName,
 *             originId: "failoverS3",
 *             s3OriginConfig: {
 *                 originAccessIdentity: _default.cloudfrontAccessIdentityPath,
 *             },
 *         },
 *     ],
 *     defaultCacheBehavior: {
 *         targetOriginId: "groupS3",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * s3_distribution = aws.cloudfront.Distribution("s3_distribution",
 *     origin_groups=[{
 *         "origin_id": "groupS3",
 *         "failover_criteria": {
 *             "status_codes": [
 *                 403,
 *                 404,
 *                 500,
 *                 502,
 *             ],
 *         },
 *         "members": [
 *             {
 *                 "origin_id": "primaryS3",
 *             },
 *             {
 *                 "origin_id": "failoverS3",
 *             },
 *         ],
 *     }],
 *     origins=[
 *         {
 *             "domain_name": primary["bucketRegionalDomainName"],
 *             "origin_id": "primaryS3",
 *             "s3_origin_config": {
 *                 "origin_access_identity": default["cloudfrontAccessIdentityPath"],
 *             },
 *         },
 *         {
 *             "domain_name": failover["bucketRegionalDomainName"],
 *             "origin_id": "failoverS3",
 *             "s3_origin_config": {
 *                 "origin_access_identity": default["cloudfrontAccessIdentityPath"],
 *             },
 *         },
 *     ],
 *     default_cache_behavior={
 *         "target_origin_id": "groupS3",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var s3Distribution = new Aws.CloudFront.Distribution("s3_distribution", new()
 *     {
 *         OriginGroups = new[]
 *         {
 *             new Aws.CloudFront.Inputs.DistributionOriginGroupArgs
 *             {
 *                 OriginId = "groupS3",
 *                 FailoverCriteria = new Aws.CloudFront.Inputs.DistributionOriginGroupFailoverCriteriaArgs
 *                 {
 *                     StatusCodes = new[]
 *                     {
 *                         403,
 *                         404,
 *                         500,
 *                         502,
 *                     },
 *                 },
 *                 Members = new[]
 *                 {
 *                     new Aws.CloudFront.Inputs.DistributionOriginGroupMemberArgs
 *                     {
 *                         OriginId = "primaryS3",
 *                     },
 *                     new Aws.CloudFront.Inputs.DistributionOriginGroupMemberArgs
 *                     {
 *                         OriginId = "failoverS3",
 *                     },
 *                 },
 *             },
 *         },
 *         Origins = new[]
 *         {
 *             new Aws.CloudFront.Inputs.DistributionOriginArgs
 *             {
 *                 DomainName = primary.BucketRegionalDomainName,
 *                 OriginId = "primaryS3",
 *                 S3OriginConfig = new Aws.CloudFront.Inputs.DistributionOriginS3OriginConfigArgs
 *                 {
 *                     OriginAccessIdentity = @default.CloudfrontAccessIdentityPath,
 *                 },
 *             },
 *             new Aws.CloudFront.Inputs.DistributionOriginArgs
 *             {
 *                 DomainName = failover.BucketRegionalDomainName,
 *                 OriginId = "failoverS3",
 *                 S3OriginConfig = new Aws.CloudFront.Inputs.DistributionOriginS3OriginConfigArgs
 *                 {
 *                     OriginAccessIdentity = @default.CloudfrontAccessIdentityPath,
 *                 },
 *             },
 *         },
 *         DefaultCacheBehavior = new Aws.CloudFront.Inputs.DistributionDefaultCacheBehaviorArgs
 *         {
 *             TargetOriginId = "groupS3",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudfront"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudfront.NewDistribution(ctx, "s3_distribution", &cloudfront.DistributionArgs{
 * 			OriginGroups: cloudfront.DistributionOriginGroupArray{
 * 				&cloudfront.DistributionOriginGroupArgs{
 * 					OriginId: pulumi.String("groupS3"),
 * 					FailoverCriteria: &cloudfront.DistributionOriginGroupFailoverCriteriaArgs{
 * 						StatusCodes: pulumi.IntArray{
 * 							pulumi.Int(403),
 * 							pulumi.Int(404),
 * 							pulumi.Int(500),
 * 							pulumi.Int(502),
 * 						},
 * 					},
 * 					Members: cloudfront.DistributionOriginGroupMemberArray{
 * 						&cloudfront.DistributionOriginGroupMemberArgs{
 * 							OriginId: pulumi.String("primaryS3"),
 * 						},
 * 						&cloudfront.DistributionOriginGroupMemberArgs{
 * 							OriginId: pulumi.String("failoverS3"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 			Origins: cloudfront.DistributionOriginArray{
 * 				&cloudfront.DistributionOriginArgs{
 * 					DomainName: pulumi.Any(primary.BucketRegionalDomainName),
 * 					OriginId:   pulumi.String("primaryS3"),
 * 					S3OriginConfig: &cloudfront.DistributionOriginS3OriginConfigArgs{
 * 						OriginAccessIdentity: pulumi.Any(_default.CloudfrontAccessIdentityPath),
 * 					},
 * 				},
 * 				&cloudfront.DistributionOriginArgs{
 * 					DomainName: pulumi.Any(failover.BucketRegionalDomainName),
 * 					OriginId:   pulumi.String("failoverS3"),
 * 					S3OriginConfig: &cloudfront.DistributionOriginS3OriginConfigArgs{
 * 						OriginAccessIdentity: pulumi.Any(_default.CloudfrontAccessIdentityPath),
 * 					},
 * 				},
 * 			},
 * 			DefaultCacheBehavior: &cloudfront.DistributionDefaultCacheBehaviorArgs{
 * 				TargetOriginId: pulumi.String("groupS3"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudfront.Distribution;
 * import com.pulumi.aws.cloudfront.DistributionArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginGroupArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginGroupFailoverCriteriaArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginS3OriginConfigArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionDefaultCacheBehaviorArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var s3Distribution = new Distribution("s3Distribution", DistributionArgs.builder()
 *             .originGroups(DistributionOriginGroupArgs.builder()
 *                 .originId("groupS3")
 *                 .failoverCriteria(DistributionOriginGroupFailoverCriteriaArgs.builder()
 *                     .statusCodes(
 *                         403,
 *                         404,
 *                         500,
 *                         502)
 *                     .build())
 *                 .members(
 *                     DistributionOriginGroupMemberArgs.builder()
 *                         .originId("primaryS3")
 *                         .build(),
 *                     DistributionOriginGroupMemberArgs.builder()
 *                         .originId("failoverS3")
 *                         .build())
 *                 .build())
 *             .origins(
 *                 DistributionOriginArgs.builder()
 *                     .domainName(primary.bucketRegionalDomainName())
 *                     .originId("primaryS3")
 *                     .s3OriginConfig(DistributionOriginS3OriginConfigArgs.builder()
 *                         .originAccessIdentity(default_.cloudfrontAccessIdentityPath())
 *                         .build())
 *                     .build(),
 *                 DistributionOriginArgs.builder()
 *                     .domainName(failover.bucketRegionalDomainName())
 *                     .originId("failoverS3")
 *                     .s3OriginConfig(DistributionOriginS3OriginConfigArgs.builder()
 *                         .originAccessIdentity(default_.cloudfrontAccessIdentityPath())
 *                         .build())
 *                     .build())
 *             .defaultCacheBehavior(DistributionDefaultCacheBehaviorArgs.builder()
 *                 .targetOriginId("groupS3")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   s3Distribution:
 *     type: aws:cloudfront:Distribution
 *     name: s3_distribution
 *     properties:
 *       originGroups:
 *         - originId: groupS3
 *           failoverCriteria:
 *             statusCodes:
 *               - 403
 *               - 404
 *               - 500
 *               - 502
 *           members:
 *             - originId: primaryS3
 *             - originId: failoverS3
 *       origins:
 *         - domainName: ${primary.bucketRegionalDomainName}
 *           originId: primaryS3
 *           s3OriginConfig:
 *             originAccessIdentity: ${default.cloudfrontAccessIdentityPath}
 *         - domainName: ${failover.bucketRegionalDomainName}
 *           originId: failoverS3
 *           s3OriginConfig:
 *             originAccessIdentity: ${default.cloudfrontAccessIdentityPath}
 *       defaultCacheBehavior:
 *         targetOriginId: groupS3
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Managed Caching Policy
 * The example below creates a CloudFront distribution with an [AWS managed caching policy](https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html).
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const s3OriginId = "myS3Origin";
 * const s3Distribution = new aws.cloudfront.Distribution("s3_distribution", {
 *     origins: [{
 *         domainName: primary.bucketRegionalDomainName,
 *         originId: "myS3Origin",
 *         s3OriginConfig: {
 *             originAccessIdentity: _default.cloudfrontAccessIdentityPath,
 *         },
 *     }],
 *     enabled: true,
 *     isIpv6Enabled: true,
 *     comment: "Some comment",
 *     defaultRootObject: "index.html",
 *     defaultCacheBehavior: {
 *         cachePolicyId: "4135ea2d-6df8-44a3-9df3-4b5a84be39ad",
 *         allowedMethods: [
 *             "GET",
 *             "HEAD",
 *             "OPTIONS",
 *         ],
 *         targetOriginId: s3OriginId,
 *     },
 *     restrictions: {
 *         geoRestriction: {
 *             restrictionType: "whitelist",
 *             locations: [
 *                 "US",
 *                 "CA",
 *                 "GB",
 *                 "DE",
 *             ],
 *         },
 *     },
 *     viewerCertificate: {
 *         cloudfrontDefaultCertificate: true,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * s3_origin_id = "myS3Origin"
 * s3_distribution = aws.cloudfront.Distribution("s3_distribution",
 *     origins=[{
 *         "domain_name": primary["bucketRegionalDomainName"],
 *         "origin_id": "myS3Origin",
 *         "s3_origin_config": {
 *             "origin_access_identity": default["cloudfrontAccessIdentityPath"],
 *         },
 *     }],
 *     enabled=True,
 *     is_ipv6_enabled=True,
 *     comment="Some comment",
 *     default_root_object="index.html",
 *     default_cache_behavior={
 *         "cache_policy_id": "4135ea2d-6df8-44a3-9df3-4b5a84be39ad",
 *         "allowed_methods": [
 *             "GET",
 *             "HEAD",
 *             "OPTIONS",
 *         ],
 *         "target_origin_id": s3_origin_id,
 *     },
 *     restrictions={
 *         "geo_restriction": {
 *             "restriction_type": "whitelist",
 *             "locations": [
 *                 "US",
 *                 "CA",
 *                 "GB",
 *                 "DE",
 *             ],
 *         },
 *     },
 *     viewer_certificate={
 *         "cloudfront_default_certificate": True,
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var s3OriginId = "myS3Origin";
 *     var s3Distribution = new Aws.CloudFront.Distribution("s3_distribution", new()
 *     {
 *         Origins = new[]
 *         {
 *             new Aws.CloudFront.Inputs.DistributionOriginArgs
 *             {
 *                 DomainName = primary.BucketRegionalDomainName,
 *                 OriginId = "myS3Origin",
 *                 S3OriginConfig = new Aws.CloudFront.Inputs.DistributionOriginS3OriginConfigArgs
 *                 {
 *                     OriginAccessIdentity = @default.CloudfrontAccessIdentityPath,
 *                 },
 *             },
 *         },
 *         Enabled = true,
 *         IsIpv6Enabled = true,
 *         Comment = "Some comment",
 *         DefaultRootObject = "index.html",
 *         DefaultCacheBehavior = new Aws.CloudFront.Inputs.DistributionDefaultCacheBehaviorArgs
 *         {
 *             CachePolicyId = "4135ea2d-6df8-44a3-9df3-4b5a84be39ad",
 *             AllowedMethods = new[]
 *             {
 *                 "GET",
 *                 "HEAD",
 *                 "OPTIONS",
 *             },
 *             TargetOriginId = s3OriginId,
 *         },
 *         Restrictions = new Aws.CloudFront.Inputs.DistributionRestrictionsArgs
 *         {
 *             GeoRestriction = new Aws.CloudFront.Inputs.DistributionRestrictionsGeoRestrictionArgs
 *             {
 *                 RestrictionType = "whitelist",
 *                 Locations = new[]
 *                 {
 *                     "US",
 *                     "CA",
 *                     "GB",
 *                     "DE",
 *                 },
 *             },
 *         },
 *         ViewerCertificate = new Aws.CloudFront.Inputs.DistributionViewerCertificateArgs
 *         {
 *             CloudfrontDefaultCertificate = true,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudfront"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		s3OriginId := "myS3Origin"
 * 		_, err := cloudfront.NewDistribution(ctx, "s3_distribution", &cloudfront.DistributionArgs{
 * 			Origins: cloudfront.DistributionOriginArray{
 * 				&cloudfront.DistributionOriginArgs{
 * 					DomainName: pulumi.Any(primary.BucketRegionalDomainName),
 * 					OriginId:   pulumi.String("myS3Origin"),
 * 					S3OriginConfig: &cloudfront.DistributionOriginS3OriginConfigArgs{
 * 						OriginAccessIdentity: pulumi.Any(_default.CloudfrontAccessIdentityPath),
 * 					},
 * 				},
 * 			},
 * 			Enabled:           pulumi.Bool(true),
 * 			IsIpv6Enabled:     pulumi.Bool(true),
 * 			Comment:           pulumi.String("Some comment"),
 * 			DefaultRootObject: pulumi.String("index.html"),
 * 			DefaultCacheBehavior: &cloudfront.DistributionDefaultCacheBehaviorArgs{
 * 				CachePolicyId: pulumi.String("4135ea2d-6df8-44a3-9df3-4b5a84be39ad"),
 * 				AllowedMethods: pulumi.StringArray{
 * 					pulumi.String("GET"),
 * 					pulumi.String("HEAD"),
 * 					pulumi.String("OPTIONS"),
 * 				},
 * 				TargetOriginId: pulumi.String(s3OriginId),
 * 			},
 * 			Restrictions: &cloudfront.DistributionRestrictionsArgs{
 * 				GeoRestriction: &cloudfront.DistributionRestrictionsGeoRestrictionArgs{
 * 					RestrictionType: pulumi.String("whitelist"),
 * 					Locations: pulumi.StringArray{
 * 						pulumi.String("US"),
 * 						pulumi.String("CA"),
 * 						pulumi.String("GB"),
 * 						pulumi.String("DE"),
 * 					},
 * 				},
 * 			},
 * 			ViewerCertificate: &cloudfront.DistributionViewerCertificateArgs{
 * 				CloudfrontDefaultCertificate: pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudfront.Distribution;
 * import com.pulumi.aws.cloudfront.DistributionArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginS3OriginConfigArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionDefaultCacheBehaviorArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionRestrictionsArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionRestrictionsGeoRestrictionArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionViewerCertificateArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var s3OriginId = "myS3Origin";
 *         var s3Distribution = new Distribution("s3Distribution", DistributionArgs.builder()
 *             .origins(DistributionOriginArgs.builder()
 *                 .domainName(primary.bucketRegionalDomainName())
 *                 .originId("myS3Origin")
 *                 .s3OriginConfig(DistributionOriginS3OriginConfigArgs.builder()
 *                     .originAccessIdentity(default_.cloudfrontAccessIdentityPath())
 *                     .build())
 *                 .build())
 *             .enabled(true)
 *             .isIpv6Enabled(true)
 *             .comment("Some comment")
 *             .defaultRootObject("index.html")
 *             .defaultCacheBehavior(DistributionDefaultCacheBehaviorArgs.builder()
 *                 .cachePolicyId("4135ea2d-6df8-44a3-9df3-4b5a84be39ad")
 *                 .allowedMethods(
 *                     "GET",
 *                     "HEAD",
 *                     "OPTIONS")
 *                 .targetOriginId(s3OriginId)
 *                 .build())
 *             .restrictions(DistributionRestrictionsArgs.builder()
 *                 .geoRestriction(DistributionRestrictionsGeoRestrictionArgs.builder()
 *                     .restrictionType("whitelist")
 *                     .locations(
 *                         "US",
 *                         "CA",
 *                         "GB",
 *                         "DE")
 *                     .build())
 *                 .build())
 *             .viewerCertificate(DistributionViewerCertificateArgs.builder()
 *                 .cloudfrontDefaultCertificate(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   s3Distribution:
 *     type: aws:cloudfront:Distribution
 *     name: s3_distribution
 *     properties:
 *       origins:
 *         - domainName: ${primary.bucketRegionalDomainName}
 *           originId: myS3Origin
 *           s3OriginConfig:
 *             originAccessIdentity: ${default.cloudfrontAccessIdentityPath}
 *       enabled: true
 *       isIpv6Enabled: true
 *       comment: Some comment
 *       defaultRootObject: index.html
 *       defaultCacheBehavior:
 *         cachePolicyId: 4135ea2d-6df8-44a3-9df3-4b5a84be39ad
 *         allowedMethods:
 *           - GET
 *           - HEAD
 *           - OPTIONS
 *         targetOriginId: ${s3OriginId}
 *       restrictions:
 *         geoRestriction:
 *           restrictionType: whitelist
 *           locations:
 *             - US
 *             - CA
 *             - GB
 *             - DE
 *       viewerCertificate:
 *         cloudfrontDefaultCertificate: true
 * variables:
 *   s3OriginId: myS3Origin
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import CloudFront Distributions using the `id`. For example:
 * ```sh
 * $ pulumi import aws:cloudfront/distribution:Distribution distribution E74FTE3EXAMPLE
 * ```
 * @property aliases
 * @property comment
 * @property continuousDeploymentPolicyId
 * @property customErrorResponses
 * @property defaultCacheBehavior
 * @property defaultRootObject
 * @property enabled `true` if any of the AWS accounts listed as trusted signers have active CloudFront key pairs
 * @property httpVersion
 * @property isIpv6Enabled
 * @property loggingConfig
 * @property orderedCacheBehaviors
 * @property originGroups
 * @property origins
 * @property priceClass
 * @property restrictions
 * @property retainOnDelete
 * @property staging
 * @property tags
 * @property viewerCertificate
 * @property waitForDeployment
 * @property webAclId
 * */*/*/*/*/*/*/*/*/*/*/*/
 */
public data class DistributionArgs(
    public val aliases: Output<List<String>>? = null,
    public val comment: Output<String>? = null,
    public val continuousDeploymentPolicyId: Output<String>? = null,
    public val customErrorResponses: Output<List<DistributionCustomErrorResponseArgs>>? = null,
    public val defaultCacheBehavior: Output<DistributionDefaultCacheBehaviorArgs>? = null,
    public val defaultRootObject: Output<String>? = null,
    public val enabled: Output<Boolean>? = null,
    public val httpVersion: Output<String>? = null,
    public val isIpv6Enabled: Output<Boolean>? = null,
    public val loggingConfig: Output<DistributionLoggingConfigArgs>? = null,
    public val orderedCacheBehaviors: Output<List<DistributionOrderedCacheBehaviorArgs>>? = null,
    public val originGroups: Output<List<DistributionOriginGroupArgs>>? = null,
    public val origins: Output<List<DistributionOriginArgs>>? = null,
    public val priceClass: Output<String>? = null,
    public val restrictions: Output<DistributionRestrictionsArgs>? = null,
    public val retainOnDelete: Output<Boolean>? = null,
    public val staging: Output<Boolean>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val viewerCertificate: Output<DistributionViewerCertificateArgs>? = null,
    public val waitForDeployment: Output<Boolean>? = null,
    public val webAclId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.cloudfront.DistributionArgs> {
    override fun toJava(): com.pulumi.aws.cloudfront.DistributionArgs =
        com.pulumi.aws.cloudfront.DistributionArgs.builder()
            .aliases(aliases?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .comment(comment?.applyValue({ args0 -> args0 }))
            .continuousDeploymentPolicyId(continuousDeploymentPolicyId?.applyValue({ args0 -> args0 }))
            .customErrorResponses(
                customErrorResponses?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .defaultCacheBehavior(
                defaultCacheBehavior?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .defaultRootObject(defaultRootObject?.applyValue({ args0 -> args0 }))
            .enabled(enabled?.applyValue({ args0 -> args0 }))
            .httpVersion(httpVersion?.applyValue({ args0 -> args0 }))
            .isIpv6Enabled(isIpv6Enabled?.applyValue({ args0 -> args0 }))
            .loggingConfig(loggingConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .orderedCacheBehaviors(
                orderedCacheBehaviors?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .originGroups(
                originGroups?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .origins(
                origins?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .priceClass(priceClass?.applyValue({ args0 -> args0 }))
            .restrictions(restrictions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .retainOnDelete(retainOnDelete?.applyValue({ args0 -> args0 }))
            .staging(staging?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .viewerCertificate(viewerCertificate?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .waitForDeployment(waitForDeployment?.applyValue({ args0 -> args0 }))
            .webAclId(webAclId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [DistributionArgs].
 */
@PulumiTagMarker
public class DistributionArgsBuilder internal constructor() {
    private var aliases: Output<List<String>>? = null

    private var comment: Output<String>? = null

    private var continuousDeploymentPolicyId: Output<String>? = null

    private var customErrorResponses: Output<List<DistributionCustomErrorResponseArgs>>? = null

    private var defaultCacheBehavior: Output<DistributionDefaultCacheBehaviorArgs>? = null

    private var defaultRootObject: Output<String>? = null

    private var enabled: Output<Boolean>? = null

    private var httpVersion: Output<String>? = null

    private var isIpv6Enabled: Output<Boolean>? = null

    private var loggingConfig: Output<DistributionLoggingConfigArgs>? = null

    private var orderedCacheBehaviors: Output<List<DistributionOrderedCacheBehaviorArgs>>? = null

    private var originGroups: Output<List<DistributionOriginGroupArgs>>? = null

    private var origins: Output<List<DistributionOriginArgs>>? = null

    private var priceClass: Output<String>? = null

    private var restrictions: Output<DistributionRestrictionsArgs>? = null

    private var retainOnDelete: Output<Boolean>? = null

    private var staging: Output<Boolean>? = null

    private var tags: Output<Map<String, String>>? = null

    private var viewerCertificate: Output<DistributionViewerCertificateArgs>? = null

    private var waitForDeployment: Output<Boolean>? = null

    private var webAclId: Output<String>? = null

    /**
     * @param value
     */
    @JvmName("acprvuvdxhbhynju")
    public suspend fun aliases(`value`: Output<List<String>>) {
        this.aliases = value
    }

    @JvmName("dsbefvwxiovrnjwq")
    public suspend fun aliases(vararg values: Output<String>) {
        this.aliases = Output.all(values.asList())
    }

    /**
     * @param values
     */
    @JvmName("xoewvtqeucsyajnv")
    public suspend fun aliases(values: List<Output<String>>) {
        this.aliases = Output.all(values)
    }

    /**
     * @param value
     */
    @JvmName("lgxpumxvibemnteo")
    public suspend fun comment(`value`: Output<String>) {
        this.comment = value
    }

    /**
     * @param value
     */
    @JvmName("pticiwnmejvpgwpy")
    public suspend fun continuousDeploymentPolicyId(`value`: Output<String>) {
        this.continuousDeploymentPolicyId = value
    }

    /**
     * @param value
     */
    @JvmName("wfhuhlfedmqvjcrj")
    public suspend fun customErrorResponses(`value`: Output<List<DistributionCustomErrorResponseArgs>>) {
        this.customErrorResponses = value
    }

    @JvmName("gxyxxwwpjhwhgokg")
    public suspend fun customErrorResponses(vararg values: Output<DistributionCustomErrorResponseArgs>) {
        this.customErrorResponses = Output.all(values.asList())
    }

    /**
     * @param values
     */
    @JvmName("wcrxtwnirhyerwmu")
    public suspend fun customErrorResponses(values: List<Output<DistributionCustomErrorResponseArgs>>) {
        this.customErrorResponses = Output.all(values)
    }

    /**
     * @param value
     */
    @JvmName("idqiibsmiyqmyqhc")
    public suspend fun defaultCacheBehavior(`value`: Output<DistributionDefaultCacheBehaviorArgs>) {
        this.defaultCacheBehavior = value
    }

    /**
     * @param value
     */
    @JvmName("wjfpamxhkmycypgc")
    public suspend fun defaultRootObject(`value`: Output<String>) {
        this.defaultRootObject = value
    }

    /**
     * @param value `true` if any of the AWS accounts listed as trusted signers have active CloudFront key pairs
     */
    @JvmName("uhsncplkkbnsctgc")
    public suspend fun enabled(`value`: Output<Boolean>) {
        this.enabled = value
    }

    /**
     * @param value
     */
    @JvmName("hiphlqqcgaikybut")
    public suspend fun httpVersion(`value`: Output<String>) {
        this.httpVersion = value
    }

    /**
     * @param value
     */
    @JvmName("ffsgubrmryobsrur")
    public suspend fun isIpv6Enabled(`value`: Output<Boolean>) {
        this.isIpv6Enabled = value
    }

    /**
     * @param value
     */
    @JvmName("gdaxxvdrouiuebkd")
    public suspend fun loggingConfig(`value`: Output<DistributionLoggingConfigArgs>) {
        this.loggingConfig = value
    }

    /**
     * @param value
     */
    @JvmName("kihexbibqkbwrikf")
    public suspend fun orderedCacheBehaviors(`value`: Output<List<DistributionOrderedCacheBehaviorArgs>>) {
        this.orderedCacheBehaviors = value
    }

    @JvmName("ctdormrawqdtsksk")
    public suspend fun orderedCacheBehaviors(vararg values: Output<DistributionOrderedCacheBehaviorArgs>) {
        this.orderedCacheBehaviors = Output.all(values.asList())
    }

    /**
     * @param values
     */
    @JvmName("qycovsmjuguhvhwl")
    public suspend fun orderedCacheBehaviors(values: List<Output<DistributionOrderedCacheBehaviorArgs>>) {
        this.orderedCacheBehaviors = Output.all(values)
    }

    /**
     * @param value
     */
    @JvmName("mibokkxufqqqfhxf")
    public suspend fun originGroups(`value`: Output<List<DistributionOriginGroupArgs>>) {
        this.originGroups = value
    }

    @JvmName("fdrokdktsrvkohmf")
    public suspend fun originGroups(vararg values: Output<DistributionOriginGroupArgs>) {
        this.originGroups = Output.all(values.asList())
    }

    /**
     * @param values
     */
    @JvmName("mtabmgiqmiaffotj")
    public suspend fun originGroups(values: List<Output<DistributionOriginGroupArgs>>) {
        this.originGroups = Output.all(values)
    }

    /**
     * @param value
     */
    @JvmName("ndfquxexcmbbkocr")
    public suspend fun origins(`value`: Output<List<DistributionOriginArgs>>) {
        this.origins = value
    }

    @JvmName("elftiaftebvooutq")
    public suspend fun origins(vararg values: Output<DistributionOriginArgs>) {
        this.origins = Output.all(values.asList())
    }

    /**
     * @param values
     */
    @JvmName("qyptcmmrjcisrucv")
    public suspend fun origins(values: List<Output<DistributionOriginArgs>>) {
        this.origins = Output.all(values)
    }

    /**
     * @param value
     */
    @JvmName("gkbwkqacaaruefir")
    public suspend fun priceClass(`value`: Output<String>) {
        this.priceClass = value
    }

    /**
     * @param value
     */
    @JvmName("syvymrsvibmmiljn")
    public suspend fun restrictions(`value`: Output<DistributionRestrictionsArgs>) {
        this.restrictions = value
    }

    /**
     * @param value
     */
    @JvmName("anukuhbwqfyusllt")
    public suspend fun retainOnDelete(`value`: Output<Boolean>) {
        this.retainOnDelete = value
    }

    /**
     * @param value
     */
    @JvmName("afkbbhyitgiosdhg")
    public suspend fun staging(`value`: Output<Boolean>) {
        this.staging = value
    }

    /**
     * @param value
     */
    @JvmName("ulpnfcqofhuvnudt")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value
     */
    @JvmName("ixnarjtnlnjmwxsk")
    public suspend fun viewerCertificate(`value`: Output<DistributionViewerCertificateArgs>) {
        this.viewerCertificate = value
    }

    /**
     * @param value
     */
    @JvmName("biakknlgqbuelaty")
    public suspend fun waitForDeployment(`value`: Output<Boolean>) {
        this.waitForDeployment = value
    }

    /**
     * @param value
     */
    @JvmName("jnmiamftcctosvvn")
    public suspend fun webAclId(`value`: Output<String>) {
        this.webAclId = value
    }

    /**
     * @param value
     */
    @JvmName("qjvvdbgbqwcybjxd")
    public suspend fun aliases(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.aliases = mapped
    }

    /**
     * @param values
     */
    @JvmName("cbxqkaprjnqkjiyh")
    public suspend fun aliases(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.aliases = mapped
    }

    /**
     * @param value
     */
    @JvmName("coilblobkbsxtxis")
    public suspend fun comment(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.comment = mapped
    }

    /**
     * @param value
     */
    @JvmName("obkmggygjswhprsh")
    public suspend fun continuousDeploymentPolicyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.continuousDeploymentPolicyId = mapped
    }

    /**
     * @param value
     */
    @JvmName("lqdmehuoufokxtiu")
    public suspend fun customErrorResponses(`value`: List<DistributionCustomErrorResponseArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customErrorResponses = mapped
    }

    /**
     * @param argument
     */
    @JvmName("wkuutupwgboyjxux")
    public suspend fun customErrorResponses(argument: List<suspend DistributionCustomErrorResponseArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            DistributionCustomErrorResponseArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.customErrorResponses = mapped
    }

    /**
     * @param argument
     */
    @JvmName("lckkcqnjqspaxsbm")
    public suspend fun customErrorResponses(vararg argument: suspend DistributionCustomErrorResponseArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            DistributionCustomErrorResponseArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.customErrorResponses = mapped
    }

    /**
     * @param argument
     */
    @JvmName("midrtrmacbjoiohk")
    public suspend fun customErrorResponses(argument: suspend DistributionCustomErrorResponseArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            DistributionCustomErrorResponseArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.customErrorResponses = mapped
    }

    /**
     * @param values
     */
    @JvmName("fnylcxxcwolqwsun")
    public suspend fun customErrorResponses(vararg values: DistributionCustomErrorResponseArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.customErrorResponses = mapped
    }

    /**
     * @param value
     */
    @JvmName("htoackovsyhxesuw")
    public suspend fun defaultCacheBehavior(`value`: DistributionDefaultCacheBehaviorArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultCacheBehavior = mapped
    }

    /**
     * @param argument
     */
    @JvmName("qsimedsnqtcwplal")
    public suspend fun defaultCacheBehavior(argument: suspend DistributionDefaultCacheBehaviorArgsBuilder.() -> Unit) {
        val toBeMapped = DistributionDefaultCacheBehaviorArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.defaultCacheBehavior = mapped
    }

    /**
     * @param value
     */
    @JvmName("xvoyelsisptudack")
    public suspend fun defaultRootObject(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultRootObject = mapped
    }

    /**
     * @param value `true` if any of the AWS accounts listed as trusted signers have active CloudFront key pairs
     */
    @JvmName("chhvhcjaogooqfod")
    public suspend fun enabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabled = mapped
    }

    /**
     * @param value
     */
    @JvmName("wesfgjamrkhgqxqn")
    public suspend fun httpVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.httpVersion = mapped
    }

    /**
     * @param value
     */
    @JvmName("pgcawcwlotknumff")
    public suspend fun isIpv6Enabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.isIpv6Enabled = mapped
    }

    /**
     * @param value
     */
    @JvmName("lccqrxxsabcmsahg")
    public suspend fun loggingConfig(`value`: DistributionLoggingConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.loggingConfig = mapped
    }

    /**
     * @param argument
     */
    @JvmName("ajxxyrvpwntjlycl")
    public suspend fun loggingConfig(argument: suspend DistributionLoggingConfigArgsBuilder.() -> Unit) {
        val toBeMapped = DistributionLoggingConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.loggingConfig = mapped
    }

    /**
     * @param value
     */
    @JvmName("yryygjgtnmucjbwt")
    public suspend fun orderedCacheBehaviors(`value`: List<DistributionOrderedCacheBehaviorArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.orderedCacheBehaviors = mapped
    }

    /**
     * @param argument
     */
    @JvmName("rvorntdohfkgebyv")
    public suspend fun orderedCacheBehaviors(argument: List<suspend DistributionOrderedCacheBehaviorArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            DistributionOrderedCacheBehaviorArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.orderedCacheBehaviors = mapped
    }

    /**
     * @param argument
     */
    @JvmName("csinbpqxkdeljsbm")
    public suspend fun orderedCacheBehaviors(vararg argument: suspend DistributionOrderedCacheBehaviorArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            DistributionOrderedCacheBehaviorArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.orderedCacheBehaviors = mapped
    }

    /**
     * @param argument
     */
    @JvmName("ktxghxguwdduwcoe")
    public suspend fun orderedCacheBehaviors(argument: suspend DistributionOrderedCacheBehaviorArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            DistributionOrderedCacheBehaviorArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.orderedCacheBehaviors = mapped
    }

    /**
     * @param values
     */
    @JvmName("itiwbkpkvajorwng")
    public suspend fun orderedCacheBehaviors(vararg values: DistributionOrderedCacheBehaviorArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.orderedCacheBehaviors = mapped
    }

    /**
     * @param value
     */
    @JvmName("qgojffdpnlycekho")
    public suspend fun originGroups(`value`: List<DistributionOriginGroupArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.originGroups = mapped
    }

    /**
     * @param argument
     */
    @JvmName("nfyrrkpnbdaeoofc")
    public suspend fun originGroups(argument: List<suspend DistributionOriginGroupArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            DistributionOriginGroupArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.originGroups = mapped
    }

    /**
     * @param argument
     */
    @JvmName("gyutfbxxouldswwu")
    public suspend fun originGroups(vararg argument: suspend DistributionOriginGroupArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            DistributionOriginGroupArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.originGroups = mapped
    }

    /**
     * @param argument
     */
    @JvmName("waktsslaabhxlpqx")
    public suspend fun originGroups(argument: suspend DistributionOriginGroupArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            DistributionOriginGroupArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.originGroups = mapped
    }

    /**
     * @param values
     */
    @JvmName("auncqrudtpdyxcaa")
    public suspend fun originGroups(vararg values: DistributionOriginGroupArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.originGroups = mapped
    }

    /**
     * @param value
     */
    @JvmName("ukaoqxsdriyavnto")
    public suspend fun origins(`value`: List<DistributionOriginArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.origins = mapped
    }

    /**
     * @param argument
     */
    @JvmName("ayjyikqsokqamslk")
    public suspend fun origins(argument: List<suspend DistributionOriginArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            DistributionOriginArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.origins = mapped
    }

    /**
     * @param argument
     */
    @JvmName("ycgathmwgueagtta")
    public suspend fun origins(vararg argument: suspend DistributionOriginArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            DistributionOriginArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.origins = mapped
    }

    /**
     * @param argument
     */
    @JvmName("qpqqteopitwmqfrm")
    public suspend fun origins(argument: suspend DistributionOriginArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(DistributionOriginArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.origins = mapped
    }

    /**
     * @param values
     */
    @JvmName("stptagcxahxqkwwt")
    public suspend fun origins(vararg values: DistributionOriginArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.origins = mapped
    }

    /**
     * @param value
     */
    @JvmName("twgeqmnwtjewctnn")
    public suspend fun priceClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.priceClass = mapped
    }

    /**
     * @param value
     */
    @JvmName("ymbwwcwyxnxyykgu")
    public suspend fun restrictions(`value`: DistributionRestrictionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restrictions = mapped
    }

    /**
     * @param argument
     */
    @JvmName("hpqwahxohescbrgq")
    public suspend fun restrictions(argument: suspend DistributionRestrictionsArgsBuilder.() -> Unit) {
        val toBeMapped = DistributionRestrictionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.restrictions = mapped
    }

    /**
     * @param value
     */
    @JvmName("lxmkrgcseugethie")
    public suspend fun retainOnDelete(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.retainOnDelete = mapped
    }

    /**
     * @param value
     */
    @JvmName("uaxywfondnsymobv")
    public suspend fun staging(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.staging = mapped
    }

    /**
     * @param value
     */
    @JvmName("odcnkubclweusdjq")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values
     */
    @JvmName("uppwemkhutfbaikb")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value
     */
    @JvmName("ceqkyndmnahudyqs")
    public suspend fun viewerCertificate(`value`: DistributionViewerCertificateArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.viewerCertificate = mapped
    }

    /**
     * @param argument
     */
    @JvmName("rbyemqxuvcnssfak")
    public suspend fun viewerCertificate(argument: suspend DistributionViewerCertificateArgsBuilder.() -> Unit) {
        val toBeMapped = DistributionViewerCertificateArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.viewerCertificate = mapped
    }

    /**
     * @param value
     */
    @JvmName("hndrxuemqsubftgx")
    public suspend fun waitForDeployment(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.waitForDeployment = mapped
    }

    /**
     * @param value
     */
    @JvmName("rnlpcyfslxesqjju")
    public suspend fun webAclId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.webAclId = mapped
    }

    internal fun build(): DistributionArgs = DistributionArgs(
        aliases = aliases,
        comment = comment,
        continuousDeploymentPolicyId = continuousDeploymentPolicyId,
        customErrorResponses = customErrorResponses,
        defaultCacheBehavior = defaultCacheBehavior,
        defaultRootObject = defaultRootObject,
        enabled = enabled,
        httpVersion = httpVersion,
        isIpv6Enabled = isIpv6Enabled,
        loggingConfig = loggingConfig,
        orderedCacheBehaviors = orderedCacheBehaviors,
        originGroups = originGroups,
        origins = origins,
        priceClass = priceClass,
        restrictions = restrictions,
        retainOnDelete = retainOnDelete,
        staging = staging,
        tags = tags,
        viewerCertificate = viewerCertificate,
        waitForDeployment = waitForDeployment,
        webAclId = webAclId,
    )
}
