@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cloudfront.kotlin

import com.pulumi.aws.cloudfront.OriginAccessIdentityArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Creates an Amazon CloudFront origin access identity.
 * For information about CloudFront distributions, see the
 * [Amazon CloudFront Developer Guide](http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Introduction.html). For more information on generating
 * origin access identities, see
 * [Using an Origin Access Identity to Restrict Access to Your Amazon S3 Content][2].
 * ## Example Usage
 * The following example below creates a CloudFront origin access identity.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloudfront.OriginAccessIdentity("example", {comment: "Some comment"});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloudfront.OriginAccessIdentity("example", comment="Some comment")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CloudFront.OriginAccessIdentity("example", new()
 *     {
 *         Comment = "Some comment",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudfront"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudfront.NewOriginAccessIdentity(ctx, "example", &cloudfront.OriginAccessIdentityArgs{
 * 			Comment: pulumi.String("Some comment"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudfront.OriginAccessIdentity;
 * import com.pulumi.aws.cloudfront.OriginAccessIdentityArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new OriginAccessIdentity("example", OriginAccessIdentityArgs.builder()
 *             .comment("Some comment")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloudfront:OriginAccessIdentity
 *     properties:
 *       comment: Some comment
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Using With CloudFront
 * Normally, when referencing an origin access identity in CloudFront, you need to
 * prefix the ID with the `origin-access-identity/cloudfront/` special path.
 * The `cloudfront_access_identity_path` allows this to be circumvented.
 * The below snippet demonstrates use with the `s3_origin_config` structure for the
 * `aws.cloudfront.Distribution` resource:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloudfront.Distribution("example", {origins: [{
 *     s3OriginConfig: {
 *         originAccessIdentity: exampleAwsCloudfrontOriginAccessIdentity.cloudfrontAccessIdentityPath,
 *     },
 * }]});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloudfront.Distribution("example", origins=[{
 *     "s3_origin_config": {
 *         "origin_access_identity": example_aws_cloudfront_origin_access_identity["cloudfrontAccessIdentityPath"],
 *     },
 * }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CloudFront.Distribution("example", new()
 *     {
 *         Origins = new[]
 *         {
 *             new Aws.CloudFront.Inputs.DistributionOriginArgs
 *             {
 *                 S3OriginConfig = new Aws.CloudFront.Inputs.DistributionOriginS3OriginConfigArgs
 *                 {
 *                     OriginAccessIdentity = exampleAwsCloudfrontOriginAccessIdentity.CloudfrontAccessIdentityPath,
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudfront"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudfront.NewDistribution(ctx, "example", &cloudfront.DistributionArgs{
 * 			Origins: cloudfront.DistributionOriginArray{
 * 				&cloudfront.DistributionOriginArgs{
 * 					S3OriginConfig: &cloudfront.DistributionOriginS3OriginConfigArgs{
 * 						OriginAccessIdentity: pulumi.Any(exampleAwsCloudfrontOriginAccessIdentity.CloudfrontAccessIdentityPath),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudfront.Distribution;
 * import com.pulumi.aws.cloudfront.DistributionArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginArgs;
 * import com.pulumi.aws.cloudfront.inputs.DistributionOriginS3OriginConfigArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Distribution("example", DistributionArgs.builder()
 *             .origins(DistributionOriginArgs.builder()
 *                 .s3OriginConfig(DistributionOriginS3OriginConfigArgs.builder()
 *                     .originAccessIdentity(exampleAwsCloudfrontOriginAccessIdentity.cloudfrontAccessIdentityPath())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloudfront:Distribution
 *     properties:
 *       origins:
 *         - s3OriginConfig:
 *             originAccessIdentity: ${exampleAwsCloudfrontOriginAccessIdentity.cloudfrontAccessIdentityPath}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Updating your bucket policy
 * Note that the AWS API may translate the `s3_canonical_user_id` `CanonicalUser`
 * principal into an `AWS` IAM ARN principal when supplied in an
 * `aws.s3.BucketV2` bucket policy, causing spurious diffs. If
 * you see this behavior, use the `iam_arn` instead:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const s3Policy = aws.iam.getPolicyDocument({
 *     statements: [{
 *         actions: ["s3:GetObject"],
 *         resources: [`${exampleAwsS3Bucket&#46;arn}/*`],
 *         principals: [{
 *             type: "AWS",
 *             identifiers: [exampleAwsCloudfrontOriginAccessIdentity&#46;iamArn],
 *         }],
 *     }],
 * });
 * const example = new aws.s3.BucketPolicy("example", {
 *     bucket: exampleAwsS3Bucket.id,
 *     policy: s3Policy.then(s3Policy => s3Policy.json),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * s3_policy = aws.iam.get_policy_document(statements=[{
 *     "actions": ["s3:GetObject"],
 *     "resources": [f"{example_aws_s3_bucket['arn']}/*"],
 *     "principals": [{
 *         "type": "AWS",
 *         "identifiers": [example_aws_cloudfront_origin_access_identity["iamArn"]],
 *     }],
 * }])
 * example = aws.s3.BucketPolicy("example",
 *     bucket=example_aws_s3_bucket["id"],
 *     policy=s3_policy.json)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var s3Policy = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "s3:GetObject",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     $"{exampleAwsS3Bucket.Arn}/*",
 *                 },
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "AWS",
 *                         Identifiers = new[]
 *                         {
 *                             exampleAwsCloudfrontOriginAccessIdentity.IamArn,
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     var example = new Aws.S3.BucketPolicy("example", new()
 *     {
 *         Bucket = exampleAwsS3Bucket.Id,
 *         Policy = s3Policy.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * s3Policy, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * Statements: []iam.GetPolicyDocumentStatement{
 * {
 * Actions: []string{
 * "s3:GetObject",
 * },
 * Resources: []string{
 * fmt.Sprintf("%v/*", exampleAwsS3Bucket.Arn),
 * },
 * Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * {
 * Type: "AWS",
 * Identifiers: interface{}{
 * exampleAwsCloudfrontOriginAccessIdentity.IamArn,
 * },
 * },
 * },
 * },
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * _, err = s3.NewBucketPolicy(ctx, "example", &s3.BucketPolicyArgs{
 * Bucket: pulumi.Any(exampleAwsS3Bucket.Id),
 * Policy: pulumi.String(s3Policy.Json),
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.s3.BucketPolicy;
 * import com.pulumi.aws.s3.BucketPolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var s3Policy = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .actions("s3:GetObject")
 *                 .resources(String.format("%s/*", exampleAwsS3Bucket.arn()))
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("AWS")
 *                     .identifiers(exampleAwsCloudfrontOriginAccessIdentity.iamArn())
 *                     .build())
 *                 .build())
 *             .build());
 *         var example = new BucketPolicy("example", BucketPolicyArgs.builder()
 *             .bucket(exampleAwsS3Bucket.id())
 *             .policy(s3Policy.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:s3:BucketPolicy
 *     properties:
 *       bucket: ${exampleAwsS3Bucket.id}
 *       policy: ${s3Policy.json}
 * variables:
 *   s3Policy:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - actions:
 *               - s3:GetObject
 *             resources:
 *               - ${exampleAwsS3Bucket.arn}/*
 *             principals:
 *               - type: AWS
 *                 identifiers:
 *                   - ${exampleAwsCloudfrontOriginAccessIdentity.iamArn}
 * ```
 * <!--End PulumiCodeChooser -->
 * [1]: http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Introduction.html
 * [2]: http://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 * ## Import
 * Using `pulumi import`, import Cloudfront Origin Access Identities using the `id`. For example:
 * ```sh
 * $ pulumi import aws:cloudfront/originAccessIdentity:OriginAccessIdentity origin_access E74FTE3AEXAMPLE
 * ```
 * @property comment An optional comment for the origin access identity.
 * */*/*/*/*/*/
 */
public data class OriginAccessIdentityArgs(
    public val comment: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.cloudfront.OriginAccessIdentityArgs> {
    override fun toJava(): com.pulumi.aws.cloudfront.OriginAccessIdentityArgs =
        com.pulumi.aws.cloudfront.OriginAccessIdentityArgs.builder()
            .comment(comment?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [OriginAccessIdentityArgs].
 */
@PulumiTagMarker
public class OriginAccessIdentityArgsBuilder internal constructor() {
    private var comment: Output<String>? = null

    /**
     * @param value An optional comment for the origin access identity.
     */
    @JvmName("kvrcakswahgfgldu")
    public suspend fun comment(`value`: Output<String>) {
        this.comment = value
    }

    /**
     * @param value An optional comment for the origin access identity.
     */
    @JvmName("qfckxruivcefplva")
    public suspend fun comment(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.comment = mapped
    }

    internal fun build(): OriginAccessIdentityArgs = OriginAccessIdentityArgs(
        comment = comment,
    )
}
