@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cloudtrail.kotlin

import com.pulumi.aws.cloudtrail.CloudtrailFunctions.getServiceAccountPlain
import com.pulumi.aws.cloudtrail.kotlin.inputs.GetServiceAccountPlainArgs
import com.pulumi.aws.cloudtrail.kotlin.inputs.GetServiceAccountPlainArgsBuilder
import com.pulumi.aws.cloudtrail.kotlin.outputs.GetServiceAccountResult
import com.pulumi.aws.cloudtrail.kotlin.outputs.GetServiceAccountResult.Companion.toKotlin
import kotlinx.coroutines.future.await
import kotlin.String
import kotlin.Suppress
import kotlin.Unit

public object CloudtrailFunctions {
    /**
     * Use this data source to get the Account ID of the [AWS CloudTrail Service Account](http://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-supported-regions.html)
     * in a given region for the purpose of allowing CloudTrail to store trail data in S3.
     * > **Note:** AWS documentation [states that](https://docs&#46;aws&#46;amazon&#46;com/awscloudtrail/latest/userguide/create-s3-bucket-policy-for-cloudtrail&#46;html#troubleshooting-s3-bucket-policy) a [service principal name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services) should be used instead of an AWS account ID in any relevant IAM policy.
     * ## Example Usage
     * <!--Start PulumiCodeChooser -->
     * ```typescript
     * import * as pulumi from "@pulumi/pulumi";
     * import * as aws from "@pulumi/aws";
     * const main = aws.cloudtrail.getServiceAccount({});
     * const bucket = new aws.s3.BucketV2("bucket", {
     *     bucket: "tf-cloudtrail-logging-test-bucket",
     *     forceDestroy: true,
     * });
     * const allowCloudtrailLogging = pulumi.all([main, bucket&#46;arn, main, bucket&#46;arn])&#46;apply(([main, bucketArn, main1, bucketArn1]) => aws.iam.getPolicyDocumentOutput({
     *     statements: [
     *         {
     *             sid: "Put bucket policy needed for trails",
     *             effect: "Allow",
     *             principals: [{
     *                 type: "AWS",
     *                 identifiers: [main&#46;arn],
     *             }],
     *             actions: ["s3:PutObject"],
     *             resources: [`${bucketArn}/*`],
     *         },
     *         {
     *             sid: "Get bucket policy needed for trails",
     *             effect: "Allow",
     *             principals: [{
     *                 type: "AWS",
     *                 identifiers: [main1&#46;arn],
     *             }],
     *             actions: ["s3:GetBucketAcl"],
     *             resources: [bucketArn1],
     *         },
     *     ],
     * }));
     * const allowCloudtrailLoggingBucketPolicy = new aws.s3.BucketPolicy("allow_cloudtrail_logging", {
     *     bucket: bucket.id,
     *     policy: allowCloudtrailLogging.apply(allowCloudtrailLogging => allowCloudtrailLogging.json),
     * });
     * ```
     * ```python
     * import pulumi
     * import pulumi_aws as aws
     * main = aws.cloudtrail.get_service_account()
     * bucket = aws.s3.BucketV2("bucket",
     *     bucket="tf-cloudtrail-logging-test-bucket",
     *     force_destroy=True)
     * allow_cloudtrail_logging = pulumi.Output.all(bucket.arn, bucket.arn).apply(lambda bucketArn, bucketArn1: aws.iam.get_policy_document_output(statements=[
     *     {
     *         "sid": "Put bucket policy needed for trails",
     *         "effect": "Allow",
     *         "principals": [{
     *             "type": "AWS",
     *             "identifiers": [main&#46;arn],
     *         }],
     *         "actions": ["s3:PutObject"],
     *         "resources": [f"{bucket_arn}/*"],
     *     },
     *     {
     *         "sid": "Get bucket policy needed for trails",
     *         "effect": "Allow",
     *         "principals": [{
     *             "type": "AWS",
     *             "identifiers": [main&#46;arn],
     *         }],
     *         "actions": ["s3:GetBucketAcl"],
     *         "resources": [bucket_arn1],
     *     },
     * ]))
     * allow_cloudtrail_logging_bucket_policy = aws.s3.BucketPolicy("allow_cloudtrail_logging",
     *     bucket=bucket.id,
     *     policy=allow_cloudtrail_logging.json)
     * ```
     * ```csharp
     * using System.Collections.Generic;
     * using System.Linq;
     * using Pulumi;
     * using Aws = Pulumi.Aws;
     * return await Deployment.RunAsync(() =>
     * {
     *     var main = Aws.CloudTrail.GetServiceAccount.Invoke();
     *     var bucket = new Aws.S3.BucketV2("bucket", new()
     *     {
     *         Bucket = "tf-cloudtrail-logging-test-bucket",
     *         ForceDestroy = true,
     *     });
     *     var allowCloudtrailLogging = Aws.Iam.GetPolicyDocument.Invoke(new()
     *     {
     *         Statements = new[]
     *         {
     *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
     *             {
     *                 Sid = "Put bucket policy needed for trails",
     *                 Effect = "Allow",
     *                 Principals = new[]
     *                 {
     *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
     *                     {
     *                         Type = "AWS",
     *                         Identifiers = new[]
     *                         {
     *                             main.Apply(getServiceAccountResult => getServiceAccountResult.Arn),
     *                         },
     *                     },
     *                 },
     *                 Actions = new[]
     *                 {
     *                     "s3:PutObject",
     *                 },
     *                 Resources = new[]
     *                 {
     *                     $"{bucket.Arn}/*",
     *                 },
     *             },
     *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
     *             {
     *                 Sid = "Get bucket policy needed for trails",
     *                 Effect = "Allow",
     *                 Principals = new[]
     *                 {
     *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
     *                     {
     *                         Type = "AWS",
     *                         Identifiers = new[]
     *                         {
     *                             main.Apply(getServiceAccountResult => getServiceAccountResult.Arn),
     *                         },
     *                     },
     *                 },
     *                 Actions = new[]
     *                 {
     *                     "s3:GetBucketAcl",
     *                 },
     *                 Resources = new[]
     *                 {
     *                     bucket.Arn,
     *                 },
     *             },
     *         },
     *     });
     *     var allowCloudtrailLoggingBucketPolicy = new Aws.S3.BucketPolicy("allow_cloudtrail_logging", new()
     *     {
     *         Bucket = bucket.Id,
     *         Policy = allowCloudtrailLogging.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
     *     });
     * });
     * ```
     * ```go
     * package main
     * import (
     * 	"fmt"
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudtrail"
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
     * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
     * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
     * )
     * func main() {
     * pulumi.Run(func(ctx *pulumi.Context) error {
     * main, err := cloudtrail.GetServiceAccount(ctx, nil, nil);
     * if err != nil {
     * return err
     * }
     * bucket, err := s3.NewBucketV2(ctx, "bucket", &s3.BucketV2Args{
     * Bucket: pulumi.String("tf-cloudtrail-logging-test-bucket"),
     * ForceDestroy: pulumi.Bool(true),
     * })
     * if err != nil {
     * return err
     * }
     * allowCloudtrailLogging := pulumi.All(bucket.Arn,bucket.Arn).ApplyT(func(_args []interface{}) (iam.GetPolicyDocumentResult, error) {
     * bucketArn := _args[0].(string)
     * bucketArn1 := _args[1].(string)
     * return iam.GetPolicyDocumentResult(interface{}(iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
     * Statements: []iam.GetPolicyDocumentStatement{
     * {
     * Sid: "Put bucket policy needed for trails",
     * Effect: "Allow",
     * Principals: []iam.GetPolicyDocumentStatementPrincipal{
     * {
     * Type: "AWS",
     * Identifiers: interface{}{
     * main.Arn,
     * },
     * },
     * },
     * Actions: []string{
     * "s3:PutObject",
     * },
     * Resources: []string{
     * fmt.Sprintf("%v/*", bucketArn),
     * },
     * },
     * {
     * Sid: "Get bucket policy needed for trails",
     * Effect: "Allow",
     * Principals: []iam.GetPolicyDocumentStatementPrincipal{
     * {
     * Type: "AWS",
     * Identifiers: interface{}{
     * main.Arn,
     * },
     * },
     * },
     * Actions: []string{
     * "s3:GetBucketAcl",
     * },
     * Resources: []string{
     * bucketArn1,
     * },
     * },
     * },
     * }, nil))), nil
     * }).(iam.GetPolicyDocumentResultOutput)
     * _, err = s3.NewBucketPolicy(ctx, "allow_cloudtrail_logging", &s3.BucketPolicyArgs{
     * Bucket: bucket.ID(),
     * Policy: pulumi.String(allowCloudtrailLogging.ApplyT(func(allowCloudtrailLogging iam.GetPolicyDocumentResult) (*string, error) {
     * return &allowCloudtrailLogging.Json, nil
     * }).(pulumi.StringPtrOutput)),
     * })
     * if err != nil {
     * return err
     * }
     * return nil
     * })
     * }
     * ```
     * ```java
     * package generated_program;
     * import com.pulumi.Context;
     * import com.pulumi.Pulumi;
     * import com.pulumi.core.Output;
     * import com.pulumi.aws.cloudtrail.CloudtrailFunctions;
     * import com.pulumi.aws.cloudtrail.inputs.GetServiceAccountArgs;
     * import com.pulumi.aws.s3.BucketV2;
     * import com.pulumi.aws.s3.BucketV2Args;
     * import com.pulumi.aws.iam.IamFunctions;
     * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
     * import com.pulumi.aws.s3.BucketPolicy;
     * import com.pulumi.aws.s3.BucketPolicyArgs;
     * import java.util.List;
     * import java.util.ArrayList;
     * import java.util.Map;
     * import java.io.File;
     * import java.nio.file.Files;
     * import java.nio.file.Paths;
     * public class App {
     *     public static void main(String[] args) {
     *         Pulumi.run(App::stack);
     *     }
     *     public static void stack(Context ctx) {
     *         final var main = CloudtrailFunctions.getServiceAccount();
     *         var bucket = new BucketV2("bucket", BucketV2Args.builder()
     *             .bucket("tf-cloudtrail-logging-test-bucket")
     *             .forceDestroy(true)
     *             .build());
     *         final var allowCloudtrailLogging = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
     *             .statements(
     *                 GetPolicyDocumentStatementArgs.builder()
     *                     .sid("Put bucket policy needed for trails")
     *                     .effect("Allow")
     *                     .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
     *                         .type("AWS")
     *                         .identifiers(main.applyValue(getServiceAccountResult -> getServiceAccountResult.arn()))
     *                         .build())
     *                     .actions("s3:PutObject")
     *                     .resources(bucket.arn().applyValue(arn -> String.format("%s/*", arn)))
     *                     .build(),
     *                 GetPolicyDocumentStatementArgs.builder()
     *                     .sid("Get bucket policy needed for trails")
     *                     .effect("Allow")
     *                     .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
     *                         .type("AWS")
     *                         .identifiers(main.applyValue(getServiceAccountResult -> getServiceAccountResult.arn()))
     *                         .build())
     *                     .actions("s3:GetBucketAcl")
     *                     .resources(bucket.arn())
     *                     .build())
     *             .build());
     *         var allowCloudtrailLoggingBucketPolicy = new BucketPolicy("allowCloudtrailLoggingBucketPolicy", BucketPolicyArgs.builder()
     *             .bucket(bucket.id())
     *             .policy(allowCloudtrailLogging.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(allowCloudtrailLogging -> allowCloudtrailLogging.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
     *             .build());
     *     }
     * }
     * ```
     * ```yaml
     * resources:
     *   bucket:
     *     type: aws:s3:BucketV2
     *     properties:
     *       bucket: tf-cloudtrail-logging-test-bucket
     *       forceDestroy: true
     *   allowCloudtrailLoggingBucketPolicy:
     *     type: aws:s3:BucketPolicy
     *     name: allow_cloudtrail_logging
     *     properties:
     *       bucket: ${bucket.id}
     *       policy: ${allowCloudtrailLogging.json}
     * variables:
     *   main:
     *     fn::invoke:
     *       Function: aws:cloudtrail:getServiceAccount
     *       Arguments: {}
     *   allowCloudtrailLogging:
     *     fn::invoke:
     *       Function: aws:iam:getPolicyDocument
     *       Arguments:
     *         statements:
     *           - sid: Put bucket policy needed for trails
     *             effect: Allow
     *             principals:
     *               - type: AWS
     *                 identifiers:
     *                   - ${main.arn}
     *             actions:
     *               - s3:PutObject
     *             resources:
     *               - ${bucket.arn}/*
     *           - sid: Get bucket policy needed for trails
     *             effect: Allow
     *             principals:
     *               - type: AWS
     *                 identifiers:
     *                   - ${main.arn}
     *             actions:
     *               - s3:GetBucketAcl
     *             resources:
     *               - ${bucket.arn}
     * ```
     * <!--End PulumiCodeChooser -->
     * @param argument A collection of arguments for invoking getServiceAccount.
     * @return A collection of values returned by getServiceAccount.
     * */*/*/*/*/*/
     */
    public suspend fun getServiceAccount(argument: GetServiceAccountPlainArgs): GetServiceAccountResult = toKotlin(getServiceAccountPlain(argument.toJava()).await())

    /**
     * @see [getServiceAccount].
     * @param region Name of the region whose AWS CloudTrail account ID is desired.
     * Defaults to the region from the AWS provider configuration.
     * @return A collection of values returned by getServiceAccount.
     */
    public suspend fun getServiceAccount(region: String? = null): GetServiceAccountResult {
        val argument = GetServiceAccountPlainArgs(
            region = region,
        )
        return toKotlin(getServiceAccountPlain(argument.toJava()).await())
    }

    /**
     * @see [getServiceAccount].
     * @param argument Builder for [com&#46;pulumi&#46;aws&#46;cloudtrail&#46;kotlin&#46;inputs&#46;GetServiceAccountPlainArgs].
     * @return A collection of values returned by getServiceAccount.
     */
    public suspend fun getServiceAccount(argument: suspend GetServiceAccountPlainArgsBuilder.() -> Unit): GetServiceAccountResult {
        val builder = GetServiceAccountPlainArgsBuilder()
        builder.argument()
        val builtArgument = builder.build()
        return toKotlin(getServiceAccountPlain(builtArgument.toJava()).await())
    }
}
