@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cloudwatch.kotlin

import com.pulumi.aws.cloudwatch.MetricAlarmArgs.builder
import com.pulumi.aws.cloudwatch.kotlin.inputs.MetricAlarmMetricQueryArgs
import com.pulumi.aws.cloudwatch.kotlin.inputs.MetricAlarmMetricQueryArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a CloudWatch Metric Alarm resource.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const foobar = new aws.cloudwatch.MetricAlarm("foobar", {
 *     name: "test-foobar5",
 *     comparisonOperator: "GreaterThanOrEqualToThreshold",
 *     evaluationPeriods: 2,
 *     metricName: "CPUUtilization",
 *     namespace: "AWS/EC2",
 *     period: 120,
 *     statistic: "Average",
 *     threshold: 80,
 *     alarmDescription: "This metric monitors ec2 cpu utilization",
 *     insufficientDataActions: [],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foobar = aws.cloudwatch.MetricAlarm("foobar",
 *     name="test-foobar5",
 *     comparison_operator="GreaterThanOrEqualToThreshold",
 *     evaluation_periods=2,
 *     metric_name="CPUUtilization",
 *     namespace="AWS/EC2",
 *     period=120,
 *     statistic="Average",
 *     threshold=80,
 *     alarm_description="This metric monitors ec2 cpu utilization",
 *     insufficient_data_actions=[])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var foobar = new Aws.CloudWatch.MetricAlarm("foobar", new()
 *     {
 *         Name = "test-foobar5",
 *         ComparisonOperator = "GreaterThanOrEqualToThreshold",
 *         EvaluationPeriods = 2,
 *         MetricName = "CPUUtilization",
 *         Namespace = "AWS/EC2",
 *         Period = 120,
 *         Statistic = "Average",
 *         Threshold = 80,
 *         AlarmDescription = "This metric monitors ec2 cpu utilization",
 *         InsufficientDataActions = new[] {},
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudwatch.NewMetricAlarm(ctx, "foobar", &cloudwatch.MetricAlarmArgs{
 * 			Name:                    pulumi.String("test-foobar5"),
 * 			ComparisonOperator:      pulumi.String("GreaterThanOrEqualToThreshold"),
 * 			EvaluationPeriods:       pulumi.Int(2),
 * 			MetricName:              pulumi.String("CPUUtilization"),
 * 			Namespace:               pulumi.String("AWS/EC2"),
 * 			Period:                  pulumi.Int(120),
 * 			Statistic:               pulumi.String("Average"),
 * 			Threshold:               pulumi.Float64(80),
 * 			AlarmDescription:        pulumi.String("This metric monitors ec2 cpu utilization"),
 * 			InsufficientDataActions: pulumi.Array{},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.MetricAlarm;
 * import com.pulumi.aws.cloudwatch.MetricAlarmArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var foobar = new MetricAlarm("foobar", MetricAlarmArgs.builder()
 *             .name("test-foobar5")
 *             .comparisonOperator("GreaterThanOrEqualToThreshold")
 *             .evaluationPeriods(2)
 *             .metricName("CPUUtilization")
 *             .namespace("AWS/EC2")
 *             .period(120)
 *             .statistic("Average")
 *             .threshold(80)
 *             .alarmDescription("This metric monitors ec2 cpu utilization")
 *             .insufficientDataActions()
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foobar:
 *     type: aws:cloudwatch:MetricAlarm
 *     properties:
 *       name: test-foobar5
 *       comparisonOperator: GreaterThanOrEqualToThreshold
 *       evaluationPeriods: 2
 *       metricName: CPUUtilization
 *       namespace: AWS/EC2
 *       period: 120
 *       statistic: Average
 *       threshold: 80
 *       alarmDescription: This metric monitors ec2 cpu utilization
 *       insufficientDataActions: []
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Example in Conjunction with Scaling Policies
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const bat = new aws.autoscaling.Policy("bat", {
 *     name: "foobar3-test",
 *     scalingAdjustment: 4,
 *     adjustmentType: "ChangeInCapacity",
 *     cooldown: 300,
 *     autoscalingGroupName: bar.name,
 * });
 * const batMetricAlarm = new aws.cloudwatch.MetricAlarm("bat", {
 *     name: "test-foobar5",
 *     comparisonOperator: "GreaterThanOrEqualToThreshold",
 *     evaluationPeriods: 2,
 *     metricName: "CPUUtilization",
 *     namespace: "AWS/EC2",
 *     period: 120,
 *     statistic: "Average",
 *     threshold: 80,
 *     dimensions: {
 *         AutoScalingGroupName: bar.name,
 *     },
 *     alarmDescription: "This metric monitors ec2 cpu utilization",
 *     alarmActions: [bat&#46;arn],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * bat = aws.autoscaling.Policy("bat",
 *     name="foobar3-test",
 *     scaling_adjustment=4,
 *     adjustment_type="ChangeInCapacity",
 *     cooldown=300,
 *     autoscaling_group_name=bar["name"])
 * bat_metric_alarm = aws.cloudwatch.MetricAlarm("bat",
 *     name="test-foobar5",
 *     comparison_operator="GreaterThanOrEqualToThreshold",
 *     evaluation_periods=2,
 *     metric_name="CPUUtilization",
 *     namespace="AWS/EC2",
 *     period=120,
 *     statistic="Average",
 *     threshold=80,
 *     dimensions={
 *         "AutoScalingGroupName": bar["name"],
 *     },
 *     alarm_description="This metric monitors ec2 cpu utilization",
 *     alarm_actions=[bat&#46;arn])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var bat = new Aws.AutoScaling.Policy("bat", new()
 *     {
 *         Name = "foobar3-test",
 *         ScalingAdjustment = 4,
 *         AdjustmentType = "ChangeInCapacity",
 *         Cooldown = 300,
 *         AutoscalingGroupName = bar.Name,
 *     });
 *     var batMetricAlarm = new Aws.CloudWatch.MetricAlarm("bat", new()
 *     {
 *         Name = "test-foobar5",
 *         ComparisonOperator = "GreaterThanOrEqualToThreshold",
 *         EvaluationPeriods = 2,
 *         MetricName = "CPUUtilization",
 *         Namespace = "AWS/EC2",
 *         Period = 120,
 *         Statistic = "Average",
 *         Threshold = 80,
 *         Dimensions =
 *         {
 *             { "AutoScalingGroupName", bar.Name },
 *         },
 *         AlarmDescription = "This metric monitors ec2 cpu utilization",
 *         AlarmActions = new[]
 *         {
 *             bat.Arn,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/autoscaling"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		bat, err := autoscaling.NewPolicy(ctx, "bat", &autoscaling.PolicyArgs{
 * 			Name:                 pulumi.String("foobar3-test"),
 * 			ScalingAdjustment:    pulumi.Int(4),
 * 			AdjustmentType:       pulumi.String("ChangeInCapacity"),
 * 			Cooldown:             pulumi.Int(300),
 * 			AutoscalingGroupName: pulumi.Any(bar.Name),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cloudwatch.NewMetricAlarm(ctx, "bat", &cloudwatch.MetricAlarmArgs{
 * 			Name:               pulumi.String("test-foobar5"),
 * 			ComparisonOperator: pulumi.String("GreaterThanOrEqualToThreshold"),
 * 			EvaluationPeriods:  pulumi.Int(2),
 * 			MetricName:         pulumi.String("CPUUtilization"),
 * 			Namespace:          pulumi.String("AWS/EC2"),
 * 			Period:             pulumi.Int(120),
 * 			Statistic:          pulumi.String("Average"),
 * 			Threshold:          pulumi.Float64(80),
 * 			Dimensions: pulumi.StringMap{
 * 				"AutoScalingGroupName": pulumi.Any(bar.Name),
 * 			},
 * 			AlarmDescription: pulumi.String("This metric monitors ec2 cpu utilization"),
 * 			AlarmActions: pulumi.Array{
 * 				bat.Arn,
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.autoscaling.Policy;
 * import com.pulumi.aws.autoscaling.PolicyArgs;
 * import com.pulumi.aws.cloudwatch.MetricAlarm;
 * import com.pulumi.aws.cloudwatch.MetricAlarmArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var bat = new Policy("bat", PolicyArgs.builder()
 *             .name("foobar3-test")
 *             .scalingAdjustment(4)
 *             .adjustmentType("ChangeInCapacity")
 *             .cooldown(300)
 *             .autoscalingGroupName(bar.name())
 *             .build());
 *         var batMetricAlarm = new MetricAlarm("batMetricAlarm", MetricAlarmArgs.builder()
 *             .name("test-foobar5")
 *             .comparisonOperator("GreaterThanOrEqualToThreshold")
 *             .evaluationPeriods(2)
 *             .metricName("CPUUtilization")
 *             .namespace("AWS/EC2")
 *             .period(120)
 *             .statistic("Average")
 *             .threshold(80)
 *             .dimensions(Map.of("AutoScalingGroupName", bar.name()))
 *             .alarmDescription("This metric monitors ec2 cpu utilization")
 *             .alarmActions(bat.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   bat:
 *     type: aws:autoscaling:Policy
 *     properties:
 *       name: foobar3-test
 *       scalingAdjustment: 4
 *       adjustmentType: ChangeInCapacity
 *       cooldown: 300
 *       autoscalingGroupName: ${bar.name}
 *   batMetricAlarm:
 *     type: aws:cloudwatch:MetricAlarm
 *     name: bat
 *     properties:
 *       name: test-foobar5
 *       comparisonOperator: GreaterThanOrEqualToThreshold
 *       evaluationPeriods: 2
 *       metricName: CPUUtilization
 *       namespace: AWS/EC2
 *       period: 120
 *       statistic: Average
 *       threshold: 80
 *       dimensions:
 *         AutoScalingGroupName: ${bar.name}
 *       alarmDescription: This metric monitors ec2 cpu utilization
 *       alarmActions:
 *         - ${bat.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Example with an Expression
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const foobar = new aws.cloudwatch.MetricAlarm("foobar", {
 *     name: "test-foobar",
 *     comparisonOperator: "GreaterThanOrEqualToThreshold",
 *     evaluationPeriods: 2,
 *     threshold: 10,
 *     alarmDescription: "Request error rate has exceeded 10%",
 *     insufficientDataActions: [],
 *     metricQueries: [
 *         {
 *             id: "e1",
 *             expression: "m2/m1*100",
 *             label: "Error Rate",
 *             returnData: true,
 *         },
 *         {
 *             id: "m1",
 *             metric: {
 *                 metricName: "RequestCount",
 *                 namespace: "AWS/ApplicationELB",
 *                 period: 120,
 *                 stat: "Sum",
 *                 unit: "Count",
 *                 dimensions: {
 *                     LoadBalancer: "app/web",
 *                 },
 *             },
 *         },
 *         {
 *             id: "m2",
 *             metric: {
 *                 metricName: "HTTPCode_ELB_5XX_Count",
 *                 namespace: "AWS/ApplicationELB",
 *                 period: 120,
 *                 stat: "Sum",
 *                 unit: "Count",
 *                 dimensions: {
 *                     LoadBalancer: "app/web",
 *                 },
 *             },
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foobar = aws.cloudwatch.MetricAlarm("foobar",
 *     name="test-foobar",
 *     comparison_operator="GreaterThanOrEqualToThreshold",
 *     evaluation_periods=2,
 *     threshold=10,
 *     alarm_description="Request error rate has exceeded 10%",
 *     insufficient_data_actions=[],
 *     metric_queries=[
 *         {
 *             "id": "e1",
 *             "expression": "m2/m1*100",
 *             "label": "Error Rate",
 *             "return_data": True,
 *         },
 *         {
 *             "id": "m1",
 *             "metric": {
 *                 "metric_name": "RequestCount",
 *                 "namespace": "AWS/ApplicationELB",
 *                 "period": 120,
 *                 "stat": "Sum",
 *                 "unit": "Count",
 *                 "dimensions": {
 *                     "load_balancer": "app/web",
 *                 },
 *             },
 *         },
 *         {
 *             "id": "m2",
 *             "metric": {
 *                 "metric_name": "HTTPCode_ELB_5XX_Count",
 *                 "namespace": "AWS/ApplicationELB",
 *                 "period": 120,
 *                 "stat": "Sum",
 *                 "unit": "Count",
 *                 "dimensions": {
 *                     "load_balancer": "app/web",
 *                 },
 *             },
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var foobar = new Aws.CloudWatch.MetricAlarm("foobar", new()
 *     {
 *         Name = "test-foobar",
 *         ComparisonOperator = "GreaterThanOrEqualToThreshold",
 *         EvaluationPeriods = 2,
 *         Threshold = 10,
 *         AlarmDescription = "Request error rate has exceeded 10%",
 *         InsufficientDataActions = new[] {},
 *         MetricQueries = new[]
 *         {
 *             new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryArgs
 *             {
 *                 Id = "e1",
 *                 Expression = "m2/m1*100",
 *                 Label = "Error Rate",
 *                 ReturnData = true,
 *             },
 *             new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryArgs
 *             {
 *                 Id = "m1",
 *                 Metric = new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryMetricArgs
 *                 {
 *                     MetricName = "RequestCount",
 *                     Namespace = "AWS/ApplicationELB",
 *                     Period = 120,
 *                     Stat = "Sum",
 *                     Unit = "Count",
 *                     Dimensions =
 *                     {
 *                         { "LoadBalancer", "app/web" },
 *                     },
 *                 },
 *             },
 *             new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryArgs
 *             {
 *                 Id = "m2",
 *                 Metric = new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryMetricArgs
 *                 {
 *                     MetricName = "HTTPCode_ELB_5XX_Count",
 *                     Namespace = "AWS/ApplicationELB",
 *                     Period = 120,
 *                     Stat = "Sum",
 *                     Unit = "Count",
 *                     Dimensions =
 *                     {
 *                         { "LoadBalancer", "app/web" },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudwatch.NewMetricAlarm(ctx, "foobar", &cloudwatch.MetricAlarmArgs{
 * 			Name:                    pulumi.String("test-foobar"),
 * 			ComparisonOperator:      pulumi.String("GreaterThanOrEqualToThreshold"),
 * 			EvaluationPeriods:       pulumi.Int(2),
 * 			Threshold:               pulumi.Float64(10),
 * 			AlarmDescription:        pulumi.String("Request error rate has exceeded 10%"),
 * 			InsufficientDataActions: pulumi.Array{},
 * 			MetricQueries: cloudwatch.MetricAlarmMetricQueryArray{
 * 				&cloudwatch.MetricAlarmMetricQueryArgs{
 * 					Id:         pulumi.String("e1"),
 * 					Expression: pulumi.String("m2/m1*100"),
 * 					Label:      pulumi.String("Error Rate"),
 * 					ReturnData: pulumi.Bool(true),
 * 				},
 * 				&cloudwatch.MetricAlarmMetricQueryArgs{
 * 					Id: pulumi.String("m1"),
 * 					Metric: &cloudwatch.MetricAlarmMetricQueryMetricArgs{
 * 						MetricName: pulumi.String("RequestCount"),
 * 						Namespace:  pulumi.String("AWS/ApplicationELB"),
 * 						Period:     pulumi.Int(120),
 * 						Stat:       pulumi.String("Sum"),
 * 						Unit:       pulumi.String("Count"),
 * 						Dimensions: pulumi.StringMap{
 * 							"LoadBalancer": pulumi.String("app/web"),
 * 						},
 * 					},
 * 				},
 * 				&cloudwatch.MetricAlarmMetricQueryArgs{
 * 					Id: pulumi.String("m2"),
 * 					Metric: &cloudwatch.MetricAlarmMetricQueryMetricArgs{
 * 						MetricName: pulumi.String("HTTPCode_ELB_5XX_Count"),
 * 						Namespace:  pulumi.String("AWS/ApplicationELB"),
 * 						Period:     pulumi.Int(120),
 * 						Stat:       pulumi.String("Sum"),
 * 						Unit:       pulumi.String("Count"),
 * 						Dimensions: pulumi.StringMap{
 * 							"LoadBalancer": pulumi.String("app/web"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.MetricAlarm;
 * import com.pulumi.aws.cloudwatch.MetricAlarmArgs;
 * import com.pulumi.aws.cloudwatch.inputs.MetricAlarmMetricQueryArgs;
 * import com.pulumi.aws.cloudwatch.inputs.MetricAlarmMetricQueryMetricArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var foobar = new MetricAlarm("foobar", MetricAlarmArgs.builder()
 *             .name("test-foobar")
 *             .comparisonOperator("GreaterThanOrEqualToThreshold")
 *             .evaluationPeriods(2)
 *             .threshold(10)
 *             .alarmDescription("Request error rate has exceeded 10%")
 *             .insufficientDataActions()
 *             .metricQueries(
 *                 MetricAlarmMetricQueryArgs.builder()
 *                     .id("e1")
 *                     .expression("m2/m1*100")
 *                     .label("Error Rate")
 *                     .returnData("true")
 *                     .build(),
 *                 MetricAlarmMetricQueryArgs.builder()
 *                     .id("m1")
 *                     .metric(MetricAlarmMetricQueryMetricArgs.builder()
 *                         .metricName("RequestCount")
 *                         .namespace("AWS/ApplicationELB")
 *                         .period(120)
 *                         .stat("Sum")
 *                         .unit("Count")
 *                         .dimensions(Map.of("LoadBalancer", "app/web"))
 *                         .build())
 *                     .build(),
 *                 MetricAlarmMetricQueryArgs.builder()
 *                     .id("m2")
 *                     .metric(MetricAlarmMetricQueryMetricArgs.builder()
 *                         .metricName("HTTPCode_ELB_5XX_Count")
 *                         .namespace("AWS/ApplicationELB")
 *                         .period(120)
 *                         .stat("Sum")
 *                         .unit("Count")
 *                         .dimensions(Map.of("LoadBalancer", "app/web"))
 *                         .build())
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foobar:
 *     type: aws:cloudwatch:MetricAlarm
 *     properties:
 *       name: test-foobar
 *       comparisonOperator: GreaterThanOrEqualToThreshold
 *       evaluationPeriods: 2
 *       threshold: 10
 *       alarmDescription: Request error rate has exceeded 10%
 *       insufficientDataActions: []
 *       metricQueries:
 *         - id: e1
 *           expression: m2/m1*100
 *           label: Error Rate
 *           returnData: 'true'
 *         - id: m1
 *           metric:
 *             metricName: RequestCount
 *             namespace: AWS/ApplicationELB
 *             period: 120
 *             stat: Sum
 *             unit: Count
 *             dimensions:
 *               LoadBalancer: app/web
 *         - id: m2
 *           metric:
 *             metricName: HTTPCode_ELB_5XX_Count
 *             namespace: AWS/ApplicationELB
 *             period: 120
 *             stat: Sum
 *             unit: Count
 *             dimensions:
 *               LoadBalancer: app/web
 * ```
 * <!--End PulumiCodeChooser -->
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const xxAnomalyDetection = new aws.cloudwatch.MetricAlarm("xx_anomaly_detection", {
 *     name: "test-foobar",
 *     comparisonOperator: "GreaterThanUpperThreshold",
 *     evaluationPeriods: 2,
 *     thresholdMetricId: "e1",
 *     alarmDescription: "This metric monitors ec2 cpu utilization",
 *     insufficientDataActions: [],
 *     metricQueries: [
 *         {
 *             id: "e1",
 *             expression: "ANOMALY_DETECTION_BAND(m1)",
 *             label: "CPUUtilization (Expected)",
 *             returnData: true,
 *         },
 *         {
 *             id: "m1",
 *             returnData: true,
 *             metric: {
 *                 metricName: "CPUUtilization",
 *                 namespace: "AWS/EC2",
 *                 period: 120,
 *                 stat: "Average",
 *                 unit: "Count",
 *                 dimensions: {
 *                     InstanceId: "i-abc123",
 *                 },
 *             },
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * xx_anomaly_detection = aws.cloudwatch.MetricAlarm("xx_anomaly_detection",
 *     name="test-foobar",
 *     comparison_operator="GreaterThanUpperThreshold",
 *     evaluation_periods=2,
 *     threshold_metric_id="e1",
 *     alarm_description="This metric monitors ec2 cpu utilization",
 *     insufficient_data_actions=[],
 *     metric_queries=[
 *         {
 *             "id": "e1",
 *             "expression": "ANOMALY_DETECTION_BAND(m1)",
 *             "label": "CPUUtilization (Expected)",
 *             "return_data": True,
 *         },
 *         {
 *             "id": "m1",
 *             "return_data": True,
 *             "metric": {
 *                 "metric_name": "CPUUtilization",
 *                 "namespace": "AWS/EC2",
 *                 "period": 120,
 *                 "stat": "Average",
 *                 "unit": "Count",
 *                 "dimensions": {
 *                     "instance_id": "i-abc123",
 *                 },
 *             },
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var xxAnomalyDetection = new Aws.CloudWatch.MetricAlarm("xx_anomaly_detection", new()
 *     {
 *         Name = "test-foobar",
 *         ComparisonOperator = "GreaterThanUpperThreshold",
 *         EvaluationPeriods = 2,
 *         ThresholdMetricId = "e1",
 *         AlarmDescription = "This metric monitors ec2 cpu utilization",
 *         InsufficientDataActions = new[] {},
 *         MetricQueries = new[]
 *         {
 *             new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryArgs
 *             {
 *                 Id = "e1",
 *                 Expression = "ANOMALY_DETECTION_BAND(m1)",
 *                 Label = "CPUUtilization (Expected)",
 *                 ReturnData = true,
 *             },
 *             new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryArgs
 *             {
 *                 Id = "m1",
 *                 ReturnData = true,
 *                 Metric = new Aws.CloudWatch.Inputs.MetricAlarmMetricQueryMetricArgs
 *                 {
 *                     MetricName = "CPUUtilization",
 *                     Namespace = "AWS/EC2",
 *                     Period = 120,
 *                     Stat = "Average",
 *                     Unit = "Count",
 *                     Dimensions =
 *                     {
 *                         { "InstanceId", "i-abc123" },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudwatch.NewMetricAlarm(ctx, "xx_anomaly_detection", &cloudwatch.MetricAlarmArgs{
 * 			Name:                    pulumi.String("test-foobar"),
 * 			ComparisonOperator:      pulumi.String("GreaterThanUpperThreshold"),
 * 			EvaluationPeriods:       pulumi.Int(2),
 * 			ThresholdMetricId:       pulumi.String("e1"),
 * 			AlarmDescription:        pulumi.String("This metric monitors ec2 cpu utilization"),
 * 			InsufficientDataActions: pulumi.Array{},
 * 			MetricQueries: cloudwatch.MetricAlarmMetricQueryArray{
 * 				&cloudwatch.MetricAlarmMetricQueryArgs{
 * 					Id:         pulumi.String("e1"),
 * 					Expression: pulumi.String("ANOMALY_DETECTION_BAND(m1)"),
 * 					Label:      pulumi.String("CPUUtilization (Expected)"),
 * 					ReturnData: pulumi.Bool(true),
 * 				},
 * 				&cloudwatch.MetricAlarmMetricQueryArgs{
 * 					Id:         pulumi.String("m1"),
 * 					ReturnData: pulumi.Bool(true),
 * 					Metric: &cloudwatch.MetricAlarmMetricQueryMetricArgs{
 * 						MetricName: pulumi.String("CPUUtilization"),
 * 						Namespace:  pulumi.String("AWS/EC2"),
 * 						Period:     pulumi.Int(120),
 * 						Stat:       pulumi.String("Average"),
 * 						Unit:       pulumi.String("Count"),
 * 						Dimensions: pulumi.StringMap{
 * 							"InstanceId": pulumi.String("i-abc123"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.MetricAlarm;
 * import com.pulumi.aws.cloudwatch.MetricAlarmArgs;
 * import com.pulumi.aws.cloudwatch.inputs.MetricAlarmMetricQueryArgs;
 * import com.pulumi.aws.cloudwatch.inputs.MetricAlarmMetricQueryMetricArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var xxAnomalyDetection = new MetricAlarm("xxAnomalyDetection", MetricAlarmArgs.builder()
 *             .name("test-foobar")
 *             .comparisonOperator("GreaterThanUpperThreshold")
 *             .evaluationPeriods(2)
 *             .thresholdMetricId("e1")
 *             .alarmDescription("This metric monitors ec2 cpu utilization")
 *             .insufficientDataActions()
 *             .metricQueries(
 *                 MetricAlarmMetricQueryArgs.builder()
 *                     .id("e1")
 *                     .expression("ANOMALY_DETECTION_BAND(m1)")
 *                     .label("CPUUtilization (Expected)")
 *                     .returnData("true")
 *                     .build(),
 *                 MetricAlarmMetricQueryArgs.builder()
 *                     .id("m1")
 *                     .returnData("true")
 *                     .metric(MetricAlarmMetricQueryMetricArgs.builder()
 *                         .metricName("CPUUtilization")
 *                         .namespace("AWS/EC2")
 *                         .period(120)
 *                         .stat("Average")
 *                         .unit("Count")
 *                         .dimensions(Map.of("InstanceId", "i-abc123"))
 *                         .build())
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   xxAnomalyDetection:
 *     type: aws:cloudwatch:MetricAlarm
 *     name: xx_anomaly_detection
 *     properties:
 *       name: test-foobar
 *       comparisonOperator: GreaterThanUpperThreshold
 *       evaluationPeriods: 2
 *       thresholdMetricId: e1
 *       alarmDescription: This metric monitors ec2 cpu utilization
 *       insufficientDataActions: []
 *       metricQueries:
 *         - id: e1
 *           expression: ANOMALY_DETECTION_BAND(m1)
 *           label: CPUUtilization (Expected)
 *           returnData: 'true'
 *         - id: m1
 *           returnData: 'true'
 *           metric:
 *             metricName: CPUUtilization
 *             namespace: AWS/EC2
 *             period: 120
 *             stat: Average
 *             unit: Count
 *             dimensions:
 *               InstanceId: i-abc123
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Example of monitoring Healthy Hosts on NLB using Target Group and NLB
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const nlbHealthyhosts = new aws.cloudwatch.MetricAlarm("nlb_healthyhosts", {
 *     name: "alarmname",
 *     comparisonOperator: "LessThanThreshold",
 *     evaluationPeriods: 1,
 *     metricName: "HealthyHostCount",
 *     namespace: "AWS/NetworkELB",
 *     period: 60,
 *     statistic: "Average",
 *     threshold: logstashServersCount,
 *     alarmDescription: "Number of healthy nodes in Target Group",
 *     actionsEnabled: true,
 *     alarmActions: [sns&#46;arn],
 *     okActions: [sns&#46;arn],
 *     dimensions: {
 *         TargetGroup: lb_tg.arnSuffix,
 *         LoadBalancer: lb.arnSuffix,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * nlb_healthyhosts = aws.cloudwatch.MetricAlarm("nlb_healthyhosts",
 *     name="alarmname",
 *     comparison_operator="LessThanThreshold",
 *     evaluation_periods=1,
 *     metric_name="HealthyHostCount",
 *     namespace="AWS/NetworkELB",
 *     period=60,
 *     statistic="Average",
 *     threshold=logstash_servers_count,
 *     alarm_description="Number of healthy nodes in Target Group",
 *     actions_enabled=True,
 *     alarm_actions=[sns["arn"]],
 *     ok_actions=[sns["arn"]],
 *     dimensions={
 *         "TargetGroup": lb_tg["arnSuffix"],
 *         "LoadBalancer": lb["arnSuffix"],
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var nlbHealthyhosts = new Aws.CloudWatch.MetricAlarm("nlb_healthyhosts", new()
 *     {
 *         Name = "alarmname",
 *         ComparisonOperator = "LessThanThreshold",
 *         EvaluationPeriods = 1,
 *         MetricName = "HealthyHostCount",
 *         Namespace = "AWS/NetworkELB",
 *         Period = 60,
 *         Statistic = "Average",
 *         Threshold = logstashServersCount,
 *         AlarmDescription = "Number of healthy nodes in Target Group",
 *         ActionsEnabled = true,
 *         AlarmActions = new[]
 *         {
 *             sns.Arn,
 *         },
 *         OkActions = new[]
 *         {
 *             sns.Arn,
 *         },
 *         Dimensions =
 *         {
 *             { "TargetGroup", lb_tg.ArnSuffix },
 *             { "LoadBalancer", lb.ArnSuffix },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudwatch.NewMetricAlarm(ctx, "nlb_healthyhosts", &cloudwatch.MetricAlarmArgs{
 * 			Name:               pulumi.String("alarmname"),
 * 			ComparisonOperator: pulumi.String("LessThanThreshold"),
 * 			EvaluationPeriods:  pulumi.Int(1),
 * 			MetricName:         pulumi.String("HealthyHostCount"),
 * 			Namespace:          pulumi.String("AWS/NetworkELB"),
 * 			Period:             pulumi.Int(60),
 * 			Statistic:          pulumi.String("Average"),
 * 			Threshold:          pulumi.Any(logstashServersCount),
 * 			AlarmDescription:   pulumi.String("Number of healthy nodes in Target Group"),
 * 			ActionsEnabled:     pulumi.Bool(true),
 * 			AlarmActions: pulumi.Array{
 * 				sns.Arn,
 * 			},
 * 			OkActions: pulumi.Array{
 * 				sns.Arn,
 * 			},
 * 			Dimensions: pulumi.StringMap{
 * 				"TargetGroup":  pulumi.Any(lb_tg.ArnSuffix),
 * 				"LoadBalancer": pulumi.Any(lb.ArnSuffix),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.MetricAlarm;
 * import com.pulumi.aws.cloudwatch.MetricAlarmArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var nlbHealthyhosts = new MetricAlarm("nlbHealthyhosts", MetricAlarmArgs.builder()
 *             .name("alarmname")
 *             .comparisonOperator("LessThanThreshold")
 *             .evaluationPeriods(1)
 *             .metricName("HealthyHostCount")
 *             .namespace("AWS/NetworkELB")
 *             .period(60)
 *             .statistic("Average")
 *             .threshold(logstashServersCount)
 *             .alarmDescription("Number of healthy nodes in Target Group")
 *             .actionsEnabled("true")
 *             .alarmActions(sns.arn())
 *             .okActions(sns.arn())
 *             .dimensions(Map.ofEntries(
 *                 Map.entry("TargetGroup", lb_tg.arnSuffix()),
 *                 Map.entry("LoadBalancer", lb.arnSuffix())
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   nlbHealthyhosts:
 *     type: aws:cloudwatch:MetricAlarm
 *     name: nlb_healthyhosts
 *     properties:
 *       name: alarmname
 *       comparisonOperator: LessThanThreshold
 *       evaluationPeriods: 1
 *       metricName: HealthyHostCount
 *       namespace: AWS/NetworkELB
 *       period: 60
 *       statistic: Average
 *       threshold: ${logstashServersCount}
 *       alarmDescription: Number of healthy nodes in Target Group
 *       actionsEnabled: 'true'
 *       alarmActions:
 *         - ${sns.arn}
 *       okActions:
 *         - ${sns.arn}
 *       dimensions:
 *         TargetGroup: ${["lb-tg"].arnSuffix}
 *         LoadBalancer: ${lb.arnSuffix}
 * ```
 * <!--End PulumiCodeChooser -->
 * > **NOTE:**  You cannot create a metric alarm consisting of both `statistic` and `extended_statistic` parameters.
 * You must choose one or the other
 * ## Import
 * Using `pulumi import`, import CloudWatch Metric Alarm using the `alarm_name`. For example:
 * ```sh
 * $ pulumi import aws:cloudwatch/metricAlarm:MetricAlarm test alarm-12345
 * ```
 * @property actionsEnabled Indicates whether or not actions should be executed during any changes to the alarm's state. Defaults to `true`.
 * @property alarmActions The list of actions to execute when this alarm transitions into an ALARM state from any other state. Each action is specified as an Amazon Resource Name (ARN).
 * @property alarmDescription The description for the alarm.
 * @property comparisonOperator The arithmetic operation to use when comparing the specified Statistic and Threshold. The specified Statistic value is used as the first operand. Either of the following is supported: `GreaterThanOrEqualToThreshold`, `GreaterThanThreshold`, `LessThanThreshold`, `LessThanOrEqualToThreshold`. Additionally, the values  `LessThanLowerOrGreaterThanUpperThreshold`, `LessThanLowerThreshold`, and `GreaterThanUpperThreshold` are used only for alarms based on anomaly detection models.
 * @property datapointsToAlarm The number of datapoints that must be breaching to trigger the alarm.
 * @property dimensions The dimensions for the alarm's associated metric.  For the list of available dimensions see the AWS documentation [here](http://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
 * @property evaluateLowSampleCountPercentiles Used only for alarms based on percentiles.
 * If you specify `ignore`, the alarm state will not change during periods with too few data points to be statistically significant.
 * If you specify `evaluate` or omit this parameter, the alarm will always be evaluated and possibly change state no matter how many data points are available.
 * The following values are supported: `ignore`, and `evaluate`.
 * @property evaluationPeriods The number of periods over which data is compared to the specified threshold.
 * @property extendedStatistic The percentile statistic for the metric associated with the alarm. Specify a value between p0.0 and p100.
 * @property insufficientDataActions The list of actions to execute when this alarm transitions into an INSUFFICIENT_DATA state from any other state. Each action is specified as an Amazon Resource Name (ARN).
 * @property metricName The name for the alarm's associated metric.
 * See docs for [supported metrics](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
 * @property metricQueries Enables you to create an alarm based on a metric math expression. You may specify at most 20.
 * @property name The descriptive name for the alarm. This name must be unique within the user's AWS account
 * @property namespace The namespace for the alarm's associated metric. See docs for the [list of namespaces](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/aws-namespaces.html).
 * See docs for [supported metrics](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
 * @property okActions The list of actions to execute when this alarm transitions into an OK state from any other state. Each action is specified as an Amazon Resource Name (ARN).
 * @property period The period in seconds over which the specified `statistic` is applied.
 * Valid values are `10`, `30`, or any multiple of `60`.
 * @property statistic The statistic to apply to the alarm's associated metric.
 * Either of the following is supported: `SampleCount`, `Average`, `Sum`, `Minimum`, `Maximum`
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * > **NOTE:**  If you specify at least one `metric_query`, you may not specify a `metric_name`, `namespace`, `period` or `statistic`. If you do not specify a `metric_query`, you must specify each of these (although you may use `extended_statistic` instead of `statistic`).
 * @property threshold The value against which the specified statistic is compared. This parameter is required for alarms based on static thresholds, but should not be used for alarms based on anomaly detection models.
 * @property thresholdMetricId If this is an alarm based on an anomaly detection model, make this value match the ID of the ANOMALY_DETECTION_BAND function.
 * @property treatMissingData Sets how this alarm is to handle missing data points. The following values are supported: `missing`, `ignore`, `breaching` and `notBreaching`. Defaults to `missing`.
 * @property unit The unit for the alarm's associated metric.
 */
public data class MetricAlarmArgs(
    public val actionsEnabled: Output<Boolean>? = null,
    public val alarmActions: Output<List<String>>? = null,
    public val alarmDescription: Output<String>? = null,
    public val comparisonOperator: Output<String>? = null,
    public val datapointsToAlarm: Output<Int>? = null,
    public val dimensions: Output<Map<String, String>>? = null,
    public val evaluateLowSampleCountPercentiles: Output<String>? = null,
    public val evaluationPeriods: Output<Int>? = null,
    public val extendedStatistic: Output<String>? = null,
    public val insufficientDataActions: Output<List<String>>? = null,
    public val metricName: Output<String>? = null,
    public val metricQueries: Output<List<MetricAlarmMetricQueryArgs>>? = null,
    public val name: Output<String>? = null,
    public val namespace: Output<String>? = null,
    public val okActions: Output<List<String>>? = null,
    public val period: Output<Int>? = null,
    public val statistic: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val threshold: Output<Double>? = null,
    public val thresholdMetricId: Output<String>? = null,
    public val treatMissingData: Output<String>? = null,
    public val unit: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.cloudwatch.MetricAlarmArgs> {
    override fun toJava(): com.pulumi.aws.cloudwatch.MetricAlarmArgs =
        com.pulumi.aws.cloudwatch.MetricAlarmArgs.builder()
            .actionsEnabled(actionsEnabled?.applyValue({ args0 -> args0 }))
            .alarmActions(alarmActions?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .alarmDescription(alarmDescription?.applyValue({ args0 -> args0 }))
            .comparisonOperator(comparisonOperator?.applyValue({ args0 -> args0 }))
            .datapointsToAlarm(datapointsToAlarm?.applyValue({ args0 -> args0 }))
            .dimensions(
                dimensions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .evaluateLowSampleCountPercentiles(
                evaluateLowSampleCountPercentiles?.applyValue({ args0 ->
                    args0
                }),
            )
            .evaluationPeriods(evaluationPeriods?.applyValue({ args0 -> args0 }))
            .extendedStatistic(extendedStatistic?.applyValue({ args0 -> args0 }))
            .insufficientDataActions(
                insufficientDataActions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .metricName(metricName?.applyValue({ args0 -> args0 }))
            .metricQueries(
                metricQueries?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .name(name?.applyValue({ args0 -> args0 }))
            .namespace(namespace?.applyValue({ args0 -> args0 }))
            .okActions(okActions?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .period(period?.applyValue({ args0 -> args0 }))
            .statistic(statistic?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .threshold(threshold?.applyValue({ args0 -> args0 }))
            .thresholdMetricId(thresholdMetricId?.applyValue({ args0 -> args0 }))
            .treatMissingData(treatMissingData?.applyValue({ args0 -> args0 }))
            .unit(unit?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [MetricAlarmArgs].
 */
@PulumiTagMarker
public class MetricAlarmArgsBuilder internal constructor() {
    private var actionsEnabled: Output<Boolean>? = null

    private var alarmActions: Output<List<String>>? = null

    private var alarmDescription: Output<String>? = null

    private var comparisonOperator: Output<String>? = null

    private var datapointsToAlarm: Output<Int>? = null

    private var dimensions: Output<Map<String, String>>? = null

    private var evaluateLowSampleCountPercentiles: Output<String>? = null

    private var evaluationPeriods: Output<Int>? = null

    private var extendedStatistic: Output<String>? = null

    private var insufficientDataActions: Output<List<String>>? = null

    private var metricName: Output<String>? = null

    private var metricQueries: Output<List<MetricAlarmMetricQueryArgs>>? = null

    private var name: Output<String>? = null

    private var namespace: Output<String>? = null

    private var okActions: Output<List<String>>? = null

    private var period: Output<Int>? = null

    private var statistic: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var threshold: Output<Double>? = null

    private var thresholdMetricId: Output<String>? = null

    private var treatMissingData: Output<String>? = null

    private var unit: Output<String>? = null

    /**
     * @param value Indicates whether or not actions should be executed during any changes to the alarm's state. Defaults to `true`.
     */
    @JvmName("ytjyhyjkojxpftnu")
    public suspend fun actionsEnabled(`value`: Output<Boolean>) {
        this.actionsEnabled = value
    }

    /**
     * @param value The list of actions to execute when this alarm transitions into an ALARM state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("rqnvlfeifpcdvxnr")
    public suspend fun alarmActions(`value`: Output<List<String>>) {
        this.alarmActions = value
    }

    @JvmName("bqodkveekschnmgy")
    public suspend fun alarmActions(vararg values: Output<String>) {
        this.alarmActions = Output.all(values.asList())
    }

    /**
     * @param values The list of actions to execute when this alarm transitions into an ALARM state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("onhjykawogfqfgop")
    public suspend fun alarmActions(values: List<Output<String>>) {
        this.alarmActions = Output.all(values)
    }

    /**
     * @param value The description for the alarm.
     */
    @JvmName("nnwjewhnpsplutgk")
    public suspend fun alarmDescription(`value`: Output<String>) {
        this.alarmDescription = value
    }

    /**
     * @param value The arithmetic operation to use when comparing the specified Statistic and Threshold. The specified Statistic value is used as the first operand. Either of the following is supported: `GreaterThanOrEqualToThreshold`, `GreaterThanThreshold`, `LessThanThreshold`, `LessThanOrEqualToThreshold`. Additionally, the values  `LessThanLowerOrGreaterThanUpperThreshold`, `LessThanLowerThreshold`, and `GreaterThanUpperThreshold` are used only for alarms based on anomaly detection models.
     */
    @JvmName("ljmrqwpgkxbekhya")
    public suspend fun comparisonOperator(`value`: Output<String>) {
        this.comparisonOperator = value
    }

    /**
     * @param value The number of datapoints that must be breaching to trigger the alarm.
     */
    @JvmName("toovvhhadwtdsxks")
    public suspend fun datapointsToAlarm(`value`: Output<Int>) {
        this.datapointsToAlarm = value
    }

    /**
     * @param value The dimensions for the alarm's associated metric.  For the list of available dimensions see the AWS documentation [here](http://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
     */
    @JvmName("kroygqkykkcepvip")
    public suspend fun dimensions(`value`: Output<Map<String, String>>) {
        this.dimensions = value
    }

    /**
     * @param value Used only for alarms based on percentiles.
     * If you specify `ignore`, the alarm state will not change during periods with too few data points to be statistically significant.
     * If you specify `evaluate` or omit this parameter, the alarm will always be evaluated and possibly change state no matter how many data points are available.
     * The following values are supported: `ignore`, and `evaluate`.
     */
    @JvmName("vkswovcqdleiutee")
    public suspend fun evaluateLowSampleCountPercentiles(`value`: Output<String>) {
        this.evaluateLowSampleCountPercentiles = value
    }

    /**
     * @param value The number of periods over which data is compared to the specified threshold.
     */
    @JvmName("jtswbcerlsutqtgk")
    public suspend fun evaluationPeriods(`value`: Output<Int>) {
        this.evaluationPeriods = value
    }

    /**
     * @param value The percentile statistic for the metric associated with the alarm. Specify a value between p0.0 and p100.
     */
    @JvmName("ddglbtcykvwebjaq")
    public suspend fun extendedStatistic(`value`: Output<String>) {
        this.extendedStatistic = value
    }

    /**
     * @param value The list of actions to execute when this alarm transitions into an INSUFFICIENT_DATA state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("pyiiajtehqklcqkc")
    public suspend fun insufficientDataActions(`value`: Output<List<String>>) {
        this.insufficientDataActions = value
    }

    @JvmName("mffeqoymuutxxqxn")
    public suspend fun insufficientDataActions(vararg values: Output<String>) {
        this.insufficientDataActions = Output.all(values.asList())
    }

    /**
     * @param values The list of actions to execute when this alarm transitions into an INSUFFICIENT_DATA state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("usnddttnwtartpyl")
    public suspend fun insufficientDataActions(values: List<Output<String>>) {
        this.insufficientDataActions = Output.all(values)
    }

    /**
     * @param value The name for the alarm's associated metric.
     * See docs for [supported metrics](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
     */
    @JvmName("drffpcsspcowuuxw")
    public suspend fun metricName(`value`: Output<String>) {
        this.metricName = value
    }

    /**
     * @param value Enables you to create an alarm based on a metric math expression. You may specify at most 20.
     */
    @JvmName("btpybobgpdvmammy")
    public suspend fun metricQueries(`value`: Output<List<MetricAlarmMetricQueryArgs>>) {
        this.metricQueries = value
    }

    @JvmName("ylhkahlemxteoole")
    public suspend fun metricQueries(vararg values: Output<MetricAlarmMetricQueryArgs>) {
        this.metricQueries = Output.all(values.asList())
    }

    /**
     * @param values Enables you to create an alarm based on a metric math expression. You may specify at most 20.
     */
    @JvmName("alhbwnwgsesbksbx")
    public suspend fun metricQueries(values: List<Output<MetricAlarmMetricQueryArgs>>) {
        this.metricQueries = Output.all(values)
    }

    /**
     * @param value The descriptive name for the alarm. This name must be unique within the user's AWS account
     */
    @JvmName("dgnludvcbkwcctpv")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The namespace for the alarm's associated metric. See docs for the [list of namespaces](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/aws-namespaces.html).
     * See docs for [supported metrics](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
     */
    @JvmName("hjdacktpnynxxuvi")
    public suspend fun namespace(`value`: Output<String>) {
        this.namespace = value
    }

    /**
     * @param value The list of actions to execute when this alarm transitions into an OK state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("svboyjdwmhhwaqjc")
    public suspend fun okActions(`value`: Output<List<String>>) {
        this.okActions = value
    }

    @JvmName("dnuprxvpxdapxran")
    public suspend fun okActions(vararg values: Output<String>) {
        this.okActions = Output.all(values.asList())
    }

    /**
     * @param values The list of actions to execute when this alarm transitions into an OK state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("ovsdfabygmgtxivs")
    public suspend fun okActions(values: List<Output<String>>) {
        this.okActions = Output.all(values)
    }

    /**
     * @param value The period in seconds over which the specified `statistic` is applied.
     * Valid values are `10`, `30`, or any multiple of `60`.
     */
    @JvmName("apattvepkataamul")
    public suspend fun period(`value`: Output<Int>) {
        this.period = value
    }

    /**
     * @param value The statistic to apply to the alarm's associated metric.
     * Either of the following is supported: `SampleCount`, `Average`, `Sum`, `Minimum`, `Maximum`
     */
    @JvmName("hljxekmpyugxdexx")
    public suspend fun statistic(`value`: Output<String>) {
        this.statistic = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     * > **NOTE:**  If you specify at least one `metric_query`, you may not specify a `metric_name`, `namespace`, `period` or `statistic`. If you do not specify a `metric_query`, you must specify each of these (although you may use `extended_statistic` instead of `statistic`).
     */
    @JvmName("ubiakqqfbfabidle")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The value against which the specified statistic is compared. This parameter is required for alarms based on static thresholds, but should not be used for alarms based on anomaly detection models.
     */
    @JvmName("nqttdlowxyvekfsc")
    public suspend fun threshold(`value`: Output<Double>) {
        this.threshold = value
    }

    /**
     * @param value If this is an alarm based on an anomaly detection model, make this value match the ID of the ANOMALY_DETECTION_BAND function.
     */
    @JvmName("qixnafpqxipvmhct")
    public suspend fun thresholdMetricId(`value`: Output<String>) {
        this.thresholdMetricId = value
    }

    /**
     * @param value Sets how this alarm is to handle missing data points. The following values are supported: `missing`, `ignore`, `breaching` and `notBreaching`. Defaults to `missing`.
     */
    @JvmName("rqvqjbntymieyylg")
    public suspend fun treatMissingData(`value`: Output<String>) {
        this.treatMissingData = value
    }

    /**
     * @param value The unit for the alarm's associated metric.
     */
    @JvmName("khlqromvfxqgmhep")
    public suspend fun unit(`value`: Output<String>) {
        this.unit = value
    }

    /**
     * @param value Indicates whether or not actions should be executed during any changes to the alarm's state. Defaults to `true`.
     */
    @JvmName("naofvxpkfqctbtqt")
    public suspend fun actionsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.actionsEnabled = mapped
    }

    /**
     * @param value The list of actions to execute when this alarm transitions into an ALARM state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("vjhxpwjnmdwubabg")
    public suspend fun alarmActions(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.alarmActions = mapped
    }

    /**
     * @param values The list of actions to execute when this alarm transitions into an ALARM state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("dnypnkukfxibhnwx")
    public suspend fun alarmActions(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.alarmActions = mapped
    }

    /**
     * @param value The description for the alarm.
     */
    @JvmName("uulyhaimkmosdgxj")
    public suspend fun alarmDescription(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.alarmDescription = mapped
    }

    /**
     * @param value The arithmetic operation to use when comparing the specified Statistic and Threshold. The specified Statistic value is used as the first operand. Either of the following is supported: `GreaterThanOrEqualToThreshold`, `GreaterThanThreshold`, `LessThanThreshold`, `LessThanOrEqualToThreshold`. Additionally, the values  `LessThanLowerOrGreaterThanUpperThreshold`, `LessThanLowerThreshold`, and `GreaterThanUpperThreshold` are used only for alarms based on anomaly detection models.
     */
    @JvmName("nwearywmiyhonvte")
    public suspend fun comparisonOperator(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.comparisonOperator = mapped
    }

    /**
     * @param value The number of datapoints that must be breaching to trigger the alarm.
     */
    @JvmName("yssivrkojwlanewc")
    public suspend fun datapointsToAlarm(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.datapointsToAlarm = mapped
    }

    /**
     * @param value The dimensions for the alarm's associated metric.  For the list of available dimensions see the AWS documentation [here](http://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
     */
    @JvmName("hvnglevhmhvvajyj")
    public suspend fun dimensions(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dimensions = mapped
    }

    /**
     * @param values The dimensions for the alarm's associated metric.  For the list of available dimensions see the AWS documentation [here](http://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
     */
    @JvmName("lvaktbrecksjwrbf")
    public fun dimensions(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.dimensions = mapped
    }

    /**
     * @param value Used only for alarms based on percentiles.
     * If you specify `ignore`, the alarm state will not change during periods with too few data points to be statistically significant.
     * If you specify `evaluate` or omit this parameter, the alarm will always be evaluated and possibly change state no matter how many data points are available.
     * The following values are supported: `ignore`, and `evaluate`.
     */
    @JvmName("toebgnvigwabqtag")
    public suspend fun evaluateLowSampleCountPercentiles(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.evaluateLowSampleCountPercentiles = mapped
    }

    /**
     * @param value The number of periods over which data is compared to the specified threshold.
     */
    @JvmName("pivqnpwoqbsrhfkg")
    public suspend fun evaluationPeriods(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.evaluationPeriods = mapped
    }

    /**
     * @param value The percentile statistic for the metric associated with the alarm. Specify a value between p0.0 and p100.
     */
    @JvmName("wvhtcdwuhxrmdsft")
    public suspend fun extendedStatistic(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.extendedStatistic = mapped
    }

    /**
     * @param value The list of actions to execute when this alarm transitions into an INSUFFICIENT_DATA state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("clmswqhbsicnhveo")
    public suspend fun insufficientDataActions(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.insufficientDataActions = mapped
    }

    /**
     * @param values The list of actions to execute when this alarm transitions into an INSUFFICIENT_DATA state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("dglcgpcgifujckmp")
    public suspend fun insufficientDataActions(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.insufficientDataActions = mapped
    }

    /**
     * @param value The name for the alarm's associated metric.
     * See docs for [supported metrics](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
     */
    @JvmName("qnankmbdfsqpcnil")
    public suspend fun metricName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.metricName = mapped
    }

    /**
     * @param value Enables you to create an alarm based on a metric math expression. You may specify at most 20.
     */
    @JvmName("wkkjqkjrjelyjipx")
    public suspend fun metricQueries(`value`: List<MetricAlarmMetricQueryArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.metricQueries = mapped
    }

    /**
     * @param argument Enables you to create an alarm based on a metric math expression. You may specify at most 20.
     */
    @JvmName("nirgycwfkrqfvkft")
    public suspend fun metricQueries(argument: List<suspend MetricAlarmMetricQueryArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            MetricAlarmMetricQueryArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.metricQueries = mapped
    }

    /**
     * @param argument Enables you to create an alarm based on a metric math expression. You may specify at most 20.
     */
    @JvmName("pyvayvwrxbksgpvn")
    public suspend fun metricQueries(vararg argument: suspend MetricAlarmMetricQueryArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            MetricAlarmMetricQueryArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.metricQueries = mapped
    }

    /**
     * @param argument Enables you to create an alarm based on a metric math expression. You may specify at most 20.
     */
    @JvmName("cvujrxxedhhjjjyo")
    public suspend fun metricQueries(argument: suspend MetricAlarmMetricQueryArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(MetricAlarmMetricQueryArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.metricQueries = mapped
    }

    /**
     * @param values Enables you to create an alarm based on a metric math expression. You may specify at most 20.
     */
    @JvmName("hybuvyodgasmkqkf")
    public suspend fun metricQueries(vararg values: MetricAlarmMetricQueryArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.metricQueries = mapped
    }

    /**
     * @param value The descriptive name for the alarm. This name must be unique within the user's AWS account
     */
    @JvmName("ivqejgawawgawlfi")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The namespace for the alarm's associated metric. See docs for the [list of namespaces](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/aws-namespaces.html).
     * See docs for [supported metrics](https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html).
     */
    @JvmName("kwjyuubtfdljlyiw")
    public suspend fun namespace(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namespace = mapped
    }

    /**
     * @param value The list of actions to execute when this alarm transitions into an OK state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("mhxdjkalkhpxajqg")
    public suspend fun okActions(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.okActions = mapped
    }

    /**
     * @param values The list of actions to execute when this alarm transitions into an OK state from any other state. Each action is specified as an Amazon Resource Name (ARN).
     */
    @JvmName("anutpxvloxnkxjhn")
    public suspend fun okActions(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.okActions = mapped
    }

    /**
     * @param value The period in seconds over which the specified `statistic` is applied.
     * Valid values are `10`, `30`, or any multiple of `60`.
     */
    @JvmName("nikmdduyncmvfsca")
    public suspend fun period(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.period = mapped
    }

    /**
     * @param value The statistic to apply to the alarm's associated metric.
     * Either of the following is supported: `SampleCount`, `Average`, `Sum`, `Minimum`, `Maximum`
     */
    @JvmName("ucoqektyrpieonrl")
    public suspend fun statistic(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.statistic = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     * > **NOTE:**  If you specify at least one `metric_query`, you may not specify a `metric_name`, `namespace`, `period` or `statistic`. If you do not specify a `metric_query`, you must specify each of these (although you may use `extended_statistic` instead of `statistic`).
     */
    @JvmName("hyslehcarmwogwrh")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     * > **NOTE:**  If you specify at least one `metric_query`, you may not specify a `metric_name`, `namespace`, `period` or `statistic`. If you do not specify a `metric_query`, you must specify each of these (although you may use `extended_statistic` instead of `statistic`).
     */
    @JvmName("nbijnatahjagbnsp")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The value against which the specified statistic is compared. This parameter is required for alarms based on static thresholds, but should not be used for alarms based on anomaly detection models.
     */
    @JvmName("rjgvpmyvwigfmaep")
    public suspend fun threshold(`value`: Double?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.threshold = mapped
    }

    /**
     * @param value If this is an alarm based on an anomaly detection model, make this value match the ID of the ANOMALY_DETECTION_BAND function.
     */
    @JvmName("imeydknknxtafosg")
    public suspend fun thresholdMetricId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.thresholdMetricId = mapped
    }

    /**
     * @param value Sets how this alarm is to handle missing data points. The following values are supported: `missing`, `ignore`, `breaching` and `notBreaching`. Defaults to `missing`.
     */
    @JvmName("thaojxauaktbjypp")
    public suspend fun treatMissingData(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.treatMissingData = mapped
    }

    /**
     * @param value The unit for the alarm's associated metric.
     */
    @JvmName("txhsducwtgpdfbun")
    public suspend fun unit(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.unit = mapped
    }

    internal fun build(): MetricAlarmArgs = MetricAlarmArgs(
        actionsEnabled = actionsEnabled,
        alarmActions = alarmActions,
        alarmDescription = alarmDescription,
        comparisonOperator = comparisonOperator,
        datapointsToAlarm = datapointsToAlarm,
        dimensions = dimensions,
        evaluateLowSampleCountPercentiles = evaluateLowSampleCountPercentiles,
        evaluationPeriods = evaluationPeriods,
        extendedStatistic = extendedStatistic,
        insufficientDataActions = insufficientDataActions,
        metricName = metricName,
        metricQueries = metricQueries,
        name = name,
        namespace = namespace,
        okActions = okActions,
        period = period,
        statistic = statistic,
        tags = tags,
        threshold = threshold,
        thresholdMetricId = thresholdMetricId,
        treatMissingData = treatMissingData,
        unit = unit,
    )
}
