@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.codebuild.kotlin

import com.pulumi.aws.codebuild.SourceCredentialArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a CodeBuild Source Credentials Resource.
 * > **NOTE:** [Codebuild only allows a single credential per given server type in a given region](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_codebuild.GitHubSourceCredentials.html). Therefore, when you define `aws.codebuild.SourceCredential`, `aws.codebuild.Project` resource defined in the same module will use it.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.codebuild.SourceCredential("example", {
 *     authType: "PERSONAL_ACCESS_TOKEN",
 *     serverType: "GITHUB",
 *     token: "example",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.codebuild.SourceCredential("example",
 *     auth_type="PERSONAL_ACCESS_TOKEN",
 *     server_type="GITHUB",
 *     token="example")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CodeBuild.SourceCredential("example", new()
 *     {
 *         AuthType = "PERSONAL_ACCESS_TOKEN",
 *         ServerType = "GITHUB",
 *         Token = "example",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/codebuild"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := codebuild.NewSourceCredential(ctx, "example", &codebuild.SourceCredentialArgs{
 * 			AuthType:   pulumi.String("PERSONAL_ACCESS_TOKEN"),
 * 			ServerType: pulumi.String("GITHUB"),
 * 			Token:      pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.codebuild.SourceCredential;
 * import com.pulumi.aws.codebuild.SourceCredentialArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new SourceCredential("example", SourceCredentialArgs.builder()
 *             .authType("PERSONAL_ACCESS_TOKEN")
 *             .serverType("GITHUB")
 *             .token("example")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:codebuild:SourceCredential
 *     properties:
 *       authType: PERSONAL_ACCESS_TOKEN
 *       serverType: GITHUB
 *       token: example
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Bitbucket Server Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.codebuild.SourceCredential("example", {
 *     authType: "BASIC_AUTH",
 *     serverType: "BITBUCKET",
 *     token: "example",
 *     userName: "test-user",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.codebuild.SourceCredential("example",
 *     auth_type="BASIC_AUTH",
 *     server_type="BITBUCKET",
 *     token="example",
 *     user_name="test-user")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CodeBuild.SourceCredential("example", new()
 *     {
 *         AuthType = "BASIC_AUTH",
 *         ServerType = "BITBUCKET",
 *         Token = "example",
 *         UserName = "test-user",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/codebuild"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := codebuild.NewSourceCredential(ctx, "example", &codebuild.SourceCredentialArgs{
 * 			AuthType:   pulumi.String("BASIC_AUTH"),
 * 			ServerType: pulumi.String("BITBUCKET"),
 * 			Token:      pulumi.String("example"),
 * 			UserName:   pulumi.String("test-user"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.codebuild.SourceCredential;
 * import com.pulumi.aws.codebuild.SourceCredentialArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new SourceCredential("example", SourceCredentialArgs.builder()
 *             .authType("BASIC_AUTH")
 *             .serverType("BITBUCKET")
 *             .token("example")
 *             .userName("test-user")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:codebuild:SourceCredential
 *     properties:
 *       authType: BASIC_AUTH
 *       serverType: BITBUCKET
 *       token: example
 *       userName: test-user
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import CodeBuild Source Credential using the CodeBuild Source Credential arn. For example:
 * ```sh
 * $ pulumi import aws:codebuild/sourceCredential:SourceCredential example arn:aws:codebuild:us-west-2:123456789:token:github
 * ```
 * @property authType The type of authentication used to connect to a GitHub, GitHub Enterprise, or Bitbucket repository. An OAUTH connection is not supported by the API.
 * @property serverType The source provider used for this project.
 * @property token For `GitHub` or `GitHub Enterprise`, this is the personal access token. For `Bitbucket`, this is the app password.
 * @property userName The Bitbucket username when the authType is `BASIC_AUTH`. This parameter is not valid for other types of source providers or connections.
 */
public data class SourceCredentialArgs(
    public val authType: Output<String>? = null,
    public val serverType: Output<String>? = null,
    public val token: Output<String>? = null,
    public val userName: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.codebuild.SourceCredentialArgs> {
    override fun toJava(): com.pulumi.aws.codebuild.SourceCredentialArgs =
        com.pulumi.aws.codebuild.SourceCredentialArgs.builder()
            .authType(authType?.applyValue({ args0 -> args0 }))
            .serverType(serverType?.applyValue({ args0 -> args0 }))
            .token(token?.applyValue({ args0 -> args0 }))
            .userName(userName?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [SourceCredentialArgs].
 */
@PulumiTagMarker
public class SourceCredentialArgsBuilder internal constructor() {
    private var authType: Output<String>? = null

    private var serverType: Output<String>? = null

    private var token: Output<String>? = null

    private var userName: Output<String>? = null

    /**
     * @param value The type of authentication used to connect to a GitHub, GitHub Enterprise, or Bitbucket repository. An OAUTH connection is not supported by the API.
     */
    @JvmName("oxgoxkehjkwfrhih")
    public suspend fun authType(`value`: Output<String>) {
        this.authType = value
    }

    /**
     * @param value The source provider used for this project.
     */
    @JvmName("wetcwfpiewxdmojq")
    public suspend fun serverType(`value`: Output<String>) {
        this.serverType = value
    }

    /**
     * @param value For `GitHub` or `GitHub Enterprise`, this is the personal access token. For `Bitbucket`, this is the app password.
     */
    @JvmName("qlafwpgpwixlmqwa")
    public suspend fun token(`value`: Output<String>) {
        this.token = value
    }

    /**
     * @param value The Bitbucket username when the authType is `BASIC_AUTH`. This parameter is not valid for other types of source providers or connections.
     */
    @JvmName("lpscffcgdrjpglcl")
    public suspend fun userName(`value`: Output<String>) {
        this.userName = value
    }

    /**
     * @param value The type of authentication used to connect to a GitHub, GitHub Enterprise, or Bitbucket repository. An OAUTH connection is not supported by the API.
     */
    @JvmName("gwtyyasuwlkkgsmh")
    public suspend fun authType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authType = mapped
    }

    /**
     * @param value The source provider used for this project.
     */
    @JvmName("grftckvlaafjoxlj")
    public suspend fun serverType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverType = mapped
    }

    /**
     * @param value For `GitHub` or `GitHub Enterprise`, this is the personal access token. For `Bitbucket`, this is the app password.
     */
    @JvmName("oobytnfgobihpejm")
    public suspend fun token(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.token = mapped
    }

    /**
     * @param value The Bitbucket username when the authType is `BASIC_AUTH`. This parameter is not valid for other types of source providers or connections.
     */
    @JvmName("jfqiaxplmuhrhqtm")
    public suspend fun userName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userName = mapped
    }

    internal fun build(): SourceCredentialArgs = SourceCredentialArgs(
        authType = authType,
        serverType = serverType,
        token = token,
        userName = userName,
    )
}
