@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.cognito.kotlin

import com.pulumi.aws.cognito.UserPoolClientArgs.builder
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientAnalyticsConfigurationArgs
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientAnalyticsConfigurationArgsBuilder
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientTokenValidityUnitsArgs
import com.pulumi.aws.cognito.kotlin.inputs.UserPoolClientTokenValidityUnitsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Provides a Cognito User Pool Client resource.
 * To manage a User Pool Client created by another service, such as when [configuring an OpenSearch Domain to use Cognito authentication](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html),
 * use the `aws.cognito.ManagedUserPoolClient` resource instead.
 * ## Example Usage
 * ### Create a basic user pool client
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const pool = new aws.cognito.UserPool("pool", {name: "pool"});
 * const client = new aws.cognito.UserPoolClient("client", {
 *     name: "client",
 *     userPoolId: pool.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * pool = aws.cognito.UserPool("pool", name="pool")
 * client = aws.cognito.UserPoolClient("client",
 *     name="client",
 *     user_pool_id=pool.id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var pool = new Aws.Cognito.UserPool("pool", new()
 *     {
 *         Name = "pool",
 *     });
 *     var client = new Aws.Cognito.UserPoolClient("client", new()
 *     {
 *         Name = "client",
 *         UserPoolId = pool.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cognito"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		pool, err := cognito.NewUserPool(ctx, "pool", &cognito.UserPoolArgs{
 * 			Name: pulumi.String("pool"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cognito.NewUserPoolClient(ctx, "client", &cognito.UserPoolClientArgs{
 * 			Name:       pulumi.String("client"),
 * 			UserPoolId: pool.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolArgs;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var pool = new UserPool("pool", UserPoolArgs.builder()
 *             .name("pool")
 *             .build());
 *         var client = new UserPoolClient("client", UserPoolClientArgs.builder()
 *             .name("client")
 *             .userPoolId(pool.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   client:
 *     type: aws:cognito:UserPoolClient
 *     properties:
 *       name: client
 *       userPoolId: ${pool.id}
 *   pool:
 *     type: aws:cognito:UserPool
 *     properties:
 *       name: pool
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Create a user pool client with no SRP authentication
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const pool = new aws.cognito.UserPool("pool", {name: "pool"});
 * const client = new aws.cognito.UserPoolClient("client", {
 *     name: "client",
 *     userPoolId: pool.id,
 *     generateSecret: true,
 *     explicitAuthFlows: ["ADMIN_NO_SRP_AUTH"],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * pool = aws.cognito.UserPool("pool", name="pool")
 * client = aws.cognito.UserPoolClient("client",
 *     name="client",
 *     user_pool_id=pool.id,
 *     generate_secret=True,
 *     explicit_auth_flows=["ADMIN_NO_SRP_AUTH"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var pool = new Aws.Cognito.UserPool("pool", new()
 *     {
 *         Name = "pool",
 *     });
 *     var client = new Aws.Cognito.UserPoolClient("client", new()
 *     {
 *         Name = "client",
 *         UserPoolId = pool.Id,
 *         GenerateSecret = true,
 *         ExplicitAuthFlows = new[]
 *         {
 *             "ADMIN_NO_SRP_AUTH",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cognito"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		pool, err := cognito.NewUserPool(ctx, "pool", &cognito.UserPoolArgs{
 * 			Name: pulumi.String("pool"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cognito.NewUserPoolClient(ctx, "client", &cognito.UserPoolClientArgs{
 * 			Name:           pulumi.String("client"),
 * 			UserPoolId:     pool.ID(),
 * 			GenerateSecret: pulumi.Bool(true),
 * 			ExplicitAuthFlows: pulumi.StringArray{
 * 				pulumi.String("ADMIN_NO_SRP_AUTH"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolArgs;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var pool = new UserPool("pool", UserPoolArgs.builder()
 *             .name("pool")
 *             .build());
 *         var client = new UserPoolClient("client", UserPoolClientArgs.builder()
 *             .name("client")
 *             .userPoolId(pool.id())
 *             .generateSecret(true)
 *             .explicitAuthFlows("ADMIN_NO_SRP_AUTH")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   client:
 *     type: aws:cognito:UserPoolClient
 *     properties:
 *       name: client
 *       userPoolId: ${pool.id}
 *       generateSecret: true
 *       explicitAuthFlows:
 *         - ADMIN_NO_SRP_AUTH
 *   pool:
 *     type: aws:cognito:UserPool
 *     properties:
 *       name: pool
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Create a user pool client with pinpoint analytics
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const testUserPool = new aws.cognito.UserPool("test", {name: "pool"});
 * const testApp = new aws.pinpoint.App("test", {name: "pinpoint"});
 * const assumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["cognito-idp&#46;amazonaws&#46;com"],
 *         }],
 *         actions: ["sts:AssumeRole"],
 *     }],
 * });
 * const testRole = new aws.iam.Role("test", {
 *     name: "role",
 *     assumeRolePolicy: assumeRole.then(assumeRole => assumeRole.json),
 * });
 * const testUserPoolClient = new aws.cognito.UserPoolClient("test", {
 *     name: "pool_client",
 *     userPoolId: testUserPool.id,
 *     analyticsConfiguration: {
 *         applicationId: testApp.applicationId,
 *         externalId: "some_id",
 *         roleArn: testRole.arn,
 *         userDataShared: true,
 *     },
 * });
 * const current = aws.getCallerIdentity({});
 * const test = aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         effect: "Allow",
 *         actions: [
 *             "mobiletargeting:UpdateEndpoint",
 *             "mobiletargeting:PutEvents",
 *         ],
 *         resources: [pulumi&#46;all([current, testApp&#46;applicationId])&#46;apply(([current, applicationId]) => `arn:aws:mobiletargeting:*:${current&#46;accountId}:apps/${applicationId}*`)],
 *     }],
 * });
 * const testRolePolicy = new aws.iam.RolePolicy("test", {
 *     name: "role_policy",
 *     role: testRole.id,
 *     policy: test.apply(test => test.json),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test_user_pool = aws.cognito.UserPool("test", name="pool")
 * test_app = aws.pinpoint.App("test", name="pinpoint")
 * assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["cognito-idp&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": ["sts:AssumeRole"],
 * }])
 * test_role = aws.iam.Role("test",
 *     name="role",
 *     assume_role_policy=assume_role.json)
 * test_user_pool_client = aws.cognito.UserPoolClient("test",
 *     name="pool_client",
 *     user_pool_id=test_user_pool.id,
 *     analytics_configuration={
 *         "application_id": test_app.application_id,
 *         "external_id": "some_id",
 *         "role_arn": test_role.arn,
 *         "user_data_shared": True,
 *     })
 * current = aws.get_caller_identity()
 * test = aws.iam.get_policy_document_output(statements=[{
 *     "effect": "Allow",
 *     "actions": [
 *         "mobiletargeting:UpdateEndpoint",
 *         "mobiletargeting:PutEvents",
 *     ],
 *     "resources": [test_app&#46;application_id&#46;apply(lambda application_id: f"arn:aws:mobiletargeting:*:{current&#46;account_id}:apps/{application_id}*")],
 * }])
 * test_role_policy = aws.iam.RolePolicy("test",
 *     name="role_policy",
 *     role=test_role.id,
 *     policy=test.json)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var testUserPool = new Aws.Cognito.UserPool("test", new()
 *     {
 *         Name = "pool",
 *     });
 *     var testApp = new Aws.Pinpoint.App("test", new()
 *     {
 *         Name = "pinpoint",
 *     });
 *     var assumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "cognito-idp.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *             },
 *         },
 *     });
 *     var testRole = new Aws.Iam.Role("test", new()
 *     {
 *         Name = "role",
 *         AssumeRolePolicy = assumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var testUserPoolClient = new Aws.Cognito.UserPoolClient("test", new()
 *     {
 *         Name = "pool_client",
 *         UserPoolId = testUserPool.Id,
 *         AnalyticsConfiguration = new Aws.Cognito.Inputs.UserPoolClientAnalyticsConfigurationArgs
 *         {
 *             ApplicationId = testApp.ApplicationId,
 *             ExternalId = "some_id",
 *             RoleArn = testRole.Arn,
 *             UserDataShared = true,
 *         },
 *     });
 *     var current = Aws.GetCallerIdentity.Invoke();
 *     var test = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Actions = new[]
 *                 {
 *                     "mobiletargeting:UpdateEndpoint",
 *                     "mobiletargeting:PutEvents",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     $"arn:aws:mobiletargeting:*:{current.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId)}:apps/{testApp.ApplicationId}*",
 *                 },
 *             },
 *         },
 *     });
 *     var testRolePolicy = new Aws.Iam.RolePolicy("test", new()
 *     {
 *         Name = "role_policy",
 *         Role = testRole.Id,
 *         Policy = test.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cognito"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/pinpoint"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		testUserPool, err := cognito.NewUserPool(ctx, "test", &cognito.UserPoolArgs{
 * 			Name: pulumi.String("pool"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		testApp, err := pinpoint.NewApp(ctx, "test", &pinpoint.AppArgs{
 * 			Name: pulumi.String("pinpoint"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		assumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"cognito-idp.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		testRole, err := iam.NewRole(ctx, "test", &iam.RoleArgs{
 * 			Name:             pulumi.String("role"),
 * 			AssumeRolePolicy: pulumi.String(assumeRole.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cognito.NewUserPoolClient(ctx, "test", &cognito.UserPoolClientArgs{
 * 			Name:       pulumi.String("pool_client"),
 * 			UserPoolId: testUserPool.ID(),
 * 			AnalyticsConfiguration: &cognito.UserPoolClientAnalyticsConfigurationArgs{
 * 				ApplicationId:  testApp.ApplicationId,
 * 				ExternalId:     pulumi.String("some_id"),
 * 				RoleArn:        testRole.Arn,
 * 				UserDataShared: pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		current, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		test := iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * 			Statements: iam.GetPolicyDocumentStatementArray{
 * 				&iam.GetPolicyDocumentStatementArgs{
 * 					Effect: pulumi.String("Allow"),
 * 					Actions: pulumi.StringArray{
 * 						pulumi.String("mobiletargeting:UpdateEndpoint"),
 * 						pulumi.String("mobiletargeting:PutEvents"),
 * 					},
 * 					Resources: pulumi.StringArray{
 * 						testApp.ApplicationId.ApplyT(func(applicationId string) (string, error) {
 * 							return fmt.Sprintf("arn:aws:mobiletargeting:*:%v:apps/%v*", current.AccountId, applicationId), nil
 * 						}).(pulumi.StringOutput),
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		_, err = iam.NewRolePolicy(ctx, "test", &iam.RolePolicyArgs{
 * 			Name: pulumi.String("role_policy"),
 * 			Role: testRole.ID(),
 * 			Policy: pulumi.String(test.ApplyT(func(test iam.GetPolicyDocumentResult) (*string, error) {
 * 				return &test.Json, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolArgs;
 * import com.pulumi.aws.pinpoint.App;
 * import com.pulumi.aws.pinpoint.AppArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import com.pulumi.aws.cognito.inputs.UserPoolClientAnalyticsConfigurationArgs;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.iam.RolePolicy;
 * import com.pulumi.aws.iam.RolePolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var testUserPool = new UserPool("testUserPool", UserPoolArgs.builder()
 *             .name("pool")
 *             .build());
 *         var testApp = new App("testApp", AppArgs.builder()
 *             .name("pinpoint")
 *             .build());
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("cognito-idp.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var testRole = new Role("testRole", RoleArgs.builder()
 *             .name("role")
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var testUserPoolClient = new UserPoolClient("testUserPoolClient", UserPoolClientArgs.builder()
 *             .name("pool_client")
 *             .userPoolId(testUserPool.id())
 *             .analyticsConfiguration(UserPoolClientAnalyticsConfigurationArgs.builder()
 *                 .applicationId(testApp.applicationId())
 *                 .externalId("some_id")
 *                 .roleArn(testRole.arn())
 *                 .userDataShared(true)
 *                 .build())
 *             .build());
 *         final var current = AwsFunctions.getCallerIdentity();
 *         final var test = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .actions(
 *                     "mobiletargeting:UpdateEndpoint",
 *                     "mobiletargeting:PutEvents")
 *                 .resources(testApp.applicationId().applyValue(applicationId -> String.format("arn:aws:mobiletargeting:*:%s:apps/%s*", current.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),applicationId)))
 *                 .build())
 *             .build());
 *         var testRolePolicy = new RolePolicy("testRolePolicy", RolePolicyArgs.builder()
 *             .name("role_policy")
 *             .role(testRole.id())
 *             .policy(test.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(test -> test.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   testUserPoolClient:
 *     type: aws:cognito:UserPoolClient
 *     name: test
 *     properties:
 *       name: pool_client
 *       userPoolId: ${testUserPool.id}
 *       analyticsConfiguration:
 *         applicationId: ${testApp.applicationId}
 *         externalId: some_id
 *         roleArn: ${testRole.arn}
 *         userDataShared: true
 *   testUserPool:
 *     type: aws:cognito:UserPool
 *     name: test
 *     properties:
 *       name: pool
 *   testApp:
 *     type: aws:pinpoint:App
 *     name: test
 *     properties:
 *       name: pinpoint
 *   testRole:
 *     type: aws:iam:Role
 *     name: test
 *     properties:
 *       name: role
 *       assumeRolePolicy: ${assumeRole.json}
 *   testRolePolicy:
 *     type: aws:iam:RolePolicy
 *     name: test
 *     properties:
 *       name: role_policy
 *       role: ${testRole.id}
 *       policy: ${test.json}
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 *   assumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - cognito-idp.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 *   test:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             actions:
 *               - mobiletargeting:UpdateEndpoint
 *               - mobiletargeting:PutEvents
 *             resources:
 *               - arn:aws:mobiletargeting:*:${current.accountId}:apps/${testApp.applicationId}*
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Create a user pool client with Cognito as the identity provider
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const pool = new aws.cognito.UserPool("pool", {name: "pool"});
 * const userpoolClient = new aws.cognito.UserPoolClient("userpool_client", {
 *     name: "client",
 *     userPoolId: pool.id,
 *     callbackUrls: ["https://example&#46;com"],
 *     allowedOauthFlowsUserPoolClient: true,
 *     allowedOauthFlows: [
 *         "code",
 *         "implicit",
 *     ],
 *     allowedOauthScopes: [
 *         "email",
 *         "openid",
 *     ],
 *     supportedIdentityProviders: ["COGNITO"],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * pool = aws.cognito.UserPool("pool", name="pool")
 * userpool_client = aws.cognito.UserPoolClient("userpool_client",
 *     name="client",
 *     user_pool_id=pool.id,
 *     callback_urls=["https://example&#46;com"],
 *     allowed_oauth_flows_user_pool_client=True,
 *     allowed_oauth_flows=[
 *         "code",
 *         "implicit",
 *     ],
 *     allowed_oauth_scopes=[
 *         "email",
 *         "openid",
 *     ],
 *     supported_identity_providers=["COGNITO"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var pool = new Aws.Cognito.UserPool("pool", new()
 *     {
 *         Name = "pool",
 *     });
 *     var userpoolClient = new Aws.Cognito.UserPoolClient("userpool_client", new()
 *     {
 *         Name = "client",
 *         UserPoolId = pool.Id,
 *         CallbackUrls = new[]
 *         {
 *             "https://example.com",
 *         },
 *         AllowedOauthFlowsUserPoolClient = true,
 *         AllowedOauthFlows = new[]
 *         {
 *             "code",
 *             "implicit",
 *         },
 *         AllowedOauthScopes = new[]
 *         {
 *             "email",
 *             "openid",
 *         },
 *         SupportedIdentityProviders = new[]
 *         {
 *             "COGNITO",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cognito"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		pool, err := cognito.NewUserPool(ctx, "pool", &cognito.UserPoolArgs{
 * 			Name: pulumi.String("pool"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cognito.NewUserPoolClient(ctx, "userpool_client", &cognito.UserPoolClientArgs{
 * 			Name:       pulumi.String("client"),
 * 			UserPoolId: pool.ID(),
 * 			CallbackUrls: pulumi.StringArray{
 * 				pulumi.String("https://example.com"),
 * 			},
 * 			AllowedOauthFlowsUserPoolClient: pulumi.Bool(true),
 * 			AllowedOauthFlows: pulumi.StringArray{
 * 				pulumi.String("code"),
 * 				pulumi.String("implicit"),
 * 			},
 * 			AllowedOauthScopes: pulumi.StringArray{
 * 				pulumi.String("email"),
 * 				pulumi.String("openid"),
 * 			},
 * 			SupportedIdentityProviders: pulumi.StringArray{
 * 				pulumi.String("COGNITO"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolArgs;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolClientArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var pool = new UserPool("pool", UserPoolArgs.builder()
 *             .name("pool")
 *             .build());
 *         var userpoolClient = new UserPoolClient("userpoolClient", UserPoolClientArgs.builder()
 *             .name("client")
 *             .userPoolId(pool.id())
 *             .callbackUrls("https://example.com")
 *             .allowedOauthFlowsUserPoolClient(true)
 *             .allowedOauthFlows(
 *                 "code",
 *                 "implicit")
 *             .allowedOauthScopes(
 *                 "email",
 *                 "openid")
 *             .supportedIdentityProviders("COGNITO")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   userpoolClient:
 *     type: aws:cognito:UserPoolClient
 *     name: userpool_client
 *     properties:
 *       name: client
 *       userPoolId: ${pool.id}
 *       callbackUrls:
 *         - https://example.com
 *       allowedOauthFlowsUserPoolClient: true
 *       allowedOauthFlows:
 *         - code
 *         - implicit
 *       allowedOauthScopes:
 *         - email
 *         - openid
 *       supportedIdentityProviders:
 *         - COGNITO
 *   pool:
 *     type: aws:cognito:UserPool
 *     properties:
 *       name: pool
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Cognito User Pool Clients using the `id` of the Cognito User Pool, and the `id` of the Cognito User Pool Client. For example:
 * ```sh
 * $ pulumi import aws:cognito/userPoolClient:UserPoolClient client us-west-2_abc123/3ho4ek12345678909nh3fmhpko
 * ```
 * @property accessTokenValidity Time limit, between 5 minutes and 1 day, after which the access token is no longer valid and cannot be used.
 * By default, the unit is hours.
 * The unit can be overridden by a value in `token_validity_units.access_token`.
 * @property allowedOauthFlows List of allowed OAuth flows (code, implicit, client_credentials).
 * @property allowedOauthFlowsUserPoolClient Whether the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
 * @property allowedOauthScopes List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
 * @property analyticsConfiguration Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
 * @property authSessionValidity Amazon Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires. Valid values between `3` and `15`. Default value is `3`.
 * @property callbackUrls List of allowed callback URLs for the identity providers.
 * @property defaultRedirectUri Default redirect URI. Must be in the list of callback URLs.
 * @property enablePropagateAdditionalUserContextData Activates the propagation of additional user context data.
 * @property enableTokenRevocation Enables or disables token revocation.
 * @property explicitAuthFlows List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
 * @property generateSecret Should an application secret be generated.
 * @property idTokenValidity Time limit, between 5 minutes and 1 day, after which the ID token is no longer valid and cannot be used.
 * By default, the unit is hours.
 * The unit can be overridden by a value in `token_validity_units.id_token`.
 * @property logoutUrls List of allowed logout URLs for the identity providers.
 * @property name Name of the application client.
 * @property preventUserExistenceErrors Choose which errors and responses are returned by Cognito APIs during authentication, account confirmation, and password recovery when the user does not exist in the user pool. When set to `ENABLED` and the user does not exist, authentication returns an error indicating either the username or password was incorrect, and account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to `LEGACY`, those APIs will return a `UserNotFoundException` exception if the user does not exist in the user pool.
 * @property readAttributes List of user pool attributes the application client can read from.
 * @property refreshTokenValidity Time limit, between 60 minutes and 10 years, after which the refresh token is no longer valid and cannot be used.
 * By default, the unit is days.
 * The unit can be overridden by a value in `token_validity_units.refresh_token`.
 * @property supportedIdentityProviders List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
 * @property tokenValidityUnits Configuration block for units in which the validity times are represented in. Detailed below.
 * @property userPoolId User pool the client belongs to.
 * The following arguments are optional:
 * @property writeAttributes List of user pool attributes the application client can write to.
 */
public data class UserPoolClientArgs(
    public val accessTokenValidity: Output<Int>? = null,
    public val allowedOauthFlows: Output<List<String>>? = null,
    public val allowedOauthFlowsUserPoolClient: Output<Boolean>? = null,
    public val allowedOauthScopes: Output<List<String>>? = null,
    public val analyticsConfiguration: Output<UserPoolClientAnalyticsConfigurationArgs>? = null,
    public val authSessionValidity: Output<Int>? = null,
    public val callbackUrls: Output<List<String>>? = null,
    public val defaultRedirectUri: Output<String>? = null,
    public val enablePropagateAdditionalUserContextData: Output<Boolean>? = null,
    public val enableTokenRevocation: Output<Boolean>? = null,
    public val explicitAuthFlows: Output<List<String>>? = null,
    public val generateSecret: Output<Boolean>? = null,
    public val idTokenValidity: Output<Int>? = null,
    public val logoutUrls: Output<List<String>>? = null,
    public val name: Output<String>? = null,
    public val preventUserExistenceErrors: Output<String>? = null,
    public val readAttributes: Output<List<String>>? = null,
    public val refreshTokenValidity: Output<Int>? = null,
    public val supportedIdentityProviders: Output<List<String>>? = null,
    public val tokenValidityUnits: Output<UserPoolClientTokenValidityUnitsArgs>? = null,
    public val userPoolId: Output<String>? = null,
    public val writeAttributes: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.cognito.UserPoolClientArgs> {
    override fun toJava(): com.pulumi.aws.cognito.UserPoolClientArgs =
        com.pulumi.aws.cognito.UserPoolClientArgs.builder()
            .accessTokenValidity(accessTokenValidity?.applyValue({ args0 -> args0 }))
            .allowedOauthFlows(allowedOauthFlows?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .allowedOauthFlowsUserPoolClient(allowedOauthFlowsUserPoolClient?.applyValue({ args0 -> args0 }))
            .allowedOauthScopes(allowedOauthScopes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .analyticsConfiguration(
                analyticsConfiguration?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .authSessionValidity(authSessionValidity?.applyValue({ args0 -> args0 }))
            .callbackUrls(callbackUrls?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .defaultRedirectUri(defaultRedirectUri?.applyValue({ args0 -> args0 }))
            .enablePropagateAdditionalUserContextData(
                enablePropagateAdditionalUserContextData?.applyValue({ args0 ->
                    args0
                }),
            )
            .enableTokenRevocation(enableTokenRevocation?.applyValue({ args0 -> args0 }))
            .explicitAuthFlows(explicitAuthFlows?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .generateSecret(generateSecret?.applyValue({ args0 -> args0 }))
            .idTokenValidity(idTokenValidity?.applyValue({ args0 -> args0 }))
            .logoutUrls(logoutUrls?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .name(name?.applyValue({ args0 -> args0 }))
            .preventUserExistenceErrors(preventUserExistenceErrors?.applyValue({ args0 -> args0 }))
            .readAttributes(readAttributes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .refreshTokenValidity(refreshTokenValidity?.applyValue({ args0 -> args0 }))
            .supportedIdentityProviders(
                supportedIdentityProviders?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tokenValidityUnits(
                tokenValidityUnits?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .userPoolId(userPoolId?.applyValue({ args0 -> args0 }))
            .writeAttributes(writeAttributes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) })).build()
}

/**
 * Builder for [UserPoolClientArgs].
 */
@PulumiTagMarker
public class UserPoolClientArgsBuilder internal constructor() {
    private var accessTokenValidity: Output<Int>? = null

    private var allowedOauthFlows: Output<List<String>>? = null

    private var allowedOauthFlowsUserPoolClient: Output<Boolean>? = null

    private var allowedOauthScopes: Output<List<String>>? = null

    private var analyticsConfiguration: Output<UserPoolClientAnalyticsConfigurationArgs>? = null

    private var authSessionValidity: Output<Int>? = null

    private var callbackUrls: Output<List<String>>? = null

    private var defaultRedirectUri: Output<String>? = null

    private var enablePropagateAdditionalUserContextData: Output<Boolean>? = null

    private var enableTokenRevocation: Output<Boolean>? = null

    private var explicitAuthFlows: Output<List<String>>? = null

    private var generateSecret: Output<Boolean>? = null

    private var idTokenValidity: Output<Int>? = null

    private var logoutUrls: Output<List<String>>? = null

    private var name: Output<String>? = null

    private var preventUserExistenceErrors: Output<String>? = null

    private var readAttributes: Output<List<String>>? = null

    private var refreshTokenValidity: Output<Int>? = null

    private var supportedIdentityProviders: Output<List<String>>? = null

    private var tokenValidityUnits: Output<UserPoolClientTokenValidityUnitsArgs>? = null

    private var userPoolId: Output<String>? = null

    private var writeAttributes: Output<List<String>>? = null

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the access token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.access_token`.
     */
    @JvmName("bgfkjnldxoslpnah")
    public suspend fun accessTokenValidity(`value`: Output<Int>) {
        this.accessTokenValidity = value
    }

    /**
     * @param value List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("gnxprbpktyoygwkt")
    public suspend fun allowedOauthFlows(`value`: Output<List<String>>) {
        this.allowedOauthFlows = value
    }

    @JvmName("oucekqybthqknbkh")
    public suspend fun allowedOauthFlows(vararg values: Output<String>) {
        this.allowedOauthFlows = Output.all(values.asList())
    }

    /**
     * @param values List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("vtetepebmmeopkqh")
    public suspend fun allowedOauthFlows(values: List<Output<String>>) {
        this.allowedOauthFlows = Output.all(values)
    }

    /**
     * @param value Whether the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
     */
    @JvmName("bepqmxmcfrpochvl")
    public suspend fun allowedOauthFlowsUserPoolClient(`value`: Output<Boolean>) {
        this.allowedOauthFlowsUserPoolClient = value
    }

    /**
     * @param value List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("qdjqjwxgcpdfuddq")
    public suspend fun allowedOauthScopes(`value`: Output<List<String>>) {
        this.allowedOauthScopes = value
    }

    @JvmName("usnwrlcuuawaduir")
    public suspend fun allowedOauthScopes(vararg values: Output<String>) {
        this.allowedOauthScopes = Output.all(values.asList())
    }

    /**
     * @param values List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("xjrjqndegkjnbdix")
    public suspend fun allowedOauthScopes(values: List<Output<String>>) {
        this.allowedOauthScopes = Output.all(values)
    }

    /**
     * @param value Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
     */
    @JvmName("plndttyeyxdoqukf")
    public suspend fun analyticsConfiguration(`value`: Output<UserPoolClientAnalyticsConfigurationArgs>) {
        this.analyticsConfiguration = value
    }

    /**
     * @param value Amazon Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires. Valid values between `3` and `15`. Default value is `3`.
     */
    @JvmName("cunumfpfsvefbxgq")
    public suspend fun authSessionValidity(`value`: Output<Int>) {
        this.authSessionValidity = value
    }

    /**
     * @param value List of allowed callback URLs for the identity providers.
     */
    @JvmName("amwuslrrslbtjstk")
    public suspend fun callbackUrls(`value`: Output<List<String>>) {
        this.callbackUrls = value
    }

    @JvmName("bbuljtogefouswto")
    public suspend fun callbackUrls(vararg values: Output<String>) {
        this.callbackUrls = Output.all(values.asList())
    }

    /**
     * @param values List of allowed callback URLs for the identity providers.
     */
    @JvmName("phqdawljeauusvsv")
    public suspend fun callbackUrls(values: List<Output<String>>) {
        this.callbackUrls = Output.all(values)
    }

    /**
     * @param value Default redirect URI. Must be in the list of callback URLs.
     */
    @JvmName("yyefmdcbyfmcibsq")
    public suspend fun defaultRedirectUri(`value`: Output<String>) {
        this.defaultRedirectUri = value
    }

    /**
     * @param value Activates the propagation of additional user context data.
     */
    @JvmName("vvigmoeybxwliefr")
    public suspend fun enablePropagateAdditionalUserContextData(`value`: Output<Boolean>) {
        this.enablePropagateAdditionalUserContextData = value
    }

    /**
     * @param value Enables or disables token revocation.
     */
    @JvmName("ngxfsabbmypwcgbd")
    public suspend fun enableTokenRevocation(`value`: Output<Boolean>) {
        this.enableTokenRevocation = value
    }

    /**
     * @param value List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("bsdohscvdkbdkbdx")
    public suspend fun explicitAuthFlows(`value`: Output<List<String>>) {
        this.explicitAuthFlows = value
    }

    @JvmName("erdxpleuouhnkjob")
    public suspend fun explicitAuthFlows(vararg values: Output<String>) {
        this.explicitAuthFlows = Output.all(values.asList())
    }

    /**
     * @param values List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("otanyihcdqvpihxl")
    public suspend fun explicitAuthFlows(values: List<Output<String>>) {
        this.explicitAuthFlows = Output.all(values)
    }

    /**
     * @param value Should an application secret be generated.
     */
    @JvmName("iaynaxgrlmurffcv")
    public suspend fun generateSecret(`value`: Output<Boolean>) {
        this.generateSecret = value
    }

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the ID token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.id_token`.
     */
    @JvmName("lwhquwlgvtopefra")
    public suspend fun idTokenValidity(`value`: Output<Int>) {
        this.idTokenValidity = value
    }

    /**
     * @param value List of allowed logout URLs for the identity providers.
     */
    @JvmName("dvulorpvgjnppepu")
    public suspend fun logoutUrls(`value`: Output<List<String>>) {
        this.logoutUrls = value
    }

    @JvmName("swvseotduygajtoo")
    public suspend fun logoutUrls(vararg values: Output<String>) {
        this.logoutUrls = Output.all(values.asList())
    }

    /**
     * @param values List of allowed logout URLs for the identity providers.
     */
    @JvmName("vuestajtwmnjbajw")
    public suspend fun logoutUrls(values: List<Output<String>>) {
        this.logoutUrls = Output.all(values)
    }

    /**
     * @param value Name of the application client.
     */
    @JvmName("vgifmphkyybkbljc")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Choose which errors and responses are returned by Cognito APIs during authentication, account confirmation, and password recovery when the user does not exist in the user pool. When set to `ENABLED` and the user does not exist, authentication returns an error indicating either the username or password was incorrect, and account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to `LEGACY`, those APIs will return a `UserNotFoundException` exception if the user does not exist in the user pool.
     */
    @JvmName("dcssohfghfnqbtfd")
    public suspend fun preventUserExistenceErrors(`value`: Output<String>) {
        this.preventUserExistenceErrors = value
    }

    /**
     * @param value List of user pool attributes the application client can read from.
     */
    @JvmName("tlqrlkdfppevxroh")
    public suspend fun readAttributes(`value`: Output<List<String>>) {
        this.readAttributes = value
    }

    @JvmName("bqfawbjiksuajwne")
    public suspend fun readAttributes(vararg values: Output<String>) {
        this.readAttributes = Output.all(values.asList())
    }

    /**
     * @param values List of user pool attributes the application client can read from.
     */
    @JvmName("wguimltkrwsosumw")
    public suspend fun readAttributes(values: List<Output<String>>) {
        this.readAttributes = Output.all(values)
    }

    /**
     * @param value Time limit, between 60 minutes and 10 years, after which the refresh token is no longer valid and cannot be used.
     * By default, the unit is days.
     * The unit can be overridden by a value in `token_validity_units.refresh_token`.
     */
    @JvmName("ktnrmadbvqetxbxn")
    public suspend fun refreshTokenValidity(`value`: Output<Int>) {
        this.refreshTokenValidity = value
    }

    /**
     * @param value List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("nvhetgevjxejjaco")
    public suspend fun supportedIdentityProviders(`value`: Output<List<String>>) {
        this.supportedIdentityProviders = value
    }

    @JvmName("kbynmytsrdwsydqk")
    public suspend fun supportedIdentityProviders(vararg values: Output<String>) {
        this.supportedIdentityProviders = Output.all(values.asList())
    }

    /**
     * @param values List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("eqqodvidlavqqneg")
    public suspend fun supportedIdentityProviders(values: List<Output<String>>) {
        this.supportedIdentityProviders = Output.all(values)
    }

    /**
     * @param value Configuration block for units in which the validity times are represented in. Detailed below.
     */
    @JvmName("mssdpttcgpotihrs")
    public suspend fun tokenValidityUnits(`value`: Output<UserPoolClientTokenValidityUnitsArgs>) {
        this.tokenValidityUnits = value
    }

    /**
     * @param value User pool the client belongs to.
     * The following arguments are optional:
     */
    @JvmName("cogagatyuuxwjpsc")
    public suspend fun userPoolId(`value`: Output<String>) {
        this.userPoolId = value
    }

    /**
     * @param value List of user pool attributes the application client can write to.
     */
    @JvmName("yfhdoehysiexaogl")
    public suspend fun writeAttributes(`value`: Output<List<String>>) {
        this.writeAttributes = value
    }

    @JvmName("bpminmyadmlwkjqd")
    public suspend fun writeAttributes(vararg values: Output<String>) {
        this.writeAttributes = Output.all(values.asList())
    }

    /**
     * @param values List of user pool attributes the application client can write to.
     */
    @JvmName("wufnqfyxkddkckae")
    public suspend fun writeAttributes(values: List<Output<String>>) {
        this.writeAttributes = Output.all(values)
    }

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the access token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.access_token`.
     */
    @JvmName("yabgwirelmwnufnw")
    public suspend fun accessTokenValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessTokenValidity = mapped
    }

    /**
     * @param value List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("hfjbvnfuwdbebwwo")
    public suspend fun allowedOauthFlows(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowedOauthFlows = mapped
    }

    /**
     * @param values List of allowed OAuth flows (code, implicit, client_credentials).
     */
    @JvmName("kmddsfqpuxdfwrek")
    public suspend fun allowedOauthFlows(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.allowedOauthFlows = mapped
    }

    /**
     * @param value Whether the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
     */
    @JvmName("tlogjhtwkbilists")
    public suspend fun allowedOauthFlowsUserPoolClient(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowedOauthFlowsUserPoolClient = mapped
    }

    /**
     * @param value List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("hkhgyclxeovhvgkp")
    public suspend fun allowedOauthScopes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowedOauthScopes = mapped
    }

    /**
     * @param values List of allowed OAuth scopes (phone, email, openid, profile, and aws.cognito.signin.user.admin).
     */
    @JvmName("vhaleyicxtbgiqok")
    public suspend fun allowedOauthScopes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.allowedOauthScopes = mapped
    }

    /**
     * @param value Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
     */
    @JvmName("puxuijedvwhvjxeb")
    public suspend fun analyticsConfiguration(`value`: UserPoolClientAnalyticsConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.analyticsConfiguration = mapped
    }

    /**
     * @param argument Configuration block for Amazon Pinpoint analytics for collecting metrics for this user pool. Detailed below.
     */
    @JvmName("vwrtihpwgaliunkk")
    public suspend fun analyticsConfiguration(argument: suspend UserPoolClientAnalyticsConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = UserPoolClientAnalyticsConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.analyticsConfiguration = mapped
    }

    /**
     * @param value Amazon Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires. Valid values between `3` and `15`. Default value is `3`.
     */
    @JvmName("qyfwuiqrufkthnwk")
    public suspend fun authSessionValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authSessionValidity = mapped
    }

    /**
     * @param value List of allowed callback URLs for the identity providers.
     */
    @JvmName("dcmtdkldceeywigy")
    public suspend fun callbackUrls(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.callbackUrls = mapped
    }

    /**
     * @param values List of allowed callback URLs for the identity providers.
     */
    @JvmName("srkbmwvhrcsjhdfp")
    public suspend fun callbackUrls(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.callbackUrls = mapped
    }

    /**
     * @param value Default redirect URI. Must be in the list of callback URLs.
     */
    @JvmName("quffufiggvbwggtj")
    public suspend fun defaultRedirectUri(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultRedirectUri = mapped
    }

    /**
     * @param value Activates the propagation of additional user context data.
     */
    @JvmName("qjtbwhkgsnbbixrg")
    public suspend fun enablePropagateAdditionalUserContextData(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enablePropagateAdditionalUserContextData = mapped
    }

    /**
     * @param value Enables or disables token revocation.
     */
    @JvmName("tqytplnjdncvdijt")
    public suspend fun enableTokenRevocation(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableTokenRevocation = mapped
    }

    /**
     * @param value List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("dkptmtrvayrlyllt")
    public suspend fun explicitAuthFlows(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.explicitAuthFlows = mapped
    }

    /**
     * @param values List of authentication flows (ADMIN_NO_SRP_AUTH, CUSTOM_AUTH_FLOW_ONLY, USER_PASSWORD_AUTH, ALLOW_ADMIN_USER_PASSWORD_AUTH, ALLOW_CUSTOM_AUTH, ALLOW_USER_PASSWORD_AUTH, ALLOW_USER_SRP_AUTH, ALLOW_REFRESH_TOKEN_AUTH).
     */
    @JvmName("mypylqlmugctmgrt")
    public suspend fun explicitAuthFlows(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.explicitAuthFlows = mapped
    }

    /**
     * @param value Should an application secret be generated.
     */
    @JvmName("mvlhxbmqgwwceoxe")
    public suspend fun generateSecret(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.generateSecret = mapped
    }

    /**
     * @param value Time limit, between 5 minutes and 1 day, after which the ID token is no longer valid and cannot be used.
     * By default, the unit is hours.
     * The unit can be overridden by a value in `token_validity_units.id_token`.
     */
    @JvmName("cdqgovpqqoemqats")
    public suspend fun idTokenValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.idTokenValidity = mapped
    }

    /**
     * @param value List of allowed logout URLs for the identity providers.
     */
    @JvmName("bbtpmiucguauhhuw")
    public suspend fun logoutUrls(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logoutUrls = mapped
    }

    /**
     * @param values List of allowed logout URLs for the identity providers.
     */
    @JvmName("rvgbyqiplsvuewkg")
    public suspend fun logoutUrls(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.logoutUrls = mapped
    }

    /**
     * @param value Name of the application client.
     */
    @JvmName("bvtpsrugvqrubbma")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Choose which errors and responses are returned by Cognito APIs during authentication, account confirmation, and password recovery when the user does not exist in the user pool. When set to `ENABLED` and the user does not exist, authentication returns an error indicating either the username or password was incorrect, and account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to `LEGACY`, those APIs will return a `UserNotFoundException` exception if the user does not exist in the user pool.
     */
    @JvmName("cosojhjshjuvpikq")
    public suspend fun preventUserExistenceErrors(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preventUserExistenceErrors = mapped
    }

    /**
     * @param value List of user pool attributes the application client can read from.
     */
    @JvmName("iavpuchqfsopkeiv")
    public suspend fun readAttributes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.readAttributes = mapped
    }

    /**
     * @param values List of user pool attributes the application client can read from.
     */
    @JvmName("hitlbnqpwmpdbptp")
    public suspend fun readAttributes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.readAttributes = mapped
    }

    /**
     * @param value Time limit, between 60 minutes and 10 years, after which the refresh token is no longer valid and cannot be used.
     * By default, the unit is days.
     * The unit can be overridden by a value in `token_validity_units.refresh_token`.
     */
    @JvmName("oqcfkdpuclekcdvc")
    public suspend fun refreshTokenValidity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.refreshTokenValidity = mapped
    }

    /**
     * @param value List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("oopebdikbadvvosy")
    public suspend fun supportedIdentityProviders(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.supportedIdentityProviders = mapped
    }

    /**
     * @param values List of provider names for the identity providers that are supported on this client. Uses the `provider_name` attribute of `aws.cognito.IdentityProvider` resource(s), or the equivalent string(s).
     */
    @JvmName("ouciyrwdpkayctva")
    public suspend fun supportedIdentityProviders(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.supportedIdentityProviders = mapped
    }

    /**
     * @param value Configuration block for units in which the validity times are represented in. Detailed below.
     */
    @JvmName("wrspyirxfddvpebl")
    public suspend fun tokenValidityUnits(`value`: UserPoolClientTokenValidityUnitsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tokenValidityUnits = mapped
    }

    /**
     * @param argument Configuration block for units in which the validity times are represented in. Detailed below.
     */
    @JvmName("qdctfnuyjwixhfws")
    public suspend fun tokenValidityUnits(argument: suspend UserPoolClientTokenValidityUnitsArgsBuilder.() -> Unit) {
        val toBeMapped = UserPoolClientTokenValidityUnitsArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.tokenValidityUnits = mapped
    }

    /**
     * @param value User pool the client belongs to.
     * The following arguments are optional:
     */
    @JvmName("osgbtvvxdkewwffx")
    public suspend fun userPoolId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userPoolId = mapped
    }

    /**
     * @param value List of user pool attributes the application client can write to.
     */
    @JvmName("baekrnhjfugpinef")
    public suspend fun writeAttributes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.writeAttributes = mapped
    }

    /**
     * @param values List of user pool attributes the application client can write to.
     */
    @JvmName("bwywevedpxhhnqcv")
    public suspend fun writeAttributes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.writeAttributes = mapped
    }

    internal fun build(): UserPoolClientArgs = UserPoolClientArgs(
        accessTokenValidity = accessTokenValidity,
        allowedOauthFlows = allowedOauthFlows,
        allowedOauthFlowsUserPoolClient = allowedOauthFlowsUserPoolClient,
        allowedOauthScopes = allowedOauthScopes,
        analyticsConfiguration = analyticsConfiguration,
        authSessionValidity = authSessionValidity,
        callbackUrls = callbackUrls,
        defaultRedirectUri = defaultRedirectUri,
        enablePropagateAdditionalUserContextData = enablePropagateAdditionalUserContextData,
        enableTokenRevocation = enableTokenRevocation,
        explicitAuthFlows = explicitAuthFlows,
        generateSecret = generateSecret,
        idTokenValidity = idTokenValidity,
        logoutUrls = logoutUrls,
        name = name,
        preventUserExistenceErrors = preventUserExistenceErrors,
        readAttributes = readAttributes,
        refreshTokenValidity = refreshTokenValidity,
        supportedIdentityProviders = supportedIdentityProviders,
        tokenValidityUnits = tokenValidityUnits,
        userPoolId = userPoolId,
        writeAttributes = writeAttributes,
    )
}
