@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.datasync.kotlin

import com.pulumi.aws.datasync.AgentArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an AWS DataSync Agent deployed on premises.
 * > **NOTE:** One of `activation_key` or `ip_address` must be provided for resource creation (agent activation). Neither is required for resource import. If using `ip_address`, this provider must be able to make an HTTP (port 80) GET request to the specified IP address from where it is running. The agent will turn off that HTTP server after activation.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.datasync.Agent("example", {
 *     ipAddress: "1.2.3.4",
 *     name: "example",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.datasync.Agent("example",
 *     ip_address="1.2.3.4",
 *     name="example")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.DataSync.Agent("example", new()
 *     {
 *         IpAddress = "1.2.3.4",
 *         Name = "example",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/datasync"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := datasync.NewAgent(ctx, "example", &datasync.AgentArgs{
 * 			IpAddress: pulumi.String("1.2.3.4"),
 * 			Name:      pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.datasync.Agent;
 * import com.pulumi.aws.datasync.AgentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Agent("example", AgentArgs.builder()
 *             .ipAddress("1.2.3.4")
 *             .name("example")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:datasync:Agent
 *     properties:
 *       ipAddress: 1.2.3.4
 *       name: example
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With VPC Endpoints
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getRegion({});
 * const exampleVpcEndpoint = new aws.ec2.VpcEndpoint("example", {
 *     serviceName: current.then(current => `com.amazonaws.${current.name}.datasync`),
 *     vpcId: exampleAwsVpc.id,
 *     securityGroupIds: [exampleAwsSecurityGroup&#46;id],
 *     subnetIds: [exampleAwsSubnet&#46;id],
 *     vpcEndpointType: "Interface",
 * });
 * const example = aws.ec2.getNetworkInterfaceOutput({
 *     id: exampleVpcEndpoint.networkInterfaceIds[0],
 * });
 * const exampleAgent = new aws.datasync.Agent("example", {
 *     ipAddress: "1.2.3.4",
 *     securityGroupArns: [exampleAwsSecurityGroup&#46;arn],
 *     subnetArns: [exampleAwsSubnet&#46;arn],
 *     vpcEndpointId: exampleVpcEndpoint.id,
 *     privateLinkEndpoint: example.apply(example => example.privateIp),
 *     name: "example",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_region()
 * example_vpc_endpoint = aws.ec2.VpcEndpoint("example",
 *     service_name=f"com.amazonaws.{current.name}.datasync",
 *     vpc_id=example_aws_vpc["id"],
 *     security_group_ids=[example_aws_security_group["id"]],
 *     subnet_ids=[example_aws_subnet["id"]],
 *     vpc_endpoint_type="Interface")
 * example = aws.ec2.get_network_interface_output(id=example_vpc_endpoint.network_interface_ids[0])
 * example_agent = aws.datasync.Agent("example",
 *     ip_address="1.2.3.4",
 *     security_group_arns=[example_aws_security_group["arn"]],
 *     subnet_arns=[example_aws_subnet["arn"]],
 *     vpc_endpoint_id=example_vpc_endpoint.id,
 *     private_link_endpoint=example.private_ip,
 *     name="example")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetRegion.Invoke();
 *     var exampleVpcEndpoint = new Aws.Ec2.VpcEndpoint("example", new()
 *     {
 *         ServiceName = $"com.amazonaws.{current.Apply(getRegionResult => getRegionResult.Name)}.datasync",
 *         VpcId = exampleAwsVpc.Id,
 *         SecurityGroupIds = new[]
 *         {
 *             exampleAwsSecurityGroup.Id,
 *         },
 *         SubnetIds = new[]
 *         {
 *             exampleAwsSubnet.Id,
 *         },
 *         VpcEndpointType = "Interface",
 *     });
 *     var example = Aws.Ec2.GetNetworkInterface.Invoke(new()
 *     {
 *         Id = exampleVpcEndpoint.NetworkInterfaceIds[0],
 *     });
 *     var exampleAgent = new Aws.DataSync.Agent("example", new()
 *     {
 *         IpAddress = "1.2.3.4",
 *         SecurityGroupArns = new[]
 *         {
 *             exampleAwsSecurityGroup.Arn,
 *         },
 *         SubnetArns = new[]
 *         {
 *             exampleAwsSubnet.Arn,
 *         },
 *         VpcEndpointId = exampleVpcEndpoint.Id,
 *         PrivateLinkEndpoint = example.Apply(getNetworkInterfaceResult => getNetworkInterfaceResult.PrivateIp),
 *         Name = "example",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/datasync"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleVpcEndpoint, err := ec2.NewVpcEndpoint(ctx, "example", &ec2.VpcEndpointArgs{
 * 			ServiceName: pulumi.Sprintf("com.amazonaws.%v.datasync", current.Name),
 * 			VpcId:       pulumi.Any(exampleAwsVpc.Id),
 * 			SecurityGroupIds: pulumi.StringArray{
 * 				exampleAwsSecurityGroup.Id,
 * 			},
 * 			SubnetIds: pulumi.StringArray{
 * 				exampleAwsSubnet.Id,
 * 			},
 * 			VpcEndpointType: pulumi.String("Interface"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example := ec2.LookupNetworkInterfaceOutput(ctx, ec2.GetNetworkInterfaceOutputArgs{
 * 			Id: exampleVpcEndpoint.NetworkInterfaceIds.ApplyT(func(networkInterfaceIds []string) (string, error) {
 * 				return networkInterfaceIds[0], nil
 * 			}).(pulumi.StringOutput),
 * 		}, nil)
 * 		_, err = datasync.NewAgent(ctx, "example", &datasync.AgentArgs{
 * 			IpAddress: pulumi.String("1.2.3.4"),
 * 			SecurityGroupArns: pulumi.StringArray{
 * 				exampleAwsSecurityGroup.Arn,
 * 			},
 * 			SubnetArns: pulumi.StringArray{
 * 				exampleAwsSubnet.Arn,
 * 			},
 * 			VpcEndpointId: exampleVpcEndpoint.ID(),
 * 			PrivateLinkEndpoint: pulumi.String(example.ApplyT(func(example ec2.GetNetworkInterfaceResult) (*string, error) {
 * 				return &example.PrivateIp, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 			Name: pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.ec2.VpcEndpoint;
 * import com.pulumi.aws.ec2.VpcEndpointArgs;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetNetworkInterfaceArgs;
 * import com.pulumi.aws.datasync.Agent;
 * import com.pulumi.aws.datasync.AgentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getRegion();
 *         var exampleVpcEndpoint = new VpcEndpoint("exampleVpcEndpoint", VpcEndpointArgs.builder()
 *             .serviceName(String.format("com.amazonaws.%s.datasync", current.applyValue(getRegionResult -> getRegionResult.name())))
 *             .vpcId(exampleAwsVpc.id())
 *             .securityGroupIds(exampleAwsSecurityGroup.id())
 *             .subnetIds(exampleAwsSubnet.id())
 *             .vpcEndpointType("Interface")
 *             .build());
 *         final var example = Ec2Functions.getNetworkInterface(GetNetworkInterfaceArgs.builder()
 *             .id(exampleVpcEndpoint.networkInterfaceIds().applyValue(networkInterfaceIds -> networkInterfaceIds[0]))
 *             .build());
 *         var exampleAgent = new Agent("exampleAgent", AgentArgs.builder()
 *             .ipAddress("1.2.3.4")
 *             .securityGroupArns(exampleAwsSecurityGroup.arn())
 *             .subnetArns(exampleAwsSubnet.arn())
 *             .vpcEndpointId(exampleVpcEndpoint.id())
 *             .privateLinkEndpoint(example.applyValue(getNetworkInterfaceResult -> getNetworkInterfaceResult).applyValue(example -> example.applyValue(getNetworkInterfaceResult -> getNetworkInterfaceResult.privateIp())))
 *             .name("example")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleAgent:
 *     type: aws:datasync:Agent
 *     name: example
 *     properties:
 *       ipAddress: 1.2.3.4
 *       securityGroupArns:
 *         - ${exampleAwsSecurityGroup.arn}
 *       subnetArns:
 *         - ${exampleAwsSubnet.arn}
 *       vpcEndpointId: ${exampleVpcEndpoint.id}
 *       privateLinkEndpoint: ${example.privateIp}
 *       name: example
 *   exampleVpcEndpoint:
 *     type: aws:ec2:VpcEndpoint
 *     name: example
 *     properties:
 *       serviceName: com.amazonaws.${current.name}.datasync
 *       vpcId: ${exampleAwsVpc.id}
 *       securityGroupIds:
 *         - ${exampleAwsSecurityGroup.id}
 *       subnetIds:
 *         - ${exampleAwsSubnet.id}
 *       vpcEndpointType: Interface
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 *   example:
 *     fn::invoke:
 *       Function: aws:ec2:getNetworkInterface
 *       Arguments:
 *         id: ${exampleVpcEndpoint.networkInterfaceIds[0]}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import `aws_datasync_agent` using the DataSync Agent Amazon Resource Name (ARN). For example:
 * ```sh
 * $ pulumi import aws:datasync/agent:Agent example arn:aws:datasync:us-east-1:123456789012:agent/agent-12345678901234567
 * ```
 * @property activationKey DataSync Agent activation key during resource creation. Conflicts with `ip_address`. If an `ip_address` is provided instead, the provider will retrieve the `activation_key` as part of the resource creation.
 * @property ipAddress DataSync Agent IP address to retrieve activation key during resource creation. Conflicts with `activation_key`. DataSync Agent must be accessible on port 80 from where the provider is running.
 * @property name Name of the DataSync Agent.
 * @property privateLinkEndpoint The IP address of the VPC endpoint the agent should connect to when retrieving an activation key during resource creation. Conflicts with `activation_key`.
 * @property securityGroupArns The ARNs of the security groups used to protect your data transfer task subnets.
 * @property subnetArns The Amazon Resource Names (ARNs) of the subnets in which DataSync will create elastic network interfaces for each data transfer task.
 * @property tags Key-value pairs of resource tags to assign to the DataSync Agent. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcEndpointId The ID of the VPC (virtual private cloud) endpoint that the agent has access to.
 */
public data class AgentArgs(
    public val activationKey: Output<String>? = null,
    public val ipAddress: Output<String>? = null,
    public val name: Output<String>? = null,
    public val privateLinkEndpoint: Output<String>? = null,
    public val securityGroupArns: Output<List<String>>? = null,
    public val subnetArns: Output<List<String>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcEndpointId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.datasync.AgentArgs> {
    override fun toJava(): com.pulumi.aws.datasync.AgentArgs =
        com.pulumi.aws.datasync.AgentArgs.builder()
            .activationKey(activationKey?.applyValue({ args0 -> args0 }))
            .ipAddress(ipAddress?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .privateLinkEndpoint(privateLinkEndpoint?.applyValue({ args0 -> args0 }))
            .securityGroupArns(securityGroupArns?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .subnetArns(subnetArns?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .vpcEndpointId(vpcEndpointId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [AgentArgs].
 */
@PulumiTagMarker
public class AgentArgsBuilder internal constructor() {
    private var activationKey: Output<String>? = null

    private var ipAddress: Output<String>? = null

    private var name: Output<String>? = null

    private var privateLinkEndpoint: Output<String>? = null

    private var securityGroupArns: Output<List<String>>? = null

    private var subnetArns: Output<List<String>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcEndpointId: Output<String>? = null

    /**
     * @param value DataSync Agent activation key during resource creation. Conflicts with `ip_address`. If an `ip_address` is provided instead, the provider will retrieve the `activation_key` as part of the resource creation.
     */
    @JvmName("ovncnkkropoeilst")
    public suspend fun activationKey(`value`: Output<String>) {
        this.activationKey = value
    }

    /**
     * @param value DataSync Agent IP address to retrieve activation key during resource creation. Conflicts with `activation_key`. DataSync Agent must be accessible on port 80 from where the provider is running.
     */
    @JvmName("dihwbhalrhcehuwj")
    public suspend fun ipAddress(`value`: Output<String>) {
        this.ipAddress = value
    }

    /**
     * @param value Name of the DataSync Agent.
     */
    @JvmName("fusnfvvpbdootjby")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The IP address of the VPC endpoint the agent should connect to when retrieving an activation key during resource creation. Conflicts with `activation_key`.
     */
    @JvmName("ymifdklmbtemhpkx")
    public suspend fun privateLinkEndpoint(`value`: Output<String>) {
        this.privateLinkEndpoint = value
    }

    /**
     * @param value The ARNs of the security groups used to protect your data transfer task subnets.
     */
    @JvmName("epdsodrboujjtrgt")
    public suspend fun securityGroupArns(`value`: Output<List<String>>) {
        this.securityGroupArns = value
    }

    @JvmName("ewuvsahqgntpchsy")
    public suspend fun securityGroupArns(vararg values: Output<String>) {
        this.securityGroupArns = Output.all(values.asList())
    }

    /**
     * @param values The ARNs of the security groups used to protect your data transfer task subnets.
     */
    @JvmName("kvxqallbjsjhawpy")
    public suspend fun securityGroupArns(values: List<Output<String>>) {
        this.securityGroupArns = Output.all(values)
    }

    /**
     * @param value The Amazon Resource Names (ARNs) of the subnets in which DataSync will create elastic network interfaces for each data transfer task.
     */
    @JvmName("vpbnxctnbmkmvpry")
    public suspend fun subnetArns(`value`: Output<List<String>>) {
        this.subnetArns = value
    }

    @JvmName("jfyuoruwunfhalbo")
    public suspend fun subnetArns(vararg values: Output<String>) {
        this.subnetArns = Output.all(values.asList())
    }

    /**
     * @param values The Amazon Resource Names (ARNs) of the subnets in which DataSync will create elastic network interfaces for each data transfer task.
     */
    @JvmName("ijxvjwjvijhhemfa")
    public suspend fun subnetArns(values: List<Output<String>>) {
        this.subnetArns = Output.all(values)
    }

    /**
     * @param value Key-value pairs of resource tags to assign to the DataSync Agent. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("tobbndphhblftslf")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The ID of the VPC (virtual private cloud) endpoint that the agent has access to.
     */
    @JvmName("wlenyblitxuxlwbh")
    public suspend fun vpcEndpointId(`value`: Output<String>) {
        this.vpcEndpointId = value
    }

    /**
     * @param value DataSync Agent activation key during resource creation. Conflicts with `ip_address`. If an `ip_address` is provided instead, the provider will retrieve the `activation_key` as part of the resource creation.
     */
    @JvmName("hwcjhpnkjxnhwbre")
    public suspend fun activationKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.activationKey = mapped
    }

    /**
     * @param value DataSync Agent IP address to retrieve activation key during resource creation. Conflicts with `activation_key`. DataSync Agent must be accessible on port 80 from where the provider is running.
     */
    @JvmName("jjcolibanrciamfn")
    public suspend fun ipAddress(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipAddress = mapped
    }

    /**
     * @param value Name of the DataSync Agent.
     */
    @JvmName("kdnomxxurpxdbpmp")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The IP address of the VPC endpoint the agent should connect to when retrieving an activation key during resource creation. Conflicts with `activation_key`.
     */
    @JvmName("jgnofjbnoktyfbew")
    public suspend fun privateLinkEndpoint(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.privateLinkEndpoint = mapped
    }

    /**
     * @param value The ARNs of the security groups used to protect your data transfer task subnets.
     */
    @JvmName("lvvkihbkgmyexlps")
    public suspend fun securityGroupArns(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroupArns = mapped
    }

    /**
     * @param values The ARNs of the security groups used to protect your data transfer task subnets.
     */
    @JvmName("yspqjofxpphcqltw")
    public suspend fun securityGroupArns(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroupArns = mapped
    }

    /**
     * @param value The Amazon Resource Names (ARNs) of the subnets in which DataSync will create elastic network interfaces for each data transfer task.
     */
    @JvmName("ehlsmtqfqxijpein")
    public suspend fun subnetArns(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetArns = mapped
    }

    /**
     * @param values The Amazon Resource Names (ARNs) of the subnets in which DataSync will create elastic network interfaces for each data transfer task.
     */
    @JvmName("drvurtifwfvfnwse")
    public suspend fun subnetArns(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.subnetArns = mapped
    }

    /**
     * @param value Key-value pairs of resource tags to assign to the DataSync Agent. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("fnpybykvemuwlujj")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value pairs of resource tags to assign to the DataSync Agent. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("uubebswsdmvdmvyg")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The ID of the VPC (virtual private cloud) endpoint that the agent has access to.
     */
    @JvmName("tcnrktukrgyenjqk")
    public suspend fun vpcEndpointId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcEndpointId = mapped
    }

    internal fun build(): AgentArgs = AgentArgs(
        activationKey = activationKey,
        ipAddress = ipAddress,
        name = name,
        privateLinkEndpoint = privateLinkEndpoint,
        securityGroupArns = securityGroupArns,
        subnetArns = subnetArns,
        tags = tags,
        vpcEndpointId = vpcEndpointId,
    )
}
