@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.directconnect.kotlin

import com.pulumi.aws.directconnect.HostedPrivateVirtualInterfaceAccepterArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a resource to manage the accepter's side of a Direct Connect hosted private virtual interface.
 * This resource accepts ownership of a private virtual interface created by another AWS account.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const accepter = aws.getCallerIdentity({});
 * // Accepter's side of the VIF.
 * const vpnGw = new aws.ec2.VpnGateway("vpn_gw", {});
 * // Creator's side of the VIF
 * const creator = new aws.directconnect.HostedPrivateVirtualInterface("creator", {
 *     connectionId: "dxcon-zzzzzzzz",
 *     ownerAccountId: accepter.then(accepter => accepter.accountId),
 *     name: "vif-foo",
 *     vlan: 4094,
 *     addressFamily: "ipv4",
 *     bgpAsn: 65352,
 * }, {
 *     dependsOn: [vpnGw],
 * });
 * const accepterHostedPrivateVirtualInterfaceAccepter = new aws.directconnect.HostedPrivateVirtualInterfaceAccepter("accepter", {
 *     virtualInterfaceId: creator.id,
 *     vpnGatewayId: vpnGw.id,
 *     tags: {
 *         Side: "Accepter",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * accepter = aws.get_caller_identity()
 * # Accepter's side of the VIF.
 * vpn_gw = aws.ec2.VpnGateway("vpn_gw")
 * # Creator's side of the VIF
 * creator = aws.directconnect.HostedPrivateVirtualInterface("creator",
 *     connection_id="dxcon-zzzzzzzz",
 *     owner_account_id=accepter.account_id,
 *     name="vif-foo",
 *     vlan=4094,
 *     address_family="ipv4",
 *     bgp_asn=65352,
 *     opts = pulumi.ResourceOptions(depends_on=[vpn_gw]))
 * accepter_hosted_private_virtual_interface_accepter = aws.directconnect.HostedPrivateVirtualInterfaceAccepter("accepter",
 *     virtual_interface_id=creator.id,
 *     vpn_gateway_id=vpn_gw.id,
 *     tags={
 *         "Side": "Accepter",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var accepter = Aws.GetCallerIdentity.Invoke();
 *     // Accepter's side of the VIF.
 *     var vpnGw = new Aws.Ec2.VpnGateway("vpn_gw");
 *     // Creator's side of the VIF
 *     var creator = new Aws.DirectConnect.HostedPrivateVirtualInterface("creator", new()
 *     {
 *         ConnectionId = "dxcon-zzzzzzzz",
 *         OwnerAccountId = accepter.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId),
 *         Name = "vif-foo",
 *         Vlan = 4094,
 *         AddressFamily = "ipv4",
 *         BgpAsn = 65352,
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             vpnGw,
 *         },
 *     });
 *     var accepterHostedPrivateVirtualInterfaceAccepter = new Aws.DirectConnect.HostedPrivateVirtualInterfaceAccepter("accepter", new()
 *     {
 *         VirtualInterfaceId = creator.Id,
 *         VpnGatewayId = vpnGw.Id,
 *         Tags =
 *         {
 *             { "Side", "Accepter" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/directconnect"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		accepter, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Accepter's side of the VIF.
 * 		vpnGw, err := ec2.NewVpnGateway(ctx, "vpn_gw", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Creator's side of the VIF
 * 		creator, err := directconnect.NewHostedPrivateVirtualInterface(ctx, "creator", &directconnect.HostedPrivateVirtualInterfaceArgs{
 * 			ConnectionId:   pulumi.String("dxcon-zzzzzzzz"),
 * 			OwnerAccountId: pulumi.String(accepter.AccountId),
 * 			Name:           pulumi.String("vif-foo"),
 * 			Vlan:           pulumi.Int(4094),
 * 			AddressFamily:  pulumi.String("ipv4"),
 * 			BgpAsn:         pulumi.Int(65352),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			vpnGw,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = directconnect.NewHostedPrivateVirtualInterfaceAccepter(ctx, "accepter", &directconnect.HostedPrivateVirtualInterfaceAccepterArgs{
 * 			VirtualInterfaceId: creator.ID(),
 * 			VpnGatewayId:       vpnGw.ID(),
 * 			Tags: pulumi.StringMap{
 * 				"Side": pulumi.String("Accepter"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.ec2.VpnGateway;
 * import com.pulumi.aws.directconnect.HostedPrivateVirtualInterface;
 * import com.pulumi.aws.directconnect.HostedPrivateVirtualInterfaceArgs;
 * import com.pulumi.aws.directconnect.HostedPrivateVirtualInterfaceAccepter;
 * import com.pulumi.aws.directconnect.HostedPrivateVirtualInterfaceAccepterArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var accepter = AwsFunctions.getCallerIdentity();
 *         // Accepter's side of the VIF.
 *         var vpnGw = new VpnGateway("vpnGw");
 *         // Creator's side of the VIF
 *         var creator = new HostedPrivateVirtualInterface("creator", HostedPrivateVirtualInterfaceArgs.builder()
 *             .connectionId("dxcon-zzzzzzzz")
 *             .ownerAccountId(accepter.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()))
 *             .name("vif-foo")
 *             .vlan(4094)
 *             .addressFamily("ipv4")
 *             .bgpAsn(65352)
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(vpnGw)
 *                 .build());
 *         var accepterHostedPrivateVirtualInterfaceAccepter = new HostedPrivateVirtualInterfaceAccepter("accepterHostedPrivateVirtualInterfaceAccepter", HostedPrivateVirtualInterfaceAccepterArgs.builder()
 *             .virtualInterfaceId(creator.id())
 *             .vpnGatewayId(vpnGw.id())
 *             .tags(Map.of("Side", "Accepter"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # Creator's side of the VIF
 *   creator:
 *     type: aws:directconnect:HostedPrivateVirtualInterface
 *     properties:
 *       connectionId: dxcon-zzzzzzzz
 *       ownerAccountId: ${accepter.accountId}
 *       name: vif-foo
 *       vlan: 4094
 *       addressFamily: ipv4
 *       bgpAsn: 65352 # The aws_dx_hosted_private_virtual_interface
 *       #   # must be destroyed before the aws_vpn_gateway.
 *     options:
 *       dependson:
 *         - ${vpnGw}
 *   # Accepter's side of the VIF.
 *   vpnGw:
 *     type: aws:ec2:VpnGateway
 *     name: vpn_gw
 *   accepterHostedPrivateVirtualInterfaceAccepter:
 *     type: aws:directconnect:HostedPrivateVirtualInterfaceAccepter
 *     name: accepter
 *     properties:
 *       virtualInterfaceId: ${creator.id}
 *       vpnGatewayId: ${vpnGw.id}
 *       tags:
 *         Side: Accepter
 * variables:
 *   accepter:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Direct Connect hosted private virtual interfaces using the VIF `id`. For example:
 * ```sh
 * $ pulumi import aws:directconnect/hostedPrivateVirtualInterfaceAccepter:HostedPrivateVirtualInterfaceAccepter test dxvif-33cc44dd
 * ```
 * @property dxGatewayId The ID of the Direct Connect gateway to which to connect the virtual interface.
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property virtualInterfaceId The ID of the Direct Connect virtual interface to accept.
 * @property vpnGatewayId The ID of the virtual private gateway to which to connect the virtual interface.
 */
public data class HostedPrivateVirtualInterfaceAccepterArgs(
    public val dxGatewayId: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val virtualInterfaceId: Output<String>? = null,
    public val vpnGatewayId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.directconnect.HostedPrivateVirtualInterfaceAccepterArgs> {
    override fun toJava(): com.pulumi.aws.directconnect.HostedPrivateVirtualInterfaceAccepterArgs =
        com.pulumi.aws.directconnect.HostedPrivateVirtualInterfaceAccepterArgs.builder()
            .dxGatewayId(dxGatewayId?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .virtualInterfaceId(virtualInterfaceId?.applyValue({ args0 -> args0 }))
            .vpnGatewayId(vpnGatewayId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [HostedPrivateVirtualInterfaceAccepterArgs].
 */
@PulumiTagMarker
public class HostedPrivateVirtualInterfaceAccepterArgsBuilder internal constructor() {
    private var dxGatewayId: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var virtualInterfaceId: Output<String>? = null

    private var vpnGatewayId: Output<String>? = null

    /**
     * @param value The ID of the Direct Connect gateway to which to connect the virtual interface.
     */
    @JvmName("etpmlbxahnomkdlh")
    public suspend fun dxGatewayId(`value`: Output<String>) {
        this.dxGatewayId = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("thocufrnghkpqwqw")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The ID of the Direct Connect virtual interface to accept.
     */
    @JvmName("ykpbqbhftkyvsvmh")
    public suspend fun virtualInterfaceId(`value`: Output<String>) {
        this.virtualInterfaceId = value
    }

    /**
     * @param value The ID of the virtual private gateway to which to connect the virtual interface.
     */
    @JvmName("vcwlkeeibiarster")
    public suspend fun vpnGatewayId(`value`: Output<String>) {
        this.vpnGatewayId = value
    }

    /**
     * @param value The ID of the Direct Connect gateway to which to connect the virtual interface.
     */
    @JvmName("skbybafoalpoethp")
    public suspend fun dxGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dxGatewayId = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("fcssfftmwqrvfmoi")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("hxkhssxcstmmfxys")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The ID of the Direct Connect virtual interface to accept.
     */
    @JvmName("lwuqyhpmpksdnbhk")
    public suspend fun virtualInterfaceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.virtualInterfaceId = mapped
    }

    /**
     * @param value The ID of the virtual private gateway to which to connect the virtual interface.
     */
    @JvmName("kqkjqasivvukwjsw")
    public suspend fun vpnGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpnGatewayId = mapped
    }

    internal fun build(): HostedPrivateVirtualInterfaceAccepterArgs =
        HostedPrivateVirtualInterfaceAccepterArgs(
            dxGatewayId = dxGatewayId,
            tags = tags,
            virtualInterfaceId = virtualInterfaceId,
            vpnGatewayId = vpnGatewayId,
        )
}
