@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.directoryservice.kotlin

import com.pulumi.aws.directoryservice.SharedDirectoryArgs.builder
import com.pulumi.aws.directoryservice.kotlin.inputs.SharedDirectoryTargetArgs
import com.pulumi.aws.directoryservice.kotlin.inputs.SharedDirectoryTargetArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * Manages a directory in your account (directory owner) shared with another account (directory consumer).
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.directoryservice.Directory("example", {
 *     name: "tf-example",
 *     password: "SuperSecretPassw0rd",
 *     type: "MicrosoftAD",
 *     edition: "Standard",
 *     vpcSettings: {
 *         vpcId: exampleAwsVpc.id,
 *         subnetIds: exampleAwsSubnet.map(__item => __item.id),
 *     },
 * });
 * const exampleSharedDirectory = new aws.directoryservice.SharedDirectory("example", {
 *     directoryId: example.id,
 *     notes: "You wanna have a catch?",
 *     target: {
 *         id: receiver.accountId,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.directoryservice.Directory("example",
 *     name="tf-example",
 *     password="SuperSecretPassw0rd",
 *     type="MicrosoftAD",
 *     edition="Standard",
 *     vpc_settings={
 *         "vpc_id": example_aws_vpc["id"],
 *         "subnet_ids": [__item["id"] for __item in example_aws_subnet],
 *     })
 * example_shared_directory = aws.directoryservice.SharedDirectory("example",
 *     directory_id=example.id,
 *     notes="You wanna have a catch?",
 *     target={
 *         "id": receiver["accountId"],
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.DirectoryService.Directory("example", new()
 *     {
 *         Name = "tf-example",
 *         Password = "SuperSecretPassw0rd",
 *         Type = "MicrosoftAD",
 *         Edition = "Standard",
 *         VpcSettings = new Aws.DirectoryService.Inputs.DirectoryVpcSettingsArgs
 *         {
 *             VpcId = exampleAwsVpc.Id,
 *             SubnetIds = exampleAwsSubnet.Select(__item => __item.Id).ToList(),
 *         },
 *     });
 *     var exampleSharedDirectory = new Aws.DirectoryService.SharedDirectory("example", new()
 *     {
 *         DirectoryId = example.Id,
 *         Notes = "You wanna have a catch?",
 *         Target = new Aws.DirectoryService.Inputs.SharedDirectoryTargetArgs
 *         {
 *             Id = receiver.AccountId,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/directoryservice"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * example, err := directoryservice.NewDirectory(ctx, "example", &directoryservice.DirectoryArgs{
 * Name: pulumi.String("tf-example"),
 * Password: pulumi.String("SuperSecretPassw0rd"),
 * Type: pulumi.String("MicrosoftAD"),
 * Edition: pulumi.String("Standard"),
 * VpcSettings: &directoryservice.DirectoryVpcSettingsArgs{
 * VpcId: pulumi.Any(exampleAwsVpc.Id),
 * SubnetIds: []pulumi.String(%!v(PANIC=Format method: fatal: A failure has occurred: unlowered splat expression @ example.pp:7,17-39)),
 * },
 * })
 * if err != nil {
 * return err
 * }
 * _, err = directoryservice.NewSharedDirectory(ctx, "example", &directoryservice.SharedDirectoryArgs{
 * DirectoryId: example.ID(),
 * Notes: pulumi.String("You wanna have a catch?"),
 * Target: &directoryservice.SharedDirectoryTargetArgs{
 * Id: pulumi.Any(receiver.AccountId),
 * },
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.directoryservice.Directory;
 * import com.pulumi.aws.directoryservice.DirectoryArgs;
 * import com.pulumi.aws.directoryservice.inputs.DirectoryVpcSettingsArgs;
 * import com.pulumi.aws.directoryservice.SharedDirectory;
 * import com.pulumi.aws.directoryservice.SharedDirectoryArgs;
 * import com.pulumi.aws.directoryservice.inputs.SharedDirectoryTargetArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Directory("example", DirectoryArgs.builder()
 *             .name("tf-example")
 *             .password("SuperSecretPassw0rd")
 *             .type("MicrosoftAD")
 *             .edition("Standard")
 *             .vpcSettings(DirectoryVpcSettingsArgs.builder()
 *                 .vpcId(exampleAwsVpc.id())
 *                 .subnetIds(exampleAwsSubnet.stream().map(element -> element.id()).collect(toList()))
 *                 .build())
 *             .build());
 *         var exampleSharedDirectory = new SharedDirectory("exampleSharedDirectory", SharedDirectoryArgs.builder()
 *             .directoryId(example.id())
 *             .notes("You wanna have a catch?")
 *             .target(SharedDirectoryTargetArgs.builder()
 *                 .id(receiver.accountId())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Directory Service Shared Directories using the owner directory ID/shared directory ID. For example:
 * ```sh
 * $ pulumi import aws:directoryservice/sharedDirectory:SharedDirectory example d-1234567890/d-9267633ece
 * ```
 * @property directoryId Identifier of the Managed Microsoft AD directory that you want to share with other accounts.
 * @property method Method used when sharing a directory. Valid values are `ORGANIZATIONS` and `HANDSHAKE`. Default is `HANDSHAKE`.
 * @property notes Message sent by the directory owner to the directory consumer to help the directory consumer administrator determine whether to approve or reject the share invitation.
 * @property target Identifier for the directory consumer account with whom the directory is to be shared. See below.
 * The following arguments are optional:
 */
public data class SharedDirectoryArgs(
    public val directoryId: Output<String>? = null,
    public val method: Output<String>? = null,
    public val notes: Output<String>? = null,
    public val target: Output<SharedDirectoryTargetArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.directoryservice.SharedDirectoryArgs> {
    override fun toJava(): com.pulumi.aws.directoryservice.SharedDirectoryArgs =
        com.pulumi.aws.directoryservice.SharedDirectoryArgs.builder()
            .directoryId(directoryId?.applyValue({ args0 -> args0 }))
            .method(method?.applyValue({ args0 -> args0 }))
            .notes(notes?.applyValue({ args0 -> args0 }))
            .target(target?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [SharedDirectoryArgs].
 */
@PulumiTagMarker
public class SharedDirectoryArgsBuilder internal constructor() {
    private var directoryId: Output<String>? = null

    private var method: Output<String>? = null

    private var notes: Output<String>? = null

    private var target: Output<SharedDirectoryTargetArgs>? = null

    /**
     * @param value Identifier of the Managed Microsoft AD directory that you want to share with other accounts.
     */
    @JvmName("wcgrjkxxvshxsaso")
    public suspend fun directoryId(`value`: Output<String>) {
        this.directoryId = value
    }

    /**
     * @param value Method used when sharing a directory. Valid values are `ORGANIZATIONS` and `HANDSHAKE`. Default is `HANDSHAKE`.
     */
    @JvmName("llrrvumgoilykvus")
    public suspend fun method(`value`: Output<String>) {
        this.method = value
    }

    /**
     * @param value Message sent by the directory owner to the directory consumer to help the directory consumer administrator determine whether to approve or reject the share invitation.
     */
    @JvmName("epnmlgchtqanmbth")
    public suspend fun notes(`value`: Output<String>) {
        this.notes = value
    }

    /**
     * @param value Identifier for the directory consumer account with whom the directory is to be shared. See below.
     * The following arguments are optional:
     */
    @JvmName("rdghchfaasvogvrk")
    public suspend fun target(`value`: Output<SharedDirectoryTargetArgs>) {
        this.target = value
    }

    /**
     * @param value Identifier of the Managed Microsoft AD directory that you want to share with other accounts.
     */
    @JvmName("jejldihbqkxslicd")
    public suspend fun directoryId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.directoryId = mapped
    }

    /**
     * @param value Method used when sharing a directory. Valid values are `ORGANIZATIONS` and `HANDSHAKE`. Default is `HANDSHAKE`.
     */
    @JvmName("heinkvdyttbdlgww")
    public suspend fun method(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.method = mapped
    }

    /**
     * @param value Message sent by the directory owner to the directory consumer to help the directory consumer administrator determine whether to approve or reject the share invitation.
     */
    @JvmName("gqhgoasmswnbawuf")
    public suspend fun notes(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.notes = mapped
    }

    /**
     * @param value Identifier for the directory consumer account with whom the directory is to be shared. See below.
     * The following arguments are optional:
     */
    @JvmName("mfouavjkkhesmyuk")
    public suspend fun target(`value`: SharedDirectoryTargetArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.target = mapped
    }

    /**
     * @param argument Identifier for the directory consumer account with whom the directory is to be shared. See below.
     * The following arguments are optional:
     */
    @JvmName("oovbvtislkhueiea")
    public suspend fun target(argument: suspend SharedDirectoryTargetArgsBuilder.() -> Unit) {
        val toBeMapped = SharedDirectoryTargetArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.target = mapped
    }

    internal fun build(): SharedDirectoryArgs = SharedDirectoryArgs(
        directoryId = directoryId,
        method = method,
        notes = notes,
        target = target,
    )
}
