@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.dynamodb.kotlin

import com.pulumi.aws.dynamodb.TableItemArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a DynamoDB table item resource
 * > **Note:** This resource is not meant to be used for managing large amounts of data in your table, it is not designed to scale.
 *   You should perform **regular backups** of all data in the table, see [AWS docs for more](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/BackupRestore.html).
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const exampleTable = new aws.dynamodb.Table("example", {
 *     name: "example-name",
 *     readCapacity: 10,
 *     writeCapacity: 10,
 *     hashKey: "exampleHashKey",
 *     attributes: [{
 *         name: "exampleHashKey",
 *         type: "S",
 *     }],
 * });
 * const example = new aws.dynamodb.TableItem("example", {
 *     tableName: exampleTable.name,
 *     hashKey: exampleTable.hashKey,
 *     item: `{
 *   "exampleHashKey": {"S": "something"},
 *   "one": {"N": "11111"},
 *   "two": {"N": "22222"},
 *   "three": {"N": "33333"},
 *   "four": {"N": "44444"}
 * }
 * `,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example_table = aws.dynamodb.Table("example",
 *     name="example-name",
 *     read_capacity=10,
 *     write_capacity=10,
 *     hash_key="exampleHashKey",
 *     attributes=[{
 *         "name": "exampleHashKey",
 *         "type": "S",
 *     }])
 * example = aws.dynamodb.TableItem("example",
 *     table_name=example_table.name,
 *     hash_key=example_table.hash_key,
 *     item="""{
 *   "exampleHashKey": {"S": "something"},
 *   "one": {"N": "11111"},
 *   "two": {"N": "22222"},
 *   "three": {"N": "33333"},
 *   "four": {"N": "44444"}
 * }
 * """)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleTable = new Aws.DynamoDB.Table("example", new()
 *     {
 *         Name = "example-name",
 *         ReadCapacity = 10,
 *         WriteCapacity = 10,
 *         HashKey = "exampleHashKey",
 *         Attributes = new[]
 *         {
 *             new Aws.DynamoDB.Inputs.TableAttributeArgs
 *             {
 *                 Name = "exampleHashKey",
 *                 Type = "S",
 *             },
 *         },
 *     });
 *     var example = new Aws.DynamoDB.TableItem("example", new()
 *     {
 *         TableName = exampleTable.Name,
 *         HashKey = exampleTable.HashKey,
 *         Item = @"{
 *   ""exampleHashKey"": {""S"": ""something""},
 *   ""one"": {""N"": ""11111""},
 *   ""two"": {""N"": ""22222""},
 *   ""three"": {""N"": ""33333""},
 *   ""four"": {""N"": ""44444""}
 * }
 * ",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/dynamodb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		exampleTable, err := dynamodb.NewTable(ctx, "example", &dynamodb.TableArgs{
 * 			Name:          pulumi.String("example-name"),
 * 			ReadCapacity:  pulumi.Int(10),
 * 			WriteCapacity: pulumi.Int(10),
 * 			HashKey:       pulumi.String("exampleHashKey"),
 * 			Attributes: dynamodb.TableAttributeArray{
 * 				&dynamodb.TableAttributeArgs{
 * 					Name: pulumi.String("exampleHashKey"),
 * 					Type: pulumi.String("S"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = dynamodb.NewTableItem(ctx, "example", &dynamodb.TableItemArgs{
 * 			TableName: exampleTable.Name,
 * 			HashKey:   exampleTable.HashKey,
 * 			Item: pulumi.String(`{
 *   "exampleHashKey": {"S": "something"},
 *   "one": {"N": "11111"},
 *   "two": {"N": "22222"},
 *   "three": {"N": "33333"},
 *   "four": {"N": "44444"}
 * }
 * `),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.dynamodb.Table;
 * import com.pulumi.aws.dynamodb.TableArgs;
 * import com.pulumi.aws.dynamodb.inputs.TableAttributeArgs;
 * import com.pulumi.aws.dynamodb.TableItem;
 * import com.pulumi.aws.dynamodb.TableItemArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleTable = new Table("exampleTable", TableArgs.builder()
 *             .name("example-name")
 *             .readCapacity(10)
 *             .writeCapacity(10)
 *             .hashKey("exampleHashKey")
 *             .attributes(TableAttributeArgs.builder()
 *                 .name("exampleHashKey")
 *                 .type("S")
 *                 .build())
 *             .build());
 *         var example = new TableItem("example", TableItemArgs.builder()
 *             .tableName(exampleTable.name())
 *             .hashKey(exampleTable.hashKey())
 *             .item("""
 * {
 *   "exampleHashKey": {"S": "something"},
 *   "one": {"N": "11111"},
 *   "two": {"N": "22222"},
 *   "three": {"N": "33333"},
 *   "four": {"N": "44444"}
 * }
 *             """)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:dynamodb:TableItem
 *     properties:
 *       tableName: ${exampleTable.name}
 *       hashKey: ${exampleTable.hashKey}
 *       item: |
 *         {
 *           "exampleHashKey": {"S": "something"},
 *           "one": {"N": "11111"},
 *           "two": {"N": "22222"},
 *           "three": {"N": "33333"},
 *           "four": {"N": "44444"}
 *         }
 *   exampleTable:
 *     type: aws:dynamodb:Table
 *     name: example
 *     properties:
 *       name: example-name
 *       readCapacity: 10
 *       writeCapacity: 10
 *       hashKey: exampleHashKey
 *       attributes:
 *         - name: exampleHashKey
 *           type: S
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * You cannot import DynamoDB table items.
 * @property hashKey Hash key to use for lookups and identification of the item
 * @property item JSON representation of a map of attribute name/value pairs, one for each attribute. Only the primary key attributes are required; you can optionally provide other attribute name-value pairs for the item.
 * @property rangeKey Range key to use for lookups and identification of the item. Required if there is range key defined in the table.
 * @property tableName Name of the table to contain the item.
 */
public data class TableItemArgs(
    public val hashKey: Output<String>? = null,
    public val item: Output<String>? = null,
    public val rangeKey: Output<String>? = null,
    public val tableName: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.dynamodb.TableItemArgs> {
    override fun toJava(): com.pulumi.aws.dynamodb.TableItemArgs =
        com.pulumi.aws.dynamodb.TableItemArgs.builder()
            .hashKey(hashKey?.applyValue({ args0 -> args0 }))
            .item(item?.applyValue({ args0 -> args0 }))
            .rangeKey(rangeKey?.applyValue({ args0 -> args0 }))
            .tableName(tableName?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [TableItemArgs].
 */
@PulumiTagMarker
public class TableItemArgsBuilder internal constructor() {
    private var hashKey: Output<String>? = null

    private var item: Output<String>? = null

    private var rangeKey: Output<String>? = null

    private var tableName: Output<String>? = null

    /**
     * @param value Hash key to use for lookups and identification of the item
     */
    @JvmName("sutbmxliljsbbgth")
    public suspend fun hashKey(`value`: Output<String>) {
        this.hashKey = value
    }

    /**
     * @param value JSON representation of a map of attribute name/value pairs, one for each attribute. Only the primary key attributes are required; you can optionally provide other attribute name-value pairs for the item.
     */
    @JvmName("vpviogokltwbutxu")
    public suspend fun item(`value`: Output<String>) {
        this.item = value
    }

    /**
     * @param value Range key to use for lookups and identification of the item. Required if there is range key defined in the table.
     */
    @JvmName("eymlvnvoaosgfnnl")
    public suspend fun rangeKey(`value`: Output<String>) {
        this.rangeKey = value
    }

    /**
     * @param value Name of the table to contain the item.
     */
    @JvmName("mjmdocedqhymeheg")
    public suspend fun tableName(`value`: Output<String>) {
        this.tableName = value
    }

    /**
     * @param value Hash key to use for lookups and identification of the item
     */
    @JvmName("tminkxaotutjlorh")
    public suspend fun hashKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.hashKey = mapped
    }

    /**
     * @param value JSON representation of a map of attribute name/value pairs, one for each attribute. Only the primary key attributes are required; you can optionally provide other attribute name-value pairs for the item.
     */
    @JvmName("lcoxccakidsgmkna")
    public suspend fun item(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.item = mapped
    }

    /**
     * @param value Range key to use for lookups and identification of the item. Required if there is range key defined in the table.
     */
    @JvmName("vofkpksbygmrgafm")
    public suspend fun rangeKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.rangeKey = mapped
    }

    /**
     * @param value Name of the table to contain the item.
     */
    @JvmName("skoppsoadkynbpdq")
    public suspend fun tableName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tableName = mapped
    }

    internal fun build(): TableItemArgs = TableItemArgs(
        hashKey = hashKey,
        item = item,
        rangeKey = rangeKey,
        tableName = tableName,
    )
}
