@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.dynamodb.kotlin

import com.pulumi.aws.dynamodb.TagArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Manages an individual DynamoDB resource tag. This resource should only be used in cases where DynamoDB resources are created outside the provider (e.g., Table replicas in other regions).
 * > **NOTE:** This tagging resource should not be combined with the resource for managing the parent resource. For example, using `aws.dynamodb.Table` and `aws.dynamodb.Tag` to manage tags of the same DynamoDB Table in the same region will cause a perpetual difference where the `aws_dynamodb_cluster` resource will try to remove the tag being added by the `aws.dynamodb.Tag` resource.
 * > **NOTE:** This tagging resource does not use the provider `ignore_tags` configuration.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * const replica = aws.getRegion({});
 * const current = aws.getRegion({});
 * const example = new aws.dynamodb.Table("example", {replicas: [{
 *     regionName: replica.then(replica => replica.name),
 * }]});
 * const test = new aws.dynamodb.Tag("test", {
 *     resourceArn: pulumi.all([example&#46;arn, current, replica])&#46;apply(([arn, current, replica]) => std.replaceOutput({
 *         text: arn,
 *         search: current.name,
 *         replace: replica.name,
 *     })).apply(invoke => invoke.result),
 *     key: "testkey",
 *     value: "testvalue",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * replica = aws.get_region()
 * current = aws.get_region()
 * example = aws.dynamodb.Table("example", replicas=[{
 *     "region_name": replica.name,
 * }])
 * test = aws.dynamodb.Tag("test",
 *     resource_arn=example.arn.apply(lambda arn: std.replace_output(text=arn,
 *         search=current.name,
 *         replace=replica.name)).apply(lambda invoke: invoke.result),
 *     key="testkey",
 *     value="testvalue")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * return await Deployment.RunAsync(() =>
 * {
 *     var replica = Aws.GetRegion.Invoke();
 *     var current = Aws.GetRegion.Invoke();
 *     var example = new Aws.DynamoDB.Table("example", new()
 *     {
 *         Replicas = new[]
 *         {
 *             new Aws.DynamoDB.Inputs.TableReplicaArgs
 *             {
 *                 RegionName = replica.Apply(getRegionResult => getRegionResult.Name),
 *             },
 *         },
 *     });
 *     var test = new Aws.DynamoDB.Tag("test", new()
 *     {
 *         ResourceArn = Output.Tuple(example.Arn, current, replica).Apply(values =>
 *         {
 *             var arn = values.Item1;
 *             var current = values.Item2;
 *             var replica = values.Item3;
 *             return Std.Replace.Invoke(new()
 *             {
 *                 Text = arn,
 *                 Search = current.Apply(getRegionResult => getRegionResult.Name),
 *                 Replace = replica.Apply(getRegionResult => getRegionResult.Name),
 *             });
 *         }).Apply(invoke => invoke.Result),
 *         Key = "testkey",
 *         Value = "testvalue",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/dynamodb"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		replica, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := dynamodb.NewTable(ctx, "example", &dynamodb.TableArgs{
 * 			Replicas: dynamodb.TableReplicaTypeArray{
 * 				&dynamodb.TableReplicaTypeArgs{
 * 					RegionName: pulumi.String(replica.Name),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = dynamodb.NewTag(ctx, "test", &dynamodb.TagArgs{
 * 			ResourceArn: pulumi.String(example.Arn.ApplyT(func(arn string) (std.ReplaceResult, error) {
 * 				return std.ReplaceResult(interface{}(std.ReplaceOutput(ctx, std.ReplaceOutputArgs{
 * 					Text:    arn,
 * 					Search:  current.Name,
 * 					Replace: replica.Name,
 * 				}, nil))), nil
 * 			}).(std.ReplaceResultOutput).ApplyT(func(invoke std.ReplaceResult) (*string, error) {
 * 				return invoke.Result, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 			Key:   pulumi.String("testkey"),
 * 			Value: pulumi.String("testvalue"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.dynamodb.Table;
 * import com.pulumi.aws.dynamodb.TableArgs;
 * import com.pulumi.aws.dynamodb.inputs.TableReplicaArgs;
 * import com.pulumi.aws.dynamodb.Tag;
 * import com.pulumi.aws.dynamodb.TagArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var replica = AwsFunctions.getRegion();
 *         final var current = AwsFunctions.getRegion();
 *         var example = new Table("example", TableArgs.builder()
 *             .replicas(TableReplicaArgs.builder()
 *                 .regionName(replica.applyValue(getRegionResult -> getRegionResult.name()))
 *                 .build())
 *             .build());
 *         var test = new Tag("test", TagArgs.builder()
 *             .resourceArn(example.arn().applyValue(arn -> StdFunctions.replace()).applyValue(invoke -> invoke.result()))
 *             .key("testkey")
 *             .value("testvalue")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:dynamodb:Table
 *     properties:
 *       replicas:
 *         - regionName: ${replica.name}
 *   test:
 *     type: aws:dynamodb:Tag
 *     properties:
 *       resourceArn:
 *         fn::invoke:
 *           Function: std:replace
 *           Arguments:
 *             text: ${example.arn}
 *             search: ${current.name}
 *             replace: ${replica.name}
 *           Return: result
 *       key: testkey
 *       value: testvalue
 * variables:
 *   replica:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import `aws_dynamodb_tag` using the DynamoDB resource identifier and key, separated by a comma (`,`). For example:
 * ```sh
 * $ pulumi import aws:dynamodb/tag:Tag example arn:aws:dynamodb:us-east-1:123456789012:table/example,Name
 * ```
 * @property key Tag name.
 * @property resourceArn Amazon Resource Name (ARN) of the DynamoDB resource to tag.
 * @property value Tag value.
 */
public data class TagArgs(
    public val key: Output<String>? = null,
    public val resourceArn: Output<String>? = null,
    public val `value`: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.dynamodb.TagArgs> {
    override fun toJava(): com.pulumi.aws.dynamodb.TagArgs = com.pulumi.aws.dynamodb.TagArgs.builder()
        .key(key?.applyValue({ args0 -> args0 }))
        .resourceArn(resourceArn?.applyValue({ args0 -> args0 }))
        .`value`(`value`?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [TagArgs].
 */
@PulumiTagMarker
public class TagArgsBuilder internal constructor() {
    private var key: Output<String>? = null

    private var resourceArn: Output<String>? = null

    private var `value`: Output<String>? = null

    /**
     * @param value Tag name.
     */
    @JvmName("odjeqqmhemkmniaf")
    public suspend fun key(`value`: Output<String>) {
        this.key = value
    }

    /**
     * @param value Amazon Resource Name (ARN) of the DynamoDB resource to tag.
     */
    @JvmName("xsoyldtubfugfckp")
    public suspend fun resourceArn(`value`: Output<String>) {
        this.resourceArn = value
    }

    /**
     * @param value Tag value.
     */
    @JvmName("xsitewgqujnyfuff")
    public suspend fun `value`(`value`: Output<String>) {
        this.`value` = value
    }

    /**
     * @param value Tag name.
     */
    @JvmName("tmrtivpstpkaggdx")
    public suspend fun key(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.key = mapped
    }

    /**
     * @param value Amazon Resource Name (ARN) of the DynamoDB resource to tag.
     */
    @JvmName("qiandvvbbpxxeagc")
    public suspend fun resourceArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceArn = mapped
    }

    /**
     * @param value Tag value.
     */
    @JvmName("iouvcqqqocbmfcvb")
    public suspend fun `value`(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.`value` = mapped
    }

    internal fun build(): TagArgs = TagArgs(
        key = key,
        resourceArn = resourceArn,
        `value` = `value`,
    )
}
