@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.EipArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an Elastic IP resource.
 * > **Note:** EIP may require IGW to exist prior to association. Use `depends_on` to set an explicit dependency on the IGW.
 * > **Note:** Do not use `network_interface` to associate the EIP to `aws.lb.LoadBalancer` or `aws.ec2.NatGateway` resources. Instead use the `allocation_id` available in those resources to allow AWS to manage the association, otherwise you will see `AuthFailure` errors.
 * ## Example Usage
 * ### Single EIP associated with an instance
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const lb = new aws.ec2.Eip("lb", {
 *     instance: web.id,
 *     domain: "vpc",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * lb = aws.ec2.Eip("lb",
 *     instance=web["id"],
 *     domain="vpc")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var lb = new Aws.Ec2.Eip("lb", new()
 *     {
 *         Instance = web.Id,
 *         Domain = "vpc",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewEip(ctx, "lb", &ec2.EipArgs{
 * 			Instance: pulumi.Any(web.Id),
 * 			Domain:   pulumi.String("vpc"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Eip;
 * import com.pulumi.aws.ec2.EipArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var lb = new Eip("lb", EipArgs.builder()
 *             .instance(web.id())
 *             .domain("vpc")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   lb:
 *     type: aws:ec2:Eip
 *     properties:
 *       instance: ${web.id}
 *       domain: vpc
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Multiple EIPs associated with a single network interface
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const multi_ip = new aws.ec2.NetworkInterface("multi-ip", {
 *     subnetId: main.id,
 *     privateIps: [
 *         "10.0.0.10",
 *         "10.0.0.11",
 *     ],
 * });
 * const one = new aws.ec2.Eip("one", {
 *     domain: "vpc",
 *     networkInterface: multi_ip.id,
 *     associateWithPrivateIp: "10.0.0.10",
 * });
 * const two = new aws.ec2.Eip("two", {
 *     domain: "vpc",
 *     networkInterface: multi_ip.id,
 *     associateWithPrivateIp: "10.0.0.11",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * multi_ip = aws.ec2.NetworkInterface("multi-ip",
 *     subnet_id=main["id"],
 *     private_ips=[
 *         "10.0.0.10",
 *         "10.0.0.11",
 *     ])
 * one = aws.ec2.Eip("one",
 *     domain="vpc",
 *     network_interface=multi_ip.id,
 *     associate_with_private_ip="10.0.0.10")
 * two = aws.ec2.Eip("two",
 *     domain="vpc",
 *     network_interface=multi_ip.id,
 *     associate_with_private_ip="10.0.0.11")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var multi_ip = new Aws.Ec2.NetworkInterface("multi-ip", new()
 *     {
 *         SubnetId = main.Id,
 *         PrivateIps = new[]
 *         {
 *             "10.0.0.10",
 *             "10.0.0.11",
 *         },
 *     });
 *     var one = new Aws.Ec2.Eip("one", new()
 *     {
 *         Domain = "vpc",
 *         NetworkInterface = multi_ip.Id,
 *         AssociateWithPrivateIp = "10.0.0.10",
 *     });
 *     var two = new Aws.Ec2.Eip("two", new()
 *     {
 *         Domain = "vpc",
 *         NetworkInterface = multi_ip.Id,
 *         AssociateWithPrivateIp = "10.0.0.11",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewNetworkInterface(ctx, "multi-ip", &ec2.NetworkInterfaceArgs{
 * 			SubnetId: pulumi.Any(main.Id),
 * 			PrivateIps: pulumi.StringArray{
 * 				pulumi.String("10.0.0.10"),
 * 				pulumi.String("10.0.0.11"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewEip(ctx, "one", &ec2.EipArgs{
 * 			Domain:                 pulumi.String("vpc"),
 * 			NetworkInterface:       multi_ip.ID(),
 * 			AssociateWithPrivateIp: pulumi.String("10.0.0.10"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewEip(ctx, "two", &ec2.EipArgs{
 * 			Domain:                 pulumi.String("vpc"),
 * 			NetworkInterface:       multi_ip.ID(),
 * 			AssociateWithPrivateIp: pulumi.String("10.0.0.11"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.NetworkInterface;
 * import com.pulumi.aws.ec2.NetworkInterfaceArgs;
 * import com.pulumi.aws.ec2.Eip;
 * import com.pulumi.aws.ec2.EipArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var multi_ip = new NetworkInterface("multi-ip", NetworkInterfaceArgs.builder()
 *             .subnetId(main.id())
 *             .privateIps(
 *                 "10.0.0.10",
 *                 "10.0.0.11")
 *             .build());
 *         var one = new Eip("one", EipArgs.builder()
 *             .domain("vpc")
 *             .networkInterface(multi_ip.id())
 *             .associateWithPrivateIp("10.0.0.10")
 *             .build());
 *         var two = new Eip("two", EipArgs.builder()
 *             .domain("vpc")
 *             .networkInterface(multi_ip.id())
 *             .associateWithPrivateIp("10.0.0.11")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   multi-ip:
 *     type: aws:ec2:NetworkInterface
 *     properties:
 *       subnetId: ${main.id}
 *       privateIps:
 *         - 10.0.0.10
 *         - 10.0.0.11
 *   one:
 *     type: aws:ec2:Eip
 *     properties:
 *       domain: vpc
 *       networkInterface: ${["multi-ip"].id}
 *       associateWithPrivateIp: 10.0.0.10
 *   two:
 *     type: aws:ec2:Eip
 *     properties:
 *       domain: vpc
 *       networkInterface: ${["multi-ip"].id}
 *       associateWithPrivateIp: 10.0.0.11
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Attaching an EIP to an Instance with a pre-assigned private ip (VPC Only)
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _default = new aws.ec2.Vpc("default", {
 *     cidrBlock: "10.0.0.0/16",
 *     enableDnsHostnames: true,
 * });
 * const gw = new aws.ec2.InternetGateway("gw", {vpcId: _default.id});
 * const myTestSubnet = new aws.ec2.Subnet("my_test_subnet", {
 *     vpcId: _default.id,
 *     cidrBlock: "10.0.0.0/24",
 *     mapPublicIpOnLaunch: true,
 * }, {
 *     dependsOn: [gw],
 * });
 * const foo = new aws.ec2.Instance("foo", {
 *     ami: "ami-5189a661",
 *     instanceType: aws.ec2.InstanceType.T2_Micro,
 *     privateIp: "10.0.0.12",
 *     subnetId: myTestSubnet.id,
 * });
 * const bar = new aws.ec2.Eip("bar", {
 *     domain: "vpc",
 *     instance: foo.id,
 *     associateWithPrivateIp: "10.0.0.12",
 * }, {
 *     dependsOn: [gw],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * default = aws.ec2.Vpc("default",
 *     cidr_block="10.0.0.0/16",
 *     enable_dns_hostnames=True)
 * gw = aws.ec2.InternetGateway("gw", vpc_id=default.id)
 * my_test_subnet = aws.ec2.Subnet("my_test_subnet",
 *     vpc_id=default.id,
 *     cidr_block="10.0.0.0/24",
 *     map_public_ip_on_launch=True,
 *     opts = pulumi.ResourceOptions(depends_on=[gw]))
 * foo = aws.ec2.Instance("foo",
 *     ami="ami-5189a661",
 *     instance_type=aws.ec2.InstanceType.T2_MICRO,
 *     private_ip="10.0.0.12",
 *     subnet_id=my_test_subnet.id)
 * bar = aws.ec2.Eip("bar",
 *     domain="vpc",
 *     instance=foo.id,
 *     associate_with_private_ip="10.0.0.12",
 *     opts = pulumi.ResourceOptions(depends_on=[gw]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @default = new Aws.Ec2.Vpc("default", new()
 *     {
 *         CidrBlock = "10.0.0.0/16",
 *         EnableDnsHostnames = true,
 *     });
 *     var gw = new Aws.Ec2.InternetGateway("gw", new()
 *     {
 *         VpcId = @default.Id,
 *     });
 *     var myTestSubnet = new Aws.Ec2.Subnet("my_test_subnet", new()
 *     {
 *         VpcId = @default.Id,
 *         CidrBlock = "10.0.0.0/24",
 *         MapPublicIpOnLaunch = true,
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             gw,
 *         },
 *     });
 *     var foo = new Aws.Ec2.Instance("foo", new()
 *     {
 *         Ami = "ami-5189a661",
 *         InstanceType = Aws.Ec2.InstanceType.T2_Micro,
 *         PrivateIp = "10.0.0.12",
 *         SubnetId = myTestSubnet.Id,
 *     });
 *     var bar = new Aws.Ec2.Eip("bar", new()
 *     {
 *         Domain = "vpc",
 *         Instance = foo.Id,
 *         AssociateWithPrivateIp = "10.0.0.12",
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             gw,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewVpc(ctx, "default", &ec2.VpcArgs{
 * 			CidrBlock:          pulumi.String("10.0.0.0/16"),
 * 			EnableDnsHostnames: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		gw, err := ec2.NewInternetGateway(ctx, "gw", &ec2.InternetGatewayArgs{
 * 			VpcId: _default.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		myTestSubnet, err := ec2.NewSubnet(ctx, "my_test_subnet", &ec2.SubnetArgs{
 * 			VpcId:               _default.ID(),
 * 			CidrBlock:           pulumi.String("10.0.0.0/24"),
 * 			MapPublicIpOnLaunch: pulumi.Bool(true),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			gw,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		foo, err := ec2.NewInstance(ctx, "foo", &ec2.InstanceArgs{
 * 			Ami:          pulumi.String("ami-5189a661"),
 * 			InstanceType: pulumi.String(ec2.InstanceType_T2_Micro),
 * 			PrivateIp:    pulumi.String("10.0.0.12"),
 * 			SubnetId:     myTestSubnet.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewEip(ctx, "bar", &ec2.EipArgs{
 * 			Domain:                 pulumi.String("vpc"),
 * 			Instance:               foo.ID(),
 * 			AssociateWithPrivateIp: pulumi.String("10.0.0.12"),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			gw,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.ec2.InternetGateway;
 * import com.pulumi.aws.ec2.InternetGatewayArgs;
 * import com.pulumi.aws.ec2.Subnet;
 * import com.pulumi.aws.ec2.SubnetArgs;
 * import com.pulumi.aws.ec2.Instance;
 * import com.pulumi.aws.ec2.InstanceArgs;
 * import com.pulumi.aws.ec2.Eip;
 * import com.pulumi.aws.ec2.EipArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Vpc("default", VpcArgs.builder()
 *             .cidrBlock("10.0.0.0/16")
 *             .enableDnsHostnames(true)
 *             .build());
 *         var gw = new InternetGateway("gw", InternetGatewayArgs.builder()
 *             .vpcId(default_.id())
 *             .build());
 *         var myTestSubnet = new Subnet("myTestSubnet", SubnetArgs.builder()
 *             .vpcId(default_.id())
 *             .cidrBlock("10.0.0.0/24")
 *             .mapPublicIpOnLaunch(true)
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(gw)
 *                 .build());
 *         var foo = new Instance("foo", InstanceArgs.builder()
 *             .ami("ami-5189a661")
 *             .instanceType("t2.micro")
 *             .privateIp("10.0.0.12")
 *             .subnetId(myTestSubnet.id())
 *             .build());
 *         var bar = new Eip("bar", EipArgs.builder()
 *             .domain("vpc")
 *             .instance(foo.id())
 *             .associateWithPrivateIp("10.0.0.12")
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(gw)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   default:
 *     type: aws:ec2:Vpc
 *     properties:
 *       cidrBlock: 10.0.0.0/16
 *       enableDnsHostnames: true
 *   gw:
 *     type: aws:ec2:InternetGateway
 *     properties:
 *       vpcId: ${default.id}
 *   myTestSubnet:
 *     type: aws:ec2:Subnet
 *     name: my_test_subnet
 *     properties:
 *       vpcId: ${default.id}
 *       cidrBlock: 10.0.0.0/24
 *       mapPublicIpOnLaunch: true
 *     options:
 *       dependson:
 *         - ${gw}
 *   foo:
 *     type: aws:ec2:Instance
 *     properties:
 *       ami: ami-5189a661
 *       instanceType: t2.micro
 *       privateIp: 10.0.0.12
 *       subnetId: ${myTestSubnet.id}
 *   bar:
 *     type: aws:ec2:Eip
 *     properties:
 *       domain: vpc
 *       instance: ${foo.id}
 *       associateWithPrivateIp: 10.0.0.12
 *     options:
 *       dependson:
 *         - ${gw}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Allocating EIP from the BYOIP pool
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const byoip_ip = new aws.ec2.Eip("byoip-ip", {
 *     domain: "vpc",
 *     publicIpv4Pool: "ipv4pool-ec2-012345",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * byoip_ip = aws.ec2.Eip("byoip-ip",
 *     domain="vpc",
 *     public_ipv4_pool="ipv4pool-ec2-012345")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var byoip_ip = new Aws.Ec2.Eip("byoip-ip", new()
 *     {
 *         Domain = "vpc",
 *         PublicIpv4Pool = "ipv4pool-ec2-012345",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewEip(ctx, "byoip-ip", &ec2.EipArgs{
 * 			Domain:         pulumi.String("vpc"),
 * 			PublicIpv4Pool: pulumi.String("ipv4pool-ec2-012345"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Eip;
 * import com.pulumi.aws.ec2.EipArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var byoip_ip = new Eip("byoip-ip", EipArgs.builder()
 *             .domain("vpc")
 *             .publicIpv4Pool("ipv4pool-ec2-012345")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   byoip-ip:
 *     type: aws:ec2:Eip
 *     properties:
 *       domain: vpc
 *       publicIpv4Pool: ipv4pool-ec2-012345
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import EIPs in a VPC using their Allocation ID. For example:
 * ```sh
 * $ pulumi import aws:ec2/eip:Eip bar eipalloc-00a10e96
 * ```
 * @property address IP address from an EC2 BYOIP pool. This option is only available for VPC EIPs.
 * @property associateWithPrivateIp User-specified primary or secondary private IP address to associate with the Elastic IP address. If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.
 * @property customerOwnedIpv4Pool ID  of a customer-owned address pool. For more on customer owned IP addressed check out [Customer-owned IP addresses guide](https://docs.aws.amazon.com/outposts/latest/userguide/outposts-networking-components.html#ip-addressing).
 * @property domain Indicates if this EIP is for use in VPC (`vpc`).
 * @property instance EC2 instance ID.
 * @property networkBorderGroup Location from which the IP address is advertised. Use this parameter to limit the address to this location.
 * @property networkInterface Network interface ID to associate with.
 * @property publicIpv4Pool EC2 IPv4 address pool identifier or `amazon`.
 * This option is only available for VPC EIPs.
 * @property tags Map of tags to assign to the resource. Tags can only be applied to EIPs in a VPC. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpc Boolean if the EIP is in a VPC or not. Use `domain` instead.
 * Defaults to `true` unless the region supports EC2-Classic.
 * > **NOTE:** You can specify either the `instance` ID or the `network_interface` ID, but not both. Including both will **not** return an error from the AWS API, but will have undefined behavior. See the relevant [AssociateAddress API Call][1] for more information.
 * > **NOTE:** Specifying both `public_ipv4_pool` and `address` won't cause an error but `address` will be used in the
 * case both options are defined as the api only requires one or the other.
 */
public data class EipArgs(
    public val address: Output<String>? = null,
    public val associateWithPrivateIp: Output<String>? = null,
    public val customerOwnedIpv4Pool: Output<String>? = null,
    public val domain: Output<String>? = null,
    public val instance: Output<String>? = null,
    public val networkBorderGroup: Output<String>? = null,
    public val networkInterface: Output<String>? = null,
    public val publicIpv4Pool: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    @Deprecated(
        message = """
  use domain attribute instead
  """,
    )
    public val vpc: Output<Boolean>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.EipArgs> {
    override fun toJava(): com.pulumi.aws.ec2.EipArgs = com.pulumi.aws.ec2.EipArgs.builder()
        .address(address?.applyValue({ args0 -> args0 }))
        .associateWithPrivateIp(associateWithPrivateIp?.applyValue({ args0 -> args0 }))
        .customerOwnedIpv4Pool(customerOwnedIpv4Pool?.applyValue({ args0 -> args0 }))
        .domain(domain?.applyValue({ args0 -> args0 }))
        .instance(instance?.applyValue({ args0 -> args0 }))
        .networkBorderGroup(networkBorderGroup?.applyValue({ args0 -> args0 }))
        .networkInterface(networkInterface?.applyValue({ args0 -> args0 }))
        .publicIpv4Pool(publicIpv4Pool?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .vpc(vpc?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [EipArgs].
 */
@PulumiTagMarker
public class EipArgsBuilder internal constructor() {
    private var address: Output<String>? = null

    private var associateWithPrivateIp: Output<String>? = null

    private var customerOwnedIpv4Pool: Output<String>? = null

    private var domain: Output<String>? = null

    private var instance: Output<String>? = null

    private var networkBorderGroup: Output<String>? = null

    private var networkInterface: Output<String>? = null

    private var publicIpv4Pool: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpc: Output<Boolean>? = null

    /**
     * @param value IP address from an EC2 BYOIP pool. This option is only available for VPC EIPs.
     */
    @JvmName("mhnwxlvvdbokfyuq")
    public suspend fun address(`value`: Output<String>) {
        this.address = value
    }

    /**
     * @param value User-specified primary or secondary private IP address to associate with the Elastic IP address. If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.
     */
    @JvmName("ymvjoqfmwiyxhbja")
    public suspend fun associateWithPrivateIp(`value`: Output<String>) {
        this.associateWithPrivateIp = value
    }

    /**
     * @param value ID  of a customer-owned address pool. For more on customer owned IP addressed check out [Customer-owned IP addresses guide](https://docs.aws.amazon.com/outposts/latest/userguide/outposts-networking-components.html#ip-addressing).
     */
    @JvmName("otaxnnnujevbykao")
    public suspend fun customerOwnedIpv4Pool(`value`: Output<String>) {
        this.customerOwnedIpv4Pool = value
    }

    /**
     * @param value Indicates if this EIP is for use in VPC (`vpc`).
     */
    @JvmName("ufenjovphsukgxjh")
    public suspend fun domain(`value`: Output<String>) {
        this.domain = value
    }

    /**
     * @param value EC2 instance ID.
     */
    @JvmName("chripfrdqvkvoryr")
    public suspend fun instance(`value`: Output<String>) {
        this.instance = value
    }

    /**
     * @param value Location from which the IP address is advertised. Use this parameter to limit the address to this location.
     */
    @JvmName("iwffdmrlebedkhrk")
    public suspend fun networkBorderGroup(`value`: Output<String>) {
        this.networkBorderGroup = value
    }

    /**
     * @param value Network interface ID to associate with.
     */
    @JvmName("xcnglhelsjfencxl")
    public suspend fun networkInterface(`value`: Output<String>) {
        this.networkInterface = value
    }

    /**
     * @param value EC2 IPv4 address pool identifier or `amazon`.
     * This option is only available for VPC EIPs.
     */
    @JvmName("ofpicbihcebkmtmp")
    public suspend fun publicIpv4Pool(`value`: Output<String>) {
        this.publicIpv4Pool = value
    }

    /**
     * @param value Map of tags to assign to the resource. Tags can only be applied to EIPs in a VPC. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ynvjmwsllyoyryjk")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Boolean if the EIP is in a VPC or not. Use `domain` instead.
     * Defaults to `true` unless the region supports EC2-Classic.
     * > **NOTE:** You can specify either the `instance` ID or the `network_interface` ID, but not both. Including both will **not** return an error from the AWS API, but will have undefined behavior. See the relevant [AssociateAddress API Call][1] for more information.
     * > **NOTE:** Specifying both `public_ipv4_pool` and `address` won't cause an error but `address` will be used in the
     * case both options are defined as the api only requires one or the other.
     */
    @Deprecated(
        message = """
  use domain attribute instead
  """,
    )
    @JvmName("shjruuktehaaastl")
    public suspend fun vpc(`value`: Output<Boolean>) {
        this.vpc = value
    }

    /**
     * @param value IP address from an EC2 BYOIP pool. This option is only available for VPC EIPs.
     */
    @JvmName("waswknuxxrlyhdkd")
    public suspend fun address(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.address = mapped
    }

    /**
     * @param value User-specified primary or secondary private IP address to associate with the Elastic IP address. If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.
     */
    @JvmName("rxvdyhomxhafxhfi")
    public suspend fun associateWithPrivateIp(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.associateWithPrivateIp = mapped
    }

    /**
     * @param value ID  of a customer-owned address pool. For more on customer owned IP addressed check out [Customer-owned IP addresses guide](https://docs.aws.amazon.com/outposts/latest/userguide/outposts-networking-components.html#ip-addressing).
     */
    @JvmName("orcbdbxgvfjtjeir")
    public suspend fun customerOwnedIpv4Pool(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerOwnedIpv4Pool = mapped
    }

    /**
     * @param value Indicates if this EIP is for use in VPC (`vpc`).
     */
    @JvmName("axqtimcyaavrnqtj")
    public suspend fun domain(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domain = mapped
    }

    /**
     * @param value EC2 instance ID.
     */
    @JvmName("ymvilomasxxcldyg")
    public suspend fun instance(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instance = mapped
    }

    /**
     * @param value Location from which the IP address is advertised. Use this parameter to limit the address to this location.
     */
    @JvmName("phpiffwilndlrkdk")
    public suspend fun networkBorderGroup(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkBorderGroup = mapped
    }

    /**
     * @param value Network interface ID to associate with.
     */
    @JvmName("cylcuktwttbxpymd")
    public suspend fun networkInterface(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkInterface = mapped
    }

    /**
     * @param value EC2 IPv4 address pool identifier or `amazon`.
     * This option is only available for VPC EIPs.
     */
    @JvmName("nrduppirmftbsupq")
    public suspend fun publicIpv4Pool(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.publicIpv4Pool = mapped
    }

    /**
     * @param value Map of tags to assign to the resource. Tags can only be applied to EIPs in a VPC. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("pcobyhxjheraurdd")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the resource. Tags can only be applied to EIPs in a VPC. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("kpwcpbehwgodurcx")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Boolean if the EIP is in a VPC or not. Use `domain` instead.
     * Defaults to `true` unless the region supports EC2-Classic.
     * > **NOTE:** You can specify either the `instance` ID or the `network_interface` ID, but not both. Including both will **not** return an error from the AWS API, but will have undefined behavior. See the relevant [AssociateAddress API Call][1] for more information.
     * > **NOTE:** Specifying both `public_ipv4_pool` and `address` won't cause an error but `address` will be used in the
     * case both options are defined as the api only requires one or the other.
     */
    @Deprecated(
        message = """
  use domain attribute instead
  """,
    )
    @JvmName("jcthxfgdsfanqxvc")
    public suspend fun vpc(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpc = mapped
    }

    internal fun build(): EipArgs = EipArgs(
        address = address,
        associateWithPrivateIp = associateWithPrivateIp,
        customerOwnedIpv4Pool = customerOwnedIpv4Pool,
        domain = domain,
        instance = instance,
        networkBorderGroup = networkBorderGroup,
        networkInterface = networkInterface,
        publicIpv4Pool = publicIpv4Pool,
        tags = tags,
        vpc = vpc,
    )
}
