@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.RouteTableArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.RouteTableRouteArgs
import com.pulumi.aws.ec2.kotlin.inputs.RouteTableRouteArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a resource to create a VPC routing table.
 * > **NOTE on Route Tables and Routes:** This provider currently
 * provides both a standalone Route resource and a Route Table resource with routes
 * defined in-line. At this time you cannot use a Route Table with in-line routes
 * in conjunction with any Route resources. Doing so will cause
 * a conflict of rule settings and will overwrite rules.
 * > **NOTE on `gateway_id` and `nat_gateway_id`:** The AWS API is very forgiving with these two
 * attributes and the `aws.ec2.RouteTable` resource can be created with a NAT ID specified as a Gateway ID attribute.
 * This _will_ lead to a permanent diff between your configuration and statefile, as the API returns the correct
 * parameters in the returned route table. If you're experiencing constant diffs in your `aws.ec2.RouteTable` resources,
 * the first thing to check is whether or not you're specifying a NAT ID instead of a Gateway ID, or vice-versa.
 * > **NOTE on `propagating_vgws` and the `aws.ec2.VpnGatewayRoutePropagation` resource:**
 * If the `propagating_vgws` argument is present, it's not supported to _also_
 * define route propagations using `aws.ec2.VpnGatewayRoutePropagation`, since
 * this resource will delete any propagating gateways not explicitly listed in
 * `propagating_vgws`. Omit this argument when defining route propagation using
 * the separate resource.
 * ## Example Usage
 * ### Basic example
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.ec2.RouteTable("example", {
 *     vpcId: exampleAwsVpc.id,
 *     routes: [
 *         {
 *             cidrBlock: "10.0.1.0/24",
 *             gatewayId: exampleAwsInternetGateway.id,
 *         },
 *         {
 *             ipv6CidrBlock: "::/0",
 *             egressOnlyGatewayId: exampleAwsEgressOnlyInternetGateway.id,
 *         },
 *     ],
 *     tags: {
 *         Name: "example",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ec2.RouteTable("example",
 *     vpc_id=example_aws_vpc["id"],
 *     routes=[
 *         {
 *             "cidr_block": "10.0.1.0/24",
 *             "gateway_id": example_aws_internet_gateway["id"],
 *         },
 *         {
 *             "ipv6_cidr_block": "::/0",
 *             "egress_only_gateway_id": example_aws_egress_only_internet_gateway["id"],
 *         },
 *     ],
 *     tags={
 *         "Name": "example",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Ec2.RouteTable("example", new()
 *     {
 *         VpcId = exampleAwsVpc.Id,
 *         Routes = new[]
 *         {
 *             new Aws.Ec2.Inputs.RouteTableRouteArgs
 *             {
 *                 CidrBlock = "10.0.1.0/24",
 *                 GatewayId = exampleAwsInternetGateway.Id,
 *             },
 *             new Aws.Ec2.Inputs.RouteTableRouteArgs
 *             {
 *                 Ipv6CidrBlock = "::/0",
 *                 EgressOnlyGatewayId = exampleAwsEgressOnlyInternetGateway.Id,
 *             },
 *         },
 *         Tags =
 *         {
 *             { "Name", "example" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewRouteTable(ctx, "example", &ec2.RouteTableArgs{
 * 			VpcId: pulumi.Any(exampleAwsVpc.Id),
 * 			Routes: ec2.RouteTableRouteArray{
 * 				&ec2.RouteTableRouteArgs{
 * 					CidrBlock: pulumi.String("10.0.1.0/24"),
 * 					GatewayId: pulumi.Any(exampleAwsInternetGateway.Id),
 * 				},
 * 				&ec2.RouteTableRouteArgs{
 * 					Ipv6CidrBlock:       pulumi.String("::/0"),
 * 					EgressOnlyGatewayId: pulumi.Any(exampleAwsEgressOnlyInternetGateway.Id),
 * 				},
 * 			},
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("example"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.RouteTable;
 * import com.pulumi.aws.ec2.RouteTableArgs;
 * import com.pulumi.aws.ec2.inputs.RouteTableRouteArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new RouteTable("example", RouteTableArgs.builder()
 *             .vpcId(exampleAwsVpc.id())
 *             .routes(
 *                 RouteTableRouteArgs.builder()
 *                     .cidrBlock("10.0.1.0/24")
 *                     .gatewayId(exampleAwsInternetGateway.id())
 *                     .build(),
 *                 RouteTableRouteArgs.builder()
 *                     .ipv6CidrBlock("::/0")
 *                     .egressOnlyGatewayId(exampleAwsEgressOnlyInternetGateway.id())
 *                     .build())
 *             .tags(Map.of("Name", "example"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:RouteTable
 *     properties:
 *       vpcId: ${exampleAwsVpc.id}
 *       routes:
 *         - cidrBlock: 10.0.1.0/24
 *           gatewayId: ${exampleAwsInternetGateway.id}
 *         - ipv6CidrBlock: ::/0
 *           egressOnlyGatewayId: ${exampleAwsEgressOnlyInternetGateway.id}
 *       tags:
 *         Name: example
 * ```
 * <!--End PulumiCodeChooser -->
 * To subsequently remove all managed routes:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.ec2.RouteTable("example", {
 *     vpcId: exampleAwsVpc.id,
 *     routes: [],
 *     tags: {
 *         Name: "example",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ec2.RouteTable("example",
 *     vpc_id=example_aws_vpc["id"],
 *     routes=[],
 *     tags={
 *         "Name": "example",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Ec2.RouteTable("example", new()
 *     {
 *         VpcId = exampleAwsVpc.Id,
 *         Routes = new[] {},
 *         Tags =
 *         {
 *             { "Name", "example" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewRouteTable(ctx, "example", &ec2.RouteTableArgs{
 * 			VpcId:  pulumi.Any(exampleAwsVpc.Id),
 * 			Routes: ec2.RouteTableRouteArray{},
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("example"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.RouteTable;
 * import com.pulumi.aws.ec2.RouteTableArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new RouteTable("example", RouteTableArgs.builder()
 *             .vpcId(exampleAwsVpc.id())
 *             .routes()
 *             .tags(Map.of("Name", "example"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:RouteTable
 *     properties:
 *       vpcId: ${exampleAwsVpc.id}
 *       routes: []
 *       tags:
 *         Name: example
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Adopting an existing local route
 * AWS creates certain routes that the AWS provider mostly ignores. You can manage them by importing or adopting them. See Import below for information on importing. This example shows adopting a route and then updating its target.
 * First, adopt an existing AWS-created route:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const test = new aws.ec2.Vpc("test", {cidrBlock: "10.1.0.0/16"});
 * const testRouteTable = new aws.ec2.RouteTable("test", {
 *     vpcId: test.id,
 *     routes: [{
 *         cidrBlock: "10.1.0.0/16",
 *         gatewayId: "local",
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test = aws.ec2.Vpc("test", cidr_block="10.1.0.0/16")
 * test_route_table = aws.ec2.RouteTable("test",
 *     vpc_id=test.id,
 *     routes=[{
 *         "cidr_block": "10.1.0.0/16",
 *         "gateway_id": "local",
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var test = new Aws.Ec2.Vpc("test", new()
 *     {
 *         CidrBlock = "10.1.0.0/16",
 *     });
 *     var testRouteTable = new Aws.Ec2.RouteTable("test", new()
 *     {
 *         VpcId = test.Id,
 *         Routes = new[]
 *         {
 *             new Aws.Ec2.Inputs.RouteTableRouteArgs
 *             {
 *                 CidrBlock = "10.1.0.0/16",
 *                 GatewayId = "local",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		test, err := ec2.NewVpc(ctx, "test", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.1.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewRouteTable(ctx, "test", &ec2.RouteTableArgs{
 * 			VpcId: test.ID(),
 * 			Routes: ec2.RouteTableRouteArray{
 * 				&ec2.RouteTableRouteArgs{
 * 					CidrBlock: pulumi.String("10.1.0.0/16"),
 * 					GatewayId: pulumi.String("local"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.ec2.RouteTable;
 * import com.pulumi.aws.ec2.RouteTableArgs;
 * import com.pulumi.aws.ec2.inputs.RouteTableRouteArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new Vpc("test", VpcArgs.builder()
 *             .cidrBlock("10.1.0.0/16")
 *             .build());
 *         var testRouteTable = new RouteTable("testRouteTable", RouteTableArgs.builder()
 *             .vpcId(test.id())
 *             .routes(RouteTableRouteArgs.builder()
 *                 .cidrBlock("10.1.0.0/16")
 *                 .gatewayId("local")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   test:
 *     type: aws:ec2:Vpc
 *     properties:
 *       cidrBlock: 10.1.0.0/16
 *   testRouteTable:
 *     type: aws:ec2:RouteTable
 *     name: test
 *     properties:
 *       vpcId: ${test.id}
 *       routes:
 *         - cidrBlock: 10.1.0.0/16
 *           gatewayId: local
 * ```
 * <!--End PulumiCodeChooser -->
 * Next, update the target of the route:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const test = new aws.ec2.Vpc("test", {cidrBlock: "10.1.0.0/16"});
 * const testSubnet = new aws.ec2.Subnet("test", {
 *     cidrBlock: "10.1.1.0/24",
 *     vpcId: test.id,
 * });
 * const testNetworkInterface = new aws.ec2.NetworkInterface("test", {subnetId: testSubnet.id});
 * const testRouteTable = new aws.ec2.RouteTable("test", {
 *     vpcId: test.id,
 *     routes: [{
 *         cidrBlock: test.cidrBlock,
 *         networkInterfaceId: testNetworkInterface.id,
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test = aws.ec2.Vpc("test", cidr_block="10.1.0.0/16")
 * test_subnet = aws.ec2.Subnet("test",
 *     cidr_block="10.1.1.0/24",
 *     vpc_id=test.id)
 * test_network_interface = aws.ec2.NetworkInterface("test", subnet_id=test_subnet.id)
 * test_route_table = aws.ec2.RouteTable("test",
 *     vpc_id=test.id,
 *     routes=[{
 *         "cidr_block": test.cidr_block,
 *         "network_interface_id": test_network_interface.id,
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var test = new Aws.Ec2.Vpc("test", new()
 *     {
 *         CidrBlock = "10.1.0.0/16",
 *     });
 *     var testSubnet = new Aws.Ec2.Subnet("test", new()
 *     {
 *         CidrBlock = "10.1.1.0/24",
 *         VpcId = test.Id,
 *     });
 *     var testNetworkInterface = new Aws.Ec2.NetworkInterface("test", new()
 *     {
 *         SubnetId = testSubnet.Id,
 *     });
 *     var testRouteTable = new Aws.Ec2.RouteTable("test", new()
 *     {
 *         VpcId = test.Id,
 *         Routes = new[]
 *         {
 *             new Aws.Ec2.Inputs.RouteTableRouteArgs
 *             {
 *                 CidrBlock = test.CidrBlock,
 *                 NetworkInterfaceId = testNetworkInterface.Id,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		test, err := ec2.NewVpc(ctx, "test", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.1.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		testSubnet, err := ec2.NewSubnet(ctx, "test", &ec2.SubnetArgs{
 * 			CidrBlock: pulumi.String("10.1.1.0/24"),
 * 			VpcId:     test.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		testNetworkInterface, err := ec2.NewNetworkInterface(ctx, "test", &ec2.NetworkInterfaceArgs{
 * 			SubnetId: testSubnet.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewRouteTable(ctx, "test", &ec2.RouteTableArgs{
 * 			VpcId: test.ID(),
 * 			Routes: ec2.RouteTableRouteArray{
 * 				&ec2.RouteTableRouteArgs{
 * 					CidrBlock:          test.CidrBlock,
 * 					NetworkInterfaceId: testNetworkInterface.ID(),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.ec2.Subnet;
 * import com.pulumi.aws.ec2.SubnetArgs;
 * import com.pulumi.aws.ec2.NetworkInterface;
 * import com.pulumi.aws.ec2.NetworkInterfaceArgs;
 * import com.pulumi.aws.ec2.RouteTable;
 * import com.pulumi.aws.ec2.RouteTableArgs;
 * import com.pulumi.aws.ec2.inputs.RouteTableRouteArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new Vpc("test", VpcArgs.builder()
 *             .cidrBlock("10.1.0.0/16")
 *             .build());
 *         var testSubnet = new Subnet("testSubnet", SubnetArgs.builder()
 *             .cidrBlock("10.1.1.0/24")
 *             .vpcId(test.id())
 *             .build());
 *         var testNetworkInterface = new NetworkInterface("testNetworkInterface", NetworkInterfaceArgs.builder()
 *             .subnetId(testSubnet.id())
 *             .build());
 *         var testRouteTable = new RouteTable("testRouteTable", RouteTableArgs.builder()
 *             .vpcId(test.id())
 *             .routes(RouteTableRouteArgs.builder()
 *                 .cidrBlock(test.cidrBlock())
 *                 .networkInterfaceId(testNetworkInterface.id())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   test:
 *     type: aws:ec2:Vpc
 *     properties:
 *       cidrBlock: 10.1.0.0/16
 *   testRouteTable:
 *     type: aws:ec2:RouteTable
 *     name: test
 *     properties:
 *       vpcId: ${test.id}
 *       routes:
 *         - cidrBlock: ${test.cidrBlock}
 *           networkInterfaceId: ${testNetworkInterface.id}
 *   testSubnet:
 *     type: aws:ec2:Subnet
 *     name: test
 *     properties:
 *       cidrBlock: 10.1.1.0/24
 *       vpcId: ${test.id}
 *   testNetworkInterface:
 *     type: aws:ec2:NetworkInterface
 *     name: test
 *     properties:
 *       subnetId: ${testSubnet.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * The target could then be updated again back to `local`.
 * ## Import
 * Using `pulumi import`, import Route Tables using the route table `id`. For example:
 * ```sh
 * $ pulumi import aws:ec2/routeTable:RouteTable public_rt rtb-4e616f6d69
 * ```
 * @property propagatingVgws A list of virtual gateways for propagation.
 * @property routes A list of route objects. Their keys are documented below.
 * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
 * @property tags A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcId The VPC ID.
 */
public data class RouteTableArgs(
    public val propagatingVgws: Output<List<String>>? = null,
    public val routes: Output<List<RouteTableRouteArgs>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.RouteTableArgs> {
    override fun toJava(): com.pulumi.aws.ec2.RouteTableArgs =
        com.pulumi.aws.ec2.RouteTableArgs.builder()
            .propagatingVgws(propagatingVgws?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .routes(routes?.applyValue({ args0 -> args0.map({ args0 -> args0.let({ args0 -> args0.toJava() }) }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .vpcId(vpcId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [RouteTableArgs].
 */
@PulumiTagMarker
public class RouteTableArgsBuilder internal constructor() {
    private var propagatingVgws: Output<List<String>>? = null

    private var routes: Output<List<RouteTableRouteArgs>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcId: Output<String>? = null

    /**
     * @param value A list of virtual gateways for propagation.
     */
    @JvmName("sgrosffejicyfyor")
    public suspend fun propagatingVgws(`value`: Output<List<String>>) {
        this.propagatingVgws = value
    }

    @JvmName("mhvewfnpcbpijflr")
    public suspend fun propagatingVgws(vararg values: Output<String>) {
        this.propagatingVgws = Output.all(values.asList())
    }

    /**
     * @param values A list of virtual gateways for propagation.
     */
    @JvmName("eldlgvwtbemsmpjc")
    public suspend fun propagatingVgws(values: List<Output<String>>) {
        this.propagatingVgws = Output.all(values)
    }

    /**
     * @param value A list of route objects. Their keys are documented below.
     * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
     */
    @JvmName("dfamwlnfaqgjbhdv")
    public suspend fun routes(`value`: Output<List<RouteTableRouteArgs>>) {
        this.routes = value
    }

    @JvmName("ihtsxrypilgvfqbe")
    public suspend fun routes(vararg values: Output<RouteTableRouteArgs>) {
        this.routes = Output.all(values.asList())
    }

    /**
     * @param values A list of route objects. Their keys are documented below.
     * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
     */
    @JvmName("xynkeuxundnsdkkg")
    public suspend fun routes(values: List<Output<RouteTableRouteArgs>>) {
        this.routes = Output.all(values)
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("xusnrydoxrqudpta")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The VPC ID.
     */
    @JvmName("fjfmqljffdbxlkak")
    public suspend fun vpcId(`value`: Output<String>) {
        this.vpcId = value
    }

    /**
     * @param value A list of virtual gateways for propagation.
     */
    @JvmName("jtmlrmdwkiaotpdj")
    public suspend fun propagatingVgws(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.propagatingVgws = mapped
    }

    /**
     * @param values A list of virtual gateways for propagation.
     */
    @JvmName("clsylibspbkbnjfx")
    public suspend fun propagatingVgws(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.propagatingVgws = mapped
    }

    /**
     * @param value A list of route objects. Their keys are documented below.
     * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
     */
    @JvmName("uokxjkpbrwiaqoyl")
    public suspend fun routes(`value`: List<RouteTableRouteArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.routes = mapped
    }

    /**
     * @param argument A list of route objects. Their keys are documented below.
     * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
     */
    @JvmName("jxoetmvvavibmhlt")
    public suspend fun routes(argument: List<suspend RouteTableRouteArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            RouteTableRouteArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.routes = mapped
    }

    /**
     * @param argument A list of route objects. Their keys are documented below.
     * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
     */
    @JvmName("smgkkqtanipyjibi")
    public suspend fun routes(vararg argument: suspend RouteTableRouteArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            RouteTableRouteArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.routes = mapped
    }

    /**
     * @param argument A list of route objects. Their keys are documented below.
     * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
     */
    @JvmName("umrliifjrsmidnvk")
    public suspend fun routes(argument: suspend RouteTableRouteArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(RouteTableRouteArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.routes = mapped
    }

    /**
     * @param values A list of route objects. Their keys are documented below.
     * This means that omitting this argument is interpreted as ignoring any existing routes. To remove all managed routes an empty list should be specified. See the example above.
     */
    @JvmName("yfhgokhcfkbqxbis")
    public suspend fun routes(vararg values: RouteTableRouteArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.routes = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ovpotbmmeveloijn")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("rwowolsjawncgxdi")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The VPC ID.
     */
    @JvmName("dxwixbmtaqwehxbw")
    public suspend fun vpcId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcId = mapped
    }

    internal fun build(): RouteTableArgs = RouteTableArgs(
        propagatingVgws = propagatingVgws,
        routes = routes,
        tags = tags,
        vpcId = vpcId,
    )
}
