@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.SecurityGroupRuleArgs.builder
import com.pulumi.aws.ec2.kotlin.enums.ProtocolType
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * Provides a security group rule resource. Represents a single `ingress` or `egress` group rule, which can be added to external Security Groups.
 * > **NOTE:** Avoid using the `aws.ec2.SecurityGroupRule` resource, as it struggles with managing multiple CIDR blocks, and, due to the historical lack of unique IDs, tags and descriptions. To avoid these problems, use the current best practice of the `aws.vpc.SecurityGroupEgressRule` and `aws.vpc.SecurityGroupIngressRule` resources with one CIDR block per rule.
 * !> **WARNING:** You should not use the `aws.ec2.SecurityGroupRule` resource in conjunction with `aws.vpc.SecurityGroupEgressRule` and `aws.vpc.SecurityGroupIngressRule` resources or with an `aws.ec2.SecurityGroup` resource that has in-line rules. Doing so may cause rule conflicts, perpetual differences, and result in rules being overwritten.
 * > **NOTE:** Setting `protocol = "all"` or `protocol = -1` with `from_port` and `to_port` will result in the EC2 API creating a security group rule with all ports open. This API behavior cannot be controlled by this provider and may generate warnings in the future.
 * > **NOTE:** Referencing Security Groups across VPC peering has certain restrictions. More information is available in the [VPC Peering User Guide](https://docs.aws.amazon.com/vpc/latest/peering/vpc-peering-security-groups.html).
 * ## Example Usage
 * Basic usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.ec2.SecurityGroupRule("example", {
 *     type: "ingress",
 *     fromPort: 0,
 *     toPort: 65535,
 *     protocol: aws.ec2.ProtocolType.TCP,
 *     cidrBlocks: [exampleAwsVpc&#46;cidrBlock],
 *     ipv6CidrBlocks: [exampleAwsVpc&#46;ipv6CidrBlock],
 *     securityGroupId: "sg-123456",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ec2.SecurityGroupRule("example",
 *     type="ingress",
 *     from_port=0,
 *     to_port=65535,
 *     protocol=aws.ec2.ProtocolType.TCP,
 *     cidr_blocks=[example_aws_vpc["cidrBlock"]],
 *     ipv6_cidr_blocks=[example_aws_vpc["ipv6CidrBlock"]],
 *     security_group_id="sg-123456")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Ec2.SecurityGroupRule("example", new()
 *     {
 *         Type = "ingress",
 *         FromPort = 0,
 *         ToPort = 65535,
 *         Protocol = Aws.Ec2.ProtocolType.TCP,
 *         CidrBlocks = new[]
 *         {
 *             exampleAwsVpc.CidrBlock,
 *         },
 *         Ipv6CidrBlocks = new[]
 *         {
 *             exampleAwsVpc.Ipv6CidrBlock,
 *         },
 *         SecurityGroupId = "sg-123456",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewSecurityGroupRule(ctx, "example", &ec2.SecurityGroupRuleArgs{
 * 			Type:     pulumi.String("ingress"),
 * 			FromPort: pulumi.Int(0),
 * 			ToPort:   pulumi.Int(65535),
 * 			Protocol: pulumi.String(ec2.ProtocolTypeTCP),
 * 			CidrBlocks: pulumi.StringArray{
 * 				exampleAwsVpc.CidrBlock,
 * 			},
 * 			Ipv6CidrBlocks: pulumi.StringArray{
 * 				exampleAwsVpc.Ipv6CidrBlock,
 * 			},
 * 			SecurityGroupId: pulumi.String("sg-123456"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.SecurityGroupRule;
 * import com.pulumi.aws.ec2.SecurityGroupRuleArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new SecurityGroupRule("example", SecurityGroupRuleArgs.builder()
 *             .type("ingress")
 *             .fromPort(0)
 *             .toPort(65535)
 *             .protocol("tcp")
 *             .cidrBlocks(exampleAwsVpc.cidrBlock())
 *             .ipv6CidrBlocks(exampleAwsVpc.ipv6CidrBlock())
 *             .securityGroupId("sg-123456")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:SecurityGroupRule
 *     properties:
 *       type: ingress
 *       fromPort: 0
 *       toPort: 65535
 *       protocol: tcp
 *       cidrBlocks:
 *         - ${exampleAwsVpc.cidrBlock}
 *       ipv6CidrBlocks:
 *         - ${exampleAwsVpc.ipv6CidrBlock}
 *       securityGroupId: sg-123456
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Usage With Prefix List IDs
 * Prefix Lists are either managed by AWS internally, or created by the customer using a
 * Managed Prefix List resource. Prefix Lists provided by
 * AWS are associated with a prefix list name, or service name, that is linked to a specific region.
 * Prefix list IDs are exported on VPC Endpoints, so you can use this format:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * // ...
 * const myEndpoint = new aws.ec2.VpcEndpoint("my_endpoint", {});
 * const allowAll = new aws.ec2.SecurityGroupRule("allow_all", {
 *     type: "egress",
 *     toPort: 0,
 *     protocol: "-1",
 *     prefixListIds: [myEndpoint&#46;prefixListId],
 *     fromPort: 0,
 *     securityGroupId: "sg-123456",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * # ...
 * my_endpoint = aws.ec2.VpcEndpoint("my_endpoint")
 * allow_all = aws.ec2.SecurityGroupRule("allow_all",
 *     type="egress",
 *     to_port=0,
 *     protocol="-1",
 *     prefix_list_ids=[my_endpoint&#46;prefix_list_id],
 *     from_port=0,
 *     security_group_id="sg-123456")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     // ...
 *     var myEndpoint = new Aws.Ec2.VpcEndpoint("my_endpoint");
 *     var allowAll = new Aws.Ec2.SecurityGroupRule("allow_all", new()
 *     {
 *         Type = "egress",
 *         ToPort = 0,
 *         Protocol = "-1",
 *         PrefixListIds = new[]
 *         {
 *             myEndpoint.PrefixListId,
 *         },
 *         FromPort = 0,
 *         SecurityGroupId = "sg-123456",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// ...
 * 		myEndpoint, err := ec2.NewVpcEndpoint(ctx, "my_endpoint", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewSecurityGroupRule(ctx, "allow_all", &ec2.SecurityGroupRuleArgs{
 * 			Type:     pulumi.String("egress"),
 * 			ToPort:   pulumi.Int(0),
 * 			Protocol: pulumi.String("-1"),
 * 			PrefixListIds: pulumi.StringArray{
 * 				myEndpoint.PrefixListId,
 * 			},
 * 			FromPort:        pulumi.Int(0),
 * 			SecurityGroupId: pulumi.String("sg-123456"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcEndpoint;
 * import com.pulumi.aws.ec2.SecurityGroupRule;
 * import com.pulumi.aws.ec2.SecurityGroupRuleArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // ...
 *         var myEndpoint = new VpcEndpoint("myEndpoint");
 *         var allowAll = new SecurityGroupRule("allowAll", SecurityGroupRuleArgs.builder()
 *             .type("egress")
 *             .toPort(0)
 *             .protocol("-1")
 *             .prefixListIds(myEndpoint.prefixListId())
 *             .fromPort(0)
 *             .securityGroupId("sg-123456")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   allowAll:
 *     type: aws:ec2:SecurityGroupRule
 *     name: allow_all
 *     properties:
 *       type: egress
 *       toPort: 0
 *       protocol: '-1'
 *       prefixListIds:
 *         - ${myEndpoint.prefixListId}
 *       fromPort: 0
 *       securityGroupId: sg-123456
 *   # ...
 *   myEndpoint:
 *     type: aws:ec2:VpcEndpoint
 *     name: my_endpoint
 * ```
 * <!--End PulumiCodeChooser -->
 * You can also find a specific Prefix List using the `aws.ec2.getPrefixList`
 * or `ec2_managed_prefix_list` data sources:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getRegion({});
 * const s3 = current.then(current => aws.ec2.getPrefixList({
 *     name: `com.amazonaws.${current.name}.s3`,
 * }));
 * const s3GatewayEgress = new aws.ec2.SecurityGroupRule("s3_gateway_egress", {
 *     description: "S3 Gateway Egress",
 *     type: "egress",
 *     securityGroupId: "sg-123456",
 *     fromPort: 443,
 *     toPort: 443,
 *     protocol: aws.ec2.ProtocolType.TCP,
 *     prefixListIds: [s3&#46;then(s3 => s3&#46;id)],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_region()
 * s3 = aws.ec2.get_prefix_list(name=f"com.amazonaws.{current.name}.s3")
 * s3_gateway_egress = aws.ec2.SecurityGroupRule("s3_gateway_egress",
 *     description="S3 Gateway Egress",
 *     type="egress",
 *     security_group_id="sg-123456",
 *     from_port=443,
 *     to_port=443,
 *     protocol=aws.ec2.ProtocolType.TCP,
 *     prefix_list_ids=[s3&#46;id])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetRegion.Invoke();
 *     var s3 = Aws.Ec2.GetPrefixList.Invoke(new()
 *     {
 *         Name = $"com.amazonaws.{current.Apply(getRegionResult => getRegionResult.Name)}.s3",
 *     });
 *     var s3GatewayEgress = new Aws.Ec2.SecurityGroupRule("s3_gateway_egress", new()
 *     {
 *         Description = "S3 Gateway Egress",
 *         Type = "egress",
 *         SecurityGroupId = "sg-123456",
 *         FromPort = 443,
 *         ToPort = 443,
 *         Protocol = Aws.Ec2.ProtocolType.TCP,
 *         PrefixListIds = new[]
 *         {
 *             s3.Apply(getPrefixListResult => getPrefixListResult.Id),
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		s3, err := ec2.GetPrefixList(ctx, &ec2.GetPrefixListArgs{
 * 			Name: pulumi.StringRef(fmt.Sprintf("com.amazonaws.%v.s3", current.Name)),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewSecurityGroupRule(ctx, "s3_gateway_egress", &ec2.SecurityGroupRuleArgs{
 * 			Description:     pulumi.String("S3 Gateway Egress"),
 * 			Type:            pulumi.String("egress"),
 * 			SecurityGroupId: pulumi.String("sg-123456"),
 * 			FromPort:        pulumi.Int(443),
 * 			ToPort:          pulumi.Int(443),
 * 			Protocol:        pulumi.String(ec2.ProtocolTypeTCP),
 * 			PrefixListIds: pulumi.StringArray{
 * 				pulumi.String(s3.Id),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetPrefixListArgs;
 * import com.pulumi.aws.ec2.SecurityGroupRule;
 * import com.pulumi.aws.ec2.SecurityGroupRuleArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getRegion();
 *         final var s3 = Ec2Functions.getPrefixList(GetPrefixListArgs.builder()
 *             .name(String.format("com.amazonaws.%s.s3", current.applyValue(getRegionResult -> getRegionResult.name())))
 *             .build());
 *         var s3GatewayEgress = new SecurityGroupRule("s3GatewayEgress", SecurityGroupRuleArgs.builder()
 *             .description("S3 Gateway Egress")
 *             .type("egress")
 *             .securityGroupId("sg-123456")
 *             .fromPort(443)
 *             .toPort(443)
 *             .protocol("tcp")
 *             .prefixListIds(s3.applyValue(getPrefixListResult -> getPrefixListResult.id()))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   s3GatewayEgress:
 *     type: aws:ec2:SecurityGroupRule
 *     name: s3_gateway_egress
 *     properties:
 *       description: S3 Gateway Egress
 *       type: egress
 *       securityGroupId: sg-123456
 *       fromPort: 443
 *       toPort: 443
 *       protocol: tcp
 *       prefixListIds:
 *         - ${s3.id}
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 *   s3:
 *     fn::invoke:
 *       Function: aws:ec2:getPrefixList
 *       Arguments:
 *         name: com.amazonaws.${current.name}.s3
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Import a rule with various IPv4 and IPv6 source CIDR blocks:
 * Import a rule, applicable to all ports, with a protocol other than TCP/UDP/ICMP/ICMPV6/ALL, e.g., Multicast Transport Protocol (MTP), using the IANA protocol number. For example: 92.
 * Import a default any/any egress rule to 0.0.0.0/0:
 * Import an egress rule with a prefix list ID destination:
 * Import a rule applicable to all protocols and ports with a security group source:
 * Import a rule that has itself and an IPv6 CIDR block as sources:
 * __Using `pulumi import` to import__ Security Group Rules using the `security_group_id`, `type`, `protocol`, `from_port`, `to_port`, and source(s)/destination(s) (such as a `cidr_block`) separated by underscores (`_`). All parts are required. For example:
 * __NOTE:__ Not all rule permissions (e.g., not all of a rule's CIDR blocks) need to be imported for this provider to manage rule permissions. However, importing some of a rule's permissions but not others, and then making changes to the rule will result in the creation of an additional rule to capture the updated permissions. Rule permissions that were not imported are left intact in the original rule.
 * Import an ingress rule in security group `sg-6e616f6d69` for TCP port 8000 with an IPv4 destination CIDR of `10.0.3.0/24`:
 * ```sh
 * $ pulumi import aws:ec2/securityGroupRule:SecurityGroupRule ingress sg-6e616f6d69_ingress_tcp_8000_8000_10.0.3.0/24
 * ```
 * Import a rule with various IPv4 and IPv6 source CIDR blocks:
 * ```sh
 * $ pulumi import aws:ec2/securityGroupRule:SecurityGroupRule ingress sg-4973616163_ingress_tcp_100_121_10.1.0.0/16_2001:db8::/48_10.2.0.0/16_2002:db8::/48
 * ```
 * Import a rule, applicable to all ports, with a protocol other than TCP/UDP/ICMP/ICMPV6/ALL, e.g., Multicast Transport Protocol (MTP), using the IANA protocol number. For example: 92.
 * ```sh
 * $ pulumi import aws:ec2/securityGroupRule:SecurityGroupRule ingress sg-6777656e646f6c796e_ingress_92_0_65536_10.0.3.0/24_10.0.4.0/24
 * ```
 * Import a default any/any egress rule to 0.0.0.0/0:
 * ```sh
 * $ pulumi import aws:ec2/securityGroupRule:SecurityGroupRule default_egress sg-6777656e646f6c796e_egress_all_0_0_0.0.0.0/0
 * ```
 * Import an egress rule with a prefix list ID destination:
 * ```sh
 * $ pulumi import aws:ec2/securityGroupRule:SecurityGroupRule egress sg-62726f6479_egress_tcp_8000_8000_pl-6469726b
 * ```
 * Import a rule applicable to all protocols and ports with a security group source:
 * ```sh
 * $ pulumi import aws:ec2/securityGroupRule:SecurityGroupRule ingress_rule sg-7472697374616e_ingress_all_0_65536_sg-6176657279
 * ```
 * Import a rule that has itself and an IPv6 CIDR block as sources:
 * ```sh
 * $ pulumi import aws:ec2/securityGroupRule:SecurityGroupRule rule_name sg-656c65616e6f72_ingress_tcp_80_80_self_2001:db8::/48
 * ```
 * @property cidrBlocks List of CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
 * @property description Description of the rule.
 * @property fromPort Start port (or ICMP type number if protocol is "icmp" or "icmpv6").
 * @property ipv6CidrBlocks List of IPv6 CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
 * @property prefixListIds List of Prefix List IDs.
 * @property protocol Protocol. If not icmp, icmpv6, tcp, udp, or all use the [protocol number](https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)
 * @property securityGroupId Security group to apply this rule to.
 * @property self Whether the security group itself will be added as a source to this ingress rule. Cannot be specified with `cidr_blocks`, `ipv6_cidr_blocks`, or `source_security_group_id`.
 * @property sourceSecurityGroupId Security group id to allow access to/from, depending on the `type`. Cannot be specified with `cidr_blocks`, `ipv6_cidr_blocks`, or `self`.
 * @property toPort End port (or ICMP code if protocol is "icmp").
 * @property type Type of rule being created. Valid options are `ingress` (inbound)
 * or `egress` (outbound).
 * The following arguments are optional:
 * > **Note** Although `cidr_blocks`, `ipv6_cidr_blocks`, `prefix_list_ids`, and `source_security_group_id` are all marked as optional, you _must_ provide one of them in order to configure the source of the traffic.
 */
public data class SecurityGroupRuleArgs(
    public val cidrBlocks: Output<List<String>>? = null,
    public val description: Output<String>? = null,
    public val fromPort: Output<Int>? = null,
    public val ipv6CidrBlocks: Output<List<String>>? = null,
    public val prefixListIds: Output<List<String>>? = null,
    public val protocol: Output<Either<String, ProtocolType>>? = null,
    public val securityGroupId: Output<String>? = null,
    public val self: Output<Boolean>? = null,
    public val sourceSecurityGroupId: Output<String>? = null,
    public val toPort: Output<Int>? = null,
    public val type: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.SecurityGroupRuleArgs> {
    override fun toJava(): com.pulumi.aws.ec2.SecurityGroupRuleArgs =
        com.pulumi.aws.ec2.SecurityGroupRuleArgs.builder()
            .cidrBlocks(cidrBlocks?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .description(description?.applyValue({ args0 -> args0 }))
            .fromPort(fromPort?.applyValue({ args0 -> args0 }))
            .ipv6CidrBlocks(ipv6CidrBlocks?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .prefixListIds(prefixListIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .protocol(
                protocol?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .securityGroupId(securityGroupId?.applyValue({ args0 -> args0 }))
            .self(self?.applyValue({ args0 -> args0 }))
            .sourceSecurityGroupId(sourceSecurityGroupId?.applyValue({ args0 -> args0 }))
            .toPort(toPort?.applyValue({ args0 -> args0 }))
            .type(type?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [SecurityGroupRuleArgs].
 */
@PulumiTagMarker
public class SecurityGroupRuleArgsBuilder internal constructor() {
    private var cidrBlocks: Output<List<String>>? = null

    private var description: Output<String>? = null

    private var fromPort: Output<Int>? = null

    private var ipv6CidrBlocks: Output<List<String>>? = null

    private var prefixListIds: Output<List<String>>? = null

    private var protocol: Output<Either<String, ProtocolType>>? = null

    private var securityGroupId: Output<String>? = null

    private var self: Output<Boolean>? = null

    private var sourceSecurityGroupId: Output<String>? = null

    private var toPort: Output<Int>? = null

    private var type: Output<String>? = null

    /**
     * @param value List of CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("fhtpmwbjxkowpsol")
    public suspend fun cidrBlocks(`value`: Output<List<String>>) {
        this.cidrBlocks = value
    }

    @JvmName("tjchrjjawxbtwkpk")
    public suspend fun cidrBlocks(vararg values: Output<String>) {
        this.cidrBlocks = Output.all(values.asList())
    }

    /**
     * @param values List of CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("ehpbbyiorriflsjy")
    public suspend fun cidrBlocks(values: List<Output<String>>) {
        this.cidrBlocks = Output.all(values)
    }

    /**
     * @param value Description of the rule.
     */
    @JvmName("panxgsrgbvffauta")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value Start port (or ICMP type number if protocol is "icmp" or "icmpv6").
     */
    @JvmName("ikabyhijgjdbpfsi")
    public suspend fun fromPort(`value`: Output<Int>) {
        this.fromPort = value
    }

    /**
     * @param value List of IPv6 CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("fxjueuwwwhlcjdst")
    public suspend fun ipv6CidrBlocks(`value`: Output<List<String>>) {
        this.ipv6CidrBlocks = value
    }

    @JvmName("eloqdtshykxenrhf")
    public suspend fun ipv6CidrBlocks(vararg values: Output<String>) {
        this.ipv6CidrBlocks = Output.all(values.asList())
    }

    /**
     * @param values List of IPv6 CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("benybyyfeaqhijdx")
    public suspend fun ipv6CidrBlocks(values: List<Output<String>>) {
        this.ipv6CidrBlocks = Output.all(values)
    }

    /**
     * @param value List of Prefix List IDs.
     */
    @JvmName("pfgtuhpjnqgrhlvi")
    public suspend fun prefixListIds(`value`: Output<List<String>>) {
        this.prefixListIds = value
    }

    @JvmName("ffyekjvapkhlimuv")
    public suspend fun prefixListIds(vararg values: Output<String>) {
        this.prefixListIds = Output.all(values.asList())
    }

    /**
     * @param values List of Prefix List IDs.
     */
    @JvmName("wpgbgqpcgbrpkbeg")
    public suspend fun prefixListIds(values: List<Output<String>>) {
        this.prefixListIds = Output.all(values)
    }

    /**
     * @param value Protocol. If not icmp, icmpv6, tcp, udp, or all use the [protocol number](https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)
     */
    @JvmName("puktdaufplxomvmv")
    public suspend fun protocol(`value`: Output<Either<String, ProtocolType>>) {
        this.protocol = value
    }

    /**
     * @param value Security group to apply this rule to.
     */
    @JvmName("shjtqestwwuyvwmw")
    public suspend fun securityGroupId(`value`: Output<String>) {
        this.securityGroupId = value
    }

    /**
     * @param value Whether the security group itself will be added as a source to this ingress rule. Cannot be specified with `cidr_blocks`, `ipv6_cidr_blocks`, or `source_security_group_id`.
     */
    @JvmName("jrynprepkhaxhcag")
    public suspend fun self(`value`: Output<Boolean>) {
        this.self = value
    }

    /**
     * @param value Security group id to allow access to/from, depending on the `type`. Cannot be specified with `cidr_blocks`, `ipv6_cidr_blocks`, or `self`.
     */
    @JvmName("cbbksmcrjuchnvdw")
    public suspend fun sourceSecurityGroupId(`value`: Output<String>) {
        this.sourceSecurityGroupId = value
    }

    /**
     * @param value End port (or ICMP code if protocol is "icmp").
     */
    @JvmName("evwxixmsvhngffkr")
    public suspend fun toPort(`value`: Output<Int>) {
        this.toPort = value
    }

    /**
     * @param value Type of rule being created. Valid options are `ingress` (inbound)
     * or `egress` (outbound).
     * The following arguments are optional:
     * > **Note** Although `cidr_blocks`, `ipv6_cidr_blocks`, `prefix_list_ids`, and `source_security_group_id` are all marked as optional, you _must_ provide one of them in order to configure the source of the traffic.
     */
    @JvmName("cciefnyqhqmvpldp")
    public suspend fun type(`value`: Output<String>) {
        this.type = value
    }

    /**
     * @param value List of CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("jpyhgibxgkwclgkk")
    public suspend fun cidrBlocks(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cidrBlocks = mapped
    }

    /**
     * @param values List of CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("ocefxtiuwcogkuho")
    public suspend fun cidrBlocks(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.cidrBlocks = mapped
    }

    /**
     * @param value Description of the rule.
     */
    @JvmName("mancalpmcaueqgvv")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value Start port (or ICMP type number if protocol is "icmp" or "icmpv6").
     */
    @JvmName("uyekytfxhqsaudiw")
    public suspend fun fromPort(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.fromPort = mapped
    }

    /**
     * @param value List of IPv6 CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("rxfsbexinslceqce")
    public suspend fun ipv6CidrBlocks(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipv6CidrBlocks = mapped
    }

    /**
     * @param values List of IPv6 CIDR blocks. Cannot be specified with `source_security_group_id` or `self`.
     */
    @JvmName("setoihiqrnuglwfb")
    public suspend fun ipv6CidrBlocks(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.ipv6CidrBlocks = mapped
    }

    /**
     * @param value List of Prefix List IDs.
     */
    @JvmName("jhrefvwkpoyvdsjk")
    public suspend fun prefixListIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.prefixListIds = mapped
    }

    /**
     * @param values List of Prefix List IDs.
     */
    @JvmName("twcliuidjsnomkpn")
    public suspend fun prefixListIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.prefixListIds = mapped
    }

    /**
     * @param value Protocol. If not icmp, icmpv6, tcp, udp, or all use the [protocol number](https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)
     */
    @JvmName("ywrdfxnsibtjbxxt")
    public suspend fun protocol(`value`: Either<String, ProtocolType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.protocol = mapped
    }

    /**
     * @param value Protocol. If not icmp, icmpv6, tcp, udp, or all use the [protocol number](https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)
     */
    @JvmName("kmjbdkautvmobwup")
    public fun protocol(`value`: String) {
        val toBeMapped = Either.ofLeft<String, ProtocolType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.protocol = mapped
    }

    /**
     * @param value Protocol. If not icmp, icmpv6, tcp, udp, or all use the [protocol number](https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)
     */
    @JvmName("qtneojkbmqbavweb")
    public fun protocol(`value`: ProtocolType) {
        val toBeMapped = Either.ofRight<String, ProtocolType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.protocol = mapped
    }

    /**
     * @param value Security group to apply this rule to.
     */
    @JvmName("axoxdtudxegaltxn")
    public suspend fun securityGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroupId = mapped
    }

    /**
     * @param value Whether the security group itself will be added as a source to this ingress rule. Cannot be specified with `cidr_blocks`, `ipv6_cidr_blocks`, or `source_security_group_id`.
     */
    @JvmName("huopkovhgxysicyu")
    public suspend fun self(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.self = mapped
    }

    /**
     * @param value Security group id to allow access to/from, depending on the `type`. Cannot be specified with `cidr_blocks`, `ipv6_cidr_blocks`, or `self`.
     */
    @JvmName("pipjdcpwoxwpfivu")
    public suspend fun sourceSecurityGroupId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceSecurityGroupId = mapped
    }

    /**
     * @param value End port (or ICMP code if protocol is "icmp").
     */
    @JvmName("cvvsasjnkqchdsqw")
    public suspend fun toPort(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.toPort = mapped
    }

    /**
     * @param value Type of rule being created. Valid options are `ingress` (inbound)
     * or `egress` (outbound).
     * The following arguments are optional:
     * > **Note** Although `cidr_blocks`, `ipv6_cidr_blocks`, `prefix_list_ids`, and `source_security_group_id` are all marked as optional, you _must_ provide one of them in order to configure the source of the traffic.
     */
    @JvmName("lyrihrnshjmacgsm")
    public suspend fun type(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.type = mapped
    }

    internal fun build(): SecurityGroupRuleArgs = SecurityGroupRuleArgs(
        cidrBlocks = cidrBlocks,
        description = description,
        fromPort = fromPort,
        ipv6CidrBlocks = ipv6CidrBlocks,
        prefixListIds = prefixListIds,
        protocol = protocol,
        securityGroupId = securityGroupId,
        self = self,
        sourceSecurityGroupId = sourceSecurityGroupId,
        toPort = toPort,
        type = type,
    )
}
