@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.SpotFleetRequestArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.SpotFleetRequestLaunchSpecificationArgs
import com.pulumi.aws.ec2.kotlin.inputs.SpotFleetRequestLaunchSpecificationArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.SpotFleetRequestLaunchTemplateConfigArgs
import com.pulumi.aws.ec2.kotlin.inputs.SpotFleetRequestLaunchTemplateConfigArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.SpotFleetRequestSpotMaintenanceStrategiesArgs
import com.pulumi.aws.ec2.kotlin.inputs.SpotFleetRequestSpotMaintenanceStrategiesArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an EC2 Spot Fleet Request resource. This allows a fleet of Spot
 * instances to be requested on the Spot market.
 * > **NOTE [AWS strongly discourages](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-best-practices.html#which-spot-request-method-to-use) the use of the legacy APIs called by this resource.
 * We recommend using the EC2 Fleet or Auto Scaling Group resources instead.
 * ## Example Usage
 * ### Using launch specifications
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * // Request a Spot fleet
 * const cheapCompute = new aws.ec2.SpotFleetRequest("cheap_compute", {
 *     iamFleetRole: "arn:aws:iam::12345678:role/spot-fleet",
 *     spotPrice: "0.03",
 *     allocationStrategy: "diversified",
 *     targetCapacity: 6,
 *     validUntil: "2019-11-04T20:44:20Z",
 *     launchSpecifications: [
 *         {
 *             instanceType: "m4.10xlarge",
 *             ami: "ami-1234",
 *             spotPrice: "2.793",
 *             placementTenancy: "dedicated",
 *             iamInstanceProfileArn: example.arn,
 *         },
 *         {
 *             instanceType: "m4.4xlarge",
 *             ami: "ami-5678",
 *             keyName: "my-key",
 *             spotPrice: "1.117",
 *             iamInstanceProfileArn: example.arn,
 *             availabilityZone: "us-west-1a",
 *             subnetId: "subnet-1234",
 *             weightedCapacity: "35",
 *             rootBlockDevices: [{
 *                 volumeSize: 300,
 *                 volumeType: "gp2",
 *             }],
 *             tags: {
 *                 Name: "spot-fleet-example",
 *             },
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * # Request a Spot fleet
 * cheap_compute = aws.ec2.SpotFleetRequest("cheap_compute",
 *     iam_fleet_role="arn:aws:iam::12345678:role/spot-fleet",
 *     spot_price="0.03",
 *     allocation_strategy="diversified",
 *     target_capacity=6,
 *     valid_until="2019-11-04T20:44:20Z",
 *     launch_specifications=[
 *         {
 *             "instance_type": "m4.10xlarge",
 *             "ami": "ami-1234",
 *             "spot_price": "2.793",
 *             "placement_tenancy": "dedicated",
 *             "iam_instance_profile_arn": example["arn"],
 *         },
 *         {
 *             "instance_type": "m4.4xlarge",
 *             "ami": "ami-5678",
 *             "key_name": "my-key",
 *             "spot_price": "1.117",
 *             "iam_instance_profile_arn": example["arn"],
 *             "availability_zone": "us-west-1a",
 *             "subnet_id": "subnet-1234",
 *             "weighted_capacity": "35",
 *             "root_block_devices": [{
 *                 "volume_size": 300,
 *                 "volume_type": "gp2",
 *             }],
 *             "tags": {
 *                 "name": "spot-fleet-example",
 *             },
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Request a Spot fleet
 *     var cheapCompute = new Aws.Ec2.SpotFleetRequest("cheap_compute", new()
 *     {
 *         IamFleetRole = "arn:aws:iam::12345678:role/spot-fleet",
 *         SpotPrice = "0.03",
 *         AllocationStrategy = "diversified",
 *         TargetCapacity = 6,
 *         ValidUntil = "2019-11-04T20:44:20Z",
 *         LaunchSpecifications = new[]
 *         {
 *             new Aws.Ec2.Inputs.SpotFleetRequestLaunchSpecificationArgs
 *             {
 *                 InstanceType = "m4.10xlarge",
 *                 Ami = "ami-1234",
 *                 SpotPrice = "2.793",
 *                 PlacementTenancy = "dedicated",
 *                 IamInstanceProfileArn = example.Arn,
 *             },
 *             new Aws.Ec2.Inputs.SpotFleetRequestLaunchSpecificationArgs
 *             {
 *                 InstanceType = "m4.4xlarge",
 *                 Ami = "ami-5678",
 *                 KeyName = "my-key",
 *                 SpotPrice = "1.117",
 *                 IamInstanceProfileArn = example.Arn,
 *                 AvailabilityZone = "us-west-1a",
 *                 SubnetId = "subnet-1234",
 *                 WeightedCapacity = "35",
 *                 RootBlockDevices = new[]
 *                 {
 *                     new Aws.Ec2.Inputs.SpotFleetRequestLaunchSpecificationRootBlockDeviceArgs
 *                     {
 *                         VolumeSize = 300,
 *                         VolumeType = "gp2",
 *                     },
 *                 },
 *                 Tags =
 *                 {
 *                     { "Name", "spot-fleet-example" },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Request a Spot fleet
 * 		_, err := ec2.NewSpotFleetRequest(ctx, "cheap_compute", &ec2.SpotFleetRequestArgs{
 * 			IamFleetRole:       pulumi.String("arn:aws:iam::12345678:role/spot-fleet"),
 * 			SpotPrice:          pulumi.String("0.03"),
 * 			AllocationStrategy: pulumi.String("diversified"),
 * 			TargetCapacity:     pulumi.Int(6),
 * 			ValidUntil:         pulumi.String("2019-11-04T20:44:20Z"),
 * 			LaunchSpecifications: ec2.SpotFleetRequestLaunchSpecificationArray{
 * 				&ec2.SpotFleetRequestLaunchSpecificationArgs{
 * 					InstanceType:          pulumi.String("m4.10xlarge"),
 * 					Ami:                   pulumi.String("ami-1234"),
 * 					SpotPrice:             pulumi.String("2.793"),
 * 					PlacementTenancy:      pulumi.String("dedicated"),
 * 					IamInstanceProfileArn: pulumi.Any(example.Arn),
 * 				},
 * 				&ec2.SpotFleetRequestLaunchSpecificationArgs{
 * 					InstanceType:          pulumi.String("m4.4xlarge"),
 * 					Ami:                   pulumi.String("ami-5678"),
 * 					KeyName:               pulumi.String("my-key"),
 * 					SpotPrice:             pulumi.String("1.117"),
 * 					IamInstanceProfileArn: pulumi.Any(example.Arn),
 * 					AvailabilityZone:      pulumi.String("us-west-1a"),
 * 					SubnetId:              pulumi.String("subnet-1234"),
 * 					WeightedCapacity:      pulumi.String("35"),
 * 					RootBlockDevices: ec2.SpotFleetRequestLaunchSpecificationRootBlockDeviceArray{
 * 						&ec2.SpotFleetRequestLaunchSpecificationRootBlockDeviceArgs{
 * 							VolumeSize: pulumi.Int(300),
 * 							VolumeType: pulumi.String("gp2"),
 * 						},
 * 					},
 * 					Tags: pulumi.StringMap{
 * 						"Name": pulumi.String("spot-fleet-example"),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.SpotFleetRequest;
 * import com.pulumi.aws.ec2.SpotFleetRequestArgs;
 * import com.pulumi.aws.ec2.inputs.SpotFleetRequestLaunchSpecificationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Request a Spot fleet
 *         var cheapCompute = new SpotFleetRequest("cheapCompute", SpotFleetRequestArgs.builder()
 *             .iamFleetRole("arn:aws:iam::12345678:role/spot-fleet")
 *             .spotPrice("0.03")
 *             .allocationStrategy("diversified")
 *             .targetCapacity(6)
 *             .validUntil("2019-11-04T20:44:20Z")
 *             .launchSpecifications(
 *                 SpotFleetRequestLaunchSpecificationArgs.builder()
 *                     .instanceType("m4.10xlarge")
 *                     .ami("ami-1234")
 *                     .spotPrice("2.793")
 *                     .placementTenancy("dedicated")
 *                     .iamInstanceProfileArn(example.arn())
 *                     .build(),
 *                 SpotFleetRequestLaunchSpecificationArgs.builder()
 *                     .instanceType("m4.4xlarge")
 *                     .ami("ami-5678")
 *                     .keyName("my-key")
 *                     .spotPrice("1.117")
 *                     .iamInstanceProfileArn(example.arn())
 *                     .availabilityZone("us-west-1a")
 *                     .subnetId("subnet-1234")
 *                     .weightedCapacity(35)
 *                     .rootBlockDevices(SpotFleetRequestLaunchSpecificationRootBlockDeviceArgs.builder()
 *                         .volumeSize("300")
 *                         .volumeType("gp2")
 *                         .build())
 *                     .tags(Map.of("Name", "spot-fleet-example"))
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # Request a Spot fleet
 *   cheapCompute:
 *     type: aws:ec2:SpotFleetRequest
 *     name: cheap_compute
 *     properties:
 *       iamFleetRole: arn:aws:iam::12345678:role/spot-fleet
 *       spotPrice: '0.03'
 *       allocationStrategy: diversified
 *       targetCapacity: 6
 *       validUntil: 2019-11-04T20:44:20Z
 *       launchSpecifications:
 *         - instanceType: m4.10xlarge
 *           ami: ami-1234
 *           spotPrice: '2.793'
 *           placementTenancy: dedicated
 *           iamInstanceProfileArn: ${example.arn}
 *         - instanceType: m4.4xlarge
 *           ami: ami-5678
 *           keyName: my-key
 *           spotPrice: '1.117'
 *           iamInstanceProfileArn: ${example.arn}
 *           availabilityZone: us-west-1a
 *           subnetId: subnet-1234
 *           weightedCapacity: 35
 *           rootBlockDevices:
 *             - volumeSize: '300'
 *               volumeType: gp2
 *           tags:
 *             Name: spot-fleet-example
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Using launch templates
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const foo = new aws.ec2.LaunchTemplate("foo", {
 *     name: "launch-template",
 *     imageId: "ami-516b9131",
 *     instanceType: "m1.small",
 *     keyName: "some-key",
 * });
 * const fooSpotFleetRequest = new aws.ec2.SpotFleetRequest("foo", {
 *     iamFleetRole: "arn:aws:iam::12345678:role/spot-fleet",
 *     spotPrice: "0.005",
 *     targetCapacity: 2,
 *     validUntil: "2019-11-04T20:44:20Z",
 *     launchTemplateConfigs: [{
 *         launchTemplateSpecification: {
 *             id: foo.id,
 *             version: foo.latestVersion,
 *         },
 *     }],
 * }, {
 *     dependsOn: [test_attach],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foo = aws.ec2.LaunchTemplate("foo",
 *     name="launch-template",
 *     image_id="ami-516b9131",
 *     instance_type="m1.small",
 *     key_name="some-key")
 * foo_spot_fleet_request = aws.ec2.SpotFleetRequest("foo",
 *     iam_fleet_role="arn:aws:iam::12345678:role/spot-fleet",
 *     spot_price="0.005",
 *     target_capacity=2,
 *     valid_until="2019-11-04T20:44:20Z",
 *     launch_template_configs=[{
 *         "launch_template_specification": {
 *             "id": foo.id,
 *             "version": foo.latest_version,
 *         },
 *     }],
 *     opts = pulumi.ResourceOptions(depends_on=[test_attach]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var foo = new Aws.Ec2.LaunchTemplate("foo", new()
 *     {
 *         Name = "launch-template",
 *         ImageId = "ami-516b9131",
 *         InstanceType = "m1.small",
 *         KeyName = "some-key",
 *     });
 *     var fooSpotFleetRequest = new Aws.Ec2.SpotFleetRequest("foo", new()
 *     {
 *         IamFleetRole = "arn:aws:iam::12345678:role/spot-fleet",
 *         SpotPrice = "0.005",
 *         TargetCapacity = 2,
 *         ValidUntil = "2019-11-04T20:44:20Z",
 *         LaunchTemplateConfigs = new[]
 *         {
 *             new Aws.Ec2.Inputs.SpotFleetRequestLaunchTemplateConfigArgs
 *             {
 *                 LaunchTemplateSpecification = new Aws.Ec2.Inputs.SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs
 *                 {
 *                     Id = foo.Id,
 *                     Version = foo.LatestVersion,
 *                 },
 *             },
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             test_attach,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		foo, err := ec2.NewLaunchTemplate(ctx, "foo", &ec2.LaunchTemplateArgs{
 * 			Name:         pulumi.String("launch-template"),
 * 			ImageId:      pulumi.String("ami-516b9131"),
 * 			InstanceType: pulumi.String("m1.small"),
 * 			KeyName:      pulumi.String("some-key"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewSpotFleetRequest(ctx, "foo", &ec2.SpotFleetRequestArgs{
 * 			IamFleetRole:   pulumi.String("arn:aws:iam::12345678:role/spot-fleet"),
 * 			SpotPrice:      pulumi.String("0.005"),
 * 			TargetCapacity: pulumi.Int(2),
 * 			ValidUntil:     pulumi.String("2019-11-04T20:44:20Z"),
 * 			LaunchTemplateConfigs: ec2.SpotFleetRequestLaunchTemplateConfigArray{
 * 				&ec2.SpotFleetRequestLaunchTemplateConfigArgs{
 * 					LaunchTemplateSpecification: &ec2.SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs{
 * 						Id:      foo.ID(),
 * 						Version: foo.LatestVersion,
 * 					},
 * 				},
 * 			},
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			test_attach,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.LaunchTemplate;
 * import com.pulumi.aws.ec2.LaunchTemplateArgs;
 * import com.pulumi.aws.ec2.SpotFleetRequest;
 * import com.pulumi.aws.ec2.SpotFleetRequestArgs;
 * import com.pulumi.aws.ec2.inputs.SpotFleetRequestLaunchTemplateConfigArgs;
 * import com.pulumi.aws.ec2.inputs.SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var foo = new LaunchTemplate("foo", LaunchTemplateArgs.builder()
 *             .name("launch-template")
 *             .imageId("ami-516b9131")
 *             .instanceType("m1.small")
 *             .keyName("some-key")
 *             .build());
 *         var fooSpotFleetRequest = new SpotFleetRequest("fooSpotFleetRequest", SpotFleetRequestArgs.builder()
 *             .iamFleetRole("arn:aws:iam::12345678:role/spot-fleet")
 *             .spotPrice("0.005")
 *             .targetCapacity(2)
 *             .validUntil("2019-11-04T20:44:20Z")
 *             .launchTemplateConfigs(SpotFleetRequestLaunchTemplateConfigArgs.builder()
 *                 .launchTemplateSpecification(SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs.builder()
 *                     .id(foo.id())
 *                     .version(foo.latestVersion())
 *                     .build())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(test_attach)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:ec2:LaunchTemplate
 *     properties:
 *       name: launch-template
 *       imageId: ami-516b9131
 *       instanceType: m1.small
 *       keyName: some-key
 *   fooSpotFleetRequest:
 *     type: aws:ec2:SpotFleetRequest
 *     name: foo
 *     properties:
 *       iamFleetRole: arn:aws:iam::12345678:role/spot-fleet
 *       spotPrice: '0.005'
 *       targetCapacity: 2
 *       validUntil: 2019-11-04T20:44:20Z
 *       launchTemplateConfigs:
 *         - launchTemplateSpecification:
 *             id: ${foo.id}
 *             version: ${foo.latestVersion}
 *     options:
 *       dependson:
 *         - ${["test-attach"]}
 * ```
 * <!--End PulumiCodeChooser -->
 * > **NOTE:** This provider does not support the functionality where multiple `subnet_id` or `availability_zone` parameters can be specified in the same
 * launch configuration block. If you want to specify multiple values, then separate launch configuration blocks should be used or launch template overrides should be configured, one per subnet:
 * ### Using multiple launch specifications
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const foo = new aws.ec2.SpotFleetRequest("foo", {
 *     iamFleetRole: "arn:aws:iam::12345678:role/spot-fleet",
 *     spotPrice: "0.005",
 *     targetCapacity: 2,
 *     validUntil: "2019-11-04T20:44:20Z",
 *     launchSpecifications: [
 *         {
 *             instanceType: "m1.small",
 *             ami: "ami-d06a90b0",
 *             keyName: "my-key",
 *             availabilityZone: "us-west-2a",
 *         },
 *         {
 *             instanceType: "m5.large",
 *             ami: "ami-d06a90b0",
 *             keyName: "my-key",
 *             availabilityZone: "us-west-2a",
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foo = aws.ec2.SpotFleetRequest("foo",
 *     iam_fleet_role="arn:aws:iam::12345678:role/spot-fleet",
 *     spot_price="0.005",
 *     target_capacity=2,
 *     valid_until="2019-11-04T20:44:20Z",
 *     launch_specifications=[
 *         {
 *             "instance_type": "m1.small",
 *             "ami": "ami-d06a90b0",
 *             "key_name": "my-key",
 *             "availability_zone": "us-west-2a",
 *         },
 *         {
 *             "instance_type": "m5.large",
 *             "ami": "ami-d06a90b0",
 *             "key_name": "my-key",
 *             "availability_zone": "us-west-2a",
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var foo = new Aws.Ec2.SpotFleetRequest("foo", new()
 *     {
 *         IamFleetRole = "arn:aws:iam::12345678:role/spot-fleet",
 *         SpotPrice = "0.005",
 *         TargetCapacity = 2,
 *         ValidUntil = "2019-11-04T20:44:20Z",
 *         LaunchSpecifications = new[]
 *         {
 *             new Aws.Ec2.Inputs.SpotFleetRequestLaunchSpecificationArgs
 *             {
 *                 InstanceType = "m1.small",
 *                 Ami = "ami-d06a90b0",
 *                 KeyName = "my-key",
 *                 AvailabilityZone = "us-west-2a",
 *             },
 *             new Aws.Ec2.Inputs.SpotFleetRequestLaunchSpecificationArgs
 *             {
 *                 InstanceType = "m5.large",
 *                 Ami = "ami-d06a90b0",
 *                 KeyName = "my-key",
 *                 AvailabilityZone = "us-west-2a",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewSpotFleetRequest(ctx, "foo", &ec2.SpotFleetRequestArgs{
 * 			IamFleetRole:   pulumi.String("arn:aws:iam::12345678:role/spot-fleet"),
 * 			SpotPrice:      pulumi.String("0.005"),
 * 			TargetCapacity: pulumi.Int(2),
 * 			ValidUntil:     pulumi.String("2019-11-04T20:44:20Z"),
 * 			LaunchSpecifications: ec2.SpotFleetRequestLaunchSpecificationArray{
 * 				&ec2.SpotFleetRequestLaunchSpecificationArgs{
 * 					InstanceType:     pulumi.String("m1.small"),
 * 					Ami:              pulumi.String("ami-d06a90b0"),
 * 					KeyName:          pulumi.String("my-key"),
 * 					AvailabilityZone: pulumi.String("us-west-2a"),
 * 				},
 * 				&ec2.SpotFleetRequestLaunchSpecificationArgs{
 * 					InstanceType:     pulumi.String("m5.large"),
 * 					Ami:              pulumi.String("ami-d06a90b0"),
 * 					KeyName:          pulumi.String("my-key"),
 * 					AvailabilityZone: pulumi.String("us-west-2a"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.SpotFleetRequest;
 * import com.pulumi.aws.ec2.SpotFleetRequestArgs;
 * import com.pulumi.aws.ec2.inputs.SpotFleetRequestLaunchSpecificationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var foo = new SpotFleetRequest("foo", SpotFleetRequestArgs.builder()
 *             .iamFleetRole("arn:aws:iam::12345678:role/spot-fleet")
 *             .spotPrice("0.005")
 *             .targetCapacity(2)
 *             .validUntil("2019-11-04T20:44:20Z")
 *             .launchSpecifications(
 *                 SpotFleetRequestLaunchSpecificationArgs.builder()
 *                     .instanceType("m1.small")
 *                     .ami("ami-d06a90b0")
 *                     .keyName("my-key")
 *                     .availabilityZone("us-west-2a")
 *                     .build(),
 *                 SpotFleetRequestLaunchSpecificationArgs.builder()
 *                     .instanceType("m5.large")
 *                     .ami("ami-d06a90b0")
 *                     .keyName("my-key")
 *                     .availabilityZone("us-west-2a")
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:ec2:SpotFleetRequest
 *     properties:
 *       iamFleetRole: arn:aws:iam::12345678:role/spot-fleet
 *       spotPrice: '0.005'
 *       targetCapacity: 2
 *       validUntil: 2019-11-04T20:44:20Z
 *       launchSpecifications:
 *         - instanceType: m1.small
 *           ami: ami-d06a90b0
 *           keyName: my-key
 *           availabilityZone: us-west-2a
 *         - instanceType: m5.large
 *           ami: ami-d06a90b0
 *           keyName: my-key
 *           availabilityZone: us-west-2a
 * ```
 * <!--End PulumiCodeChooser -->
 * > In this example, we use a `dynamic` block to define zero or more `launch_specification` blocks, producing one for each element in the list of subnet ids.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const config = new pulumi.Config();
 * const subnets = config.requireObject("subnets");
 * const example = new aws.ec2.SpotFleetRequest("example", {
 *     launchSpecifications: .map(s => ({
 *         subnetId: s[1],
 *     })).map((v, k) => ({key: k, value: v})).map(entry => ({
 *         ami: "ami-1234",
 *         instanceType: "m4.4xlarge",
 *         subnetId: entry.value.subnetId,
 *         vpcSecurityGroupIds: "sg-123456",
 *         rootBlockDevices: [{
 *             volumeSize: 8,
 *             volumeType: "gp2",
 *             deleteOnTermination: true,
 *         }],
 *         tags: {
 *             Name: "Spot Node",
 *             tag_builder: "builder",
 *         },
 *     })),
 *     iamFleetRole: "arn:aws:iam::12345678:role/spot-fleet",
 *     targetCapacity: 3,
 *     validUntil: "2019-11-04T20:44:20Z",
 *     allocationStrategy: "lowestPrice",
 *     fleetType: "request",
 *     waitForFulfillment: true,
 *     terminateInstancesWithExpiration: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * config = pulumi.Config()
 * subnets = config.require_object("subnets")
 * example = aws.ec2.SpotFleetRequest("example",
 *     launch_specifications=[{
 *         "ami": "ami-1234",
 *         "instance_type": "m4.4xlarge",
 *         "subnet_id": entry["value"]["subnetId"],
 *         "vpc_security_group_ids": "sg-123456",
 *         "root_block_devices": [{
 *             "volume_size": 8,
 *             "volume_type": "gp2",
 *             "delete_on_termination": True,
 *         }],
 *         "tags": {
 *             "name": "Spot Node",
 *             "tag_builder": "builder",
 *         },
 *     } for entry in [{"key": k, "value": v} for k, v in [{
 *         "subnetId": s[1],
 *     } for s in subnets]]],
 *     iam_fleet_role="arn:aws:iam::12345678:role/spot-fleet",
 *     target_capacity=3,
 *     valid_until="2019-11-04T20:44:20Z",
 *     allocation_strategy="lowestPrice",
 *     fleet_type="request",
 *     wait_for_fulfillment=True,
 *     terminate_instances_with_expiration=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var subnets = config.RequireObject<dynamic>("subnets");
 *     var example = new Aws.Ec2.SpotFleetRequest("example", new()
 *     {
 *         LaunchSpecifications = .Select(s =>
 *         {
 *             return
 *             {
 *                 { "subnetId", s[1] },
 *             };
 *         }).ToList().Select((v, k) => new { Key = k, Value = v }).Select(entry =>
 *         {
 *             return new Aws.Ec2.Inputs.SpotFleetRequestLaunchSpecificationArgs
 *             {
 *                 Ami = "ami-1234",
 *                 InstanceType = "m4.4xlarge",
 *                 SubnetId = entry.Value.SubnetId,
 *                 VpcSecurityGroupIds = "sg-123456",
 *                 RootBlockDevices = new[]
 *                 {
 *                     new Aws.Ec2.Inputs.SpotFleetRequestLaunchSpecificationRootBlockDeviceArgs
 *                     {
 *                         VolumeSize = 8,
 *                         VolumeType = "gp2",
 *                         DeleteOnTermination = true,
 *                     },
 *                 },
 *                 Tags =
 *                 {
 *                     { "Name", "Spot Node" },
 *                     { "tag_builder", "builder" },
 *                 },
 *             };
 *         }).ToList(),
 *         IamFleetRole = "arn:aws:iam::12345678:role/spot-fleet",
 *         TargetCapacity = 3,
 *         ValidUntil = "2019-11-04T20:44:20Z",
 *         AllocationStrategy = "lowestPrice",
 *         FleetType = "request",
 *         WaitForFulfillment = true,
 *         TerminateInstancesWithExpiration = true,
 *     });
 * });
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Using multiple launch configurations
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = aws.ec2.getSubnets({
 *     filters: [{
 *         name: "vpc-id",
 *         values: [vpcId],
 *     }],
 * });
 * const foo = new aws.ec2.LaunchTemplate("foo", {
 *     name: "launch-template",
 *     imageId: "ami-516b9131",
 *     instanceType: "m1.small",
 *     keyName: "some-key",
 * });
 * const fooSpotFleetRequest = new aws.ec2.SpotFleetRequest("foo", {
 *     iamFleetRole: "arn:aws:iam::12345678:role/spot-fleet",
 *     spotPrice: "0.005",
 *     targetCapacity: 2,
 *     validUntil: "2019-11-04T20:44:20Z",
 *     launchTemplateConfigs: [{
 *         launchTemplateSpecification: {
 *             id: foo.id,
 *             version: foo.latestVersion,
 *         },
 *         overrides: [
 *             {
 *                 subnetId: example.then(example => example.ids?.[0]),
 *             },
 *             {
 *                 subnetId: example.then(example => example.ids?.[1]),
 *             },
 *             {
 *                 subnetId: example.then(example => example.ids?.[2]),
 *             },
 *         ],
 *     }],
 * }, {
 *     dependsOn: [test_attach],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ec2.get_subnets(filters=[{
 *     "name": "vpc-id",
 *     "values": [vpc_id],
 * }])
 * foo = aws.ec2.LaunchTemplate("foo",
 *     name="launch-template",
 *     image_id="ami-516b9131",
 *     instance_type="m1.small",
 *     key_name="some-key")
 * foo_spot_fleet_request = aws.ec2.SpotFleetRequest("foo",
 *     iam_fleet_role="arn:aws:iam::12345678:role/spot-fleet",
 *     spot_price="0.005",
 *     target_capacity=2,
 *     valid_until="2019-11-04T20:44:20Z",
 *     launch_template_configs=[{
 *         "launch_template_specification": {
 *             "id": foo.id,
 *             "version": foo.latest_version,
 *         },
 *         "overrides": [
 *             {
 *                 "subnet_id": example.ids[0],
 *             },
 *             {
 *                 "subnet_id": example.ids[1],
 *             },
 *             {
 *                 "subnet_id": example.ids[2],
 *             },
 *         ],
 *     }],
 *     opts = pulumi.ResourceOptions(depends_on=[test_attach]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = Aws.Ec2.GetSubnets.Invoke(new()
 *     {
 *         Filters = new[]
 *         {
 *             new Aws.Ec2.Inputs.GetSubnetsFilterInputArgs
 *             {
 *                 Name = "vpc-id",
 *                 Values = new[]
 *                 {
 *                     vpcId,
 *                 },
 *             },
 *         },
 *     });
 *     var foo = new Aws.Ec2.LaunchTemplate("foo", new()
 *     {
 *         Name = "launch-template",
 *         ImageId = "ami-516b9131",
 *         InstanceType = "m1.small",
 *         KeyName = "some-key",
 *     });
 *     var fooSpotFleetRequest = new Aws.Ec2.SpotFleetRequest("foo", new()
 *     {
 *         IamFleetRole = "arn:aws:iam::12345678:role/spot-fleet",
 *         SpotPrice = "0.005",
 *         TargetCapacity = 2,
 *         ValidUntil = "2019-11-04T20:44:20Z",
 *         LaunchTemplateConfigs = new[]
 *         {
 *             new Aws.Ec2.Inputs.SpotFleetRequestLaunchTemplateConfigArgs
 *             {
 *                 LaunchTemplateSpecification = new Aws.Ec2.Inputs.SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs
 *                 {
 *                     Id = foo.Id,
 *                     Version = foo.LatestVersion,
 *                 },
 *                 Overrides = new[]
 *                 {
 *                     new Aws.Ec2.Inputs.SpotFleetRequestLaunchTemplateConfigOverrideArgs
 *                     {
 *                         SubnetId = example.Apply(getSubnetsResult => getSubnetsResult.Ids[0]),
 *                     },
 *                     new Aws.Ec2.Inputs.SpotFleetRequestLaunchTemplateConfigOverrideArgs
 *                     {
 *                         SubnetId = example.Apply(getSubnetsResult => getSubnetsResult.Ids[1]),
 *                     },
 *                     new Aws.Ec2.Inputs.SpotFleetRequestLaunchTemplateConfigOverrideArgs
 *                     {
 *                         SubnetId = example.Apply(getSubnetsResult => getSubnetsResult.Ids[2]),
 *                     },
 *                 },
 *             },
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             test_attach,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * example, err := ec2.GetSubnets(ctx, &ec2.GetSubnetsArgs{
 * Filters: []ec2.GetSubnetsFilter{
 * {
 * Name: "vpc-id",
 * Values: interface{}{
 * vpcId,
 * },
 * },
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * foo, err := ec2.NewLaunchTemplate(ctx, "foo", &ec2.LaunchTemplateArgs{
 * Name: pulumi.String("launch-template"),
 * ImageId: pulumi.String("ami-516b9131"),
 * InstanceType: pulumi.String("m1.small"),
 * KeyName: pulumi.String("some-key"),
 * })
 * if err != nil {
 * return err
 * }
 * _, err = ec2.NewSpotFleetRequest(ctx, "foo", &ec2.SpotFleetRequestArgs{
 * IamFleetRole: pulumi.String("arn:aws:iam::12345678:role/spot-fleet"),
 * SpotPrice: pulumi.String("0.005"),
 * TargetCapacity: pulumi.Int(2),
 * ValidUntil: pulumi.String("2019-11-04T20:44:20Z"),
 * LaunchTemplateConfigs: ec2.SpotFleetRequestLaunchTemplateConfigArray{
 * &ec2.SpotFleetRequestLaunchTemplateConfigArgs{
 * LaunchTemplateSpecification: &ec2.SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs{
 * Id: foo.ID(),
 * Version: foo.LatestVersion,
 * },
 * Overrides: ec2.SpotFleetRequestLaunchTemplateConfigOverrideArray{
 * &ec2.SpotFleetRequestLaunchTemplateConfigOverrideArgs{
 * SubnetId: pulumi.String(example.Ids[0]),
 * },
 * &ec2.SpotFleetRequestLaunchTemplateConfigOverrideArgs{
 * SubnetId: pulumi.String(example.Ids[1]),
 * },
 * &ec2.SpotFleetRequestLaunchTemplateConfigOverrideArgs{
 * SubnetId: pulumi.String(example.Ids[2]),
 * },
 * },
 * },
 * },
 * }, pulumi.DependsOn([]pulumi.Resource{
 * test_attach,
 * }))
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetSubnetsArgs;
 * import com.pulumi.aws.ec2.LaunchTemplate;
 * import com.pulumi.aws.ec2.LaunchTemplateArgs;
 * import com.pulumi.aws.ec2.SpotFleetRequest;
 * import com.pulumi.aws.ec2.SpotFleetRequestArgs;
 * import com.pulumi.aws.ec2.inputs.SpotFleetRequestLaunchTemplateConfigArgs;
 * import com.pulumi.aws.ec2.inputs.SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = Ec2Functions.getSubnets(GetSubnetsArgs.builder()
 *             .filters(GetSubnetsFilterArgs.builder()
 *                 .name("vpc-id")
 *                 .values(vpcId)
 *                 .build())
 *             .build());
 *         var foo = new LaunchTemplate("foo", LaunchTemplateArgs.builder()
 *             .name("launch-template")
 *             .imageId("ami-516b9131")
 *             .instanceType("m1.small")
 *             .keyName("some-key")
 *             .build());
 *         var fooSpotFleetRequest = new SpotFleetRequest("fooSpotFleetRequest", SpotFleetRequestArgs.builder()
 *             .iamFleetRole("arn:aws:iam::12345678:role/spot-fleet")
 *             .spotPrice("0.005")
 *             .targetCapacity(2)
 *             .validUntil("2019-11-04T20:44:20Z")
 *             .launchTemplateConfigs(SpotFleetRequestLaunchTemplateConfigArgs.builder()
 *                 .launchTemplateSpecification(SpotFleetRequestLaunchTemplateConfigLaunchTemplateSpecificationArgs.builder()
 *                     .id(foo.id())
 *                     .version(foo.latestVersion())
 *                     .build())
 *                 .overrides(
 *                     SpotFleetRequestLaunchTemplateConfigOverrideArgs.builder()
 *                         .subnetId(example.applyValue(getSubnetsResult -> getSubnetsResult.ids()[0]))
 *                         .build(),
 *                     SpotFleetRequestLaunchTemplateConfigOverrideArgs.builder()
 *                         .subnetId(example.applyValue(getSubnetsResult -> getSubnetsResult.ids()[1]))
 *                         .build(),
 *                     SpotFleetRequestLaunchTemplateConfigOverrideArgs.builder()
 *                         .subnetId(example.applyValue(getSubnetsResult -> getSubnetsResult.ids()[2]))
 *                         .build())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(test_attach)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:ec2:LaunchTemplate
 *     properties:
 *       name: launch-template
 *       imageId: ami-516b9131
 *       instanceType: m1.small
 *       keyName: some-key
 *   fooSpotFleetRequest:
 *     type: aws:ec2:SpotFleetRequest
 *     name: foo
 *     properties:
 *       iamFleetRole: arn:aws:iam::12345678:role/spot-fleet
 *       spotPrice: '0.005'
 *       targetCapacity: 2
 *       validUntil: 2019-11-04T20:44:20Z
 *       launchTemplateConfigs:
 *         - launchTemplateSpecification:
 *             id: ${foo.id}
 *             version: ${foo.latestVersion}
 *           overrides:
 *             - subnetId: ${example.ids[0]}
 *             - subnetId: ${example.ids[1]}
 *             - subnetId: ${example.ids[2]}
 *     options:
 *       dependson:
 *         - ${["test-attach"]}
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: aws:ec2:getSubnets
 *       Arguments:
 *         filters:
 *           - name: vpc-id
 *             values:
 *               - ${vpcId}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Spot Fleet Requests using `id`. For example:
 * ```sh
 * $ pulumi import aws:ec2/spotFleetRequest:SpotFleetRequest fleet sfr-005e9ec8-5546-4c31-b317-31a62325411e
 * ```
 * @property allocationStrategy Indicates how to allocate the target capacity across
 * the Spot pools specified by the Spot fleet request. Valid values: `lowestPrice`, `diversified`, `capacityOptimized`, `capacityOptimizedPrioritized`, and `priceCapacityOptimized`. The default is
 * `lowestPrice`.
 * @property context Reserved.
 * @property excessCapacityTerminationPolicy Indicates whether running Spot
 * instances should be terminated if the target capacity of the Spot fleet
 * request is decreased below the current size of the Spot fleet.
 * @property fleetType The type of fleet request. Indicates whether the Spot Fleet only requests the target
 * capacity or also attempts to maintain it. Default is `maintain`.
 * @property iamFleetRole Grants the Spot fleet permission to terminate
 * Spot instances on your behalf when you cancel its Spot fleet request using
 * CancelSpotFleetRequests or when the Spot fleet request expires, if you set
 * terminateInstancesWithExpiration.
 * @property instanceInterruptionBehaviour Indicates whether a Spot
 * instance stops or terminates when it is interrupted. Default is
 * `terminate`.
 * @property instancePoolsToUseCount The number of Spot pools across which to allocate your target Spot capacity.
 * Valid only when `allocation_strategy` is set to `lowestPrice`. Spot Fleet selects
 * the cheapest Spot pools and evenly allocates your target Spot capacity across
 * the number of Spot pools that you specify.
 * @property launchSpecifications Used to define the launch configuration of the
 * spot-fleet request. Can be specified multiple times to define different bids
 * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
 * **Note**: This takes in similar but not
 * identical inputs as `aws.ec2.Instance`.  There are limitations on
 * what you can specify. See the list of officially supported inputs in the
 * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
 * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
 * @property launchTemplateConfigs Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
 * @property loadBalancers A list of elastic load balancer names to add to the Spot fleet.
 * @property onDemandAllocationStrategy The order of the launch template overrides to use in fulfilling On-Demand capacity. the possible values are: `lowestPrice` and `prioritized`. the default is `lowestPrice`.
 * @property onDemandMaxTotalPrice The maximum amount per hour for On-Demand Instances that you're willing to pay. When the maximum amount you're willing to pay is reached, the fleet stops launching instances even if it hasn’t met the target capacity.
 * @property onDemandTargetCapacity The number of On-Demand units to request. If the request type is `maintain`, you can specify a target capacity of 0 and add capacity later.
 * @property replaceUnhealthyInstances Indicates whether Spot fleet should replace unhealthy instances. Default `false`.
 * @property spotMaintenanceStrategies Nested argument containing maintenance strategies for managing your Spot Instances that are at an elevated risk of being interrupted. Defined below.
 * @property spotPrice The maximum bid price per unit hour.
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property targetCapacity The number of units to request. You can choose to set the
 * target capacity in terms of instances or a performance characteristic that is
 * important to your application workload, such as vCPUs, memory, or I/O.
 * @property targetCapacityUnitType The unit for the target capacity. This can only be done with `instance_requirements` defined
 * @property targetGroupArns A list of `aws.alb.TargetGroup` ARNs, for use with Application Load Balancing.
 * @property terminateInstancesOnDelete Indicates whether running Spot
 * instances should be terminated when the resource is deleted (and the Spot fleet request cancelled).
 * If no value is specified, the value of the `terminate_instances_with_expiration` argument is used.
 * @property terminateInstancesWithExpiration Indicates whether running Spot
 * instances should be terminated when the Spot fleet request expires.
 * @property validFrom The start date and time of the request, in UTC [RFC3339](https://tools.ietf.org/html/rfc3339#section-5.8) format(for example, YYYY-MM-DDTHH:MM:SSZ). The default is to start fulfilling the request immediately.
 * @property validUntil The end date and time of the request, in UTC [RFC3339](https://tools.ietf.org/html/rfc3339#section-5.8) format(for example, YYYY-MM-DDTHH:MM:SSZ). At this point, no new Spot instance requests are placed or enabled to fulfill the request.
 * @property waitForFulfillment If set, this provider will
 * wait for the Spot Request to be fulfilled, and will throw an error if the
 * timeout of 10m is reached.
 */
public data class SpotFleetRequestArgs(
    public val allocationStrategy: Output<String>? = null,
    public val context: Output<String>? = null,
    public val excessCapacityTerminationPolicy: Output<String>? = null,
    public val fleetType: Output<String>? = null,
    public val iamFleetRole: Output<String>? = null,
    public val instanceInterruptionBehaviour: Output<String>? = null,
    public val instancePoolsToUseCount: Output<Int>? = null,
    public val launchSpecifications: Output<List<SpotFleetRequestLaunchSpecificationArgs>>? = null,
    public val launchTemplateConfigs: Output<List<SpotFleetRequestLaunchTemplateConfigArgs>>? = null,
    public val loadBalancers: Output<List<String>>? = null,
    public val onDemandAllocationStrategy: Output<String>? = null,
    public val onDemandMaxTotalPrice: Output<String>? = null,
    public val onDemandTargetCapacity: Output<Int>? = null,
    public val replaceUnhealthyInstances: Output<Boolean>? = null,
    public val spotMaintenanceStrategies: Output<SpotFleetRequestSpotMaintenanceStrategiesArgs>? =
        null,
    public val spotPrice: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val targetCapacity: Output<Int>? = null,
    public val targetCapacityUnitType: Output<String>? = null,
    public val targetGroupArns: Output<List<String>>? = null,
    public val terminateInstancesOnDelete: Output<String>? = null,
    public val terminateInstancesWithExpiration: Output<Boolean>? = null,
    public val validFrom: Output<String>? = null,
    public val validUntil: Output<String>? = null,
    public val waitForFulfillment: Output<Boolean>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.SpotFleetRequestArgs> {
    override fun toJava(): com.pulumi.aws.ec2.SpotFleetRequestArgs =
        com.pulumi.aws.ec2.SpotFleetRequestArgs.builder()
            .allocationStrategy(allocationStrategy?.applyValue({ args0 -> args0 }))
            .context(context?.applyValue({ args0 -> args0 }))
            .excessCapacityTerminationPolicy(excessCapacityTerminationPolicy?.applyValue({ args0 -> args0 }))
            .fleetType(fleetType?.applyValue({ args0 -> args0 }))
            .iamFleetRole(iamFleetRole?.applyValue({ args0 -> args0 }))
            .instanceInterruptionBehaviour(instanceInterruptionBehaviour?.applyValue({ args0 -> args0 }))
            .instancePoolsToUseCount(instancePoolsToUseCount?.applyValue({ args0 -> args0 }))
            .launchSpecifications(
                launchSpecifications?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .launchTemplateConfigs(
                launchTemplateConfigs?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .loadBalancers(loadBalancers?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .onDemandAllocationStrategy(onDemandAllocationStrategy?.applyValue({ args0 -> args0 }))
            .onDemandMaxTotalPrice(onDemandMaxTotalPrice?.applyValue({ args0 -> args0 }))
            .onDemandTargetCapacity(onDemandTargetCapacity?.applyValue({ args0 -> args0 }))
            .replaceUnhealthyInstances(replaceUnhealthyInstances?.applyValue({ args0 -> args0 }))
            .spotMaintenanceStrategies(
                spotMaintenanceStrategies?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .spotPrice(spotPrice?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .targetCapacity(targetCapacity?.applyValue({ args0 -> args0 }))
            .targetCapacityUnitType(targetCapacityUnitType?.applyValue({ args0 -> args0 }))
            .targetGroupArns(targetGroupArns?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .terminateInstancesOnDelete(terminateInstancesOnDelete?.applyValue({ args0 -> args0 }))
            .terminateInstancesWithExpiration(terminateInstancesWithExpiration?.applyValue({ args0 -> args0 }))
            .validFrom(validFrom?.applyValue({ args0 -> args0 }))
            .validUntil(validUntil?.applyValue({ args0 -> args0 }))
            .waitForFulfillment(waitForFulfillment?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [SpotFleetRequestArgs].
 */
@PulumiTagMarker
public class SpotFleetRequestArgsBuilder internal constructor() {
    private var allocationStrategy: Output<String>? = null

    private var context: Output<String>? = null

    private var excessCapacityTerminationPolicy: Output<String>? = null

    private var fleetType: Output<String>? = null

    private var iamFleetRole: Output<String>? = null

    private var instanceInterruptionBehaviour: Output<String>? = null

    private var instancePoolsToUseCount: Output<Int>? = null

    private var launchSpecifications: Output<List<SpotFleetRequestLaunchSpecificationArgs>>? = null

    private var launchTemplateConfigs: Output<List<SpotFleetRequestLaunchTemplateConfigArgs>>? = null

    private var loadBalancers: Output<List<String>>? = null

    private var onDemandAllocationStrategy: Output<String>? = null

    private var onDemandMaxTotalPrice: Output<String>? = null

    private var onDemandTargetCapacity: Output<Int>? = null

    private var replaceUnhealthyInstances: Output<Boolean>? = null

    private var spotMaintenanceStrategies: Output<SpotFleetRequestSpotMaintenanceStrategiesArgs>? =
        null

    private var spotPrice: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var targetCapacity: Output<Int>? = null

    private var targetCapacityUnitType: Output<String>? = null

    private var targetGroupArns: Output<List<String>>? = null

    private var terminateInstancesOnDelete: Output<String>? = null

    private var terminateInstancesWithExpiration: Output<Boolean>? = null

    private var validFrom: Output<String>? = null

    private var validUntil: Output<String>? = null

    private var waitForFulfillment: Output<Boolean>? = null

    /**
     * @param value Indicates how to allocate the target capacity across
     * the Spot pools specified by the Spot fleet request. Valid values: `lowestPrice`, `diversified`, `capacityOptimized`, `capacityOptimizedPrioritized`, and `priceCapacityOptimized`. The default is
     * `lowestPrice`.
     */
    @JvmName("dqqibhpajtgxohsi")
    public suspend fun allocationStrategy(`value`: Output<String>) {
        this.allocationStrategy = value
    }

    /**
     * @param value Reserved.
     */
    @JvmName("mtgqsdmfqxfhidxe")
    public suspend fun context(`value`: Output<String>) {
        this.context = value
    }

    /**
     * @param value Indicates whether running Spot
     * instances should be terminated if the target capacity of the Spot fleet
     * request is decreased below the current size of the Spot fleet.
     */
    @JvmName("fgpipkpnrafdtamg")
    public suspend fun excessCapacityTerminationPolicy(`value`: Output<String>) {
        this.excessCapacityTerminationPolicy = value
    }

    /**
     * @param value The type of fleet request. Indicates whether the Spot Fleet only requests the target
     * capacity or also attempts to maintain it. Default is `maintain`.
     */
    @JvmName("dgwkyoxpkprsgfti")
    public suspend fun fleetType(`value`: Output<String>) {
        this.fleetType = value
    }

    /**
     * @param value Grants the Spot fleet permission to terminate
     * Spot instances on your behalf when you cancel its Spot fleet request using
     * CancelSpotFleetRequests or when the Spot fleet request expires, if you set
     * terminateInstancesWithExpiration.
     */
    @JvmName("obrvbswwxxbojlwo")
    public suspend fun iamFleetRole(`value`: Output<String>) {
        this.iamFleetRole = value
    }

    /**
     * @param value Indicates whether a Spot
     * instance stops or terminates when it is interrupted. Default is
     * `terminate`.
     */
    @JvmName("hglwphrkqugeriuq")
    public suspend fun instanceInterruptionBehaviour(`value`: Output<String>) {
        this.instanceInterruptionBehaviour = value
    }

    /**
     * @param value The number of Spot pools across which to allocate your target Spot capacity.
     * Valid only when `allocation_strategy` is set to `lowestPrice`. Spot Fleet selects
     * the cheapest Spot pools and evenly allocates your target Spot capacity across
     * the number of Spot pools that you specify.
     */
    @JvmName("absngpglcuhidfuf")
    public suspend fun instancePoolsToUseCount(`value`: Output<Int>) {
        this.instancePoolsToUseCount = value
    }

    /**
     * @param value Used to define the launch configuration of the
     * spot-fleet request. Can be specified multiple times to define different bids
     * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
     * **Note**: This takes in similar but not
     * identical inputs as `aws.ec2.Instance`.  There are limitations on
     * what you can specify. See the list of officially supported inputs in the
     * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
     * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
     */
    @JvmName("yeoauqptskcrbmkh")
    public suspend fun launchSpecifications(`value`: Output<List<SpotFleetRequestLaunchSpecificationArgs>>) {
        this.launchSpecifications = value
    }

    @JvmName("wuopvxuliliqfepu")
    public suspend fun launchSpecifications(vararg values: Output<SpotFleetRequestLaunchSpecificationArgs>) {
        this.launchSpecifications = Output.all(values.asList())
    }

    /**
     * @param values Used to define the launch configuration of the
     * spot-fleet request. Can be specified multiple times to define different bids
     * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
     * **Note**: This takes in similar but not
     * identical inputs as `aws.ec2.Instance`.  There are limitations on
     * what you can specify. See the list of officially supported inputs in the
     * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
     * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
     */
    @JvmName("wawptlbapahedpep")
    public suspend fun launchSpecifications(values: List<Output<SpotFleetRequestLaunchSpecificationArgs>>) {
        this.launchSpecifications = Output.all(values)
    }

    /**
     * @param value Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
     */
    @JvmName("hwkhbspkadrxbvkf")
    public suspend fun launchTemplateConfigs(`value`: Output<List<SpotFleetRequestLaunchTemplateConfigArgs>>) {
        this.launchTemplateConfigs = value
    }

    @JvmName("icpnddiogqmtxlqn")
    public suspend fun launchTemplateConfigs(vararg values: Output<SpotFleetRequestLaunchTemplateConfigArgs>) {
        this.launchTemplateConfigs = Output.all(values.asList())
    }

    /**
     * @param values Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
     */
    @JvmName("rnuocpnblknxrttw")
    public suspend fun launchTemplateConfigs(values: List<Output<SpotFleetRequestLaunchTemplateConfigArgs>>) {
        this.launchTemplateConfigs = Output.all(values)
    }

    /**
     * @param value A list of elastic load balancer names to add to the Spot fleet.
     */
    @JvmName("icspwlvvarjkaepy")
    public suspend fun loadBalancers(`value`: Output<List<String>>) {
        this.loadBalancers = value
    }

    @JvmName("vnntfgxndtuwsirc")
    public suspend fun loadBalancers(vararg values: Output<String>) {
        this.loadBalancers = Output.all(values.asList())
    }

    /**
     * @param values A list of elastic load balancer names to add to the Spot fleet.
     */
    @JvmName("rvhlsgvaqrvvqkus")
    public suspend fun loadBalancers(values: List<Output<String>>) {
        this.loadBalancers = Output.all(values)
    }

    /**
     * @param value The order of the launch template overrides to use in fulfilling On-Demand capacity. the possible values are: `lowestPrice` and `prioritized`. the default is `lowestPrice`.
     */
    @JvmName("xxehsajcceoluiqp")
    public suspend fun onDemandAllocationStrategy(`value`: Output<String>) {
        this.onDemandAllocationStrategy = value
    }

    /**
     * @param value The maximum amount per hour for On-Demand Instances that you're willing to pay. When the maximum amount you're willing to pay is reached, the fleet stops launching instances even if it hasn’t met the target capacity.
     */
    @JvmName("nkvbbwdnjwpokyin")
    public suspend fun onDemandMaxTotalPrice(`value`: Output<String>) {
        this.onDemandMaxTotalPrice = value
    }

    /**
     * @param value The number of On-Demand units to request. If the request type is `maintain`, you can specify a target capacity of 0 and add capacity later.
     */
    @JvmName("ynqxepvsgmiluwnk")
    public suspend fun onDemandTargetCapacity(`value`: Output<Int>) {
        this.onDemandTargetCapacity = value
    }

    /**
     * @param value Indicates whether Spot fleet should replace unhealthy instances. Default `false`.
     */
    @JvmName("unvclkuytchgixml")
    public suspend fun replaceUnhealthyInstances(`value`: Output<Boolean>) {
        this.replaceUnhealthyInstances = value
    }

    /**
     * @param value Nested argument containing maintenance strategies for managing your Spot Instances that are at an elevated risk of being interrupted. Defined below.
     */
    @JvmName("rwxmhlbbvxaawbrh")
    public suspend fun spotMaintenanceStrategies(`value`: Output<SpotFleetRequestSpotMaintenanceStrategiesArgs>) {
        this.spotMaintenanceStrategies = value
    }

    /**
     * @param value The maximum bid price per unit hour.
     */
    @JvmName("emdsokeaxpufwnoj")
    public suspend fun spotPrice(`value`: Output<String>) {
        this.spotPrice = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vqjijxuwwtqvjnyw")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The number of units to request. You can choose to set the
     * target capacity in terms of instances or a performance characteristic that is
     * important to your application workload, such as vCPUs, memory, or I/O.
     */
    @JvmName("fahxxnbispcsjlyj")
    public suspend fun targetCapacity(`value`: Output<Int>) {
        this.targetCapacity = value
    }

    /**
     * @param value The unit for the target capacity. This can only be done with `instance_requirements` defined
     */
    @JvmName("avkiqrmmnknhfowr")
    public suspend fun targetCapacityUnitType(`value`: Output<String>) {
        this.targetCapacityUnitType = value
    }

    /**
     * @param value A list of `aws.alb.TargetGroup` ARNs, for use with Application Load Balancing.
     */
    @JvmName("lphomtnoaogebssy")
    public suspend fun targetGroupArns(`value`: Output<List<String>>) {
        this.targetGroupArns = value
    }

    @JvmName("odprniywjjpcwucc")
    public suspend fun targetGroupArns(vararg values: Output<String>) {
        this.targetGroupArns = Output.all(values.asList())
    }

    /**
     * @param values A list of `aws.alb.TargetGroup` ARNs, for use with Application Load Balancing.
     */
    @JvmName("bdifyajfwkohmkml")
    public suspend fun targetGroupArns(values: List<Output<String>>) {
        this.targetGroupArns = Output.all(values)
    }

    /**
     * @param value Indicates whether running Spot
     * instances should be terminated when the resource is deleted (and the Spot fleet request cancelled).
     * If no value is specified, the value of the `terminate_instances_with_expiration` argument is used.
     */
    @JvmName("crpsgkcetyubnwdx")
    public suspend fun terminateInstancesOnDelete(`value`: Output<String>) {
        this.terminateInstancesOnDelete = value
    }

    /**
     * @param value Indicates whether running Spot
     * instances should be terminated when the Spot fleet request expires.
     */
    @JvmName("saniarivvfwdyjil")
    public suspend fun terminateInstancesWithExpiration(`value`: Output<Boolean>) {
        this.terminateInstancesWithExpiration = value
    }

    /**
     * @param value The start date and time of the request, in UTC [RFC3339](https://tools.ietf.org/html/rfc3339#section-5.8) format(for example, YYYY-MM-DDTHH:MM:SSZ). The default is to start fulfilling the request immediately.
     */
    @JvmName("pslnhlopfqkkfwxe")
    public suspend fun validFrom(`value`: Output<String>) {
        this.validFrom = value
    }

    /**
     * @param value The end date and time of the request, in UTC [RFC3339](https://tools.ietf.org/html/rfc3339#section-5.8) format(for example, YYYY-MM-DDTHH:MM:SSZ). At this point, no new Spot instance requests are placed or enabled to fulfill the request.
     */
    @JvmName("swpmeillthorubkh")
    public suspend fun validUntil(`value`: Output<String>) {
        this.validUntil = value
    }

    /**
     * @param value If set, this provider will
     * wait for the Spot Request to be fulfilled, and will throw an error if the
     * timeout of 10m is reached.
     */
    @JvmName("jgrpgyxwwakecjhp")
    public suspend fun waitForFulfillment(`value`: Output<Boolean>) {
        this.waitForFulfillment = value
    }

    /**
     * @param value Indicates how to allocate the target capacity across
     * the Spot pools specified by the Spot fleet request. Valid values: `lowestPrice`, `diversified`, `capacityOptimized`, `capacityOptimizedPrioritized`, and `priceCapacityOptimized`. The default is
     * `lowestPrice`.
     */
    @JvmName("dofrraofnptrdesl")
    public suspend fun allocationStrategy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocationStrategy = mapped
    }

    /**
     * @param value Reserved.
     */
    @JvmName("hdixhqmamvfyiqru")
    public suspend fun context(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.context = mapped
    }

    /**
     * @param value Indicates whether running Spot
     * instances should be terminated if the target capacity of the Spot fleet
     * request is decreased below the current size of the Spot fleet.
     */
    @JvmName("hjtgfgowchdtokur")
    public suspend fun excessCapacityTerminationPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.excessCapacityTerminationPolicy = mapped
    }

    /**
     * @param value The type of fleet request. Indicates whether the Spot Fleet only requests the target
     * capacity or also attempts to maintain it. Default is `maintain`.
     */
    @JvmName("fgojsoqpsuqxqyex")
    public suspend fun fleetType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.fleetType = mapped
    }

    /**
     * @param value Grants the Spot fleet permission to terminate
     * Spot instances on your behalf when you cancel its Spot fleet request using
     * CancelSpotFleetRequests or when the Spot fleet request expires, if you set
     * terminateInstancesWithExpiration.
     */
    @JvmName("rcrxmkkxjknmcpyq")
    public suspend fun iamFleetRole(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamFleetRole = mapped
    }

    /**
     * @param value Indicates whether a Spot
     * instance stops or terminates when it is interrupted. Default is
     * `terminate`.
     */
    @JvmName("nbvkytgmrkjwnwdi")
    public suspend fun instanceInterruptionBehaviour(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceInterruptionBehaviour = mapped
    }

    /**
     * @param value The number of Spot pools across which to allocate your target Spot capacity.
     * Valid only when `allocation_strategy` is set to `lowestPrice`. Spot Fleet selects
     * the cheapest Spot pools and evenly allocates your target Spot capacity across
     * the number of Spot pools that you specify.
     */
    @JvmName("myuriqnntnbcgdwa")
    public suspend fun instancePoolsToUseCount(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instancePoolsToUseCount = mapped
    }

    /**
     * @param value Used to define the launch configuration of the
     * spot-fleet request. Can be specified multiple times to define different bids
     * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
     * **Note**: This takes in similar but not
     * identical inputs as `aws.ec2.Instance`.  There are limitations on
     * what you can specify. See the list of officially supported inputs in the
     * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
     * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
     */
    @JvmName("yimmqqefvfbmkgat")
    public suspend fun launchSpecifications(`value`: List<SpotFleetRequestLaunchSpecificationArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.launchSpecifications = mapped
    }

    /**
     * @param argument Used to define the launch configuration of the
     * spot-fleet request. Can be specified multiple times to define different bids
     * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
     * **Note**: This takes in similar but not
     * identical inputs as `aws.ec2.Instance`.  There are limitations on
     * what you can specify. See the list of officially supported inputs in the
     * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
     * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
     */
    @JvmName("xubmohxeyjwownda")
    public suspend fun launchSpecifications(argument: List<suspend SpotFleetRequestLaunchSpecificationArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            SpotFleetRequestLaunchSpecificationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.launchSpecifications = mapped
    }

    /**
     * @param argument Used to define the launch configuration of the
     * spot-fleet request. Can be specified multiple times to define different bids
     * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
     * **Note**: This takes in similar but not
     * identical inputs as `aws.ec2.Instance`.  There are limitations on
     * what you can specify. See the list of officially supported inputs in the
     * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
     * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
     */
    @JvmName("ieowxiflhahyyvtd")
    public suspend fun launchSpecifications(vararg argument: suspend SpotFleetRequestLaunchSpecificationArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            SpotFleetRequestLaunchSpecificationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.launchSpecifications = mapped
    }

    /**
     * @param argument Used to define the launch configuration of the
     * spot-fleet request. Can be specified multiple times to define different bids
     * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
     * **Note**: This takes in similar but not
     * identical inputs as `aws.ec2.Instance`.  There are limitations on
     * what you can specify. See the list of officially supported inputs in the
     * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
     * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
     */
    @JvmName("akrljbbjdrfwbpqs")
    public suspend fun launchSpecifications(argument: suspend SpotFleetRequestLaunchSpecificationArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            SpotFleetRequestLaunchSpecificationArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.launchSpecifications = mapped
    }

    /**
     * @param values Used to define the launch configuration of the
     * spot-fleet request. Can be specified multiple times to define different bids
     * across different markets and instance types. Conflicts with `launch_template_config`. At least one of `launch_specification` or `launch_template_config` is required.
     * **Note**: This takes in similar but not
     * identical inputs as `aws.ec2.Instance`.  There are limitations on
     * what you can specify. See the list of officially supported inputs in the
     * [reference documentation](http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SpotFleetLaunchSpecification.html). Any normal `aws.ec2.Instance` parameter that corresponds to those inputs may be used and it have
     * a additional parameter `iam_instance_profile_arn` takes `aws.iam.InstanceProfile` attribute `arn` as input.
     */
    @JvmName("biseptvbmoyjeijs")
    public suspend fun launchSpecifications(vararg values: SpotFleetRequestLaunchSpecificationArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.launchSpecifications = mapped
    }

    /**
     * @param value Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
     */
    @JvmName("wkdblwwldnuklvsg")
    public suspend fun launchTemplateConfigs(`value`: List<SpotFleetRequestLaunchTemplateConfigArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.launchTemplateConfigs = mapped
    }

    /**
     * @param argument Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
     */
    @JvmName("arxbcpncddegqkhe")
    public suspend fun launchTemplateConfigs(argument: List<suspend SpotFleetRequestLaunchTemplateConfigArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            SpotFleetRequestLaunchTemplateConfigArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.launchTemplateConfigs = mapped
    }

    /**
     * @param argument Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
     */
    @JvmName("kxienimmqhawxbdj")
    public suspend fun launchTemplateConfigs(vararg argument: suspend SpotFleetRequestLaunchTemplateConfigArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            SpotFleetRequestLaunchTemplateConfigArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.launchTemplateConfigs = mapped
    }

    /**
     * @param argument Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
     */
    @JvmName("wmmgvgfgmhtyvqgt")
    public suspend fun launchTemplateConfigs(argument: suspend SpotFleetRequestLaunchTemplateConfigArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            SpotFleetRequestLaunchTemplateConfigArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.launchTemplateConfigs = mapped
    }

    /**
     * @param values Launch template configuration block. See Launch Template Configs below for more details. Conflicts with `launch_specification`. At least one of `launch_specification` or `launch_template_config` is required.
     */
    @JvmName("syadgsrjyxwlkuku")
    public suspend fun launchTemplateConfigs(vararg values: SpotFleetRequestLaunchTemplateConfigArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.launchTemplateConfigs = mapped
    }

    /**
     * @param value A list of elastic load balancer names to add to the Spot fleet.
     */
    @JvmName("qptlugdxdkmksifb")
    public suspend fun loadBalancers(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.loadBalancers = mapped
    }

    /**
     * @param values A list of elastic load balancer names to add to the Spot fleet.
     */
    @JvmName("bsohgmhpqcmbdkui")
    public suspend fun loadBalancers(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.loadBalancers = mapped
    }

    /**
     * @param value The order of the launch template overrides to use in fulfilling On-Demand capacity. the possible values are: `lowestPrice` and `prioritized`. the default is `lowestPrice`.
     */
    @JvmName("hlseabsyrbvugaso")
    public suspend fun onDemandAllocationStrategy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.onDemandAllocationStrategy = mapped
    }

    /**
     * @param value The maximum amount per hour for On-Demand Instances that you're willing to pay. When the maximum amount you're willing to pay is reached, the fleet stops launching instances even if it hasn’t met the target capacity.
     */
    @JvmName("kttcgomnowkaileq")
    public suspend fun onDemandMaxTotalPrice(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.onDemandMaxTotalPrice = mapped
    }

    /**
     * @param value The number of On-Demand units to request. If the request type is `maintain`, you can specify a target capacity of 0 and add capacity later.
     */
    @JvmName("umbnihkfhjyjxpoi")
    public suspend fun onDemandTargetCapacity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.onDemandTargetCapacity = mapped
    }

    /**
     * @param value Indicates whether Spot fleet should replace unhealthy instances. Default `false`.
     */
    @JvmName("dntxvqseanjcnpnj")
    public suspend fun replaceUnhealthyInstances(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replaceUnhealthyInstances = mapped
    }

    /**
     * @param value Nested argument containing maintenance strategies for managing your Spot Instances that are at an elevated risk of being interrupted. Defined below.
     */
    @JvmName("omyagjptadtegoyg")
    public suspend fun spotMaintenanceStrategies(`value`: SpotFleetRequestSpotMaintenanceStrategiesArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.spotMaintenanceStrategies = mapped
    }

    /**
     * @param argument Nested argument containing maintenance strategies for managing your Spot Instances that are at an elevated risk of being interrupted. Defined below.
     */
    @JvmName("uqendvhnmkdunbjs")
    public suspend fun spotMaintenanceStrategies(argument: suspend SpotFleetRequestSpotMaintenanceStrategiesArgsBuilder.() -> Unit) {
        val toBeMapped = SpotFleetRequestSpotMaintenanceStrategiesArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.spotMaintenanceStrategies = mapped
    }

    /**
     * @param value The maximum bid price per unit hour.
     */
    @JvmName("oauxdhkdbbrpsxhs")
    public suspend fun spotPrice(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.spotPrice = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("paodkfrlesoehiam")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("wtgrfbvhcjhocoev")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The number of units to request. You can choose to set the
     * target capacity in terms of instances or a performance characteristic that is
     * important to your application workload, such as vCPUs, memory, or I/O.
     */
    @JvmName("ilpvuseipajwsdua")
    public suspend fun targetCapacity(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetCapacity = mapped
    }

    /**
     * @param value The unit for the target capacity. This can only be done with `instance_requirements` defined
     */
    @JvmName("jshpdlhcubpxuabi")
    public suspend fun targetCapacityUnitType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetCapacityUnitType = mapped
    }

    /**
     * @param value A list of `aws.alb.TargetGroup` ARNs, for use with Application Load Balancing.
     */
    @JvmName("qkqihytgqibqjjgv")
    public suspend fun targetGroupArns(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetGroupArns = mapped
    }

    /**
     * @param values A list of `aws.alb.TargetGroup` ARNs, for use with Application Load Balancing.
     */
    @JvmName("htjgxgcgvlqkustm")
    public suspend fun targetGroupArns(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.targetGroupArns = mapped
    }

    /**
     * @param value Indicates whether running Spot
     * instances should be terminated when the resource is deleted (and the Spot fleet request cancelled).
     * If no value is specified, the value of the `terminate_instances_with_expiration` argument is used.
     */
    @JvmName("pyxnarcnufnoneba")
    public suspend fun terminateInstancesOnDelete(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.terminateInstancesOnDelete = mapped
    }

    /**
     * @param value Indicates whether running Spot
     * instances should be terminated when the Spot fleet request expires.
     */
    @JvmName("gsrbepnuljivtoeh")
    public suspend fun terminateInstancesWithExpiration(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.terminateInstancesWithExpiration = mapped
    }

    /**
     * @param value The start date and time of the request, in UTC [RFC3339](https://tools.ietf.org/html/rfc3339#section-5.8) format(for example, YYYY-MM-DDTHH:MM:SSZ). The default is to start fulfilling the request immediately.
     */
    @JvmName("hpnlwatwujotdllv")
    public suspend fun validFrom(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.validFrom = mapped
    }

    /**
     * @param value The end date and time of the request, in UTC [RFC3339](https://tools.ietf.org/html/rfc3339#section-5.8) format(for example, YYYY-MM-DDTHH:MM:SSZ). At this point, no new Spot instance requests are placed or enabled to fulfill the request.
     */
    @JvmName("cdrnpggltxhpctsj")
    public suspend fun validUntil(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.validUntil = mapped
    }

    /**
     * @param value If set, this provider will
     * wait for the Spot Request to be fulfilled, and will throw an error if the
     * timeout of 10m is reached.
     */
    @JvmName("wxnikvoqtonkrfdw")
    public suspend fun waitForFulfillment(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.waitForFulfillment = mapped
    }

    internal fun build(): SpotFleetRequestArgs = SpotFleetRequestArgs(
        allocationStrategy = allocationStrategy,
        context = context,
        excessCapacityTerminationPolicy = excessCapacityTerminationPolicy,
        fleetType = fleetType,
        iamFleetRole = iamFleetRole,
        instanceInterruptionBehaviour = instanceInterruptionBehaviour,
        instancePoolsToUseCount = instancePoolsToUseCount,
        launchSpecifications = launchSpecifications,
        launchTemplateConfigs = launchTemplateConfigs,
        loadBalancers = loadBalancers,
        onDemandAllocationStrategy = onDemandAllocationStrategy,
        onDemandMaxTotalPrice = onDemandMaxTotalPrice,
        onDemandTargetCapacity = onDemandTargetCapacity,
        replaceUnhealthyInstances = replaceUnhealthyInstances,
        spotMaintenanceStrategies = spotMaintenanceStrategies,
        spotPrice = spotPrice,
        tags = tags,
        targetCapacity = targetCapacity,
        targetCapacityUnitType = targetCapacityUnitType,
        targetGroupArns = targetGroupArns,
        terminateInstancesOnDelete = terminateInstancesOnDelete,
        terminateInstancesWithExpiration = terminateInstancesWithExpiration,
        validFrom = validFrom,
        validUntil = validUntil,
        waitForFulfillment = waitForFulfillment,
    )
}
