@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.SubnetArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an VPC subnet resource.
 * > **NOTE:** Due to [AWS Lambda improved VPC networking changes that began deploying in September 2019](https://aws.amazon.com/blogs/compute/announcing-improved-vpc-networking-for-aws-lambda-functions/), subnets associated with Lambda Functions can take up to 45 minutes to successfully delete.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const main = new aws.ec2.Subnet("main", {
 *     vpcId: mainAwsVpc.id,
 *     cidrBlock: "10.0.1.0/24",
 *     tags: {
 *         Name: "Main",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * main = aws.ec2.Subnet("main",
 *     vpc_id=main_aws_vpc["id"],
 *     cidr_block="10.0.1.0/24",
 *     tags={
 *         "Name": "Main",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var main = new Aws.Ec2.Subnet("main", new()
 *     {
 *         VpcId = mainAwsVpc.Id,
 *         CidrBlock = "10.0.1.0/24",
 *         Tags =
 *         {
 *             { "Name", "Main" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewSubnet(ctx, "main", &ec2.SubnetArgs{
 * 			VpcId:     pulumi.Any(mainAwsVpc.Id),
 * 			CidrBlock: pulumi.String("10.0.1.0/24"),
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("Main"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Subnet;
 * import com.pulumi.aws.ec2.SubnetArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var main = new Subnet("main", SubnetArgs.builder()
 *             .vpcId(mainAwsVpc.id())
 *             .cidrBlock("10.0.1.0/24")
 *             .tags(Map.of("Name", "Main"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   main:
 *     type: aws:ec2:Subnet
 *     properties:
 *       vpcId: ${mainAwsVpc.id}
 *       cidrBlock: 10.0.1.0/24
 *       tags:
 *         Name: Main
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Subnets In Secondary VPC CIDR Blocks
 * When managing subnets in one of a VPC's secondary CIDR blocks created using a `aws.ec2.VpcIpv4CidrBlockAssociation`
 * resource, it is recommended to reference that resource's `vpc_id` attribute to ensure correct dependency ordering.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const secondaryCidr = new aws.ec2.VpcIpv4CidrBlockAssociation("secondary_cidr", {
 *     vpcId: main.id,
 *     cidrBlock: "172.20.0.0/16",
 * });
 * const inSecondaryCidr = new aws.ec2.Subnet("in_secondary_cidr", {
 *     vpcId: secondaryCidr.vpcId,
 *     cidrBlock: "172.20.0.0/24",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * secondary_cidr = aws.ec2.VpcIpv4CidrBlockAssociation("secondary_cidr",
 *     vpc_id=main["id"],
 *     cidr_block="172.20.0.0/16")
 * in_secondary_cidr = aws.ec2.Subnet("in_secondary_cidr",
 *     vpc_id=secondary_cidr.vpc_id,
 *     cidr_block="172.20.0.0/24")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var secondaryCidr = new Aws.Ec2.VpcIpv4CidrBlockAssociation("secondary_cidr", new()
 *     {
 *         VpcId = main.Id,
 *         CidrBlock = "172.20.0.0/16",
 *     });
 *     var inSecondaryCidr = new Aws.Ec2.Subnet("in_secondary_cidr", new()
 *     {
 *         VpcId = secondaryCidr.VpcId,
 *         CidrBlock = "172.20.0.0/24",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		secondaryCidr, err := ec2.NewVpcIpv4CidrBlockAssociation(ctx, "secondary_cidr", &ec2.VpcIpv4CidrBlockAssociationArgs{
 * 			VpcId:     pulumi.Any(main.Id),
 * 			CidrBlock: pulumi.String("172.20.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewSubnet(ctx, "in_secondary_cidr", &ec2.SubnetArgs{
 * 			VpcId:     secondaryCidr.VpcId,
 * 			CidrBlock: pulumi.String("172.20.0.0/24"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcIpv4CidrBlockAssociation;
 * import com.pulumi.aws.ec2.VpcIpv4CidrBlockAssociationArgs;
 * import com.pulumi.aws.ec2.Subnet;
 * import com.pulumi.aws.ec2.SubnetArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var secondaryCidr = new VpcIpv4CidrBlockAssociation("secondaryCidr", VpcIpv4CidrBlockAssociationArgs.builder()
 *             .vpcId(main.id())
 *             .cidrBlock("172.20.0.0/16")
 *             .build());
 *         var inSecondaryCidr = new Subnet("inSecondaryCidr", SubnetArgs.builder()
 *             .vpcId(secondaryCidr.vpcId())
 *             .cidrBlock("172.20.0.0/24")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   secondaryCidr:
 *     type: aws:ec2:VpcIpv4CidrBlockAssociation
 *     name: secondary_cidr
 *     properties:
 *       vpcId: ${main.id}
 *       cidrBlock: 172.20.0.0/16
 *   inSecondaryCidr:
 *     type: aws:ec2:Subnet
 *     name: in_secondary_cidr
 *     properties:
 *       vpcId: ${secondaryCidr.vpcId}
 *       cidrBlock: 172.20.0.0/24
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import subnets using the subnet `id`. For example:
 * ```sh
 * $ pulumi import aws:ec2/subnet:Subnet public_subnet subnet-9d4a7b6c
 * ```
 * @property assignIpv6AddressOnCreation Specify true to indicate
 * that network interfaces created in the specified subnet should be
 * assigned an IPv6 address. Default is `false`
 * @property availabilityZone AZ for the subnet.
 * @property availabilityZoneId AZ ID of the subnet. This argument is not supported in all regions or partitions. If necessary, use `availability_zone` instead.
 * @property cidrBlock The IPv4 CIDR block for the subnet.
 * @property customerOwnedIpv4Pool The customer owned IPv4 address pool. Typically used with the `map_customer_owned_ip_on_launch` argument. The `outpost_arn` argument must be specified when configured.
 * @property enableDns64 Indicates whether DNS queries made to the Amazon-provided DNS Resolver in this subnet should return synthetic IPv6 addresses for IPv4-only destinations. Default: `false`.
 * @property enableLniAtDeviceIndex Indicates the device position for local network interfaces in this subnet. For example, 1 indicates local network interfaces in this subnet are the secondary network interface (eth1). A local network interface cannot be the primary network interface (eth0).
 * @property enableResourceNameDnsARecordOnLaunch Indicates whether to respond to DNS queries for instance hostnames with DNS A records. Default: `false`.
 * @property enableResourceNameDnsAaaaRecordOnLaunch Indicates whether to respond to DNS queries for instance hostnames with DNS AAAA records. Default: `false`.
 * @property ipv6CidrBlock The IPv6 network range for the subnet,
 * in CIDR notation. The subnet size must use a /64 prefix length.
 * @property ipv6Native Indicates whether to create an IPv6-only subnet. Default: `false`.
 * @property mapCustomerOwnedIpOnLaunch Specify `true` to indicate that network interfaces created in the subnet should be assigned a customer owned IP address. The `customer_owned_ipv4_pool` and `outpost_arn` arguments must be specified when set to `true`. Default is `false`.
 * @property mapPublicIpOnLaunch Specify true to indicate
 * that instances launched into the subnet should be assigned
 * a public IP address. Default is `false`.
 * @property outpostArn The Amazon Resource Name (ARN) of the Outpost.
 * @property privateDnsHostnameTypeOnLaunch The type of hostnames to assign to instances in the subnet at launch. For IPv6-only subnets, an instance DNS name must be based on the instance ID. For dual-stack and IPv4-only subnets, you can specify whether DNS names use the instance IPv4 address or the instance ID. Valid values: `ip-name`, `resource-name`.
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcId The VPC ID.
 */
public data class SubnetArgs(
    public val assignIpv6AddressOnCreation: Output<Boolean>? = null,
    public val availabilityZone: Output<String>? = null,
    public val availabilityZoneId: Output<String>? = null,
    public val cidrBlock: Output<String>? = null,
    public val customerOwnedIpv4Pool: Output<String>? = null,
    public val enableDns64: Output<Boolean>? = null,
    public val enableLniAtDeviceIndex: Output<Int>? = null,
    public val enableResourceNameDnsARecordOnLaunch: Output<Boolean>? = null,
    public val enableResourceNameDnsAaaaRecordOnLaunch: Output<Boolean>? = null,
    public val ipv6CidrBlock: Output<String>? = null,
    public val ipv6Native: Output<Boolean>? = null,
    public val mapCustomerOwnedIpOnLaunch: Output<Boolean>? = null,
    public val mapPublicIpOnLaunch: Output<Boolean>? = null,
    public val outpostArn: Output<String>? = null,
    public val privateDnsHostnameTypeOnLaunch: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.SubnetArgs> {
    override fun toJava(): com.pulumi.aws.ec2.SubnetArgs = com.pulumi.aws.ec2.SubnetArgs.builder()
        .assignIpv6AddressOnCreation(assignIpv6AddressOnCreation?.applyValue({ args0 -> args0 }))
        .availabilityZone(availabilityZone?.applyValue({ args0 -> args0 }))
        .availabilityZoneId(availabilityZoneId?.applyValue({ args0 -> args0 }))
        .cidrBlock(cidrBlock?.applyValue({ args0 -> args0 }))
        .customerOwnedIpv4Pool(customerOwnedIpv4Pool?.applyValue({ args0 -> args0 }))
        .enableDns64(enableDns64?.applyValue({ args0 -> args0 }))
        .enableLniAtDeviceIndex(enableLniAtDeviceIndex?.applyValue({ args0 -> args0 }))
        .enableResourceNameDnsARecordOnLaunch(
            enableResourceNameDnsARecordOnLaunch?.applyValue({ args0 ->
                args0
            }),
        )
        .enableResourceNameDnsAaaaRecordOnLaunch(
            enableResourceNameDnsAaaaRecordOnLaunch?.applyValue({ args0 ->
                args0
            }),
        )
        .ipv6CidrBlock(ipv6CidrBlock?.applyValue({ args0 -> args0 }))
        .ipv6Native(ipv6Native?.applyValue({ args0 -> args0 }))
        .mapCustomerOwnedIpOnLaunch(mapCustomerOwnedIpOnLaunch?.applyValue({ args0 -> args0 }))
        .mapPublicIpOnLaunch(mapPublicIpOnLaunch?.applyValue({ args0 -> args0 }))
        .outpostArn(outpostArn?.applyValue({ args0 -> args0 }))
        .privateDnsHostnameTypeOnLaunch(privateDnsHostnameTypeOnLaunch?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .vpcId(vpcId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [SubnetArgs].
 */
@PulumiTagMarker
public class SubnetArgsBuilder internal constructor() {
    private var assignIpv6AddressOnCreation: Output<Boolean>? = null

    private var availabilityZone: Output<String>? = null

    private var availabilityZoneId: Output<String>? = null

    private var cidrBlock: Output<String>? = null

    private var customerOwnedIpv4Pool: Output<String>? = null

    private var enableDns64: Output<Boolean>? = null

    private var enableLniAtDeviceIndex: Output<Int>? = null

    private var enableResourceNameDnsARecordOnLaunch: Output<Boolean>? = null

    private var enableResourceNameDnsAaaaRecordOnLaunch: Output<Boolean>? = null

    private var ipv6CidrBlock: Output<String>? = null

    private var ipv6Native: Output<Boolean>? = null

    private var mapCustomerOwnedIpOnLaunch: Output<Boolean>? = null

    private var mapPublicIpOnLaunch: Output<Boolean>? = null

    private var outpostArn: Output<String>? = null

    private var privateDnsHostnameTypeOnLaunch: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcId: Output<String>? = null

    /**
     * @param value Specify true to indicate
     * that network interfaces created in the specified subnet should be
     * assigned an IPv6 address. Default is `false`
     */
    @JvmName("lxjagvtqlbkfxast")
    public suspend fun assignIpv6AddressOnCreation(`value`: Output<Boolean>) {
        this.assignIpv6AddressOnCreation = value
    }

    /**
     * @param value AZ for the subnet.
     */
    @JvmName("benihdiraqvnhalh")
    public suspend fun availabilityZone(`value`: Output<String>) {
        this.availabilityZone = value
    }

    /**
     * @param value AZ ID of the subnet. This argument is not supported in all regions or partitions. If necessary, use `availability_zone` instead.
     */
    @JvmName("kdijdkmbomkriiwp")
    public suspend fun availabilityZoneId(`value`: Output<String>) {
        this.availabilityZoneId = value
    }

    /**
     * @param value The IPv4 CIDR block for the subnet.
     */
    @JvmName("qplcktelduqatolf")
    public suspend fun cidrBlock(`value`: Output<String>) {
        this.cidrBlock = value
    }

    /**
     * @param value The customer owned IPv4 address pool. Typically used with the `map_customer_owned_ip_on_launch` argument. The `outpost_arn` argument must be specified when configured.
     */
    @JvmName("pyhvydjhfcmtsfby")
    public suspend fun customerOwnedIpv4Pool(`value`: Output<String>) {
        this.customerOwnedIpv4Pool = value
    }

    /**
     * @param value Indicates whether DNS queries made to the Amazon-provided DNS Resolver in this subnet should return synthetic IPv6 addresses for IPv4-only destinations. Default: `false`.
     */
    @JvmName("jexmxpqjjdunpvkt")
    public suspend fun enableDns64(`value`: Output<Boolean>) {
        this.enableDns64 = value
    }

    /**
     * @param value Indicates the device position for local network interfaces in this subnet. For example, 1 indicates local network interfaces in this subnet are the secondary network interface (eth1). A local network interface cannot be the primary network interface (eth0).
     */
    @JvmName("apomyujhcnkubgpa")
    public suspend fun enableLniAtDeviceIndex(`value`: Output<Int>) {
        this.enableLniAtDeviceIndex = value
    }

    /**
     * @param value Indicates whether to respond to DNS queries for instance hostnames with DNS A records. Default: `false`.
     */
    @JvmName("enmmehdplqpqjrjl")
    public suspend fun enableResourceNameDnsARecordOnLaunch(`value`: Output<Boolean>) {
        this.enableResourceNameDnsARecordOnLaunch = value
    }

    /**
     * @param value Indicates whether to respond to DNS queries for instance hostnames with DNS AAAA records. Default: `false`.
     */
    @JvmName("uvyrvobawwqlrern")
    public suspend fun enableResourceNameDnsAaaaRecordOnLaunch(`value`: Output<Boolean>) {
        this.enableResourceNameDnsAaaaRecordOnLaunch = value
    }

    /**
     * @param value The IPv6 network range for the subnet,
     * in CIDR notation. The subnet size must use a /64 prefix length.
     */
    @JvmName("cebdlbglswgpsset")
    public suspend fun ipv6CidrBlock(`value`: Output<String>) {
        this.ipv6CidrBlock = value
    }

    /**
     * @param value Indicates whether to create an IPv6-only subnet. Default: `false`.
     */
    @JvmName("ipygraprfmysqiim")
    public suspend fun ipv6Native(`value`: Output<Boolean>) {
        this.ipv6Native = value
    }

    /**
     * @param value Specify `true` to indicate that network interfaces created in the subnet should be assigned a customer owned IP address. The `customer_owned_ipv4_pool` and `outpost_arn` arguments must be specified when set to `true`. Default is `false`.
     */
    @JvmName("kmuiqkkegikswwfc")
    public suspend fun mapCustomerOwnedIpOnLaunch(`value`: Output<Boolean>) {
        this.mapCustomerOwnedIpOnLaunch = value
    }

    /**
     * @param value Specify true to indicate
     * that instances launched into the subnet should be assigned
     * a public IP address. Default is `false`.
     */
    @JvmName("xkcehnygflrqgijd")
    public suspend fun mapPublicIpOnLaunch(`value`: Output<Boolean>) {
        this.mapPublicIpOnLaunch = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the Outpost.
     */
    @JvmName("xokuyivyqwvpcgqp")
    public suspend fun outpostArn(`value`: Output<String>) {
        this.outpostArn = value
    }

    /**
     * @param value The type of hostnames to assign to instances in the subnet at launch. For IPv6-only subnets, an instance DNS name must be based on the instance ID. For dual-stack and IPv4-only subnets, you can specify whether DNS names use the instance IPv4 address or the instance ID. Valid values: `ip-name`, `resource-name`.
     */
    @JvmName("sgamaayebwycapow")
    public suspend fun privateDnsHostnameTypeOnLaunch(`value`: Output<String>) {
        this.privateDnsHostnameTypeOnLaunch = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("iqilwxfhfhmuetsr")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The VPC ID.
     */
    @JvmName("fwoonuqqradvqexb")
    public suspend fun vpcId(`value`: Output<String>) {
        this.vpcId = value
    }

    /**
     * @param value Specify true to indicate
     * that network interfaces created in the specified subnet should be
     * assigned an IPv6 address. Default is `false`
     */
    @JvmName("mprulvabpdjeydae")
    public suspend fun assignIpv6AddressOnCreation(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.assignIpv6AddressOnCreation = mapped
    }

    /**
     * @param value AZ for the subnet.
     */
    @JvmName("pmwgxvetbvralysf")
    public suspend fun availabilityZone(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZone = mapped
    }

    /**
     * @param value AZ ID of the subnet. This argument is not supported in all regions or partitions. If necessary, use `availability_zone` instead.
     */
    @JvmName("cnvfhxfqmqxavxdb")
    public suspend fun availabilityZoneId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZoneId = mapped
    }

    /**
     * @param value The IPv4 CIDR block for the subnet.
     */
    @JvmName("xbffngmpdidnsyjl")
    public suspend fun cidrBlock(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cidrBlock = mapped
    }

    /**
     * @param value The customer owned IPv4 address pool. Typically used with the `map_customer_owned_ip_on_launch` argument. The `outpost_arn` argument must be specified when configured.
     */
    @JvmName("soihgvwcluolxlxo")
    public suspend fun customerOwnedIpv4Pool(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerOwnedIpv4Pool = mapped
    }

    /**
     * @param value Indicates whether DNS queries made to the Amazon-provided DNS Resolver in this subnet should return synthetic IPv6 addresses for IPv4-only destinations. Default: `false`.
     */
    @JvmName("rcaudnrjksimfuau")
    public suspend fun enableDns64(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableDns64 = mapped
    }

    /**
     * @param value Indicates the device position for local network interfaces in this subnet. For example, 1 indicates local network interfaces in this subnet are the secondary network interface (eth1). A local network interface cannot be the primary network interface (eth0).
     */
    @JvmName("rtythfarysibehrb")
    public suspend fun enableLniAtDeviceIndex(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableLniAtDeviceIndex = mapped
    }

    /**
     * @param value Indicates whether to respond to DNS queries for instance hostnames with DNS A records. Default: `false`.
     */
    @JvmName("sumirdsibkgehdfw")
    public suspend fun enableResourceNameDnsARecordOnLaunch(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableResourceNameDnsARecordOnLaunch = mapped
    }

    /**
     * @param value Indicates whether to respond to DNS queries for instance hostnames with DNS AAAA records. Default: `false`.
     */
    @JvmName("rpvxxhrrbdmuwadb")
    public suspend fun enableResourceNameDnsAaaaRecordOnLaunch(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableResourceNameDnsAaaaRecordOnLaunch = mapped
    }

    /**
     * @param value The IPv6 network range for the subnet,
     * in CIDR notation. The subnet size must use a /64 prefix length.
     */
    @JvmName("lptbwpluwbdoxipk")
    public suspend fun ipv6CidrBlock(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipv6CidrBlock = mapped
    }

    /**
     * @param value Indicates whether to create an IPv6-only subnet. Default: `false`.
     */
    @JvmName("fgktalgiwpxxgwpk")
    public suspend fun ipv6Native(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipv6Native = mapped
    }

    /**
     * @param value Specify `true` to indicate that network interfaces created in the subnet should be assigned a customer owned IP address. The `customer_owned_ipv4_pool` and `outpost_arn` arguments must be specified when set to `true`. Default is `false`.
     */
    @JvmName("eacfgcoqcmdthyhk")
    public suspend fun mapCustomerOwnedIpOnLaunch(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.mapCustomerOwnedIpOnLaunch = mapped
    }

    /**
     * @param value Specify true to indicate
     * that instances launched into the subnet should be assigned
     * a public IP address. Default is `false`.
     */
    @JvmName("wluanuvcyyplqgor")
    public suspend fun mapPublicIpOnLaunch(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.mapPublicIpOnLaunch = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the Outpost.
     */
    @JvmName("pjokmptmihcpqcqd")
    public suspend fun outpostArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.outpostArn = mapped
    }

    /**
     * @param value The type of hostnames to assign to instances in the subnet at launch. For IPv6-only subnets, an instance DNS name must be based on the instance ID. For dual-stack and IPv4-only subnets, you can specify whether DNS names use the instance IPv4 address or the instance ID. Valid values: `ip-name`, `resource-name`.
     */
    @JvmName("aowmsgvuccjldjjo")
    public suspend fun privateDnsHostnameTypeOnLaunch(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.privateDnsHostnameTypeOnLaunch = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vrjxrspnripauthd")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("nkdkevdhdsqomeco")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The VPC ID.
     */
    @JvmName("jxusjdftrkjhlevf")
    public suspend fun vpcId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcId = mapped
    }

    internal fun build(): SubnetArgs = SubnetArgs(
        assignIpv6AddressOnCreation = assignIpv6AddressOnCreation,
        availabilityZone = availabilityZone,
        availabilityZoneId = availabilityZoneId,
        cidrBlock = cidrBlock,
        customerOwnedIpv4Pool = customerOwnedIpv4Pool,
        enableDns64 = enableDns64,
        enableLniAtDeviceIndex = enableLniAtDeviceIndex,
        enableResourceNameDnsARecordOnLaunch = enableResourceNameDnsARecordOnLaunch,
        enableResourceNameDnsAaaaRecordOnLaunch = enableResourceNameDnsAaaaRecordOnLaunch,
        ipv6CidrBlock = ipv6CidrBlock,
        ipv6Native = ipv6Native,
        mapCustomerOwnedIpOnLaunch = mapCustomerOwnedIpOnLaunch,
        mapPublicIpOnLaunch = mapPublicIpOnLaunch,
        outpostArn = outpostArn,
        privateDnsHostnameTypeOnLaunch = privateDnsHostnameTypeOnLaunch,
        tags = tags,
        vpcId = vpcId,
    )
}
