@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.VpcEndpointArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.VpcEndpointDnsOptionsArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpcEndpointDnsOptionsArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.VpcEndpointSubnetConfigurationArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpcEndpointSubnetConfigurationArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a VPC Endpoint resource.
 * > **NOTE on VPC Endpoints and VPC Endpoint Associations:** The provider provides both standalone VPC Endpoint Associations for
 * Route Tables - (an association between a VPC endpoint and a single `route_table_id`),
 * Security Groups - (an association between a VPC endpoint and a single `security_group_id`),
 * and Subnets - (an association between a VPC endpoint and a single `subnet_id`) and
 * a VPC Endpoint resource with `route_table_ids` and `subnet_ids` attributes.
 * Do not use the same resource ID in both a VPC Endpoint resource and a VPC Endpoint Association resource.
 * Doing so will cause a conflict of associations and will overwrite the association.
 * ## Example Usage
 * ### Basic
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const s3 = new aws.ec2.VpcEndpoint("s3", {
 *     vpcId: main.id,
 *     serviceName: "com.amazonaws.us-west-2.s3",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * s3 = aws.ec2.VpcEndpoint("s3",
 *     vpc_id=main["id"],
 *     service_name="com.amazonaws.us-west-2.s3")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var s3 = new Aws.Ec2.VpcEndpoint("s3", new()
 *     {
 *         VpcId = main.Id,
 *         ServiceName = "com.amazonaws.us-west-2.s3",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewVpcEndpoint(ctx, "s3", &ec2.VpcEndpointArgs{
 * 			VpcId:       pulumi.Any(main.Id),
 * 			ServiceName: pulumi.String("com.amazonaws.us-west-2.s3"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcEndpoint;
 * import com.pulumi.aws.ec2.VpcEndpointArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var s3 = new VpcEndpoint("s3", VpcEndpointArgs.builder()
 *             .vpcId(main.id())
 *             .serviceName("com.amazonaws.us-west-2.s3")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   s3:
 *     type: aws:ec2:VpcEndpoint
 *     properties:
 *       vpcId: ${main.id}
 *       serviceName: com.amazonaws.us-west-2.s3
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Basic w/ Tags
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const s3 = new aws.ec2.VpcEndpoint("s3", {
 *     vpcId: main.id,
 *     serviceName: "com.amazonaws.us-west-2.s3",
 *     tags: {
 *         Environment: "test",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * s3 = aws.ec2.VpcEndpoint("s3",
 *     vpc_id=main["id"],
 *     service_name="com.amazonaws.us-west-2.s3",
 *     tags={
 *         "Environment": "test",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var s3 = new Aws.Ec2.VpcEndpoint("s3", new()
 *     {
 *         VpcId = main.Id,
 *         ServiceName = "com.amazonaws.us-west-2.s3",
 *         Tags =
 *         {
 *             { "Environment", "test" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewVpcEndpoint(ctx, "s3", &ec2.VpcEndpointArgs{
 * 			VpcId:       pulumi.Any(main.Id),
 * 			ServiceName: pulumi.String("com.amazonaws.us-west-2.s3"),
 * 			Tags: pulumi.StringMap{
 * 				"Environment": pulumi.String("test"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcEndpoint;
 * import com.pulumi.aws.ec2.VpcEndpointArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var s3 = new VpcEndpoint("s3", VpcEndpointArgs.builder()
 *             .vpcId(main.id())
 *             .serviceName("com.amazonaws.us-west-2.s3")
 *             .tags(Map.of("Environment", "test"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   s3:
 *     type: aws:ec2:VpcEndpoint
 *     properties:
 *       vpcId: ${main.id}
 *       serviceName: com.amazonaws.us-west-2.s3
 *       tags:
 *         Environment: test
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Interface Endpoint Type
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const ec2 = new aws.ec2.VpcEndpoint("ec2", {
 *     vpcId: main.id,
 *     serviceName: "com.amazonaws.us-west-2.ec2",
 *     vpcEndpointType: "Interface",
 *     securityGroupIds: [sg1&#46;id],
 *     privateDnsEnabled: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * ec2 = aws.ec2.VpcEndpoint("ec2",
 *     vpc_id=main["id"],
 *     service_name="com.amazonaws.us-west-2.ec2",
 *     vpc_endpoint_type="Interface",
 *     security_group_ids=[sg1["id"]],
 *     private_dns_enabled=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var ec2 = new Aws.Ec2.VpcEndpoint("ec2", new()
 *     {
 *         VpcId = main.Id,
 *         ServiceName = "com.amazonaws.us-west-2.ec2",
 *         VpcEndpointType = "Interface",
 *         SecurityGroupIds = new[]
 *         {
 *             sg1.Id,
 *         },
 *         PrivateDnsEnabled = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewVpcEndpoint(ctx, "ec2", &ec2.VpcEndpointArgs{
 * 			VpcId:           pulumi.Any(main.Id),
 * 			ServiceName:     pulumi.String("com.amazonaws.us-west-2.ec2"),
 * 			VpcEndpointType: pulumi.String("Interface"),
 * 			SecurityGroupIds: pulumi.StringArray{
 * 				sg1.Id,
 * 			},
 * 			PrivateDnsEnabled: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcEndpoint;
 * import com.pulumi.aws.ec2.VpcEndpointArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var ec2 = new VpcEndpoint("ec2", VpcEndpointArgs.builder()
 *             .vpcId(main.id())
 *             .serviceName("com.amazonaws.us-west-2.ec2")
 *             .vpcEndpointType("Interface")
 *             .securityGroupIds(sg1.id())
 *             .privateDnsEnabled(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   ec2:
 *     type: aws:ec2:VpcEndpoint
 *     properties:
 *       vpcId: ${main.id}
 *       serviceName: com.amazonaws.us-west-2.ec2
 *       vpcEndpointType: Interface
 *       securityGroupIds:
 *         - ${sg1.id}
 *       privateDnsEnabled: true
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Interface Endpoint Type with User-Defined IP Address
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const ec2 = new aws.ec2.VpcEndpoint("ec2", {
 *     vpcId: example.id,
 *     serviceName: "com.amazonaws.us-west-2.ec2",
 *     vpcEndpointType: "Interface",
 *     subnetConfigurations: [
 *         {
 *             ipv4: "10.0.1.10",
 *             subnetId: example1.id,
 *         },
 *         {
 *             ipv4: "10.0.2.10",
 *             subnetId: example2.id,
 *         },
 *     ],
 *     subnetIds: [
 *         example1.id,
 *         example2.id,
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * ec2 = aws.ec2.VpcEndpoint("ec2",
 *     vpc_id=example["id"],
 *     service_name="com.amazonaws.us-west-2.ec2",
 *     vpc_endpoint_type="Interface",
 *     subnet_configurations=[
 *         {
 *             "ipv4": "10.0.1.10",
 *             "subnet_id": example1["id"],
 *         },
 *         {
 *             "ipv4": "10.0.2.10",
 *             "subnet_id": example2["id"],
 *         },
 *     ],
 *     subnet_ids=[
 *         example1["id"],
 *         example2["id"],
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var ec2 = new Aws.Ec2.VpcEndpoint("ec2", new()
 *     {
 *         VpcId = example.Id,
 *         ServiceName = "com.amazonaws.us-west-2.ec2",
 *         VpcEndpointType = "Interface",
 *         SubnetConfigurations = new[]
 *         {
 *             new Aws.Ec2.Inputs.VpcEndpointSubnetConfigurationArgs
 *             {
 *                 Ipv4 = "10.0.1.10",
 *                 SubnetId = example1.Id,
 *             },
 *             new Aws.Ec2.Inputs.VpcEndpointSubnetConfigurationArgs
 *             {
 *                 Ipv4 = "10.0.2.10",
 *                 SubnetId = example2.Id,
 *             },
 *         },
 *         SubnetIds = new[]
 *         {
 *             example1.Id,
 *             example2.Id,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewVpcEndpoint(ctx, "ec2", &ec2.VpcEndpointArgs{
 * 			VpcId:           pulumi.Any(example.Id),
 * 			ServiceName:     pulumi.String("com.amazonaws.us-west-2.ec2"),
 * 			VpcEndpointType: pulumi.String("Interface"),
 * 			SubnetConfigurations: ec2.VpcEndpointSubnetConfigurationArray{
 * 				&ec2.VpcEndpointSubnetConfigurationArgs{
 * 					Ipv4:     pulumi.String("10.0.1.10"),
 * 					SubnetId: pulumi.Any(example1.Id),
 * 				},
 * 				&ec2.VpcEndpointSubnetConfigurationArgs{
 * 					Ipv4:     pulumi.String("10.0.2.10"),
 * 					SubnetId: pulumi.Any(example2.Id),
 * 				},
 * 			},
 * 			SubnetIds: pulumi.StringArray{
 * 				example1.Id,
 * 				example2.Id,
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcEndpoint;
 * import com.pulumi.aws.ec2.VpcEndpointArgs;
 * import com.pulumi.aws.ec2.inputs.VpcEndpointSubnetConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var ec2 = new VpcEndpoint("ec2", VpcEndpointArgs.builder()
 *             .vpcId(example.id())
 *             .serviceName("com.amazonaws.us-west-2.ec2")
 *             .vpcEndpointType("Interface")
 *             .subnetConfigurations(
 *                 VpcEndpointSubnetConfigurationArgs.builder()
 *                     .ipv4("10.0.1.10")
 *                     .subnetId(example1.id())
 *                     .build(),
 *                 VpcEndpointSubnetConfigurationArgs.builder()
 *                     .ipv4("10.0.2.10")
 *                     .subnetId(example2.id())
 *                     .build())
 *             .subnetIds(
 *                 example1.id(),
 *                 example2.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   ec2:
 *     type: aws:ec2:VpcEndpoint
 *     properties:
 *       vpcId: ${example.id}
 *       serviceName: com.amazonaws.us-west-2.ec2
 *       vpcEndpointType: Interface
 *       subnetConfigurations:
 *         - ipv4: 10.0.1.10
 *           subnetId: ${example1.id}
 *         - ipv4: 10.0.2.10
 *           subnetId: ${example2.id}
 *       subnetIds:
 *         - ${example1.id}
 *         - ${example2.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Gateway Load Balancer Endpoint Type
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getCallerIdentity({});
 * const example = new aws.ec2.VpcEndpointService("example", {
 *     acceptanceRequired: false,
 *     allowedPrincipals: [current&#46;then(current => current&#46;arn)],
 *     gatewayLoadBalancerArns: [exampleAwsLb&#46;arn],
 * });
 * const exampleVpcEndpoint = new aws.ec2.VpcEndpoint("example", {
 *     serviceName: example.serviceName,
 *     subnetIds: [exampleAwsSubnet&#46;id],
 *     vpcEndpointType: example.serviceType,
 *     vpcId: exampleAwsVpc.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_caller_identity()
 * example = aws.ec2.VpcEndpointService("example",
 *     acceptance_required=False,
 *     allowed_principals=[current&#46;arn],
 *     gateway_load_balancer_arns=[example_aws_lb["arn"]])
 * example_vpc_endpoint = aws.ec2.VpcEndpoint("example",
 *     service_name=example.service_name,
 *     subnet_ids=[example_aws_subnet["id"]],
 *     vpc_endpoint_type=example.service_type,
 *     vpc_id=example_aws_vpc["id"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetCallerIdentity.Invoke();
 *     var example = new Aws.Ec2.VpcEndpointService("example", new()
 *     {
 *         AcceptanceRequired = false,
 *         AllowedPrincipals = new[]
 *         {
 *             current.Apply(getCallerIdentityResult => getCallerIdentityResult.Arn),
 *         },
 *         GatewayLoadBalancerArns = new[]
 *         {
 *             exampleAwsLb.Arn,
 *         },
 *     });
 *     var exampleVpcEndpoint = new Aws.Ec2.VpcEndpoint("example", new()
 *     {
 *         ServiceName = example.ServiceName,
 *         SubnetIds = new[]
 *         {
 *             exampleAwsSubnet.Id,
 *         },
 *         VpcEndpointType = example.ServiceType,
 *         VpcId = exampleAwsVpc.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := ec2.NewVpcEndpointService(ctx, "example", &ec2.VpcEndpointServiceArgs{
 * 			AcceptanceRequired: pulumi.Bool(false),
 * 			AllowedPrincipals: pulumi.StringArray{
 * 				pulumi.String(current.Arn),
 * 			},
 * 			GatewayLoadBalancerArns: pulumi.StringArray{
 * 				exampleAwsLb.Arn,
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcEndpoint(ctx, "example", &ec2.VpcEndpointArgs{
 * 			ServiceName: example.ServiceName,
 * 			SubnetIds: pulumi.StringArray{
 * 				exampleAwsSubnet.Id,
 * 			},
 * 			VpcEndpointType: example.ServiceType,
 * 			VpcId:           pulumi.Any(exampleAwsVpc.Id),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.ec2.VpcEndpointService;
 * import com.pulumi.aws.ec2.VpcEndpointServiceArgs;
 * import com.pulumi.aws.ec2.VpcEndpoint;
 * import com.pulumi.aws.ec2.VpcEndpointArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getCallerIdentity();
 *         var example = new VpcEndpointService("example", VpcEndpointServiceArgs.builder()
 *             .acceptanceRequired(false)
 *             .allowedPrincipals(current.applyValue(getCallerIdentityResult -> getCallerIdentityResult.arn()))
 *             .gatewayLoadBalancerArns(exampleAwsLb.arn())
 *             .build());
 *         var exampleVpcEndpoint = new VpcEndpoint("exampleVpcEndpoint", VpcEndpointArgs.builder()
 *             .serviceName(example.serviceName())
 *             .subnetIds(exampleAwsSubnet.id())
 *             .vpcEndpointType(example.serviceType())
 *             .vpcId(exampleAwsVpc.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:VpcEndpointService
 *     properties:
 *       acceptanceRequired: false
 *       allowedPrincipals:
 *         - ${current.arn}
 *       gatewayLoadBalancerArns:
 *         - ${exampleAwsLb.arn}
 *   exampleVpcEndpoint:
 *     type: aws:ec2:VpcEndpoint
 *     name: example
 *     properties:
 *       serviceName: ${example.serviceName}
 *       subnetIds:
 *         - ${exampleAwsSubnet.id}
 *       vpcEndpointType: ${example.serviceType}
 *       vpcId: ${exampleAwsVpc.id}
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import VPC Endpoints using the VPC endpoint `id`. For example:
 * ```sh
 * $ pulumi import aws:ec2/vpcEndpoint:VpcEndpoint endpoint1 vpce-3ecf2a57
 * ```
 * @property autoAccept Accept the VPC endpoint (the VPC endpoint and service need to be in the same AWS account).
 * @property dnsOptions The DNS options for the endpoint. See dns_options below.
 * @property ipAddressType The IP address type for the endpoint. Valid values are `ipv4`, `dualstack`, and `ipv6`.
 * @property policy A policy to attach to the endpoint that controls access to the service. This is a JSON formatted string. Defaults to full access. All `Gateway` and some `Interface` endpoints support policies - see the [relevant AWS documentation](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) for more details.
 * @property privateDnsEnabled Whether or not to associate a private hosted zone with the specified VPC. Applicable for endpoints of type `Interface`. Most users will want this enabled to allow services within the VPC to automatically use the endpoint.
 * Defaults to `false`.
 * @property routeTableIds One or more route table IDs. Applicable for endpoints of type `Gateway`.
 * @property securityGroupIds The ID of one or more security groups to associate with the network interface. Applicable for endpoints of type `Interface`.
 * If no security groups are specified, the VPC's [default security group](https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html#DefaultSecurityGroup) is associated with the endpoint.
 * @property serviceName The service name. For AWS services the service name is usually in the form `com.amazonaws.<region>.<service>` (the SageMaker Notebook service is an exception to this rule, the service name is in the form `aws.sagemaker.<region>.notebook`).
 * @property subnetConfigurations Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
 * @property subnetIds The ID of one or more subnets in which to create a network interface for the endpoint. Applicable for endpoints of type `GatewayLoadBalancer` and `Interface`. Interface type endpoints cannot function without being assigned to a subnet.
 * @property tags A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcEndpointType The VPC endpoint type, `Gateway`, `GatewayLoadBalancer`, or `Interface`. Defaults to `Gateway`.
 * @property vpcId The ID of the VPC in which the endpoint will be used.
 */
public data class VpcEndpointArgs(
    public val autoAccept: Output<Boolean>? = null,
    public val dnsOptions: Output<VpcEndpointDnsOptionsArgs>? = null,
    public val ipAddressType: Output<String>? = null,
    public val policy: Output<String>? = null,
    public val privateDnsEnabled: Output<Boolean>? = null,
    public val routeTableIds: Output<List<String>>? = null,
    public val securityGroupIds: Output<List<String>>? = null,
    public val serviceName: Output<String>? = null,
    public val subnetConfigurations: Output<List<VpcEndpointSubnetConfigurationArgs>>? = null,
    public val subnetIds: Output<List<String>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcEndpointType: Output<String>? = null,
    public val vpcId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.VpcEndpointArgs> {
    override fun toJava(): com.pulumi.aws.ec2.VpcEndpointArgs =
        com.pulumi.aws.ec2.VpcEndpointArgs.builder()
            .autoAccept(autoAccept?.applyValue({ args0 -> args0 }))
            .dnsOptions(dnsOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .ipAddressType(ipAddressType?.applyValue({ args0 -> args0 }))
            .policy(policy?.applyValue({ args0 -> args0 }))
            .privateDnsEnabled(privateDnsEnabled?.applyValue({ args0 -> args0 }))
            .routeTableIds(routeTableIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .securityGroupIds(securityGroupIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .serviceName(serviceName?.applyValue({ args0 -> args0 }))
            .subnetConfigurations(
                subnetConfigurations?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .subnetIds(subnetIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .vpcEndpointType(vpcEndpointType?.applyValue({ args0 -> args0 }))
            .vpcId(vpcId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [VpcEndpointArgs].
 */
@PulumiTagMarker
public class VpcEndpointArgsBuilder internal constructor() {
    private var autoAccept: Output<Boolean>? = null

    private var dnsOptions: Output<VpcEndpointDnsOptionsArgs>? = null

    private var ipAddressType: Output<String>? = null

    private var policy: Output<String>? = null

    private var privateDnsEnabled: Output<Boolean>? = null

    private var routeTableIds: Output<List<String>>? = null

    private var securityGroupIds: Output<List<String>>? = null

    private var serviceName: Output<String>? = null

    private var subnetConfigurations: Output<List<VpcEndpointSubnetConfigurationArgs>>? = null

    private var subnetIds: Output<List<String>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcEndpointType: Output<String>? = null

    private var vpcId: Output<String>? = null

    /**
     * @param value Accept the VPC endpoint (the VPC endpoint and service need to be in the same AWS account).
     */
    @JvmName("isoecswbesgtkjej")
    public suspend fun autoAccept(`value`: Output<Boolean>) {
        this.autoAccept = value
    }

    /**
     * @param value The DNS options for the endpoint. See dns_options below.
     */
    @JvmName("ahsajxxmgojpytrc")
    public suspend fun dnsOptions(`value`: Output<VpcEndpointDnsOptionsArgs>) {
        this.dnsOptions = value
    }

    /**
     * @param value The IP address type for the endpoint. Valid values are `ipv4`, `dualstack`, and `ipv6`.
     */
    @JvmName("kbipqvyobcynbcxx")
    public suspend fun ipAddressType(`value`: Output<String>) {
        this.ipAddressType = value
    }

    /**
     * @param value A policy to attach to the endpoint that controls access to the service. This is a JSON formatted string. Defaults to full access. All `Gateway` and some `Interface` endpoints support policies - see the [relevant AWS documentation](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) for more details.
     */
    @JvmName("lhgrfqgsqwtkmndl")
    public suspend fun policy(`value`: Output<String>) {
        this.policy = value
    }

    /**
     * @param value Whether or not to associate a private hosted zone with the specified VPC. Applicable for endpoints of type `Interface`. Most users will want this enabled to allow services within the VPC to automatically use the endpoint.
     * Defaults to `false`.
     */
    @JvmName("oiuhimyohgfdagwh")
    public suspend fun privateDnsEnabled(`value`: Output<Boolean>) {
        this.privateDnsEnabled = value
    }

    /**
     * @param value One or more route table IDs. Applicable for endpoints of type `Gateway`.
     */
    @JvmName("uusjlvtthbkcrqvq")
    public suspend fun routeTableIds(`value`: Output<List<String>>) {
        this.routeTableIds = value
    }

    @JvmName("qrmntjqxpoisujsi")
    public suspend fun routeTableIds(vararg values: Output<String>) {
        this.routeTableIds = Output.all(values.asList())
    }

    /**
     * @param values One or more route table IDs. Applicable for endpoints of type `Gateway`.
     */
    @JvmName("kcfjphqddxwfdaya")
    public suspend fun routeTableIds(values: List<Output<String>>) {
        this.routeTableIds = Output.all(values)
    }

    /**
     * @param value The ID of one or more security groups to associate with the network interface. Applicable for endpoints of type `Interface`.
     * If no security groups are specified, the VPC's [default security group](https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html#DefaultSecurityGroup) is associated with the endpoint.
     */
    @JvmName("ybkanlpewhlyouuc")
    public suspend fun securityGroupIds(`value`: Output<List<String>>) {
        this.securityGroupIds = value
    }

    @JvmName("nbivlfalxttlltbn")
    public suspend fun securityGroupIds(vararg values: Output<String>) {
        this.securityGroupIds = Output.all(values.asList())
    }

    /**
     * @param values The ID of one or more security groups to associate with the network interface. Applicable for endpoints of type `Interface`.
     * If no security groups are specified, the VPC's [default security group](https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html#DefaultSecurityGroup) is associated with the endpoint.
     */
    @JvmName("owxqqrelniaubpek")
    public suspend fun securityGroupIds(values: List<Output<String>>) {
        this.securityGroupIds = Output.all(values)
    }

    /**
     * @param value The service name. For AWS services the service name is usually in the form `com.amazonaws.<region>.<service>` (the SageMaker Notebook service is an exception to this rule, the service name is in the form `aws.sagemaker.<region>.notebook`).
     */
    @JvmName("vhhdraauvsvbcqri")
    public suspend fun serviceName(`value`: Output<String>) {
        this.serviceName = value
    }

    /**
     * @param value Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
     */
    @JvmName("ntqpomyvnkmbcvmx")
    public suspend fun subnetConfigurations(`value`: Output<List<VpcEndpointSubnetConfigurationArgs>>) {
        this.subnetConfigurations = value
    }

    @JvmName("vinrlyvckjolmkpa")
    public suspend fun subnetConfigurations(vararg values: Output<VpcEndpointSubnetConfigurationArgs>) {
        this.subnetConfigurations = Output.all(values.asList())
    }

    /**
     * @param values Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
     */
    @JvmName("iherjykcrdrttips")
    public suspend fun subnetConfigurations(values: List<Output<VpcEndpointSubnetConfigurationArgs>>) {
        this.subnetConfigurations = Output.all(values)
    }

    /**
     * @param value The ID of one or more subnets in which to create a network interface for the endpoint. Applicable for endpoints of type `GatewayLoadBalancer` and `Interface`. Interface type endpoints cannot function without being assigned to a subnet.
     */
    @JvmName("yhjgqoruaghhkjdx")
    public suspend fun subnetIds(`value`: Output<List<String>>) {
        this.subnetIds = value
    }

    @JvmName("ytjmsfgbugmmxxxv")
    public suspend fun subnetIds(vararg values: Output<String>) {
        this.subnetIds = Output.all(values.asList())
    }

    /**
     * @param values The ID of one or more subnets in which to create a network interface for the endpoint. Applicable for endpoints of type `GatewayLoadBalancer` and `Interface`. Interface type endpoints cannot function without being assigned to a subnet.
     */
    @JvmName("vmnsqcumjsmcmwhg")
    public suspend fun subnetIds(values: List<Output<String>>) {
        this.subnetIds = Output.all(values)
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("algpkhoiqinafhnb")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The VPC endpoint type, `Gateway`, `GatewayLoadBalancer`, or `Interface`. Defaults to `Gateway`.
     */
    @JvmName("imoqjtefsowamojr")
    public suspend fun vpcEndpointType(`value`: Output<String>) {
        this.vpcEndpointType = value
    }

    /**
     * @param value The ID of the VPC in which the endpoint will be used.
     */
    @JvmName("jrnfsjcimtswbxqm")
    public suspend fun vpcId(`value`: Output<String>) {
        this.vpcId = value
    }

    /**
     * @param value Accept the VPC endpoint (the VPC endpoint and service need to be in the same AWS account).
     */
    @JvmName("srdjnbkamilxnhbk")
    public suspend fun autoAccept(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoAccept = mapped
    }

    /**
     * @param value The DNS options for the endpoint. See dns_options below.
     */
    @JvmName("jincviijlxruqqyy")
    public suspend fun dnsOptions(`value`: VpcEndpointDnsOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dnsOptions = mapped
    }

    /**
     * @param argument The DNS options for the endpoint. See dns_options below.
     */
    @JvmName("shscrknusuoafrjb")
    public suspend fun dnsOptions(argument: suspend VpcEndpointDnsOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = VpcEndpointDnsOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.dnsOptions = mapped
    }

    /**
     * @param value The IP address type for the endpoint. Valid values are `ipv4`, `dualstack`, and `ipv6`.
     */
    @JvmName("rkmmkwitxhmevimg")
    public suspend fun ipAddressType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipAddressType = mapped
    }

    /**
     * @param value A policy to attach to the endpoint that controls access to the service. This is a JSON formatted string. Defaults to full access. All `Gateway` and some `Interface` endpoints support policies - see the [relevant AWS documentation](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) for more details.
     */
    @JvmName("dkmmgoksgpftrtpb")
    public suspend fun policy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policy = mapped
    }

    /**
     * @param value Whether or not to associate a private hosted zone with the specified VPC. Applicable for endpoints of type `Interface`. Most users will want this enabled to allow services within the VPC to automatically use the endpoint.
     * Defaults to `false`.
     */
    @JvmName("lfoeokanllatmkkx")
    public suspend fun privateDnsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.privateDnsEnabled = mapped
    }

    /**
     * @param value One or more route table IDs. Applicable for endpoints of type `Gateway`.
     */
    @JvmName("hbkpojqkyunbjrty")
    public suspend fun routeTableIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.routeTableIds = mapped
    }

    /**
     * @param values One or more route table IDs. Applicable for endpoints of type `Gateway`.
     */
    @JvmName("ogmtbcoflwhivrdo")
    public suspend fun routeTableIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.routeTableIds = mapped
    }

    /**
     * @param value The ID of one or more security groups to associate with the network interface. Applicable for endpoints of type `Interface`.
     * If no security groups are specified, the VPC's [default security group](https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html#DefaultSecurityGroup) is associated with the endpoint.
     */
    @JvmName("uudvmjrhpjvupftv")
    public suspend fun securityGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroupIds = mapped
    }

    /**
     * @param values The ID of one or more security groups to associate with the network interface. Applicable for endpoints of type `Interface`.
     * If no security groups are specified, the VPC's [default security group](https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html#DefaultSecurityGroup) is associated with the endpoint.
     */
    @JvmName("drvwsvfcslvqcjsp")
    public suspend fun securityGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroupIds = mapped
    }

    /**
     * @param value The service name. For AWS services the service name is usually in the form `com.amazonaws.<region>.<service>` (the SageMaker Notebook service is an exception to this rule, the service name is in the form `aws.sagemaker.<region>.notebook`).
     */
    @JvmName("wxayviqtbwbsikcs")
    public suspend fun serviceName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serviceName = mapped
    }

    /**
     * @param value Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
     */
    @JvmName("ipngdvmutepslfqs")
    public suspend fun subnetConfigurations(`value`: List<VpcEndpointSubnetConfigurationArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetConfigurations = mapped
    }

    /**
     * @param argument Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
     */
    @JvmName("ddwstxegkqcwfibi")
    public suspend fun subnetConfigurations(argument: List<suspend VpcEndpointSubnetConfigurationArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            VpcEndpointSubnetConfigurationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.subnetConfigurations = mapped
    }

    /**
     * @param argument Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
     */
    @JvmName("jjdashtlilhjyapo")
    public suspend fun subnetConfigurations(vararg argument: suspend VpcEndpointSubnetConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            VpcEndpointSubnetConfigurationArgsBuilder().applySuspend { it() }.build()
        }
        val mapped = of(toBeMapped)
        this.subnetConfigurations = mapped
    }

    /**
     * @param argument Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
     */
    @JvmName("ohjfdtsjwqgshwbd")
    public suspend fun subnetConfigurations(argument: suspend VpcEndpointSubnetConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            VpcEndpointSubnetConfigurationArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.subnetConfigurations = mapped
    }

    /**
     * @param values Subnet configuration for the endpoint, used to select specific IPv4 and/or IPv6 addresses to the endpoint. See subnet_configuration below.
     */
    @JvmName("xoxlwcnkgqpkjivv")
    public suspend fun subnetConfigurations(vararg values: VpcEndpointSubnetConfigurationArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.subnetConfigurations = mapped
    }

    /**
     * @param value The ID of one or more subnets in which to create a network interface for the endpoint. Applicable for endpoints of type `GatewayLoadBalancer` and `Interface`. Interface type endpoints cannot function without being assigned to a subnet.
     */
    @JvmName("olnqmpfqhtagvmtj")
    public suspend fun subnetIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetIds = mapped
    }

    /**
     * @param values The ID of one or more subnets in which to create a network interface for the endpoint. Applicable for endpoints of type `GatewayLoadBalancer` and `Interface`. Interface type endpoints cannot function without being assigned to a subnet.
     */
    @JvmName("oqctudkqqakrlkau")
    public suspend fun subnetIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.subnetIds = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("rsexlfnnjmsxhoby")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("iqexjidolphaoelv")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The VPC endpoint type, `Gateway`, `GatewayLoadBalancer`, or `Interface`. Defaults to `Gateway`.
     */
    @JvmName("tqjkiggrawyimesy")
    public suspend fun vpcEndpointType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcEndpointType = mapped
    }

    /**
     * @param value The ID of the VPC in which the endpoint will be used.
     */
    @JvmName("pcxgwoxhafbqhvgf")
    public suspend fun vpcId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcId = mapped
    }

    internal fun build(): VpcEndpointArgs = VpcEndpointArgs(
        autoAccept = autoAccept,
        dnsOptions = dnsOptions,
        ipAddressType = ipAddressType,
        policy = policy,
        privateDnsEnabled = privateDnsEnabled,
        routeTableIds = routeTableIds,
        securityGroupIds = securityGroupIds,
        serviceName = serviceName,
        subnetConfigurations = subnetConfigurations,
        subnetIds = subnetIds,
        tags = tags,
        vpcEndpointType = vpcEndpointType,
        vpcId = vpcId,
    )
}
