@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.VpcIpamPoolArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an IP address pool resource for IPAM.
 * ## Example Usage
 * Basic usage:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getRegion({});
 * const example = new aws.ec2.VpcIpam("example", {operatingRegions: [{
 *     regionName: current.then(current => current.name),
 * }]});
 * const exampleVpcIpamPool = new aws.ec2.VpcIpamPool("example", {
 *     addressFamily: "ipv4",
 *     ipamScopeId: example.privateDefaultScopeId,
 *     locale: current.then(current => current.name),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_region()
 * example = aws.ec2.VpcIpam("example", operating_regions=[{
 *     "region_name": current.name,
 * }])
 * example_vpc_ipam_pool = aws.ec2.VpcIpamPool("example",
 *     address_family="ipv4",
 *     ipam_scope_id=example.private_default_scope_id,
 *     locale=current.name)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetRegion.Invoke();
 *     var example = new Aws.Ec2.VpcIpam("example", new()
 *     {
 *         OperatingRegions = new[]
 *         {
 *             new Aws.Ec2.Inputs.VpcIpamOperatingRegionArgs
 *             {
 *                 RegionName = current.Apply(getRegionResult => getRegionResult.Name),
 *             },
 *         },
 *     });
 *     var exampleVpcIpamPool = new Aws.Ec2.VpcIpamPool("example", new()
 *     {
 *         AddressFamily = "ipv4",
 *         IpamScopeId = example.PrivateDefaultScopeId,
 *         Locale = current.Apply(getRegionResult => getRegionResult.Name),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := ec2.NewVpcIpam(ctx, "example", &ec2.VpcIpamArgs{
 * 			OperatingRegions: ec2.VpcIpamOperatingRegionArray{
 * 				&ec2.VpcIpamOperatingRegionArgs{
 * 					RegionName: pulumi.String(current.Name),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcIpamPool(ctx, "example", &ec2.VpcIpamPoolArgs{
 * 			AddressFamily: pulumi.String("ipv4"),
 * 			IpamScopeId:   example.PrivateDefaultScopeId,
 * 			Locale:        pulumi.String(current.Name),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpam;
 * import com.pulumi.aws.ec2.VpcIpamArgs;
 * import com.pulumi.aws.ec2.inputs.VpcIpamOperatingRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpamPool;
 * import com.pulumi.aws.ec2.VpcIpamPoolArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getRegion();
 *         var example = new VpcIpam("example", VpcIpamArgs.builder()
 *             .operatingRegions(VpcIpamOperatingRegionArgs.builder()
 *                 .regionName(current.applyValue(getRegionResult -> getRegionResult.name()))
 *                 .build())
 *             .build());
 *         var exampleVpcIpamPool = new VpcIpamPool("exampleVpcIpamPool", VpcIpamPoolArgs.builder()
 *             .addressFamily("ipv4")
 *             .ipamScopeId(example.privateDefaultScopeId())
 *             .locale(current.applyValue(getRegionResult -> getRegionResult.name()))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:VpcIpam
 *     properties:
 *       operatingRegions:
 *         - regionName: ${current.name}
 *   exampleVpcIpamPool:
 *     type: aws:ec2:VpcIpamPool
 *     name: example
 *     properties:
 *       addressFamily: ipv4
 *       ipamScopeId: ${example.privateDefaultScopeId}
 *       locale: ${current.name}
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * Nested Pools:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getRegion({});
 * const example = new aws.ec2.VpcIpam("example", {operatingRegions: [{
 *     regionName: current.then(current => current.name),
 * }]});
 * const parent = new aws.ec2.VpcIpamPool("parent", {
 *     addressFamily: "ipv4",
 *     ipamScopeId: example.privateDefaultScopeId,
 * });
 * const parentTest = new aws.ec2.VpcIpamPoolCidr("parent_test", {
 *     ipamPoolId: parent.id,
 *     cidr: "172.20.0.0/16",
 * });
 * const child = new aws.ec2.VpcIpamPool("child", {
 *     addressFamily: "ipv4",
 *     ipamScopeId: example.privateDefaultScopeId,
 *     locale: current.then(current => current.name),
 *     sourceIpamPoolId: parent.id,
 * });
 * const childTest = new aws.ec2.VpcIpamPoolCidr("child_test", {
 *     ipamPoolId: child.id,
 *     cidr: "172.20.0.0/24",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_region()
 * example = aws.ec2.VpcIpam("example", operating_regions=[{
 *     "region_name": current.name,
 * }])
 * parent = aws.ec2.VpcIpamPool("parent",
 *     address_family="ipv4",
 *     ipam_scope_id=example.private_default_scope_id)
 * parent_test = aws.ec2.VpcIpamPoolCidr("parent_test",
 *     ipam_pool_id=parent.id,
 *     cidr="172.20.0.0/16")
 * child = aws.ec2.VpcIpamPool("child",
 *     address_family="ipv4",
 *     ipam_scope_id=example.private_default_scope_id,
 *     locale=current.name,
 *     source_ipam_pool_id=parent.id)
 * child_test = aws.ec2.VpcIpamPoolCidr("child_test",
 *     ipam_pool_id=child.id,
 *     cidr="172.20.0.0/24")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetRegion.Invoke();
 *     var example = new Aws.Ec2.VpcIpam("example", new()
 *     {
 *         OperatingRegions = new[]
 *         {
 *             new Aws.Ec2.Inputs.VpcIpamOperatingRegionArgs
 *             {
 *                 RegionName = current.Apply(getRegionResult => getRegionResult.Name),
 *             },
 *         },
 *     });
 *     var parent = new Aws.Ec2.VpcIpamPool("parent", new()
 *     {
 *         AddressFamily = "ipv4",
 *         IpamScopeId = example.PrivateDefaultScopeId,
 *     });
 *     var parentTest = new Aws.Ec2.VpcIpamPoolCidr("parent_test", new()
 *     {
 *         IpamPoolId = parent.Id,
 *         Cidr = "172.20.0.0/16",
 *     });
 *     var child = new Aws.Ec2.VpcIpamPool("child", new()
 *     {
 *         AddressFamily = "ipv4",
 *         IpamScopeId = example.PrivateDefaultScopeId,
 *         Locale = current.Apply(getRegionResult => getRegionResult.Name),
 *         SourceIpamPoolId = parent.Id,
 *     });
 *     var childTest = new Aws.Ec2.VpcIpamPoolCidr("child_test", new()
 *     {
 *         IpamPoolId = child.Id,
 *         Cidr = "172.20.0.0/24",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := ec2.NewVpcIpam(ctx, "example", &ec2.VpcIpamArgs{
 * 			OperatingRegions: ec2.VpcIpamOperatingRegionArray{
 * 				&ec2.VpcIpamOperatingRegionArgs{
 * 					RegionName: pulumi.String(current.Name),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		parent, err := ec2.NewVpcIpamPool(ctx, "parent", &ec2.VpcIpamPoolArgs{
 * 			AddressFamily: pulumi.String("ipv4"),
 * 			IpamScopeId:   example.PrivateDefaultScopeId,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcIpamPoolCidr(ctx, "parent_test", &ec2.VpcIpamPoolCidrArgs{
 * 			IpamPoolId: parent.ID(),
 * 			Cidr:       pulumi.String("172.20.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		child, err := ec2.NewVpcIpamPool(ctx, "child", &ec2.VpcIpamPoolArgs{
 * 			AddressFamily:    pulumi.String("ipv4"),
 * 			IpamScopeId:      example.PrivateDefaultScopeId,
 * 			Locale:           pulumi.String(current.Name),
 * 			SourceIpamPoolId: parent.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcIpamPoolCidr(ctx, "child_test", &ec2.VpcIpamPoolCidrArgs{
 * 			IpamPoolId: child.ID(),
 * 			Cidr:       pulumi.String("172.20.0.0/24"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpam;
 * import com.pulumi.aws.ec2.VpcIpamArgs;
 * import com.pulumi.aws.ec2.inputs.VpcIpamOperatingRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpamPool;
 * import com.pulumi.aws.ec2.VpcIpamPoolArgs;
 * import com.pulumi.aws.ec2.VpcIpamPoolCidr;
 * import com.pulumi.aws.ec2.VpcIpamPoolCidrArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getRegion();
 *         var example = new VpcIpam("example", VpcIpamArgs.builder()
 *             .operatingRegions(VpcIpamOperatingRegionArgs.builder()
 *                 .regionName(current.applyValue(getRegionResult -> getRegionResult.name()))
 *                 .build())
 *             .build());
 *         var parent = new VpcIpamPool("parent", VpcIpamPoolArgs.builder()
 *             .addressFamily("ipv4")
 *             .ipamScopeId(example.privateDefaultScopeId())
 *             .build());
 *         var parentTest = new VpcIpamPoolCidr("parentTest", VpcIpamPoolCidrArgs.builder()
 *             .ipamPoolId(parent.id())
 *             .cidr("172.20.0.0/16")
 *             .build());
 *         var child = new VpcIpamPool("child", VpcIpamPoolArgs.builder()
 *             .addressFamily("ipv4")
 *             .ipamScopeId(example.privateDefaultScopeId())
 *             .locale(current.applyValue(getRegionResult -> getRegionResult.name()))
 *             .sourceIpamPoolId(parent.id())
 *             .build());
 *         var childTest = new VpcIpamPoolCidr("childTest", VpcIpamPoolCidrArgs.builder()
 *             .ipamPoolId(child.id())
 *             .cidr("172.20.0.0/24")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:VpcIpam
 *     properties:
 *       operatingRegions:
 *         - regionName: ${current.name}
 *   parent:
 *     type: aws:ec2:VpcIpamPool
 *     properties:
 *       addressFamily: ipv4
 *       ipamScopeId: ${example.privateDefaultScopeId}
 *   parentTest:
 *     type: aws:ec2:VpcIpamPoolCidr
 *     name: parent_test
 *     properties:
 *       ipamPoolId: ${parent.id}
 *       cidr: 172.20.0.0/16
 *   child:
 *     type: aws:ec2:VpcIpamPool
 *     properties:
 *       addressFamily: ipv4
 *       ipamScopeId: ${example.privateDefaultScopeId}
 *       locale: ${current.name}
 *       sourceIpamPoolId: ${parent.id}
 *   childTest:
 *     type: aws:ec2:VpcIpamPoolCidr
 *     name: child_test
 *     properties:
 *       ipamPoolId: ${child.id}
 *       cidr: 172.20.0.0/24
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import IPAMs using the IPAM pool `id`. For example:
 * ```sh
 * $ pulumi import aws:ec2/vpcIpamPool:VpcIpamPool example ipam-pool-0958f95207d978e1e
 * ```
 * @property addressFamily The IP protocol assigned to this pool. You must choose either IPv4 or IPv6 protocol for a pool.
 * @property allocationDefaultNetmaskLength A default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16 (unless you provide a different netmask value when you create the new allocation).
 * @property allocationMaxNetmaskLength The maximum netmask length that will be required for CIDR allocations in this pool.
 * @property allocationMinNetmaskLength The minimum netmask length that will be required for CIDR allocations in this pool.
 * @property allocationResourceTags Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.
 * @property autoImport If you include this argument, IPAM automatically imports any VPCs you have in your scope that fall
 * within the CIDR range in the pool.
 * @property awsService Limits which AWS service the pool can be used in. Only useable on public scopes. Valid Values: `ec2`.
 * @property cascade Enables you to quickly delete an IPAM pool and all resources within that pool, including provisioned CIDRs, allocations, and other pools.
 * @property description A description for the IPAM pool.
 * @property ipamScopeId The ID of the scope in which you would like to create the IPAM pool.
 * @property locale The locale in which you would like to create the IPAM pool. Locale is the Region where you want to make an IPAM pool available for allocations. You can only create pools with locales that match the operating Regions of the IPAM. You can only create VPCs from a pool whose locale matches the VPC's Region. Possible values: Any AWS region, such as `us-east-1`.
 * @property publicIpSource The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the public scope. Valid values are `byoip` or `amazon`. Default is `byoip`.
 * @property publiclyAdvertisable Defines whether or not IPv6 pool space is publicly advertisable over the internet. This argument is required if `address_family = "ipv6"` and `public_ip_source = "byoip"`, default is `false`. This option is not available for IPv4 pool space or if `public_ip_source = "amazon"`.
 * @property sourceIpamPoolId The ID of the source IPAM pool. Use this argument to create a child pool within an existing pool.
 * @property tags A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class VpcIpamPoolArgs(
    public val addressFamily: Output<String>? = null,
    public val allocationDefaultNetmaskLength: Output<Int>? = null,
    public val allocationMaxNetmaskLength: Output<Int>? = null,
    public val allocationMinNetmaskLength: Output<Int>? = null,
    public val allocationResourceTags: Output<Map<String, String>>? = null,
    public val autoImport: Output<Boolean>? = null,
    public val awsService: Output<String>? = null,
    public val cascade: Output<Boolean>? = null,
    public val description: Output<String>? = null,
    public val ipamScopeId: Output<String>? = null,
    public val locale: Output<String>? = null,
    public val publicIpSource: Output<String>? = null,
    public val publiclyAdvertisable: Output<Boolean>? = null,
    public val sourceIpamPoolId: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.VpcIpamPoolArgs> {
    override fun toJava(): com.pulumi.aws.ec2.VpcIpamPoolArgs =
        com.pulumi.aws.ec2.VpcIpamPoolArgs.builder()
            .addressFamily(addressFamily?.applyValue({ args0 -> args0 }))
            .allocationDefaultNetmaskLength(allocationDefaultNetmaskLength?.applyValue({ args0 -> args0 }))
            .allocationMaxNetmaskLength(allocationMaxNetmaskLength?.applyValue({ args0 -> args0 }))
            .allocationMinNetmaskLength(allocationMinNetmaskLength?.applyValue({ args0 -> args0 }))
            .allocationResourceTags(
                allocationResourceTags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .autoImport(autoImport?.applyValue({ args0 -> args0 }))
            .awsService(awsService?.applyValue({ args0 -> args0 }))
            .cascade(cascade?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .ipamScopeId(ipamScopeId?.applyValue({ args0 -> args0 }))
            .locale(locale?.applyValue({ args0 -> args0 }))
            .publicIpSource(publicIpSource?.applyValue({ args0 -> args0 }))
            .publiclyAdvertisable(publiclyAdvertisable?.applyValue({ args0 -> args0 }))
            .sourceIpamPoolId(sourceIpamPoolId?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [VpcIpamPoolArgs].
 */
@PulumiTagMarker
public class VpcIpamPoolArgsBuilder internal constructor() {
    private var addressFamily: Output<String>? = null

    private var allocationDefaultNetmaskLength: Output<Int>? = null

    private var allocationMaxNetmaskLength: Output<Int>? = null

    private var allocationMinNetmaskLength: Output<Int>? = null

    private var allocationResourceTags: Output<Map<String, String>>? = null

    private var autoImport: Output<Boolean>? = null

    private var awsService: Output<String>? = null

    private var cascade: Output<Boolean>? = null

    private var description: Output<String>? = null

    private var ipamScopeId: Output<String>? = null

    private var locale: Output<String>? = null

    private var publicIpSource: Output<String>? = null

    private var publiclyAdvertisable: Output<Boolean>? = null

    private var sourceIpamPoolId: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value The IP protocol assigned to this pool. You must choose either IPv4 or IPv6 protocol for a pool.
     */
    @JvmName("eixdfwcspakhvybe")
    public suspend fun addressFamily(`value`: Output<String>) {
        this.addressFamily = value
    }

    /**
     * @param value A default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16 (unless you provide a different netmask value when you create the new allocation).
     */
    @JvmName("wqtmcpbrebrnjlsk")
    public suspend fun allocationDefaultNetmaskLength(`value`: Output<Int>) {
        this.allocationDefaultNetmaskLength = value
    }

    /**
     * @param value The maximum netmask length that will be required for CIDR allocations in this pool.
     */
    @JvmName("tqrxkedenblulehm")
    public suspend fun allocationMaxNetmaskLength(`value`: Output<Int>) {
        this.allocationMaxNetmaskLength = value
    }

    /**
     * @param value The minimum netmask length that will be required for CIDR allocations in this pool.
     */
    @JvmName("ogxrvgdpyciqicvv")
    public suspend fun allocationMinNetmaskLength(`value`: Output<Int>) {
        this.allocationMinNetmaskLength = value
    }

    /**
     * @param value Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.
     */
    @JvmName("yaymynpmvxvmresl")
    public suspend fun allocationResourceTags(`value`: Output<Map<String, String>>) {
        this.allocationResourceTags = value
    }

    /**
     * @param value If you include this argument, IPAM automatically imports any VPCs you have in your scope that fall
     * within the CIDR range in the pool.
     */
    @JvmName("dctppeqyxejqwkng")
    public suspend fun autoImport(`value`: Output<Boolean>) {
        this.autoImport = value
    }

    /**
     * @param value Limits which AWS service the pool can be used in. Only useable on public scopes. Valid Values: `ec2`.
     */
    @JvmName("cffxwhnfqkamhggp")
    public suspend fun awsService(`value`: Output<String>) {
        this.awsService = value
    }

    /**
     * @param value Enables you to quickly delete an IPAM pool and all resources within that pool, including provisioned CIDRs, allocations, and other pools.
     */
    @JvmName("fmrsquydbxuacfrg")
    public suspend fun cascade(`value`: Output<Boolean>) {
        this.cascade = value
    }

    /**
     * @param value A description for the IPAM pool.
     */
    @JvmName("fvjxakviaribrijy")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The ID of the scope in which you would like to create the IPAM pool.
     */
    @JvmName("ncclrabacmiegnhc")
    public suspend fun ipamScopeId(`value`: Output<String>) {
        this.ipamScopeId = value
    }

    /**
     * @param value The locale in which you would like to create the IPAM pool. Locale is the Region where you want to make an IPAM pool available for allocations. You can only create pools with locales that match the operating Regions of the IPAM. You can only create VPCs from a pool whose locale matches the VPC's Region. Possible values: Any AWS region, such as `us-east-1`.
     */
    @JvmName("alnmtoteakwihina")
    public suspend fun locale(`value`: Output<String>) {
        this.locale = value
    }

    /**
     * @param value The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the public scope. Valid values are `byoip` or `amazon`. Default is `byoip`.
     */
    @JvmName("jaatmkxyjexuypgb")
    public suspend fun publicIpSource(`value`: Output<String>) {
        this.publicIpSource = value
    }

    /**
     * @param value Defines whether or not IPv6 pool space is publicly advertisable over the internet. This argument is required if `address_family = "ipv6"` and `public_ip_source = "byoip"`, default is `false`. This option is not available for IPv4 pool space or if `public_ip_source = "amazon"`.
     */
    @JvmName("bcirpywdcanjiboq")
    public suspend fun publiclyAdvertisable(`value`: Output<Boolean>) {
        this.publiclyAdvertisable = value
    }

    /**
     * @param value The ID of the source IPAM pool. Use this argument to create a child pool within an existing pool.
     */
    @JvmName("bafuqvsmlfxuveom")
    public suspend fun sourceIpamPoolId(`value`: Output<String>) {
        this.sourceIpamPoolId = value
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("nihskhptoeowivup")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The IP protocol assigned to this pool. You must choose either IPv4 or IPv6 protocol for a pool.
     */
    @JvmName("jnwjtspvexiumsmp")
    public suspend fun addressFamily(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.addressFamily = mapped
    }

    /**
     * @param value A default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16 (unless you provide a different netmask value when you create the new allocation).
     */
    @JvmName("fjaewmoiksqcivrg")
    public suspend fun allocationDefaultNetmaskLength(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocationDefaultNetmaskLength = mapped
    }

    /**
     * @param value The maximum netmask length that will be required for CIDR allocations in this pool.
     */
    @JvmName("bejlyurihfaiqont")
    public suspend fun allocationMaxNetmaskLength(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocationMaxNetmaskLength = mapped
    }

    /**
     * @param value The minimum netmask length that will be required for CIDR allocations in this pool.
     */
    @JvmName("hipyxiwfgswdevnt")
    public suspend fun allocationMinNetmaskLength(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocationMinNetmaskLength = mapped
    }

    /**
     * @param value Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.
     */
    @JvmName("aporweswaiphrsvp")
    public suspend fun allocationResourceTags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocationResourceTags = mapped
    }

    /**
     * @param values Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.
     */
    @JvmName("qojnspwrqypcynjt")
    public fun allocationResourceTags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.allocationResourceTags = mapped
    }

    /**
     * @param value If you include this argument, IPAM automatically imports any VPCs you have in your scope that fall
     * within the CIDR range in the pool.
     */
    @JvmName("xygyuviccdfdijks")
    public suspend fun autoImport(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoImport = mapped
    }

    /**
     * @param value Limits which AWS service the pool can be used in. Only useable on public scopes. Valid Values: `ec2`.
     */
    @JvmName("oypsyvohkrghnrxj")
    public suspend fun awsService(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.awsService = mapped
    }

    /**
     * @param value Enables you to quickly delete an IPAM pool and all resources within that pool, including provisioned CIDRs, allocations, and other pools.
     */
    @JvmName("mullpkjiyurijkiv")
    public suspend fun cascade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cascade = mapped
    }

    /**
     * @param value A description for the IPAM pool.
     */
    @JvmName("okxhxbajbdhlktgu")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The ID of the scope in which you would like to create the IPAM pool.
     */
    @JvmName("poosjyexhthansdx")
    public suspend fun ipamScopeId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipamScopeId = mapped
    }

    /**
     * @param value The locale in which you would like to create the IPAM pool. Locale is the Region where you want to make an IPAM pool available for allocations. You can only create pools with locales that match the operating Regions of the IPAM. You can only create VPCs from a pool whose locale matches the VPC's Region. Possible values: Any AWS region, such as `us-east-1`.
     */
    @JvmName("iqamropmvvvmyeim")
    public suspend fun locale(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.locale = mapped
    }

    /**
     * @param value The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the public scope. Valid values are `byoip` or `amazon`. Default is `byoip`.
     */
    @JvmName("vlucjevytesadbwu")
    public suspend fun publicIpSource(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.publicIpSource = mapped
    }

    /**
     * @param value Defines whether or not IPv6 pool space is publicly advertisable over the internet. This argument is required if `address_family = "ipv6"` and `public_ip_source = "byoip"`, default is `false`. This option is not available for IPv4 pool space or if `public_ip_source = "amazon"`.
     */
    @JvmName("widbppsvmngfuxqj")
    public suspend fun publiclyAdvertisable(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.publiclyAdvertisable = mapped
    }

    /**
     * @param value The ID of the source IPAM pool. Use this argument to create a child pool within an existing pool.
     */
    @JvmName("drjxvqtkmxchpnlc")
    public suspend fun sourceIpamPoolId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceIpamPoolId = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("togwblmnpqrlohlv")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("elgutrllopgsjhkr")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): VpcIpamPoolArgs = VpcIpamPoolArgs(
        addressFamily = addressFamily,
        allocationDefaultNetmaskLength = allocationDefaultNetmaskLength,
        allocationMaxNetmaskLength = allocationMaxNetmaskLength,
        allocationMinNetmaskLength = allocationMinNetmaskLength,
        allocationResourceTags = allocationResourceTags,
        autoImport = autoImport,
        awsService = awsService,
        cascade = cascade,
        description = description,
        ipamScopeId = ipamScopeId,
        locale = locale,
        publicIpSource = publicIpSource,
        publiclyAdvertisable = publiclyAdvertisable,
        sourceIpamPoolId = sourceIpamPoolId,
        tags = tags,
    )
}
