@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.VpcIpamPoolCidrArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.VpcIpamPoolCidrCidrAuthorizationContextArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpcIpamPoolCidrCidrAuthorizationContextArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * Provisions a CIDR from an IPAM address pool.
 * > **NOTE:** Provisioning Public IPv4 or Public IPv6 require [steps outside the scope of this resource](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#prepare-for-byoip). The resource accepts `message` and `signature` as part of the `cidr_authorization_context` attribute but those must be generated ahead of time. Public IPv6 CIDRs that are provisioned into a Pool with `publicly_advertisable = true` and all public IPv4 CIDRs also require creating a Route Origin Authorization (ROA) object in your Regional Internet Registry (RIR).
 * > **NOTE:** In order to deprovision CIDRs all Allocations must be released. Allocations created by a VPC take up to 30 minutes to be released. However, for IPAM to properly manage the removal of allocation records created by VPCs and other resources, you must [grant it permissions](https://docs.aws.amazon.com/vpc/latest/ipam/choose-single-user-or-orgs-ipam.html) in
 * either a single account or organizationally. If you are unable to deprovision a cidr after waiting over 30 minutes, you may be missing the Service Linked Role.
 * ## Example Usage
 * Basic usage:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getRegion({});
 * const example = new aws.ec2.VpcIpam("example", {operatingRegions: [{
 *     regionName: current.then(current => current.name),
 * }]});
 * const exampleVpcIpamPool = new aws.ec2.VpcIpamPool("example", {
 *     addressFamily: "ipv4",
 *     ipamScopeId: example.privateDefaultScopeId,
 *     locale: current.then(current => current.name),
 * });
 * const exampleVpcIpamPoolCidr = new aws.ec2.VpcIpamPoolCidr("example", {
 *     ipamPoolId: exampleVpcIpamPool.id,
 *     cidr: "172.20.0.0/16",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_region()
 * example = aws.ec2.VpcIpam("example", operating_regions=[{
 *     "region_name": current.name,
 * }])
 * example_vpc_ipam_pool = aws.ec2.VpcIpamPool("example",
 *     address_family="ipv4",
 *     ipam_scope_id=example.private_default_scope_id,
 *     locale=current.name)
 * example_vpc_ipam_pool_cidr = aws.ec2.VpcIpamPoolCidr("example",
 *     ipam_pool_id=example_vpc_ipam_pool.id,
 *     cidr="172.20.0.0/16")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetRegion.Invoke();
 *     var example = new Aws.Ec2.VpcIpam("example", new()
 *     {
 *         OperatingRegions = new[]
 *         {
 *             new Aws.Ec2.Inputs.VpcIpamOperatingRegionArgs
 *             {
 *                 RegionName = current.Apply(getRegionResult => getRegionResult.Name),
 *             },
 *         },
 *     });
 *     var exampleVpcIpamPool = new Aws.Ec2.VpcIpamPool("example", new()
 *     {
 *         AddressFamily = "ipv4",
 *         IpamScopeId = example.PrivateDefaultScopeId,
 *         Locale = current.Apply(getRegionResult => getRegionResult.Name),
 *     });
 *     var exampleVpcIpamPoolCidr = new Aws.Ec2.VpcIpamPoolCidr("example", new()
 *     {
 *         IpamPoolId = exampleVpcIpamPool.Id,
 *         Cidr = "172.20.0.0/16",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := ec2.NewVpcIpam(ctx, "example", &ec2.VpcIpamArgs{
 * 			OperatingRegions: ec2.VpcIpamOperatingRegionArray{
 * 				&ec2.VpcIpamOperatingRegionArgs{
 * 					RegionName: pulumi.String(current.Name),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleVpcIpamPool, err := ec2.NewVpcIpamPool(ctx, "example", &ec2.VpcIpamPoolArgs{
 * 			AddressFamily: pulumi.String("ipv4"),
 * 			IpamScopeId:   example.PrivateDefaultScopeId,
 * 			Locale:        pulumi.String(current.Name),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcIpamPoolCidr(ctx, "example", &ec2.VpcIpamPoolCidrArgs{
 * 			IpamPoolId: exampleVpcIpamPool.ID(),
 * 			Cidr:       pulumi.String("172.20.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpam;
 * import com.pulumi.aws.ec2.VpcIpamArgs;
 * import com.pulumi.aws.ec2.inputs.VpcIpamOperatingRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpamPool;
 * import com.pulumi.aws.ec2.VpcIpamPoolArgs;
 * import com.pulumi.aws.ec2.VpcIpamPoolCidr;
 * import com.pulumi.aws.ec2.VpcIpamPoolCidrArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getRegion();
 *         var example = new VpcIpam("example", VpcIpamArgs.builder()
 *             .operatingRegions(VpcIpamOperatingRegionArgs.builder()
 *                 .regionName(current.applyValue(getRegionResult -> getRegionResult.name()))
 *                 .build())
 *             .build());
 *         var exampleVpcIpamPool = new VpcIpamPool("exampleVpcIpamPool", VpcIpamPoolArgs.builder()
 *             .addressFamily("ipv4")
 *             .ipamScopeId(example.privateDefaultScopeId())
 *             .locale(current.applyValue(getRegionResult -> getRegionResult.name()))
 *             .build());
 *         var exampleVpcIpamPoolCidr = new VpcIpamPoolCidr("exampleVpcIpamPoolCidr", VpcIpamPoolCidrArgs.builder()
 *             .ipamPoolId(exampleVpcIpamPool.id())
 *             .cidr("172.20.0.0/16")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:VpcIpam
 *     properties:
 *       operatingRegions:
 *         - regionName: ${current.name}
 *   exampleVpcIpamPool:
 *     type: aws:ec2:VpcIpamPool
 *     name: example
 *     properties:
 *       addressFamily: ipv4
 *       ipamScopeId: ${example.privateDefaultScopeId}
 *       locale: ${current.name}
 *   exampleVpcIpamPoolCidr:
 *     type: aws:ec2:VpcIpamPoolCidr
 *     name: example
 *     properties:
 *       ipamPoolId: ${exampleVpcIpamPool.id}
 *       cidr: 172.20.0.0/16
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * Provision Public IPv6 Pool CIDRs:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getRegion({});
 * const example = new aws.ec2.VpcIpam("example", {operatingRegions: [{
 *     regionName: current.then(current => current.name),
 * }]});
 * const ipv6TestPublic = new aws.ec2.VpcIpamPool("ipv6_test_public", {
 *     addressFamily: "ipv6",
 *     ipamScopeId: example.publicDefaultScopeId,
 *     locale: "us-east-1",
 *     description: "public ipv6",
 *     publiclyAdvertisable: false,
 *     publicIpSource: "amazon",
 *     awsService: "ec2",
 * });
 * const ipv6TestPublicVpcIpamPoolCidr = new aws.ec2.VpcIpamPoolCidr("ipv6_test_public", {
 *     ipamPoolId: ipv6TestPublic.id,
 *     netmaskLength: 52,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_region()
 * example = aws.ec2.VpcIpam("example", operating_regions=[{
 *     "region_name": current.name,
 * }])
 * ipv6_test_public = aws.ec2.VpcIpamPool("ipv6_test_public",
 *     address_family="ipv6",
 *     ipam_scope_id=example.public_default_scope_id,
 *     locale="us-east-1",
 *     description="public ipv6",
 *     publicly_advertisable=False,
 *     public_ip_source="amazon",
 *     aws_service="ec2")
 * ipv6_test_public_vpc_ipam_pool_cidr = aws.ec2.VpcIpamPoolCidr("ipv6_test_public",
 *     ipam_pool_id=ipv6_test_public.id,
 *     netmask_length=52)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetRegion.Invoke();
 *     var example = new Aws.Ec2.VpcIpam("example", new()
 *     {
 *         OperatingRegions = new[]
 *         {
 *             new Aws.Ec2.Inputs.VpcIpamOperatingRegionArgs
 *             {
 *                 RegionName = current.Apply(getRegionResult => getRegionResult.Name),
 *             },
 *         },
 *     });
 *     var ipv6TestPublic = new Aws.Ec2.VpcIpamPool("ipv6_test_public", new()
 *     {
 *         AddressFamily = "ipv6",
 *         IpamScopeId = example.PublicDefaultScopeId,
 *         Locale = "us-east-1",
 *         Description = "public ipv6",
 *         PubliclyAdvertisable = false,
 *         PublicIpSource = "amazon",
 *         AwsService = "ec2",
 *     });
 *     var ipv6TestPublicVpcIpamPoolCidr = new Aws.Ec2.VpcIpamPoolCidr("ipv6_test_public", new()
 *     {
 *         IpamPoolId = ipv6TestPublic.Id,
 *         NetmaskLength = 52,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := ec2.NewVpcIpam(ctx, "example", &ec2.VpcIpamArgs{
 * 			OperatingRegions: ec2.VpcIpamOperatingRegionArray{
 * 				&ec2.VpcIpamOperatingRegionArgs{
 * 					RegionName: pulumi.String(current.Name),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		ipv6TestPublic, err := ec2.NewVpcIpamPool(ctx, "ipv6_test_public", &ec2.VpcIpamPoolArgs{
 * 			AddressFamily:        pulumi.String("ipv6"),
 * 			IpamScopeId:          example.PublicDefaultScopeId,
 * 			Locale:               pulumi.String("us-east-1"),
 * 			Description:          pulumi.String("public ipv6"),
 * 			PubliclyAdvertisable: pulumi.Bool(false),
 * 			PublicIpSource:       pulumi.String("amazon"),
 * 			AwsService:           pulumi.String("ec2"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcIpamPoolCidr(ctx, "ipv6_test_public", &ec2.VpcIpamPoolCidrArgs{
 * 			IpamPoolId:    ipv6TestPublic.ID(),
 * 			NetmaskLength: pulumi.Int(52),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpam;
 * import com.pulumi.aws.ec2.VpcIpamArgs;
 * import com.pulumi.aws.ec2.inputs.VpcIpamOperatingRegionArgs;
 * import com.pulumi.aws.ec2.VpcIpamPool;
 * import com.pulumi.aws.ec2.VpcIpamPoolArgs;
 * import com.pulumi.aws.ec2.VpcIpamPoolCidr;
 * import com.pulumi.aws.ec2.VpcIpamPoolCidrArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getRegion();
 *         var example = new VpcIpam("example", VpcIpamArgs.builder()
 *             .operatingRegions(VpcIpamOperatingRegionArgs.builder()
 *                 .regionName(current.applyValue(getRegionResult -> getRegionResult.name()))
 *                 .build())
 *             .build());
 *         var ipv6TestPublic = new VpcIpamPool("ipv6TestPublic", VpcIpamPoolArgs.builder()
 *             .addressFamily("ipv6")
 *             .ipamScopeId(example.publicDefaultScopeId())
 *             .locale("us-east-1")
 *             .description("public ipv6")
 *             .publiclyAdvertisable(false)
 *             .publicIpSource("amazon")
 *             .awsService("ec2")
 *             .build());
 *         var ipv6TestPublicVpcIpamPoolCidr = new VpcIpamPoolCidr("ipv6TestPublicVpcIpamPoolCidr", VpcIpamPoolCidrArgs.builder()
 *             .ipamPoolId(ipv6TestPublic.id())
 *             .netmaskLength(52)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:VpcIpam
 *     properties:
 *       operatingRegions:
 *         - regionName: ${current.name}
 *   ipv6TestPublic:
 *     type: aws:ec2:VpcIpamPool
 *     name: ipv6_test_public
 *     properties:
 *       addressFamily: ipv6
 *       ipamScopeId: ${example.publicDefaultScopeId}
 *       locale: us-east-1
 *       description: public ipv6
 *       publiclyAdvertisable: false
 *       publicIpSource: amazon
 *       awsService: ec2
 *   ipv6TestPublicVpcIpamPoolCidr:
 *     type: aws:ec2:VpcIpamPoolCidr
 *     name: ipv6_test_public
 *     properties:
 *       ipamPoolId: ${ipv6TestPublic.id}
 *       netmaskLength: 52
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import IPAMs using the `<cidr>_<ipam-pool-id>`. For example:
 * __NOTE:__ Do not use the IPAM Pool Cidr ID as this was introduced after the resource already existed.
 * ```sh
 * $ pulumi import aws:ec2/vpcIpamPoolCidr:VpcIpamPoolCidr example 172.20.0.0/24_ipam-pool-0e634f5a1517cccdc
 * ```
 * @property cidr The CIDR you want to assign to the pool. Conflicts with `netmask_length`.
 * @property cidrAuthorizationContext A signed document that proves that you are authorized to bring the specified IP address range to Amazon using BYOIP. This is not stored in the state file. See cidr_authorization_context for more information.
 * @property ipamPoolId The ID of the pool to which you want to assign a CIDR.
 * @property netmaskLength If provided, the cidr provisioned into the specified pool will be the next available cidr given this declared netmask length. Conflicts with `cidr`.
 */
public data class VpcIpamPoolCidrArgs(
    public val cidr: Output<String>? = null,
    public val cidrAuthorizationContext: Output<VpcIpamPoolCidrCidrAuthorizationContextArgs>? = null,
    public val ipamPoolId: Output<String>? = null,
    public val netmaskLength: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.VpcIpamPoolCidrArgs> {
    override fun toJava(): com.pulumi.aws.ec2.VpcIpamPoolCidrArgs =
        com.pulumi.aws.ec2.VpcIpamPoolCidrArgs.builder()
            .cidr(cidr?.applyValue({ args0 -> args0 }))
            .cidrAuthorizationContext(
                cidrAuthorizationContext?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .ipamPoolId(ipamPoolId?.applyValue({ args0 -> args0 }))
            .netmaskLength(netmaskLength?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [VpcIpamPoolCidrArgs].
 */
@PulumiTagMarker
public class VpcIpamPoolCidrArgsBuilder internal constructor() {
    private var cidr: Output<String>? = null

    private var cidrAuthorizationContext: Output<VpcIpamPoolCidrCidrAuthorizationContextArgs>? = null

    private var ipamPoolId: Output<String>? = null

    private var netmaskLength: Output<Int>? = null

    /**
     * @param value The CIDR you want to assign to the pool. Conflicts with `netmask_length`.
     */
    @JvmName("ahovmyhiuafvojbt")
    public suspend fun cidr(`value`: Output<String>) {
        this.cidr = value
    }

    /**
     * @param value A signed document that proves that you are authorized to bring the specified IP address range to Amazon using BYOIP. This is not stored in the state file. See cidr_authorization_context for more information.
     */
    @JvmName("eiavwoyglanvxkyq")
    public suspend fun cidrAuthorizationContext(`value`: Output<VpcIpamPoolCidrCidrAuthorizationContextArgs>) {
        this.cidrAuthorizationContext = value
    }

    /**
     * @param value The ID of the pool to which you want to assign a CIDR.
     */
    @JvmName("wtoxeynlmfwxksps")
    public suspend fun ipamPoolId(`value`: Output<String>) {
        this.ipamPoolId = value
    }

    /**
     * @param value If provided, the cidr provisioned into the specified pool will be the next available cidr given this declared netmask length. Conflicts with `cidr`.
     */
    @JvmName("hxlrcyxcggflmdji")
    public suspend fun netmaskLength(`value`: Output<Int>) {
        this.netmaskLength = value
    }

    /**
     * @param value The CIDR you want to assign to the pool. Conflicts with `netmask_length`.
     */
    @JvmName("cudcxwhbljosfpdp")
    public suspend fun cidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cidr = mapped
    }

    /**
     * @param value A signed document that proves that you are authorized to bring the specified IP address range to Amazon using BYOIP. This is not stored in the state file. See cidr_authorization_context for more information.
     */
    @JvmName("sdtfhqnofnlucthm")
    public suspend fun cidrAuthorizationContext(`value`: VpcIpamPoolCidrCidrAuthorizationContextArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cidrAuthorizationContext = mapped
    }

    /**
     * @param argument A signed document that proves that you are authorized to bring the specified IP address range to Amazon using BYOIP. This is not stored in the state file. See cidr_authorization_context for more information.
     */
    @JvmName("uhuyneicaegocqje")
    public suspend fun cidrAuthorizationContext(argument: suspend VpcIpamPoolCidrCidrAuthorizationContextArgsBuilder.() -> Unit) {
        val toBeMapped = VpcIpamPoolCidrCidrAuthorizationContextArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.cidrAuthorizationContext = mapped
    }

    /**
     * @param value The ID of the pool to which you want to assign a CIDR.
     */
    @JvmName("upmqgucuhspaasov")
    public suspend fun ipamPoolId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipamPoolId = mapped
    }

    /**
     * @param value If provided, the cidr provisioned into the specified pool will be the next available cidr given this declared netmask length. Conflicts with `cidr`.
     */
    @JvmName("qkylrsqxtaofssfk")
    public suspend fun netmaskLength(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.netmaskLength = mapped
    }

    internal fun build(): VpcIpamPoolCidrArgs = VpcIpamPoolCidrArgs(
        cidr = cidr,
        cidrAuthorizationContext = cidrAuthorizationContext,
        ipamPoolId = ipamPoolId,
        netmaskLength = netmaskLength,
    )
}
