@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.VpcPeeringConnectionAccepterArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionAccepterAccepterArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionAccepterAccepterArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionAccepterRequesterArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionAccepterRequesterArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a resource to manage the accepter's side of a VPC Peering Connection.
 * When a cross-account (requester's AWS account differs from the accepter's AWS account) or an inter-region
 * VPC Peering Connection is created, a VPC Peering Connection resource is automatically created in the
 * accepter's account.
 * The requester can use the `aws.ec2.VpcPeeringConnection` resource to manage its side of the connection
 * and the accepter can use the `aws.ec2.VpcPeeringConnectionAccepter` resource to "adopt" its side of the
 * connection into management.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const main = new aws.ec2.Vpc("main", {cidrBlock: "10.0.0.0/16"});
 * const peerVpc = new aws.ec2.Vpc("peer", {cidrBlock: "10.1.0.0/16"});
 * const peer = aws.getCallerIdentity({});
 * // Requester's side of the connection.
 * const peerVpcPeeringConnection = new aws.ec2.VpcPeeringConnection("peer", {
 *     vpcId: main.id,
 *     peerVpcId: peerVpc.id,
 *     peerOwnerId: peer.then(peer => peer.accountId),
 *     peerRegion: "us-west-2",
 *     autoAccept: false,
 *     tags: {
 *         Side: "Requester",
 *     },
 * });
 * // Accepter's side of the connection.
 * const peerVpcPeeringConnectionAccepter = new aws.ec2.VpcPeeringConnectionAccepter("peer", {
 *     vpcPeeringConnectionId: peerVpcPeeringConnection.id,
 *     autoAccept: true,
 *     tags: {
 *         Side: "Accepter",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * main = aws.ec2.Vpc("main", cidr_block="10.0.0.0/16")
 * peer_vpc = aws.ec2.Vpc("peer", cidr_block="10.1.0.0/16")
 * peer = aws.get_caller_identity()
 * # Requester's side of the connection.
 * peer_vpc_peering_connection = aws.ec2.VpcPeeringConnection("peer",
 *     vpc_id=main.id,
 *     peer_vpc_id=peer_vpc.id,
 *     peer_owner_id=peer.account_id,
 *     peer_region="us-west-2",
 *     auto_accept=False,
 *     tags={
 *         "Side": "Requester",
 *     })
 * # Accepter's side of the connection.
 * peer_vpc_peering_connection_accepter = aws.ec2.VpcPeeringConnectionAccepter("peer",
 *     vpc_peering_connection_id=peer_vpc_peering_connection.id,
 *     auto_accept=True,
 *     tags={
 *         "Side": "Accepter",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var main = new Aws.Ec2.Vpc("main", new()
 *     {
 *         CidrBlock = "10.0.0.0/16",
 *     });
 *     var peerVpc = new Aws.Ec2.Vpc("peer", new()
 *     {
 *         CidrBlock = "10.1.0.0/16",
 *     });
 *     var peer = Aws.GetCallerIdentity.Invoke();
 *     // Requester's side of the connection.
 *     var peerVpcPeeringConnection = new Aws.Ec2.VpcPeeringConnection("peer", new()
 *     {
 *         VpcId = main.Id,
 *         PeerVpcId = peerVpc.Id,
 *         PeerOwnerId = peer.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId),
 *         PeerRegion = "us-west-2",
 *         AutoAccept = false,
 *         Tags =
 *         {
 *             { "Side", "Requester" },
 *         },
 *     });
 *     // Accepter's side of the connection.
 *     var peerVpcPeeringConnectionAccepter = new Aws.Ec2.VpcPeeringConnectionAccepter("peer", new()
 *     {
 *         VpcPeeringConnectionId = peerVpcPeeringConnection.Id,
 *         AutoAccept = true,
 *         Tags =
 *         {
 *             { "Side", "Accepter" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		main, err := ec2.NewVpc(ctx, "main", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.0.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		peerVpc, err := ec2.NewVpc(ctx, "peer", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.1.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		peer, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Requester's side of the connection.
 * 		peerVpcPeeringConnection, err := ec2.NewVpcPeeringConnection(ctx, "peer", &ec2.VpcPeeringConnectionArgs{
 * 			VpcId:       main.ID(),
 * 			PeerVpcId:   peerVpc.ID(),
 * 			PeerOwnerId: pulumi.String(peer.AccountId),
 * 			PeerRegion:  pulumi.String("us-west-2"),
 * 			AutoAccept:  pulumi.Bool(false),
 * 			Tags: pulumi.StringMap{
 * 				"Side": pulumi.String("Requester"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Accepter's side of the connection.
 * 		_, err = ec2.NewVpcPeeringConnectionAccepter(ctx, "peer", &ec2.VpcPeeringConnectionAccepterArgs{
 * 			VpcPeeringConnectionId: peerVpcPeeringConnection.ID(),
 * 			AutoAccept:             pulumi.Bool(true),
 * 			Tags: pulumi.StringMap{
 * 				"Side": pulumi.String("Accepter"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.ec2.VpcPeeringConnection;
 * import com.pulumi.aws.ec2.VpcPeeringConnectionArgs;
 * import com.pulumi.aws.ec2.VpcPeeringConnectionAccepter;
 * import com.pulumi.aws.ec2.VpcPeeringConnectionAccepterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var main = new Vpc("main", VpcArgs.builder()
 *             .cidrBlock("10.0.0.0/16")
 *             .build());
 *         var peerVpc = new Vpc("peerVpc", VpcArgs.builder()
 *             .cidrBlock("10.1.0.0/16")
 *             .build());
 *         final var peer = AwsFunctions.getCallerIdentity();
 *         // Requester's side of the connection.
 *         var peerVpcPeeringConnection = new VpcPeeringConnection("peerVpcPeeringConnection", VpcPeeringConnectionArgs.builder()
 *             .vpcId(main.id())
 *             .peerVpcId(peerVpc.id())
 *             .peerOwnerId(peer.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()))
 *             .peerRegion("us-west-2")
 *             .autoAccept(false)
 *             .tags(Map.of("Side", "Requester"))
 *             .build());
 *         // Accepter's side of the connection.
 *         var peerVpcPeeringConnectionAccepter = new VpcPeeringConnectionAccepter("peerVpcPeeringConnectionAccepter", VpcPeeringConnectionAccepterArgs.builder()
 *             .vpcPeeringConnectionId(peerVpcPeeringConnection.id())
 *             .autoAccept(true)
 *             .tags(Map.of("Side", "Accepter"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   main:
 *     type: aws:ec2:Vpc
 *     properties:
 *       cidrBlock: 10.0.0.0/16
 *   peerVpc:
 *     type: aws:ec2:Vpc
 *     name: peer
 *     properties:
 *       cidrBlock: 10.1.0.0/16
 *   # Requester's side of the connection.
 *   peerVpcPeeringConnection:
 *     type: aws:ec2:VpcPeeringConnection
 *     name: peer
 *     properties:
 *       vpcId: ${main.id}
 *       peerVpcId: ${peerVpc.id}
 *       peerOwnerId: ${peer.accountId}
 *       peerRegion: us-west-2
 *       autoAccept: false
 *       tags:
 *         Side: Requester
 *   # Accepter's side of the connection.
 *   peerVpcPeeringConnectionAccepter:
 *     type: aws:ec2:VpcPeeringConnectionAccepter
 *     name: peer
 *     properties:
 *       vpcPeeringConnectionId: ${peerVpcPeeringConnection.id}
 *       autoAccept: true
 *       tags:
 *         Side: Accepter
 * variables:
 *   peer:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import VPC Peering Connection Accepters using the Peering Connection ID. For example:
 * ```sh
 * $ pulumi import aws:ec2/vpcPeeringConnectionAccepter:VpcPeeringConnectionAccepter example pcx-12345678
 * ```
 * Certain resource arguments, like `auto_accept`, do not have an EC2 API method for reading the information after peering connection creation. If the argument is set in the Pulumi program on an imported resource, Pulumi will always show a difference. To workaround this behavior, either omit the argument from the Pulumi program or use `ignore_changes` to hide the difference. For example:
 * @property accepter A configuration block that describes [VPC Peering Connection]
 * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the accepter VPC.
 * @property autoAccept Whether or not to accept the peering request. Defaults to `false`.
 * @property requester A configuration block that describes [VPC Peering Connection]
 * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the requester VPC.
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcPeeringConnectionId The VPC Peering Connection ID to manage.
 */
public data class VpcPeeringConnectionAccepterArgs(
    public val accepter: Output<VpcPeeringConnectionAccepterAccepterArgs>? = null,
    public val autoAccept: Output<Boolean>? = null,
    public val requester: Output<VpcPeeringConnectionAccepterRequesterArgs>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcPeeringConnectionId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.VpcPeeringConnectionAccepterArgs> {
    override fun toJava(): com.pulumi.aws.ec2.VpcPeeringConnectionAccepterArgs =
        com.pulumi.aws.ec2.VpcPeeringConnectionAccepterArgs.builder()
            .accepter(accepter?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .autoAccept(autoAccept?.applyValue({ args0 -> args0 }))
            .requester(requester?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .vpcPeeringConnectionId(vpcPeeringConnectionId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [VpcPeeringConnectionAccepterArgs].
 */
@PulumiTagMarker
public class VpcPeeringConnectionAccepterArgsBuilder internal constructor() {
    private var accepter: Output<VpcPeeringConnectionAccepterAccepterArgs>? = null

    private var autoAccept: Output<Boolean>? = null

    private var requester: Output<VpcPeeringConnectionAccepterRequesterArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcPeeringConnectionId: Output<String>? = null

    /**
     * @param value A configuration block that describes [VPC Peering Connection]
     * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the accepter VPC.
     */
    @JvmName("ntdmgxqimjifvdph")
    public suspend fun accepter(`value`: Output<VpcPeeringConnectionAccepterAccepterArgs>) {
        this.accepter = value
    }

    /**
     * @param value Whether or not to accept the peering request. Defaults to `false`.
     */
    @JvmName("kvsqqqblqtxuckuv")
    public suspend fun autoAccept(`value`: Output<Boolean>) {
        this.autoAccept = value
    }

    /**
     * @param value A configuration block that describes [VPC Peering Connection]
     * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the requester VPC.
     */
    @JvmName("kdvxiwyqhpdccixj")
    public suspend fun requester(`value`: Output<VpcPeeringConnectionAccepterRequesterArgs>) {
        this.requester = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("xgywabmlpfhpnupy")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The VPC Peering Connection ID to manage.
     */
    @JvmName("nkmdnbiwwovmcamw")
    public suspend fun vpcPeeringConnectionId(`value`: Output<String>) {
        this.vpcPeeringConnectionId = value
    }

    /**
     * @param value A configuration block that describes [VPC Peering Connection]
     * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the accepter VPC.
     */
    @JvmName("bkbeflhjdyyioyfj")
    public suspend fun accepter(`value`: VpcPeeringConnectionAccepterAccepterArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accepter = mapped
    }

    /**
     * @param argument A configuration block that describes [VPC Peering Connection]
     * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the accepter VPC.
     */
    @JvmName("mtimgltkkuwnmwto")
    public suspend fun accepter(argument: suspend VpcPeeringConnectionAccepterAccepterArgsBuilder.() -> Unit) {
        val toBeMapped = VpcPeeringConnectionAccepterAccepterArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.accepter = mapped
    }

    /**
     * @param value Whether or not to accept the peering request. Defaults to `false`.
     */
    @JvmName("avcaubdjaqsnywgu")
    public suspend fun autoAccept(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoAccept = mapped
    }

    /**
     * @param value A configuration block that describes [VPC Peering Connection]
     * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the requester VPC.
     */
    @JvmName("udbugsntwnwdquhx")
    public suspend fun requester(`value`: VpcPeeringConnectionAccepterRequesterArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requester = mapped
    }

    /**
     * @param argument A configuration block that describes [VPC Peering Connection]
     * (https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options set for the requester VPC.
     */
    @JvmName("mqegqbhbolmmnltb")
    public suspend fun requester(argument: suspend VpcPeeringConnectionAccepterRequesterArgsBuilder.() -> Unit) {
        val toBeMapped = VpcPeeringConnectionAccepterRequesterArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.requester = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jxxrnupngpclocfm")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("kcolcfbgybuatmgi")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The VPC Peering Connection ID to manage.
     */
    @JvmName("neekjdqclyfghmof")
    public suspend fun vpcPeeringConnectionId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcPeeringConnectionId = mapped
    }

    internal fun build(): VpcPeeringConnectionAccepterArgs = VpcPeeringConnectionAccepterArgs(
        accepter = accepter,
        autoAccept = autoAccept,
        requester = requester,
        tags = tags,
        vpcPeeringConnectionId = vpcPeeringConnectionId,
    )
}
