@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.VpcPeeringConnectionArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionAccepterArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionAccepterArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionRequesterArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpcPeeringConnectionRequesterArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a resource to manage a VPC peering connection.
 * > **NOTE on VPC Peering Connections and VPC Peering Connection Options:** This provider provides
 * both a standalone VPC Peering Connection Options and a VPC Peering Connection
 * resource with `accepter` and `requester` attributes. Do not manage options for the same VPC peering
 * connection in both a VPC Peering Connection resource and a VPC Peering Connection Options resource.
 * Doing so will cause a conflict of options and will overwrite the options.
 * Using a VPC Peering Connection Options resource decouples management of the connection options from
 * management of the VPC Peering Connection and allows options to be set correctly in cross-account scenarios.
 * > **Note:** For cross-account (requester's AWS account differs from the accepter's AWS account) or inter-region
 * VPC Peering Connections use the `aws.ec2.VpcPeeringConnection` resource to manage the requester's side of the
 * connection and use the `aws.ec2.VpcPeeringConnectionAccepter` resource to manage the accepter's side of the connection.
 * > **Note:** Creating multiple `aws.ec2.VpcPeeringConnection` resources with the same `peer_vpc_id` and `vpc_id` will not produce an error. Instead, AWS will return the connection `id` that already exists, resulting in multiple `aws.ec2.VpcPeeringConnection` resources with the same `id`.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const foo = new aws.ec2.VpcPeeringConnection("foo", {
 *     peerOwnerId: peerOwnerId,
 *     peerVpcId: bar.id,
 *     vpcId: fooAwsVpc.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foo = aws.ec2.VpcPeeringConnection("foo",
 *     peer_owner_id=peer_owner_id,
 *     peer_vpc_id=bar["id"],
 *     vpc_id=foo_aws_vpc["id"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var foo = new Aws.Ec2.VpcPeeringConnection("foo", new()
 *     {
 *         PeerOwnerId = peerOwnerId,
 *         PeerVpcId = bar.Id,
 *         VpcId = fooAwsVpc.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewVpcPeeringConnection(ctx, "foo", &ec2.VpcPeeringConnectionArgs{
 * 			PeerOwnerId: pulumi.Any(peerOwnerId),
 * 			PeerVpcId:   pulumi.Any(bar.Id),
 * 			VpcId:       pulumi.Any(fooAwsVpc.Id),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcPeeringConnection;
 * import com.pulumi.aws.ec2.VpcPeeringConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var foo = new VpcPeeringConnection("foo", VpcPeeringConnectionArgs.builder()
 *             .peerOwnerId(peerOwnerId)
 *             .peerVpcId(bar.id())
 *             .vpcId(fooAwsVpc.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:ec2:VpcPeeringConnection
 *     properties:
 *       peerOwnerId: ${peerOwnerId}
 *       peerVpcId: ${bar.id}
 *       vpcId: ${fooAwsVpc.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * Basic usage with connection options:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const foo = new aws.ec2.VpcPeeringConnection("foo", {
 *     peerOwnerId: peerOwnerId,
 *     peerVpcId: bar.id,
 *     vpcId: fooAwsVpc.id,
 *     accepter: {
 *         allowRemoteVpcDnsResolution: true,
 *     },
 *     requester: {
 *         allowRemoteVpcDnsResolution: true,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foo = aws.ec2.VpcPeeringConnection("foo",
 *     peer_owner_id=peer_owner_id,
 *     peer_vpc_id=bar["id"],
 *     vpc_id=foo_aws_vpc["id"],
 *     accepter={
 *         "allow_remote_vpc_dns_resolution": True,
 *     },
 *     requester={
 *         "allow_remote_vpc_dns_resolution": True,
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var foo = new Aws.Ec2.VpcPeeringConnection("foo", new()
 *     {
 *         PeerOwnerId = peerOwnerId,
 *         PeerVpcId = bar.Id,
 *         VpcId = fooAwsVpc.Id,
 *         Accepter = new Aws.Ec2.Inputs.VpcPeeringConnectionAccepterArgs
 *         {
 *             AllowRemoteVpcDnsResolution = true,
 *         },
 *         Requester = new Aws.Ec2.Inputs.VpcPeeringConnectionRequesterArgs
 *         {
 *             AllowRemoteVpcDnsResolution = true,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := ec2.NewVpcPeeringConnection(ctx, "foo", &ec2.VpcPeeringConnectionArgs{
 * 			PeerOwnerId: pulumi.Any(peerOwnerId),
 * 			PeerVpcId:   pulumi.Any(bar.Id),
 * 			VpcId:       pulumi.Any(fooAwsVpc.Id),
 * 			Accepter: &ec2.VpcPeeringConnectionAccepterTypeArgs{
 * 				AllowRemoteVpcDnsResolution: pulumi.Bool(true),
 * 			},
 * 			Requester: &ec2.VpcPeeringConnectionRequesterArgs{
 * 				AllowRemoteVpcDnsResolution: pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.VpcPeeringConnection;
 * import com.pulumi.aws.ec2.VpcPeeringConnectionArgs;
 * import com.pulumi.aws.ec2.inputs.VpcPeeringConnectionAccepterArgs;
 * import com.pulumi.aws.ec2.inputs.VpcPeeringConnectionRequesterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var foo = new VpcPeeringConnection("foo", VpcPeeringConnectionArgs.builder()
 *             .peerOwnerId(peerOwnerId)
 *             .peerVpcId(bar.id())
 *             .vpcId(fooAwsVpc.id())
 *             .accepter(VpcPeeringConnectionAccepterArgs.builder()
 *                 .allowRemoteVpcDnsResolution(true)
 *                 .build())
 *             .requester(VpcPeeringConnectionRequesterArgs.builder()
 *                 .allowRemoteVpcDnsResolution(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:ec2:VpcPeeringConnection
 *     properties:
 *       peerOwnerId: ${peerOwnerId}
 *       peerVpcId: ${bar.id}
 *       vpcId: ${fooAwsVpc.id}
 *       accepter:
 *         allowRemoteVpcDnsResolution: true
 *       requester:
 *         allowRemoteVpcDnsResolution: true
 * ```
 * <!--End PulumiCodeChooser -->
 * Basic usage with tags:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const fooVpc = new aws.ec2.Vpc("foo", {cidrBlock: "10.1.0.0/16"});
 * const bar = new aws.ec2.Vpc("bar", {cidrBlock: "10.2.0.0/16"});
 * const foo = new aws.ec2.VpcPeeringConnection("foo", {
 *     peerOwnerId: peerOwnerId,
 *     peerVpcId: bar.id,
 *     vpcId: fooVpc.id,
 *     autoAccept: true,
 *     tags: {
 *         Name: "VPC Peering between foo and bar",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foo_vpc = aws.ec2.Vpc("foo", cidr_block="10.1.0.0/16")
 * bar = aws.ec2.Vpc("bar", cidr_block="10.2.0.0/16")
 * foo = aws.ec2.VpcPeeringConnection("foo",
 *     peer_owner_id=peer_owner_id,
 *     peer_vpc_id=bar.id,
 *     vpc_id=foo_vpc.id,
 *     auto_accept=True,
 *     tags={
 *         "Name": "VPC Peering between foo and bar",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var fooVpc = new Aws.Ec2.Vpc("foo", new()
 *     {
 *         CidrBlock = "10.1.0.0/16",
 *     });
 *     var bar = new Aws.Ec2.Vpc("bar", new()
 *     {
 *         CidrBlock = "10.2.0.0/16",
 *     });
 *     var foo = new Aws.Ec2.VpcPeeringConnection("foo", new()
 *     {
 *         PeerOwnerId = peerOwnerId,
 *         PeerVpcId = bar.Id,
 *         VpcId = fooVpc.Id,
 *         AutoAccept = true,
 *         Tags =
 *         {
 *             { "Name", "VPC Peering between foo and bar" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		fooVpc, err := ec2.NewVpc(ctx, "foo", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.1.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		bar, err := ec2.NewVpc(ctx, "bar", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.2.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcPeeringConnection(ctx, "foo", &ec2.VpcPeeringConnectionArgs{
 * 			PeerOwnerId: pulumi.Any(peerOwnerId),
 * 			PeerVpcId:   bar.ID(),
 * 			VpcId:       fooVpc.ID(),
 * 			AutoAccept:  pulumi.Bool(true),
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("VPC Peering between foo and bar"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.ec2.VpcPeeringConnection;
 * import com.pulumi.aws.ec2.VpcPeeringConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var fooVpc = new Vpc("fooVpc", VpcArgs.builder()
 *             .cidrBlock("10.1.0.0/16")
 *             .build());
 *         var bar = new Vpc("bar", VpcArgs.builder()
 *             .cidrBlock("10.2.0.0/16")
 *             .build());
 *         var foo = new VpcPeeringConnection("foo", VpcPeeringConnectionArgs.builder()
 *             .peerOwnerId(peerOwnerId)
 *             .peerVpcId(bar.id())
 *             .vpcId(fooVpc.id())
 *             .autoAccept(true)
 *             .tags(Map.of("Name", "VPC Peering between foo and bar"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:ec2:VpcPeeringConnection
 *     properties:
 *       peerOwnerId: ${peerOwnerId}
 *       peerVpcId: ${bar.id}
 *       vpcId: ${fooVpc.id}
 *       autoAccept: true
 *       tags:
 *         Name: VPC Peering between foo and bar
 *   fooVpc:
 *     type: aws:ec2:Vpc
 *     name: foo
 *     properties:
 *       cidrBlock: 10.1.0.0/16
 *   bar:
 *     type: aws:ec2:Vpc
 *     properties:
 *       cidrBlock: 10.2.0.0/16
 * ```
 * <!--End PulumiCodeChooser -->
 * Basic usage with region:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const fooVpc = new aws.ec2.Vpc("foo", {cidrBlock: "10.1.0.0/16"});
 * const bar = new aws.ec2.Vpc("bar", {cidrBlock: "10.2.0.0/16"});
 * const foo = new aws.ec2.VpcPeeringConnection("foo", {
 *     peerOwnerId: peerOwnerId,
 *     peerVpcId: bar.id,
 *     vpcId: fooVpc.id,
 *     peerRegion: "us-east-1",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * foo_vpc = aws.ec2.Vpc("foo", cidr_block="10.1.0.0/16")
 * bar = aws.ec2.Vpc("bar", cidr_block="10.2.0.0/16")
 * foo = aws.ec2.VpcPeeringConnection("foo",
 *     peer_owner_id=peer_owner_id,
 *     peer_vpc_id=bar.id,
 *     vpc_id=foo_vpc.id,
 *     peer_region="us-east-1")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var fooVpc = new Aws.Ec2.Vpc("foo", new()
 *     {
 *         CidrBlock = "10.1.0.0/16",
 *     });
 *     var bar = new Aws.Ec2.Vpc("bar", new()
 *     {
 *         CidrBlock = "10.2.0.0/16",
 *     });
 *     var foo = new Aws.Ec2.VpcPeeringConnection("foo", new()
 *     {
 *         PeerOwnerId = peerOwnerId,
 *         PeerVpcId = bar.Id,
 *         VpcId = fooVpc.Id,
 *         PeerRegion = "us-east-1",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		fooVpc, err := ec2.NewVpc(ctx, "foo", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.1.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		bar, err := ec2.NewVpc(ctx, "bar", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.2.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpcPeeringConnection(ctx, "foo", &ec2.VpcPeeringConnectionArgs{
 * 			PeerOwnerId: pulumi.Any(peerOwnerId),
 * 			PeerVpcId:   bar.ID(),
 * 			VpcId:       fooVpc.ID(),
 * 			PeerRegion:  pulumi.String("us-east-1"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.ec2.VpcPeeringConnection;
 * import com.pulumi.aws.ec2.VpcPeeringConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var fooVpc = new Vpc("fooVpc", VpcArgs.builder()
 *             .cidrBlock("10.1.0.0/16")
 *             .build());
 *         var bar = new Vpc("bar", VpcArgs.builder()
 *             .cidrBlock("10.2.0.0/16")
 *             .build());
 *         var foo = new VpcPeeringConnection("foo", VpcPeeringConnectionArgs.builder()
 *             .peerOwnerId(peerOwnerId)
 *             .peerVpcId(bar.id())
 *             .vpcId(fooVpc.id())
 *             .peerRegion("us-east-1")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:ec2:VpcPeeringConnection
 *     properties:
 *       peerOwnerId: ${peerOwnerId}
 *       peerVpcId: ${bar.id}
 *       vpcId: ${fooVpc.id}
 *       peerRegion: us-east-1
 *   fooVpc:
 *     type: aws:ec2:Vpc
 *     name: foo
 *     properties:
 *       cidrBlock: 10.1.0.0/16
 *   bar:
 *     type: aws:ec2:Vpc
 *     properties:
 *       cidrBlock: 10.2.0.0/16
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Notes
 * If both VPCs are not in the same AWS account and region do not enable the `auto_accept` attribute.
 * The accepter can manage its side of the connection using the `aws.ec2.VpcPeeringConnectionAccepter` resource
 * or accept the connection manually using the AWS Management Console, AWS CLI, through SDKs, etc.
 * ## Import
 * Using `pulumi import`, import VPC Peering resources using the VPC peering `id`. For example:
 * ```sh
 * $ pulumi import aws:ec2/vpcPeeringConnection:VpcPeeringConnection test_connection pcx-111aaa111
 * ```
 * @property accepter An optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that accepts
 * the peering connection (a maximum of one).
 * @property autoAccept Accept the peering (both VPCs need to be in the same AWS account and region).
 * @property peerOwnerId The AWS account ID of the target peer VPC.
 * Defaults to the account ID the [AWS provider][1] is currently connected to, so must be managed if connecting cross-account.
 * @property peerRegion The region of the accepter VPC of the VPC Peering Connection. `auto_accept` must be `false`,
 * and use the `aws.ec2.VpcPeeringConnectionAccepter` to manage the accepter side.
 * @property peerVpcId The ID of the target VPC with which you are creating the VPC Peering Connection.
 * @property requester A optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that requests
 * the peering connection (a maximum of one).
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcId The ID of the requester VPC.
 */
public data class VpcPeeringConnectionArgs(
    public val accepter: Output<VpcPeeringConnectionAccepterArgs>? = null,
    public val autoAccept: Output<Boolean>? = null,
    public val peerOwnerId: Output<String>? = null,
    public val peerRegion: Output<String>? = null,
    public val peerVpcId: Output<String>? = null,
    public val requester: Output<VpcPeeringConnectionRequesterArgs>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.VpcPeeringConnectionArgs> {
    override fun toJava(): com.pulumi.aws.ec2.VpcPeeringConnectionArgs =
        com.pulumi.aws.ec2.VpcPeeringConnectionArgs.builder()
            .accepter(accepter?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .autoAccept(autoAccept?.applyValue({ args0 -> args0 }))
            .peerOwnerId(peerOwnerId?.applyValue({ args0 -> args0 }))
            .peerRegion(peerRegion?.applyValue({ args0 -> args0 }))
            .peerVpcId(peerVpcId?.applyValue({ args0 -> args0 }))
            .requester(requester?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .vpcId(vpcId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [VpcPeeringConnectionArgs].
 */
@PulumiTagMarker
public class VpcPeeringConnectionArgsBuilder internal constructor() {
    private var accepter: Output<VpcPeeringConnectionAccepterArgs>? = null

    private var autoAccept: Output<Boolean>? = null

    private var peerOwnerId: Output<String>? = null

    private var peerRegion: Output<String>? = null

    private var peerVpcId: Output<String>? = null

    private var requester: Output<VpcPeeringConnectionRequesterArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcId: Output<String>? = null

    /**
     * @param value An optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that accepts
     * the peering connection (a maximum of one).
     */
    @JvmName("otviimvsuvkycdng")
    public suspend fun accepter(`value`: Output<VpcPeeringConnectionAccepterArgs>) {
        this.accepter = value
    }

    /**
     * @param value Accept the peering (both VPCs need to be in the same AWS account and region).
     */
    @JvmName("sxqjamnjdyedweat")
    public suspend fun autoAccept(`value`: Output<Boolean>) {
        this.autoAccept = value
    }

    /**
     * @param value The AWS account ID of the target peer VPC.
     * Defaults to the account ID the [AWS provider][1] is currently connected to, so must be managed if connecting cross-account.
     */
    @JvmName("vllfhwgrauxvdhej")
    public suspend fun peerOwnerId(`value`: Output<String>) {
        this.peerOwnerId = value
    }

    /**
     * @param value The region of the accepter VPC of the VPC Peering Connection. `auto_accept` must be `false`,
     * and use the `aws.ec2.VpcPeeringConnectionAccepter` to manage the accepter side.
     */
    @JvmName("tkumajghsjtkysji")
    public suspend fun peerRegion(`value`: Output<String>) {
        this.peerRegion = value
    }

    /**
     * @param value The ID of the target VPC with which you are creating the VPC Peering Connection.
     */
    @JvmName("mcrsiuaiarqvccgs")
    public suspend fun peerVpcId(`value`: Output<String>) {
        this.peerVpcId = value
    }

    /**
     * @param value A optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that requests
     * the peering connection (a maximum of one).
     */
    @JvmName("ajqfbnqtxtdoqghv")
    public suspend fun requester(`value`: Output<VpcPeeringConnectionRequesterArgs>) {
        this.requester = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("fohxfbwpqniawveb")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The ID of the requester VPC.
     */
    @JvmName("itkbjnqqppyjspki")
    public suspend fun vpcId(`value`: Output<String>) {
        this.vpcId = value
    }

    /**
     * @param value An optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that accepts
     * the peering connection (a maximum of one).
     */
    @JvmName("uhhxfwowbirraccq")
    public suspend fun accepter(`value`: VpcPeeringConnectionAccepterArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accepter = mapped
    }

    /**
     * @param argument An optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that accepts
     * the peering connection (a maximum of one).
     */
    @JvmName("vhigqxemoqkoukxm")
    public suspend fun accepter(argument: suspend VpcPeeringConnectionAccepterArgsBuilder.() -> Unit) {
        val toBeMapped = VpcPeeringConnectionAccepterArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.accepter = mapped
    }

    /**
     * @param value Accept the peering (both VPCs need to be in the same AWS account and region).
     */
    @JvmName("yjbkrmcjuxyfsbar")
    public suspend fun autoAccept(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoAccept = mapped
    }

    /**
     * @param value The AWS account ID of the target peer VPC.
     * Defaults to the account ID the [AWS provider][1] is currently connected to, so must be managed if connecting cross-account.
     */
    @JvmName("qrvngmhyrgcayyxl")
    public suspend fun peerOwnerId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.peerOwnerId = mapped
    }

    /**
     * @param value The region of the accepter VPC of the VPC Peering Connection. `auto_accept` must be `false`,
     * and use the `aws.ec2.VpcPeeringConnectionAccepter` to manage the accepter side.
     */
    @JvmName("iqjwqnlsiqkfvgee")
    public suspend fun peerRegion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.peerRegion = mapped
    }

    /**
     * @param value The ID of the target VPC with which you are creating the VPC Peering Connection.
     */
    @JvmName("sgtuquycfburlocj")
    public suspend fun peerVpcId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.peerVpcId = mapped
    }

    /**
     * @param value A optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that requests
     * the peering connection (a maximum of one).
     */
    @JvmName("osqixhtexoufqsme")
    public suspend fun requester(`value`: VpcPeeringConnectionRequesterArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requester = mapped
    }

    /**
     * @param argument A optional configuration block that allows for [VPC Peering Connection](https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html) options to be set for the VPC that requests
     * the peering connection (a maximum of one).
     */
    @JvmName("ytgjaxyhchyusefy")
    public suspend fun requester(argument: suspend VpcPeeringConnectionRequesterArgsBuilder.() -> Unit) {
        val toBeMapped = VpcPeeringConnectionRequesterArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.requester = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("shsvkewpcnhbvwel")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ppneegyntwlqktsh")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The ID of the requester VPC.
     */
    @JvmName("gvdojjbhrlgeqxnc")
    public suspend fun vpcId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcId = mapped
    }

    internal fun build(): VpcPeeringConnectionArgs = VpcPeeringConnectionArgs(
        accepter = accepter,
        autoAccept = autoAccept,
        peerOwnerId = peerOwnerId,
        peerRegion = peerRegion,
        peerVpcId = peerVpcId,
        requester = requester,
        tags = tags,
        vpcId = vpcId,
    )
}
