@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ec2.kotlin

import com.pulumi.aws.ec2.VpnConnectionArgs.builder
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel1LogOptionsArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel1LogOptionsArgsBuilder
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel2LogOptionsArgs
import com.pulumi.aws.ec2.kotlin.inputs.VpnConnectionTunnel2LogOptionsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages a Site-to-Site VPN connection. A Site-to-Site VPN connection is an Internet Protocol security (IPsec) VPN connection between a VPC and an on-premises network.
 * Any new Site-to-Site VPN connection that you create is an [AWS VPN connection](https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-categories.html).
 * > **Note:** The CIDR blocks in the arguments `tunnel1_inside_cidr` and `tunnel2_inside_cidr` must have a prefix of /30 and be a part of a specific range.
 * [Read more about this in the AWS documentation](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_VpnTunnelOptionsSpecification.html).
 * ## Example Usage
 * ### EC2 Transit Gateway
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.ec2transitgateway.TransitGateway("example", {});
 * const exampleCustomerGateway = new aws.ec2.CustomerGateway("example", {
 *     bgpAsn: "65000",
 *     ipAddress: "172.0.0.1",
 *     type: "ipsec.1",
 * });
 * const exampleVpnConnection = new aws.ec2.VpnConnection("example", {
 *     customerGatewayId: exampleCustomerGateway.id,
 *     transitGatewayId: example.id,
 *     type: exampleCustomerGateway.type,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ec2transitgateway.TransitGateway("example")
 * example_customer_gateway = aws.ec2.CustomerGateway("example",
 *     bgp_asn="65000",
 *     ip_address="172.0.0.1",
 *     type="ipsec.1")
 * example_vpn_connection = aws.ec2.VpnConnection("example",
 *     customer_gateway_id=example_customer_gateway.id,
 *     transit_gateway_id=example.id,
 *     type=example_customer_gateway.type)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Ec2TransitGateway.TransitGateway("example");
 *     var exampleCustomerGateway = new Aws.Ec2.CustomerGateway("example", new()
 *     {
 *         BgpAsn = "65000",
 *         IpAddress = "172.0.0.1",
 *         Type = "ipsec.1",
 *     });
 *     var exampleVpnConnection = new Aws.Ec2.VpnConnection("example", new()
 *     {
 *         CustomerGatewayId = exampleCustomerGateway.Id,
 *         TransitGatewayId = example.Id,
 *         Type = exampleCustomerGateway.Type,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2transitgateway"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := ec2transitgateway.NewTransitGateway(ctx, "example", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleCustomerGateway, err := ec2.NewCustomerGateway(ctx, "example", &ec2.CustomerGatewayArgs{
 * 			BgpAsn:    pulumi.String("65000"),
 * 			IpAddress: pulumi.String("172.0.0.1"),
 * 			Type:      pulumi.String("ipsec.1"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpnConnection(ctx, "example", &ec2.VpnConnectionArgs{
 * 			CustomerGatewayId: exampleCustomerGateway.ID(),
 * 			TransitGatewayId:  example.ID(),
 * 			Type:              exampleCustomerGateway.Type,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2transitgateway.TransitGateway;
 * import com.pulumi.aws.ec2.CustomerGateway;
 * import com.pulumi.aws.ec2.CustomerGatewayArgs;
 * import com.pulumi.aws.ec2.VpnConnection;
 * import com.pulumi.aws.ec2.VpnConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new TransitGateway("example");
 *         var exampleCustomerGateway = new CustomerGateway("exampleCustomerGateway", CustomerGatewayArgs.builder()
 *             .bgpAsn(65000)
 *             .ipAddress("172.0.0.1")
 *             .type("ipsec.1")
 *             .build());
 *         var exampleVpnConnection = new VpnConnection("exampleVpnConnection", VpnConnectionArgs.builder()
 *             .customerGatewayId(exampleCustomerGateway.id())
 *             .transitGatewayId(example.id())
 *             .type(exampleCustomerGateway.type())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2transitgateway:TransitGateway
 *   exampleCustomerGateway:
 *     type: aws:ec2:CustomerGateway
 *     name: example
 *     properties:
 *       bgpAsn: 65000
 *       ipAddress: 172.0.0.1
 *       type: ipsec.1
 *   exampleVpnConnection:
 *     type: aws:ec2:VpnConnection
 *     name: example
 *     properties:
 *       customerGatewayId: ${exampleCustomerGateway.id}
 *       transitGatewayId: ${example.id}
 *       type: ${exampleCustomerGateway.type}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Virtual Private Gateway
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const vpc = new aws.ec2.Vpc("vpc", {cidrBlock: "10.0.0.0/16"});
 * const vpnGateway = new aws.ec2.VpnGateway("vpn_gateway", {vpcId: vpc.id});
 * const customerGateway = new aws.ec2.CustomerGateway("customer_gateway", {
 *     bgpAsn: "65000",
 *     ipAddress: "172.0.0.1",
 *     type: "ipsec.1",
 * });
 * const main = new aws.ec2.VpnConnection("main", {
 *     vpnGatewayId: vpnGateway.id,
 *     customerGatewayId: customerGateway.id,
 *     type: "ipsec.1",
 *     staticRoutesOnly: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * vpc = aws.ec2.Vpc("vpc", cidr_block="10.0.0.0/16")
 * vpn_gateway = aws.ec2.VpnGateway("vpn_gateway", vpc_id=vpc.id)
 * customer_gateway = aws.ec2.CustomerGateway("customer_gateway",
 *     bgp_asn="65000",
 *     ip_address="172.0.0.1",
 *     type="ipsec.1")
 * main = aws.ec2.VpnConnection("main",
 *     vpn_gateway_id=vpn_gateway.id,
 *     customer_gateway_id=customer_gateway.id,
 *     type="ipsec.1",
 *     static_routes_only=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var vpc = new Aws.Ec2.Vpc("vpc", new()
 *     {
 *         CidrBlock = "10.0.0.0/16",
 *     });
 *     var vpnGateway = new Aws.Ec2.VpnGateway("vpn_gateway", new()
 *     {
 *         VpcId = vpc.Id,
 *     });
 *     var customerGateway = new Aws.Ec2.CustomerGateway("customer_gateway", new()
 *     {
 *         BgpAsn = "65000",
 *         IpAddress = "172.0.0.1",
 *         Type = "ipsec.1",
 *     });
 *     var main = new Aws.Ec2.VpnConnection("main", new()
 *     {
 *         VpnGatewayId = vpnGateway.Id,
 *         CustomerGatewayId = customerGateway.Id,
 *         Type = "ipsec.1",
 *         StaticRoutesOnly = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		vpc, err := ec2.NewVpc(ctx, "vpc", &ec2.VpcArgs{
 * 			CidrBlock: pulumi.String("10.0.0.0/16"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		vpnGateway, err := ec2.NewVpnGateway(ctx, "vpn_gateway", &ec2.VpnGatewayArgs{
 * 			VpcId: vpc.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		customerGateway, err := ec2.NewCustomerGateway(ctx, "customer_gateway", &ec2.CustomerGatewayArgs{
 * 			BgpAsn:    pulumi.String("65000"),
 * 			IpAddress: pulumi.String("172.0.0.1"),
 * 			Type:      pulumi.String("ipsec.1"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ec2.NewVpnConnection(ctx, "main", &ec2.VpnConnectionArgs{
 * 			VpnGatewayId:      vpnGateway.ID(),
 * 			CustomerGatewayId: customerGateway.ID(),
 * 			Type:              pulumi.String("ipsec.1"),
 * 			StaticRoutesOnly:  pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Vpc;
 * import com.pulumi.aws.ec2.VpcArgs;
 * import com.pulumi.aws.ec2.VpnGateway;
 * import com.pulumi.aws.ec2.VpnGatewayArgs;
 * import com.pulumi.aws.ec2.CustomerGateway;
 * import com.pulumi.aws.ec2.CustomerGatewayArgs;
 * import com.pulumi.aws.ec2.VpnConnection;
 * import com.pulumi.aws.ec2.VpnConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var vpc = new Vpc("vpc", VpcArgs.builder()
 *             .cidrBlock("10.0.0.0/16")
 *             .build());
 *         var vpnGateway = new VpnGateway("vpnGateway", VpnGatewayArgs.builder()
 *             .vpcId(vpc.id())
 *             .build());
 *         var customerGateway = new CustomerGateway("customerGateway", CustomerGatewayArgs.builder()
 *             .bgpAsn(65000)
 *             .ipAddress("172.0.0.1")
 *             .type("ipsec.1")
 *             .build());
 *         var main = new VpnConnection("main", VpnConnectionArgs.builder()
 *             .vpnGatewayId(vpnGateway.id())
 *             .customerGatewayId(customerGateway.id())
 *             .type("ipsec.1")
 *             .staticRoutesOnly(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   vpc:
 *     type: aws:ec2:Vpc
 *     properties:
 *       cidrBlock: 10.0.0.0/16
 *   vpnGateway:
 *     type: aws:ec2:VpnGateway
 *     name: vpn_gateway
 *     properties:
 *       vpcId: ${vpc.id}
 *   customerGateway:
 *     type: aws:ec2:CustomerGateway
 *     name: customer_gateway
 *     properties:
 *       bgpAsn: 65000
 *       ipAddress: 172.0.0.1
 *       type: ipsec.1
 *   main:
 *     type: aws:ec2:VpnConnection
 *     properties:
 *       vpnGatewayId: ${vpnGateway.id}
 *       customerGatewayId: ${customerGateway.id}
 *       type: ipsec.1
 *       staticRoutesOnly: true
 * ```
 * <!--End PulumiCodeChooser -->
 * ### AWS Site to Site Private VPN
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const exampleGateway = new aws.directconnect.Gateway("example", {
 *     name: "example_ipsec_vpn_example",
 *     amazonSideAsn: "64512",
 * });
 * const exampleTransitGateway = new aws.ec2transitgateway.TransitGateway("example", {
 *     amazonSideAsn: 64513,
 *     description: "example_ipsec_vpn_example",
 *     transitGatewayCidrBlocks: ["10&#46;0&#46;0&#46;0/24"],
 * });
 * const exampleCustomerGateway = new aws.ec2.CustomerGateway("example", {
 *     bgpAsn: "64514",
 *     ipAddress: "10.0.0.1",
 *     type: "ipsec.1",
 *     tags: {
 *         Name: "example_ipsec_vpn_example",
 *     },
 * });
 * const exampleGatewayAssociation = new aws.directconnect.GatewayAssociation("example", {
 *     dxGatewayId: exampleGateway.id,
 *     associatedGatewayId: exampleTransitGateway.id,
 *     allowedPrefixes: ["10&#46;0&#46;0&#46;0/8"],
 * });
 * const example = aws.ec2transitgateway.getDirectConnectGatewayAttachmentOutput({
 *     transitGatewayId: exampleTransitGateway.id,
 *     dxGatewayId: exampleGateway.id,
 * });
 * const exampleVpnConnection = new aws.ec2.VpnConnection("example", {
 *     customerGatewayId: exampleCustomerGateway.id,
 *     outsideIpAddressType: "PrivateIpv4",
 *     transitGatewayId: exampleTransitGateway.id,
 *     transportTransitGatewayAttachmentId: example.apply(example => example.id),
 *     type: "ipsec.1",
 *     tags: {
 *         Name: "example_ipsec_vpn_example",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example_gateway = aws.directconnect.Gateway("example",
 *     name="example_ipsec_vpn_example",
 *     amazon_side_asn="64512")
 * example_transit_gateway = aws.ec2transitgateway.TransitGateway("example",
 *     amazon_side_asn=64513,
 *     description="example_ipsec_vpn_example",
 *     transit_gateway_cidr_blocks=["10&#46;0&#46;0&#46;0/24"])
 * example_customer_gateway = aws.ec2.CustomerGateway("example",
 *     bgp_asn="64514",
 *     ip_address="10.0.0.1",
 *     type="ipsec.1",
 *     tags={
 *         "Name": "example_ipsec_vpn_example",
 *     })
 * example_gateway_association = aws.directconnect.GatewayAssociation("example",
 *     dx_gateway_id=example_gateway.id,
 *     associated_gateway_id=example_transit_gateway.id,
 *     allowed_prefixes=["10&#46;0&#46;0&#46;0/8"])
 * example = aws.ec2transitgateway.get_direct_connect_gateway_attachment_output(transit_gateway_id=example_transit_gateway.id,
 *     dx_gateway_id=example_gateway.id)
 * example_vpn_connection = aws.ec2.VpnConnection("example",
 *     customer_gateway_id=example_customer_gateway.id,
 *     outside_ip_address_type="PrivateIpv4",
 *     transit_gateway_id=example_transit_gateway.id,
 *     transport_transit_gateway_attachment_id=example.id,
 *     type="ipsec.1",
 *     tags={
 *         "Name": "example_ipsec_vpn_example",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleGateway = new Aws.DirectConnect.Gateway("example", new()
 *     {
 *         Name = "example_ipsec_vpn_example",
 *         AmazonSideAsn = "64512",
 *     });
 *     var exampleTransitGateway = new Aws.Ec2TransitGateway.TransitGateway("example", new()
 *     {
 *         AmazonSideAsn = 64513,
 *         Description = "example_ipsec_vpn_example",
 *         TransitGatewayCidrBlocks = new[]
 *         {
 *             "10.0.0.0/24",
 *         },
 *     });
 *     var exampleCustomerGateway = new Aws.Ec2.CustomerGateway("example", new()
 *     {
 *         BgpAsn = "64514",
 *         IpAddress = "10.0.0.1",
 *         Type = "ipsec.1",
 *         Tags =
 *         {
 *             { "Name", "example_ipsec_vpn_example" },
 *         },
 *     });
 *     var exampleGatewayAssociation = new Aws.DirectConnect.GatewayAssociation("example", new()
 *     {
 *         DxGatewayId = exampleGateway.Id,
 *         AssociatedGatewayId = exampleTransitGateway.Id,
 *         AllowedPrefixes = new[]
 *         {
 *             "10.0.0.0/8",
 *         },
 *     });
 *     var example = Aws.Ec2TransitGateway.GetDirectConnectGatewayAttachment.Invoke(new()
 *     {
 *         TransitGatewayId = exampleTransitGateway.Id,
 *         DxGatewayId = exampleGateway.Id,
 *     });
 *     var exampleVpnConnection = new Aws.Ec2.VpnConnection("example", new()
 *     {
 *         CustomerGatewayId = exampleCustomerGateway.Id,
 *         OutsideIpAddressType = "PrivateIpv4",
 *         TransitGatewayId = exampleTransitGateway.Id,
 *         TransportTransitGatewayAttachmentId = example.Apply(getDirectConnectGatewayAttachmentResult => getDirectConnectGatewayAttachmentResult.Id),
 *         Type = "ipsec.1",
 *         Tags =
 *         {
 *             { "Name", "example_ipsec_vpn_example" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/directconnect"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2transitgateway"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		exampleGateway, err := directconnect.NewGateway(ctx, "example", &directconnect.GatewayArgs{
 * 			Name:          pulumi.String("example_ipsec_vpn_example"),
 * 			AmazonSideAsn: pulumi.String("64512"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleTransitGateway, err := ec2transitgateway.NewTransitGateway(ctx, "example", &ec2transitgateway.TransitGatewayArgs{
 * 			AmazonSideAsn: pulumi.Int(64513),
 * 			Description:   pulumi.String("example_ipsec_vpn_example"),
 * 			TransitGatewayCidrBlocks: pulumi.StringArray{
 * 				pulumi.String("10.0.0.0/24"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleCustomerGateway, err := ec2.NewCustomerGateway(ctx, "example", &ec2.CustomerGatewayArgs{
 * 			BgpAsn:    pulumi.String("64514"),
 * 			IpAddress: pulumi.String("10.0.0.1"),
 * 			Type:      pulumi.String("ipsec.1"),
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("example_ipsec_vpn_example"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = directconnect.NewGatewayAssociation(ctx, "example", &directconnect.GatewayAssociationArgs{
 * 			DxGatewayId:         exampleGateway.ID(),
 * 			AssociatedGatewayId: exampleTransitGateway.ID(),
 * 			AllowedPrefixes: pulumi.StringArray{
 * 				pulumi.String("10.0.0.0/8"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example := ec2transitgateway.GetDirectConnectGatewayAttachmentOutput(ctx, ec2transitgateway.GetDirectConnectGatewayAttachmentOutputArgs{
 * 			TransitGatewayId: exampleTransitGateway.ID(),
 * 			DxGatewayId:      exampleGateway.ID(),
 * 		}, nil)
 * 		_, err = ec2.NewVpnConnection(ctx, "example", &ec2.VpnConnectionArgs{
 * 			CustomerGatewayId:    exampleCustomerGateway.ID(),
 * 			OutsideIpAddressType: pulumi.String("PrivateIpv4"),
 * 			TransitGatewayId:     exampleTransitGateway.ID(),
 * 			TransportTransitGatewayAttachmentId: pulumi.String(example.ApplyT(func(example ec2transitgateway.GetDirectConnectGatewayAttachmentResult) (*string, error) {
 * 				return &example.Id, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 			Type: pulumi.String("ipsec.1"),
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("example_ipsec_vpn_example"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.directconnect.Gateway;
 * import com.pulumi.aws.directconnect.GatewayArgs;
 * import com.pulumi.aws.ec2transitgateway.TransitGateway;
 * import com.pulumi.aws.ec2transitgateway.TransitGatewayArgs;
 * import com.pulumi.aws.ec2.CustomerGateway;
 * import com.pulumi.aws.ec2.CustomerGatewayArgs;
 * import com.pulumi.aws.directconnect.GatewayAssociation;
 * import com.pulumi.aws.directconnect.GatewayAssociationArgs;
 * import com.pulumi.aws.ec2transitgateway.Ec2transitgatewayFunctions;
 * import com.pulumi.aws.ec2transitgateway.inputs.GetDirectConnectGatewayAttachmentArgs;
 * import com.pulumi.aws.ec2.VpnConnection;
 * import com.pulumi.aws.ec2.VpnConnectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleGateway = new Gateway("exampleGateway", GatewayArgs.builder()
 *             .name("example_ipsec_vpn_example")
 *             .amazonSideAsn("64512")
 *             .build());
 *         var exampleTransitGateway = new TransitGateway("exampleTransitGateway", TransitGatewayArgs.builder()
 *             .amazonSideAsn("64513")
 *             .description("example_ipsec_vpn_example")
 *             .transitGatewayCidrBlocks("10.0.0.0/24")
 *             .build());
 *         var exampleCustomerGateway = new CustomerGateway("exampleCustomerGateway", CustomerGatewayArgs.builder()
 *             .bgpAsn(64514)
 *             .ipAddress("10.0.0.1")
 *             .type("ipsec.1")
 *             .tags(Map.of("Name", "example_ipsec_vpn_example"))
 *             .build());
 *         var exampleGatewayAssociation = new GatewayAssociation("exampleGatewayAssociation", GatewayAssociationArgs.builder()
 *             .dxGatewayId(exampleGateway.id())
 *             .associatedGatewayId(exampleTransitGateway.id())
 *             .allowedPrefixes("10.0.0.0/8")
 *             .build());
 *         final var example = Ec2transitgatewayFunctions.getDirectConnectGatewayAttachment(GetDirectConnectGatewayAttachmentArgs.builder()
 *             .transitGatewayId(exampleTransitGateway.id())
 *             .dxGatewayId(exampleGateway.id())
 *             .build());
 *         var exampleVpnConnection = new VpnConnection("exampleVpnConnection", VpnConnectionArgs.builder()
 *             .customerGatewayId(exampleCustomerGateway.id())
 *             .outsideIpAddressType("PrivateIpv4")
 *             .transitGatewayId(exampleTransitGateway.id())
 *             .transportTransitGatewayAttachmentId(example.applyValue(getDirectConnectGatewayAttachmentResult -> getDirectConnectGatewayAttachmentResult).applyValue(example -> example.applyValue(getDirectConnectGatewayAttachmentResult -> getDirectConnectGatewayAttachmentResult.id())))
 *             .type("ipsec.1")
 *             .tags(Map.of("Name", "example_ipsec_vpn_example"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleGateway:
 *     type: aws:directconnect:Gateway
 *     name: example
 *     properties:
 *       name: example_ipsec_vpn_example
 *       amazonSideAsn: '64512'
 *   exampleTransitGateway:
 *     type: aws:ec2transitgateway:TransitGateway
 *     name: example
 *     properties:
 *       amazonSideAsn: '64513'
 *       description: example_ipsec_vpn_example
 *       transitGatewayCidrBlocks:
 *         - 10.0.0.0/24
 *   exampleCustomerGateway:
 *     type: aws:ec2:CustomerGateway
 *     name: example
 *     properties:
 *       bgpAsn: 64514
 *       ipAddress: 10.0.0.1
 *       type: ipsec.1
 *       tags:
 *         Name: example_ipsec_vpn_example
 *   exampleGatewayAssociation:
 *     type: aws:directconnect:GatewayAssociation
 *     name: example
 *     properties:
 *       dxGatewayId: ${exampleGateway.id}
 *       associatedGatewayId: ${exampleTransitGateway.id}
 *       allowedPrefixes:
 *         - 10.0.0.0/8
 *   exampleVpnConnection:
 *     type: aws:ec2:VpnConnection
 *     name: example
 *     properties:
 *       customerGatewayId: ${exampleCustomerGateway.id}
 *       outsideIpAddressType: PrivateIpv4
 *       transitGatewayId: ${exampleTransitGateway.id}
 *       transportTransitGatewayAttachmentId: ${example.id}
 *       type: ipsec.1
 *       tags:
 *         Name: example_ipsec_vpn_example
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: aws:ec2transitgateway:getDirectConnectGatewayAttachment
 *       Arguments:
 *         transitGatewayId: ${exampleTransitGateway.id}
 *         dxGatewayId: ${exampleGateway.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import VPN Connections using the VPN connection `id`. For example:
 * ```sh
 * $ pulumi import aws:ec2/vpnConnection:VpnConnection testvpnconnection vpn-40f41529
 * ```
 * @property customerGatewayId The ID of the customer gateway.
 * @property enableAcceleration Indicate whether to enable acceleration for the VPN connection. Supports only EC2 Transit Gateway.
 * @property localIpv4NetworkCidr The IPv4 CIDR on the customer gateway (on-premises) side of the VPN connection.
 * @property localIpv6NetworkCidr The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
 * @property outsideIpAddressType Indicates if a Public S2S VPN or Private S2S VPN over AWS Direct Connect. Valid values are `PublicIpv4 | PrivateIpv4`
 * @property remoteIpv4NetworkCidr The IPv4 CIDR on the AWS side of the VPN connection.
 * @property remoteIpv6NetworkCidr The IPv6 CIDR on the AWS side of the VPN connection.
 * @property staticRoutesOnly Whether the VPN connection uses static routes exclusively. Static routes must be used for devices that don't support BGP.
 * @property tags Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property transitGatewayId The ID of the EC2 Transit Gateway.
 * @property transportTransitGatewayAttachmentId . The attachment ID of the Transit Gateway attachment to Direct Connect Gateway. The ID is obtained through a data source only.
 * @property tunnel1DpdTimeoutAction The action to take after DPD timeout occurs for the first VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
 * @property tunnel1DpdTimeoutSeconds The number of seconds after which a DPD timeout occurs for the first VPN tunnel. Valid value is equal or higher than `30`.
 * @property tunnel1EnableTunnelLifecycleControl Turn on or off tunnel endpoint lifecycle control feature for the first VPN tunnel. Valid values are `true | false`.
 * @property tunnel1IkeVersions The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
 * @property tunnel1InsideCidr The CIDR block of the inside IP addresses for the first VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
 * @property tunnel1InsideIpv6Cidr The range of inside IPv6 addresses for the first VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
 * @property tunnel1LogOptions Options for logging VPN tunnel activity. See Log Options below for more details.
 * @property tunnel1Phase1DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel1Phase1EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel1Phase1IntegrityAlgorithms One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel1Phase1LifetimeSeconds The lifetime for phase 1 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `28800`.
 * @property tunnel1Phase2DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel1Phase2EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel1Phase2IntegrityAlgorithms List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel1Phase2LifetimeSeconds The lifetime for phase 2 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `3600`.
 * @property tunnel1PresharedKey The preshared key of the first VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
 * @property tunnel1RekeyFuzzPercentage The percentage of the rekey window for the first VPN tunnel (determined by `tunnel1_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
 * @property tunnel1RekeyMarginTimeSeconds The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the first VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel1_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel1_phase2_lifetime_seconds`.
 * @property tunnel1ReplayWindowSize The number of packets in an IKE replay window for the first VPN tunnel. Valid value is between `64` and `2048`.
 * @property tunnel1StartupAction The action to take when the establishing the tunnel for the first VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
 * @property tunnel2DpdTimeoutAction The action to take after DPD timeout occurs for the second VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
 * @property tunnel2DpdTimeoutSeconds The number of seconds after which a DPD timeout occurs for the second VPN tunnel. Valid value is equal or higher than `30`.
 * @property tunnel2EnableTunnelLifecycleControl Turn on or off tunnel endpoint lifecycle control feature for the second VPN tunnel. Valid values are `true | false`.
 * @property tunnel2IkeVersions The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
 * @property tunnel2InsideCidr The CIDR block of the inside IP addresses for the second VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
 * @property tunnel2InsideIpv6Cidr The range of inside IPv6 addresses for the second VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
 * @property tunnel2LogOptions Options for logging VPN tunnel activity. See Log Options below for more details.
 * @property tunnel2Phase1DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel2Phase1EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel2Phase1IntegrityAlgorithms One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel2Phase1LifetimeSeconds The lifetime for phase 1 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `28800`.
 * @property tunnel2Phase2DhGroupNumbers List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
 * @property tunnel2Phase2EncryptionAlgorithms List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
 * @property tunnel2Phase2IntegrityAlgorithms List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
 * @property tunnel2Phase2LifetimeSeconds The lifetime for phase 2 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `3600`.
 * @property tunnel2PresharedKey The preshared key of the second VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
 * @property tunnel2RekeyFuzzPercentage The percentage of the rekey window for the second VPN tunnel (determined by `tunnel2_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
 * @property tunnel2RekeyMarginTimeSeconds The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the second VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel2_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel2_phase2_lifetime_seconds`.
 * @property tunnel2ReplayWindowSize The number of packets in an IKE replay window for the second VPN tunnel. Valid value is between `64` and `2048`.
 * @property tunnel2StartupAction The action to take when the establishing the tunnel for the second VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
 * @property tunnelInsideIpVersion Indicate whether the VPN tunnels process IPv4 or IPv6 traffic. Valid values are `ipv4 | ipv6`. `ipv6` Supports only EC2 Transit Gateway.
 * @property type The type of VPN connection. The only type AWS supports at this time is "ipsec.1".
 * @property vpnGatewayId The ID of the Virtual Private Gateway.
 */
public data class VpnConnectionArgs(
    public val customerGatewayId: Output<String>? = null,
    public val enableAcceleration: Output<Boolean>? = null,
    public val localIpv4NetworkCidr: Output<String>? = null,
    public val localIpv6NetworkCidr: Output<String>? = null,
    public val outsideIpAddressType: Output<String>? = null,
    public val remoteIpv4NetworkCidr: Output<String>? = null,
    public val remoteIpv6NetworkCidr: Output<String>? = null,
    public val staticRoutesOnly: Output<Boolean>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val transitGatewayId: Output<String>? = null,
    public val transportTransitGatewayAttachmentId: Output<String>? = null,
    public val tunnel1DpdTimeoutAction: Output<String>? = null,
    public val tunnel1DpdTimeoutSeconds: Output<Int>? = null,
    public val tunnel1EnableTunnelLifecycleControl: Output<Boolean>? = null,
    public val tunnel1IkeVersions: Output<List<String>>? = null,
    public val tunnel1InsideCidr: Output<String>? = null,
    public val tunnel1InsideIpv6Cidr: Output<String>? = null,
    public val tunnel1LogOptions: Output<VpnConnectionTunnel1LogOptionsArgs>? = null,
    public val tunnel1Phase1DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel1Phase1EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase1IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase1LifetimeSeconds: Output<Int>? = null,
    public val tunnel1Phase2DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel1Phase2EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase2IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel1Phase2LifetimeSeconds: Output<Int>? = null,
    public val tunnel1PresharedKey: Output<String>? = null,
    public val tunnel1RekeyFuzzPercentage: Output<Int>? = null,
    public val tunnel1RekeyMarginTimeSeconds: Output<Int>? = null,
    public val tunnel1ReplayWindowSize: Output<Int>? = null,
    public val tunnel1StartupAction: Output<String>? = null,
    public val tunnel2DpdTimeoutAction: Output<String>? = null,
    public val tunnel2DpdTimeoutSeconds: Output<Int>? = null,
    public val tunnel2EnableTunnelLifecycleControl: Output<Boolean>? = null,
    public val tunnel2IkeVersions: Output<List<String>>? = null,
    public val tunnel2InsideCidr: Output<String>? = null,
    public val tunnel2InsideIpv6Cidr: Output<String>? = null,
    public val tunnel2LogOptions: Output<VpnConnectionTunnel2LogOptionsArgs>? = null,
    public val tunnel2Phase1DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel2Phase1EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase1IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase1LifetimeSeconds: Output<Int>? = null,
    public val tunnel2Phase2DhGroupNumbers: Output<List<Int>>? = null,
    public val tunnel2Phase2EncryptionAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase2IntegrityAlgorithms: Output<List<String>>? = null,
    public val tunnel2Phase2LifetimeSeconds: Output<Int>? = null,
    public val tunnel2PresharedKey: Output<String>? = null,
    public val tunnel2RekeyFuzzPercentage: Output<Int>? = null,
    public val tunnel2RekeyMarginTimeSeconds: Output<Int>? = null,
    public val tunnel2ReplayWindowSize: Output<Int>? = null,
    public val tunnel2StartupAction: Output<String>? = null,
    public val tunnelInsideIpVersion: Output<String>? = null,
    public val type: Output<String>? = null,
    public val vpnGatewayId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ec2.VpnConnectionArgs> {
    override fun toJava(): com.pulumi.aws.ec2.VpnConnectionArgs =
        com.pulumi.aws.ec2.VpnConnectionArgs.builder()
            .customerGatewayId(customerGatewayId?.applyValue({ args0 -> args0 }))
            .enableAcceleration(enableAcceleration?.applyValue({ args0 -> args0 }))
            .localIpv4NetworkCidr(localIpv4NetworkCidr?.applyValue({ args0 -> args0 }))
            .localIpv6NetworkCidr(localIpv6NetworkCidr?.applyValue({ args0 -> args0 }))
            .outsideIpAddressType(outsideIpAddressType?.applyValue({ args0 -> args0 }))
            .remoteIpv4NetworkCidr(remoteIpv4NetworkCidr?.applyValue({ args0 -> args0 }))
            .remoteIpv6NetworkCidr(remoteIpv6NetworkCidr?.applyValue({ args0 -> args0 }))
            .staticRoutesOnly(staticRoutesOnly?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .transitGatewayId(transitGatewayId?.applyValue({ args0 -> args0 }))
            .transportTransitGatewayAttachmentId(
                transportTransitGatewayAttachmentId?.applyValue({ args0 ->
                    args0
                }),
            )
            .tunnel1DpdTimeoutAction(tunnel1DpdTimeoutAction?.applyValue({ args0 -> args0 }))
            .tunnel1DpdTimeoutSeconds(tunnel1DpdTimeoutSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1EnableTunnelLifecycleControl(
                tunnel1EnableTunnelLifecycleControl?.applyValue({ args0 ->
                    args0
                }),
            )
            .tunnel1IkeVersions(tunnel1IkeVersions?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .tunnel1InsideCidr(tunnel1InsideCidr?.applyValue({ args0 -> args0 }))
            .tunnel1InsideIpv6Cidr(tunnel1InsideIpv6Cidr?.applyValue({ args0 -> args0 }))
            .tunnel1LogOptions(tunnel1LogOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tunnel1Phase1DhGroupNumbers(
                tunnel1Phase1DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel1Phase1EncryptionAlgorithms(
                tunnel1Phase1EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase1IntegrityAlgorithms(
                tunnel1Phase1IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase1LifetimeSeconds(tunnel1Phase1LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1Phase2DhGroupNumbers(
                tunnel1Phase2DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel1Phase2EncryptionAlgorithms(
                tunnel1Phase2EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase2IntegrityAlgorithms(
                tunnel1Phase2IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel1Phase2LifetimeSeconds(tunnel1Phase2LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1PresharedKey(tunnel1PresharedKey?.applyValue({ args0 -> args0 }))
            .tunnel1RekeyFuzzPercentage(tunnel1RekeyFuzzPercentage?.applyValue({ args0 -> args0 }))
            .tunnel1RekeyMarginTimeSeconds(tunnel1RekeyMarginTimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel1ReplayWindowSize(tunnel1ReplayWindowSize?.applyValue({ args0 -> args0 }))
            .tunnel1StartupAction(tunnel1StartupAction?.applyValue({ args0 -> args0 }))
            .tunnel2DpdTimeoutAction(tunnel2DpdTimeoutAction?.applyValue({ args0 -> args0 }))
            .tunnel2DpdTimeoutSeconds(tunnel2DpdTimeoutSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2EnableTunnelLifecycleControl(
                tunnel2EnableTunnelLifecycleControl?.applyValue({ args0 ->
                    args0
                }),
            )
            .tunnel2IkeVersions(tunnel2IkeVersions?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .tunnel2InsideCidr(tunnel2InsideCidr?.applyValue({ args0 -> args0 }))
            .tunnel2InsideIpv6Cidr(tunnel2InsideIpv6Cidr?.applyValue({ args0 -> args0 }))
            .tunnel2LogOptions(tunnel2LogOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tunnel2Phase1DhGroupNumbers(
                tunnel2Phase1DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel2Phase1EncryptionAlgorithms(
                tunnel2Phase1EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase1IntegrityAlgorithms(
                tunnel2Phase1IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase1LifetimeSeconds(tunnel2Phase1LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2Phase2DhGroupNumbers(
                tunnel2Phase2DhGroupNumbers?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tunnel2Phase2EncryptionAlgorithms(
                tunnel2Phase2EncryptionAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase2IntegrityAlgorithms(
                tunnel2Phase2IntegrityAlgorithms?.applyValue({ args0 ->
                    args0.map({ args0 -> args0 })
                }),
            )
            .tunnel2Phase2LifetimeSeconds(tunnel2Phase2LifetimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2PresharedKey(tunnel2PresharedKey?.applyValue({ args0 -> args0 }))
            .tunnel2RekeyFuzzPercentage(tunnel2RekeyFuzzPercentage?.applyValue({ args0 -> args0 }))
            .tunnel2RekeyMarginTimeSeconds(tunnel2RekeyMarginTimeSeconds?.applyValue({ args0 -> args0 }))
            .tunnel2ReplayWindowSize(tunnel2ReplayWindowSize?.applyValue({ args0 -> args0 }))
            .tunnel2StartupAction(tunnel2StartupAction?.applyValue({ args0 -> args0 }))
            .tunnelInsideIpVersion(tunnelInsideIpVersion?.applyValue({ args0 -> args0 }))
            .type(type?.applyValue({ args0 -> args0 }))
            .vpnGatewayId(vpnGatewayId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [VpnConnectionArgs].
 */
@PulumiTagMarker
public class VpnConnectionArgsBuilder internal constructor() {
    private var customerGatewayId: Output<String>? = null

    private var enableAcceleration: Output<Boolean>? = null

    private var localIpv4NetworkCidr: Output<String>? = null

    private var localIpv6NetworkCidr: Output<String>? = null

    private var outsideIpAddressType: Output<String>? = null

    private var remoteIpv4NetworkCidr: Output<String>? = null

    private var remoteIpv6NetworkCidr: Output<String>? = null

    private var staticRoutesOnly: Output<Boolean>? = null

    private var tags: Output<Map<String, String>>? = null

    private var transitGatewayId: Output<String>? = null

    private var transportTransitGatewayAttachmentId: Output<String>? = null

    private var tunnel1DpdTimeoutAction: Output<String>? = null

    private var tunnel1DpdTimeoutSeconds: Output<Int>? = null

    private var tunnel1EnableTunnelLifecycleControl: Output<Boolean>? = null

    private var tunnel1IkeVersions: Output<List<String>>? = null

    private var tunnel1InsideCidr: Output<String>? = null

    private var tunnel1InsideIpv6Cidr: Output<String>? = null

    private var tunnel1LogOptions: Output<VpnConnectionTunnel1LogOptionsArgs>? = null

    private var tunnel1Phase1DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel1Phase1EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase1IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase1LifetimeSeconds: Output<Int>? = null

    private var tunnel1Phase2DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel1Phase2EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase2IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel1Phase2LifetimeSeconds: Output<Int>? = null

    private var tunnel1PresharedKey: Output<String>? = null

    private var tunnel1RekeyFuzzPercentage: Output<Int>? = null

    private var tunnel1RekeyMarginTimeSeconds: Output<Int>? = null

    private var tunnel1ReplayWindowSize: Output<Int>? = null

    private var tunnel1StartupAction: Output<String>? = null

    private var tunnel2DpdTimeoutAction: Output<String>? = null

    private var tunnel2DpdTimeoutSeconds: Output<Int>? = null

    private var tunnel2EnableTunnelLifecycleControl: Output<Boolean>? = null

    private var tunnel2IkeVersions: Output<List<String>>? = null

    private var tunnel2InsideCidr: Output<String>? = null

    private var tunnel2InsideIpv6Cidr: Output<String>? = null

    private var tunnel2LogOptions: Output<VpnConnectionTunnel2LogOptionsArgs>? = null

    private var tunnel2Phase1DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel2Phase1EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase1IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase1LifetimeSeconds: Output<Int>? = null

    private var tunnel2Phase2DhGroupNumbers: Output<List<Int>>? = null

    private var tunnel2Phase2EncryptionAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase2IntegrityAlgorithms: Output<List<String>>? = null

    private var tunnel2Phase2LifetimeSeconds: Output<Int>? = null

    private var tunnel2PresharedKey: Output<String>? = null

    private var tunnel2RekeyFuzzPercentage: Output<Int>? = null

    private var tunnel2RekeyMarginTimeSeconds: Output<Int>? = null

    private var tunnel2ReplayWindowSize: Output<Int>? = null

    private var tunnel2StartupAction: Output<String>? = null

    private var tunnelInsideIpVersion: Output<String>? = null

    private var type: Output<String>? = null

    private var vpnGatewayId: Output<String>? = null

    /**
     * @param value The ID of the customer gateway.
     */
    @JvmName("dokxrddpndhvdkcr")
    public suspend fun customerGatewayId(`value`: Output<String>) {
        this.customerGatewayId = value
    }

    /**
     * @param value Indicate whether to enable acceleration for the VPN connection. Supports only EC2 Transit Gateway.
     */
    @JvmName("hwknlbmroxsbyxue")
    public suspend fun enableAcceleration(`value`: Output<Boolean>) {
        this.enableAcceleration = value
    }

    /**
     * @param value The IPv4 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("cjtvuufmvonsfclf")
    public suspend fun localIpv4NetworkCidr(`value`: Output<String>) {
        this.localIpv4NetworkCidr = value
    }

    /**
     * @param value The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("ammwjpjmxuvyrjht")
    public suspend fun localIpv6NetworkCidr(`value`: Output<String>) {
        this.localIpv6NetworkCidr = value
    }

    /**
     * @param value Indicates if a Public S2S VPN or Private S2S VPN over AWS Direct Connect. Valid values are `PublicIpv4 | PrivateIpv4`
     */
    @JvmName("bhwqwkrhkcbcjffn")
    public suspend fun outsideIpAddressType(`value`: Output<String>) {
        this.outsideIpAddressType = value
    }

    /**
     * @param value The IPv4 CIDR on the AWS side of the VPN connection.
     */
    @JvmName("pwcyvkkfakwobehb")
    public suspend fun remoteIpv4NetworkCidr(`value`: Output<String>) {
        this.remoteIpv4NetworkCidr = value
    }

    /**
     * @param value The IPv6 CIDR on the AWS side of the VPN connection.
     */
    @JvmName("fcpyimywanwpulwx")
    public suspend fun remoteIpv6NetworkCidr(`value`: Output<String>) {
        this.remoteIpv6NetworkCidr = value
    }

    /**
     * @param value Whether the VPN connection uses static routes exclusively. Static routes must be used for devices that don't support BGP.
     */
    @JvmName("qxxlukpoqbmuyfge")
    public suspend fun staticRoutesOnly(`value`: Output<Boolean>) {
        this.staticRoutesOnly = value
    }

    /**
     * @param value Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("dgmkuilqvfukmaha")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The ID of the EC2 Transit Gateway.
     */
    @JvmName("eitaeagxcctqbmkp")
    public suspend fun transitGatewayId(`value`: Output<String>) {
        this.transitGatewayId = value
    }

    /**
     * @param value . The attachment ID of the Transit Gateway attachment to Direct Connect Gateway. The ID is obtained through a data source only.
     */
    @JvmName("xffhmlpjrffdngdw")
    public suspend fun transportTransitGatewayAttachmentId(`value`: Output<String>) {
        this.transportTransitGatewayAttachmentId = value
    }

    /**
     * @param value The action to take after DPD timeout occurs for the first VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("qsvhfliddbkyrwsd")
    public suspend fun tunnel1DpdTimeoutAction(`value`: Output<String>) {
        this.tunnel1DpdTimeoutAction = value
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the first VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("kklwyrnwxcfnrmhh")
    public suspend fun tunnel1DpdTimeoutSeconds(`value`: Output<Int>) {
        this.tunnel1DpdTimeoutSeconds = value
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the first VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("xdvkmfkxccngdivc")
    public suspend fun tunnel1EnableTunnelLifecycleControl(`value`: Output<Boolean>) {
        this.tunnel1EnableTunnelLifecycleControl = value
    }

    /**
     * @param value The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("mnyytryahdlffoeh")
    public suspend fun tunnel1IkeVersions(`value`: Output<List<String>>) {
        this.tunnel1IkeVersions = value
    }

    @JvmName("njgaslgtpmvqfoyr")
    public suspend fun tunnel1IkeVersions(vararg values: Output<String>) {
        this.tunnel1IkeVersions = Output.all(values.asList())
    }

    /**
     * @param values The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("atxsuowrwwspscmb")
    public suspend fun tunnel1IkeVersions(values: List<Output<String>>) {
        this.tunnel1IkeVersions = Output.all(values)
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the first VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("btqfjgdojwsrrqrl")
    public suspend fun tunnel1InsideCidr(`value`: Output<String>) {
        this.tunnel1InsideCidr = value
    }

    /**
     * @param value The range of inside IPv6 addresses for the first VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("sittedmiyjjubqpn")
    public suspend fun tunnel1InsideIpv6Cidr(`value`: Output<String>) {
        this.tunnel1InsideIpv6Cidr = value
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("xmrkstwsownvecqo")
    public suspend fun tunnel1LogOptions(`value`: Output<VpnConnectionTunnel1LogOptionsArgs>) {
        this.tunnel1LogOptions = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("agsmbhhjwtjkmhaa")
    public suspend fun tunnel1Phase1DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel1Phase1DhGroupNumbers = value
    }

    @JvmName("quycnvqlflgyrhuw")
    public suspend fun tunnel1Phase1DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel1Phase1DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("qyoubgciivvisnhc")
    public suspend fun tunnel1Phase1DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel1Phase1DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("bhbckjriqxdaihst")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase1EncryptionAlgorithms = value
    }

    @JvmName("swfdegsifkyebcqd")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase1EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("gqqujmieaweehpjc")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase1EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("trmjfqafsrfhcdar")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase1IntegrityAlgorithms = value
    }

    @JvmName("bkyvxjnabidldndh")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase1IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("mvpfqcssjvpwuyvi")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase1IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("ftlsxyqdvfqckaum")
    public suspend fun tunnel1Phase1LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel1Phase1LifetimeSeconds = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("ogdtctanyehfmshf")
    public suspend fun tunnel1Phase2DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel1Phase2DhGroupNumbers = value
    }

    @JvmName("ugpnoqbueslfobjk")
    public suspend fun tunnel1Phase2DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel1Phase2DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("lihrebnnispfxoax")
    public suspend fun tunnel1Phase2DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel1Phase2DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("brnjhhrstnlqgqxg")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase2EncryptionAlgorithms = value
    }

    @JvmName("gcnqesqwchfdiawt")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase2EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("bibxcalnxewbvrry")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase2EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("nnofpewkgxctqhpu")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel1Phase2IntegrityAlgorithms = value
    }

    @JvmName("sgldcxebmvpqwewj")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel1Phase2IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("jurhccaslpeocyvt")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel1Phase2IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("wfhmgiucjkkhxrvh")
    public suspend fun tunnel1Phase2LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel1Phase2LifetimeSeconds = value
    }

    /**
     * @param value The preshared key of the first VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("fyiwprskgnbfhwtu")
    public suspend fun tunnel1PresharedKey(`value`: Output<String>) {
        this.tunnel1PresharedKey = value
    }

    /**
     * @param value The percentage of the rekey window for the first VPN tunnel (determined by `tunnel1_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("pgpplsoqienruijo")
    public suspend fun tunnel1RekeyFuzzPercentage(`value`: Output<Int>) {
        this.tunnel1RekeyFuzzPercentage = value
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the first VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel1_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel1_phase2_lifetime_seconds`.
     */
    @JvmName("rdwicqedwjgqtbyp")
    public suspend fun tunnel1RekeyMarginTimeSeconds(`value`: Output<Int>) {
        this.tunnel1RekeyMarginTimeSeconds = value
    }

    /**
     * @param value The number of packets in an IKE replay window for the first VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("giracwbukjrwqrlg")
    public suspend fun tunnel1ReplayWindowSize(`value`: Output<Int>) {
        this.tunnel1ReplayWindowSize = value
    }

    /**
     * @param value The action to take when the establishing the tunnel for the first VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("aaxapkjrujpicggu")
    public suspend fun tunnel1StartupAction(`value`: Output<String>) {
        this.tunnel1StartupAction = value
    }

    /**
     * @param value The action to take after DPD timeout occurs for the second VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("nvdxgxcndubgtvat")
    public suspend fun tunnel2DpdTimeoutAction(`value`: Output<String>) {
        this.tunnel2DpdTimeoutAction = value
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the second VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("grtorwkmnximvsyf")
    public suspend fun tunnel2DpdTimeoutSeconds(`value`: Output<Int>) {
        this.tunnel2DpdTimeoutSeconds = value
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the second VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("mmjhvxnvflxrqkuv")
    public suspend fun tunnel2EnableTunnelLifecycleControl(`value`: Output<Boolean>) {
        this.tunnel2EnableTunnelLifecycleControl = value
    }

    /**
     * @param value The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("rqgkcnfeppnrtydf")
    public suspend fun tunnel2IkeVersions(`value`: Output<List<String>>) {
        this.tunnel2IkeVersions = value
    }

    @JvmName("jmupljmxgdopfbxs")
    public suspend fun tunnel2IkeVersions(vararg values: Output<String>) {
        this.tunnel2IkeVersions = Output.all(values.asList())
    }

    /**
     * @param values The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("ofspdwbsxqbmecdf")
    public suspend fun tunnel2IkeVersions(values: List<Output<String>>) {
        this.tunnel2IkeVersions = Output.all(values)
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the second VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("qufklftyveukowsi")
    public suspend fun tunnel2InsideCidr(`value`: Output<String>) {
        this.tunnel2InsideCidr = value
    }

    /**
     * @param value The range of inside IPv6 addresses for the second VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("otwoxpqbywatgioi")
    public suspend fun tunnel2InsideIpv6Cidr(`value`: Output<String>) {
        this.tunnel2InsideIpv6Cidr = value
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("eilqpjkcckhgolkr")
    public suspend fun tunnel2LogOptions(`value`: Output<VpnConnectionTunnel2LogOptionsArgs>) {
        this.tunnel2LogOptions = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("cndufjsfxqopgcus")
    public suspend fun tunnel2Phase1DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel2Phase1DhGroupNumbers = value
    }

    @JvmName("ejnsgchoshyaelhm")
    public suspend fun tunnel2Phase1DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel2Phase1DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("yeecayywwncrktsw")
    public suspend fun tunnel2Phase1DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel2Phase1DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("qpquoipboxpnadbi")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase1EncryptionAlgorithms = value
    }

    @JvmName("eldnvghyhtddjlnl")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase1EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("ditsrljcksgvghtp")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase1EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("lokqrxgyqwkwqcxh")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase1IntegrityAlgorithms = value
    }

    @JvmName("mkkushteclpsbgqe")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase1IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("ljqojqmlppaygmhp")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase1IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("ahxqejlxclgucoqr")
    public suspend fun tunnel2Phase1LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel2Phase1LifetimeSeconds = value
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("japgntkllyfrrnby")
    public suspend fun tunnel2Phase2DhGroupNumbers(`value`: Output<List<Int>>) {
        this.tunnel2Phase2DhGroupNumbers = value
    }

    @JvmName("oqgwqrhybnvvexpk")
    public suspend fun tunnel2Phase2DhGroupNumbers(vararg values: Output<Int>) {
        this.tunnel2Phase2DhGroupNumbers = Output.all(values.asList())
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("offrdytbrapvmgym")
    public suspend fun tunnel2Phase2DhGroupNumbers(values: List<Output<Int>>) {
        this.tunnel2Phase2DhGroupNumbers = Output.all(values)
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("uwxritukkqnaiood")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase2EncryptionAlgorithms = value
    }

    @JvmName("bsmggoecsrdycrwk")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase2EncryptionAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("uncfdhucldgsjxit")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase2EncryptionAlgorithms = Output.all(values)
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("koplnvyfvwdruxjs")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(`value`: Output<List<String>>) {
        this.tunnel2Phase2IntegrityAlgorithms = value
    }

    @JvmName("uvhcpidqkglgkoln")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(vararg values: Output<String>) {
        this.tunnel2Phase2IntegrityAlgorithms = Output.all(values.asList())
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("xiugnrkplwgtqfgh")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(values: List<Output<String>>) {
        this.tunnel2Phase2IntegrityAlgorithms = Output.all(values)
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("iiyqluctkgyqojuv")
    public suspend fun tunnel2Phase2LifetimeSeconds(`value`: Output<Int>) {
        this.tunnel2Phase2LifetimeSeconds = value
    }

    /**
     * @param value The preshared key of the second VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("oqvldtvbvwfskoot")
    public suspend fun tunnel2PresharedKey(`value`: Output<String>) {
        this.tunnel2PresharedKey = value
    }

    /**
     * @param value The percentage of the rekey window for the second VPN tunnel (determined by `tunnel2_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("qohnqfuukejbojor")
    public suspend fun tunnel2RekeyFuzzPercentage(`value`: Output<Int>) {
        this.tunnel2RekeyFuzzPercentage = value
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the second VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel2_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel2_phase2_lifetime_seconds`.
     */
    @JvmName("stjxewebwwacbvct")
    public suspend fun tunnel2RekeyMarginTimeSeconds(`value`: Output<Int>) {
        this.tunnel2RekeyMarginTimeSeconds = value
    }

    /**
     * @param value The number of packets in an IKE replay window for the second VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("mbddecjfvytaxvvw")
    public suspend fun tunnel2ReplayWindowSize(`value`: Output<Int>) {
        this.tunnel2ReplayWindowSize = value
    }

    /**
     * @param value The action to take when the establishing the tunnel for the second VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("uafsodaciylukvnt")
    public suspend fun tunnel2StartupAction(`value`: Output<String>) {
        this.tunnel2StartupAction = value
    }

    /**
     * @param value Indicate whether the VPN tunnels process IPv4 or IPv6 traffic. Valid values are `ipv4 | ipv6`. `ipv6` Supports only EC2 Transit Gateway.
     */
    @JvmName("vsutjgbquskklhvs")
    public suspend fun tunnelInsideIpVersion(`value`: Output<String>) {
        this.tunnelInsideIpVersion = value
    }

    /**
     * @param value The type of VPN connection. The only type AWS supports at this time is "ipsec.1".
     */
    @JvmName("wohpuwsgiiyswiuk")
    public suspend fun type(`value`: Output<String>) {
        this.type = value
    }

    /**
     * @param value The ID of the Virtual Private Gateway.
     */
    @JvmName("tiiudkhiamqwofnb")
    public suspend fun vpnGatewayId(`value`: Output<String>) {
        this.vpnGatewayId = value
    }

    /**
     * @param value The ID of the customer gateway.
     */
    @JvmName("egctupehlffyovmc")
    public suspend fun customerGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerGatewayId = mapped
    }

    /**
     * @param value Indicate whether to enable acceleration for the VPN connection. Supports only EC2 Transit Gateway.
     */
    @JvmName("ekvdyudbcovjrtxi")
    public suspend fun enableAcceleration(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableAcceleration = mapped
    }

    /**
     * @param value The IPv4 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("ohcaiwpwwolqgoej")
    public suspend fun localIpv4NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.localIpv4NetworkCidr = mapped
    }

    /**
     * @param value The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.
     */
    @JvmName("qwcdkbtipgrafrhd")
    public suspend fun localIpv6NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.localIpv6NetworkCidr = mapped
    }

    /**
     * @param value Indicates if a Public S2S VPN or Private S2S VPN over AWS Direct Connect. Valid values are `PublicIpv4 | PrivateIpv4`
     */
    @JvmName("obehvgxjcrutmllw")
    public suspend fun outsideIpAddressType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.outsideIpAddressType = mapped
    }

    /**
     * @param value The IPv4 CIDR on the AWS side of the VPN connection.
     */
    @JvmName("jigcujibvufvsjto")
    public suspend fun remoteIpv4NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.remoteIpv4NetworkCidr = mapped
    }

    /**
     * @param value The IPv6 CIDR on the AWS side of the VPN connection.
     */
    @JvmName("rjkraweqwhqpbokd")
    public suspend fun remoteIpv6NetworkCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.remoteIpv6NetworkCidr = mapped
    }

    /**
     * @param value Whether the VPN connection uses static routes exclusively. Static routes must be used for devices that don't support BGP.
     */
    @JvmName("ulxxntdgrpbwyruv")
    public suspend fun staticRoutesOnly(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.staticRoutesOnly = mapped
    }

    /**
     * @param value Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("drykfgmqpihkyjbt")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Tags to apply to the connection. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ahadinfjdjbuuibs")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The ID of the EC2 Transit Gateway.
     */
    @JvmName("xptsspvjnjagiaju")
    public suspend fun transitGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transitGatewayId = mapped
    }

    /**
     * @param value . The attachment ID of the Transit Gateway attachment to Direct Connect Gateway. The ID is obtained through a data source only.
     */
    @JvmName("nnwenxfhnfwsttei")
    public suspend fun transportTransitGatewayAttachmentId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.transportTransitGatewayAttachmentId = mapped
    }

    /**
     * @param value The action to take after DPD timeout occurs for the first VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("neepudbntuasklnv")
    public suspend fun tunnel1DpdTimeoutAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1DpdTimeoutAction = mapped
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the first VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("ytoirxxcqquxpvfm")
    public suspend fun tunnel1DpdTimeoutSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1DpdTimeoutSeconds = mapped
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the first VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("jioqgnjirifcaeto")
    public suspend fun tunnel1EnableTunnelLifecycleControl(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1EnableTunnelLifecycleControl = mapped
    }

    /**
     * @param value The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("iqgavvqenknvxiro")
    public suspend fun tunnel1IkeVersions(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1IkeVersions = mapped
    }

    /**
     * @param values The IKE versions that are permitted for the first VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("owmnjymyqptptnvk")
    public suspend fun tunnel1IkeVersions(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1IkeVersions = mapped
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the first VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("syrtjqsqsiljtvxp")
    public suspend fun tunnel1InsideCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1InsideCidr = mapped
    }

    /**
     * @param value The range of inside IPv6 addresses for the first VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("biqklidtdebwqrcv")
    public suspend fun tunnel1InsideIpv6Cidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1InsideIpv6Cidr = mapped
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("qaiivcwascccgpqq")
    public suspend fun tunnel1LogOptions(`value`: VpnConnectionTunnel1LogOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1LogOptions = mapped
    }

    /**
     * @param argument Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("sjlxiiifbqejiwbw")
    public suspend fun tunnel1LogOptions(argument: suspend VpnConnectionTunnel1LogOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = VpnConnectionTunnel1LogOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.tunnel1LogOptions = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("hayvjbvsxagkrmjo")
    public suspend fun tunnel1Phase1DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("ikowbwrswulpikyx")
    public suspend fun tunnel1Phase1DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase1DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("ootpiejwlcpbhpsv")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("hsrbnsvyydcrnxjb")
    public suspend fun tunnel1Phase1EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("xydkucnncfqnsehe")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the first VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("xaausajnbnwcyfin")
    public suspend fun tunnel1Phase1IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("ejhqiumyvndgtgqm")
    public suspend fun tunnel1Phase1LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase1LifetimeSeconds = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("kfasurapfyvmdytv")
    public suspend fun tunnel1Phase2DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("wehkbkuecvfhospr")
    public suspend fun tunnel1Phase2DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase2DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("eijoqigottcfxdiv")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("lchnkumqxnvbrjmw")
    public suspend fun tunnel1Phase2EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("rissdxwvbbifyymj")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the first VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("xursmdqkwonyqpcb")
    public suspend fun tunnel1Phase2IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel1Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the first VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("srxpoknybisjpnxm")
    public suspend fun tunnel1Phase2LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1Phase2LifetimeSeconds = mapped
    }

    /**
     * @param value The preshared key of the first VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("cvonkepyqdhxvqsg")
    public suspend fun tunnel1PresharedKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1PresharedKey = mapped
    }

    /**
     * @param value The percentage of the rekey window for the first VPN tunnel (determined by `tunnel1_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("qqpyqcruqcmrblll")
    public suspend fun tunnel1RekeyFuzzPercentage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1RekeyFuzzPercentage = mapped
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the first VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel1_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel1_phase2_lifetime_seconds`.
     */
    @JvmName("wgbvpaemhncqgmmx")
    public suspend fun tunnel1RekeyMarginTimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1RekeyMarginTimeSeconds = mapped
    }

    /**
     * @param value The number of packets in an IKE replay window for the first VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("dygywdvvjdfdfbqf")
    public suspend fun tunnel1ReplayWindowSize(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1ReplayWindowSize = mapped
    }

    /**
     * @param value The action to take when the establishing the tunnel for the first VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("gsbdmklyjdtyogxy")
    public suspend fun tunnel1StartupAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel1StartupAction = mapped
    }

    /**
     * @param value The action to take after DPD timeout occurs for the second VPN tunnel. Specify restart to restart the IKE initiation. Specify clear to end the IKE session. Valid values are `clear | none | restart`.
     */
    @JvmName("ntjmilabbdowkfeu")
    public suspend fun tunnel2DpdTimeoutAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2DpdTimeoutAction = mapped
    }

    /**
     * @param value The number of seconds after which a DPD timeout occurs for the second VPN tunnel. Valid value is equal or higher than `30`.
     */
    @JvmName("tnthgtstuvkyvave")
    public suspend fun tunnel2DpdTimeoutSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2DpdTimeoutSeconds = mapped
    }

    /**
     * @param value Turn on or off tunnel endpoint lifecycle control feature for the second VPN tunnel. Valid values are `true | false`.
     */
    @JvmName("amvyilwkujyaqiqi")
    public suspend fun tunnel2EnableTunnelLifecycleControl(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2EnableTunnelLifecycleControl = mapped
    }

    /**
     * @param value The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("bbbonquejxdrqfre")
    public suspend fun tunnel2IkeVersions(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2IkeVersions = mapped
    }

    /**
     * @param values The IKE versions that are permitted for the second VPN tunnel. Valid values are `ikev1 | ikev2`.
     */
    @JvmName("yxvkoaiihjlvgvci")
    public suspend fun tunnel2IkeVersions(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2IkeVersions = mapped
    }

    /**
     * @param value The CIDR block of the inside IP addresses for the second VPN tunnel. Valid value is a size /30 CIDR block from the 169.254.0.0/16 range.
     */
    @JvmName("rgtnlwmudnybdjqs")
    public suspend fun tunnel2InsideCidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2InsideCidr = mapped
    }

    /**
     * @param value The range of inside IPv6 addresses for the second VPN tunnel. Supports only EC2 Transit Gateway. Valid value is a size /126 CIDR block from the local fd00::/8 range.
     */
    @JvmName("dmktaruayeqvfmdf")
    public suspend fun tunnel2InsideIpv6Cidr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2InsideIpv6Cidr = mapped
    }

    /**
     * @param value Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("suyffnwhpgvlogeg")
    public suspend fun tunnel2LogOptions(`value`: VpnConnectionTunnel2LogOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2LogOptions = mapped
    }

    /**
     * @param argument Options for logging VPN tunnel activity. See Log Options below for more details.
     */
    @JvmName("dxlvnapgnkcqblds")
    public suspend fun tunnel2LogOptions(argument: suspend VpnConnectionTunnel2LogOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = VpnConnectionTunnel2LogOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.tunnel2LogOptions = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("osoxsrmtkptqubcg")
    public suspend fun tunnel2Phase1DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are ` 2 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("dwiyutjahmerhbjw")
    public suspend fun tunnel2Phase1DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase1DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("fwkvatdaustssuyr")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("brgpehehuvelkcji")
    public suspend fun tunnel2Phase1EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase1EncryptionAlgorithms = mapped
    }

    /**
     * @param value One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("gorxrlrwcdvhmupf")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param values One or more integrity algorithms that are permitted for the second VPN tunnel for phase 1 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("gpsqrkgekllphujb")
    public suspend fun tunnel2Phase1IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase1IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 1 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `28800`.
     */
    @JvmName("ntlqwqghnotadnfk")
    public suspend fun tunnel2Phase1LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase1LifetimeSeconds = mapped
    }

    /**
     * @param value List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("jvviqqtyyugcdnmu")
    public suspend fun tunnel2Phase2DhGroupNumbers(`value`: List<Int>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2DhGroupNumbers = mapped
    }

    /**
     * @param values List of one or more Diffie-Hellman group numbers that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `2 | 5 | 14 | 15 | 16 | 17 | 18 | 19 | 20 | 21 | 22 | 23 | 24`.
     */
    @JvmName("puhqmxocpqwdqdir")
    public suspend fun tunnel2Phase2DhGroupNumbers(vararg values: Int) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase2DhGroupNumbers = mapped
    }

    /**
     * @param value List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("oojaduxnwqoqwesp")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param values List of one or more encryption algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `AES128 | AES256 | AES128-GCM-16 | AES256-GCM-16`.
     */
    @JvmName("hmyyxrogyjvuieae")
    public suspend fun tunnel2Phase2EncryptionAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase2EncryptionAlgorithms = mapped
    }

    /**
     * @param value List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("rsrujbxykityjudn")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param values List of one or more integrity algorithms that are permitted for the second VPN tunnel for phase 2 IKE negotiations. Valid values are `SHA1 | SHA2-256 | SHA2-384 | SHA2-512`.
     */
    @JvmName("pmtlmcgmonplveyr")
    public suspend fun tunnel2Phase2IntegrityAlgorithms(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tunnel2Phase2IntegrityAlgorithms = mapped
    }

    /**
     * @param value The lifetime for phase 2 of the IKE negotiation for the second VPN tunnel, in seconds. Valid value is between `900` and `3600`.
     */
    @JvmName("npirxoxygkwcqjlq")
    public suspend fun tunnel2Phase2LifetimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2Phase2LifetimeSeconds = mapped
    }

    /**
     * @param value The preshared key of the second VPN tunnel. The preshared key must be between 8 and 64 characters in length and cannot start with zero(0). Allowed characters are alphanumeric characters, periods(.) and underscores(_).
     */
    @JvmName("poselfnvtgjbylrn")
    public suspend fun tunnel2PresharedKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2PresharedKey = mapped
    }

    /**
     * @param value The percentage of the rekey window for the second VPN tunnel (determined by `tunnel2_rekey_margin_time_seconds`) during which the rekey time is randomly selected. Valid value is between `0` and `100`.
     */
    @JvmName("ahicdqnbvnfswlmt")
    public suspend fun tunnel2RekeyFuzzPercentage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2RekeyFuzzPercentage = mapped
    }

    /**
     * @param value The margin time, in seconds, before the phase 2 lifetime expires, during which the AWS side of the second VPN connection performs an IKE rekey. The exact time of the rekey is randomly selected based on the value for `tunnel2_rekey_fuzz_percentage`. Valid value is between `60` and half of `tunnel2_phase2_lifetime_seconds`.
     */
    @JvmName("ojndfgdeupxdskvn")
    public suspend fun tunnel2RekeyMarginTimeSeconds(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2RekeyMarginTimeSeconds = mapped
    }

    /**
     * @param value The number of packets in an IKE replay window for the second VPN tunnel. Valid value is between `64` and `2048`.
     */
    @JvmName("feisicyaxsoedknd")
    public suspend fun tunnel2ReplayWindowSize(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2ReplayWindowSize = mapped
    }

    /**
     * @param value The action to take when the establishing the tunnel for the second VPN connection. By default, your customer gateway device must initiate the IKE negotiation and bring up the tunnel. Specify start for AWS to initiate the IKE negotiation. Valid values are `add | start`.
     */
    @JvmName("oryfmesdjudhllal")
    public suspend fun tunnel2StartupAction(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnel2StartupAction = mapped
    }

    /**
     * @param value Indicate whether the VPN tunnels process IPv4 or IPv6 traffic. Valid values are `ipv4 | ipv6`. `ipv6` Supports only EC2 Transit Gateway.
     */
    @JvmName("kxcragprweqtvaou")
    public suspend fun tunnelInsideIpVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tunnelInsideIpVersion = mapped
    }

    /**
     * @param value The type of VPN connection. The only type AWS supports at this time is "ipsec.1".
     */
    @JvmName("vlnbjafwyjcbhxqu")
    public suspend fun type(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.type = mapped
    }

    /**
     * @param value The ID of the Virtual Private Gateway.
     */
    @JvmName("mlooigcdejkhqcue")
    public suspend fun vpnGatewayId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpnGatewayId = mapped
    }

    internal fun build(): VpnConnectionArgs = VpnConnectionArgs(
        customerGatewayId = customerGatewayId,
        enableAcceleration = enableAcceleration,
        localIpv4NetworkCidr = localIpv4NetworkCidr,
        localIpv6NetworkCidr = localIpv6NetworkCidr,
        outsideIpAddressType = outsideIpAddressType,
        remoteIpv4NetworkCidr = remoteIpv4NetworkCidr,
        remoteIpv6NetworkCidr = remoteIpv6NetworkCidr,
        staticRoutesOnly = staticRoutesOnly,
        tags = tags,
        transitGatewayId = transitGatewayId,
        transportTransitGatewayAttachmentId = transportTransitGatewayAttachmentId,
        tunnel1DpdTimeoutAction = tunnel1DpdTimeoutAction,
        tunnel1DpdTimeoutSeconds = tunnel1DpdTimeoutSeconds,
        tunnel1EnableTunnelLifecycleControl = tunnel1EnableTunnelLifecycleControl,
        tunnel1IkeVersions = tunnel1IkeVersions,
        tunnel1InsideCidr = tunnel1InsideCidr,
        tunnel1InsideIpv6Cidr = tunnel1InsideIpv6Cidr,
        tunnel1LogOptions = tunnel1LogOptions,
        tunnel1Phase1DhGroupNumbers = tunnel1Phase1DhGroupNumbers,
        tunnel1Phase1EncryptionAlgorithms = tunnel1Phase1EncryptionAlgorithms,
        tunnel1Phase1IntegrityAlgorithms = tunnel1Phase1IntegrityAlgorithms,
        tunnel1Phase1LifetimeSeconds = tunnel1Phase1LifetimeSeconds,
        tunnel1Phase2DhGroupNumbers = tunnel1Phase2DhGroupNumbers,
        tunnel1Phase2EncryptionAlgorithms = tunnel1Phase2EncryptionAlgorithms,
        tunnel1Phase2IntegrityAlgorithms = tunnel1Phase2IntegrityAlgorithms,
        tunnel1Phase2LifetimeSeconds = tunnel1Phase2LifetimeSeconds,
        tunnel1PresharedKey = tunnel1PresharedKey,
        tunnel1RekeyFuzzPercentage = tunnel1RekeyFuzzPercentage,
        tunnel1RekeyMarginTimeSeconds = tunnel1RekeyMarginTimeSeconds,
        tunnel1ReplayWindowSize = tunnel1ReplayWindowSize,
        tunnel1StartupAction = tunnel1StartupAction,
        tunnel2DpdTimeoutAction = tunnel2DpdTimeoutAction,
        tunnel2DpdTimeoutSeconds = tunnel2DpdTimeoutSeconds,
        tunnel2EnableTunnelLifecycleControl = tunnel2EnableTunnelLifecycleControl,
        tunnel2IkeVersions = tunnel2IkeVersions,
        tunnel2InsideCidr = tunnel2InsideCidr,
        tunnel2InsideIpv6Cidr = tunnel2InsideIpv6Cidr,
        tunnel2LogOptions = tunnel2LogOptions,
        tunnel2Phase1DhGroupNumbers = tunnel2Phase1DhGroupNumbers,
        tunnel2Phase1EncryptionAlgorithms = tunnel2Phase1EncryptionAlgorithms,
        tunnel2Phase1IntegrityAlgorithms = tunnel2Phase1IntegrityAlgorithms,
        tunnel2Phase1LifetimeSeconds = tunnel2Phase1LifetimeSeconds,
        tunnel2Phase2DhGroupNumbers = tunnel2Phase2DhGroupNumbers,
        tunnel2Phase2EncryptionAlgorithms = tunnel2Phase2EncryptionAlgorithms,
        tunnel2Phase2IntegrityAlgorithms = tunnel2Phase2IntegrityAlgorithms,
        tunnel2Phase2LifetimeSeconds = tunnel2Phase2LifetimeSeconds,
        tunnel2PresharedKey = tunnel2PresharedKey,
        tunnel2RekeyFuzzPercentage = tunnel2RekeyFuzzPercentage,
        tunnel2RekeyMarginTimeSeconds = tunnel2RekeyMarginTimeSeconds,
        tunnel2ReplayWindowSize = tunnel2ReplayWindowSize,
        tunnel2StartupAction = tunnel2StartupAction,
        tunnelInsideIpVersion = tunnelInsideIpVersion,
        type = type,
        vpnGatewayId = vpnGatewayId,
    )
}
