@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.eks.kotlin

import com.pulumi.aws.eks.AddonArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an EKS add-on.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.eks.Addon("example", {
 *     clusterName: exampleAwsEksCluster.name,
 *     addonName: "vpc-cni",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.eks.Addon("example",
 *     cluster_name=example_aws_eks_cluster["name"],
 *     addon_name="vpc-cni")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Eks.Addon("example", new()
 *     {
 *         ClusterName = exampleAwsEksCluster.Name,
 *         AddonName = "vpc-cni",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := eks.NewAddon(ctx, "example", &eks.AddonArgs{
 * 			ClusterName: pulumi.Any(exampleAwsEksCluster.Name),
 * 			AddonName:   pulumi.String("vpc-cni"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Addon;
 * import com.pulumi.aws.eks.AddonArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Addon("example", AddonArgs.builder()
 *             .clusterName(exampleAwsEksCluster.name())
 *             .addonName("vpc-cni")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:eks:Addon
 *     properties:
 *       clusterName: ${exampleAwsEksCluster.name}
 *       addonName: vpc-cni
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Example Update add-on usage with resolve_conflicts_on_update and PRESERVE
 * `resolve_conflicts_on_update` with `PRESERVE` can be used to retain the config changes applied to the add-on with kubectl while upgrading to a newer version of the add-on.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.eks.Addon("example", {
 *     clusterName: exampleAwsEksCluster.name,
 *     addonName: "coredns",
 *     addonVersion: "v1.10.1-eksbuild.1",
 *     resolveConflictsOnUpdate: "PRESERVE",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.eks.Addon("example",
 *     cluster_name=example_aws_eks_cluster["name"],
 *     addon_name="coredns",
 *     addon_version="v1.10.1-eksbuild.1",
 *     resolve_conflicts_on_update="PRESERVE")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Eks.Addon("example", new()
 *     {
 *         ClusterName = exampleAwsEksCluster.Name,
 *         AddonName = "coredns",
 *         AddonVersion = "v1.10.1-eksbuild.1",
 *         ResolveConflictsOnUpdate = "PRESERVE",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := eks.NewAddon(ctx, "example", &eks.AddonArgs{
 * 			ClusterName:              pulumi.Any(exampleAwsEksCluster.Name),
 * 			AddonName:                pulumi.String("coredns"),
 * 			AddonVersion:             pulumi.String("v1.10.1-eksbuild.1"),
 * 			ResolveConflictsOnUpdate: pulumi.String("PRESERVE"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Addon;
 * import com.pulumi.aws.eks.AddonArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Addon("example", AddonArgs.builder()
 *             .clusterName(exampleAwsEksCluster.name())
 *             .addonName("coredns")
 *             .addonVersion("v1.10.1-eksbuild.1")
 *             .resolveConflictsOnUpdate("PRESERVE")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:eks:Addon
 *     properties:
 *       clusterName: ${exampleAwsEksCluster.name}
 *       addonName: coredns
 *       addonVersion: v1.10.1-eksbuild.1
 *       resolveConflictsOnUpdate: PRESERVE
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Example add-on usage with custom configuration_values
 * Custom add-on configuration can be passed using `configuration_values` as a single JSON string while creating or updating the add-on.
 * > **Note:** `configuration_values` is a single JSON string should match the valid JSON schema for each add-on with specific version.
 * To find the correct JSON schema for each add-on can be extracted using [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html) call.
 * This below is an example for extracting the `configuration_values` schema for `coredns`.
 * ```bash
 *  aws eks describe-addon-configuration \
 *  --addon-name coredns \
 *  --addon-version v1.10.1-eksbuild.1
 * ```
 * Example to create a `coredns` managed addon with custom `configuration_values`.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.eks.Addon("example", {
 *     clusterName: "mycluster",
 *     addonName: "coredns",
 *     addonVersion: "v1.10.1-eksbuild.1",
 *     resolveConflictsOnCreate: "OVERWRITE",
 *     configurationValues: JSON.stringify({
 *         replicaCount: 4,
 *         resources: {
 *             limits: {
 *                 cpu: "100m",
 *                 memory: "150Mi",
 *             },
 *             requests: {
 *                 cpu: "100m",
 *                 memory: "150Mi",
 *             },
 *         },
 *     }),
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * example = aws.eks.Addon("example",
 *     cluster_name="mycluster",
 *     addon_name="coredns",
 *     addon_version="v1.10.1-eksbuild.1",
 *     resolve_conflicts_on_create="OVERWRITE",
 *     configuration_values=json.dumps({
 *         "replicaCount": 4,
 *         "resources": {
 *             "limits": {
 *                 "cpu": "100m",
 *                 "memory": "150Mi",
 *             },
 *             "requests": {
 *                 "cpu": "100m",
 *                 "memory": "150Mi",
 *             },
 *         },
 *     }))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Eks.Addon("example", new()
 *     {
 *         ClusterName = "mycluster",
 *         AddonName = "coredns",
 *         AddonVersion = "v1.10.1-eksbuild.1",
 *         ResolveConflictsOnCreate = "OVERWRITE",
 *         ConfigurationValues = JsonSerializer.Serialize(new Dictionary<string, object?>
 *         {
 *             ["replicaCount"] = 4,
 *             ["resources"] = new Dictionary<string, object?>
 *             {
 *                 ["limits"] = new Dictionary<string, object?>
 *                 {
 *                     ["cpu"] = "100m",
 *                     ["memory"] = "150Mi",
 *                 },
 *                 ["requests"] = new Dictionary<string, object?>
 *                 {
 *                     ["cpu"] = "100m",
 *                     ["memory"] = "150Mi",
 *                 },
 *             },
 *         }),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 			"replicaCount": 4,
 * 			"resources": map[string]interface{}{
 * 				"limits": map[string]interface{}{
 * 					"cpu":    "100m",
 * 					"memory": "150Mi",
 * 				},
 * 				"requests": map[string]interface{}{
 * 					"cpu":    "100m",
 * 					"memory": "150Mi",
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		json0 := string(tmpJSON0)
 * 		_, err = eks.NewAddon(ctx, "example", &eks.AddonArgs{
 * 			ClusterName:              pulumi.String("mycluster"),
 * 			AddonName:                pulumi.String("coredns"),
 * 			AddonVersion:             pulumi.String("v1.10.1-eksbuild.1"),
 * 			ResolveConflictsOnCreate: pulumi.String("OVERWRITE"),
 * 			ConfigurationValues:      pulumi.String(json0),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Addon;
 * import com.pulumi.aws.eks.AddonArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Addon("example", AddonArgs.builder()
 *             .clusterName("mycluster")
 *             .addonName("coredns")
 *             .addonVersion("v1.10.1-eksbuild.1")
 *             .resolveConflictsOnCreate("OVERWRITE")
 *             .configurationValues(serializeJson(
 *                 jsonObject(
 *                     jsonProperty("replicaCount", 4),
 *                     jsonProperty("resources", jsonObject(
 *                         jsonProperty("limits", jsonObject(
 *                             jsonProperty("cpu", "100m"),
 *                             jsonProperty("memory", "150Mi")
 *                         )),
 *                         jsonProperty("requests", jsonObject(
 *                             jsonProperty("cpu", "100m"),
 *                             jsonProperty("memory", "150Mi")
 *                         ))
 *                     ))
 *                 )))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:eks:Addon
 *     properties:
 *       clusterName: mycluster
 *       addonName: coredns
 *       addonVersion: v1.10.1-eksbuild.1
 *       resolveConflictsOnCreate: OVERWRITE
 *       configurationValues:
 *         fn::toJSON:
 *           replicaCount: 4
 *           resources:
 *             limits:
 *               cpu: 100m
 *               memory: 150Mi
 *             requests:
 *               cpu: 100m
 *               memory: 150Mi
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Example IAM Role for EKS Addon "vpc-cni" with AWS managed policy
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * import * as tls from "@pulumi/tls";
 * const exampleCluster = new aws.eks.Cluster("example", {});
 * const example = exampleCluster.identities.apply(identities => tls.getCertificateOutput({
 *     url: identities[0]&#46;oidcs?&#46;[0]?.issuer,
 * }));
 * const exampleOpenIdConnectProvider = new aws.iam.OpenIdConnectProvider("example", {
 *     clientIdLists: ["sts&#46;amazonaws&#46;com"],
 *     thumbprintLists: [example&#46;apply(example => example&#46;certificates?&#46;[0]?&#46;sha1Fingerprint)],
 *     url: exampleCluster.identities.apply(identities => identities[0]&#46;oidcs?&#46;[0]?.issuer),
 * });
 * const exampleAssumeRolePolicy = aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         actions: ["sts:AssumeRoleWithWebIdentity"],
 *         effect: "Allow",
 *         conditions: [{
 *             test: "StringEquals",
 *             variable: std.replaceOutput({
 *                 text: exampleOpenIdConnectProvider.url,
 *                 search: "https://",
 *                 replace: "",
 *             }).apply(invoke => `${invoke.result}:sub`),
 *             values: ["system:serviceaccount:kube-system:aws-node"],
 *         }],
 *         principals: [{
 *             identifiers: [exampleOpenIdConnectProvider&#46;arn],
 *             type: "Federated",
 *         }],
 *     }],
 * });
 * const exampleRole = new aws.iam.Role("example", {
 *     assumeRolePolicy: exampleAssumeRolePolicy.apply(exampleAssumeRolePolicy => exampleAssumeRolePolicy.json),
 *     name: "example-vpc-cni-role",
 * });
 * const exampleRolePolicyAttachment = new aws.iam.RolePolicyAttachment("example", {
 *     policyArn: "arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy",
 *     role: exampleRole.name,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * import pulumi_tls as tls
 * example_cluster = aws.eks.Cluster("example")
 * example = example_cluster.identities.apply(lambda identities: tls.get_certificate_output(url=identities[0]&#46;oidcs[0].issuer))
 * example_open_id_connect_provider = aws.iam.OpenIdConnectProvider("example",
 *     client_id_lists=["sts&#46;amazonaws&#46;com"],
 *     thumbprint_lists=[example&#46;certificates[0]&#46;sha1_fingerprint],
 *     url=example_cluster.identities[0]&#46;oidcs[0].issuer)
 * example_assume_role_policy = aws.iam.get_policy_document_output(statements=[{
 *     "actions": ["sts:AssumeRoleWithWebIdentity"],
 *     "effect": "Allow",
 *     "conditions": [{
 *         "test": "StringEquals",
 *         "variable": std.replace_output(text=example_open_id_connect_provider.url,
 *             search="https://",
 *             replace="").apply(lambda invoke: f"{invoke.result}:sub"),
 *         "values": ["system:serviceaccount:kube-system:aws-node"],
 *     }],
 *     "principals": [{
 *         "identifiers": [example_open_id_connect_provider&#46;arn],
 *         "type": "Federated",
 *     }],
 * }])
 * example_role = aws.iam.Role("example",
 *     assume_role_policy=example_assume_role_policy.json,
 *     name="example-vpc-cni-role")
 * example_role_policy_attachment = aws.iam.RolePolicyAttachment("example",
 *     policy_arn="arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy",
 *     role=example_role.name)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * using Tls = Pulumi.Tls;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleCluster = new Aws.Eks.Cluster("example");
 *     var example = Tls.GetCertificate.Invoke(new()
 *     {
 *         Url = exampleCluster.Identities[0]&#46;Oidcs[0]?.Issuer,
 *     });
 *     var exampleOpenIdConnectProvider = new Aws.Iam.OpenIdConnectProvider("example", new()
 *     {
 *         ClientIdLists = new[]
 *         {
 *             "sts.amazonaws.com",
 *         },
 *         ThumbprintLists = new[]
 *         {
 *             example.Apply(getCertificateResult => getCertificateResult.Certificates[0]?.Sha1Fingerprint),
 *         },
 *         Url = exampleCluster.Identities.Apply(identities => identities[0]&#46;Oidcs[0]?.Issuer),
 *     });
 *     var exampleAssumeRolePolicy = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRoleWithWebIdentity",
 *                 },
 *                 Effect = "Allow",
 *                 Conditions = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementConditionInputArgs
 *                     {
 *                         Test = "StringEquals",
 *                         Variable = $"{Std.Replace.Invoke(new()
 *                         {
 *                             Text = exampleOpenIdConnectProvider.Url,
 *                             Search = "https://",
 *                             Replace = "",
 *                         }).Result}:sub",
 *                         Values = new[]
 *                         {
 *                             "system:serviceaccount:kube-system:aws-node",
 *                         },
 *                     },
 *                 },
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Identifiers = new[]
 *                         {
 *                             exampleOpenIdConnectProvider.Arn,
 *                         },
 *                         Type = "Federated",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     var exampleRole = new Aws.Iam.Role("example", new()
 *     {
 *         AssumeRolePolicy = exampleAssumeRolePolicy.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *         Name = "example-vpc-cni-role",
 *     });
 *     var exampleRolePolicyAttachment = new Aws.Iam.RolePolicyAttachment("example", new()
 *     {
 *         PolicyArn = "arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy",
 *         Role = exampleRole.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi-tls/sdk/v4/go/tls"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		exampleCluster, err := eks.NewCluster(ctx, "example", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example := exampleCluster.Identities.ApplyT(func(identities []eks.ClusterIdentity) (tls.GetCertificateResult, error) {
 * 			return tls.GetCertificateResult(interface{}(tls.GetCertificateOutput(ctx, tls.GetCertificateOutputArgs{
 * 				Url: identities[0]&#46;Oidcs[0].Issuer,
 * 			}, nil))), nil
 * 		}).(tls.GetCertificateResultOutput)
 * 		exampleOpenIdConnectProvider, err := iam.NewOpenIdConnectProvider(ctx, "example", &iam.OpenIdConnectProviderArgs{
 * 			ClientIdLists: pulumi.StringArray{
 * 				pulumi.String("sts.amazonaws.com"),
 * 			},
 * 			ThumbprintLists: pulumi.StringArray{
 * 				pulumi.String(example.ApplyT(func(example tls.GetCertificateResult) (*string, error) {
 * 					return &example.Certificates[0].Sha1Fingerprint, nil
 * 				}).(pulumi.StringPtrOutput)),
 * 			},
 * 			Url: pulumi.String(exampleCluster.Identities.ApplyT(func(identities []eks.ClusterIdentity) (*string, error) {
 * 				return &identities[0]&#46;Oidcs[0].Issuer, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleAssumeRolePolicy := iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * 			Statements: iam.GetPolicyDocumentStatementArray{
 * 				&iam.GetPolicyDocumentStatementArgs{
 * 					Actions: pulumi.StringArray{
 * 						pulumi.String("sts:AssumeRoleWithWebIdentity"),
 * 					},
 * 					Effect: pulumi.String("Allow"),
 * 					Conditions: iam.GetPolicyDocumentStatementConditionArray{
 * 						&iam.GetPolicyDocumentStatementConditionArgs{
 * 							Test: pulumi.String("StringEquals"),
 * 							Variable: std.ReplaceOutput(ctx, std.ReplaceOutputArgs{
 * 								Text:    exampleOpenIdConnectProvider.Url,
 * 								Search:  pulumi.String("https://"),
 * 								Replace: pulumi.String(""),
 * 							}, nil).ApplyT(func(invoke std.ReplaceResult) (string, error) {
 * 								return fmt.Sprintf("%v:sub", invoke.Result), nil
 * 							}).(pulumi.StringOutput),
 * 							Values: pulumi.StringArray{
 * 								pulumi.String("system:serviceaccount:kube-system:aws-node"),
 * 							},
 * 						},
 * 					},
 * 					Principals: iam.GetPolicyDocumentStatementPrincipalArray{
 * 						&iam.GetPolicyDocumentStatementPrincipalArgs{
 * 							Identifiers: pulumi.StringArray{
 * 								exampleOpenIdConnectProvider.Arn,
 * 							},
 * 							Type: pulumi.String("Federated"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		exampleRole, err := iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			AssumeRolePolicy: pulumi.String(exampleAssumeRolePolicy.ApplyT(func(exampleAssumeRolePolicy iam.GetPolicyDocumentResult) (*string, error) {
 * 				return &exampleAssumeRolePolicy.Json, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 			Name: pulumi.String("example-vpc-cni-role"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iam.NewRolePolicyAttachment(ctx, "example", &iam.RolePolicyAttachmentArgs{
 * 			PolicyArn: pulumi.String("arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy"),
 * 			Role:      exampleRole.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.tls.TlsFunctions;
 * import com.pulumi.tls.inputs.GetCertificateArgs;
 * import com.pulumi.aws.iam.OpenIdConnectProvider;
 * import com.pulumi.aws.iam.OpenIdConnectProviderArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleCluster = new Cluster("exampleCluster");
 *         final var example = TlsFunctions.getCertificate(GetCertificateArgs.builder()
 *             .url(exampleCluster.identities().applyValue(identities -> identities[0]&#46;oidcs()[0].issuer()))
 *             .build());
 *         var exampleOpenIdConnectProvider = new OpenIdConnectProvider("exampleOpenIdConnectProvider", OpenIdConnectProviderArgs.builder()
 *             .clientIdLists("sts.amazonaws.com")
 *             .thumbprintLists(example.applyValue(getCertificateResult -> getCertificateResult).applyValue(example -> example.applyValue(getCertificateResult -> getCertificateResult.certificates()[0].sha1Fingerprint())))
 *             .url(exampleCluster.identities().applyValue(identities -> identities[0]&#46;oidcs()[0].issuer()))
 *             .build());
 *         final var exampleAssumeRolePolicy = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .actions("sts:AssumeRoleWithWebIdentity")
 *                 .effect("Allow")
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("StringEquals")
 *                     .variable(StdFunctions.replace().applyValue(invoke -> String.format("%s:sub", invoke.result())))
 *                     .values("system:serviceaccount:kube-system:aws-node")
 *                     .build())
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .identifiers(exampleOpenIdConnectProvider.arn())
 *                     .type("Federated")
 *                     .build())
 *                 .build())
 *             .build());
 *         var exampleRole = new Role("exampleRole", RoleArgs.builder()
 *             .assumeRolePolicy(exampleAssumeRolePolicy.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(exampleAssumeRolePolicy -> exampleAssumeRolePolicy.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .name("example-vpc-cni-role")
 *             .build());
 *         var exampleRolePolicyAttachment = new RolePolicyAttachment("exampleRolePolicyAttachment", RolePolicyAttachmentArgs.builder()
 *             .policyArn("arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy")
 *             .role(exampleRole.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleCluster:
 *     type: aws:eks:Cluster
 *     name: example
 *   exampleOpenIdConnectProvider:
 *     type: aws:iam:OpenIdConnectProvider
 *     name: example
 *     properties:
 *       clientIdLists:
 *         - sts.amazonaws.com
 *       thumbprintLists:
 *         - ${example.certificates[0].sha1Fingerprint}
 *       url: ${exampleCluster.identities[0]&#46;oidcs[0].issuer}
 *   exampleRole:
 *     type: aws:iam:Role
 *     name: example
 *     properties:
 *       assumeRolePolicy: ${exampleAssumeRolePolicy.json}
 *       name: example-vpc-cni-role
 *   exampleRolePolicyAttachment:
 *     type: aws:iam:RolePolicyAttachment
 *     name: example
 *     properties:
 *       policyArn: arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy
 *       role: ${exampleRole.name}
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: tls:getCertificate
 *       Arguments:
 *         url: ${exampleCluster.identities[0]&#46;oidcs[0].issuer}
 *   exampleAssumeRolePolicy:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - actions:
 *               - sts:AssumeRoleWithWebIdentity
 *             effect: Allow
 *             conditions:
 *               - test: StringEquals
 *                 variable:
 *                   fn::join:
 *                     -
 *                     - - fn::invoke:
 *                           Function: std:replace
 *                           Arguments:
 *                             text: ${exampleOpenIdConnectProvider.url}
 *                             search: https://
 *                             replace:
 *                           Return: result
 *                       - :sub
 *                 values:
 *                   - system:serviceaccount:kube-system:aws-node
 *             principals:
 *               - identifiers:
 *                   - ${exampleOpenIdConnectProvider.arn}
 *                 type: Federated
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import EKS add-on using the `cluster_name` and `addon_name` separated by a colon (`:`). For example:
 * ```sh
 * $ pulumi import aws:eks/addon:Addon my_eks_addon my_cluster_name:my_addon_name
 * ```
 * @property addonName Name of the EKS add-on. The name must match one of
 * the names returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
 * @property addonVersion The version of the EKS add-on. The version must
 * match one of the versions returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
 * @property clusterName Name of the EKS Cluster.
 * The following arguments are optional:
 * @property configurationValues custom configuration values for addons with single JSON string. This JSON string value must match the JSON schema derived from [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html).
 * @property preserve Indicates if you want to preserve the created resources when deleting the EKS add-on.
 * @property resolveConflicts Define how to resolve parameter value conflicts when migrating an existing add-on to an Amazon EKS add-on or when applying version updates to the add-on. Valid values are `NONE`, `OVERWRITE` and `PRESERVE`. Note that `PRESERVE` is only valid on addon update, not for initial addon creation. If you need to set this to `PRESERVE`, use the `resolve_conflicts_on_create` and `resolve_conflicts_on_update` attributes instead. For more details check [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
 * @property resolveConflictsOnCreate How to resolve field value conflicts when migrating a self-managed add-on to an Amazon EKS add-on. Valid values are `NONE` and `OVERWRITE`. For more details see the [CreateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html) API Docs.
 * @property resolveConflictsOnUpdate How to resolve field value conflicts for an Amazon EKS add-on if you've changed a value from the Amazon EKS default value. Valid values are `NONE`, `OVERWRITE`, and `PRESERVE`. For more details see the [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
 * @property serviceAccountRoleArn The Amazon Resource Name (ARN) of an
 * existing IAM role to bind to the add-on's service account. The role must be
 * assigned the IAM permissions required by the add-on. If you don't specify
 * an existing IAM role, then the add-on uses the permissions assigned to the node
 * IAM role. For more information, see [Amazon EKS node IAM role](https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html)
 * in the Amazon EKS User Guide.
 * > **Note:** To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC)
 * provider created for your cluster. For more information, [see Enabling IAM roles
 * for service accounts on your cluster](https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html)
 * in the Amazon EKS User Guide.
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class AddonArgs(
    public val addonName: Output<String>? = null,
    public val addonVersion: Output<String>? = null,
    public val clusterName: Output<String>? = null,
    public val configurationValues: Output<String>? = null,
    public val preserve: Output<Boolean>? = null,
    @Deprecated(
        message = """
  The "resolve_conflicts" attribute can't be set to "PRESERVE" on initial resource creation. Use
      "resolve_conflicts_on_create" and/or "resolve_conflicts_on_update" instead
  """,
    )
    public val resolveConflicts: Output<String>? = null,
    public val resolveConflictsOnCreate: Output<String>? = null,
    public val resolveConflictsOnUpdate: Output<String>? = null,
    public val serviceAccountRoleArn: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.eks.AddonArgs> {
    override fun toJava(): com.pulumi.aws.eks.AddonArgs = com.pulumi.aws.eks.AddonArgs.builder()
        .addonName(addonName?.applyValue({ args0 -> args0 }))
        .addonVersion(addonVersion?.applyValue({ args0 -> args0 }))
        .clusterName(clusterName?.applyValue({ args0 -> args0 }))
        .configurationValues(configurationValues?.applyValue({ args0 -> args0 }))
        .preserve(preserve?.applyValue({ args0 -> args0 }))
        .resolveConflicts(resolveConflicts?.applyValue({ args0 -> args0 }))
        .resolveConflictsOnCreate(resolveConflictsOnCreate?.applyValue({ args0 -> args0 }))
        .resolveConflictsOnUpdate(resolveConflictsOnUpdate?.applyValue({ args0 -> args0 }))
        .serviceAccountRoleArn(serviceAccountRoleArn?.applyValue({ args0 -> args0 }))
        .tags(
            tags?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }),
        ).build()
}

/**
 * Builder for [AddonArgs].
 */
@PulumiTagMarker
public class AddonArgsBuilder internal constructor() {
    private var addonName: Output<String>? = null

    private var addonVersion: Output<String>? = null

    private var clusterName: Output<String>? = null

    private var configurationValues: Output<String>? = null

    private var preserve: Output<Boolean>? = null

    private var resolveConflicts: Output<String>? = null

    private var resolveConflictsOnCreate: Output<String>? = null

    private var resolveConflictsOnUpdate: Output<String>? = null

    private var serviceAccountRoleArn: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value Name of the EKS add-on. The name must match one of
     * the names returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("hhshlobajmjftcft")
    public suspend fun addonName(`value`: Output<String>) {
        this.addonName = value
    }

    /**
     * @param value The version of the EKS add-on. The version must
     * match one of the versions returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("wgybrcrelaiuimxh")
    public suspend fun addonVersion(`value`: Output<String>) {
        this.addonVersion = value
    }

    /**
     * @param value Name of the EKS Cluster.
     * The following arguments are optional:
     */
    @JvmName("frxeqwecnaggajia")
    public suspend fun clusterName(`value`: Output<String>) {
        this.clusterName = value
    }

    /**
     * @param value custom configuration values for addons with single JSON string. This JSON string value must match the JSON schema derived from [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html).
     */
    @JvmName("snhrrknxiiunjvtd")
    public suspend fun configurationValues(`value`: Output<String>) {
        this.configurationValues = value
    }

    /**
     * @param value Indicates if you want to preserve the created resources when deleting the EKS add-on.
     */
    @JvmName("qlghwpyniviogqef")
    public suspend fun preserve(`value`: Output<Boolean>) {
        this.preserve = value
    }

    /**
     * @param value Define how to resolve parameter value conflicts when migrating an existing add-on to an Amazon EKS add-on or when applying version updates to the add-on. Valid values are `NONE`, `OVERWRITE` and `PRESERVE`. Note that `PRESERVE` is only valid on addon update, not for initial addon creation. If you need to set this to `PRESERVE`, use the `resolve_conflicts_on_create` and `resolve_conflicts_on_update` attributes instead. For more details check [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @Deprecated(
        message = """
  The "resolve_conflicts" attribute can't be set to "PRESERVE" on initial resource creation. Use
      "resolve_conflicts_on_create" and/or "resolve_conflicts_on_update" instead
  """,
    )
    @JvmName("rchbumxrttxkxmrd")
    public suspend fun resolveConflicts(`value`: Output<String>) {
        this.resolveConflicts = value
    }

    /**
     * @param value How to resolve field value conflicts when migrating a self-managed add-on to an Amazon EKS add-on. Valid values are `NONE` and `OVERWRITE`. For more details see the [CreateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html) API Docs.
     */
    @JvmName("mywjemxpqqobphjg")
    public suspend fun resolveConflictsOnCreate(`value`: Output<String>) {
        this.resolveConflictsOnCreate = value
    }

    /**
     * @param value How to resolve field value conflicts for an Amazon EKS add-on if you've changed a value from the Amazon EKS default value. Valid values are `NONE`, `OVERWRITE`, and `PRESERVE`. For more details see the [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @JvmName("cpgutkqoumuqddom")
    public suspend fun resolveConflictsOnUpdate(`value`: Output<String>) {
        this.resolveConflictsOnUpdate = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of an
     * existing IAM role to bind to the add-on's service account. The role must be
     * assigned the IAM permissions required by the add-on. If you don't specify
     * an existing IAM role, then the add-on uses the permissions assigned to the node
     * IAM role. For more information, see [Amazon EKS node IAM role](https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html)
     * in the Amazon EKS User Guide.
     * > **Note:** To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC)
     * provider created for your cluster. For more information, [see Enabling IAM roles
   * for service accounts on your cluster](https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html)
     * in the Amazon EKS User Guide.
     */
    @JvmName("rhajflatknwotlos")
    public suspend fun serviceAccountRoleArn(`value`: Output<String>) {
        this.serviceAccountRoleArn = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ghjmtmaxwrohyooa")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Name of the EKS add-on. The name must match one of
     * the names returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("mfqeydtgidhviviw")
    public suspend fun addonName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.addonName = mapped
    }

    /**
     * @param value The version of the EKS add-on. The version must
     * match one of the versions returned by [describe-addon-versions](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-versions.html).
     */
    @JvmName("usqaedifdcoyjhik")
    public suspend fun addonVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.addonVersion = mapped
    }

    /**
     * @param value Name of the EKS Cluster.
     * The following arguments are optional:
     */
    @JvmName("kighmflxvsedfrcn")
    public suspend fun clusterName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterName = mapped
    }

    /**
     * @param value custom configuration values for addons with single JSON string. This JSON string value must match the JSON schema derived from [describe-addon-configuration](https://docs.aws.amazon.com/cli/latest/reference/eks/describe-addon-configuration.html).
     */
    @JvmName("sjsyhvrmloddeque")
    public suspend fun configurationValues(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.configurationValues = mapped
    }

    /**
     * @param value Indicates if you want to preserve the created resources when deleting the EKS add-on.
     */
    @JvmName("rttrlddspbvhtbqk")
    public suspend fun preserve(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preserve = mapped
    }

    /**
     * @param value Define how to resolve parameter value conflicts when migrating an existing add-on to an Amazon EKS add-on or when applying version updates to the add-on. Valid values are `NONE`, `OVERWRITE` and `PRESERVE`. Note that `PRESERVE` is only valid on addon update, not for initial addon creation. If you need to set this to `PRESERVE`, use the `resolve_conflicts_on_create` and `resolve_conflicts_on_update` attributes instead. For more details check [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @Deprecated(
        message = """
  The "resolve_conflicts" attribute can't be set to "PRESERVE" on initial resource creation. Use
      "resolve_conflicts_on_create" and/or "resolve_conflicts_on_update" instead
  """,
    )
    @JvmName("bvadbtivevqygewh")
    public suspend fun resolveConflicts(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resolveConflicts = mapped
    }

    /**
     * @param value How to resolve field value conflicts when migrating a self-managed add-on to an Amazon EKS add-on. Valid values are `NONE` and `OVERWRITE`. For more details see the [CreateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html) API Docs.
     */
    @JvmName("pbmvxcwftcncyckl")
    public suspend fun resolveConflictsOnCreate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resolveConflictsOnCreate = mapped
    }

    /**
     * @param value How to resolve field value conflicts for an Amazon EKS add-on if you've changed a value from the Amazon EKS default value. Valid values are `NONE`, `OVERWRITE`, and `PRESERVE`. For more details see the [UpdateAddon](https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html) API Docs.
     */
    @JvmName("xflwxphygbwxmswj")
    public suspend fun resolveConflictsOnUpdate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resolveConflictsOnUpdate = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of an
     * existing IAM role to bind to the add-on's service account. The role must be
     * assigned the IAM permissions required by the add-on. If you don't specify
     * an existing IAM role, then the add-on uses the permissions assigned to the node
     * IAM role. For more information, see [Amazon EKS node IAM role](https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html)
     * in the Amazon EKS User Guide.
     * > **Note:** To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC)
     * provider created for your cluster. For more information, [see Enabling IAM roles
   * for service accounts on your cluster](https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html)
     * in the Amazon EKS User Guide.
     */
    @JvmName("wgdnawxqwqwpoaqa")
    public suspend fun serviceAccountRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serviceAccountRoleArn = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("nfsmuliefqwyaspt")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("pxqcuaieijocofek")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): AddonArgs = AddonArgs(
        addonName = addonName,
        addonVersion = addonVersion,
        clusterName = clusterName,
        configurationValues = configurationValues,
        preserve = preserve,
        resolveConflicts = resolveConflicts,
        resolveConflictsOnCreate = resolveConflictsOnCreate,
        resolveConflictsOnUpdate = resolveConflictsOnUpdate,
        serviceAccountRoleArn = serviceAccountRoleArn,
        tags = tags,
    )
}
