@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.eks.kotlin

import com.pulumi.aws.eks.ClusterArgs.builder
import com.pulumi.aws.eks.kotlin.inputs.ClusterAccessConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterAccessConfigArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterEncryptionConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterEncryptionConfigArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterKubernetesNetworkConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterKubernetesNetworkConfigArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterOutpostConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterOutpostConfigArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterUpgradePolicyArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterUpgradePolicyArgsBuilder
import com.pulumi.aws.eks.kotlin.inputs.ClusterVpcConfigArgs
import com.pulumi.aws.eks.kotlin.inputs.ClusterVpcConfigArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an EKS Cluster.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * export = async () => {
 *     const example = new aws.eks.Cluster("example", {
 *         name: "example",
 *         roleArn: exampleAwsIamRole.arn,
 *         vpcConfig: {
 *             subnetIds: [
 *                 example1.id,
 *                 example2.id,
 *             ],
 *         },
 *     }, {
 *         dependsOn: [
 *             example_AmazonEKSClusterPolicy,
 *             example_AmazonEKSVPCResourceController,
 *         ],
 *     });
 *     return {
 *         endpoint: example.endpoint,
 *         "kubeconfig-certificate-authority-data": example.certificateAuthority.apply(certificateAuthority => certificateAuthority.data),
 *     };
 * }
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.eks.Cluster("example",
 *     name="example",
 *     role_arn=example_aws_iam_role["arn"],
 *     vpc_config={
 *         "subnet_ids": [
 *             example1["id"],
 *             example2["id"],
 *         ],
 *     },
 *     opts = pulumi.ResourceOptions(depends_on=[
 *             example__amazon_eks_cluster_policy,
 *             example__amazon_eksvpc_resource_controller,
 *         ]))
 * pulumi.export("endpoint", example.endpoint)
 * pulumi.export("kubeconfig-certificate-authority-data", example.certificate_authority.data)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Eks.Cluster("example", new()
 *     {
 *         Name = "example",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         VpcConfig = new Aws.Eks.Inputs.ClusterVpcConfigArgs
 *         {
 *             SubnetIds = new[]
 *             {
 *                 example1.Id,
 *                 example2.Id,
 *             },
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             example_AmazonEKSClusterPolicy,
 *             example_AmazonEKSVPCResourceController,
 *         },
 *     });
 *     return new Dictionary<string, object?>
 *     {
 *         ["endpoint"] = example.Endpoint,
 *         ["kubeconfig-certificate-authority-data"] = example.CertificateAuthority.Apply(certificateAuthority => certificateAuthority.Data),
 *     };
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := eks.NewCluster(ctx, "example", &eks.ClusterArgs{
 * 			Name:    pulumi.String("example"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			VpcConfig: &eks.ClusterVpcConfigArgs{
 * 				SubnetIds: pulumi.StringArray{
 * 					example1.Id,
 * 					example2.Id,
 * 				},
 * 			},
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			example_AmazonEKSClusterPolicy,
 * 			example_AmazonEKSVPCResourceController,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		ctx.Export("endpoint", example.Endpoint)
 * 		ctx.Export("kubeconfig-certificate-authority-data", example.CertificateAuthority.ApplyT(func(certificateAuthority eks.ClusterCertificateAuthority) (*string, error) {
 * 			return &certificateAuthority.Data, nil
 * 		}).(pulumi.StringPtrOutput))
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.aws.eks.ClusterArgs;
 * import com.pulumi.aws.eks.inputs.ClusterVpcConfigArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .name("example")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .vpcConfig(ClusterVpcConfigArgs.builder()
 *                 .subnetIds(
 *                     example1.id(),
 *                     example2.id())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(
 *                     example_AmazonEKSClusterPolicy,
 *                     example_AmazonEKSVPCResourceController)
 *                 .build());
 *         ctx.export("endpoint", example.endpoint());
 *         ctx.export("kubeconfig-certificate-authority-data", example.certificateAuthority().applyValue(certificateAuthority -> certificateAuthority.data()));
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:eks:Cluster
 *     properties:
 *       name: example
 *       roleArn: ${exampleAwsIamRole.arn}
 *       vpcConfig:
 *         subnetIds:
 *           - ${example1.id}
 *           - ${example2.id}
 *     options:
 *       dependson:
 *         - ${["example-AmazonEKSClusterPolicy"]}
 *         - ${["example-AmazonEKSVPCResourceController"]}
 * outputs:
 *   endpoint: ${example.endpoint}
 *   kubeconfig-certificate-authority-data: ${example.certificateAuthority.data}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Example IAM Role for EKS Cluster
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const assumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["eks&#46;amazonaws&#46;com"],
 *         }],
 *         actions: ["sts:AssumeRole"],
 *     }],
 * });
 * const example = new aws.iam.Role("example", {
 *     name: "eks-cluster-example",
 *     assumeRolePolicy: assumeRole.then(assumeRole => assumeRole.json),
 * });
 * const example_AmazonEKSClusterPolicy = new aws.iam.RolePolicyAttachment("example-AmazonEKSClusterPolicy", {
 *     policyArn: "arn:aws:iam::aws:policy/AmazonEKSClusterPolicy",
 *     role: example.name,
 * });
 * // Optionally, enable Security Groups for Pods
 * // Reference: https://docs.aws.amazon.com/eks/latest/userguide/security-groups-for-pods.html
 * const example_AmazonEKSVPCResourceController = new aws.iam.RolePolicyAttachment("example-AmazonEKSVPCResourceController", {
 *     policyArn: "arn:aws:iam::aws:policy/AmazonEKSVPCResourceController",
 *     role: example.name,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["eks&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": ["sts:AssumeRole"],
 * }])
 * example = aws.iam.Role("example",
 *     name="eks-cluster-example",
 *     assume_role_policy=assume_role.json)
 * example__amazon_eks_cluster_policy = aws.iam.RolePolicyAttachment("example-AmazonEKSClusterPolicy",
 *     policy_arn="arn:aws:iam::aws:policy/AmazonEKSClusterPolicy",
 *     role=example.name)
 * # Optionally, enable Security Groups for Pods
 * # Reference: https://docs.aws.amazon.com/eks/latest/userguide/security-groups-for-pods.html
 * example__amazon_eksvpc_resource_controller = aws.iam.RolePolicyAttachment("example-AmazonEKSVPCResourceController",
 *     policy_arn="arn:aws:iam::aws:policy/AmazonEKSVPCResourceController",
 *     role=example.name)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var assumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "eks.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *             },
 *         },
 *     });
 *     var example = new Aws.Iam.Role("example", new()
 *     {
 *         Name = "eks-cluster-example",
 *         AssumeRolePolicy = assumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var example_AmazonEKSClusterPolicy = new Aws.Iam.RolePolicyAttachment("example-AmazonEKSClusterPolicy", new()
 *     {
 *         PolicyArn = "arn:aws:iam::aws:policy/AmazonEKSClusterPolicy",
 *         Role = example.Name,
 *     });
 *     // Optionally, enable Security Groups for Pods
 *     // Reference: https://docs.aws.amazon.com/eks/latest/userguide/security-groups-for-pods.html
 *     var example_AmazonEKSVPCResourceController = new Aws.Iam.RolePolicyAttachment("example-AmazonEKSVPCResourceController", new()
 *     {
 *         PolicyArn = "arn:aws:iam::aws:policy/AmazonEKSVPCResourceController",
 *         Role = example.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		assumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"eks.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			Name:             pulumi.String("eks-cluster-example"),
 * 			AssumeRolePolicy: pulumi.String(assumeRole.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iam.NewRolePolicyAttachment(ctx, "example-AmazonEKSClusterPolicy", &iam.RolePolicyAttachmentArgs{
 * 			PolicyArn: pulumi.String("arn:aws:iam::aws:policy/AmazonEKSClusterPolicy"),
 * 			Role:      example.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Optionally, enable Security Groups for Pods
 * 		// Reference: https://docs.aws.amazon.com/eks/latest/userguide/security-groups-for-pods.html
 * 		_, err = iam.NewRolePolicyAttachment(ctx, "example-AmazonEKSVPCResourceController", &iam.RolePolicyAttachmentArgs{
 * 			PolicyArn: pulumi.String("arn:aws:iam::aws:policy/AmazonEKSVPCResourceController"),
 * 			Role:      example.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("eks.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var example = new Role("example", RoleArgs.builder()
 *             .name("eks-cluster-example")
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var example_AmazonEKSClusterPolicy = new RolePolicyAttachment("example-AmazonEKSClusterPolicy", RolePolicyAttachmentArgs.builder()
 *             .policyArn("arn:aws:iam::aws:policy/AmazonEKSClusterPolicy")
 *             .role(example.name())
 *             .build());
 *         // Optionally, enable Security Groups for Pods
 *         // Reference: https://docs.aws.amazon.com/eks/latest/userguide/security-groups-for-pods.html
 *         var example_AmazonEKSVPCResourceController = new RolePolicyAttachment("example-AmazonEKSVPCResourceController", RolePolicyAttachmentArgs.builder()
 *             .policyArn("arn:aws:iam::aws:policy/AmazonEKSVPCResourceController")
 *             .role(example.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:iam:Role
 *     properties:
 *       name: eks-cluster-example
 *       assumeRolePolicy: ${assumeRole.json}
 *   example-AmazonEKSClusterPolicy:
 *     type: aws:iam:RolePolicyAttachment
 *     properties:
 *       policyArn: arn:aws:iam::aws:policy/AmazonEKSClusterPolicy
 *       role: ${example.name}
 *   # Optionally, enable Security Groups for Pods
 *   # Reference: https://docs.aws.amazon.com/eks/latest/userguide/security-groups-for-pods.html
 *   example-AmazonEKSVPCResourceController:
 *     type: aws:iam:RolePolicyAttachment
 *     properties:
 *       policyArn: arn:aws:iam::aws:policy/AmazonEKSVPCResourceController
 *       role: ${example.name}
 * variables:
 *   assumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - eks.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Enabling Control Plane Logging
 * [EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html) can be enabled via the `enabled_cluster_log_types` argument. To manage the CloudWatch Log Group retention period, the `aws.cloudwatch.LogGroup` resource can be used.
 * > The below configuration uses [`dependsOn`](https://www.pulumi.com/docs/intro/concepts/programming-model/#dependson) to prevent ordering issues with EKS automatically creating the log group first and a variable for naming consistency. Other ordering and naming methodologies may be more appropriate for your environment.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const config = new pulumi.Config();
 * const clusterName = config.get("clusterName") || "example";
 * const exampleLogGroup = new aws.cloudwatch.LogGroup("example", {
 *     name: `/aws/eks/${clusterName}/cluster`,
 *     retentionInDays: 7,
 * });
 * const example = new aws.eks.Cluster("example", {
 *     enabledClusterLogTypes: [
 *         "api",
 *         "audit",
 *     ],
 *     name: clusterName,
 * }, {
 *     dependsOn: [exampleLogGroup],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * config = pulumi.Config()
 * cluster_name = config.get("clusterName")
 * if cluster_name is None:
 *     cluster_name = "example"
 * example_log_group = aws.cloudwatch.LogGroup("example",
 *     name=f"/aws/eks/{cluster_name}/cluster",
 *     retention_in_days=7)
 * example = aws.eks.Cluster("example",
 *     enabled_cluster_log_types=[
 *         "api",
 *         "audit",
 *     ],
 *     name=cluster_name,
 *     opts = pulumi.ResourceOptions(depends_on=[example_log_group]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var clusterName = config.Get("clusterName") ?? "example";
 *     var exampleLogGroup = new Aws.CloudWatch.LogGroup("example", new()
 *     {
 *         Name = $"/aws/eks/{clusterName}/cluster",
 *         RetentionInDays = 7,
 *     });
 *     var example = new Aws.Eks.Cluster("example", new()
 *     {
 *         EnabledClusterLogTypes = new[]
 *         {
 *             "api",
 *             "audit",
 *         },
 *         Name = clusterName,
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             exampleLogGroup,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		clusterName := "example"
 * 		if param := cfg.Get("clusterName"); param != "" {
 * 			clusterName = param
 * 		}
 * 		exampleLogGroup, err := cloudwatch.NewLogGroup(ctx, "example", &cloudwatch.LogGroupArgs{
 * 			Name:            pulumi.Sprintf("/aws/eks/%v/cluster", clusterName),
 * 			RetentionInDays: pulumi.Int(7),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = eks.NewCluster(ctx, "example", &eks.ClusterArgs{
 * 			EnabledClusterLogTypes: pulumi.StringArray{
 * 				pulumi.String("api"),
 * 				pulumi.String("audit"),
 * 			},
 * 			Name: pulumi.String(clusterName),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			exampleLogGroup,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.cloudwatch.LogGroupArgs;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.aws.eks.ClusterArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var clusterName = config.get("clusterName").orElse("example");
 *         var exampleLogGroup = new LogGroup("exampleLogGroup", LogGroupArgs.builder()
 *             .name(String.format("/aws/eks/%s/cluster", clusterName))
 *             .retentionInDays(7)
 *             .build());
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .enabledClusterLogTypes(
 *                 "api",
 *                 "audit")
 *             .name(clusterName)
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleLogGroup)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   clusterName:
 *     type: string
 *     default: example
 * resources:
 *   example:
 *     type: aws:eks:Cluster
 *     properties:
 *       enabledClusterLogTypes:
 *         - api
 *         - audit
 *       name: ${clusterName}
 *     options:
 *       dependson:
 *         - ${exampleLogGroup}
 *   exampleLogGroup:
 *     type: aws:cloudwatch:LogGroup
 *     name: example
 *     properties:
 *       name: /aws/eks/${clusterName}/cluster
 *       retentionInDays: 7 # ... potentially other configuration ...
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Enabling IAM Roles for Service Accounts
 * For more information about this feature, see the [EKS User Guide](https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html).
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * import * as tls from "@pulumi/tls";
 * const exampleCluster = new aws.eks.Cluster("example", {});
 * const example = exampleCluster.identities.apply(identities => tls.getCertificateOutput({
 *     url: identities[0]&#46;oidcs?&#46;[0]?.issuer,
 * }));
 * const exampleOpenIdConnectProvider = new aws.iam.OpenIdConnectProvider("example", {
 *     clientIdLists: ["sts&#46;amazonaws&#46;com"],
 *     thumbprintLists: [example&#46;apply(example => example&#46;certificates?&#46;[0]?&#46;sha1Fingerprint)],
 *     url: example.apply(example => example.url),
 * });
 * const exampleAssumeRolePolicy = aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         actions: ["sts:AssumeRoleWithWebIdentity"],
 *         effect: "Allow",
 *         conditions: [{
 *             test: "StringEquals",
 *             variable: std.replaceOutput({
 *                 text: exampleOpenIdConnectProvider.url,
 *                 search: "https://",
 *                 replace: "",
 *             }).apply(invoke => `${invoke.result}:sub`),
 *             values: ["system:serviceaccount:kube-system:aws-node"],
 *         }],
 *         principals: [{
 *             identifiers: [exampleOpenIdConnectProvider&#46;arn],
 *             type: "Federated",
 *         }],
 *     }],
 * });
 * const exampleRole = new aws.iam.Role("example", {
 *     assumeRolePolicy: exampleAssumeRolePolicy.apply(exampleAssumeRolePolicy => exampleAssumeRolePolicy.json),
 *     name: "example",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * import pulumi_tls as tls
 * example_cluster = aws.eks.Cluster("example")
 * example = example_cluster.identities.apply(lambda identities: tls.get_certificate_output(url=identities[0]&#46;oidcs[0].issuer))
 * example_open_id_connect_provider = aws.iam.OpenIdConnectProvider("example",
 *     client_id_lists=["sts&#46;amazonaws&#46;com"],
 *     thumbprint_lists=[example&#46;certificates[0]&#46;sha1_fingerprint],
 *     url=example.url)
 * example_assume_role_policy = aws.iam.get_policy_document_output(statements=[{
 *     "actions": ["sts:AssumeRoleWithWebIdentity"],
 *     "effect": "Allow",
 *     "conditions": [{
 *         "test": "StringEquals",
 *         "variable": std.replace_output(text=example_open_id_connect_provider.url,
 *             search="https://",
 *             replace="").apply(lambda invoke: f"{invoke.result}:sub"),
 *         "values": ["system:serviceaccount:kube-system:aws-node"],
 *     }],
 *     "principals": [{
 *         "identifiers": [example_open_id_connect_provider&#46;arn],
 *         "type": "Federated",
 *     }],
 * }])
 * example_role = aws.iam.Role("example",
 *     assume_role_policy=example_assume_role_policy.json,
 *     name="example")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * using Tls = Pulumi.Tls;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleCluster = new Aws.Eks.Cluster("example");
 *     var example = Tls.GetCertificate.Invoke(new()
 *     {
 *         Url = exampleCluster.Identities[0]&#46;Oidcs[0]?.Issuer,
 *     });
 *     var exampleOpenIdConnectProvider = new Aws.Iam.OpenIdConnectProvider("example", new()
 *     {
 *         ClientIdLists = new[]
 *         {
 *             "sts.amazonaws.com",
 *         },
 *         ThumbprintLists = new[]
 *         {
 *             example.Apply(getCertificateResult => getCertificateResult.Certificates[0]?.Sha1Fingerprint),
 *         },
 *         Url = example.Apply(getCertificateResult => getCertificateResult.Url),
 *     });
 *     var exampleAssumeRolePolicy = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRoleWithWebIdentity",
 *                 },
 *                 Effect = "Allow",
 *                 Conditions = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementConditionInputArgs
 *                     {
 *                         Test = "StringEquals",
 *                         Variable = $"{Std.Replace.Invoke(new()
 *                         {
 *                             Text = exampleOpenIdConnectProvider.Url,
 *                             Search = "https://",
 *                             Replace = "",
 *                         }).Result}:sub",
 *                         Values = new[]
 *                         {
 *                             "system:serviceaccount:kube-system:aws-node",
 *                         },
 *                     },
 *                 },
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Identifiers = new[]
 *                         {
 *                             exampleOpenIdConnectProvider.Arn,
 *                         },
 *                         Type = "Federated",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     var exampleRole = new Aws.Iam.Role("example", new()
 *     {
 *         AssumeRolePolicy = exampleAssumeRolePolicy.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *         Name = "example",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi-tls/sdk/v4/go/tls"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		exampleCluster, err := eks.NewCluster(ctx, "example", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example := exampleCluster.Identities.ApplyT(func(identities []eks.ClusterIdentity) (tls.GetCertificateResult, error) {
 * 			return tls.GetCertificateResult(interface{}(tls.GetCertificateOutput(ctx, tls.GetCertificateOutputArgs{
 * 				Url: identities[0]&#46;Oidcs[0].Issuer,
 * 			}, nil))), nil
 * 		}).(tls.GetCertificateResultOutput)
 * 		exampleOpenIdConnectProvider, err := iam.NewOpenIdConnectProvider(ctx, "example", &iam.OpenIdConnectProviderArgs{
 * 			ClientIdLists: pulumi.StringArray{
 * 				pulumi.String("sts.amazonaws.com"),
 * 			},
 * 			ThumbprintLists: pulumi.StringArray{
 * 				pulumi.String(example.ApplyT(func(example tls.GetCertificateResult) (*string, error) {
 * 					return &example.Certificates[0].Sha1Fingerprint, nil
 * 				}).(pulumi.StringPtrOutput)),
 * 			},
 * 			Url: pulumi.String(example.ApplyT(func(example tls.GetCertificateResult) (*string, error) {
 * 				return &example.Url, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleAssumeRolePolicy := iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * 			Statements: iam.GetPolicyDocumentStatementArray{
 * 				&iam.GetPolicyDocumentStatementArgs{
 * 					Actions: pulumi.StringArray{
 * 						pulumi.String("sts:AssumeRoleWithWebIdentity"),
 * 					},
 * 					Effect: pulumi.String("Allow"),
 * 					Conditions: iam.GetPolicyDocumentStatementConditionArray{
 * 						&iam.GetPolicyDocumentStatementConditionArgs{
 * 							Test: pulumi.String("StringEquals"),
 * 							Variable: std.ReplaceOutput(ctx, std.ReplaceOutputArgs{
 * 								Text:    exampleOpenIdConnectProvider.Url,
 * 								Search:  pulumi.String("https://"),
 * 								Replace: pulumi.String(""),
 * 							}, nil).ApplyT(func(invoke std.ReplaceResult) (string, error) {
 * 								return fmt.Sprintf("%v:sub", invoke.Result), nil
 * 							}).(pulumi.StringOutput),
 * 							Values: pulumi.StringArray{
 * 								pulumi.String("system:serviceaccount:kube-system:aws-node"),
 * 							},
 * 						},
 * 					},
 * 					Principals: iam.GetPolicyDocumentStatementPrincipalArray{
 * 						&iam.GetPolicyDocumentStatementPrincipalArgs{
 * 							Identifiers: pulumi.StringArray{
 * 								exampleOpenIdConnectProvider.Arn,
 * 							},
 * 							Type: pulumi.String("Federated"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		_, err = iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			AssumeRolePolicy: pulumi.String(exampleAssumeRolePolicy.ApplyT(func(exampleAssumeRolePolicy iam.GetPolicyDocumentResult) (*string, error) {
 * 				return &exampleAssumeRolePolicy.Json, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 			Name: pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.tls.TlsFunctions;
 * import com.pulumi.tls.inputs.GetCertificateArgs;
 * import com.pulumi.aws.iam.OpenIdConnectProvider;
 * import com.pulumi.aws.iam.OpenIdConnectProviderArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleCluster = new Cluster("exampleCluster");
 *         final var example = TlsFunctions.getCertificate(GetCertificateArgs.builder()
 *             .url(exampleCluster.identities().applyValue(identities -> identities[0]&#46;oidcs()[0].issuer()))
 *             .build());
 *         var exampleOpenIdConnectProvider = new OpenIdConnectProvider("exampleOpenIdConnectProvider", OpenIdConnectProviderArgs.builder()
 *             .clientIdLists("sts.amazonaws.com")
 *             .thumbprintLists(example.applyValue(getCertificateResult -> getCertificateResult).applyValue(example -> example.applyValue(getCertificateResult -> getCertificateResult.certificates()[0].sha1Fingerprint())))
 *             .url(example.applyValue(getCertificateResult -> getCertificateResult).applyValue(example -> example.applyValue(getCertificateResult -> getCertificateResult.url())))
 *             .build());
 *         final var exampleAssumeRolePolicy = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .actions("sts:AssumeRoleWithWebIdentity")
 *                 .effect("Allow")
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("StringEquals")
 *                     .variable(StdFunctions.replace().applyValue(invoke -> String.format("%s:sub", invoke.result())))
 *                     .values("system:serviceaccount:kube-system:aws-node")
 *                     .build())
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .identifiers(exampleOpenIdConnectProvider.arn())
 *                     .type("Federated")
 *                     .build())
 *                 .build())
 *             .build());
 *         var exampleRole = new Role("exampleRole", RoleArgs.builder()
 *             .assumeRolePolicy(exampleAssumeRolePolicy.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(exampleAssumeRolePolicy -> exampleAssumeRolePolicy.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .name("example")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleCluster:
 *     type: aws:eks:Cluster
 *     name: example
 *   exampleOpenIdConnectProvider:
 *     type: aws:iam:OpenIdConnectProvider
 *     name: example
 *     properties:
 *       clientIdLists:
 *         - sts.amazonaws.com
 *       thumbprintLists:
 *         - ${example.certificates[0].sha1Fingerprint}
 *       url: ${example.url}
 *   exampleRole:
 *     type: aws:iam:Role
 *     name: example
 *     properties:
 *       assumeRolePolicy: ${exampleAssumeRolePolicy.json}
 *       name: example
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: tls:getCertificate
 *       Arguments:
 *         url: ${exampleCluster.identities[0]&#46;oidcs[0].issuer}
 *   exampleAssumeRolePolicy:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - actions:
 *               - sts:AssumeRoleWithWebIdentity
 *             effect: Allow
 *             conditions:
 *               - test: StringEquals
 *                 variable:
 *                   fn::join:
 *                     -
 *                     - - fn::invoke:
 *                           Function: std:replace
 *                           Arguments:
 *                             text: ${exampleOpenIdConnectProvider.url}
 *                             search: https://
 *                             replace:
 *                           Return: result
 *                       - :sub
 *                 values:
 *                   - system:serviceaccount:kube-system:aws-node
 *             principals:
 *               - identifiers:
 *                   - ${exampleOpenIdConnectProvider.arn}
 *                 type: Federated
 * ```
 * <!--End PulumiCodeChooser -->
 * ### EKS Cluster on AWS Outpost
 * [Creating a local Amazon EKS cluster on an AWS Outpost](https://docs.aws.amazon.com/eks/latest/userguide/create-cluster-outpost.html)
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.iam.Role("example", {
 *     assumeRolePolicy: exampleAssumeRolePolicy.json,
 *     name: "example",
 * });
 * const exampleCluster = new aws.eks.Cluster("example", {
 *     name: "example-cluster",
 *     roleArn: example.arn,
 *     vpcConfig: {
 *         endpointPrivateAccess: true,
 *         endpointPublicAccess: false,
 *     },
 *     outpostConfig: {
 *         controlPlaneInstanceType: "m5d.large",
 *         outpostArns: [exampleAwsOutpostsOutpost&#46;arn],
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.iam.Role("example",
 *     assume_role_policy=example_assume_role_policy["json"],
 *     name="example")
 * example_cluster = aws.eks.Cluster("example",
 *     name="example-cluster",
 *     role_arn=example.arn,
 *     vpc_config={
 *         "endpoint_private_access": True,
 *         "endpoint_public_access": False,
 *     },
 *     outpost_config={
 *         "control_plane_instance_type": "m5d.large",
 *         "outpost_arns": [example_aws_outposts_outpost["arn"]],
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Iam.Role("example", new()
 *     {
 *         AssumeRolePolicy = exampleAssumeRolePolicy.Json,
 *         Name = "example",
 *     });
 *     var exampleCluster = new Aws.Eks.Cluster("example", new()
 *     {
 *         Name = "example-cluster",
 *         RoleArn = example.Arn,
 *         VpcConfig = new Aws.Eks.Inputs.ClusterVpcConfigArgs
 *         {
 *             EndpointPrivateAccess = true,
 *             EndpointPublicAccess = false,
 *         },
 *         OutpostConfig = new Aws.Eks.Inputs.ClusterOutpostConfigArgs
 *         {
 *             ControlPlaneInstanceType = "m5d.large",
 *             OutpostArns = new[]
 *             {
 *                 exampleAwsOutpostsOutpost.Arn,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			AssumeRolePolicy: pulumi.Any(exampleAssumeRolePolicy.Json),
 * 			Name:             pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = eks.NewCluster(ctx, "example", &eks.ClusterArgs{
 * 			Name:    pulumi.String("example-cluster"),
 * 			RoleArn: example.Arn,
 * 			VpcConfig: &eks.ClusterVpcConfigArgs{
 * 				EndpointPrivateAccess: pulumi.Bool(true),
 * 				EndpointPublicAccess:  pulumi.Bool(false),
 * 			},
 * 			OutpostConfig: &eks.ClusterOutpostConfigArgs{
 * 				ControlPlaneInstanceType: pulumi.String("m5d.large"),
 * 				OutpostArns: pulumi.StringArray{
 * 					exampleAwsOutpostsOutpost.Arn,
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.aws.eks.ClusterArgs;
 * import com.pulumi.aws.eks.inputs.ClusterVpcConfigArgs;
 * import com.pulumi.aws.eks.inputs.ClusterOutpostConfigArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Role("example", RoleArgs.builder()
 *             .assumeRolePolicy(exampleAssumeRolePolicy.json())
 *             .name("example")
 *             .build());
 *         var exampleCluster = new Cluster("exampleCluster", ClusterArgs.builder()
 *             .name("example-cluster")
 *             .roleArn(example.arn())
 *             .vpcConfig(ClusterVpcConfigArgs.builder()
 *                 .endpointPrivateAccess(true)
 *                 .endpointPublicAccess(false)
 *                 .build())
 *             .outpostConfig(ClusterOutpostConfigArgs.builder()
 *                 .controlPlaneInstanceType("m5d.large")
 *                 .outpostArns(exampleAwsOutpostsOutpost.arn())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:iam:Role
 *     properties:
 *       assumeRolePolicy: ${exampleAssumeRolePolicy.json}
 *       name: example
 *   exampleCluster:
 *     type: aws:eks:Cluster
 *     name: example
 *     properties:
 *       name: example-cluster
 *       roleArn: ${example.arn}
 *       vpcConfig:
 *         endpointPrivateAccess: true
 *         endpointPublicAccess: false
 *       outpostConfig:
 *         controlPlaneInstanceType: m5d.large
 *         outpostArns:
 *           - ${exampleAwsOutpostsOutpost.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### EKS Cluster with Access Config
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.iam.Role("example", {
 *     assumeRolePolicy: exampleAssumeRolePolicy.json,
 *     name: "example",
 * });
 * const exampleCluster = new aws.eks.Cluster("example", {
 *     name: "example-cluster",
 *     roleArn: example.arn,
 *     vpcConfig: {
 *         endpointPrivateAccess: true,
 *         endpointPublicAccess: false,
 *     },
 *     accessConfig: {
 *         authenticationMode: "CONFIG_MAP",
 *         bootstrapClusterCreatorAdminPermissions: true,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.iam.Role("example",
 *     assume_role_policy=example_assume_role_policy["json"],
 *     name="example")
 * example_cluster = aws.eks.Cluster("example",
 *     name="example-cluster",
 *     role_arn=example.arn,
 *     vpc_config={
 *         "endpoint_private_access": True,
 *         "endpoint_public_access": False,
 *     },
 *     access_config={
 *         "authentication_mode": "CONFIG_MAP",
 *         "bootstrap_cluster_creator_admin_permissions": True,
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Iam.Role("example", new()
 *     {
 *         AssumeRolePolicy = exampleAssumeRolePolicy.Json,
 *         Name = "example",
 *     });
 *     var exampleCluster = new Aws.Eks.Cluster("example", new()
 *     {
 *         Name = "example-cluster",
 *         RoleArn = example.Arn,
 *         VpcConfig = new Aws.Eks.Inputs.ClusterVpcConfigArgs
 *         {
 *             EndpointPrivateAccess = true,
 *             EndpointPublicAccess = false,
 *         },
 *         AccessConfig = new Aws.Eks.Inputs.ClusterAccessConfigArgs
 *         {
 *             AuthenticationMode = "CONFIG_MAP",
 *             BootstrapClusterCreatorAdminPermissions = true,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			AssumeRolePolicy: pulumi.Any(exampleAssumeRolePolicy.Json),
 * 			Name:             pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = eks.NewCluster(ctx, "example", &eks.ClusterArgs{
 * 			Name:    pulumi.String("example-cluster"),
 * 			RoleArn: example.Arn,
 * 			VpcConfig: &eks.ClusterVpcConfigArgs{
 * 				EndpointPrivateAccess: pulumi.Bool(true),
 * 				EndpointPublicAccess:  pulumi.Bool(false),
 * 			},
 * 			AccessConfig: &eks.ClusterAccessConfigArgs{
 * 				AuthenticationMode:                      pulumi.String("CONFIG_MAP"),
 * 				BootstrapClusterCreatorAdminPermissions: pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.eks.Cluster;
 * import com.pulumi.aws.eks.ClusterArgs;
 * import com.pulumi.aws.eks.inputs.ClusterVpcConfigArgs;
 * import com.pulumi.aws.eks.inputs.ClusterAccessConfigArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Role("example", RoleArgs.builder()
 *             .assumeRolePolicy(exampleAssumeRolePolicy.json())
 *             .name("example")
 *             .build());
 *         var exampleCluster = new Cluster("exampleCluster", ClusterArgs.builder()
 *             .name("example-cluster")
 *             .roleArn(example.arn())
 *             .vpcConfig(ClusterVpcConfigArgs.builder()
 *                 .endpointPrivateAccess(true)
 *                 .endpointPublicAccess(false)
 *                 .build())
 *             .accessConfig(ClusterAccessConfigArgs.builder()
 *                 .authenticationMode("CONFIG_MAP")
 *                 .bootstrapClusterCreatorAdminPermissions(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:iam:Role
 *     properties:
 *       assumeRolePolicy: ${exampleAssumeRolePolicy.json}
 *       name: example
 *   exampleCluster:
 *     type: aws:eks:Cluster
 *     name: example
 *     properties:
 *       name: example-cluster
 *       roleArn: ${example.arn}
 *       vpcConfig:
 *         endpointPrivateAccess: true
 *         endpointPublicAccess: false
 *       accessConfig:
 *         authenticationMode: CONFIG_MAP
 *         bootstrapClusterCreatorAdminPermissions: true
 * ```
 * <!--End PulumiCodeChooser -->
 * After adding inline IAM Policies (e.g., `aws.iam.RolePolicy` resource) or attaching IAM Policies (e.g., `aws.iam.Policy` resource and `aws.iam.RolePolicyAttachment` resource) with the desired permissions to the IAM Role, annotate the Kubernetes service account (e.g., `kubernetes_service_account` resource) and recreate any pods.
 * ## Import
 * Using `pulumi import`, import EKS Clusters using the `name`. For example:
 * ```sh
 * $ pulumi import aws:eks/cluster:Cluster my_cluster my_cluster
 * ```
 * @property accessConfig Configuration block for the access config associated with your cluster, see [Amazon EKS Access Entries](https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html).
 * @property bootstrapSelfManagedAddons Install default unmanaged add-ons, such as `aws-cni`, `kube-proxy`, and CoreDNS during cluster creation. If `false`, you must manually install desired add-ons. Changing this value will force a new cluster to be created. Defaults to `true`.
 * @property defaultAddonsToRemoves
 * @property enabledClusterLogTypes List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
 * @property encryptionConfig Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
 * @property kubernetesNetworkConfig Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
 * @property name Name of the cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]*$`).
 * @property outpostConfig Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
 * @property roleArn ARN of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Ensure the resource configuration includes explicit dependencies on the IAM Role permissions by adding `depends_on` if using the `aws.iam.RolePolicy` resource or `aws.iam.RolePolicyAttachment` resource, otherwise EKS cannot delete EKS managed EC2 infrastructure such as Security Groups on EKS Cluster deletion.
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property upgradePolicy Configuration block for the support policy to use for the cluster.  See upgrade_policy for details.
 * @property version Desired Kubernetes master version. If you do not specify a value, the latest available version at resource creation is used and no upgrades will occur except those automatically triggered by EKS. The value must be configured and increased to upgrade the version when desired. Downgrades are not supported by EKS.
 * @property vpcConfig Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
 * The following arguments are optional:
 */
public data class ClusterArgs(
    public val accessConfig: Output<ClusterAccessConfigArgs>? = null,
    public val bootstrapSelfManagedAddons: Output<Boolean>? = null,
    public val defaultAddonsToRemoves: Output<List<String>>? = null,
    public val enabledClusterLogTypes: Output<List<String>>? = null,
    public val encryptionConfig: Output<ClusterEncryptionConfigArgs>? = null,
    public val kubernetesNetworkConfig: Output<ClusterKubernetesNetworkConfigArgs>? = null,
    public val name: Output<String>? = null,
    public val outpostConfig: Output<ClusterOutpostConfigArgs>? = null,
    public val roleArn: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val upgradePolicy: Output<ClusterUpgradePolicyArgs>? = null,
    public val version: Output<String>? = null,
    public val vpcConfig: Output<ClusterVpcConfigArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.eks.ClusterArgs> {
    override fun toJava(): com.pulumi.aws.eks.ClusterArgs = com.pulumi.aws.eks.ClusterArgs.builder()
        .accessConfig(accessConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .bootstrapSelfManagedAddons(bootstrapSelfManagedAddons?.applyValue({ args0 -> args0 }))
        .defaultAddonsToRemoves(
            defaultAddonsToRemoves?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        )
        .enabledClusterLogTypes(
            enabledClusterLogTypes?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        )
        .encryptionConfig(encryptionConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .kubernetesNetworkConfig(
            kubernetesNetworkConfig?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .name(name?.applyValue({ args0 -> args0 }))
        .outpostConfig(outpostConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .roleArn(roleArn?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .upgradePolicy(upgradePolicy?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .version(version?.applyValue({ args0 -> args0 }))
        .vpcConfig(vpcConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [ClusterArgs].
 */
@PulumiTagMarker
public class ClusterArgsBuilder internal constructor() {
    private var accessConfig: Output<ClusterAccessConfigArgs>? = null

    private var bootstrapSelfManagedAddons: Output<Boolean>? = null

    private var defaultAddonsToRemoves: Output<List<String>>? = null

    private var enabledClusterLogTypes: Output<List<String>>? = null

    private var encryptionConfig: Output<ClusterEncryptionConfigArgs>? = null

    private var kubernetesNetworkConfig: Output<ClusterKubernetesNetworkConfigArgs>? = null

    private var name: Output<String>? = null

    private var outpostConfig: Output<ClusterOutpostConfigArgs>? = null

    private var roleArn: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var upgradePolicy: Output<ClusterUpgradePolicyArgs>? = null

    private var version: Output<String>? = null

    private var vpcConfig: Output<ClusterVpcConfigArgs>? = null

    /**
     * @param value Configuration block for the access config associated with your cluster, see [Amazon EKS Access Entries](https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html).
     */
    @JvmName("duswyawlojhtrvkd")
    public suspend fun accessConfig(`value`: Output<ClusterAccessConfigArgs>) {
        this.accessConfig = value
    }

    /**
     * @param value Install default unmanaged add-ons, such as `aws-cni`, `kube-proxy`, and CoreDNS during cluster creation. If `false`, you must manually install desired add-ons. Changing this value will force a new cluster to be created. Defaults to `true`.
     */
    @JvmName("xsathmnoqjewihhk")
    public suspend fun bootstrapSelfManagedAddons(`value`: Output<Boolean>) {
        this.bootstrapSelfManagedAddons = value
    }

    /**
     * @param value
     */
    @JvmName("dxknoaficibuvyiw")
    public suspend fun defaultAddonsToRemoves(`value`: Output<List<String>>) {
        this.defaultAddonsToRemoves = value
    }

    @JvmName("yxketqbxmlabxhwn")
    public suspend fun defaultAddonsToRemoves(vararg values: Output<String>) {
        this.defaultAddonsToRemoves = Output.all(values.asList())
    }

    /**
     * @param values
     */
    @JvmName("hwnuftdxshowiayx")
    public suspend fun defaultAddonsToRemoves(values: List<Output<String>>) {
        this.defaultAddonsToRemoves = Output.all(values)
    }

    /**
     * @param value List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("btvkahafuaeuwofa")
    public suspend fun enabledClusterLogTypes(`value`: Output<List<String>>) {
        this.enabledClusterLogTypes = value
    }

    @JvmName("ckwfusvpelfcvryi")
    public suspend fun enabledClusterLogTypes(vararg values: Output<String>) {
        this.enabledClusterLogTypes = Output.all(values.asList())
    }

    /**
     * @param values List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("ylfhkomfcpvqfdxr")
    public suspend fun enabledClusterLogTypes(values: List<Output<String>>) {
        this.enabledClusterLogTypes = Output.all(values)
    }

    /**
     * @param value Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
     */
    @JvmName("hvebeykfcglpiwlu")
    public suspend fun encryptionConfig(`value`: Output<ClusterEncryptionConfigArgs>) {
        this.encryptionConfig = value
    }

    /**
     * @param value Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("yjtmoanhsobnnbdq")
    public suspend fun kubernetesNetworkConfig(`value`: Output<ClusterKubernetesNetworkConfigArgs>) {
        this.kubernetesNetworkConfig = value
    }

    /**
     * @param value Name of the cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]*$`).
     */
    @JvmName("enfxxetibjiojgmn")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
     */
    @JvmName("kkswbgkbgjrpkhol")
    public suspend fun outpostConfig(`value`: Output<ClusterOutpostConfigArgs>) {
        this.outpostConfig = value
    }

    /**
     * @param value ARN of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Ensure the resource configuration includes explicit dependencies on the IAM Role permissions by adding `depends_on` if using the `aws.iam.RolePolicy` resource or `aws.iam.RolePolicyAttachment` resource, otherwise EKS cannot delete EKS managed EC2 infrastructure such as Security Groups on EKS Cluster deletion.
     */
    @JvmName("mehothliipavrtbh")
    public suspend fun roleArn(`value`: Output<String>) {
        this.roleArn = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vyvghbjeaiwwjxtx")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Configuration block for the support policy to use for the cluster.  See upgrade_policy for details.
     */
    @JvmName("jkiwvwclfxookrro")
    public suspend fun upgradePolicy(`value`: Output<ClusterUpgradePolicyArgs>) {
        this.upgradePolicy = value
    }

    /**
     * @param value Desired Kubernetes master version. If you do not specify a value, the latest available version at resource creation is used and no upgrades will occur except those automatically triggered by EKS. The value must be configured and increased to upgrade the version when desired. Downgrades are not supported by EKS.
     */
    @JvmName("btxmvnotmaxvoljm")
    public suspend fun version(`value`: Output<String>) {
        this.version = value
    }

    /**
     * @param value Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
     * The following arguments are optional:
     */
    @JvmName("wdwwfytutkchnngi")
    public suspend fun vpcConfig(`value`: Output<ClusterVpcConfigArgs>) {
        this.vpcConfig = value
    }

    /**
     * @param value Configuration block for the access config associated with your cluster, see [Amazon EKS Access Entries](https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html).
     */
    @JvmName("xuchclkokdgksexk")
    public suspend fun accessConfig(`value`: ClusterAccessConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessConfig = mapped
    }

    /**
     * @param argument Configuration block for the access config associated with your cluster, see [Amazon EKS Access Entries](https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html).
     */
    @JvmName("wyoflqrvxdsbqajx")
    public suspend fun accessConfig(argument: suspend ClusterAccessConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterAccessConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.accessConfig = mapped
    }

    /**
     * @param value Install default unmanaged add-ons, such as `aws-cni`, `kube-proxy`, and CoreDNS during cluster creation. If `false`, you must manually install desired add-ons. Changing this value will force a new cluster to be created. Defaults to `true`.
     */
    @JvmName("nykeljxgkxnpeipv")
    public suspend fun bootstrapSelfManagedAddons(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.bootstrapSelfManagedAddons = mapped
    }

    /**
     * @param value
     */
    @JvmName("nnjdcmxyufrmksmw")
    public suspend fun defaultAddonsToRemoves(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultAddonsToRemoves = mapped
    }

    /**
     * @param values
     */
    @JvmName("uumlgunkxlpebspy")
    public suspend fun defaultAddonsToRemoves(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.defaultAddonsToRemoves = mapped
    }

    /**
     * @param value List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("asnvftymibtetapf")
    public suspend fun enabledClusterLogTypes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabledClusterLogTypes = mapped
    }

    /**
     * @param values List of the desired control plane logging to enable. For more information, see [Amazon EKS Control Plane Logging](https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html).
     */
    @JvmName("hiymwepdntjtsldq")
    public suspend fun enabledClusterLogTypes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.enabledClusterLogTypes = mapped
    }

    /**
     * @param value Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
     */
    @JvmName("jlufvjlhecivpmeq")
    public suspend fun encryptionConfig(`value`: ClusterEncryptionConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.encryptionConfig = mapped
    }

    /**
     * @param argument Configuration block with encryption configuration for the cluster. Only available on Kubernetes 1.13 and above clusters created after March 6, 2020. Detailed below.
     */
    @JvmName("ripljosylkkleeuf")
    public suspend fun encryptionConfig(argument: suspend ClusterEncryptionConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterEncryptionConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.encryptionConfig = mapped
    }

    /**
     * @param value Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("yukviypduwwqdmea")
    public suspend fun kubernetesNetworkConfig(`value`: ClusterKubernetesNetworkConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kubernetesNetworkConfig = mapped
    }

    /**
     * @param argument Configuration block with kubernetes network configuration for the cluster. Detailed below. If removed, this provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("obpwqihlbhrolbfi")
    public suspend fun kubernetesNetworkConfig(argument: suspend ClusterKubernetesNetworkConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterKubernetesNetworkConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.kubernetesNetworkConfig = mapped
    }

    /**
     * @param value Name of the cluster. Must be between 1-100 characters in length. Must begin with an alphanumeric character, and must only contain alphanumeric characters, dashes and underscores (`^[0-9A-Za-z][A-Za-z0-9\-_]*$`).
     */
    @JvmName("ramowfdklndjdken")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
     */
    @JvmName("opgnevnwheucuxag")
    public suspend fun outpostConfig(`value`: ClusterOutpostConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.outpostConfig = mapped
    }

    /**
     * @param argument Configuration block representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This block isn't available for creating Amazon EKS clusters on the AWS cloud.
     */
    @JvmName("gpvuhmdivdvrpnfx")
    public suspend fun outpostConfig(argument: suspend ClusterOutpostConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterOutpostConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.outpostConfig = mapped
    }

    /**
     * @param value ARN of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Ensure the resource configuration includes explicit dependencies on the IAM Role permissions by adding `depends_on` if using the `aws.iam.RolePolicy` resource or `aws.iam.RolePolicyAttachment` resource, otherwise EKS cannot delete EKS managed EC2 infrastructure such as Security Groups on EKS Cluster deletion.
     */
    @JvmName("xwblxrmmtcmhrygh")
    public suspend fun roleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.roleArn = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ovdapiiqxtlgnvhs")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("yvltkgifebfllrqi")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Configuration block for the support policy to use for the cluster.  See upgrade_policy for details.
     */
    @JvmName("wehnneyrthxknaif")
    public suspend fun upgradePolicy(`value`: ClusterUpgradePolicyArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.upgradePolicy = mapped
    }

    /**
     * @param argument Configuration block for the support policy to use for the cluster.  See upgrade_policy for details.
     */
    @JvmName("emookiqrjmhbmpcn")
    public suspend fun upgradePolicy(argument: suspend ClusterUpgradePolicyArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterUpgradePolicyArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.upgradePolicy = mapped
    }

    /**
     * @param value Desired Kubernetes master version. If you do not specify a value, the latest available version at resource creation is used and no upgrades will occur except those automatically triggered by EKS. The value must be configured and increased to upgrade the version when desired. Downgrades are not supported by EKS.
     */
    @JvmName("nfmhfylqvlpyworf")
    public suspend fun version(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.version = mapped
    }

    /**
     * @param value Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
     * The following arguments are optional:
     */
    @JvmName("jbshfupifwrplsos")
    public suspend fun vpcConfig(`value`: ClusterVpcConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcConfig = mapped
    }

    /**
     * @param argument Configuration block for the VPC associated with your cluster. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see [Cluster VPC Considerations](https://docs&#46;aws&#46;amazon&#46;com/eks/latest/userguide/network_reqs&#46;html) and [Cluster Security Group Considerations](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html) in the Amazon EKS User Guide. Detailed below. Also contains attributes detailed in the Attributes section.
     * The following arguments are optional:
     */
    @JvmName("oimtempbqagkdxsb")
    public suspend fun vpcConfig(argument: suspend ClusterVpcConfigArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterVpcConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.vpcConfig = mapped
    }

    internal fun build(): ClusterArgs = ClusterArgs(
        accessConfig = accessConfig,
        bootstrapSelfManagedAddons = bootstrapSelfManagedAddons,
        defaultAddonsToRemoves = defaultAddonsToRemoves,
        enabledClusterLogTypes = enabledClusterLogTypes,
        encryptionConfig = encryptionConfig,
        kubernetesNetworkConfig = kubernetesNetworkConfig,
        name = name,
        outpostConfig = outpostConfig,
        roleArn = roleArn,
        tags = tags,
        upgradePolicy = upgradePolicy,
        version = version,
        vpcConfig = vpcConfig,
    )
}
