@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.eks.kotlin

import com.pulumi.aws.eks.PodIdentityAssociationArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Resource for managing an AWS EKS (Elastic Kubernetes) Pod Identity Association.
 * Creates an EKS Pod Identity association between a service account in an Amazon EKS cluster and an IAM role with EKS Pod Identity. Use EKS Pod Identity to give temporary IAM credentials to pods and the credentials are rotated automatically.
 * Amazon EKS Pod Identity associations provide the ability to manage credentials for your applications, similar to the way that EC2 instance profiles provide credentials to Amazon EC2 instances.
 * If a pod uses a service account that has an association, Amazon EKS sets environment variables in the containers of the pod. The environment variables configure the Amazon Web Services SDKs, including the Command Line Interface, to use the EKS Pod Identity credentials.
 * Pod Identity is a simpler method than IAM roles for service accounts, as this method doesn’t use OIDC identity providers. Additionally, you can configure a role for Pod Identity once, and reuse it across clusters.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const assumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["pods&#46;eks&#46;amazonaws&#46;com"],
 *         }],
 *         actions: [
 *             "sts:AssumeRole",
 *             "sts:TagSession",
 *         ],
 *     }],
 * });
 * const example = new aws.iam.Role("example", {
 *     name: "eks-pod-identity-example",
 *     assumeRolePolicy: assumeRole.then(assumeRole => assumeRole.json),
 * });
 * const exampleS3 = new aws.iam.RolePolicyAttachment("example_s3", {
 *     policyArn: "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess",
 *     role: example.name,
 * });
 * const examplePodIdentityAssociation = new aws.eks.PodIdentityAssociation("example", {
 *     clusterName: exampleAwsEksCluster.name,
 *     namespace: "example",
 *     serviceAccount: "example-sa",
 *     roleArn: example.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["pods&#46;eks&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": [
 *         "sts:AssumeRole",
 *         "sts:TagSession",
 *     ],
 * }])
 * example = aws.iam.Role("example",
 *     name="eks-pod-identity-example",
 *     assume_role_policy=assume_role.json)
 * example_s3 = aws.iam.RolePolicyAttachment("example_s3",
 *     policy_arn="arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess",
 *     role=example.name)
 * example_pod_identity_association = aws.eks.PodIdentityAssociation("example",
 *     cluster_name=example_aws_eks_cluster["name"],
 *     namespace="example",
 *     service_account="example-sa",
 *     role_arn=example.arn)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var assumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "pods.eks.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                     "sts:TagSession",
 *                 },
 *             },
 *         },
 *     });
 *     var example = new Aws.Iam.Role("example", new()
 *     {
 *         Name = "eks-pod-identity-example",
 *         AssumeRolePolicy = assumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var exampleS3 = new Aws.Iam.RolePolicyAttachment("example_s3", new()
 *     {
 *         PolicyArn = "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess",
 *         Role = example.Name,
 *     });
 *     var examplePodIdentityAssociation = new Aws.Eks.PodIdentityAssociation("example", new()
 *     {
 *         ClusterName = exampleAwsEksCluster.Name,
 *         Namespace = "example",
 *         ServiceAccount = "example-sa",
 *         RoleArn = example.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/eks"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		assumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"pods.eks.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 						"sts:TagSession",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			Name:             pulumi.String("eks-pod-identity-example"),
 * 			AssumeRolePolicy: pulumi.String(assumeRole.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iam.NewRolePolicyAttachment(ctx, "example_s3", &iam.RolePolicyAttachmentArgs{
 * 			PolicyArn: pulumi.String("arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess"),
 * 			Role:      example.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = eks.NewPodIdentityAssociation(ctx, "example", &eks.PodIdentityAssociationArgs{
 * 			ClusterName:    pulumi.Any(exampleAwsEksCluster.Name),
 * 			Namespace:      pulumi.String("example"),
 * 			ServiceAccount: pulumi.String("example-sa"),
 * 			RoleArn:        example.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.iam.RolePolicyAttachment;
 * import com.pulumi.aws.iam.RolePolicyAttachmentArgs;
 * import com.pulumi.aws.eks.PodIdentityAssociation;
 * import com.pulumi.aws.eks.PodIdentityAssociationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("pods.eks.amazonaws.com")
 *                     .build())
 *                 .actions(
 *                     "sts:AssumeRole",
 *                     "sts:TagSession")
 *                 .build())
 *             .build());
 *         var example = new Role("example", RoleArgs.builder()
 *             .name("eks-pod-identity-example")
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var exampleS3 = new RolePolicyAttachment("exampleS3", RolePolicyAttachmentArgs.builder()
 *             .policyArn("arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess")
 *             .role(example.name())
 *             .build());
 *         var examplePodIdentityAssociation = new PodIdentityAssociation("examplePodIdentityAssociation", PodIdentityAssociationArgs.builder()
 *             .clusterName(exampleAwsEksCluster.name())
 *             .namespace("example")
 *             .serviceAccount("example-sa")
 *             .roleArn(example.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:iam:Role
 *     properties:
 *       name: eks-pod-identity-example
 *       assumeRolePolicy: ${assumeRole.json}
 *   exampleS3:
 *     type: aws:iam:RolePolicyAttachment
 *     name: example_s3
 *     properties:
 *       policyArn: arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess
 *       role: ${example.name}
 *   examplePodIdentityAssociation:
 *     type: aws:eks:PodIdentityAssociation
 *     name: example
 *     properties:
 *       clusterName: ${exampleAwsEksCluster.name}
 *       namespace: example
 *       serviceAccount: example-sa
 *       roleArn: ${example.arn}
 * variables:
 *   assumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - pods.eks.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 *               - sts:TagSession
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import EKS (Elastic Kubernetes) Pod Identity Association using the `cluster_name` and `association_id` separated by a comma (`,`). For example:
 * ```sh
 * $ pulumi import aws:eks/podIdentityAssociation:PodIdentityAssociation example example,a-12345678
 * ```
 * @property clusterName The name of the cluster to create the association in.
 * @property namespace The name of the Kubernetes namespace inside the cluster to create the association in. The service account and the pods that use the service account must be in this namespace.
 * @property roleArn The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.
 * @property serviceAccount The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
 * The following arguments are optional:
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class PodIdentityAssociationArgs(
    public val clusterName: Output<String>? = null,
    public val namespace: Output<String>? = null,
    public val roleArn: Output<String>? = null,
    public val serviceAccount: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.eks.PodIdentityAssociationArgs> {
    override fun toJava(): com.pulumi.aws.eks.PodIdentityAssociationArgs =
        com.pulumi.aws.eks.PodIdentityAssociationArgs.builder()
            .clusterName(clusterName?.applyValue({ args0 -> args0 }))
            .namespace(namespace?.applyValue({ args0 -> args0 }))
            .roleArn(roleArn?.applyValue({ args0 -> args0 }))
            .serviceAccount(serviceAccount?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [PodIdentityAssociationArgs].
 */
@PulumiTagMarker
public class PodIdentityAssociationArgsBuilder internal constructor() {
    private var clusterName: Output<String>? = null

    private var namespace: Output<String>? = null

    private var roleArn: Output<String>? = null

    private var serviceAccount: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value The name of the cluster to create the association in.
     */
    @JvmName("npvbunapddgsjjnl")
    public suspend fun clusterName(`value`: Output<String>) {
        this.clusterName = value
    }

    /**
     * @param value The name of the Kubernetes namespace inside the cluster to create the association in. The service account and the pods that use the service account must be in this namespace.
     */
    @JvmName("rpdikdlikmwkkqsq")
    public suspend fun namespace(`value`: Output<String>) {
        this.namespace = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.
     */
    @JvmName("rnfqyrsmrvxugtep")
    public suspend fun roleArn(`value`: Output<String>) {
        this.roleArn = value
    }

    /**
     * @param value The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
     * The following arguments are optional:
     */
    @JvmName("iphhtqmshkjagcrf")
    public suspend fun serviceAccount(`value`: Output<String>) {
        this.serviceAccount = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("obuxhxawcbdukglp")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The name of the cluster to create the association in.
     */
    @JvmName("obauaeibfaekogwg")
    public suspend fun clusterName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterName = mapped
    }

    /**
     * @param value The name of the Kubernetes namespace inside the cluster to create the association in. The service account and the pods that use the service account must be in this namespace.
     */
    @JvmName("owyeyfnxjercerky")
    public suspend fun namespace(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namespace = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.
     */
    @JvmName("pboggcgaqqhckqus")
    public suspend fun roleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.roleArn = mapped
    }

    /**
     * @param value The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
     * The following arguments are optional:
     */
    @JvmName("oaraeibgudubnjwh")
    public suspend fun serviceAccount(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serviceAccount = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("apmkehxmmdvmqptx")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("onehslfnxoofpema")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): PodIdentityAssociationArgs = PodIdentityAssociationArgs(
        clusterName = clusterName,
        namespace = namespace,
        roleArn = roleArn,
        serviceAccount = serviceAccount,
        tags = tags,
    )
}
