@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.elasticsearch.kotlin

import com.pulumi.aws.elasticsearch.DomainArgs.builder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainAdvancedSecurityOptionsArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainAdvancedSecurityOptionsArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainAutoTuneOptionsArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainAutoTuneOptionsArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainClusterConfigArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainClusterConfigArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainCognitoOptionsArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainCognitoOptionsArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainDomainEndpointOptionsArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainDomainEndpointOptionsArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainEbsOptionsArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainEbsOptionsArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainEncryptAtRestArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainEncryptAtRestArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainLogPublishingOptionArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainLogPublishingOptionArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainNodeToNodeEncryptionArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainNodeToNodeEncryptionArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainSnapshotOptionsArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainSnapshotOptionsArgsBuilder
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainVpcOptionsArgs
import com.pulumi.aws.elasticsearch.kotlin.inputs.DomainVpcOptionsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages an AWS Elasticsearch Domain.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.elasticsearch.Domain("example", {
 *     domainName: "example",
 *     elasticsearchVersion: "7.10",
 *     clusterConfig: {
 *         instanceType: "r4.large.elasticsearch",
 *     },
 *     tags: {
 *         Domain: "TestDomain",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.elasticsearch.Domain("example",
 *     domain_name="example",
 *     elasticsearch_version="7.10",
 *     cluster_config={
 *         "instance_type": "r4.large.elasticsearch",
 *     },
 *     tags={
 *         "Domain": "TestDomain",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.ElasticSearch.Domain("example", new()
 *     {
 *         DomainName = "example",
 *         ElasticsearchVersion = "7.10",
 *         ClusterConfig = new Aws.ElasticSearch.Inputs.DomainClusterConfigArgs
 *         {
 *             InstanceType = "r4.large.elasticsearch",
 *         },
 *         Tags =
 *         {
 *             { "Domain", "TestDomain" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/elasticsearch"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := elasticsearch.NewDomain(ctx, "example", &elasticsearch.DomainArgs{
 * 			DomainName:           pulumi.String("example"),
 * 			ElasticsearchVersion: pulumi.String("7.10"),
 * 			ClusterConfig: &elasticsearch.DomainClusterConfigArgs{
 * 				InstanceType: pulumi.String("r4.large.elasticsearch"),
 * 			},
 * 			Tags: pulumi.StringMap{
 * 				"Domain": pulumi.String("TestDomain"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.elasticsearch.Domain;
 * import com.pulumi.aws.elasticsearch.DomainArgs;
 * import com.pulumi.aws.elasticsearch.inputs.DomainClusterConfigArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Domain("example", DomainArgs.builder()
 *             .domainName("example")
 *             .elasticsearchVersion("7.10")
 *             .clusterConfig(DomainClusterConfigArgs.builder()
 *                 .instanceType("r4.large.elasticsearch")
 *                 .build())
 *             .tags(Map.of("Domain", "TestDomain"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:elasticsearch:Domain
 *     properties:
 *       domainName: example
 *       elasticsearchVersion: '7.10'
 *       clusterConfig:
 *         instanceType: r4.large.elasticsearch
 *       tags:
 *         Domain: TestDomain
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Access Policy
 * > See also: `aws.elasticsearch.DomainPolicy` resource
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const config = new pulumi.Config();
 * const domain = config.get("domain") || "tf-test";
 * const current = aws.getRegion({});
 * const currentGetCallerIdentity = aws.getCallerIdentity({});
 * const example = new aws.elasticsearch.Domain("example", {
 *     domainName: domain,
 *     accessPolicies: Promise.all([current, currentGetCallerIdentity])&#46;then(([current, currentGetCallerIdentity]) => `{
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Action": "es:*",
 *       "Principal": "*",
 *       "Effect": "Allow",
 *       "Resource": "arn:aws:es:${current.name}:${currentGetCallerIdentity.accountId}:domain/${domain}/*",
 *       "Condition": {
 *         "IpAddress": {"aws:SourceIp": ["66&#46;193&#46;100&#46;22/32"]}
 *       }
 *     }
 *   ]
 * }
 * `),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * config = pulumi.Config()
 * domain = config.get("domain")
 * if domain is None:
 *     domain = "tf-test"
 * current = aws.get_region()
 * current_get_caller_identity = aws.get_caller_identity()
 * example = aws.elasticsearch.Domain("example",
 *     domain_name=domain,
 *     access_policies=f"""{{
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {{
 *       "Action": "es:*",
 *       "Principal": "*",
 *       "Effect": "Allow",
 *       "Resource": "arn:aws:es:{current.name}:{current_get_caller_identity.account_id}:domain/{domain}/*",
 *       "Condition": {{
 *         "IpAddress": {{"aws:SourceIp": ["66&#46;193&#46;100&#46;22/32"]}}
 *       }}
 *     }}
 *   ]
 * }}
 * """)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var domain = config.Get("domain") ?? "tf-test";
 *     var current = Aws.GetRegion.Invoke();
 *     var currentGetCallerIdentity = Aws.GetCallerIdentity.Invoke();
 *     var example = new Aws.ElasticSearch.Domain("example", new()
 *     {
 *         DomainName = domain,
 *         AccessPolicies = Output.Tuple(current, currentGetCallerIdentity).Apply(values =>
 *         {
 *             var current = values.Item1;
 *             var currentGetCallerIdentity = values.Item2;
 *             return @$"{{
 *   ""Version"": ""2012-10-17"",
 *   ""Statement"": [
 *     {{
 *       ""Action"": ""es:*"",
 *       ""Principal"": ""*"",
 *       ""Effect"": ""Allow"",
 *       ""Resource"": ""arn:aws:es:{current.Apply(getRegionResult => getRegionResult.Name)}:{currentGetCallerIdentity.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId)}:domain/{domain}/*"",
 *       ""Condition"": {{
 *         ""IpAddress"": {{""aws:SourceIp"": [""66&#46;193&#46;100&#46;22/32""]}}
 *       }}
 *     }}
 *   ]
 * }}
 * ";
 *         }),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/elasticsearch"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		cfg := config.New(ctx, "")
 * 		domain := "tf-test"
 * 		if param := cfg.Get("domain"); param != "" {
 * 			domain = param
 * 		}
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		currentGetCallerIdentity, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = elasticsearch.NewDomain(ctx, "example", &elasticsearch.DomainArgs{
 * 			DomainName: pulumi.String(domain),
 * 			AccessPolicies: pulumi.Any(fmt.Sprintf(`{
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Action": "es:*",
 *       "Principal": "*",
 *       "Effect": "Allow",
 *       "Resource": "arn:aws:es:%v:%v:domain/%v/*",
 *       "Condition": {
 *         "IpAddress": {"aws:SourceIp": ["66&#46;193&#46;100&#46;22/32"]}
 *       }
 *     }
 *   ]
 * }
 * `, current.Name, currentGetCallerIdentity.AccountId, domain)),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.elasticsearch.Domain;
 * import com.pulumi.aws.elasticsearch.DomainArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var domain = config.get("domain").orElse("tf-test");
 *         final var current = AwsFunctions.getRegion();
 *         final var currentGetCallerIdentity = AwsFunctions.getCallerIdentity();
 *         var example = new Domain("example", DomainArgs.builder()
 *             .domainName(domain)
 *             .accessPolicies("""
 * {
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Action": "es:*",
 *       "Principal": "*",
 *       "Effect": "Allow",
 *       "Resource": "arn:aws:es:%s:%s:domain/%s/*",
 *       "Condition": {
 *         "IpAddress": {"aws:SourceIp": ["66&#46;193&#46;100&#46;22/32"]}
 *       }
 *     }
 *   ]
 * }
 * ", current.applyValue(getRegionResult -> getRegionResult.name()),currentGetCallerIdentity.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),domain))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   domain:
 *     type: string
 *     default: tf-test
 * resources:
 *   example:
 *     type: aws:elasticsearch:Domain
 *     properties:
 *       domainName: ${domain}
 *       accessPolicies: |
 *         {
 *           "Version": "2012-10-17",
 *           "Statement": [
 *             {
 *               "Action": "es:*",
 *               "Principal": "*",
 *               "Effect": "Allow",
 *               "Resource": "arn:aws:es:${current.name}:${currentGetCallerIdentity.accountId}:domain/${domain}/*",
 *               "Condition": {
 *                 "IpAddress": {"aws:SourceIp": ["66&#46;193&#46;100&#46;22/32"]}
 *               }
 *             }
 *           ]
 *         }
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 *   currentGetCallerIdentity:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Log Publishing to CloudWatch Logs
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const exampleLogGroup = new aws.cloudwatch.LogGroup("example", {name: "example"});
 * const example = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["es&#46;amazonaws&#46;com"],
 *         }],
 *         actions: [
 *             "logs:PutLogEvents",
 *             "logs:PutLogEventsBatch",
 *             "logs:CreateLogStream",
 *         ],
 *         resources: ["arn:aws:logs:*"],
 *     }],
 * });
 * const exampleLogResourcePolicy = new aws.cloudwatch.LogResourcePolicy("example", {
 *     policyName: "example",
 *     policyDocument: example.then(example => example.json),
 * });
 * const exampleDomain = new aws.elasticsearch.Domain("example", {logPublishingOptions: [{
 *     cloudwatchLogGroupArn: exampleLogGroup.arn,
 *     logType: "INDEX_SLOW_LOGS",
 * }]});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example_log_group = aws.cloudwatch.LogGroup("example", name="example")
 * example = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["es&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": [
 *         "logs:PutLogEvents",
 *         "logs:PutLogEventsBatch",
 *         "logs:CreateLogStream",
 *     ],
 *     "resources": ["arn:aws:logs:*"],
 * }])
 * example_log_resource_policy = aws.cloudwatch.LogResourcePolicy("example",
 *     policy_name="example",
 *     policy_document=example.json)
 * example_domain = aws.elasticsearch.Domain("example", log_publishing_options=[{
 *     "cloudwatch_log_group_arn": example_log_group.arn,
 *     "log_type": "INDEX_SLOW_LOGS",
 * }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleLogGroup = new Aws.CloudWatch.LogGroup("example", new()
 *     {
 *         Name = "example",
 *     });
 *     var example = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "es.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "logs:PutLogEvents",
 *                     "logs:PutLogEventsBatch",
 *                     "logs:CreateLogStream",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     "arn:aws:logs:*",
 *                 },
 *             },
 *         },
 *     });
 *     var exampleLogResourcePolicy = new Aws.CloudWatch.LogResourcePolicy("example", new()
 *     {
 *         PolicyName = "example",
 *         PolicyDocument = example.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var exampleDomain = new Aws.ElasticSearch.Domain("example", new()
 *     {
 *         LogPublishingOptions = new[]
 *         {
 *             new Aws.ElasticSearch.Inputs.DomainLogPublishingOptionArgs
 *             {
 *                 CloudwatchLogGroupArn = exampleLogGroup.Arn,
 *                 LogType = "INDEX_SLOW_LOGS",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/elasticsearch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		exampleLogGroup, err := cloudwatch.NewLogGroup(ctx, "example", &cloudwatch.LogGroupArgs{
 * 			Name: pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"es.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"logs:PutLogEvents",
 * 						"logs:PutLogEventsBatch",
 * 						"logs:CreateLogStream",
 * 					},
 * 					Resources: []string{
 * 						"arn:aws:logs:*",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cloudwatch.NewLogResourcePolicy(ctx, "example", &cloudwatch.LogResourcePolicyArgs{
 * 			PolicyName:     pulumi.String("example"),
 * 			PolicyDocument: pulumi.String(example.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = elasticsearch.NewDomain(ctx, "example", &elasticsearch.DomainArgs{
 * 			LogPublishingOptions: elasticsearch.DomainLogPublishingOptionArray{
 * 				&elasticsearch.DomainLogPublishingOptionArgs{
 * 					CloudwatchLogGroupArn: exampleLogGroup.Arn,
 * 					LogType:               pulumi.String("INDEX_SLOW_LOGS"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.cloudwatch.LogGroupArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.cloudwatch.LogResourcePolicy;
 * import com.pulumi.aws.cloudwatch.LogResourcePolicyArgs;
 * import com.pulumi.aws.elasticsearch.Domain;
 * import com.pulumi.aws.elasticsearch.DomainArgs;
 * import com.pulumi.aws.elasticsearch.inputs.DomainLogPublishingOptionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleLogGroup = new LogGroup("exampleLogGroup", LogGroupArgs.builder()
 *             .name("example")
 *             .build());
 *         final var example = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("es.amazonaws.com")
 *                     .build())
 *                 .actions(
 *                     "logs:PutLogEvents",
 *                     "logs:PutLogEventsBatch",
 *                     "logs:CreateLogStream")
 *                 .resources("arn:aws:logs:*")
 *                 .build())
 *             .build());
 *         var exampleLogResourcePolicy = new LogResourcePolicy("exampleLogResourcePolicy", LogResourcePolicyArgs.builder()
 *             .policyName("example")
 *             .policyDocument(example.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var exampleDomain = new Domain("exampleDomain", DomainArgs.builder()
 *             .logPublishingOptions(DomainLogPublishingOptionArgs.builder()
 *                 .cloudwatchLogGroupArn(exampleLogGroup.arn())
 *                 .logType("INDEX_SLOW_LOGS")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleLogGroup:
 *     type: aws:cloudwatch:LogGroup
 *     name: example
 *     properties:
 *       name: example
 *   exampleLogResourcePolicy:
 *     type: aws:cloudwatch:LogResourcePolicy
 *     name: example
 *     properties:
 *       policyName: example
 *       policyDocument: ${example.json}
 *   exampleDomain:
 *     type: aws:elasticsearch:Domain
 *     name: example
 *     properties:
 *       logPublishingOptions:
 *         - cloudwatchLogGroupArn: ${exampleLogGroup.arn}
 *           logType: INDEX_SLOW_LOGS
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - es.amazonaws.com
 *             actions:
 *               - logs:PutLogEvents
 *               - logs:PutLogEventsBatch
 *               - logs:CreateLogStream
 *             resources:
 *               - arn:aws:logs:*
 * ```
 * <!--End PulumiCodeChooser -->
 * ### VPC based ES
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const config = new pulumi.Config();
 * const vpc = config.requireObject("vpc");
 * const domain = config.get("domain") || "tf-test";
 * const selected = aws.ec2.getVpc({
 *     tags: {
 *         Name: vpc,
 *     },
 * });
 * const selectedGetSubnets = selected.then(selected => aws.ec2.getSubnets({
 *     filters: [{
 *         name: "vpc-id",
 *         values: [selected&#46;id],
 *     }],
 *     tags: {
 *         Tier: "private",
 *     },
 * }));
 * const current = aws.getRegion({});
 * const currentGetCallerIdentity = aws.getCallerIdentity({});
 * const es = new aws.ec2.SecurityGroup("es", {
 *     name: `${vpc}-elasticsearch-${domain}`,
 *     description: "Managed by Pulumi",
 *     vpcId: selected.then(selected => selected.id),
 *     ingress: [{
 *         fromPort: 443,
 *         toPort: 443,
 *         protocol: "tcp",
 *         cidrBlocks: [selected&#46;then(selected => selected&#46;cidrBlock)],
 *     }],
 * });
 * const esServiceLinkedRole = new aws.iam.ServiceLinkedRole("es", {awsServiceName: "opensearchservice.amazonaws.com"});
 * const esDomain = new aws.elasticsearch.Domain("es", {
 *     domainName: domain,
 *     elasticsearchVersion: "6.3",
 *     clusterConfig: {
 *         instanceType: "m4.large.elasticsearch",
 *         zoneAwarenessEnabled: true,
 *     },
 *     vpcOptions: {
 *         subnetIds: [
 *             selectedGetSubnets.then(selectedGetSubnets => selectedGetSubnets.ids?.[0]),
 *             selectedGetSubnets.then(selectedGetSubnets => selectedGetSubnets.ids?.[1]),
 *         ],
 *         securityGroupIds: [es&#46;id],
 *     },
 *     advancedOptions: {
 *         "rest.action.multi.allow_explicit_index": "true",
 *     },
 *     accessPolicies: Promise.all([current, currentGetCallerIdentity])&#46;then(([current, currentGetCallerIdentity]) => `{
 * 	"Version": "2012-10-17",
 * 	"Statement": [
 * 		{
 * 			"Action": "es:*",
 * 			"Principal": "*",
 * 			"Effect": "Allow",
 * 			"Resource": "arn:aws:es:${current.name}:${currentGetCallerIdentity.accountId}:domain/${domain}/*"
 * 		}
 * 	]
 * }
 * `),
 *     tags: {
 *         Domain: "TestDomain",
 *     },
 * }, {
 *     dependsOn: [esServiceLinkedRole],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * config = pulumi.Config()
 * vpc = config.require_object("vpc")
 * domain = config.get("domain")
 * if domain is None:
 *     domain = "tf-test"
 * selected = aws.ec2.get_vpc(tags={
 *     "Name": vpc,
 * })
 * selected_get_subnets = aws.ec2.get_subnets(filters=[{
 *         "name": "vpc-id",
 *         "values": [selected&#46;id],
 *     }],
 *     tags={
 *         "Tier": "private",
 *     })
 * current = aws.get_region()
 * current_get_caller_identity = aws.get_caller_identity()
 * es = aws.ec2.SecurityGroup("es",
 *     name=f"{vpc}-elasticsearch-{domain}",
 *     description="Managed by Pulumi",
 *     vpc_id=selected.id,
 *     ingress=[{
 *         "from_port": 443,
 *         "to_port": 443,
 *         "protocol": "tcp",
 *         "cidr_blocks": [selected&#46;cidr_block],
 *     }])
 * es_service_linked_role = aws.iam.ServiceLinkedRole("es", aws_service_name="opensearchservice.amazonaws.com")
 * es_domain = aws.elasticsearch.Domain("es",
 *     domain_name=domain,
 *     elasticsearch_version="6.3",
 *     cluster_config={
 *         "instance_type": "m4.large.elasticsearch",
 *         "zone_awareness_enabled": True,
 *     },
 *     vpc_options={
 *         "subnet_ids": [
 *             selected_get_subnets.ids[0],
 *             selected_get_subnets.ids[1],
 *         ],
 *         "security_group_ids": [es&#46;id],
 *     },
 *     advanced_options={
 *         "rest.action.multi.allow_explicit_index": "true",
 *     },
 *     access_policies=f"""{{
 * \x09"Version": "2012-10-17",
 * \x09"Statement": [
 * \x09\x09{{
 * \x09\x09\x09"Action": "es:*",
 * \x09\x09\x09"Principal": "*",
 * \x09\x09\x09"Effect": "Allow",
 * \x09\x09\x09"Resource": "arn:aws:es:{current.name}:{current_get_caller_identity.account_id}:domain/{domain}/*"
 * \x09\x09}}
 * \x09]
 * }}
 * """,
 *     tags={
 *         "Domain": "TestDomain",
 *     },
 *     opts = pulumi.ResourceOptions(depends_on=[es_service_linked_role]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var vpc = config.RequireObject<dynamic>("vpc");
 *     var domain = config.Get("domain") ?? "tf-test";
 *     var selected = Aws.Ec2.GetVpc.Invoke(new()
 *     {
 *         Tags =
 *         {
 *             { "Name", vpc },
 *         },
 *     });
 *     var selectedGetSubnets = Aws.Ec2.GetSubnets.Invoke(new()
 *     {
 *         Filters = new[]
 *         {
 *             new Aws.Ec2.Inputs.GetSubnetsFilterInputArgs
 *             {
 *                 Name = "vpc-id",
 *                 Values = new[]
 *                 {
 *                     selected.Apply(getVpcResult => getVpcResult.Id),
 *                 },
 *             },
 *         },
 *         Tags =
 *         {
 *             { "Tier", "private" },
 *         },
 *     });
 *     var current = Aws.GetRegion.Invoke();
 *     var currentGetCallerIdentity = Aws.GetCallerIdentity.Invoke();
 *     var es = new Aws.Ec2.SecurityGroup("es", new()
 *     {
 *         Name = $"{vpc}-elasticsearch-{domain}",
 *         Description = "Managed by Pulumi",
 *         VpcId = selected.Apply(getVpcResult => getVpcResult.Id),
 *         Ingress = new[]
 *         {
 *             new Aws.Ec2.Inputs.SecurityGroupIngressArgs
 *             {
 *                 FromPort = 443,
 *                 ToPort = 443,
 *                 Protocol = "tcp",
 *                 CidrBlocks = new[]
 *                 {
 *                     selected.Apply(getVpcResult => getVpcResult.CidrBlock),
 *                 },
 *             },
 *         },
 *     });
 *     var esServiceLinkedRole = new Aws.Iam.ServiceLinkedRole("es", new()
 *     {
 *         AwsServiceName = "opensearchservice.amazonaws.com",
 *     });
 *     var esDomain = new Aws.ElasticSearch.Domain("es", new()
 *     {
 *         DomainName = domain,
 *         ElasticsearchVersion = "6.3",
 *         ClusterConfig = new Aws.ElasticSearch.Inputs.DomainClusterConfigArgs
 *         {
 *             InstanceType = "m4.large.elasticsearch",
 *             ZoneAwarenessEnabled = true,
 *         },
 *         VpcOptions = new Aws.ElasticSearch.Inputs.DomainVpcOptionsArgs
 *         {
 *             SubnetIds = new[]
 *             {
 *                 selectedGetSubnets.Apply(getSubnetsResult => getSubnetsResult.Ids[0]),
 *                 selectedGetSubnets.Apply(getSubnetsResult => getSubnetsResult.Ids[1]),
 *             },
 *             SecurityGroupIds = new[]
 *             {
 *                 es.Id,
 *             },
 *         },
 *         AdvancedOptions =
 *         {
 *             { "rest.action.multi.allow_explicit_index", "true" },
 *         },
 *         AccessPolicies = Output.Tuple(current, currentGetCallerIdentity).Apply(values =>
 *         {
 *             var current = values.Item1;
 *             var currentGetCallerIdentity = values.Item2;
 *             return @$"{{
 * 	""Version"": ""2012-10-17"",
 * 	""Statement"": [
 * 		{{
 * 			""Action"": ""es:*"",
 * 			""Principal"": ""*"",
 * 			""Effect"": ""Allow"",
 * 			""Resource"": ""arn:aws:es:{current.Apply(getRegionResult => getRegionResult.Name)}:{currentGetCallerIdentity.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId)}:domain/{domain}/*""
 * 		}}
 * 	]
 * }}
 * ";
 *         }),
 *         Tags =
 *         {
 *             { "Domain", "TestDomain" },
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             esServiceLinkedRole,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/elasticsearch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * cfg := config.New(ctx, "")
 * vpc := cfg.RequireObject("vpc")
 * domain := "tf-test";
 * if param := cfg.Get("domain"); param != ""{
 * domain = param
 * }
 * selected, err := ec2.LookupVpc(ctx, &ec2.LookupVpcArgs{
 * Tags: pulumi.StringMap{
 * "Name": vpc,
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * selectedGetSubnets, err := ec2.GetSubnets(ctx, &ec2.GetSubnetsArgs{
 * Filters: []ec2.GetSubnetsFilter{
 * {
 * Name: "vpc-id",
 * Values: interface{}{
 * selected.Id,
 * },
 * },
 * },
 * Tags: map[string]interface{}{
 * "Tier": "private",
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * current, err := aws.GetRegion(ctx, nil, nil);
 * if err != nil {
 * return err
 * }
 * currentGetCallerIdentity, err := aws.GetCallerIdentity(ctx, nil, nil);
 * if err != nil {
 * return err
 * }
 * es, err := ec2.NewSecurityGroup(ctx, "es", &ec2.SecurityGroupArgs{
 * Name: pulumi.Sprintf("%v-elasticsearch-%v", vpc, domain),
 * Description: pulumi.String("Managed by Pulumi"),
 * VpcId: pulumi.String(selected.Id),
 * Ingress: ec2.SecurityGroupIngressArray{
 * &ec2.SecurityGroupIngressArgs{
 * FromPort: pulumi.Int(443),
 * ToPort: pulumi.Int(443),
 * Protocol: pulumi.String("tcp"),
 * CidrBlocks: pulumi.StringArray{
 * pulumi.String(selected.CidrBlock),
 * },
 * },
 * },
 * })
 * if err != nil {
 * return err
 * }
 * esServiceLinkedRole, err := iam.NewServiceLinkedRole(ctx, "es", &iam.ServiceLinkedRoleArgs{
 * AwsServiceName: pulumi.String("opensearchservice.amazonaws.com"),
 * })
 * if err != nil {
 * return err
 * }
 * _, err = elasticsearch.NewDomain(ctx, "es", &elasticsearch.DomainArgs{
 * DomainName: pulumi.String(domain),
 * ElasticsearchVersion: pulumi.String("6.3"),
 * ClusterConfig: &elasticsearch.DomainClusterConfigArgs{
 * InstanceType: pulumi.String("m4.large.elasticsearch"),
 * ZoneAwarenessEnabled: pulumi.Bool(true),
 * },
 * VpcOptions: &elasticsearch.DomainVpcOptionsArgs{
 * SubnetIds: pulumi.StringArray{
 * pulumi.String(selectedGetSubnets.Ids[0]),
 * pulumi.String(selectedGetSubnets.Ids[1]),
 * },
 * SecurityGroupIds: pulumi.StringArray{
 * es.ID(),
 * },
 * },
 * AdvancedOptions: pulumi.StringMap{
 * "rest.action.multi.allow_explicit_index": pulumi.String("true"),
 * },
 * AccessPolicies: pulumi.Any(fmt.Sprintf(`{
 * 	"Version": "2012-10-17",
 * 	"Statement": [
 * 		{
 * 			"Action": "es:*",
 * 			"Principal": "*",
 * 			"Effect": "Allow",
 * 			"Resource": "arn:aws:es:%v:%v:domain/%v/*"
 * 		}
 * 	]
 * }
 * `, current.Name, currentGetCallerIdentity.AccountId, domain)),
 * Tags: pulumi.StringMap{
 * "Domain": pulumi.String("TestDomain"),
 * },
 * }, pulumi.DependsOn([]pulumi.Resource{
 * esServiceLinkedRole,
 * }))
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.Ec2Functions;
 * import com.pulumi.aws.ec2.inputs.GetVpcArgs;
 * import com.pulumi.aws.ec2.inputs.GetSubnetsArgs;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.ec2.SecurityGroup;
 * import com.pulumi.aws.ec2.SecurityGroupArgs;
 * import com.pulumi.aws.ec2.inputs.SecurityGroupIngressArgs;
 * import com.pulumi.aws.iam.ServiceLinkedRole;
 * import com.pulumi.aws.iam.ServiceLinkedRoleArgs;
 * import com.pulumi.aws.elasticsearch.Domain;
 * import com.pulumi.aws.elasticsearch.DomainArgs;
 * import com.pulumi.aws.elasticsearch.inputs.DomainClusterConfigArgs;
 * import com.pulumi.aws.elasticsearch.inputs.DomainVpcOptionsArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var vpc = config.get("vpc");
 *         final var domain = config.get("domain").orElse("tf-test");
 *         final var selected = Ec2Functions.getVpc(GetVpcArgs.builder()
 *             .tags(Map.of("Name", vpc))
 *             .build());
 *         final var selectedGetSubnets = Ec2Functions.getSubnets(GetSubnetsArgs.builder()
 *             .filters(GetSubnetsFilterArgs.builder()
 *                 .name("vpc-id")
 *                 .values(selected.applyValue(getVpcResult -> getVpcResult.id()))
 *                 .build())
 *             .tags(Map.of("Tier", "private"))
 *             .build());
 *         final var current = AwsFunctions.getRegion();
 *         final var currentGetCallerIdentity = AwsFunctions.getCallerIdentity();
 *         var es = new SecurityGroup("es", SecurityGroupArgs.builder()
 *             .name(String.format("%s-elasticsearch-%s", vpc,domain))
 *             .description("Managed by Pulumi")
 *             .vpcId(selected.applyValue(getVpcResult -> getVpcResult.id()))
 *             .ingress(SecurityGroupIngressArgs.builder()
 *                 .fromPort(443)
 *                 .toPort(443)
 *                 .protocol("tcp")
 *                 .cidrBlocks(selected.applyValue(getVpcResult -> getVpcResult.cidrBlock()))
 *                 .build())
 *             .build());
 *         var esServiceLinkedRole = new ServiceLinkedRole("esServiceLinkedRole", ServiceLinkedRoleArgs.builder()
 *             .awsServiceName("opensearchservice.amazonaws.com")
 *             .build());
 *         var esDomain = new Domain("esDomain", DomainArgs.builder()
 *             .domainName(domain)
 *             .elasticsearchVersion("6.3")
 *             .clusterConfig(DomainClusterConfigArgs.builder()
 *                 .instanceType("m4.large.elasticsearch")
 *                 .zoneAwarenessEnabled(true)
 *                 .build())
 *             .vpcOptions(DomainVpcOptionsArgs.builder()
 *                 .subnetIds(
 *                     selectedGetSubnets.applyValue(getSubnetsResult -> getSubnetsResult.ids()[0]),
 *                     selectedGetSubnets.applyValue(getSubnetsResult -> getSubnetsResult.ids()[1]))
 *                 .securityGroupIds(es.id())
 *                 .build())
 *             .advancedOptions(Map.of("rest.action.multi.allow_explicit_index", "true"))
 *             .accessPolicies("""
 * {
 * 	"Version": "2012-10-17",
 * 	"Statement": [
 * 		{
 * 			"Action": "es:*",
 * 			"Principal": "*",
 * 			"Effect": "Allow",
 * 			"Resource": "arn:aws:es:%s:%s:domain/%s/*"
 * 		}
 * 	]
 * }
 * ", current.applyValue(getRegionResult -> getRegionResult.name()),currentGetCallerIdentity.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),domain))
 *             .tags(Map.of("Domain", "TestDomain"))
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(esServiceLinkedRole)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   vpc:
 *     type: dynamic
 *   domain:
 *     type: string
 *     default: tf-test
 * resources:
 *   es:
 *     type: aws:ec2:SecurityGroup
 *     properties:
 *       name: ${vpc}-elasticsearch-${domain}
 *       description: Managed by Pulumi
 *       vpcId: ${selected.id}
 *       ingress:
 *         - fromPort: 443
 *           toPort: 443
 *           protocol: tcp
 *           cidrBlocks:
 *             - ${selected.cidrBlock}
 *   esServiceLinkedRole:
 *     type: aws:iam:ServiceLinkedRole
 *     name: es
 *     properties:
 *       awsServiceName: opensearchservice.amazonaws.com
 *   esDomain:
 *     type: aws:elasticsearch:Domain
 *     name: es
 *     properties:
 *       domainName: ${domain}
 *       elasticsearchVersion: '6.3'
 *       clusterConfig:
 *         instanceType: m4.large.elasticsearch
 *         zoneAwarenessEnabled: true
 *       vpcOptions:
 *         subnetIds:
 *           - ${selectedGetSubnets.ids[0]}
 *           - ${selectedGetSubnets.ids[1]}
 *         securityGroupIds:
 *           - ${es.id}
 *       advancedOptions:
 *         rest.action.multi.allow_explicit_index: 'true'
 *       accessPolicies: |
 *         {
 *         	"Version": "2012-10-17",
 *         	"Statement": [
 *         		{
 *         			"Action": "es:*",
 *         			"Principal": "*",
 *         			"Effect": "Allow",
 *         			"Resource": "arn:aws:es:${current.name}:${currentGetCallerIdentity.accountId}:domain/${domain}/*"
 *         		}
 *         	]
 *         }
 *       tags:
 *         Domain: TestDomain
 *     options:
 *       dependson:
 *         - ${esServiceLinkedRole}
 * variables:
 *   selected:
 *     fn::invoke:
 *       Function: aws:ec2:getVpc
 *       Arguments:
 *         tags:
 *           Name: ${vpc}
 *   selectedGetSubnets:
 *     fn::invoke:
 *       Function: aws:ec2:getSubnets
 *       Arguments:
 *         filters:
 *           - name: vpc-id
 *             values:
 *               - ${selected.id}
 *         tags:
 *           Tier: private
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 *   currentGetCallerIdentity:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Elasticsearch domains using the `domain_name`. For example:
 * ```sh
 * $ pulumi import aws:elasticsearch/domain:Domain example domain_name
 * ```
 * @property accessPolicies IAM policy document specifying the access policies for the domain.
 * @property advancedOptions Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your Elasticsearch domain on every apply.
 * @property advancedSecurityOptions Configuration block for [fine-grained access control](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/fgac.html). Detailed below.
 * @property autoTuneOptions Configuration block for the Auto-Tune options of the domain. Detailed below.
 * @property clusterConfig Configuration block for the cluster of the domain. Detailed below.
 * @property cognitoOptions Configuration block for authenticating Kibana with Cognito. Detailed below.
 * @property domainEndpointOptions Configuration block for domain endpoint HTTP(S) related options. Detailed below.
 * @property domainName Name of the domain.
 * The following arguments are optional:
 * @property ebsOptions Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/elasticsearch-service/pricing/). Detailed below.
 * @property elasticsearchVersion Version of Elasticsearch to deploy. Defaults to `1.5`.
 * @property encryptAtRest Configuration block for encrypt at rest options. Only available for [certain instance types](http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html). Detailed below.
 * @property logPublishingOptions Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
 * @property nodeToNodeEncryption Configuration block for node-to-node encryption options. Detailed below.
 * @property snapshotOptions Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running Elasticsearch 5.3 and later, Amazon ES takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions of Elasticsearch, Amazon ES takes daily automated snapshots.
 * @property tags Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcOptions Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html#es-vpc-limitations)). Detailed below.
 * */*/*/*/*/*/*/*/*/*/*/*/
 */
public data class DomainArgs(
    public val accessPolicies: Output<String>? = null,
    public val advancedOptions: Output<Map<String, String>>? = null,
    public val advancedSecurityOptions: Output<DomainAdvancedSecurityOptionsArgs>? = null,
    public val autoTuneOptions: Output<DomainAutoTuneOptionsArgs>? = null,
    public val clusterConfig: Output<DomainClusterConfigArgs>? = null,
    public val cognitoOptions: Output<DomainCognitoOptionsArgs>? = null,
    public val domainEndpointOptions: Output<DomainDomainEndpointOptionsArgs>? = null,
    public val domainName: Output<String>? = null,
    public val ebsOptions: Output<DomainEbsOptionsArgs>? = null,
    public val elasticsearchVersion: Output<String>? = null,
    public val encryptAtRest: Output<DomainEncryptAtRestArgs>? = null,
    public val logPublishingOptions: Output<List<DomainLogPublishingOptionArgs>>? = null,
    public val nodeToNodeEncryption: Output<DomainNodeToNodeEncryptionArgs>? = null,
    public val snapshotOptions: Output<DomainSnapshotOptionsArgs>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcOptions: Output<DomainVpcOptionsArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.elasticsearch.DomainArgs> {
    override fun toJava(): com.pulumi.aws.elasticsearch.DomainArgs =
        com.pulumi.aws.elasticsearch.DomainArgs.builder()
            .accessPolicies(accessPolicies?.applyValue({ args0 -> args0 }))
            .advancedOptions(
                advancedOptions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .advancedSecurityOptions(
                advancedSecurityOptions?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .autoTuneOptions(autoTuneOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .clusterConfig(clusterConfig?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .cognitoOptions(cognitoOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .domainEndpointOptions(
                domainEndpointOptions?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .domainName(domainName?.applyValue({ args0 -> args0 }))
            .ebsOptions(ebsOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .elasticsearchVersion(elasticsearchVersion?.applyValue({ args0 -> args0 }))
            .encryptAtRest(encryptAtRest?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .logPublishingOptions(
                logPublishingOptions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .nodeToNodeEncryption(
                nodeToNodeEncryption?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .snapshotOptions(snapshotOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .vpcOptions(vpcOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [DomainArgs].
 */
@PulumiTagMarker
public class DomainArgsBuilder internal constructor() {
    private var accessPolicies: Output<String>? = null

    private var advancedOptions: Output<Map<String, String>>? = null

    private var advancedSecurityOptions: Output<DomainAdvancedSecurityOptionsArgs>? = null

    private var autoTuneOptions: Output<DomainAutoTuneOptionsArgs>? = null

    private var clusterConfig: Output<DomainClusterConfigArgs>? = null

    private var cognitoOptions: Output<DomainCognitoOptionsArgs>? = null

    private var domainEndpointOptions: Output<DomainDomainEndpointOptionsArgs>? = null

    private var domainName: Output<String>? = null

    private var ebsOptions: Output<DomainEbsOptionsArgs>? = null

    private var elasticsearchVersion: Output<String>? = null

    private var encryptAtRest: Output<DomainEncryptAtRestArgs>? = null

    private var logPublishingOptions: Output<List<DomainLogPublishingOptionArgs>>? = null

    private var nodeToNodeEncryption: Output<DomainNodeToNodeEncryptionArgs>? = null

    private var snapshotOptions: Output<DomainSnapshotOptionsArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcOptions: Output<DomainVpcOptionsArgs>? = null

    /**
     * @param value IAM policy document specifying the access policies for the domain.
     */
    @JvmName("eilqoyslvvdyurhf")
    public suspend fun accessPolicies(`value`: Output<String>) {
        this.accessPolicies = value
    }

    /**
     * @param value Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your Elasticsearch domain on every apply.
     */
    @JvmName("gdbsstcagiqskwat")
    public suspend fun advancedOptions(`value`: Output<Map<String, String>>) {
        this.advancedOptions = value
    }

    /**
     * @param value Configuration block for [fine-grained access control](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/fgac.html). Detailed below.
     */
    @JvmName("yqkorydomwgqgwrp")
    public suspend fun advancedSecurityOptions(`value`: Output<DomainAdvancedSecurityOptionsArgs>) {
        this.advancedSecurityOptions = value
    }

    /**
     * @param value Configuration block for the Auto-Tune options of the domain. Detailed below.
     */
    @JvmName("wughiasevgayrdtb")
    public suspend fun autoTuneOptions(`value`: Output<DomainAutoTuneOptionsArgs>) {
        this.autoTuneOptions = value
    }

    /**
     * @param value Configuration block for the cluster of the domain. Detailed below.
     */
    @JvmName("tcyqsfqobrpelssc")
    public suspend fun clusterConfig(`value`: Output<DomainClusterConfigArgs>) {
        this.clusterConfig = value
    }

    /**
     * @param value Configuration block for authenticating Kibana with Cognito. Detailed below.
     */
    @JvmName("jtollokofhebiyyf")
    public suspend fun cognitoOptions(`value`: Output<DomainCognitoOptionsArgs>) {
        this.cognitoOptions = value
    }

    /**
     * @param value Configuration block for domain endpoint HTTP(S) related options. Detailed below.
     */
    @JvmName("gjfriijqsppbhxhf")
    public suspend fun domainEndpointOptions(`value`: Output<DomainDomainEndpointOptionsArgs>) {
        this.domainEndpointOptions = value
    }

    /**
     * @param value Name of the domain.
     * The following arguments are optional:
     */
    @JvmName("efsnrpymcayabrow")
    public suspend fun domainName(`value`: Output<String>) {
        this.domainName = value
    }

    /**
     * @param value Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/elasticsearch-service/pricing/). Detailed below.
     */
    @JvmName("yccpnwgrctgdnbxd")
    public suspend fun ebsOptions(`value`: Output<DomainEbsOptionsArgs>) {
        this.ebsOptions = value
    }

    /**
     * @param value Version of Elasticsearch to deploy. Defaults to `1.5`.
     */
    @JvmName("thnynhxmiinalqky")
    public suspend fun elasticsearchVersion(`value`: Output<String>) {
        this.elasticsearchVersion = value
    }

    /**
     * @param value Configuration block for encrypt at rest options. Only available for [certain instance types](http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html). Detailed below.
     */
    @JvmName("oiupaiesxmavpbqo")
    public suspend fun encryptAtRest(`value`: Output<DomainEncryptAtRestArgs>) {
        this.encryptAtRest = value
    }

    /**
     * @param value Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("wqkqlwxbrckcrvaa")
    public suspend fun logPublishingOptions(`value`: Output<List<DomainLogPublishingOptionArgs>>) {
        this.logPublishingOptions = value
    }

    @JvmName("gdhqkhntpifhutme")
    public suspend fun logPublishingOptions(vararg values: Output<DomainLogPublishingOptionArgs>) {
        this.logPublishingOptions = Output.all(values.asList())
    }

    /**
     * @param values Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("fngmpjtshgiabfdm")
    public suspend fun logPublishingOptions(values: List<Output<DomainLogPublishingOptionArgs>>) {
        this.logPublishingOptions = Output.all(values)
    }

    /**
     * @param value Configuration block for node-to-node encryption options. Detailed below.
     */
    @JvmName("ivuqvmuetgxhlxmc")
    public suspend fun nodeToNodeEncryption(`value`: Output<DomainNodeToNodeEncryptionArgs>) {
        this.nodeToNodeEncryption = value
    }

    /**
     * @param value Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running Elasticsearch 5.3 and later, Amazon ES takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions of Elasticsearch, Amazon ES takes daily automated snapshots.
     */
    @JvmName("wrhwkwmlwoytkefn")
    public suspend fun snapshotOptions(`value`: Output<DomainSnapshotOptionsArgs>) {
        this.snapshotOptions = value
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("oqhnmkxvcinsweiq")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html#es-vpc-limitations)). Detailed below.
     */
    @JvmName("gcjwtcxfjsvooheu")
    public suspend fun vpcOptions(`value`: Output<DomainVpcOptionsArgs>) {
        this.vpcOptions = value
    }

    /**
     * @param value IAM policy document specifying the access policies for the domain.
     */
    @JvmName("rleryouoqctpmvnx")
    public suspend fun accessPolicies(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessPolicies = mapped
    }

    /**
     * @param value Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your Elasticsearch domain on every apply.
     */
    @JvmName("fwjxjyehjxjollqo")
    public suspend fun advancedOptions(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.advancedOptions = mapped
    }

    /**
     * @param values Key-value string pairs to specify advanced configuration options. Note that the values for these configuration options must be strings (wrapped in quotes) or they may be wrong and cause a perpetual diff, causing the provider to want to recreate your Elasticsearch domain on every apply.
     */
    @JvmName("ccuycfnflapiakha")
    public fun advancedOptions(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.advancedOptions = mapped
    }

    /**
     * @param value Configuration block for [fine-grained access control](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/fgac.html). Detailed below.
     */
    @JvmName("xknwlyxlhwejibnu")
    public suspend fun advancedSecurityOptions(`value`: DomainAdvancedSecurityOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.advancedSecurityOptions = mapped
    }

    /**
     * @param argument Configuration block for [fine-grained access control](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/fgac.html). Detailed below.
     */
    @JvmName("kajinohuwytnvimn")
    public suspend fun advancedSecurityOptions(argument: suspend DomainAdvancedSecurityOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainAdvancedSecurityOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.advancedSecurityOptions = mapped
    }

    /**
     * @param value Configuration block for the Auto-Tune options of the domain. Detailed below.
     */
    @JvmName("cfhlkqjsucvajgoi")
    public suspend fun autoTuneOptions(`value`: DomainAutoTuneOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoTuneOptions = mapped
    }

    /**
     * @param argument Configuration block for the Auto-Tune options of the domain. Detailed below.
     */
    @JvmName("xbhcqwcyhnhybhts")
    public suspend fun autoTuneOptions(argument: suspend DomainAutoTuneOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainAutoTuneOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.autoTuneOptions = mapped
    }

    /**
     * @param value Configuration block for the cluster of the domain. Detailed below.
     */
    @JvmName("anfnurduvewvllgo")
    public suspend fun clusterConfig(`value`: DomainClusterConfigArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterConfig = mapped
    }

    /**
     * @param argument Configuration block for the cluster of the domain. Detailed below.
     */
    @JvmName("isiplndkqfkahluj")
    public suspend fun clusterConfig(argument: suspend DomainClusterConfigArgsBuilder.() -> Unit) {
        val toBeMapped = DomainClusterConfigArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.clusterConfig = mapped
    }

    /**
     * @param value Configuration block for authenticating Kibana with Cognito. Detailed below.
     */
    @JvmName("mpikjdccmrpkyvaf")
    public suspend fun cognitoOptions(`value`: DomainCognitoOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cognitoOptions = mapped
    }

    /**
     * @param argument Configuration block for authenticating Kibana with Cognito. Detailed below.
     */
    @JvmName("qaygnrbmcxovanrd")
    public suspend fun cognitoOptions(argument: suspend DomainCognitoOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainCognitoOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.cognitoOptions = mapped
    }

    /**
     * @param value Configuration block for domain endpoint HTTP(S) related options. Detailed below.
     */
    @JvmName("dprbwkvktpwqnlux")
    public suspend fun domainEndpointOptions(`value`: DomainDomainEndpointOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainEndpointOptions = mapped
    }

    /**
     * @param argument Configuration block for domain endpoint HTTP(S) related options. Detailed below.
     */
    @JvmName("dyvjdfonpeppbywp")
    public suspend fun domainEndpointOptions(argument: suspend DomainDomainEndpointOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainDomainEndpointOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.domainEndpointOptions = mapped
    }

    /**
     * @param value Name of the domain.
     * The following arguments are optional:
     */
    @JvmName("mvasaydjkxseshpa")
    public suspend fun domainName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainName = mapped
    }

    /**
     * @param value Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/elasticsearch-service/pricing/). Detailed below.
     */
    @JvmName("olkbbmeocuptlpuf")
    public suspend fun ebsOptions(`value`: DomainEbsOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ebsOptions = mapped
    }

    /**
     * @param argument Configuration block for EBS related options, may be required based on chosen [instance size](https://aws.amazon.com/elasticsearch-service/pricing/). Detailed below.
     */
    @JvmName("tdgualstravvabfb")
    public suspend fun ebsOptions(argument: suspend DomainEbsOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainEbsOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.ebsOptions = mapped
    }

    /**
     * @param value Version of Elasticsearch to deploy. Defaults to `1.5`.
     */
    @JvmName("sfmblenpkhhdxrlk")
    public suspend fun elasticsearchVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.elasticsearchVersion = mapped
    }

    /**
     * @param value Configuration block for encrypt at rest options. Only available for [certain instance types](http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html). Detailed below.
     */
    @JvmName("wtijlliqmimwpepf")
    public suspend fun encryptAtRest(`value`: DomainEncryptAtRestArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.encryptAtRest = mapped
    }

    /**
     * @param argument Configuration block for encrypt at rest options. Only available for [certain instance types](http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html). Detailed below.
     */
    @JvmName("ytirilxsdctvlwgi")
    public suspend fun encryptAtRest(argument: suspend DomainEncryptAtRestArgsBuilder.() -> Unit) {
        val toBeMapped = DomainEncryptAtRestArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.encryptAtRest = mapped
    }

    /**
     * @param value Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("oxdhskahvwbpyakc")
    public suspend fun logPublishingOptions(`value`: List<DomainLogPublishingOptionArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logPublishingOptions = mapped
    }

    /**
     * @param argument Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("syyxplhxvscdamap")
    public suspend fun logPublishingOptions(argument: List<suspend DomainLogPublishingOptionArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            DomainLogPublishingOptionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.logPublishingOptions = mapped
    }

    /**
     * @param argument Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("aylioodqjpvbbtnt")
    public suspend fun logPublishingOptions(vararg argument: suspend DomainLogPublishingOptionArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            DomainLogPublishingOptionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.logPublishingOptions = mapped
    }

    /**
     * @param argument Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("xstyxftcxwdqtmtj")
    public suspend fun logPublishingOptions(argument: suspend DomainLogPublishingOptionArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            DomainLogPublishingOptionArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.logPublishingOptions = mapped
    }

    /**
     * @param values Configuration block for publishing slow and application logs to CloudWatch Logs. This block can be declared multiple times, for each log_type, within the same resource. Detailed below.
     */
    @JvmName("lfcwhluadvhcpeoy")
    public suspend fun logPublishingOptions(vararg values: DomainLogPublishingOptionArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.logPublishingOptions = mapped
    }

    /**
     * @param value Configuration block for node-to-node encryption options. Detailed below.
     */
    @JvmName("ebnyxmxphfsajtcp")
    public suspend fun nodeToNodeEncryption(`value`: DomainNodeToNodeEncryptionArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.nodeToNodeEncryption = mapped
    }

    /**
     * @param argument Configuration block for node-to-node encryption options. Detailed below.
     */
    @JvmName("otruogslrxqlllnt")
    public suspend fun nodeToNodeEncryption(argument: suspend DomainNodeToNodeEncryptionArgsBuilder.() -> Unit) {
        val toBeMapped = DomainNodeToNodeEncryptionArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.nodeToNodeEncryption = mapped
    }

    /**
     * @param value Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running Elasticsearch 5.3 and later, Amazon ES takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions of Elasticsearch, Amazon ES takes daily automated snapshots.
     */
    @JvmName("qobontfinbwuajbn")
    public suspend fun snapshotOptions(`value`: DomainSnapshotOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotOptions = mapped
    }

    /**
     * @param argument Configuration block for snapshot related options. Detailed below. DEPRECATED. For domains running Elasticsearch 5.3 and later, Amazon ES takes hourly automated snapshots, making this setting irrelevant. For domains running earlier versions of Elasticsearch, Amazon ES takes daily automated snapshots.
     */
    @JvmName("usqfrigdhkcmgxdl")
    public suspend fun snapshotOptions(argument: suspend DomainSnapshotOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainSnapshotOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.snapshotOptions = mapped
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("nkvlcjiodrloxtiv")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("irfhrkwqhoopjkpj")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html#es-vpc-limitations)). Detailed below.
     */
    @JvmName("xvhpkiqfpxmkcyww")
    public suspend fun vpcOptions(`value`: DomainVpcOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcOptions = mapped
    }

    /**
     * @param argument Configuration block for VPC related options. Adding or removing this configuration forces a new resource ([documentation](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html#es-vpc-limitations)). Detailed below.
     */
    @JvmName("auaqlltrggngefne")
    public suspend fun vpcOptions(argument: suspend DomainVpcOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = DomainVpcOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.vpcOptions = mapped
    }

    internal fun build(): DomainArgs = DomainArgs(
        accessPolicies = accessPolicies,
        advancedOptions = advancedOptions,
        advancedSecurityOptions = advancedSecurityOptions,
        autoTuneOptions = autoTuneOptions,
        clusterConfig = clusterConfig,
        cognitoOptions = cognitoOptions,
        domainEndpointOptions = domainEndpointOptions,
        domainName = domainName,
        ebsOptions = ebsOptions,
        elasticsearchVersion = elasticsearchVersion,
        encryptAtRest = encryptAtRest,
        logPublishingOptions = logPublishingOptions,
        nodeToNodeEncryption = nodeToNodeEncryption,
        snapshotOptions = snapshotOptions,
        tags = tags,
        vpcOptions = vpcOptions,
    )
}
