@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.glue.kotlin

import com.pulumi.aws.glue.JobArgs.builder
import com.pulumi.aws.glue.kotlin.inputs.JobCommandArgs
import com.pulumi.aws.glue.kotlin.inputs.JobCommandArgsBuilder
import com.pulumi.aws.glue.kotlin.inputs.JobExecutionPropertyArgs
import com.pulumi.aws.glue.kotlin.inputs.JobExecutionPropertyArgsBuilder
import com.pulumi.aws.glue.kotlin.inputs.JobNotificationPropertyArgs
import com.pulumi.aws.glue.kotlin.inputs.JobNotificationPropertyArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Double
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a Glue Job resource.
 * > Glue functionality, such as monitoring and logging of jobs, is typically managed with the `default_arguments` argument. See the [Special Parameters Used by AWS Glue](https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html) topic in the Glue developer guide for additional information.
 * ## Example Usage
 * ### Python Job
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.glue.Job("example", {
 *     name: "example",
 *     roleArn: exampleAwsIamRole.arn,
 *     command: {
 *         scriptLocation: `s3://${exampleAwsS3Bucket.bucket}/example.py`,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.glue.Job("example",
 *     name="example",
 *     role_arn=example_aws_iam_role["arn"],
 *     command={
 *         "script_location": f"s3://{example_aws_s3_bucket['bucket']}/example.py",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Glue.Job("example", new()
 *     {
 *         Name = "example",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Command = new Aws.Glue.Inputs.JobCommandArgs
 *         {
 *             ScriptLocation = $"s3://{exampleAwsS3Bucket.Bucket}/example.py",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/glue"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := glue.NewJob(ctx, "example", &glue.JobArgs{
 * 			Name:    pulumi.String("example"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Command: &glue.JobCommandArgs{
 * 				ScriptLocation: pulumi.Sprintf("s3://%v/example.py", exampleAwsS3Bucket.Bucket),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.glue.Job;
 * import com.pulumi.aws.glue.JobArgs;
 * import com.pulumi.aws.glue.inputs.JobCommandArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Job("example", JobArgs.builder()
 *             .name("example")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .command(JobCommandArgs.builder()
 *                 .scriptLocation(String.format("s3://%s/example.py", exampleAwsS3Bucket.bucket()))
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:glue:Job
 *     properties:
 *       name: example
 *       roleArn: ${exampleAwsIamRole.arn}
 *       command:
 *         scriptLocation: s3://${exampleAwsS3Bucket.bucket}/example.py
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Ray Job
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.glue.Job("example", {
 *     name: "example",
 *     roleArn: exampleAwsIamRole.arn,
 *     glueVersion: "4.0",
 *     workerType: "Z.2X",
 *     command: {
 *         name: "glueray",
 *         pythonVersion: "3.9",
 *         runtime: "Ray2.4",
 *         scriptLocation: `s3://${exampleAwsS3Bucket.bucket}/example.py`,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.glue.Job("example",
 *     name="example",
 *     role_arn=example_aws_iam_role["arn"],
 *     glue_version="4.0",
 *     worker_type="Z.2X",
 *     command={
 *         "name": "glueray",
 *         "python_version": "3.9",
 *         "runtime": "Ray2.4",
 *         "script_location": f"s3://{example_aws_s3_bucket['bucket']}/example.py",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Glue.Job("example", new()
 *     {
 *         Name = "example",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         GlueVersion = "4.0",
 *         WorkerType = "Z.2X",
 *         Command = new Aws.Glue.Inputs.JobCommandArgs
 *         {
 *             Name = "glueray",
 *             PythonVersion = "3.9",
 *             Runtime = "Ray2.4",
 *             ScriptLocation = $"s3://{exampleAwsS3Bucket.Bucket}/example.py",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/glue"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := glue.NewJob(ctx, "example", &glue.JobArgs{
 * 			Name:        pulumi.String("example"),
 * 			RoleArn:     pulumi.Any(exampleAwsIamRole.Arn),
 * 			GlueVersion: pulumi.String("4.0"),
 * 			WorkerType:  pulumi.String("Z.2X"),
 * 			Command: &glue.JobCommandArgs{
 * 				Name:           pulumi.String("glueray"),
 * 				PythonVersion:  pulumi.String("3.9"),
 * 				Runtime:        pulumi.String("Ray2.4"),
 * 				ScriptLocation: pulumi.Sprintf("s3://%v/example.py", exampleAwsS3Bucket.Bucket),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.glue.Job;
 * import com.pulumi.aws.glue.JobArgs;
 * import com.pulumi.aws.glue.inputs.JobCommandArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Job("example", JobArgs.builder()
 *             .name("example")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .glueVersion("4.0")
 *             .workerType("Z.2X")
 *             .command(JobCommandArgs.builder()
 *                 .name("glueray")
 *                 .pythonVersion("3.9")
 *                 .runtime("Ray2.4")
 *                 .scriptLocation(String.format("s3://%s/example.py", exampleAwsS3Bucket.bucket()))
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:glue:Job
 *     properties:
 *       name: example
 *       roleArn: ${exampleAwsIamRole.arn}
 *       glueVersion: '4.0'
 *       workerType: Z.2X
 *       command:
 *         name: glueray
 *         pythonVersion: '3.9'
 *         runtime: Ray2.4
 *         scriptLocation: s3://${exampleAwsS3Bucket.bucket}/example.py
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Scala Job
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.glue.Job("example", {
 *     name: "example",
 *     roleArn: exampleAwsIamRole.arn,
 *     command: {
 *         scriptLocation: `s3://${exampleAwsS3Bucket.bucket}/example.scala`,
 *     },
 *     defaultArguments: {
 *         "--job-language": "scala",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.glue.Job("example",
 *     name="example",
 *     role_arn=example_aws_iam_role["arn"],
 *     command={
 *         "script_location": f"s3://{example_aws_s3_bucket['bucket']}/example.scala",
 *     },
 *     default_arguments={
 *         "--job-language": "scala",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Glue.Job("example", new()
 *     {
 *         Name = "example",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Command = new Aws.Glue.Inputs.JobCommandArgs
 *         {
 *             ScriptLocation = $"s3://{exampleAwsS3Bucket.Bucket}/example.scala",
 *         },
 *         DefaultArguments =
 *         {
 *             { "--job-language", "scala" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/glue"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := glue.NewJob(ctx, "example", &glue.JobArgs{
 * 			Name:    pulumi.String("example"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Command: &glue.JobCommandArgs{
 * 				ScriptLocation: pulumi.Sprintf("s3://%v/example.scala", exampleAwsS3Bucket.Bucket),
 * 			},
 * 			DefaultArguments: pulumi.StringMap{
 * 				"--job-language": pulumi.String("scala"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.glue.Job;
 * import com.pulumi.aws.glue.JobArgs;
 * import com.pulumi.aws.glue.inputs.JobCommandArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Job("example", JobArgs.builder()
 *             .name("example")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .command(JobCommandArgs.builder()
 *                 .scriptLocation(String.format("s3://%s/example.scala", exampleAwsS3Bucket.bucket()))
 *                 .build())
 *             .defaultArguments(Map.of("--job-language", "scala"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:glue:Job
 *     properties:
 *       name: example
 *       roleArn: ${exampleAwsIamRole.arn}
 *       command:
 *         scriptLocation: s3://${exampleAwsS3Bucket.bucket}/example.scala
 *       defaultArguments:
 *         --job-language: scala
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Streaming Job
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.glue.Job("example", {
 *     name: "example streaming job",
 *     roleArn: exampleAwsIamRole.arn,
 *     command: {
 *         name: "gluestreaming",
 *         scriptLocation: `s3://${exampleAwsS3Bucket.bucket}/example.script`,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.glue.Job("example",
 *     name="example streaming job",
 *     role_arn=example_aws_iam_role["arn"],
 *     command={
 *         "name": "gluestreaming",
 *         "script_location": f"s3://{example_aws_s3_bucket['bucket']}/example.script",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Glue.Job("example", new()
 *     {
 *         Name = "example streaming job",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Command = new Aws.Glue.Inputs.JobCommandArgs
 *         {
 *             Name = "gluestreaming",
 *             ScriptLocation = $"s3://{exampleAwsS3Bucket.Bucket}/example.script",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/glue"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := glue.NewJob(ctx, "example", &glue.JobArgs{
 * 			Name:    pulumi.String("example streaming job"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Command: &glue.JobCommandArgs{
 * 				Name:           pulumi.String("gluestreaming"),
 * 				ScriptLocation: pulumi.Sprintf("s3://%v/example.script", exampleAwsS3Bucket.Bucket),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.glue.Job;
 * import com.pulumi.aws.glue.JobArgs;
 * import com.pulumi.aws.glue.inputs.JobCommandArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Job("example", JobArgs.builder()
 *             .name("example streaming job")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .command(JobCommandArgs.builder()
 *                 .name("gluestreaming")
 *                 .scriptLocation(String.format("s3://%s/example.script", exampleAwsS3Bucket.bucket()))
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:glue:Job
 *     properties:
 *       name: example streaming job
 *       roleArn: ${exampleAwsIamRole.arn}
 *       command:
 *         name: gluestreaming
 *         scriptLocation: s3://${exampleAwsS3Bucket.bucket}/example.script
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Enabling CloudWatch Logs and Metrics
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloudwatch.LogGroup("example", {
 *     name: "example",
 *     retentionInDays: 14,
 * });
 * const exampleJob = new aws.glue.Job("example", {defaultArguments: {
 *     "--continuous-log-logGroup": example.name,
 *     "--enable-continuous-cloudwatch-log": "true",
 *     "--enable-continuous-log-filter": "true",
 *     "--enable-metrics": "",
 * }});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloudwatch.LogGroup("example",
 *     name="example",
 *     retention_in_days=14)
 * example_job = aws.glue.Job("example", default_arguments={
 *     "--continuous-log-logGroup": example.name,
 *     "--enable-continuous-cloudwatch-log": "true",
 *     "--enable-continuous-log-filter": "true",
 *     "--enable-metrics": "",
 * })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CloudWatch.LogGroup("example", new()
 *     {
 *         Name = "example",
 *         RetentionInDays = 14,
 *     });
 *     var exampleJob = new Aws.Glue.Job("example", new()
 *     {
 *         DefaultArguments =
 *         {
 *             { "--continuous-log-logGroup", example.Name },
 *             { "--enable-continuous-cloudwatch-log", "true" },
 *             { "--enable-continuous-log-filter", "true" },
 *             { "--enable-metrics", "" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/glue"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := cloudwatch.NewLogGroup(ctx, "example", &cloudwatch.LogGroupArgs{
 * 			Name:            pulumi.String("example"),
 * 			RetentionInDays: pulumi.Int(14),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = glue.NewJob(ctx, "example", &glue.JobArgs{
 * 			DefaultArguments: pulumi.StringMap{
 * 				"--continuous-log-logGroup":          example.Name,
 * 				"--enable-continuous-cloudwatch-log": pulumi.String("true"),
 * 				"--enable-continuous-log-filter":     pulumi.String("true"),
 * 				"--enable-metrics":                   pulumi.String(""),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.cloudwatch.LogGroupArgs;
 * import com.pulumi.aws.glue.Job;
 * import com.pulumi.aws.glue.JobArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new LogGroup("example", LogGroupArgs.builder()
 *             .name("example")
 *             .retentionInDays(14)
 *             .build());
 *         var exampleJob = new Job("exampleJob", JobArgs.builder()
 *             .defaultArguments(Map.ofEntries(
 *                 Map.entry("--continuous-log-logGroup", example.name()),
 *                 Map.entry("--enable-continuous-cloudwatch-log", "true"),
 *                 Map.entry("--enable-continuous-log-filter", "true"),
 *                 Map.entry("--enable-metrics", "")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloudwatch:LogGroup
 *     properties:
 *       name: example
 *       retentionInDays: 14
 *   exampleJob:
 *     type: aws:glue:Job
 *     name: example
 *     properties:
 *       defaultArguments:
 *         --continuous-log-logGroup: ${example.name}
 *         --enable-continuous-cloudwatch-log: 'true'
 *         --enable-continuous-log-filter: 'true'
 *         --enable-metrics:
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Glue Jobs using `name`. For example:
 * ```sh
 * $ pulumi import aws:glue/job:Job MyJob MyJob
 * ```
 * @property command The command of the job. Defined below.
 * @property connections The list of connections used for this job.
 * @property defaultArguments The map of default arguments for this job. You can specify arguments here that your own job-execution script consumes, as well as arguments that AWS Glue itself consumes. For information about how to specify and consume your own Job arguments, see the [Calling AWS Glue APIs in Python](http://docs&#46;aws&#46;amazon&#46;com/glue/latest/dg/aws-glue-programming-python-calling&#46;html) topic in the developer guide&#46; For information about the key-value pairs that AWS Glue consumes to set up your job, see the [Special Parameters Used by AWS Glue](http://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-glue-arguments.html) topic in the developer guide.
 * @property description Description of the job.
 * @property executionClass Indicates whether the job is run with a standard or flexible execution class. The standard execution class is ideal for time-sensitive workloads that require fast job startup and dedicated resources. Valid value: `FLEX`, `STANDARD`.
 * @property executionProperty Execution property of the job. Defined below.
 * @property glueVersion The version of glue to use, for example "1.0". Ray jobs should set this to 4.0 or greater. For information about available versions, see the [AWS Glue Release Notes](https://docs.aws.amazon.com/glue/latest/dg/release-notes.html).
 * @property maintenanceWindow Specifies the day of the week and hour for the maintenance window for streaming jobs.
 * @property maxCapacity The maximum number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. `Required` when `pythonshell` is set, accept either `0.0625` or `1.0`. Use `number_of_workers` and `worker_type` arguments instead with `glue_version` `2.0` and above.
 * @property maxRetries The maximum number of times to retry this job if it fails.
 * @property name The name you assign to this job. It must be unique in your account.
 * @property nonOverridableArguments Non-overridable arguments for this job, specified as name-value pairs.
 * @property notificationProperty Notification property of the job. Defined below.
 * @property numberOfWorkers The number of workers of a defined workerType that are allocated when a job runs.
 * @property roleArn The ARN of the IAM role associated with this job.
 * @property securityConfiguration The name of the Security Configuration to be associated with the job.
 * @property tags Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property timeout The job timeout in minutes. The default is 2880 minutes (48 hours) for `glueetl` and `pythonshell` jobs, and null (unlimited) for `gluestreaming` jobs.
 * @property workerType The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, G.2X, or G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.
 * * For the Standard worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
 * * For the G.1X worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker. Recommended for memory-intensive jobs.
 * * For the G.2X worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker. Recommended for memory-intensive jobs.
 * * For the G.4X worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk (approximately 235GB free), and provides 1 executor per worker. Recommended for memory-intensive jobs. Only available for Glue version 3.0. Available AWS Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
 * * For the G.8X worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB disk (approximately 487GB free), and provides 1 executor per worker. Recommended for memory-intensive jobs. Only available for Glue version 3.0. Available AWS Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
 * * For the G.025X worker type, each worker maps to 0.25 DPU (2 vCPU, 4GB of memory, 64 GB disk), and provides 1 executor per worker. Recommended for low volume streaming jobs. Only available for Glue version 3.0.
 * * For the Z.2X worker type, each worker maps to 2 M-DPU (8vCPU, 64 GB of m emory, 128 GB disk), and provides up to 8 Ray workers based on the autoscaler.
 */
public data class JobArgs(
    public val command: Output<JobCommandArgs>? = null,
    public val connections: Output<List<String>>? = null,
    public val defaultArguments: Output<Map<String, String>>? = null,
    public val description: Output<String>? = null,
    public val executionClass: Output<String>? = null,
    public val executionProperty: Output<JobExecutionPropertyArgs>? = null,
    public val glueVersion: Output<String>? = null,
    public val maintenanceWindow: Output<String>? = null,
    public val maxCapacity: Output<Double>? = null,
    public val maxRetries: Output<Int>? = null,
    public val name: Output<String>? = null,
    public val nonOverridableArguments: Output<Map<String, String>>? = null,
    public val notificationProperty: Output<JobNotificationPropertyArgs>? = null,
    public val numberOfWorkers: Output<Int>? = null,
    public val roleArn: Output<String>? = null,
    public val securityConfiguration: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val timeout: Output<Int>? = null,
    public val workerType: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.glue.JobArgs> {
    override fun toJava(): com.pulumi.aws.glue.JobArgs = com.pulumi.aws.glue.JobArgs.builder()
        .command(command?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .connections(connections?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .defaultArguments(
            defaultArguments?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }),
        )
        .description(description?.applyValue({ args0 -> args0 }))
        .executionClass(executionClass?.applyValue({ args0 -> args0 }))
        .executionProperty(executionProperty?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .glueVersion(glueVersion?.applyValue({ args0 -> args0 }))
        .maintenanceWindow(maintenanceWindow?.applyValue({ args0 -> args0 }))
        .maxCapacity(maxCapacity?.applyValue({ args0 -> args0 }))
        .maxRetries(maxRetries?.applyValue({ args0 -> args0 }))
        .name(name?.applyValue({ args0 -> args0 }))
        .nonOverridableArguments(
            nonOverridableArguments?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }),
        )
        .notificationProperty(
            notificationProperty?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .numberOfWorkers(numberOfWorkers?.applyValue({ args0 -> args0 }))
        .roleArn(roleArn?.applyValue({ args0 -> args0 }))
        .securityConfiguration(securityConfiguration?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .timeout(timeout?.applyValue({ args0 -> args0 }))
        .workerType(workerType?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [JobArgs].
 */
@PulumiTagMarker
public class JobArgsBuilder internal constructor() {
    private var command: Output<JobCommandArgs>? = null

    private var connections: Output<List<String>>? = null

    private var defaultArguments: Output<Map<String, String>>? = null

    private var description: Output<String>? = null

    private var executionClass: Output<String>? = null

    private var executionProperty: Output<JobExecutionPropertyArgs>? = null

    private var glueVersion: Output<String>? = null

    private var maintenanceWindow: Output<String>? = null

    private var maxCapacity: Output<Double>? = null

    private var maxRetries: Output<Int>? = null

    private var name: Output<String>? = null

    private var nonOverridableArguments: Output<Map<String, String>>? = null

    private var notificationProperty: Output<JobNotificationPropertyArgs>? = null

    private var numberOfWorkers: Output<Int>? = null

    private var roleArn: Output<String>? = null

    private var securityConfiguration: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var timeout: Output<Int>? = null

    private var workerType: Output<String>? = null

    /**
     * @param value The command of the job. Defined below.
     */
    @JvmName("vyaxykxdnaxmbxii")
    public suspend fun command(`value`: Output<JobCommandArgs>) {
        this.command = value
    }

    /**
     * @param value The list of connections used for this job.
     */
    @JvmName("cqtyopbwivurdtjo")
    public suspend fun connections(`value`: Output<List<String>>) {
        this.connections = value
    }

    @JvmName("nybjbvdrenadbqmo")
    public suspend fun connections(vararg values: Output<String>) {
        this.connections = Output.all(values.asList())
    }

    /**
     * @param values The list of connections used for this job.
     */
    @JvmName("ippurdbocoqyejqw")
    public suspend fun connections(values: List<Output<String>>) {
        this.connections = Output.all(values)
    }

    /**
     * @param value The map of default arguments for this job. You can specify arguments here that your own job-execution script consumes, as well as arguments that AWS Glue itself consumes. For information about how to specify and consume your own Job arguments, see the [Calling AWS Glue APIs in Python](http://docs&#46;aws&#46;amazon&#46;com/glue/latest/dg/aws-glue-programming-python-calling&#46;html) topic in the developer guide&#46; For information about the key-value pairs that AWS Glue consumes to set up your job, see the [Special Parameters Used by AWS Glue](http://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-glue-arguments.html) topic in the developer guide.
     */
    @JvmName("qapjmqkeqcflohmt")
    public suspend fun defaultArguments(`value`: Output<Map<String, String>>) {
        this.defaultArguments = value
    }

    /**
     * @param value Description of the job.
     */
    @JvmName("pwnbkjpwyvxqrwyi")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value Indicates whether the job is run with a standard or flexible execution class. The standard execution class is ideal for time-sensitive workloads that require fast job startup and dedicated resources. Valid value: `FLEX`, `STANDARD`.
     */
    @JvmName("qvtmbubdhunttkup")
    public suspend fun executionClass(`value`: Output<String>) {
        this.executionClass = value
    }

    /**
     * @param value Execution property of the job. Defined below.
     */
    @JvmName("fykdymtignfcrohi")
    public suspend fun executionProperty(`value`: Output<JobExecutionPropertyArgs>) {
        this.executionProperty = value
    }

    /**
     * @param value The version of glue to use, for example "1.0". Ray jobs should set this to 4.0 or greater. For information about available versions, see the [AWS Glue Release Notes](https://docs.aws.amazon.com/glue/latest/dg/release-notes.html).
     */
    @JvmName("olvrygmhnwygkkoe")
    public suspend fun glueVersion(`value`: Output<String>) {
        this.glueVersion = value
    }

    /**
     * @param value Specifies the day of the week and hour for the maintenance window for streaming jobs.
     */
    @JvmName("ripinyhaknyurrms")
    public suspend fun maintenanceWindow(`value`: Output<String>) {
        this.maintenanceWindow = value
    }

    /**
     * @param value The maximum number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. `Required` when `pythonshell` is set, accept either `0.0625` or `1.0`. Use `number_of_workers` and `worker_type` arguments instead with `glue_version` `2.0` and above.
     */
    @JvmName("jgsvfpbkhcwwjtal")
    public suspend fun maxCapacity(`value`: Output<Double>) {
        this.maxCapacity = value
    }

    /**
     * @param value The maximum number of times to retry this job if it fails.
     */
    @JvmName("qgcyomolprubnsul")
    public suspend fun maxRetries(`value`: Output<Int>) {
        this.maxRetries = value
    }

    /**
     * @param value The name you assign to this job. It must be unique in your account.
     */
    @JvmName("iusaxgkjfekcmpbh")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Non-overridable arguments for this job, specified as name-value pairs.
     */
    @JvmName("oifkadidcqqxlcfg")
    public suspend fun nonOverridableArguments(`value`: Output<Map<String, String>>) {
        this.nonOverridableArguments = value
    }

    /**
     * @param value Notification property of the job. Defined below.
     */
    @JvmName("kgkehofhtfggcjoy")
    public suspend fun notificationProperty(`value`: Output<JobNotificationPropertyArgs>) {
        this.notificationProperty = value
    }

    /**
     * @param value The number of workers of a defined workerType that are allocated when a job runs.
     */
    @JvmName("nyvngapuhnrjmiod")
    public suspend fun numberOfWorkers(`value`: Output<Int>) {
        this.numberOfWorkers = value
    }

    /**
     * @param value The ARN of the IAM role associated with this job.
     */
    @JvmName("tylquhyjbrhulfve")
    public suspend fun roleArn(`value`: Output<String>) {
        this.roleArn = value
    }

    /**
     * @param value The name of the Security Configuration to be associated with the job.
     */
    @JvmName("eqhjcxmkptpgcamm")
    public suspend fun securityConfiguration(`value`: Output<String>) {
        this.securityConfiguration = value
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cdpgdvrrjfntcnoq")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The job timeout in minutes. The default is 2880 minutes (48 hours) for `glueetl` and `pythonshell` jobs, and null (unlimited) for `gluestreaming` jobs.
     */
    @JvmName("qaesbogchcqllgdd")
    public suspend fun timeout(`value`: Output<Int>) {
        this.timeout = value
    }

    /**
     * @param value The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, G.2X, or G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.
     * * For the Standard worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
     * * For the G.1X worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker. Recommended for memory-intensive jobs.
     * * For the G.2X worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker. Recommended for memory-intensive jobs.
     * * For the G.4X worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk (approximately 235GB free), and provides 1 executor per worker. Recommended for memory-intensive jobs. Only available for Glue version 3.0. Available AWS Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * * For the G.8X worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB disk (approximately 487GB free), and provides 1 executor per worker. Recommended for memory-intensive jobs. Only available for Glue version 3.0. Available AWS Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * * For the G.025X worker type, each worker maps to 0.25 DPU (2 vCPU, 4GB of memory, 64 GB disk), and provides 1 executor per worker. Recommended for low volume streaming jobs. Only available for Glue version 3.0.
     * * For the Z.2X worker type, each worker maps to 2 M-DPU (8vCPU, 64 GB of m emory, 128 GB disk), and provides up to 8 Ray workers based on the autoscaler.
     */
    @JvmName("vobtlwojxqutsbgt")
    public suspend fun workerType(`value`: Output<String>) {
        this.workerType = value
    }

    /**
     * @param value The command of the job. Defined below.
     */
    @JvmName("atylrtkxryfxhila")
    public suspend fun command(`value`: JobCommandArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.command = mapped
    }

    /**
     * @param argument The command of the job. Defined below.
     */
    @JvmName("nkrsdgnrmnkutfmj")
    public suspend fun command(argument: suspend JobCommandArgsBuilder.() -> Unit) {
        val toBeMapped = JobCommandArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.command = mapped
    }

    /**
     * @param value The list of connections used for this job.
     */
    @JvmName("oqdjlojcrkpvsjhu")
    public suspend fun connections(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.connections = mapped
    }

    /**
     * @param values The list of connections used for this job.
     */
    @JvmName("yhylhmyqpkshxued")
    public suspend fun connections(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.connections = mapped
    }

    /**
     * @param value The map of default arguments for this job. You can specify arguments here that your own job-execution script consumes, as well as arguments that AWS Glue itself consumes. For information about how to specify and consume your own Job arguments, see the [Calling AWS Glue APIs in Python](http://docs&#46;aws&#46;amazon&#46;com/glue/latest/dg/aws-glue-programming-python-calling&#46;html) topic in the developer guide&#46; For information about the key-value pairs that AWS Glue consumes to set up your job, see the [Special Parameters Used by AWS Glue](http://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-glue-arguments.html) topic in the developer guide.
     */
    @JvmName("vvmmjwnpuiivjpjn")
    public suspend fun defaultArguments(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultArguments = mapped
    }

    /**
     * @param values The map of default arguments for this job. You can specify arguments here that your own job-execution script consumes, as well as arguments that AWS Glue itself consumes. For information about how to specify and consume your own Job arguments, see the [Calling AWS Glue APIs in Python](http://docs&#46;aws&#46;amazon&#46;com/glue/latest/dg/aws-glue-programming-python-calling&#46;html) topic in the developer guide&#46; For information about the key-value pairs that AWS Glue consumes to set up your job, see the [Special Parameters Used by AWS Glue](http://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-glue-arguments.html) topic in the developer guide.
     */
    @JvmName("fgsfkmyrwmkylttv")
    public fun defaultArguments(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.defaultArguments = mapped
    }

    /**
     * @param value Description of the job.
     */
    @JvmName("iphekkvewbexhbaa")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value Indicates whether the job is run with a standard or flexible execution class. The standard execution class is ideal for time-sensitive workloads that require fast job startup and dedicated resources. Valid value: `FLEX`, `STANDARD`.
     */
    @JvmName("huhgbrxkmmysetea")
    public suspend fun executionClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.executionClass = mapped
    }

    /**
     * @param value Execution property of the job. Defined below.
     */
    @JvmName("sndysbdafjuqpqwj")
    public suspend fun executionProperty(`value`: JobExecutionPropertyArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.executionProperty = mapped
    }

    /**
     * @param argument Execution property of the job. Defined below.
     */
    @JvmName("lugmdnoxhiwggrnq")
    public suspend fun executionProperty(argument: suspend JobExecutionPropertyArgsBuilder.() -> Unit) {
        val toBeMapped = JobExecutionPropertyArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.executionProperty = mapped
    }

    /**
     * @param value The version of glue to use, for example "1.0". Ray jobs should set this to 4.0 or greater. For information about available versions, see the [AWS Glue Release Notes](https://docs.aws.amazon.com/glue/latest/dg/release-notes.html).
     */
    @JvmName("qfglvflmmdlijnrm")
    public suspend fun glueVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.glueVersion = mapped
    }

    /**
     * @param value Specifies the day of the week and hour for the maintenance window for streaming jobs.
     */
    @JvmName("cajcyjxuylffnucg")
    public suspend fun maintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintenanceWindow = mapped
    }

    /**
     * @param value The maximum number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. `Required` when `pythonshell` is set, accept either `0.0625` or `1.0`. Use `number_of_workers` and `worker_type` arguments instead with `glue_version` `2.0` and above.
     */
    @JvmName("hkoauvaenyhngppw")
    public suspend fun maxCapacity(`value`: Double?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxCapacity = mapped
    }

    /**
     * @param value The maximum number of times to retry this job if it fails.
     */
    @JvmName("ycpjsfaawjybgdqx")
    public suspend fun maxRetries(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxRetries = mapped
    }

    /**
     * @param value The name you assign to this job. It must be unique in your account.
     */
    @JvmName("cgwhasaukkputloh")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Non-overridable arguments for this job, specified as name-value pairs.
     */
    @JvmName("xboomvcjkgbhlkjj")
    public suspend fun nonOverridableArguments(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.nonOverridableArguments = mapped
    }

    /**
     * @param values Non-overridable arguments for this job, specified as name-value pairs.
     */
    @JvmName("xcfgfshjlnpdfumo")
    public fun nonOverridableArguments(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.nonOverridableArguments = mapped
    }

    /**
     * @param value Notification property of the job. Defined below.
     */
    @JvmName("ooybleislwmnroup")
    public suspend fun notificationProperty(`value`: JobNotificationPropertyArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.notificationProperty = mapped
    }

    /**
     * @param argument Notification property of the job. Defined below.
     */
    @JvmName("bcsiqhsfepimupeo")
    public suspend fun notificationProperty(argument: suspend JobNotificationPropertyArgsBuilder.() -> Unit) {
        val toBeMapped = JobNotificationPropertyArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.notificationProperty = mapped
    }

    /**
     * @param value The number of workers of a defined workerType that are allocated when a job runs.
     */
    @JvmName("ygmlkitcjvyfrceg")
    public suspend fun numberOfWorkers(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.numberOfWorkers = mapped
    }

    /**
     * @param value The ARN of the IAM role associated with this job.
     */
    @JvmName("efcdiqelweqcasyo")
    public suspend fun roleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.roleArn = mapped
    }

    /**
     * @param value The name of the Security Configuration to be associated with the job.
     */
    @JvmName("xhsukfxlvxfuousn")
    public suspend fun securityConfiguration(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityConfiguration = mapped
    }

    /**
     * @param value Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("oapybucvnaltljnb")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("bfxbofvqpxaaamgs")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value The job timeout in minutes. The default is 2880 minutes (48 hours) for `glueetl` and `pythonshell` jobs, and null (unlimited) for `gluestreaming` jobs.
     */
    @JvmName("yuawfbnhiqcupvqf")
    public suspend fun timeout(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.timeout = mapped
    }

    /**
     * @param value The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, G.2X, or G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.
     * * For the Standard worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
     * * For the G.1X worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker. Recommended for memory-intensive jobs.
     * * For the G.2X worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker. Recommended for memory-intensive jobs.
     * * For the G.4X worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk (approximately 235GB free), and provides 1 executor per worker. Recommended for memory-intensive jobs. Only available for Glue version 3.0. Available AWS Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * * For the G.8X worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB disk (approximately 487GB free), and provides 1 executor per worker. Recommended for memory-intensive jobs. Only available for Glue version 3.0. Available AWS Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * * For the G.025X worker type, each worker maps to 0.25 DPU (2 vCPU, 4GB of memory, 64 GB disk), and provides 1 executor per worker. Recommended for low volume streaming jobs. Only available for Glue version 3.0.
     * * For the Z.2X worker type, each worker maps to 2 M-DPU (8vCPU, 64 GB of m emory, 128 GB disk), and provides up to 8 Ray workers based on the autoscaler.
     */
    @JvmName("fvmcrgdqugmunmwx")
    public suspend fun workerType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.workerType = mapped
    }

    internal fun build(): JobArgs = JobArgs(
        command = command,
        connections = connections,
        defaultArguments = defaultArguments,
        description = description,
        executionClass = executionClass,
        executionProperty = executionProperty,
        glueVersion = glueVersion,
        maintenanceWindow = maintenanceWindow,
        maxCapacity = maxCapacity,
        maxRetries = maxRetries,
        name = name,
        nonOverridableArguments = nonOverridableArguments,
        notificationProperty = notificationProperty,
        numberOfWorkers = numberOfWorkers,
        roleArn = roleArn,
        securityConfiguration = securityConfiguration,
        tags = tags,
        timeout = timeout,
        workerType = workerType,
    )
}
