@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.guardduty.kotlin

import com.pulumi.aws.guardduty.ThreatIntelSetArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a resource to manage a GuardDuty ThreatIntelSet.
 * > **Note:** Currently in GuardDuty, users from member accounts cannot upload and further manage ThreatIntelSets. ThreatIntelSets that are uploaded by the primary account are imposed on GuardDuty functionality in its member accounts. See the [GuardDuty API Documentation](https://docs.aws.amazon.com/guardduty/latest/ug/create-threat-intel-set.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const primary = new aws.guardduty.Detector("primary", {enable: true});
 * const bucket = new aws.s3.BucketV2("bucket", {});
 * const bucketAcl = new aws.s3.BucketAclV2("bucket_acl", {
 *     bucket: bucket.id,
 *     acl: "private",
 * });
 * const myThreatIntelSet = new aws.s3.BucketObjectv2("MyThreatIntelSet", {
 *     acl: "public-read",
 *     content: "10.0.0.0/8\n",
 *     bucket: bucket.id,
 *     key: "MyThreatIntelSet",
 * });
 * const myThreatIntelSetThreatIntelSet = new aws.guardduty.ThreatIntelSet("MyThreatIntelSet", {
 *     activate: true,
 *     detectorId: primary.id,
 *     format: "TXT",
 *     location: pulumi.interpolate`https://s3.amazonaws.com/${myThreatIntelSet.bucket}/${myThreatIntelSet.key}`,
 *     name: "MyThreatIntelSet",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * primary = aws.guardduty.Detector("primary", enable=True)
 * bucket = aws.s3.BucketV2("bucket")
 * bucket_acl = aws.s3.BucketAclV2("bucket_acl",
 *     bucket=bucket.id,
 *     acl="private")
 * my_threat_intel_set = aws.s3.BucketObjectv2("MyThreatIntelSet",
 *     acl="public-read",
 *     content="10.0.0.0/8\n",
 *     bucket=bucket.id,
 *     key="MyThreatIntelSet")
 * my_threat_intel_set_threat_intel_set = aws.guardduty.ThreatIntelSet("MyThreatIntelSet",
 *     activate=True,
 *     detector_id=primary.id,
 *     format="TXT",
 *     location=pulumi.Output.all(my_threat_intel_set.bucket, my_threat_intel_set.key).apply(lambda bucket, key: f"https://s3.amazonaws.com/{bucket}/{key}"),
 *     name="MyThreatIntelSet")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var primary = new Aws.GuardDuty.Detector("primary", new()
 *     {
 *         Enable = true,
 *     });
 *     var bucket = new Aws.S3.BucketV2("bucket");
 *     var bucketAcl = new Aws.S3.BucketAclV2("bucket_acl", new()
 *     {
 *         Bucket = bucket.Id,
 *         Acl = "private",
 *     });
 *     var myThreatIntelSet = new Aws.S3.BucketObjectv2("MyThreatIntelSet", new()
 *     {
 *         Acl = "public-read",
 *         Content = @"10.0.0.0/8
 * ",
 *         Bucket = bucket.Id,
 *         Key = "MyThreatIntelSet",
 *     });
 *     var myThreatIntelSetThreatIntelSet = new Aws.GuardDuty.ThreatIntelSet("MyThreatIntelSet", new()
 *     {
 *         Activate = true,
 *         DetectorId = primary.Id,
 *         Format = "TXT",
 *         Location = Output.Tuple(myThreatIntelSet.Bucket, myThreatIntelSet.Key).Apply(values =>
 *         {
 *             var bucket = values.Item1;
 *             var key = values.Item2;
 *             return $"https://s3.amazonaws.com/{bucket}/{key}";
 *         }),
 *         Name = "MyThreatIntelSet",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/guardduty"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		primary, err := guardduty.NewDetector(ctx, "primary", &guardduty.DetectorArgs{
 * 			Enable: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		bucket, err := s3.NewBucketV2(ctx, "bucket", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketAclV2(ctx, "bucket_acl", &s3.BucketAclV2Args{
 * 			Bucket: bucket.ID(),
 * 			Acl:    pulumi.String("private"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		myThreatIntelSet, err := s3.NewBucketObjectv2(ctx, "MyThreatIntelSet", &s3.BucketObjectv2Args{
 * 			Acl:     pulumi.String("public-read"),
 * 			Content: pulumi.String("10.0.0.0/8\n"),
 * 			Bucket:  bucket.ID(),
 * 			Key:     pulumi.String("MyThreatIntelSet"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = guardduty.NewThreatIntelSet(ctx, "MyThreatIntelSet", &guardduty.ThreatIntelSetArgs{
 * 			Activate:   pulumi.Bool(true),
 * 			DetectorId: primary.ID(),
 * 			Format:     pulumi.String("TXT"),
 * 			Location: pulumi.All(myThreatIntelSet.Bucket, myThreatIntelSet.Key).ApplyT(func(_args []interface{}) (string, error) {
 * 				bucket := _args[0].(string)
 * 				key := _args[1].(string)
 * 				return fmt.Sprintf("https://s3.amazonaws.com/%v/%v", bucket, key), nil
 * 			}).(pulumi.StringOutput),
 * 			Name: pulumi.String("MyThreatIntelSet"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.guardduty.Detector;
 * import com.pulumi.aws.guardduty.DetectorArgs;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketAclV2;
 * import com.pulumi.aws.s3.BucketAclV2Args;
 * import com.pulumi.aws.s3.BucketObjectv2;
 * import com.pulumi.aws.s3.BucketObjectv2Args;
 * import com.pulumi.aws.guardduty.ThreatIntelSet;
 * import com.pulumi.aws.guardduty.ThreatIntelSetArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var primary = new Detector("primary", DetectorArgs.builder()
 *             .enable(true)
 *             .build());
 *         var bucket = new BucketV2("bucket");
 *         var bucketAcl = new BucketAclV2("bucketAcl", BucketAclV2Args.builder()
 *             .bucket(bucket.id())
 *             .acl("private")
 *             .build());
 *         var myThreatIntelSet = new BucketObjectv2("myThreatIntelSet", BucketObjectv2Args.builder()
 *             .acl("public-read")
 *             .content("""
 * 10.0.0.0/8
 *             """)
 *             .bucket(bucket.id())
 *             .key("MyThreatIntelSet")
 *             .build());
 *         var myThreatIntelSetThreatIntelSet = new ThreatIntelSet("myThreatIntelSetThreatIntelSet", ThreatIntelSetArgs.builder()
 *             .activate(true)
 *             .detectorId(primary.id())
 *             .format("TXT")
 *             .location(Output.tuple(myThreatIntelSet.bucket(), myThreatIntelSet.key()).applyValue(values -> {
 *                 var bucket = values.t1;
 *                 var key = values.t2;
 *                 return String.format("https://s3.amazonaws.com/%s/%s", bucket,key);
 *             }))
 *             .name("MyThreatIntelSet")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   primary:
 *     type: aws:guardduty:Detector
 *     properties:
 *       enable: true
 *   bucket:
 *     type: aws:s3:BucketV2
 *   bucketAcl:
 *     type: aws:s3:BucketAclV2
 *     name: bucket_acl
 *     properties:
 *       bucket: ${bucket.id}
 *       acl: private
 *   myThreatIntelSet:
 *     type: aws:s3:BucketObjectv2
 *     name: MyThreatIntelSet
 *     properties:
 *       acl: public-read
 *       content: |
 *         10.0.0.0/8
 *       bucket: ${bucket.id}
 *       key: MyThreatIntelSet
 *   myThreatIntelSetThreatIntelSet:
 *     type: aws:guardduty:ThreatIntelSet
 *     name: MyThreatIntelSet
 *     properties:
 *       activate: true
 *       detectorId: ${primary.id}
 *       format: TXT
 *       location: https://s3.amazonaws.com/${myThreatIntelSet.bucket}/${myThreatIntelSet.key}
 *       name: MyThreatIntelSet
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import GuardDuty ThreatIntelSet using the primary GuardDuty detector ID and ThreatIntelSetID. For example:
 * ```sh
 * $ pulumi import aws:guardduty/threatIntelSet:ThreatIntelSet MyThreatIntelSet 00b00fd5aecc0ab60a708659477e9617:123456789012
 * ```
 * @property activate Specifies whether GuardDuty is to start using the uploaded ThreatIntelSet.
 * @property detectorId The detector ID of the GuardDuty.
 * @property format The format of the file that contains the ThreatIntelSet. Valid values: `TXT` | `STIX` | `OTX_CSV` | `ALIEN_VAULT` | `PROOF_POINT` | `FIRE_EYE`
 * @property location The URI of the file that contains the ThreatIntelSet.
 * @property name The friendly name to identify the ThreatIntelSet.
 * @property tags Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class ThreatIntelSetArgs(
    public val activate: Output<Boolean>? = null,
    public val detectorId: Output<String>? = null,
    public val format: Output<String>? = null,
    public val location: Output<String>? = null,
    public val name: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.guardduty.ThreatIntelSetArgs> {
    override fun toJava(): com.pulumi.aws.guardduty.ThreatIntelSetArgs =
        com.pulumi.aws.guardduty.ThreatIntelSetArgs.builder()
            .activate(activate?.applyValue({ args0 -> args0 }))
            .detectorId(detectorId?.applyValue({ args0 -> args0 }))
            .format(format?.applyValue({ args0 -> args0 }))
            .location(location?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [ThreatIntelSetArgs].
 */
@PulumiTagMarker
public class ThreatIntelSetArgsBuilder internal constructor() {
    private var activate: Output<Boolean>? = null

    private var detectorId: Output<String>? = null

    private var format: Output<String>? = null

    private var location: Output<String>? = null

    private var name: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value Specifies whether GuardDuty is to start using the uploaded ThreatIntelSet.
     */
    @JvmName("yseufcbaefgolajx")
    public suspend fun activate(`value`: Output<Boolean>) {
        this.activate = value
    }

    /**
     * @param value The detector ID of the GuardDuty.
     */
    @JvmName("jyoslsuiimkrnebm")
    public suspend fun detectorId(`value`: Output<String>) {
        this.detectorId = value
    }

    /**
     * @param value The format of the file that contains the ThreatIntelSet. Valid values: `TXT` | `STIX` | `OTX_CSV` | `ALIEN_VAULT` | `PROOF_POINT` | `FIRE_EYE`
     */
    @JvmName("unlpxeelijwemnod")
    public suspend fun format(`value`: Output<String>) {
        this.format = value
    }

    /**
     * @param value The URI of the file that contains the ThreatIntelSet.
     */
    @JvmName("iqurpekhkcigpmlo")
    public suspend fun location(`value`: Output<String>) {
        this.location = value
    }

    /**
     * @param value The friendly name to identify the ThreatIntelSet.
     */
    @JvmName("wxnoaxvjgljqgsqg")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("uwftvjycrrvuapfe")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Specifies whether GuardDuty is to start using the uploaded ThreatIntelSet.
     */
    @JvmName("bkyxpdgtsapalhxd")
    public suspend fun activate(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.activate = mapped
    }

    /**
     * @param value The detector ID of the GuardDuty.
     */
    @JvmName("gelaooysvuauhdbc")
    public suspend fun detectorId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.detectorId = mapped
    }

    /**
     * @param value The format of the file that contains the ThreatIntelSet. Valid values: `TXT` | `STIX` | `OTX_CSV` | `ALIEN_VAULT` | `PROOF_POINT` | `FIRE_EYE`
     */
    @JvmName("ohbyjsenjlndkbkr")
    public suspend fun format(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.format = mapped
    }

    /**
     * @param value The URI of the file that contains the ThreatIntelSet.
     */
    @JvmName("pknjdclniccvnabu")
    public suspend fun location(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.location = mapped
    }

    /**
     * @param value The friendly name to identify the ThreatIntelSet.
     */
    @JvmName("kljlpnsfqclcxbni")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("gwfxbfgynykxfdmd")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vcininjbjyyrwjbh")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): ThreatIntelSetArgs = ThreatIntelSetArgs(
        activate = activate,
        detectorId = detectorId,
        format = format,
        location = location,
        name = name,
        tags = tags,
    )
}
