@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.iot.kotlin

import com.pulumi.aws.iot.CertificateArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Creates and manages an AWS IoT certificate.
 * ## Example Usage
 * ### With CSR
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * const cert = new aws.iot.Certificate("cert", {
 *     csr: std.file({
 *         input: "/my/csr.pem",
 *     }).then(invoke => invoke.result),
 *     active: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * cert = aws.iot.Certificate("cert",
 *     csr=std.file(input="/my/csr.pem").result,
 *     active=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * return await Deployment.RunAsync(() =>
 * {
 *     var cert = new Aws.Iot.Certificate("cert", new()
 *     {
 *         Csr = Std.File.Invoke(new()
 *         {
 *             Input = "/my/csr.pem",
 *         }).Apply(invoke => invoke.Result),
 *         Active = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iot"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		invokeFile, err := std.File(ctx, &std.FileArgs{
 * 			Input: "/my/csr.pem",
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iot.NewCertificate(ctx, "cert", &iot.CertificateArgs{
 * 			Csr:    pulumi.String(invokeFile.Result),
 * 			Active: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iot.Certificate;
 * import com.pulumi.aws.iot.CertificateArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var cert = new Certificate("cert", CertificateArgs.builder()
 *             .csr(StdFunctions.file(FileArgs.builder()
 *                 .input("/my/csr.pem")
 *                 .build()).result())
 *             .active(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   cert:
 *     type: aws:iot:Certificate
 *     properties:
 *       csr:
 *         fn::invoke:
 *           Function: std:file
 *           Arguments:
 *             input: /my/csr.pem
 *           Return: result
 *       active: true
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Without CSR
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const cert = new aws.iot.Certificate("cert", {active: true});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * cert = aws.iot.Certificate("cert", active=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var cert = new Aws.Iot.Certificate("cert", new()
 *     {
 *         Active = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iot"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := iot.NewCertificate(ctx, "cert", &iot.CertificateArgs{
 * 			Active: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iot.Certificate;
 * import com.pulumi.aws.iot.CertificateArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var cert = new Certificate("cert", CertificateArgs.builder()
 *             .active(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   cert:
 *     type: aws:iot:Certificate
 *     properties:
 *       active: true
 * ```
 * <!--End PulumiCodeChooser -->
 * ### From existing certificate without a CA
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * const cert = new aws.iot.Certificate("cert", {
 *     certificatePem: std.file({
 *         input: "/my/cert.pem",
 *     }).then(invoke => invoke.result),
 *     active: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * cert = aws.iot.Certificate("cert",
 *     certificate_pem=std.file(input="/my/cert.pem").result,
 *     active=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * return await Deployment.RunAsync(() =>
 * {
 *     var cert = new Aws.Iot.Certificate("cert", new()
 *     {
 *         CertificatePem = Std.File.Invoke(new()
 *         {
 *             Input = "/my/cert.pem",
 *         }).Apply(invoke => invoke.Result),
 *         Active = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iot"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		invokeFile, err := std.File(ctx, &std.FileArgs{
 * 			Input: "/my/cert.pem",
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = iot.NewCertificate(ctx, "cert", &iot.CertificateArgs{
 * 			CertificatePem: pulumi.String(invokeFile.Result),
 * 			Active:         pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iot.Certificate;
 * import com.pulumi.aws.iot.CertificateArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var cert = new Certificate("cert", CertificateArgs.builder()
 *             .certificatePem(StdFunctions.file(FileArgs.builder()
 *                 .input("/my/cert.pem")
 *                 .build()).result())
 *             .active(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   cert:
 *     type: aws:iot:Certificate
 *     properties:
 *       certificatePem:
 *         fn::invoke:
 *           Function: std:file
 *           Arguments:
 *             input: /my/cert.pem
 *           Return: result
 *       active: true
 * ```
 * <!--End PulumiCodeChooser -->
 * @property active Boolean flag to indicate if the certificate should be active
 * @property caPem The CA certificate for the certificate to be registered. If this is set, the CA needs to be registered with AWS IoT beforehand.
 * @property certificatePem The certificate to be registered. If `ca_pem` is unspecified, review
 * [RegisterCertificateWithoutCA](https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html).
 * If `ca_pem` is specified, review
 * [RegisterCertificate](https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html)
 * for more information on registering a certificate.
 * @property csr The certificate signing request. Review
 * [CreateCertificateFromCsr](https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html)
 * for more information on generating a certificate from a certificate signing request (CSR).
 * If none is specified both the certificate and keys will be generated, review [CreateKeysAndCertificate](https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html)
 * for more information on generating keys and a certificate.
 */
public data class CertificateArgs(
    public val active: Output<Boolean>? = null,
    public val caPem: Output<String>? = null,
    public val certificatePem: Output<String>? = null,
    public val csr: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.iot.CertificateArgs> {
    override fun toJava(): com.pulumi.aws.iot.CertificateArgs =
        com.pulumi.aws.iot.CertificateArgs.builder()
            .active(active?.applyValue({ args0 -> args0 }))
            .caPem(caPem?.applyValue({ args0 -> args0 }))
            .certificatePem(certificatePem?.applyValue({ args0 -> args0 }))
            .csr(csr?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [CertificateArgs].
 */
@PulumiTagMarker
public class CertificateArgsBuilder internal constructor() {
    private var active: Output<Boolean>? = null

    private var caPem: Output<String>? = null

    private var certificatePem: Output<String>? = null

    private var csr: Output<String>? = null

    /**
     * @param value Boolean flag to indicate if the certificate should be active
     */
    @JvmName("pnfrlrmrkwbukdgq")
    public suspend fun active(`value`: Output<Boolean>) {
        this.active = value
    }

    /**
     * @param value The CA certificate for the certificate to be registered. If this is set, the CA needs to be registered with AWS IoT beforehand.
     */
    @JvmName("rsarjrhcadvbytaq")
    public suspend fun caPem(`value`: Output<String>) {
        this.caPem = value
    }

    /**
     * @param value The certificate to be registered. If `ca_pem` is unspecified, review
     * [RegisterCertificateWithoutCA](https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html).
     * If `ca_pem` is specified, review
     * [RegisterCertificate](https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html)
     * for more information on registering a certificate.
     */
    @JvmName("paqgncqajustgyft")
    public suspend fun certificatePem(`value`: Output<String>) {
        this.certificatePem = value
    }

    /**
     * @param value The certificate signing request. Review
     * [CreateCertificateFromCsr](https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html)
     * for more information on generating a certificate from a certificate signing request (CSR).
     * If none is specified both the certificate and keys will be generated, review [CreateKeysAndCertificate](https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html)
     * for more information on generating keys and a certificate.
     */
    @JvmName("efbqxppqsjaamcfy")
    public suspend fun csr(`value`: Output<String>) {
        this.csr = value
    }

    /**
     * @param value Boolean flag to indicate if the certificate should be active
     */
    @JvmName("erhbtrisqodprjiw")
    public suspend fun active(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.active = mapped
    }

    /**
     * @param value The CA certificate for the certificate to be registered. If this is set, the CA needs to be registered with AWS IoT beforehand.
     */
    @JvmName("hjgxdrcebxqadybl")
    public suspend fun caPem(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.caPem = mapped
    }

    /**
     * @param value The certificate to be registered. If `ca_pem` is unspecified, review
     * [RegisterCertificateWithoutCA](https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html).
     * If `ca_pem` is specified, review
     * [RegisterCertificate](https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html)
     * for more information on registering a certificate.
     */
    @JvmName("ouadjpikxsonbxkg")
    public suspend fun certificatePem(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.certificatePem = mapped
    }

    /**
     * @param value The certificate signing request. Review
     * [CreateCertificateFromCsr](https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html)
     * for more information on generating a certificate from a certificate signing request (CSR).
     * If none is specified both the certificate and keys will be generated, review [CreateKeysAndCertificate](https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html)
     * for more information on generating keys and a certificate.
     */
    @JvmName("auwqnltsxmkhimkm")
    public suspend fun csr(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.csr = mapped
    }

    internal fun build(): CertificateArgs = CertificateArgs(
        active = active,
        caPem = caPem,
        certificatePem = certificatePem,
        csr = csr,
    )
}
