@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.kendra.kotlin

import com.pulumi.aws.kendra.kotlin.outputs.DataSourceConfiguration
import com.pulumi.aws.kendra.kotlin.outputs.DataSourceCustomDocumentEnrichmentConfiguration
import com.pulumi.core.Output
import com.pulumi.kotlin.KotlinCustomResource
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.ResourceMapper
import com.pulumi.kotlin.options.CustomResourceOptions
import com.pulumi.kotlin.options.CustomResourceOptionsBuilder
import com.pulumi.resources.Resource
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import com.pulumi.aws.kendra.kotlin.outputs.DataSourceConfiguration.Companion.toKotlin as dataSourceConfigurationToKotlin
import com.pulumi.aws.kendra.kotlin.outputs.DataSourceCustomDocumentEnrichmentConfiguration.Companion.toKotlin as dataSourceCustomDocumentEnrichmentConfigurationToKotlin

/**
 * Builder for [DataSource].
 */
@PulumiTagMarker
public class DataSourceResourceBuilder internal constructor() {
    public var name: String? = null

    public var args: DataSourceArgs = DataSourceArgs()

    public var opts: CustomResourceOptions = CustomResourceOptions()

    /**
     * @param name The _unique_ name of the resulting resource.
     */
    public fun name(`value`: String) {
        this.name = value
    }

    /**
     * @param block The arguments to use to populate this resource's properties.
     */
    public suspend fun args(block: suspend DataSourceArgsBuilder.() -> Unit) {
        val builder = DataSourceArgsBuilder()
        block(builder)
        this.args = builder.build()
    }

    /**
     * @param block A bag of options that control this resource's behavior.
     */
    public suspend fun opts(block: suspend CustomResourceOptionsBuilder.() -> Unit) {
        this.opts = com.pulumi.kotlin.options.CustomResourceOptions.opts(block)
    }

    internal fun build(): DataSource {
        val builtJavaResource = com.pulumi.aws.kendra.DataSource(
            this.name,
            this.args.toJava(),
            this.opts.toJava(),
        )
        return DataSource(builtJavaResource)
    }
}

/**
 * Resource for managing an AWS Kendra Data Source.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     description: "example",
 *     languageCode: "en",
 *     type: "CUSTOM",
 *     tags: {
 *         hello: "world",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     description="example",
 *     language_code="en",
 *     type="CUSTOM",
 *     tags={
 *         "hello": "world",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Description = "example",
 *         LanguageCode = "en",
 *         Type = "CUSTOM",
 *         Tags =
 *         {
 *             { "hello", "world" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId:      pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:         pulumi.String("example"),
 * 			Description:  pulumi.String("example"),
 * 			LanguageCode: pulumi.String("en"),
 * 			Type:         pulumi.String("CUSTOM"),
 * 			Tags: pulumi.StringMap{
 * 				"hello": pulumi.String("world"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .description("example")
 *             .languageCode("en")
 *             .type("CUSTOM")
 *             .tags(Map.of("hello", "world"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       description: example
 *       languageCode: en
 *       type: CUSTOM
 *       tags:
 *         hello: world
 * ```
 * <!--End PulumiCodeChooser -->
 * ### S3 Connector
 * ### With Schedule
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "S3",
 *     roleArn: exampleAwsIamRole.arn,
 *     schedule: "cron(9 10 1 * ? *)",
 *     configuration: {
 *         s3Configuration: {
 *             bucketName: exampleAwsS3Bucket.id,
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="S3",
 *     role_arn=example_aws_iam_role["arn"],
 *     schedule="cron(9 10 1 * ? *)",
 *     configuration={
 *         "s3_configuration": {
 *             "bucket_name": example_aws_s3_bucket["id"],
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "S3",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Schedule = "cron(9 10 1 * ? *)",
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             S3Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationS3ConfigurationArgs
 *             {
 *                 BucketName = exampleAwsS3Bucket.Id,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId:  pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:     pulumi.String("example"),
 * 			Type:     pulumi.String("S3"),
 * 			RoleArn:  pulumi.Any(exampleAwsIamRole.Arn),
 * 			Schedule: pulumi.String("cron(9 10 1 * ? *)"),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				S3Configuration: &kendra.DataSourceConfigurationS3ConfigurationArgs{
 * 					BucketName: pulumi.Any(exampleAwsS3Bucket.Id),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationS3ConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("S3")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .schedule("cron(9 10 1 * ? *)")
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .s3Configuration(DataSourceConfigurationS3ConfigurationArgs.builder()
 *                     .bucketName(exampleAwsS3Bucket.id())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: S3
 *       roleArn: ${exampleAwsIamRole.arn}
 *       schedule: cron(9 10 1 * ? *)
 *       configuration:
 *         s3Configuration:
 *           bucketName: ${exampleAwsS3Bucket.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Access Control List
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "S3",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         s3Configuration: {
 *             bucketName: exampleAwsS3Bucket.id,
 *             accessControlListConfiguration: {
 *                 keyPath: `s3://${exampleAwsS3Bucket.id}/path-1`,
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="S3",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "s3_configuration": {
 *             "bucket_name": example_aws_s3_bucket["id"],
 *             "access_control_list_configuration": {
 *                 "key_path": f"s3://{example_aws_s3_bucket['id']}/path-1",
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "S3",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             S3Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationS3ConfigurationArgs
 *             {
 *                 BucketName = exampleAwsS3Bucket.Id,
 *                 AccessControlListConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationS3ConfigurationAccessControlListConfigurationArgs
 *                 {
 *                     KeyPath = $"s3://{exampleAwsS3Bucket.Id}/path-1",
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("S3"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				S3Configuration: &kendra.DataSourceConfigurationS3ConfigurationArgs{
 * 					BucketName: pulumi.Any(exampleAwsS3Bucket.Id),
 * 					AccessControlListConfiguration: &kendra.DataSourceConfigurationS3ConfigurationAccessControlListConfigurationArgs{
 * 						KeyPath: pulumi.Sprintf("s3://%v/path-1", exampleAwsS3Bucket.Id),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationS3ConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationS3ConfigurationAccessControlListConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("S3")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .s3Configuration(DataSourceConfigurationS3ConfigurationArgs.builder()
 *                     .bucketName(exampleAwsS3Bucket.id())
 *                     .accessControlListConfiguration(DataSourceConfigurationS3ConfigurationAccessControlListConfigurationArgs.builder()
 *                         .keyPath(String.format("s3://%s/path-1", exampleAwsS3Bucket.id()))
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: S3
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         s3Configuration:
 *           bucketName: ${exampleAwsS3Bucket.id}
 *           accessControlListConfiguration:
 *             keyPath: s3://${exampleAwsS3Bucket.id}/path-1
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Documents Metadata Configuration
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "S3",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         s3Configuration: {
 *             bucketName: exampleAwsS3Bucket.id,
 *             exclusionPatterns: ["example"],
 *             inclusionPatterns: ["hello"],
 *             inclusionPrefixes: ["world"],
 *             documentsMetadataConfiguration: {
 *                 s3Prefix: "example",
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="S3",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "s3_configuration": {
 *             "bucket_name": example_aws_s3_bucket["id"],
 *             "exclusion_patterns": ["example"],
 *             "inclusion_patterns": ["hello"],
 *             "inclusion_prefixes": ["world"],
 *             "documents_metadata_configuration": {
 *                 "s3_prefix": "example",
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "S3",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             S3Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationS3ConfigurationArgs
 *             {
 *                 BucketName = exampleAwsS3Bucket.Id,
 *                 ExclusionPatterns = new[]
 *                 {
 *                     "example",
 *                 },
 *                 InclusionPatterns = new[]
 *                 {
 *                     "hello",
 *                 },
 *                 InclusionPrefixes = new[]
 *                 {
 *                     "world",
 *                 },
 *                 DocumentsMetadataConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationS3ConfigurationDocumentsMetadataConfigurationArgs
 *                 {
 *                     S3Prefix = "example",
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("S3"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				S3Configuration: &kendra.DataSourceConfigurationS3ConfigurationArgs{
 * 					BucketName: pulumi.Any(exampleAwsS3Bucket.Id),
 * 					ExclusionPatterns: pulumi.StringArray{
 * 						pulumi.String("example"),
 * 					},
 * 					InclusionPatterns: pulumi.StringArray{
 * 						pulumi.String("hello"),
 * 					},
 * 					InclusionPrefixes: pulumi.StringArray{
 * 						pulumi.String("world"),
 * 					},
 * 					DocumentsMetadataConfiguration: &kendra.DataSourceConfigurationS3ConfigurationDocumentsMetadataConfigurationArgs{
 * 						S3Prefix: pulumi.String("example"),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationS3ConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationS3ConfigurationDocumentsMetadataConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("S3")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .s3Configuration(DataSourceConfigurationS3ConfigurationArgs.builder()
 *                     .bucketName(exampleAwsS3Bucket.id())
 *                     .exclusionPatterns("example")
 *                     .inclusionPatterns("hello")
 *                     .inclusionPrefixes("world")
 *                     .documentsMetadataConfiguration(DataSourceConfigurationS3ConfigurationDocumentsMetadataConfigurationArgs.builder()
 *                         .s3Prefix("example")
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: S3
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         s3Configuration:
 *           bucketName: ${exampleAwsS3Bucket.id}
 *           exclusionPatterns:
 *             - example
 *           inclusionPatterns:
 *             - hello
 *           inclusionPrefixes:
 *             - world
 *           documentsMetadataConfiguration:
 *             s3Prefix: example
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Web Crawler Connector
 * ### With Seed URLs
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           urls:
 *             seedUrlConfiguration:
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Site Maps
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             urls: {
 *                 siteMapsConfiguration: {
 *                     siteMaps: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "urls": {
 *                 "site_maps_configuration": {
 *                     "site_maps": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SiteMapsConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSiteMapsConfigurationArgs
 *                     {
 *                         SiteMaps = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SiteMapsConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSiteMapsConfigurationArgs{
 * 							SiteMaps: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSiteMapsConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .siteMapsConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSiteMapsConfigurationArgs.builder()
 *                             .siteMaps("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           urls:
 *             siteMapsConfiguration:
 *               siteMaps:
 *                 - REPLACE_WITH_YOUR_URL
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Web Crawler Mode
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     webCrawlerMode: "SUBDOMAINS",
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "web_crawler_mode": "SUBDOMAINS",
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         WebCrawlerMode = "SUBDOMAINS",
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							WebCrawlerMode: pulumi.String("SUBDOMAINS"),
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .webCrawlerMode("SUBDOMAINS")
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           urls:
 *             seedUrlConfiguration:
 *               webCrawlerMode: SUBDOMAINS
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Authentication Configuration
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             authenticationConfiguration: {
 *                 basicAuthentications: [{
 *                     credentials: exampleAwsSecretsmanagerSecret.arn,
 *                     host: "a.example.com",
 *                     port: 443,
 *                 }],
 *             },
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * }, {
 *     dependsOn: [exampleAwsSecretsmanagerSecretVersion],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "authentication_configuration": {
 *                 "basic_authentications": [{
 *                     "credentials": example_aws_secretsmanager_secret["arn"],
 *                     "host": "a.example.com",
 *                     "port": 443,
 *                 }],
 *             },
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 *     opts = pulumi.ResourceOptions(depends_on=[example_aws_secretsmanager_secret_version]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 AuthenticationConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationArgs
 *                 {
 *                     BasicAuthentications = new[]
 *                     {
 *                         new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationBasicAuthenticationArgs
 *                         {
 *                             Credentials = exampleAwsSecretsmanagerSecret.Arn,
 *                             Host = "a.example.com",
 *                             Port = 443,
 *                         },
 *                     },
 *                 },
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             exampleAwsSecretsmanagerSecretVersion,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					AuthenticationConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationArgs{
 * 						BasicAuthentications: kendra.DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationBasicAuthenticationArray{
 * 							&kendra.DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationBasicAuthenticationArgs{
 * 								Credentials: pulumi.Any(exampleAwsSecretsmanagerSecret.Arn),
 * 								Host:        pulumi.String("a.example.com"),
 * 								Port:        pulumi.Int(443),
 * 							},
 * 						},
 * 					},
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			exampleAwsSecretsmanagerSecretVersion,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .authenticationConfiguration(DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationArgs.builder()
 *                         .basicAuthentications(DataSourceConfigurationWebCrawlerConfigurationAuthenticationConfigurationBasicAuthenticationArgs.builder()
 *                             .credentials(exampleAwsSecretsmanagerSecret.arn())
 *                             .host("a.example.com")
 *                             .port("443")
 *                             .build())
 *                         .build())
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleAwsSecretsmanagerSecretVersion)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           authenticationConfiguration:
 *             basicAuthentications:
 *               - credentials: ${exampleAwsSecretsmanagerSecret.arn}
 *                 host: a.example.com
 *                 port: '443'
 *           urls:
 *             seedUrlConfiguration:
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 *     options:
 *       dependson:
 *         - ${exampleAwsSecretsmanagerSecretVersion}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Crawl Depth
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             crawlDepth: 3,
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "crawl_depth": 3,
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 CrawlDepth = 3,
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					CrawlDepth: pulumi.Int(3),
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .crawlDepth(3)
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           crawlDepth: 3
 *           urls:
 *             seedUrlConfiguration:
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Max Links Per Page
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             maxLinksPerPage: 100,
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "max_links_per_page": 100,
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 MaxLinksPerPage = 100,
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					MaxLinksPerPage: pulumi.Int(100),
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .maxLinksPerPage(100)
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           maxLinksPerPage: 100
 *           urls:
 *             seedUrlConfiguration:
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Max Urls Per Minute Crawl Rate
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             maxUrlsPerMinuteCrawlRate: 300,
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "max_urls_per_minute_crawl_rate": 300,
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 MaxUrlsPerMinuteCrawlRate = 300,
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					MaxUrlsPerMinuteCrawlRate: pulumi.Int(300),
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .maxUrlsPerMinuteCrawlRate(300)
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           maxUrlsPerMinuteCrawlRate: 300
 *           urls:
 *             seedUrlConfiguration:
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Proxy Configuration
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             proxyConfiguration: {
 *                 credentials: exampleAwsSecretsmanagerSecret.arn,
 *                 host: "a.example.com",
 *                 port: 443,
 *             },
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * }, {
 *     dependsOn: [exampleAwsSecretsmanagerSecretVersion],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "proxy_configuration": {
 *                 "credentials": example_aws_secretsmanager_secret["arn"],
 *                 "host": "a.example.com",
 *                 "port": 443,
 *             },
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 *     opts = pulumi.ResourceOptions(depends_on=[example_aws_secretsmanager_secret_version]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 ProxyConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationProxyConfigurationArgs
 *                 {
 *                     Credentials = exampleAwsSecretsmanagerSecret.Arn,
 *                     Host = "a.example.com",
 *                     Port = 443,
 *                 },
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             exampleAwsSecretsmanagerSecretVersion,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					ProxyConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationProxyConfigurationArgs{
 * 						Credentials: pulumi.Any(exampleAwsSecretsmanagerSecret.Arn),
 * 						Host:        pulumi.String("a.example.com"),
 * 						Port:        pulumi.Int(443),
 * 					},
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			exampleAwsSecretsmanagerSecretVersion,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationProxyConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .proxyConfiguration(DataSourceConfigurationWebCrawlerConfigurationProxyConfigurationArgs.builder()
 *                         .credentials(exampleAwsSecretsmanagerSecret.arn())
 *                         .host("a.example.com")
 *                         .port("443")
 *                         .build())
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleAwsSecretsmanagerSecretVersion)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           proxyConfiguration:
 *             credentials: ${exampleAwsSecretsmanagerSecret.arn}
 *             host: a.example.com
 *             port: '443'
 *           urls:
 *             seedUrlConfiguration:
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 *     options:
 *       dependson:
 *         - ${exampleAwsSecretsmanagerSecretVersion}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With URL Exclusion and Inclusion Patterns
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kendra.DataSource("example", {
 *     indexId: exampleAwsKendraIndex.id,
 *     name: "example",
 *     type: "WEBCRAWLER",
 *     roleArn: exampleAwsIamRole.arn,
 *     configuration: {
 *         webCrawlerConfiguration: {
 *             urlExclusionPatterns: ["example"],
 *             urlInclusionPatterns: ["hello"],
 *             urls: {
 *                 seedUrlConfiguration: {
 *                     seedUrls: ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kendra.DataSource("example",
 *     index_id=example_aws_kendra_index["id"],
 *     name="example",
 *     type="WEBCRAWLER",
 *     role_arn=example_aws_iam_role["arn"],
 *     configuration={
 *         "web_crawler_configuration": {
 *             "url_exclusion_patterns": ["example"],
 *             "url_inclusion_patterns": ["hello"],
 *             "urls": {
 *                 "seed_url_configuration": {
 *                     "seed_urls": ["REPLACE_WITH_YOUR_URL"],
 *                 },
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kendra.DataSource("example", new()
 *     {
 *         IndexId = exampleAwsKendraIndex.Id,
 *         Name = "example",
 *         Type = "WEBCRAWLER",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Configuration = new Aws.Kendra.Inputs.DataSourceConfigurationArgs
 *         {
 *             WebCrawlerConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationArgs
 *             {
 *                 UrlExclusionPatterns = new[]
 *                 {
 *                     "example",
 *                 },
 *                 UrlInclusionPatterns = new[]
 *                 {
 *                     "hello",
 *                 },
 *                 Urls = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs
 *                 {
 *                     SeedUrlConfiguration = new Aws.Kendra.Inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs
 *                     {
 *                         SeedUrls = new[]
 *                         {
 *                             "REPLACE_WITH_YOUR_URL",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kendra"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kendra.NewDataSource(ctx, "example", &kendra.DataSourceArgs{
 * 			IndexId: pulumi.Any(exampleAwsKendraIndex.Id),
 * 			Name:    pulumi.String("example"),
 * 			Type:    pulumi.String("WEBCRAWLER"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Configuration: &kendra.DataSourceConfigurationArgs{
 * 				WebCrawlerConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationArgs{
 * 					UrlExclusionPatterns: pulumi.StringArray{
 * 						pulumi.String("example"),
 * 					},
 * 					UrlInclusionPatterns: pulumi.StringArray{
 * 						pulumi.String("hello"),
 * 					},
 * 					Urls: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs{
 * 						SeedUrlConfiguration: &kendra.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs{
 * 							SeedUrls: pulumi.StringArray{
 * 								pulumi.String("REPLACE_WITH_YOUR_URL"),
 * 							},
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kendra.DataSource;
 * import com.pulumi.aws.kendra.DataSourceArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsArgs;
 * import com.pulumi.aws.kendra.inputs.DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DataSource("example", DataSourceArgs.builder()
 *             .indexId(exampleAwsKendraIndex.id())
 *             .name("example")
 *             .type("WEBCRAWLER")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .configuration(DataSourceConfigurationArgs.builder()
 *                 .webCrawlerConfiguration(DataSourceConfigurationWebCrawlerConfigurationArgs.builder()
 *                     .urlExclusionPatterns("example")
 *                     .urlInclusionPatterns("hello")
 *                     .urls(DataSourceConfigurationWebCrawlerConfigurationUrlsArgs.builder()
 *                         .seedUrlConfiguration(DataSourceConfigurationWebCrawlerConfigurationUrlsSeedUrlConfigurationArgs.builder()
 *                             .seedUrls("REPLACE_WITH_YOUR_URL")
 *                             .build())
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kendra:DataSource
 *     properties:
 *       indexId: ${exampleAwsKendraIndex.id}
 *       name: example
 *       type: WEBCRAWLER
 *       roleArn: ${exampleAwsIamRole.arn}
 *       configuration:
 *         webCrawlerConfiguration:
 *           urlExclusionPatterns:
 *             - example
 *           urlInclusionPatterns:
 *             - hello
 *           urls:
 *             seedUrlConfiguration:
 *               seedUrls:
 *                 - REPLACE_WITH_YOUR_URL
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Kendra Data Source using the unique identifiers of the data_source and index separated by a slash (`/`). For example:
 * ```sh
 * $ pulumi import aws:kendra/dataSource:DataSource example 1045d08d-66ef-4882-b3ed-dfb7df183e90/b34dfdf7-1f2b-4704-9581-79e00296845f
 * ```
 */
public class DataSource internal constructor(
    override val javaResource: com.pulumi.aws.kendra.DataSource,
) : KotlinCustomResource(javaResource, DataSourceMapper) {
    /**
     * ARN of the Data Source.
     */
    public val arn: Output<String>
        get() = javaResource.arn().applyValue({ args0 -> args0 })

    /**
     * A block with the configuration information to connect to your Data Source repository. You can't specify the `configuration` block when the `type` parameter is set to `CUSTOM`. Detailed below.
     */
    public val configuration: Output<DataSourceConfiguration>?
        get() = javaResource.configuration().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.let({ args0 -> dataSourceConfigurationToKotlin(args0) })
            }).orElse(null)
        })

    /**
     * The Unix timestamp of when the Data Source was created.
     */
    public val createdAt: Output<String>
        get() = javaResource.createdAt().applyValue({ args0 -> args0 })

    /**
     * A block with the configuration information for altering document metadata and content during the document ingestion process. For more information on how to create, modify and delete document metadata, or make other content alterations when you ingest documents into Amazon Kendra, see [Customizing document metadata during the ingestion process](https://docs.aws.amazon.com/kendra/latest/dg/custom-document-enrichment.html). Detailed below.
     */
    public val customDocumentEnrichmentConfiguration:
        Output<DataSourceCustomDocumentEnrichmentConfiguration>?
        get() = javaResource.customDocumentEnrichmentConfiguration().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.let({ args0 ->
                    dataSourceCustomDocumentEnrichmentConfigurationToKotlin(args0)
                })
            }).orElse(null)
        })

    /**
     * The unique identifiers of the Data Source.
     */
    public val dataSourceId: Output<String>
        get() = javaResource.dataSourceId().applyValue({ args0 -> args0 })

    /**
     * A description for the Data Source connector.
     */
    public val description: Output<String>?
        get() = javaResource.description().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * When the Status field value is `FAILED`, the ErrorMessage field contains a description of the error that caused the Data Source to fail.
     */
    public val errorMessage: Output<String>
        get() = javaResource.errorMessage().applyValue({ args0 -> args0 })

    /**
     * The identifier of the index for your Amazon Kendra data source.
     */
    public val indexId: Output<String>
        get() = javaResource.indexId().applyValue({ args0 -> args0 })

    /**
     * The code for a language. This allows you to support a language for all documents when creating the Data Source connector. English is supported by default. For more information on supported languages, including their codes, see [Adding documents in languages other than English](https://docs.aws.amazon.com/kendra/latest/dg/in-adding-languages.html).
     */
    public val languageCode: Output<String>
        get() = javaResource.languageCode().applyValue({ args0 -> args0 })

    /**
     * A name for your data source connector.
     */
    public val name: Output<String>
        get() = javaResource.name().applyValue({ args0 -> args0 })

    /**
     * The Amazon Resource Name (ARN) of a role with permission to access the data source connector. For more information, see [IAM roles for Amazon Kendra](https://docs.aws.amazon.com/kendra/latest/dg/iam-roles.html). You can't specify the `role_arn` parameter when the `type` parameter is set to `CUSTOM`. The `role_arn` parameter is required for all other data sources.
     */
    public val roleArn: Output<String>?
        get() = javaResource.roleArn().applyValue({ args0 -> args0.map({ args0 -> args0 }).orElse(null) })

    /**
     * Sets the frequency for Amazon Kendra to check the documents in your Data Source repository and update the index. If you don't set a schedule Amazon Kendra will not periodically update the index. You can call the `StartDataSourceSyncJob` API to update the index.
     */
    public val schedule: Output<String>?
        get() = javaResource.schedule().applyValue({ args0 -> args0.map({ args0 -> args0 }).orElse(null) })

    /**
     * The current status of the Data Source. When the status is `ACTIVE` the Data Source is ready to use. When the status is `FAILED`, the `error_message` field contains the reason that the Data Source failed.
     */
    public val status: Output<String>
        get() = javaResource.status().applyValue({ args0 -> args0 })

    /**
     * Key-value map of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    public val tags: Output<Map<String, String>>?
        get() = javaResource.tags().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }).orElse(null)
        })

    /**
     * A map of tags assigned to the resource, including those inherited from the provider `default_tags` configuration block.
     */
    @Deprecated(
        message = """
  Please use `tags` instead.
  """,
    )
    public val tagsAll: Output<Map<String, String>>
        get() = javaResource.tagsAll().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.key.to(args0.value)
            }).toMap()
        })

    /**
     * The type of data source repository. For an updated list of values, refer to [Valid Values for Type](https://docs.aws.amazon.com/kendra/latest/dg/API_CreateDataSource.html#Kendra-CreateDataSource-request-Type).
     * The following arguments are optional:
     */
    public val type: Output<String>
        get() = javaResource.type().applyValue({ args0 -> args0 })

    /**
     * The Unix timestamp of when the Data Source was last updated.
     */
    public val updatedAt: Output<String>
        get() = javaResource.updatedAt().applyValue({ args0 -> args0 })
}

public object DataSourceMapper : ResourceMapper<DataSource> {
    override fun supportsMappingOfType(javaResource: Resource): Boolean =
        com.pulumi.aws.kendra.DataSource::class == javaResource::class

    override fun map(javaResource: Resource): DataSource = DataSource(
        javaResource as
            com.pulumi.aws.kendra.DataSource,
    )
}

/**
 * @see [DataSource].
 * @param name The _unique_ name of the resulting resource.
 * @param block Builder for [DataSource].
 */
public suspend fun dataSource(name: String, block: suspend DataSourceResourceBuilder.() -> Unit): DataSource {
    val builder = DataSourceResourceBuilder()
    builder.name(name)
    block(builder)
    return builder.build()
}

/**
 * @see [DataSource].
 * @param name The _unique_ name of the resulting resource.
 */
public fun dataSource(name: String): DataSource {
    val builder = DataSourceResourceBuilder()
    builder.name(name)
    return builder.build()
}
