@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.kms.kotlin

import com.pulumi.aws.kms.AliasArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides an alias for a KMS customer master key. AWS Console enforces 1-to-1 mapping between aliases & keys,
 * but API (hence this provider too) allows you to create as many aliases as
 * the [account limits](http://docs.aws.amazon.com/kms/latest/developerguide/limits.html) allow you.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const a = new aws.kms.Key("a", {});
 * const aAlias = new aws.kms.Alias("a", {
 *     name: "alias/my-key-alias",
 *     targetKeyId: a.keyId,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * a = aws.kms.Key("a")
 * a_alias = aws.kms.Alias("a",
 *     name="alias/my-key-alias",
 *     target_key_id=a.key_id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var a = new Aws.Kms.Key("a");
 *     var aAlias = new Aws.Kms.Alias("a", new()
 *     {
 *         Name = "alias/my-key-alias",
 *         TargetKeyId = a.KeyId,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		a, err := kms.NewKey(ctx, "a", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = kms.NewAlias(ctx, "a", &kms.AliasArgs{
 * 			Name:        pulumi.String("alias/my-key-alias"),
 * 			TargetKeyId: a.KeyId,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.Alias;
 * import com.pulumi.aws.kms.AliasArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var a = new Key("a");
 *         var aAlias = new Alias("aAlias", AliasArgs.builder()
 *             .name("alias/my-key-alias")
 *             .targetKeyId(a.keyId())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   a:
 *     type: aws:kms:Key
 *   aAlias:
 *     type: aws:kms:Alias
 *     name: a
 *     properties:
 *       name: alias/my-key-alias
 *       targetKeyId: ${a.keyId}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import KMS aliases using the `name`. For example:
 * ```sh
 * $ pulumi import aws:kms/alias:Alias a alias/my-key-alias
 * ```
 * @property name The display name of the alias. The name must start with the word "alias" followed by a forward slash (alias/)
 * @property namePrefix Creates an unique alias beginning with the specified prefix.
 * The name must start with the word "alias" followed by a forward slash (alias/).  Conflicts with `name`.
 * @property targetKeyId Identifier for the key for which the alias is for, can be either an ARN or key_id.
 */
public data class AliasArgs(
    public val name: Output<String>? = null,
    public val namePrefix: Output<String>? = null,
    public val targetKeyId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.kms.AliasArgs> {
    override fun toJava(): com.pulumi.aws.kms.AliasArgs = com.pulumi.aws.kms.AliasArgs.builder()
        .name(name?.applyValue({ args0 -> args0 }))
        .namePrefix(namePrefix?.applyValue({ args0 -> args0 }))
        .targetKeyId(targetKeyId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [AliasArgs].
 */
@PulumiTagMarker
public class AliasArgsBuilder internal constructor() {
    private var name: Output<String>? = null

    private var namePrefix: Output<String>? = null

    private var targetKeyId: Output<String>? = null

    /**
     * @param value The display name of the alias. The name must start with the word "alias" followed by a forward slash (alias/)
     */
    @JvmName("shpveceqbdxrnjrt")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Creates an unique alias beginning with the specified prefix.
     * The name must start with the word "alias" followed by a forward slash (alias/).  Conflicts with `name`.
     */
    @JvmName("aiirgldmcliohkhj")
    public suspend fun namePrefix(`value`: Output<String>) {
        this.namePrefix = value
    }

    /**
     * @param value Identifier for the key for which the alias is for, can be either an ARN or key_id.
     */
    @JvmName("pfnpofqqwdcjjyrk")
    public suspend fun targetKeyId(`value`: Output<String>) {
        this.targetKeyId = value
    }

    /**
     * @param value The display name of the alias. The name must start with the word "alias" followed by a forward slash (alias/)
     */
    @JvmName("fofpsjhpigplodyu")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Creates an unique alias beginning with the specified prefix.
     * The name must start with the word "alias" followed by a forward slash (alias/).  Conflicts with `name`.
     */
    @JvmName("rcjfhtehuowkdksg")
    public suspend fun namePrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namePrefix = mapped
    }

    /**
     * @param value Identifier for the key for which the alias is for, can be either an ARN or key_id.
     */
    @JvmName("yvecdmmsnlcwjlos")
    public suspend fun targetKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetKeyId = mapped
    }

    internal fun build(): AliasArgs = AliasArgs(
        name = name,
        namePrefix = namePrefix,
        targetKeyId = targetKeyId,
    )
}
