@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.kms.kotlin

import com.pulumi.aws.kms.ReplicaExternalKeyArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages a KMS multi-Region replica key that uses external key material.
 * See the [AWS KMS Developer Guide](https://docs.aws.amazon.com/kms/latest/developerguide/multi-region-keys-import.html) for more information on importing key material into multi-Region keys.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const primary = new aws.kms.ExternalKey("primary", {
 *     description: "Multi-Region primary key",
 *     deletionWindowInDays: 30,
 *     multiRegion: true,
 *     enabled: true,
 *     keyMaterialBase64: "...",
 * });
 * const replica = new aws.kms.ReplicaExternalKey("replica", {
 *     description: "Multi-Region replica key",
 *     deletionWindowInDays: 7,
 *     primaryKeyArn: primaryAwsKmsExternal.arn,
 *     keyMaterialBase64: "...",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * primary = aws.kms.ExternalKey("primary",
 *     description="Multi-Region primary key",
 *     deletion_window_in_days=30,
 *     multi_region=True,
 *     enabled=True,
 *     key_material_base64="...")
 * replica = aws.kms.ReplicaExternalKey("replica",
 *     description="Multi-Region replica key",
 *     deletion_window_in_days=7,
 *     primary_key_arn=primary_aws_kms_external["arn"],
 *     key_material_base64="...")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var primary = new Aws.Kms.ExternalKey("primary", new()
 *     {
 *         Description = "Multi-Region primary key",
 *         DeletionWindowInDays = 30,
 *         MultiRegion = true,
 *         Enabled = true,
 *         KeyMaterialBase64 = "...",
 *     });
 *     var replica = new Aws.Kms.ReplicaExternalKey("replica", new()
 *     {
 *         Description = "Multi-Region replica key",
 *         DeletionWindowInDays = 7,
 *         PrimaryKeyArn = primaryAwsKmsExternal.Arn,
 *         KeyMaterialBase64 = "...",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := kms.NewExternalKey(ctx, "primary", &kms.ExternalKeyArgs{
 * 			Description:          pulumi.String("Multi-Region primary key"),
 * 			DeletionWindowInDays: pulumi.Int(30),
 * 			MultiRegion:          pulumi.Bool(true),
 * 			Enabled:              pulumi.Bool(true),
 * 			KeyMaterialBase64:    pulumi.String("..."),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = kms.NewReplicaExternalKey(ctx, "replica", &kms.ReplicaExternalKeyArgs{
 * 			Description:          pulumi.String("Multi-Region replica key"),
 * 			DeletionWindowInDays: pulumi.Int(7),
 * 			PrimaryKeyArn:        pulumi.Any(primaryAwsKmsExternal.Arn),
 * 			KeyMaterialBase64:    pulumi.String("..."),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.ExternalKey;
 * import com.pulumi.aws.kms.ExternalKeyArgs;
 * import com.pulumi.aws.kms.ReplicaExternalKey;
 * import com.pulumi.aws.kms.ReplicaExternalKeyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var primary = new ExternalKey("primary", ExternalKeyArgs.builder()
 *             .description("Multi-Region primary key")
 *             .deletionWindowInDays(30)
 *             .multiRegion(true)
 *             .enabled(true)
 *             .keyMaterialBase64("...")
 *             .build());
 *         var replica = new ReplicaExternalKey("replica", ReplicaExternalKeyArgs.builder()
 *             .description("Multi-Region replica key")
 *             .deletionWindowInDays(7)
 *             .primaryKeyArn(primaryAwsKmsExternal.arn())
 *             .keyMaterialBase64("...")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   primary:
 *     type: aws:kms:ExternalKey
 *     properties:
 *       description: Multi-Region primary key
 *       deletionWindowInDays: 30
 *       multiRegion: true
 *       enabled: true
 *       keyMaterialBase64: '...'
 *   replica:
 *     type: aws:kms:ReplicaExternalKey
 *     properties:
 *       description: Multi-Region replica key
 *       deletionWindowInDays: 7
 *       primaryKeyArn: ${primaryAwsKmsExternal.arn}
 *       keyMaterialBase64: '...'
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import KMS multi-Region replica keys using the `id`. For example:
 * ```sh
 * $ pulumi import aws:kms/replicaExternalKey:ReplicaExternalKey example 1234abcd-12ab-34cd-56ef-1234567890ab
 * ```
 * @property bypassPolicyLockoutSafetyCheck A flag to indicate whether to bypass the key policy lockout safety check.
 * Setting this value to true increases the risk that the KMS key becomes unmanageable. Do not set this value to true indiscriminately.
 * For more information, refer to the scenario in the [Default Key Policy](https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam) section in the _AWS Key Management Service Developer Guide_.
 * The default value is `false`.
 * @property deletionWindowInDays The waiting period, specified in number of days. After the waiting period ends, AWS KMS deletes the KMS key.
 * If you specify a value, it must be between `7` and `30`, inclusive. If you do not specify a value, it defaults to `30`.
 * @property description A description of the KMS key.
 * @property enabled Specifies whether the replica key is enabled. Disabled KMS keys cannot be used in cryptographic operations. Keys pending import can only be `false`. Imported keys default to `true` unless expired.
 * @property keyMaterialBase64 Base64 encoded 256-bit symmetric encryption key material to import. The KMS key is permanently associated with this key material. The same key material can be [reimported](https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html#reimport-key-material), but you cannot import different key material.
 * @property policy The key policy to attach to the KMS key. If you do not specify a key policy, AWS KMS attaches the [default key policy](https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default) to the KMS key.
 * @property primaryKeyArn The ARN of the multi-Region primary key to replicate. The primary key must be in a different AWS Region of the same AWS Partition. You can create only one replica of a given primary key in each AWS Region.
 * @property tags A map of tags to assign to the replica key. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property validTo Time at which the imported key material expires. When the key material expires, AWS KMS deletes the key material and the key becomes unusable. If not specified, key material does not expire. Valid values: [RFC3339 time string](https://tools.ietf.org/html/rfc3339#section-5.8) (`YYYY-MM-DDTHH:MM:SSZ`)
 */
public data class ReplicaExternalKeyArgs(
    public val bypassPolicyLockoutSafetyCheck: Output<Boolean>? = null,
    public val deletionWindowInDays: Output<Int>? = null,
    public val description: Output<String>? = null,
    public val enabled: Output<Boolean>? = null,
    public val keyMaterialBase64: Output<String>? = null,
    public val policy: Output<String>? = null,
    public val primaryKeyArn: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val validTo: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.kms.ReplicaExternalKeyArgs> {
    override fun toJava(): com.pulumi.aws.kms.ReplicaExternalKeyArgs =
        com.pulumi.aws.kms.ReplicaExternalKeyArgs.builder()
            .bypassPolicyLockoutSafetyCheck(bypassPolicyLockoutSafetyCheck?.applyValue({ args0 -> args0 }))
            .deletionWindowInDays(deletionWindowInDays?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .enabled(enabled?.applyValue({ args0 -> args0 }))
            .keyMaterialBase64(keyMaterialBase64?.applyValue({ args0 -> args0 }))
            .policy(policy?.applyValue({ args0 -> args0 }))
            .primaryKeyArn(primaryKeyArn?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .validTo(validTo?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ReplicaExternalKeyArgs].
 */
@PulumiTagMarker
public class ReplicaExternalKeyArgsBuilder internal constructor() {
    private var bypassPolicyLockoutSafetyCheck: Output<Boolean>? = null

    private var deletionWindowInDays: Output<Int>? = null

    private var description: Output<String>? = null

    private var enabled: Output<Boolean>? = null

    private var keyMaterialBase64: Output<String>? = null

    private var policy: Output<String>? = null

    private var primaryKeyArn: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var validTo: Output<String>? = null

    /**
     * @param value A flag to indicate whether to bypass the key policy lockout safety check.
     * Setting this value to true increases the risk that the KMS key becomes unmanageable. Do not set this value to true indiscriminately.
     * For more information, refer to the scenario in the [Default Key Policy](https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam) section in the _AWS Key Management Service Developer Guide_.
     * The default value is `false`.
     */
    @JvmName("lmoorcxcoggkxmcx")
    public suspend fun bypassPolicyLockoutSafetyCheck(`value`: Output<Boolean>) {
        this.bypassPolicyLockoutSafetyCheck = value
    }

    /**
     * @param value The waiting period, specified in number of days. After the waiting period ends, AWS KMS deletes the KMS key.
     * If you specify a value, it must be between `7` and `30`, inclusive. If you do not specify a value, it defaults to `30`.
     */
    @JvmName("todfxmxyrboqtiaf")
    public suspend fun deletionWindowInDays(`value`: Output<Int>) {
        this.deletionWindowInDays = value
    }

    /**
     * @param value A description of the KMS key.
     */
    @JvmName("hrlqjknegphfialm")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value Specifies whether the replica key is enabled. Disabled KMS keys cannot be used in cryptographic operations. Keys pending import can only be `false`. Imported keys default to `true` unless expired.
     */
    @JvmName("usalgfksgiavoicx")
    public suspend fun enabled(`value`: Output<Boolean>) {
        this.enabled = value
    }

    /**
     * @param value Base64 encoded 256-bit symmetric encryption key material to import. The KMS key is permanently associated with this key material. The same key material can be [reimported](https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html#reimport-key-material), but you cannot import different key material.
     */
    @JvmName("kmbapyvdpjrvxaja")
    public suspend fun keyMaterialBase64(`value`: Output<String>) {
        this.keyMaterialBase64 = value
    }

    /**
     * @param value The key policy to attach to the KMS key. If you do not specify a key policy, AWS KMS attaches the [default key policy](https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default) to the KMS key.
     */
    @JvmName("jwqfhbghqikllqsr")
    public suspend fun policy(`value`: Output<String>) {
        this.policy = value
    }

    /**
     * @param value The ARN of the multi-Region primary key to replicate. The primary key must be in a different AWS Region of the same AWS Partition. You can create only one replica of a given primary key in each AWS Region.
     */
    @JvmName("vjwkqfnkxgdgdxtn")
    public suspend fun primaryKeyArn(`value`: Output<String>) {
        this.primaryKeyArn = value
    }

    /**
     * @param value A map of tags to assign to the replica key. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("nqgetongkqyytsmj")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Time at which the imported key material expires. When the key material expires, AWS KMS deletes the key material and the key becomes unusable. If not specified, key material does not expire. Valid values: [RFC3339 time string](https://tools.ietf.org/html/rfc3339#section-5.8) (`YYYY-MM-DDTHH:MM:SSZ`)
     */
    @JvmName("cjvsnlxxtxsugmnh")
    public suspend fun validTo(`value`: Output<String>) {
        this.validTo = value
    }

    /**
     * @param value A flag to indicate whether to bypass the key policy lockout safety check.
     * Setting this value to true increases the risk that the KMS key becomes unmanageable. Do not set this value to true indiscriminately.
     * For more information, refer to the scenario in the [Default Key Policy](https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam) section in the _AWS Key Management Service Developer Guide_.
     * The default value is `false`.
     */
    @JvmName("ypktrlxxbtddtdnt")
    public suspend fun bypassPolicyLockoutSafetyCheck(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.bypassPolicyLockoutSafetyCheck = mapped
    }

    /**
     * @param value The waiting period, specified in number of days. After the waiting period ends, AWS KMS deletes the KMS key.
     * If you specify a value, it must be between `7` and `30`, inclusive. If you do not specify a value, it defaults to `30`.
     */
    @JvmName("wuusqqlworhsjcfm")
    public suspend fun deletionWindowInDays(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionWindowInDays = mapped
    }

    /**
     * @param value A description of the KMS key.
     */
    @JvmName("blhakfjqrhqtmion")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value Specifies whether the replica key is enabled. Disabled KMS keys cannot be used in cryptographic operations. Keys pending import can only be `false`. Imported keys default to `true` unless expired.
     */
    @JvmName("vdsgrkmgudmjkuts")
    public suspend fun enabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabled = mapped
    }

    /**
     * @param value Base64 encoded 256-bit symmetric encryption key material to import. The KMS key is permanently associated with this key material. The same key material can be [reimported](https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html#reimport-key-material), but you cannot import different key material.
     */
    @JvmName("ctdmlnqrgydrnsyf")
    public suspend fun keyMaterialBase64(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.keyMaterialBase64 = mapped
    }

    /**
     * @param value The key policy to attach to the KMS key. If you do not specify a key policy, AWS KMS attaches the [default key policy](https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default) to the KMS key.
     */
    @JvmName("axymckadkixobvig")
    public suspend fun policy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policy = mapped
    }

    /**
     * @param value The ARN of the multi-Region primary key to replicate. The primary key must be in a different AWS Region of the same AWS Partition. You can create only one replica of a given primary key in each AWS Region.
     */
    @JvmName("rsrttumndvbrtpdi")
    public suspend fun primaryKeyArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.primaryKeyArn = mapped
    }

    /**
     * @param value A map of tags to assign to the replica key. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ylckvjkqsfbiersl")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the replica key. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cucgtstawchaqcyu")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Time at which the imported key material expires. When the key material expires, AWS KMS deletes the key material and the key becomes unusable. If not specified, key material does not expire. Valid values: [RFC3339 time string](https://tools.ietf.org/html/rfc3339#section-5.8) (`YYYY-MM-DDTHH:MM:SSZ`)
     */
    @JvmName("addckoftcilvbmad")
    public suspend fun validTo(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.validTo = mapped
    }

    internal fun build(): ReplicaExternalKeyArgs = ReplicaExternalKeyArgs(
        bypassPolicyLockoutSafetyCheck = bypassPolicyLockoutSafetyCheck,
        deletionWindowInDays = deletionWindowInDays,
        description = description,
        enabled = enabled,
        keyMaterialBase64 = keyMaterialBase64,
        policy = policy,
        primaryKeyArn = primaryKeyArn,
        tags = tags,
        validTo = validTo,
    )
}
