@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.lambda.kotlin

import com.pulumi.aws.lambda.CodeSigningConfigArgs.builder
import com.pulumi.aws.lambda.kotlin.inputs.CodeSigningConfigAllowedPublishersArgs
import com.pulumi.aws.lambda.kotlin.inputs.CodeSigningConfigAllowedPublishersArgsBuilder
import com.pulumi.aws.lambda.kotlin.inputs.CodeSigningConfigPoliciesArgs
import com.pulumi.aws.lambda.kotlin.inputs.CodeSigningConfigPoliciesArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * Provides a Lambda Code Signing Config resource. A code signing configuration defines a list of allowed signing profiles and defines the code-signing validation policy (action to be taken if deployment validation checks fail).
 * For information about Lambda code signing configurations and how to use them, see [configuring code signing for Lambda functions](https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const newCsc = new aws.lambda.CodeSigningConfig("new_csc", {
 *     allowedPublishers: {
 *         signingProfileVersionArns: [
 *             example1.arn,
 *             example2.arn,
 *         ],
 *     },
 *     policies: {
 *         untrustedArtifactOnDeployment: "Warn",
 *     },
 *     description: "My awesome code signing config.",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * new_csc = aws.lambda_.CodeSigningConfig("new_csc",
 *     allowed_publishers={
 *         "signing_profile_version_arns": [
 *             example1["arn"],
 *             example2["arn"],
 *         ],
 *     },
 *     policies={
 *         "untrusted_artifact_on_deployment": "Warn",
 *     },
 *     description="My awesome code signing config.")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var newCsc = new Aws.Lambda.CodeSigningConfig("new_csc", new()
 *     {
 *         AllowedPublishers = new Aws.Lambda.Inputs.CodeSigningConfigAllowedPublishersArgs
 *         {
 *             SigningProfileVersionArns = new[]
 *             {
 *                 example1.Arn,
 *                 example2.Arn,
 *             },
 *         },
 *         Policies = new Aws.Lambda.Inputs.CodeSigningConfigPoliciesArgs
 *         {
 *             UntrustedArtifactOnDeployment = "Warn",
 *         },
 *         Description = "My awesome code signing config.",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lambda"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := lambda.NewCodeSigningConfig(ctx, "new_csc", &lambda.CodeSigningConfigArgs{
 * 			AllowedPublishers: &lambda.CodeSigningConfigAllowedPublishersArgs{
 * 				SigningProfileVersionArns: pulumi.StringArray{
 * 					example1.Arn,
 * 					example2.Arn,
 * 				},
 * 			},
 * 			Policies: &lambda.CodeSigningConfigPoliciesArgs{
 * 				UntrustedArtifactOnDeployment: pulumi.String("Warn"),
 * 			},
 * 			Description: pulumi.String("My awesome code signing config."),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lambda.CodeSigningConfig;
 * import com.pulumi.aws.lambda.CodeSigningConfigArgs;
 * import com.pulumi.aws.lambda.inputs.CodeSigningConfigAllowedPublishersArgs;
 * import com.pulumi.aws.lambda.inputs.CodeSigningConfigPoliciesArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var newCsc = new CodeSigningConfig("newCsc", CodeSigningConfigArgs.builder()
 *             .allowedPublishers(CodeSigningConfigAllowedPublishersArgs.builder()
 *                 .signingProfileVersionArns(
 *                     example1.arn(),
 *                     example2.arn())
 *                 .build())
 *             .policies(CodeSigningConfigPoliciesArgs.builder()
 *                 .untrustedArtifactOnDeployment("Warn")
 *                 .build())
 *             .description("My awesome code signing config.")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   newCsc:
 *     type: aws:lambda:CodeSigningConfig
 *     name: new_csc
 *     properties:
 *       allowedPublishers:
 *         signingProfileVersionArns:
 *           - ${example1.arn}
 *           - ${example2.arn}
 *       policies:
 *         untrustedArtifactOnDeployment: Warn
 *       description: My awesome code signing config.
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Code Signing Configs using their ARN. For example:
 * ```sh
 * $ pulumi import aws:lambda/codeSigningConfig:CodeSigningConfig imported_csc arn:aws:lambda:us-west-2:123456789012:code-signing-config:csc-0f6c334abcdea4d8b
 * ```
 * @property allowedPublishers A configuration block of allowed publishers as signing profiles for this code signing configuration. Detailed below.
 * @property description Descriptive name for this code signing configuration.
 * @property policies A configuration block of code signing policies that define the actions to take if the validation checks fail. Detailed below.
 */
public data class CodeSigningConfigArgs(
    public val allowedPublishers: Output<CodeSigningConfigAllowedPublishersArgs>? = null,
    public val description: Output<String>? = null,
    public val policies: Output<CodeSigningConfigPoliciesArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.lambda.CodeSigningConfigArgs> {
    override fun toJava(): com.pulumi.aws.lambda.CodeSigningConfigArgs =
        com.pulumi.aws.lambda.CodeSigningConfigArgs.builder()
            .allowedPublishers(allowedPublishers?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .description(description?.applyValue({ args0 -> args0 }))
            .policies(policies?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) })).build()
}

/**
 * Builder for [CodeSigningConfigArgs].
 */
@PulumiTagMarker
public class CodeSigningConfigArgsBuilder internal constructor() {
    private var allowedPublishers: Output<CodeSigningConfigAllowedPublishersArgs>? = null

    private var description: Output<String>? = null

    private var policies: Output<CodeSigningConfigPoliciesArgs>? = null

    /**
     * @param value A configuration block of allowed publishers as signing profiles for this code signing configuration. Detailed below.
     */
    @JvmName("omdyfblnlimfpats")
    public suspend fun allowedPublishers(`value`: Output<CodeSigningConfigAllowedPublishersArgs>) {
        this.allowedPublishers = value
    }

    /**
     * @param value Descriptive name for this code signing configuration.
     */
    @JvmName("qwqvuhcbecyehkej")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value A configuration block of code signing policies that define the actions to take if the validation checks fail. Detailed below.
     */
    @JvmName("okxpivmeprmteeok")
    public suspend fun policies(`value`: Output<CodeSigningConfigPoliciesArgs>) {
        this.policies = value
    }

    /**
     * @param value A configuration block of allowed publishers as signing profiles for this code signing configuration. Detailed below.
     */
    @JvmName("buacgpddqfydnoro")
    public suspend fun allowedPublishers(`value`: CodeSigningConfigAllowedPublishersArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowedPublishers = mapped
    }

    /**
     * @param argument A configuration block of allowed publishers as signing profiles for this code signing configuration. Detailed below.
     */
    @JvmName("lnwrptfsnhwouyim")
    public suspend fun allowedPublishers(argument: suspend CodeSigningConfigAllowedPublishersArgsBuilder.() -> Unit) {
        val toBeMapped = CodeSigningConfigAllowedPublishersArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.allowedPublishers = mapped
    }

    /**
     * @param value Descriptive name for this code signing configuration.
     */
    @JvmName("syomwhgogxlfpaie")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value A configuration block of code signing policies that define the actions to take if the validation checks fail. Detailed below.
     */
    @JvmName("ifbsejsbcemwytsr")
    public suspend fun policies(`value`: CodeSigningConfigPoliciesArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policies = mapped
    }

    /**
     * @param argument A configuration block of code signing policies that define the actions to take if the validation checks fail. Detailed below.
     */
    @JvmName("tgcibuugyviwuifv")
    public suspend fun policies(argument: suspend CodeSigningConfigPoliciesArgsBuilder.() -> Unit) {
        val toBeMapped = CodeSigningConfigPoliciesArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.policies = mapped
    }

    internal fun build(): CodeSigningConfigArgs = CodeSigningConfigArgs(
        allowedPublishers = allowedPublishers,
        description = description,
        policies = policies,
    )
}
