@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.lambda.kotlin

import com.pulumi.aws.lambda.PermissionArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Gives an external source (like an EventBridge Rule, SNS, or S3) permission to access the Lambda function.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const iamForLambda = new aws.iam.Role("iam_for_lambda", {
 *     name: "iam_for_lambda",
 *     assumeRolePolicy: JSON.stringify({
 *         Version: "2012-10-17",
 *         Statement: [{
 *             Action: "sts:AssumeRole",
 *             Effect: "Allow",
 *             Sid: "",
 *             Principal: {
 *                 Service: "lambda.amazonaws.com",
 *             },
 *         }],
 *     }),
 * });
 * const testLambda = new aws.lambda.Function("test_lambda", {
 *     code: new pulumi.asset.FileArchive("lambdatest.zip"),
 *     name: "lambda_function_name",
 *     role: iamForLambda.arn,
 *     handler: "exports.handler",
 *     runtime: aws.lambda.Runtime.NodeJS16dX,
 * });
 * const testAlias = new aws.lambda.Alias("test_alias", {
 *     name: "testalias",
 *     description: "a sample description",
 *     functionName: testLambda.name,
 *     functionVersion: "$LATEST",
 * });
 * const allowCloudwatch = new aws.lambda.Permission("allow_cloudwatch", {
 *     statementId: "AllowExecutionFromCloudWatch",
 *     action: "lambda:InvokeFunction",
 *     "function": testLambda.name,
 *     principal: "events.amazonaws.com",
 *     sourceArn: "arn:aws:events:eu-west-1:111122223333:rule/RunDaily",
 *     qualifier: testAlias.name,
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * iam_for_lambda = aws.iam.Role("iam_for_lambda",
 *     name="iam_for_lambda",
 *     assume_role_policy=json.dumps({
 *         "Version": "2012-10-17",
 *         "Statement": [{
 *             "Action": "sts:AssumeRole",
 *             "Effect": "Allow",
 *             "Sid": "",
 *             "Principal": {
 *                 "Service": "lambda.amazonaws.com",
 *             },
 *         }],
 *     }))
 * test_lambda = aws.lambda_.Function("test_lambda",
 *     code=pulumi.FileArchive("lambdatest.zip"),
 *     name="lambda_function_name",
 *     role=iam_for_lambda.arn,
 *     handler="exports.handler",
 *     runtime=aws.lambda_.Runtime.NODE_JS16D_X)
 * test_alias = aws.lambda_.Alias("test_alias",
 *     name="testalias",
 *     description="a sample description",
 *     function_name=test_lambda.name,
 *     function_version="$LATEST")
 * allow_cloudwatch = aws.lambda_.Permission("allow_cloudwatch",
 *     statement_id="AllowExecutionFromCloudWatch",
 *     action="lambda:InvokeFunction",
 *     function=test_lambda.name,
 *     principal="events.amazonaws.com",
 *     source_arn="arn:aws:events:eu-west-1:111122223333:rule/RunDaily",
 *     qualifier=test_alias.name)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var iamForLambda = new Aws.Iam.Role("iam_for_lambda", new()
 *     {
 *         Name = "iam_for_lambda",
 *         AssumeRolePolicy = JsonSerializer.Serialize(new Dictionary<string, object?>
 *         {
 *             ["Version"] = "2012-10-17",
 *             ["Statement"] = new[]
 *             {
 *                 new Dictionary<string, object?>
 *                 {
 *                     ["Action"] = "sts:AssumeRole",
 *                     ["Effect"] = "Allow",
 *                     ["Sid"] = "",
 *                     ["Principal"] = new Dictionary<string, object?>
 *                     {
 *                         ["Service"] = "lambda.amazonaws.com",
 *                     },
 *                 },
 *             },
 *         }),
 *     });
 *     var testLambda = new Aws.Lambda.Function("test_lambda", new()
 *     {
 *         Code = new FileArchive("lambdatest.zip"),
 *         Name = "lambda_function_name",
 *         Role = iamForLambda.Arn,
 *         Handler = "exports.handler",
 *         Runtime = Aws.Lambda.Runtime.NodeJS16dX,
 *     });
 *     var testAlias = new Aws.Lambda.Alias("test_alias", new()
 *     {
 *         Name = "testalias",
 *         Description = "a sample description",
 *         FunctionName = testLambda.Name,
 *         FunctionVersion = "$LATEST",
 *     });
 *     var allowCloudwatch = new Aws.Lambda.Permission("allow_cloudwatch", new()
 *     {
 *         StatementId = "AllowExecutionFromCloudWatch",
 *         Action = "lambda:InvokeFunction",
 *         Function = testLambda.Name,
 *         Principal = "events.amazonaws.com",
 *         SourceArn = "arn:aws:events:eu-west-1:111122223333:rule/RunDaily",
 *         Qualifier = testAlias.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lambda"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 			"Version": "2012-10-17",
 * 			"Statement": []map[string]interface{}{
 * 				map[string]interface{}{
 * 					"Action": "sts:AssumeRole",
 * 					"Effect": "Allow",
 * 					"Sid":    "",
 * 					"Principal": map[string]interface{}{
 * 						"Service": "lambda.amazonaws.com",
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		json0 := string(tmpJSON0)
 * 		iamForLambda, err := iam.NewRole(ctx, "iam_for_lambda", &iam.RoleArgs{
 * 			Name:             pulumi.String("iam_for_lambda"),
 * 			AssumeRolePolicy: pulumi.String(json0),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		testLambda, err := lambda.NewFunction(ctx, "test_lambda", &lambda.FunctionArgs{
 * 			Code:    pulumi.NewFileArchive("lambdatest.zip"),
 * 			Name:    pulumi.String("lambda_function_name"),
 * 			Role:    iamForLambda.Arn,
 * 			Handler: pulumi.String("exports.handler"),
 * 			Runtime: pulumi.String(lambda.RuntimeNodeJS16dX),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		testAlias, err := lambda.NewAlias(ctx, "test_alias", &lambda.AliasArgs{
 * 			Name:            pulumi.String("testalias"),
 * 			Description:     pulumi.String("a sample description"),
 * 			FunctionName:    testLambda.Name,
 * 			FunctionVersion: pulumi.String("$LATEST"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lambda.NewPermission(ctx, "allow_cloudwatch", &lambda.PermissionArgs{
 * 			StatementId: pulumi.String("AllowExecutionFromCloudWatch"),
 * 			Action:      pulumi.String("lambda:InvokeFunction"),
 * 			Function:    testLambda.Name,
 * 			Principal:   pulumi.String("events.amazonaws.com"),
 * 			SourceArn:   pulumi.String("arn:aws:events:eu-west-1:111122223333:rule/RunDaily"),
 * 			Qualifier:   testAlias.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.lambda.Function;
 * import com.pulumi.aws.lambda.FunctionArgs;
 * import com.pulumi.aws.lambda.Alias;
 * import com.pulumi.aws.lambda.AliasArgs;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import com.pulumi.asset.FileArchive;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var iamForLambda = new Role("iamForLambda", RoleArgs.builder()
 *             .name("iam_for_lambda")
 *             .assumeRolePolicy(serializeJson(
 *                 jsonObject(
 *                     jsonProperty("Version", "2012-10-17"),
 *                     jsonProperty("Statement", jsonArray(jsonObject(
 *                         jsonProperty("Action", "sts:AssumeRole"),
 *                         jsonProperty("Effect", "Allow"),
 *                         jsonProperty("Sid", ""),
 *                         jsonProperty("Principal", jsonObject(
 *                             jsonProperty("Service", "lambda.amazonaws.com")
 *                         ))
 *                     )))
 *                 )))
 *             .build());
 *         var testLambda = new Function("testLambda", FunctionArgs.builder()
 *             .code(new FileArchive("lambdatest.zip"))
 *             .name("lambda_function_name")
 *             .role(iamForLambda.arn())
 *             .handler("exports.handler")
 *             .runtime("nodejs16.x")
 *             .build());
 *         var testAlias = new Alias("testAlias", AliasArgs.builder()
 *             .name("testalias")
 *             .description("a sample description")
 *             .functionName(testLambda.name())
 *             .functionVersion("$LATEST")
 *             .build());
 *         var allowCloudwatch = new Permission("allowCloudwatch", PermissionArgs.builder()
 *             .statementId("AllowExecutionFromCloudWatch")
 *             .action("lambda:InvokeFunction")
 *             .function(testLambda.name())
 *             .principal("events.amazonaws.com")
 *             .sourceArn("arn:aws:events:eu-west-1:111122223333:rule/RunDaily")
 *             .qualifier(testAlias.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   allowCloudwatch:
 *     type: aws:lambda:Permission
 *     name: allow_cloudwatch
 *     properties:
 *       statementId: AllowExecutionFromCloudWatch
 *       action: lambda:InvokeFunction
 *       function: ${testLambda.name}
 *       principal: events.amazonaws.com
 *       sourceArn: arn:aws:events:eu-west-1:111122223333:rule/RunDaily
 *       qualifier: ${testAlias.name}
 *   testAlias:
 *     type: aws:lambda:Alias
 *     name: test_alias
 *     properties:
 *       name: testalias
 *       description: a sample description
 *       functionName: ${testLambda.name}
 *       functionVersion: $LATEST
 *   testLambda:
 *     type: aws:lambda:Function
 *     name: test_lambda
 *     properties:
 *       code:
 *         fn::FileArchive: lambdatest.zip
 *       name: lambda_function_name
 *       role: ${iamForLambda.arn}
 *       handler: exports.handler
 *       runtime: nodejs16.x
 *   iamForLambda:
 *     type: aws:iam:Role
 *     name: iam_for_lambda
 *     properties:
 *       name: iam_for_lambda
 *       assumeRolePolicy:
 *         fn::toJSON:
 *           Version: 2012-10-17
 *           Statement:
 *             - Action: sts:AssumeRole
 *               Effect: Allow
 *               Sid:
 *               Principal:
 *                 Service: lambda.amazonaws.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With SNS
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _default = new aws.sns.Topic("default", {name: "call-lambda-maybe"});
 * const defaultRole = new aws.iam.Role("default", {
 *     name: "iam_for_lambda_with_sns",
 *     assumeRolePolicy: JSON.stringify({
 *         Version: "2012-10-17",
 *         Statement: [{
 *             Action: "sts:AssumeRole",
 *             Effect: "Allow",
 *             Sid: "",
 *             Principal: {
 *                 Service: "lambda.amazonaws.com",
 *             },
 *         }],
 *     }),
 * });
 * const func = new aws.lambda.Function("func", {
 *     code: new pulumi.asset.FileArchive("lambdatest.zip"),
 *     name: "lambda_called_from_sns",
 *     role: defaultRole.arn,
 *     handler: "exports.handler",
 *     runtime: aws.lambda.Runtime.Python3d7,
 * });
 * const withSns = new aws.lambda.Permission("with_sns", {
 *     statementId: "AllowExecutionFromSNS",
 *     action: "lambda:InvokeFunction",
 *     "function": func.name,
 *     principal: "sns.amazonaws.com",
 *     sourceArn: _default.arn,
 * });
 * const lambda = new aws.sns.TopicSubscription("lambda", {
 *     topic: _default.arn,
 *     protocol: "lambda",
 *     endpoint: func.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * default = aws.sns.Topic("default", name="call-lambda-maybe")
 * default_role = aws.iam.Role("default",
 *     name="iam_for_lambda_with_sns",
 *     assume_role_policy=json.dumps({
 *         "Version": "2012-10-17",
 *         "Statement": [{
 *             "Action": "sts:AssumeRole",
 *             "Effect": "Allow",
 *             "Sid": "",
 *             "Principal": {
 *                 "Service": "lambda.amazonaws.com",
 *             },
 *         }],
 *     }))
 * func = aws.lambda_.Function("func",
 *     code=pulumi.FileArchive("lambdatest.zip"),
 *     name="lambda_called_from_sns",
 *     role=default_role.arn,
 *     handler="exports.handler",
 *     runtime=aws.lambda_.Runtime.PYTHON3D7)
 * with_sns = aws.lambda_.Permission("with_sns",
 *     statement_id="AllowExecutionFromSNS",
 *     action="lambda:InvokeFunction",
 *     function=func.name,
 *     principal="sns.amazonaws.com",
 *     source_arn=default.arn)
 * lambda_ = aws.sns.TopicSubscription("lambda",
 *     topic=default.arn,
 *     protocol="lambda",
 *     endpoint=func.arn)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @default = new Aws.Sns.Topic("default", new()
 *     {
 *         Name = "call-lambda-maybe",
 *     });
 *     var defaultRole = new Aws.Iam.Role("default", new()
 *     {
 *         Name = "iam_for_lambda_with_sns",
 *         AssumeRolePolicy = JsonSerializer.Serialize(new Dictionary<string, object?>
 *         {
 *             ["Version"] = "2012-10-17",
 *             ["Statement"] = new[]
 *             {
 *                 new Dictionary<string, object?>
 *                 {
 *                     ["Action"] = "sts:AssumeRole",
 *                     ["Effect"] = "Allow",
 *                     ["Sid"] = "",
 *                     ["Principal"] = new Dictionary<string, object?>
 *                     {
 *                         ["Service"] = "lambda.amazonaws.com",
 *                     },
 *                 },
 *             },
 *         }),
 *     });
 *     var func = new Aws.Lambda.Function("func", new()
 *     {
 *         Code = new FileArchive("lambdatest.zip"),
 *         Name = "lambda_called_from_sns",
 *         Role = defaultRole.Arn,
 *         Handler = "exports.handler",
 *         Runtime = Aws.Lambda.Runtime.Python3d7,
 *     });
 *     var withSns = new Aws.Lambda.Permission("with_sns", new()
 *     {
 *         StatementId = "AllowExecutionFromSNS",
 *         Action = "lambda:InvokeFunction",
 *         Function = func.Name,
 *         Principal = "sns.amazonaws.com",
 *         SourceArn = @default.Arn,
 *     });
 *     var lambda = new Aws.Sns.TopicSubscription("lambda", new()
 *     {
 *         Topic = @default.Arn,
 *         Protocol = "lambda",
 *         Endpoint = func.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lambda"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sns.NewTopic(ctx, "default", &sns.TopicArgs{
 * 			Name: pulumi.String("call-lambda-maybe"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 			"Version": "2012-10-17",
 * 			"Statement": []map[string]interface{}{
 * 				map[string]interface{}{
 * 					"Action": "sts:AssumeRole",
 * 					"Effect": "Allow",
 * 					"Sid":    "",
 * 					"Principal": map[string]interface{}{
 * 						"Service": "lambda.amazonaws.com",
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		json0 := string(tmpJSON0)
 * 		defaultRole, err := iam.NewRole(ctx, "default", &iam.RoleArgs{
 * 			Name:             pulumi.String("iam_for_lambda_with_sns"),
 * 			AssumeRolePolicy: pulumi.String(json0),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lambda.NewFunction(ctx, "func", &lambda.FunctionArgs{
 * 			Code:    pulumi.NewFileArchive("lambdatest.zip"),
 * 			Name:    pulumi.String("lambda_called_from_sns"),
 * 			Role:    defaultRole.Arn,
 * 			Handler: pulumi.String("exports.handler"),
 * 			Runtime: pulumi.String(lambda.RuntimePython3d7),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lambda.NewPermission(ctx, "with_sns", &lambda.PermissionArgs{
 * 			StatementId: pulumi.String("AllowExecutionFromSNS"),
 * 			Action:      pulumi.String("lambda:InvokeFunction"),
 * 			Function:    _func.Name,
 * 			Principal:   pulumi.String("sns.amazonaws.com"),
 * 			SourceArn:   _default.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = sns.NewTopicSubscription(ctx, "lambda", &sns.TopicSubscriptionArgs{
 * 			Topic:    _default.Arn,
 * 			Protocol: pulumi.String("lambda"),
 * 			Endpoint: _func.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sns.Topic;
 * import com.pulumi.aws.sns.TopicArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.lambda.Function;
 * import com.pulumi.aws.lambda.FunctionArgs;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import com.pulumi.aws.sns.TopicSubscription;
 * import com.pulumi.aws.sns.TopicSubscriptionArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import com.pulumi.asset.FileArchive;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Topic("default", TopicArgs.builder()
 *             .name("call-lambda-maybe")
 *             .build());
 *         var defaultRole = new Role("defaultRole", RoleArgs.builder()
 *             .name("iam_for_lambda_with_sns")
 *             .assumeRolePolicy(serializeJson(
 *                 jsonObject(
 *                     jsonProperty("Version", "2012-10-17"),
 *                     jsonProperty("Statement", jsonArray(jsonObject(
 *                         jsonProperty("Action", "sts:AssumeRole"),
 *                         jsonProperty("Effect", "Allow"),
 *                         jsonProperty("Sid", ""),
 *                         jsonProperty("Principal", jsonObject(
 *                             jsonProperty("Service", "lambda.amazonaws.com")
 *                         ))
 *                     )))
 *                 )))
 *             .build());
 *         var func = new Function("func", FunctionArgs.builder()
 *             .code(new FileArchive("lambdatest.zip"))
 *             .name("lambda_called_from_sns")
 *             .role(defaultRole.arn())
 *             .handler("exports.handler")
 *             .runtime("python3.7")
 *             .build());
 *         var withSns = new Permission("withSns", PermissionArgs.builder()
 *             .statementId("AllowExecutionFromSNS")
 *             .action("lambda:InvokeFunction")
 *             .function(func.name())
 *             .principal("sns.amazonaws.com")
 *             .sourceArn(default_.arn())
 *             .build());
 *         var lambda = new TopicSubscription("lambda", TopicSubscriptionArgs.builder()
 *             .topic(default_.arn())
 *             .protocol("lambda")
 *             .endpoint(func.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   withSns:
 *     type: aws:lambda:Permission
 *     name: with_sns
 *     properties:
 *       statementId: AllowExecutionFromSNS
 *       action: lambda:InvokeFunction
 *       function: ${func.name}
 *       principal: sns.amazonaws.com
 *       sourceArn: ${default.arn}
 *   default:
 *     type: aws:sns:Topic
 *     properties:
 *       name: call-lambda-maybe
 *   lambda:
 *     type: aws:sns:TopicSubscription
 *     properties:
 *       topic: ${default.arn}
 *       protocol: lambda
 *       endpoint: ${func.arn}
 *   func:
 *     type: aws:lambda:Function
 *     properties:
 *       code:
 *         fn::FileArchive: lambdatest.zip
 *       name: lambda_called_from_sns
 *       role: ${defaultRole.arn}
 *       handler: exports.handler
 *       runtime: python3.7
 *   defaultRole:
 *     type: aws:iam:Role
 *     name: default
 *     properties:
 *       name: iam_for_lambda_with_sns
 *       assumeRolePolicy:
 *         fn::toJSON:
 *           Version: 2012-10-17
 *           Statement:
 *             - Action: sts:AssumeRole
 *               Effect: Allow
 *               Sid:
 *               Principal:
 *                 Service: lambda.amazonaws.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With API Gateway REST API
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const myDemoAPI = new aws.apigateway.RestApi("MyDemoAPI", {
 *     name: "MyDemoAPI",
 *     description: "This is my API for demonstration purposes",
 * });
 * const lambdaPermission = new aws.lambda.Permission("lambda_permission", {
 *     statementId: "AllowMyDemoAPIInvoke",
 *     action: "lambda:InvokeFunction",
 *     "function": "MyDemoFunction",
 *     principal: "apigateway.amazonaws.com",
 *     sourceArn: pulumi.interpolate`${myDemoAPI.executionArn}/*`,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * my_demo_api = aws.apigateway.RestApi("MyDemoAPI",
 *     name="MyDemoAPI",
 *     description="This is my API for demonstration purposes")
 * lambda_permission = aws.lambda_.Permission("lambda_permission",
 *     statement_id="AllowMyDemoAPIInvoke",
 *     action="lambda:InvokeFunction",
 *     function="MyDemoFunction",
 *     principal="apigateway.amazonaws.com",
 *     source_arn=my_demo_api.execution_arn.apply(lambda execution_arn: f"{execution_arn}/*"))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var myDemoAPI = new Aws.ApiGateway.RestApi("MyDemoAPI", new()
 *     {
 *         Name = "MyDemoAPI",
 *         Description = "This is my API for demonstration purposes",
 *     });
 *     var lambdaPermission = new Aws.Lambda.Permission("lambda_permission", new()
 *     {
 *         StatementId = "AllowMyDemoAPIInvoke",
 *         Action = "lambda:InvokeFunction",
 *         Function = "MyDemoFunction",
 *         Principal = "apigateway.amazonaws.com",
 *         SourceArn = myDemoAPI.ExecutionArn.Apply(executionArn => $"{executionArn}/*"),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/apigateway"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lambda"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		myDemoAPI, err := apigateway.NewRestApi(ctx, "MyDemoAPI", &apigateway.RestApiArgs{
 * 			Name:        pulumi.String("MyDemoAPI"),
 * 			Description: pulumi.String("This is my API for demonstration purposes"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lambda.NewPermission(ctx, "lambda_permission", &lambda.PermissionArgs{
 * 			StatementId: pulumi.String("AllowMyDemoAPIInvoke"),
 * 			Action:      pulumi.String("lambda:InvokeFunction"),
 * 			Function:    pulumi.Any("MyDemoFunction"),
 * 			Principal:   pulumi.String("apigateway.amazonaws.com"),
 * 			SourceArn: myDemoAPI.ExecutionArn.ApplyT(func(executionArn string) (string, error) {
 * 				return fmt.Sprintf("%v/*", executionArn), nil
 * 			}).(pulumi.StringOutput),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.apigateway.RestApi;
 * import com.pulumi.aws.apigateway.RestApiArgs;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var myDemoAPI = new RestApi("myDemoAPI", RestApiArgs.builder()
 *             .name("MyDemoAPI")
 *             .description("This is my API for demonstration purposes")
 *             .build());
 *         var lambdaPermission = new Permission("lambdaPermission", PermissionArgs.builder()
 *             .statementId("AllowMyDemoAPIInvoke")
 *             .action("lambda:InvokeFunction")
 *             .function("MyDemoFunction")
 *             .principal("apigateway.amazonaws.com")
 *             .sourceArn(myDemoAPI.executionArn().applyValue(executionArn -> String.format("%s/*", executionArn)))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   myDemoAPI:
 *     type: aws:apigateway:RestApi
 *     name: MyDemoAPI
 *     properties:
 *       name: MyDemoAPI
 *       description: This is my API for demonstration purposes
 *   lambdaPermission:
 *     type: aws:lambda:Permission
 *     name: lambda_permission
 *     properties:
 *       statementId: AllowMyDemoAPIInvoke
 *       action: lambda:InvokeFunction
 *       function: MyDemoFunction
 *       principal: apigateway.amazonaws.com
 *       sourceArn: ${myDemoAPI.executionArn}/*
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With CloudWatch Log Group
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _default = new aws.cloudwatch.LogGroup("default", {name: "/default"});
 * const assumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["lambda&#46;amazonaws&#46;com"],
 *         }],
 *         actions: ["sts:AssumeRole"],
 *     }],
 * });
 * const defaultRole = new aws.iam.Role("default", {
 *     name: "iam_for_lambda_called_from_cloudwatch_logs",
 *     assumeRolePolicy: assumeRole.then(assumeRole => assumeRole.json),
 * });
 * const loggingFunction = new aws.lambda.Function("logging", {
 *     code: new pulumi.asset.FileArchive("lamba_logging.zip"),
 *     name: "lambda_called_from_cloudwatch_logs",
 *     handler: "exports.handler",
 *     role: defaultRole.arn,
 *     runtime: aws.lambda.Runtime.Python3d7,
 * });
 * const logging = new aws.lambda.Permission("logging", {
 *     action: "lambda:InvokeFunction",
 *     "function": loggingFunction.name,
 *     principal: "logs.eu-west-1.amazonaws.com",
 *     sourceArn: pulumi.interpolate`${_default.arn}:*`,
 * });
 * const loggingLogSubscriptionFilter = new aws.cloudwatch.LogSubscriptionFilter("logging", {
 *     destinationArn: loggingFunction.arn,
 *     filterPattern: "",
 *     logGroup: _default.name,
 *     name: "logging_default",
 * }, {
 *     dependsOn: [logging],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * default = aws.cloudwatch.LogGroup("default", name="/default")
 * assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["lambda&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": ["sts:AssumeRole"],
 * }])
 * default_role = aws.iam.Role("default",
 *     name="iam_for_lambda_called_from_cloudwatch_logs",
 *     assume_role_policy=assume_role.json)
 * logging_function = aws.lambda_.Function("logging",
 *     code=pulumi.FileArchive("lamba_logging.zip"),
 *     name="lambda_called_from_cloudwatch_logs",
 *     handler="exports.handler",
 *     role=default_role.arn,
 *     runtime=aws.lambda_.Runtime.PYTHON3D7)
 * logging = aws.lambda_.Permission("logging",
 *     action="lambda:InvokeFunction",
 *     function=logging_function.name,
 *     principal="logs.eu-west-1.amazonaws.com",
 *     source_arn=default.arn.apply(lambda arn: f"{arn}:*"))
 * logging_log_subscription_filter = aws.cloudwatch.LogSubscriptionFilter("logging",
 *     destination_arn=logging_function.arn,
 *     filter_pattern="",
 *     log_group=default.name,
 *     name="logging_default",
 *     opts = pulumi.ResourceOptions(depends_on=[logging]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @default = new Aws.CloudWatch.LogGroup("default", new()
 *     {
 *         Name = "/default",
 *     });
 *     var assumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "lambda.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *             },
 *         },
 *     });
 *     var defaultRole = new Aws.Iam.Role("default", new()
 *     {
 *         Name = "iam_for_lambda_called_from_cloudwatch_logs",
 *         AssumeRolePolicy = assumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 *     var loggingFunction = new Aws.Lambda.Function("logging", new()
 *     {
 *         Code = new FileArchive("lamba_logging.zip"),
 *         Name = "lambda_called_from_cloudwatch_logs",
 *         Handler = "exports.handler",
 *         Role = defaultRole.Arn,
 *         Runtime = Aws.Lambda.Runtime.Python3d7,
 *     });
 *     var logging = new Aws.Lambda.Permission("logging", new()
 *     {
 *         Action = "lambda:InvokeFunction",
 *         Function = loggingFunction.Name,
 *         Principal = "logs.eu-west-1.amazonaws.com",
 *         SourceArn = @default.Arn.Apply(arn => $"{arn}:*"),
 *     });
 *     var loggingLogSubscriptionFilter = new Aws.CloudWatch.LogSubscriptionFilter("logging", new()
 *     {
 *         DestinationArn = loggingFunction.Arn,
 *         FilterPattern = "",
 *         LogGroup = @default.Name,
 *         Name = "logging_default",
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             logging,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lambda"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudwatch.NewLogGroup(ctx, "default", &cloudwatch.LogGroupArgs{
 * 			Name: pulumi.String("/default"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		assumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"lambda.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		defaultRole, err := iam.NewRole(ctx, "default", &iam.RoleArgs{
 * 			Name:             pulumi.String("iam_for_lambda_called_from_cloudwatch_logs"),
 * 			AssumeRolePolicy: pulumi.String(assumeRole.Json),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		loggingFunction, err := lambda.NewFunction(ctx, "logging", &lambda.FunctionArgs{
 * 			Code:    pulumi.NewFileArchive("lamba_logging.zip"),
 * 			Name:    pulumi.String("lambda_called_from_cloudwatch_logs"),
 * 			Handler: pulumi.String("exports.handler"),
 * 			Role:    defaultRole.Arn,
 * 			Runtime: pulumi.String(lambda.RuntimePython3d7),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		logging, err := lambda.NewPermission(ctx, "logging", &lambda.PermissionArgs{
 * 			Action:    pulumi.String("lambda:InvokeFunction"),
 * 			Function:  loggingFunction.Name,
 * 			Principal: pulumi.String("logs.eu-west-1.amazonaws.com"),
 * 			SourceArn: _default.Arn.ApplyT(func(arn string) (string, error) {
 * 				return fmt.Sprintf("%v:*", arn), nil
 * 			}).(pulumi.StringOutput),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = cloudwatch.NewLogSubscriptionFilter(ctx, "logging", &cloudwatch.LogSubscriptionFilterArgs{
 * 			DestinationArn: loggingFunction.Arn,
 * 			FilterPattern:  pulumi.String(""),
 * 			LogGroup:       _default.Name,
 * 			Name:           pulumi.String("logging_default"),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			logging,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.cloudwatch.LogGroupArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.lambda.Function;
 * import com.pulumi.aws.lambda.FunctionArgs;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import com.pulumi.aws.cloudwatch.LogSubscriptionFilter;
 * import com.pulumi.aws.cloudwatch.LogSubscriptionFilterArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import com.pulumi.asset.FileArchive;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new LogGroup("default", LogGroupArgs.builder()
 *             .name("/default")
 *             .build());
 *         final var assumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("lambda.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var defaultRole = new Role("defaultRole", RoleArgs.builder()
 *             .name("iam_for_lambda_called_from_cloudwatch_logs")
 *             .assumeRolePolicy(assumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .build());
 *         var loggingFunction = new Function("loggingFunction", FunctionArgs.builder()
 *             .code(new FileArchive("lamba_logging.zip"))
 *             .name("lambda_called_from_cloudwatch_logs")
 *             .handler("exports.handler")
 *             .role(defaultRole.arn())
 *             .runtime("python3.7")
 *             .build());
 *         var logging = new Permission("logging", PermissionArgs.builder()
 *             .action("lambda:InvokeFunction")
 *             .function(loggingFunction.name())
 *             .principal("logs.eu-west-1.amazonaws.com")
 *             .sourceArn(default_.arn().applyValue(arn -> String.format("%s:*", arn)))
 *             .build());
 *         var loggingLogSubscriptionFilter = new LogSubscriptionFilter("loggingLogSubscriptionFilter", LogSubscriptionFilterArgs.builder()
 *             .destinationArn(loggingFunction.arn())
 *             .filterPattern("")
 *             .logGroup(default_.name())
 *             .name("logging_default")
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(logging)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   logging:
 *     type: aws:lambda:Permission
 *     properties:
 *       action: lambda:InvokeFunction
 *       function: ${loggingFunction.name}
 *       principal: logs.eu-west-1.amazonaws.com
 *       sourceArn: ${default.arn}:*
 *   default:
 *     type: aws:cloudwatch:LogGroup
 *     properties:
 *       name: /default
 *   loggingLogSubscriptionFilter:
 *     type: aws:cloudwatch:LogSubscriptionFilter
 *     name: logging
 *     properties:
 *       destinationArn: ${loggingFunction.arn}
 *       filterPattern:
 *       logGroup: ${default.name}
 *       name: logging_default
 *     options:
 *       dependson:
 *         - ${logging}
 *   loggingFunction:
 *     type: aws:lambda:Function
 *     name: logging
 *     properties:
 *       code:
 *         fn::FileArchive: lamba_logging.zip
 *       name: lambda_called_from_cloudwatch_logs
 *       handler: exports.handler
 *       role: ${defaultRole.arn}
 *       runtime: python3.7
 *   defaultRole:
 *     type: aws:iam:Role
 *     name: default
 *     properties:
 *       name: iam_for_lambda_called_from_cloudwatch_logs
 *       assumeRolePolicy: ${assumeRole.json}
 * variables:
 *   assumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - lambda.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Cross-Account Invocation Policy
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const url = new aws.lambda.FunctionUrl("url", {
 *     functionName: example.functionName,
 *     authorizationType: "AWS_IAM",
 * });
 * const urlPermission = new aws.lambda.Permission("url", {
 *     action: "lambda:InvokeFunctionUrl",
 *     "function": example.functionName,
 *     principal: "arn:aws:iam::444455556666:role/example",
 *     sourceAccount: "444455556666",
 *     functionUrlAuthType: "AWS_IAM",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * url = aws.lambda_.FunctionUrl("url",
 *     function_name=example["functionName"],
 *     authorization_type="AWS_IAM")
 * url_permission = aws.lambda_.Permission("url",
 *     action="lambda:InvokeFunctionUrl",
 *     function=example["functionName"],
 *     principal="arn:aws:iam::444455556666:role/example",
 *     source_account="444455556666",
 *     function_url_auth_type="AWS_IAM")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var url = new Aws.Lambda.FunctionUrl("url", new()
 *     {
 *         FunctionName = example.FunctionName,
 *         AuthorizationType = "AWS_IAM",
 *     });
 *     var urlPermission = new Aws.Lambda.Permission("url", new()
 *     {
 *         Action = "lambda:InvokeFunctionUrl",
 *         Function = example.FunctionName,
 *         Principal = "arn:aws:iam::444455556666:role/example",
 *         SourceAccount = "444455556666",
 *         FunctionUrlAuthType = "AWS_IAM",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lambda"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := lambda.NewFunctionUrl(ctx, "url", &lambda.FunctionUrlArgs{
 * 			FunctionName:      pulumi.Any(example.FunctionName),
 * 			AuthorizationType: pulumi.String("AWS_IAM"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lambda.NewPermission(ctx, "url", &lambda.PermissionArgs{
 * 			Action:              pulumi.String("lambda:InvokeFunctionUrl"),
 * 			Function:            pulumi.Any(example.FunctionName),
 * 			Principal:           pulumi.String("arn:aws:iam::444455556666:role/example"),
 * 			SourceAccount:       pulumi.String("444455556666"),
 * 			FunctionUrlAuthType: pulumi.String("AWS_IAM"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lambda.FunctionUrl;
 * import com.pulumi.aws.lambda.FunctionUrlArgs;
 * import com.pulumi.aws.lambda.Permission;
 * import com.pulumi.aws.lambda.PermissionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var url = new FunctionUrl("url", FunctionUrlArgs.builder()
 *             .functionName(example.functionName())
 *             .authorizationType("AWS_IAM")
 *             .build());
 *         var urlPermission = new Permission("urlPermission", PermissionArgs.builder()
 *             .action("lambda:InvokeFunctionUrl")
 *             .function(example.functionName())
 *             .principal("arn:aws:iam::444455556666:role/example")
 *             .sourceAccount("444455556666")
 *             .functionUrlAuthType("AWS_IAM")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   url:
 *     type: aws:lambda:FunctionUrl
 *     properties:
 *       functionName: ${example.functionName}
 *       authorizationType: AWS_IAM
 *   urlPermission:
 *     type: aws:lambda:Permission
 *     name: url
 *     properties:
 *       action: lambda:InvokeFunctionUrl
 *       function: ${example.functionName}
 *       principal: arn:aws:iam::444455556666:role/example
 *       sourceAccount: '444455556666'
 *       functionUrlAuthType: AWS_IAM
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Lambda permission statements using function_name/statement_id with an optional qualifier. For example:
 * ```sh
 * $ pulumi import aws:lambda/permission:Permission test_lambda_permission my_test_lambda_function/AllowExecutionFromCloudWatch
 * ```
 * ```sh
 * $ pulumi import aws:lambda/permission:Permission test_lambda_permission my_test_lambda_function:qualifier_name/AllowExecutionFromCloudWatch
 * ```
 * @property action The AWS Lambda action you want to allow in this statement. (e.g., `lambda:InvokeFunction`)
 * @property eventSourceToken The Event Source Token to validate.  Used with [Alexa Skills](https://developer.amazon.com/docs/custom-skills/host-a-custom-skill-as-an-aws-lambda-function.html#use-aws-cli).
 * @property function Name of the Lambda function whose resource policy you are updating
 * @property functionUrlAuthType Lambda Function URLs [authentication type](https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html). Valid values are: `AWS_IAM` or `NONE`. Only supported for `lambda:InvokeFunctionUrl` action.
 * @property principal The principal who is getting this permission e.g., `s3.amazonaws.com`, an AWS account ID, or AWS IAM principal, or AWS service principal such as `events.amazonaws.com` or `sns.amazonaws.com`.
 * @property principalOrgId The identifier for your organization in AWS Organizations. Use this to grant permissions to all the AWS accounts under this organization.
 * [1]: https://developer.amazon.com/docs/custom-skills/host-a-custom-skill-as-an-aws-lambda-function.html#use-aws-cli
 * [2]: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-control-access-using-iam-policies-to-invoke-api.html
 * [3]: https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html
 * @property qualifier Query parameter to specify function version or alias name. The permission will then apply to the specific qualified ARN e.g., `arn:aws:lambda:aws-region:acct-id:function:function-name:2`
 * @property sourceAccount This parameter is used when allowing cross-account access, or for S3 and SES. The AWS account ID (without a hyphen) of the source owner.
 * @property sourceArn When the principal is an AWS service, the ARN of the specific resource within that service to grant permission to.
 * Without this, any resource from `principal` will be granted permission – even if that resource is from another account.
 * For S3, this should be the ARN of the S3 Bucket.
 * For EventBridge events, this should be the ARN of the EventBridge Rule.
 * For API Gateway, this should be the ARN of the API, as described [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-control-access-using-iam-policies-to-invoke-api.html).
 * @property statementId A unique statement identifier. By default generated by the provider.
 * @property statementIdPrefix A statement identifier prefix. The provider will generate a unique suffix. Conflicts with `statement_id`.
 * */*/*/*/*/*/
 */
public data class PermissionArgs(
    public val action: Output<String>? = null,
    public val eventSourceToken: Output<String>? = null,
    public val function: Output<String>? = null,
    public val functionUrlAuthType: Output<String>? = null,
    public val principal: Output<String>? = null,
    public val principalOrgId: Output<String>? = null,
    public val qualifier: Output<String>? = null,
    public val sourceAccount: Output<String>? = null,
    public val sourceArn: Output<String>? = null,
    public val statementId: Output<String>? = null,
    public val statementIdPrefix: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.lambda.PermissionArgs> {
    override fun toJava(): com.pulumi.aws.lambda.PermissionArgs =
        com.pulumi.aws.lambda.PermissionArgs.builder()
            .action(action?.applyValue({ args0 -> args0 }))
            .eventSourceToken(eventSourceToken?.applyValue({ args0 -> args0 }))
            .function(function?.applyValue({ args0 -> args0 }))
            .functionUrlAuthType(functionUrlAuthType?.applyValue({ args0 -> args0 }))
            .principal(principal?.applyValue({ args0 -> args0 }))
            .principalOrgId(principalOrgId?.applyValue({ args0 -> args0 }))
            .qualifier(qualifier?.applyValue({ args0 -> args0 }))
            .sourceAccount(sourceAccount?.applyValue({ args0 -> args0 }))
            .sourceArn(sourceArn?.applyValue({ args0 -> args0 }))
            .statementId(statementId?.applyValue({ args0 -> args0 }))
            .statementIdPrefix(statementIdPrefix?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [PermissionArgs].
 */
@PulumiTagMarker
public class PermissionArgsBuilder internal constructor() {
    private var action: Output<String>? = null

    private var eventSourceToken: Output<String>? = null

    private var function: Output<String>? = null

    private var functionUrlAuthType: Output<String>? = null

    private var principal: Output<String>? = null

    private var principalOrgId: Output<String>? = null

    private var qualifier: Output<String>? = null

    private var sourceAccount: Output<String>? = null

    private var sourceArn: Output<String>? = null

    private var statementId: Output<String>? = null

    private var statementIdPrefix: Output<String>? = null

    /**
     * @param value The AWS Lambda action you want to allow in this statement. (e.g., `lambda:InvokeFunction`)
     */
    @JvmName("lgmeutfdusocivkj")
    public suspend fun action(`value`: Output<String>) {
        this.action = value
    }

    /**
     * @param value The Event Source Token to validate.  Used with [Alexa Skills](https://developer.amazon.com/docs/custom-skills/host-a-custom-skill-as-an-aws-lambda-function.html#use-aws-cli).
     */
    @JvmName("egjxfksrtbravedt")
    public suspend fun eventSourceToken(`value`: Output<String>) {
        this.eventSourceToken = value
    }

    /**
     * @param value Name of the Lambda function whose resource policy you are updating
     */
    @JvmName("chblrmxvhknalqwk")
    public suspend fun function(`value`: Output<String>) {
        this.function = value
    }

    /**
     * @param value Lambda Function URLs [authentication type](https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html). Valid values are: `AWS_IAM` or `NONE`. Only supported for `lambda:InvokeFunctionUrl` action.
     */
    @JvmName("pphkkudxuqacwcql")
    public suspend fun functionUrlAuthType(`value`: Output<String>) {
        this.functionUrlAuthType = value
    }

    /**
     * @param value The principal who is getting this permission e.g., `s3.amazonaws.com`, an AWS account ID, or AWS IAM principal, or AWS service principal such as `events.amazonaws.com` or `sns.amazonaws.com`.
     */
    @JvmName("wiuuydxeucrdalsq")
    public suspend fun principal(`value`: Output<String>) {
        this.principal = value
    }

    /**
     * @param value The identifier for your organization in AWS Organizations. Use this to grant permissions to all the AWS accounts under this organization.
     * [1]: https://developer.amazon.com/docs/custom-skills/host-a-custom-skill-as-an-aws-lambda-function.html#use-aws-cli
     * [2]: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-control-access-using-iam-policies-to-invoke-api.html
     * [3]: https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html
     */
    @JvmName("rnbkgaaccaatdpdd")
    public suspend fun principalOrgId(`value`: Output<String>) {
        this.principalOrgId = value
    }

    /**
     * @param value Query parameter to specify function version or alias name. The permission will then apply to the specific qualified ARN e.g., `arn:aws:lambda:aws-region:acct-id:function:function-name:2`
     */
    @JvmName("rqsbkbkfxrxeqcdw")
    public suspend fun qualifier(`value`: Output<String>) {
        this.qualifier = value
    }

    /**
     * @param value This parameter is used when allowing cross-account access, or for S3 and SES. The AWS account ID (without a hyphen) of the source owner.
     */
    @JvmName("nyydhmnqugnybata")
    public suspend fun sourceAccount(`value`: Output<String>) {
        this.sourceAccount = value
    }

    /**
     * @param value When the principal is an AWS service, the ARN of the specific resource within that service to grant permission to.
     * Without this, any resource from `principal` will be granted permission – even if that resource is from another account.
     * For S3, this should be the ARN of the S3 Bucket.
     * For EventBridge events, this should be the ARN of the EventBridge Rule.
     * For API Gateway, this should be the ARN of the API, as described [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-control-access-using-iam-policies-to-invoke-api.html).
     */
    @JvmName("qhrbyjjsfvbruefo")
    public suspend fun sourceArn(`value`: Output<String>) {
        this.sourceArn = value
    }

    /**
     * @param value A unique statement identifier. By default generated by the provider.
     */
    @JvmName("jfagolxdlptwavku")
    public suspend fun statementId(`value`: Output<String>) {
        this.statementId = value
    }

    /**
     * @param value A statement identifier prefix. The provider will generate a unique suffix. Conflicts with `statement_id`.
     */
    @JvmName("yjvcjfpaucjvjcdr")
    public suspend fun statementIdPrefix(`value`: Output<String>) {
        this.statementIdPrefix = value
    }

    /**
     * @param value The AWS Lambda action you want to allow in this statement. (e.g., `lambda:InvokeFunction`)
     */
    @JvmName("wqwwfvaqdlyykjco")
    public suspend fun action(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.action = mapped
    }

    /**
     * @param value The Event Source Token to validate.  Used with [Alexa Skills](https://developer.amazon.com/docs/custom-skills/host-a-custom-skill-as-an-aws-lambda-function.html#use-aws-cli).
     */
    @JvmName("sdvbhlfmucclscup")
    public suspend fun eventSourceToken(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.eventSourceToken = mapped
    }

    /**
     * @param value Name of the Lambda function whose resource policy you are updating
     */
    @JvmName("hfstpfrpdukqmgpl")
    public suspend fun function(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.function = mapped
    }

    /**
     * @param value Lambda Function URLs [authentication type](https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html). Valid values are: `AWS_IAM` or `NONE`. Only supported for `lambda:InvokeFunctionUrl` action.
     */
    @JvmName("pklogvfaemprxbqn")
    public suspend fun functionUrlAuthType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.functionUrlAuthType = mapped
    }

    /**
     * @param value The principal who is getting this permission e.g., `s3.amazonaws.com`, an AWS account ID, or AWS IAM principal, or AWS service principal such as `events.amazonaws.com` or `sns.amazonaws.com`.
     */
    @JvmName("xjgjcuqlaldwfshb")
    public suspend fun principal(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.principal = mapped
    }

    /**
     * @param value The identifier for your organization in AWS Organizations. Use this to grant permissions to all the AWS accounts under this organization.
     * [1]: https://developer.amazon.com/docs/custom-skills/host-a-custom-skill-as-an-aws-lambda-function.html#use-aws-cli
     * [2]: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-control-access-using-iam-policies-to-invoke-api.html
     * [3]: https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html
     */
    @JvmName("qodkmvabueitkxsd")
    public suspend fun principalOrgId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.principalOrgId = mapped
    }

    /**
     * @param value Query parameter to specify function version or alias name. The permission will then apply to the specific qualified ARN e.g., `arn:aws:lambda:aws-region:acct-id:function:function-name:2`
     */
    @JvmName("cdstyelqsqkkngwj")
    public suspend fun qualifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.qualifier = mapped
    }

    /**
     * @param value This parameter is used when allowing cross-account access, or for S3 and SES. The AWS account ID (without a hyphen) of the source owner.
     */
    @JvmName("iqsytxgmycdorwqm")
    public suspend fun sourceAccount(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceAccount = mapped
    }

    /**
     * @param value When the principal is an AWS service, the ARN of the specific resource within that service to grant permission to.
     * Without this, any resource from `principal` will be granted permission – even if that resource is from another account.
     * For S3, this should be the ARN of the S3 Bucket.
     * For EventBridge events, this should be the ARN of the EventBridge Rule.
     * For API Gateway, this should be the ARN of the API, as described [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-control-access-using-iam-policies-to-invoke-api.html).
     */
    @JvmName("febhirhbplxkcksp")
    public suspend fun sourceArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceArn = mapped
    }

    /**
     * @param value A unique statement identifier. By default generated by the provider.
     */
    @JvmName("qlduapokdfbkkklx")
    public suspend fun statementId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.statementId = mapped
    }

    /**
     * @param value A statement identifier prefix. The provider will generate a unique suffix. Conflicts with `statement_id`.
     */
    @JvmName("ysmprjheclnshjrr")
    public suspend fun statementIdPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.statementIdPrefix = mapped
    }

    internal fun build(): PermissionArgs = PermissionArgs(
        action = action,
        eventSourceToken = eventSourceToken,
        function = function,
        functionUrlAuthType = functionUrlAuthType,
        principal = principal,
        principalOrgId = principalOrgId,
        qualifier = qualifier,
        sourceAccount = sourceAccount,
        sourceArn = sourceArn,
        statementId = statementId,
        statementIdPrefix = statementIdPrefix,
    )
}
