@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.lb.kotlin

import com.pulumi.aws.lb.ListenerArgs.builder
import com.pulumi.aws.lb.kotlin.inputs.ListenerDefaultActionArgs
import com.pulumi.aws.lb.kotlin.inputs.ListenerDefaultActionArgsBuilder
import com.pulumi.aws.lb.kotlin.inputs.ListenerMutualAuthenticationArgs
import com.pulumi.aws.lb.kotlin.inputs.ListenerMutualAuthenticationArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a Load Balancer Listener resource.
 * > **Note:** `aws.alb.Listener` is known as `aws.lb.Listener`. The functionality is identical.
 * ## Example Usage
 * ### Forward Action
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const frontEnd = new aws.lb.LoadBalancer("front_end", {});
 * const frontEndTargetGroup = new aws.lb.TargetGroup("front_end", {});
 * const frontEndListener = new aws.lb.Listener("front_end", {
 *     loadBalancerArn: frontEnd.arn,
 *     port: 443,
 *     protocol: "HTTPS",
 *     sslPolicy: "ELBSecurityPolicy-2016-08",
 *     certificateArn: "arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4",
 *     defaultActions: [{
 *         type: "forward",
 *         targetGroupArn: frontEndTargetGroup.arn,
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * front_end = aws.lb.LoadBalancer("front_end")
 * front_end_target_group = aws.lb.TargetGroup("front_end")
 * front_end_listener = aws.lb.Listener("front_end",
 *     load_balancer_arn=front_end.arn,
 *     port=443,
 *     protocol="HTTPS",
 *     ssl_policy="ELBSecurityPolicy-2016-08",
 *     certificate_arn="arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4",
 *     default_actions=[{
 *         "type": "forward",
 *         "target_group_arn": front_end_target_group.arn,
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var frontEnd = new Aws.LB.LoadBalancer("front_end");
 *     var frontEndTargetGroup = new Aws.LB.TargetGroup("front_end");
 *     var frontEndListener = new Aws.LB.Listener("front_end", new()
 *     {
 *         LoadBalancerArn = frontEnd.Arn,
 *         Port = 443,
 *         Protocol = "HTTPS",
 *         SslPolicy = "ELBSecurityPolicy-2016-08",
 *         CertificateArn = "arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4",
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = frontEndTargetGroup.Arn,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		frontEnd, err := lb.NewLoadBalancer(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		frontEndTargetGroup, err := lb.NewTargetGroup(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListener(ctx, "front_end", &lb.ListenerArgs{
 * 			LoadBalancerArn: frontEnd.Arn,
 * 			Port:            pulumi.Int(443),
 * 			Protocol:        pulumi.String("HTTPS"),
 * 			SslPolicy:       pulumi.String("ELBSecurityPolicy-2016-08"),
 * 			CertificateArn:  pulumi.String("arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4"),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: frontEndTargetGroup.Arn,
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.TargetGroup;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var frontEnd = new LoadBalancer("frontEnd");
 *         var frontEndTargetGroup = new TargetGroup("frontEndTargetGroup");
 *         var frontEndListener = new Listener("frontEndListener", ListenerArgs.builder()
 *             .loadBalancerArn(frontEnd.arn())
 *             .port("443")
 *             .protocol("HTTPS")
 *             .sslPolicy("ELBSecurityPolicy-2016-08")
 *             .certificateArn("arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4")
 *             .defaultActions(ListenerDefaultActionArgs.builder()
 *                 .type("forward")
 *                 .targetGroupArn(frontEndTargetGroup.arn())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   frontEnd:
 *     type: aws:lb:LoadBalancer
 *     name: front_end
 *   frontEndTargetGroup:
 *     type: aws:lb:TargetGroup
 *     name: front_end
 *   frontEndListener:
 *     type: aws:lb:Listener
 *     name: front_end
 *     properties:
 *       loadBalancerArn: ${frontEnd.arn}
 *       port: '443'
 *       protocol: HTTPS
 *       sslPolicy: ELBSecurityPolicy-2016-08
 *       certificateArn: arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4
 *       defaultActions:
 *         - type: forward
 *           targetGroupArn: ${frontEndTargetGroup.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * To a NLB:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const frontEnd = new aws.lb.Listener("front_end", {
 *     loadBalancerArn: frontEndAwsLb.arn,
 *     port: 443,
 *     protocol: "TLS",
 *     certificateArn: "arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4",
 *     alpnPolicy: "HTTP2Preferred",
 *     defaultActions: [{
 *         type: "forward",
 *         targetGroupArn: frontEndAwsLbTargetGroup.arn,
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * front_end = aws.lb.Listener("front_end",
 *     load_balancer_arn=front_end_aws_lb["arn"],
 *     port=443,
 *     protocol="TLS",
 *     certificate_arn="arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4",
 *     alpn_policy="HTTP2Preferred",
 *     default_actions=[{
 *         "type": "forward",
 *         "target_group_arn": front_end_aws_lb_target_group["arn"],
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var frontEnd = new Aws.LB.Listener("front_end", new()
 *     {
 *         LoadBalancerArn = frontEndAwsLb.Arn,
 *         Port = 443,
 *         Protocol = "TLS",
 *         CertificateArn = "arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4",
 *         AlpnPolicy = "HTTP2Preferred",
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = frontEndAwsLbTargetGroup.Arn,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := lb.NewListener(ctx, "front_end", &lb.ListenerArgs{
 * 			LoadBalancerArn: pulumi.Any(frontEndAwsLb.Arn),
 * 			Port:            pulumi.Int(443),
 * 			Protocol:        pulumi.String("TLS"),
 * 			CertificateArn:  pulumi.String("arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4"),
 * 			AlpnPolicy:      pulumi.String("HTTP2Preferred"),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: pulumi.Any(frontEndAwsLbTargetGroup.Arn),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var frontEnd = new Listener("frontEnd", ListenerArgs.builder()
 *             .loadBalancerArn(frontEndAwsLb.arn())
 *             .port("443")
 *             .protocol("TLS")
 *             .certificateArn("arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4")
 *             .alpnPolicy("HTTP2Preferred")
 *             .defaultActions(ListenerDefaultActionArgs.builder()
 *                 .type("forward")
 *                 .targetGroupArn(frontEndAwsLbTargetGroup.arn())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   frontEnd:
 *     type: aws:lb:Listener
 *     name: front_end
 *     properties:
 *       loadBalancerArn: ${frontEndAwsLb.arn}
 *       port: '443'
 *       protocol: TLS
 *       certificateArn: arn:aws:iam::187416307283:server-certificate/test_cert_rab3wuqwgja25ct3n4jdj2tzu4
 *       alpnPolicy: HTTP2Preferred
 *       defaultActions:
 *         - type: forward
 *           targetGroupArn: ${frontEndAwsLbTargetGroup.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Redirect Action
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const frontEnd = new aws.lb.LoadBalancer("front_end", {});
 * const frontEndListener = new aws.lb.Listener("front_end", {
 *     loadBalancerArn: frontEnd.arn,
 *     port: 80,
 *     protocol: "HTTP",
 *     defaultActions: [{
 *         type: "redirect",
 *         redirect: {
 *             port: "443",
 *             protocol: "HTTPS",
 *             statusCode: "HTTP_301",
 *         },
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * front_end = aws.lb.LoadBalancer("front_end")
 * front_end_listener = aws.lb.Listener("front_end",
 *     load_balancer_arn=front_end.arn,
 *     port=80,
 *     protocol="HTTP",
 *     default_actions=[{
 *         "type": "redirect",
 *         "redirect": {
 *             "port": "443",
 *             "protocol": "HTTPS",
 *             "status_code": "HTTP_301",
 *         },
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var frontEnd = new Aws.LB.LoadBalancer("front_end");
 *     var frontEndListener = new Aws.LB.Listener("front_end", new()
 *     {
 *         LoadBalancerArn = frontEnd.Arn,
 *         Port = 80,
 *         Protocol = "HTTP",
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "redirect",
 *                 Redirect = new Aws.LB.Inputs.ListenerDefaultActionRedirectArgs
 *                 {
 *                     Port = "443",
 *                     Protocol = "HTTPS",
 *                     StatusCode = "HTTP_301",
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		frontEnd, err := lb.NewLoadBalancer(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListener(ctx, "front_end", &lb.ListenerArgs{
 * 			LoadBalancerArn: frontEnd.Arn,
 * 			Port:            pulumi.Int(80),
 * 			Protocol:        pulumi.String("HTTP"),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type: pulumi.String("redirect"),
 * 					Redirect: &lb.ListenerDefaultActionRedirectArgs{
 * 						Port:       pulumi.String("443"),
 * 						Protocol:   pulumi.String("HTTPS"),
 * 						StatusCode: pulumi.String("HTTP_301"),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionRedirectArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var frontEnd = new LoadBalancer("frontEnd");
 *         var frontEndListener = new Listener("frontEndListener", ListenerArgs.builder()
 *             .loadBalancerArn(frontEnd.arn())
 *             .port("80")
 *             .protocol("HTTP")
 *             .defaultActions(ListenerDefaultActionArgs.builder()
 *                 .type("redirect")
 *                 .redirect(ListenerDefaultActionRedirectArgs.builder()
 *                     .port("443")
 *                     .protocol("HTTPS")
 *                     .statusCode("HTTP_301")
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   frontEnd:
 *     type: aws:lb:LoadBalancer
 *     name: front_end
 *   frontEndListener:
 *     type: aws:lb:Listener
 *     name: front_end
 *     properties:
 *       loadBalancerArn: ${frontEnd.arn}
 *       port: '80'
 *       protocol: HTTP
 *       defaultActions:
 *         - type: redirect
 *           redirect:
 *             port: '443'
 *             protocol: HTTPS
 *             statusCode: HTTP_301
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Fixed-response Action
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const frontEnd = new aws.lb.LoadBalancer("front_end", {});
 * const frontEndListener = new aws.lb.Listener("front_end", {
 *     loadBalancerArn: frontEnd.arn,
 *     port: 80,
 *     protocol: "HTTP",
 *     defaultActions: [{
 *         type: "fixed-response",
 *         fixedResponse: {
 *             contentType: "text/plain",
 *             messageBody: "Fixed response content",
 *             statusCode: "200",
 *         },
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * front_end = aws.lb.LoadBalancer("front_end")
 * front_end_listener = aws.lb.Listener("front_end",
 *     load_balancer_arn=front_end.arn,
 *     port=80,
 *     protocol="HTTP",
 *     default_actions=[{
 *         "type": "fixed-response",
 *         "fixed_response": {
 *             "content_type": "text/plain",
 *             "message_body": "Fixed response content",
 *             "status_code": "200",
 *         },
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var frontEnd = new Aws.LB.LoadBalancer("front_end");
 *     var frontEndListener = new Aws.LB.Listener("front_end", new()
 *     {
 *         LoadBalancerArn = frontEnd.Arn,
 *         Port = 80,
 *         Protocol = "HTTP",
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "fixed-response",
 *                 FixedResponse = new Aws.LB.Inputs.ListenerDefaultActionFixedResponseArgs
 *                 {
 *                     ContentType = "text/plain",
 *                     MessageBody = "Fixed response content",
 *                     StatusCode = "200",
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		frontEnd, err := lb.NewLoadBalancer(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListener(ctx, "front_end", &lb.ListenerArgs{
 * 			LoadBalancerArn: frontEnd.Arn,
 * 			Port:            pulumi.Int(80),
 * 			Protocol:        pulumi.String("HTTP"),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type: pulumi.String("fixed-response"),
 * 					FixedResponse: &lb.ListenerDefaultActionFixedResponseArgs{
 * 						ContentType: pulumi.String("text/plain"),
 * 						MessageBody: pulumi.String("Fixed response content"),
 * 						StatusCode:  pulumi.String("200"),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionFixedResponseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var frontEnd = new LoadBalancer("frontEnd");
 *         var frontEndListener = new Listener("frontEndListener", ListenerArgs.builder()
 *             .loadBalancerArn(frontEnd.arn())
 *             .port("80")
 *             .protocol("HTTP")
 *             .defaultActions(ListenerDefaultActionArgs.builder()
 *                 .type("fixed-response")
 *                 .fixedResponse(ListenerDefaultActionFixedResponseArgs.builder()
 *                     .contentType("text/plain")
 *                     .messageBody("Fixed response content")
 *                     .statusCode("200")
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   frontEnd:
 *     type: aws:lb:LoadBalancer
 *     name: front_end
 *   frontEndListener:
 *     type: aws:lb:Listener
 *     name: front_end
 *     properties:
 *       loadBalancerArn: ${frontEnd.arn}
 *       port: '80'
 *       protocol: HTTP
 *       defaultActions:
 *         - type: fixed-response
 *           fixedResponse:
 *             contentType: text/plain
 *             messageBody: Fixed response content
 *             statusCode: '200'
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Authenticate-cognito Action
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const frontEnd = new aws.lb.LoadBalancer("front_end", {});
 * const frontEndTargetGroup = new aws.lb.TargetGroup("front_end", {});
 * const pool = new aws.cognito.UserPool("pool", {});
 * const client = new aws.cognito.UserPoolClient("client", {});
 * const domain = new aws.cognito.UserPoolDomain("domain", {});
 * const frontEndListener = new aws.lb.Listener("front_end", {
 *     loadBalancerArn: frontEnd.arn,
 *     port: 80,
 *     protocol: "HTTP",
 *     defaultActions: [
 *         {
 *             type: "authenticate-cognito",
 *             authenticateCognito: {
 *                 userPoolArn: pool.arn,
 *                 userPoolClientId: client.id,
 *                 userPoolDomain: domain.domain,
 *             },
 *         },
 *         {
 *             type: "forward",
 *             targetGroupArn: frontEndTargetGroup.arn,
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * front_end = aws.lb.LoadBalancer("front_end")
 * front_end_target_group = aws.lb.TargetGroup("front_end")
 * pool = aws.cognito.UserPool("pool")
 * client = aws.cognito.UserPoolClient("client")
 * domain = aws.cognito.UserPoolDomain("domain")
 * front_end_listener = aws.lb.Listener("front_end",
 *     load_balancer_arn=front_end.arn,
 *     port=80,
 *     protocol="HTTP",
 *     default_actions=[
 *         {
 *             "type": "authenticate-cognito",
 *             "authenticate_cognito": {
 *                 "user_pool_arn": pool.arn,
 *                 "user_pool_client_id": client.id,
 *                 "user_pool_domain": domain.domain,
 *             },
 *         },
 *         {
 *             "type": "forward",
 *             "target_group_arn": front_end_target_group.arn,
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var frontEnd = new Aws.LB.LoadBalancer("front_end");
 *     var frontEndTargetGroup = new Aws.LB.TargetGroup("front_end");
 *     var pool = new Aws.Cognito.UserPool("pool");
 *     var client = new Aws.Cognito.UserPoolClient("client");
 *     var domain = new Aws.Cognito.UserPoolDomain("domain");
 *     var frontEndListener = new Aws.LB.Listener("front_end", new()
 *     {
 *         LoadBalancerArn = frontEnd.Arn,
 *         Port = 80,
 *         Protocol = "HTTP",
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "authenticate-cognito",
 *                 AuthenticateCognito = new Aws.LB.Inputs.ListenerDefaultActionAuthenticateCognitoArgs
 *                 {
 *                     UserPoolArn = pool.Arn,
 *                     UserPoolClientId = client.Id,
 *                     UserPoolDomain = domain.Domain,
 *                 },
 *             },
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = frontEndTargetGroup.Arn,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cognito"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		frontEnd, err := lb.NewLoadBalancer(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		frontEndTargetGroup, err := lb.NewTargetGroup(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		pool, err := cognito.NewUserPool(ctx, "pool", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		client, err := cognito.NewUserPoolClient(ctx, "client", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		domain, err := cognito.NewUserPoolDomain(ctx, "domain", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListener(ctx, "front_end", &lb.ListenerArgs{
 * 			LoadBalancerArn: frontEnd.Arn,
 * 			Port:            pulumi.Int(80),
 * 			Protocol:        pulumi.String("HTTP"),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type: pulumi.String("authenticate-cognito"),
 * 					AuthenticateCognito: &lb.ListenerDefaultActionAuthenticateCognitoArgs{
 * 						UserPoolArn:      pool.Arn,
 * 						UserPoolClientId: client.ID(),
 * 						UserPoolDomain:   domain.Domain,
 * 					},
 * 				},
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: frontEndTargetGroup.Arn,
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.TargetGroup;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolDomain;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionAuthenticateCognitoArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var frontEnd = new LoadBalancer("frontEnd");
 *         var frontEndTargetGroup = new TargetGroup("frontEndTargetGroup");
 *         var pool = new UserPool("pool");
 *         var client = new UserPoolClient("client");
 *         var domain = new UserPoolDomain("domain");
 *         var frontEndListener = new Listener("frontEndListener", ListenerArgs.builder()
 *             .loadBalancerArn(frontEnd.arn())
 *             .port("80")
 *             .protocol("HTTP")
 *             .defaultActions(
 *                 ListenerDefaultActionArgs.builder()
 *                     .type("authenticate-cognito")
 *                     .authenticateCognito(ListenerDefaultActionAuthenticateCognitoArgs.builder()
 *                         .userPoolArn(pool.arn())
 *                         .userPoolClientId(client.id())
 *                         .userPoolDomain(domain.domain())
 *                         .build())
 *                     .build(),
 *                 ListenerDefaultActionArgs.builder()
 *                     .type("forward")
 *                     .targetGroupArn(frontEndTargetGroup.arn())
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   frontEnd:
 *     type: aws:lb:LoadBalancer
 *     name: front_end
 *   frontEndTargetGroup:
 *     type: aws:lb:TargetGroup
 *     name: front_end
 *   pool:
 *     type: aws:cognito:UserPool
 *   client:
 *     type: aws:cognito:UserPoolClient
 *   domain:
 *     type: aws:cognito:UserPoolDomain
 *   frontEndListener:
 *     type: aws:lb:Listener
 *     name: front_end
 *     properties:
 *       loadBalancerArn: ${frontEnd.arn}
 *       port: '80'
 *       protocol: HTTP
 *       defaultActions:
 *         - type: authenticate-cognito
 *           authenticateCognito:
 *             userPoolArn: ${pool.arn}
 *             userPoolClientId: ${client.id}
 *             userPoolDomain: ${domain.domain}
 *         - type: forward
 *           targetGroupArn: ${frontEndTargetGroup.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Authenticate-OIDC Action
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const frontEnd = new aws.lb.LoadBalancer("front_end", {});
 * const frontEndTargetGroup = new aws.lb.TargetGroup("front_end", {});
 * const frontEndListener = new aws.lb.Listener("front_end", {
 *     loadBalancerArn: frontEnd.arn,
 *     port: 80,
 *     protocol: "HTTP",
 *     defaultActions: [
 *         {
 *             type: "authenticate-oidc",
 *             authenticateOidc: {
 *                 authorizationEndpoint: "https://example.com/authorization_endpoint",
 *                 clientId: "client_id",
 *                 clientSecret: "client_secret",
 *                 issuer: "https://example.com",
 *                 tokenEndpoint: "https://example.com/token_endpoint",
 *                 userInfoEndpoint: "https://example.com/user_info_endpoint",
 *             },
 *         },
 *         {
 *             type: "forward",
 *             targetGroupArn: frontEndTargetGroup.arn,
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * front_end = aws.lb.LoadBalancer("front_end")
 * front_end_target_group = aws.lb.TargetGroup("front_end")
 * front_end_listener = aws.lb.Listener("front_end",
 *     load_balancer_arn=front_end.arn,
 *     port=80,
 *     protocol="HTTP",
 *     default_actions=[
 *         {
 *             "type": "authenticate-oidc",
 *             "authenticate_oidc": {
 *                 "authorization_endpoint": "https://example.com/authorization_endpoint",
 *                 "client_id": "client_id",
 *                 "client_secret": "client_secret",
 *                 "issuer": "https://example.com",
 *                 "token_endpoint": "https://example.com/token_endpoint",
 *                 "user_info_endpoint": "https://example.com/user_info_endpoint",
 *             },
 *         },
 *         {
 *             "type": "forward",
 *             "target_group_arn": front_end_target_group.arn,
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var frontEnd = new Aws.LB.LoadBalancer("front_end");
 *     var frontEndTargetGroup = new Aws.LB.TargetGroup("front_end");
 *     var frontEndListener = new Aws.LB.Listener("front_end", new()
 *     {
 *         LoadBalancerArn = frontEnd.Arn,
 *         Port = 80,
 *         Protocol = "HTTP",
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "authenticate-oidc",
 *                 AuthenticateOidc = new Aws.LB.Inputs.ListenerDefaultActionAuthenticateOidcArgs
 *                 {
 *                     AuthorizationEndpoint = "https://example.com/authorization_endpoint",
 *                     ClientId = "client_id",
 *                     ClientSecret = "client_secret",
 *                     Issuer = "https://example.com",
 *                     TokenEndpoint = "https://example.com/token_endpoint",
 *                     UserInfoEndpoint = "https://example.com/user_info_endpoint",
 *                 },
 *             },
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = frontEndTargetGroup.Arn,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		frontEnd, err := lb.NewLoadBalancer(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		frontEndTargetGroup, err := lb.NewTargetGroup(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListener(ctx, "front_end", &lb.ListenerArgs{
 * 			LoadBalancerArn: frontEnd.Arn,
 * 			Port:            pulumi.Int(80),
 * 			Protocol:        pulumi.String("HTTP"),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type: pulumi.String("authenticate-oidc"),
 * 					AuthenticateOidc: &lb.ListenerDefaultActionAuthenticateOidcArgs{
 * 						AuthorizationEndpoint: pulumi.String("https://example.com/authorization_endpoint"),
 * 						ClientId:              pulumi.String("client_id"),
 * 						ClientSecret:          pulumi.String("client_secret"),
 * 						Issuer:                pulumi.String("https://example.com"),
 * 						TokenEndpoint:         pulumi.String("https://example.com/token_endpoint"),
 * 						UserInfoEndpoint:      pulumi.String("https://example.com/user_info_endpoint"),
 * 					},
 * 				},
 * 				&lb.ListenerDefaultActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: frontEndTargetGroup.Arn,
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.TargetGroup;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionAuthenticateOidcArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var frontEnd = new LoadBalancer("frontEnd");
 *         var frontEndTargetGroup = new TargetGroup("frontEndTargetGroup");
 *         var frontEndListener = new Listener("frontEndListener", ListenerArgs.builder()
 *             .loadBalancerArn(frontEnd.arn())
 *             .port("80")
 *             .protocol("HTTP")
 *             .defaultActions(
 *                 ListenerDefaultActionArgs.builder()
 *                     .type("authenticate-oidc")
 *                     .authenticateOidc(ListenerDefaultActionAuthenticateOidcArgs.builder()
 *                         .authorizationEndpoint("https://example.com/authorization_endpoint")
 *                         .clientId("client_id")
 *                         .clientSecret("client_secret")
 *                         .issuer("https://example.com")
 *                         .tokenEndpoint("https://example.com/token_endpoint")
 *                         .userInfoEndpoint("https://example.com/user_info_endpoint")
 *                         .build())
 *                     .build(),
 *                 ListenerDefaultActionArgs.builder()
 *                     .type("forward")
 *                     .targetGroupArn(frontEndTargetGroup.arn())
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   frontEnd:
 *     type: aws:lb:LoadBalancer
 *     name: front_end
 *   frontEndTargetGroup:
 *     type: aws:lb:TargetGroup
 *     name: front_end
 *   frontEndListener:
 *     type: aws:lb:Listener
 *     name: front_end
 *     properties:
 *       loadBalancerArn: ${frontEnd.arn}
 *       port: '80'
 *       protocol: HTTP
 *       defaultActions:
 *         - type: authenticate-oidc
 *           authenticateOidc:
 *             authorizationEndpoint: https://example.com/authorization_endpoint
 *             clientId: client_id
 *             clientSecret: client_secret
 *             issuer: https://example.com
 *             tokenEndpoint: https://example.com/token_endpoint
 *             userInfoEndpoint: https://example.com/user_info_endpoint
 *         - type: forward
 *           targetGroupArn: ${frontEndTargetGroup.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Gateway Load Balancer Listener
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.lb.LoadBalancer("example", {
 *     loadBalancerType: "gateway",
 *     name: "example",
 *     subnetMappings: [{
 *         subnetId: exampleAwsSubnet.id,
 *     }],
 * });
 * const exampleTargetGroup = new aws.lb.TargetGroup("example", {
 *     name: "example",
 *     port: 6081,
 *     protocol: "GENEVE",
 *     vpcId: exampleAwsVpc.id,
 *     healthCheck: {
 *         port: "80",
 *         protocol: "HTTP",
 *     },
 * });
 * const exampleListener = new aws.lb.Listener("example", {
 *     loadBalancerArn: example.id,
 *     defaultActions: [{
 *         targetGroupArn: exampleTargetGroup.id,
 *         type: "forward",
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.lb.LoadBalancer("example",
 *     load_balancer_type="gateway",
 *     name="example",
 *     subnet_mappings=[{
 *         "subnet_id": example_aws_subnet["id"],
 *     }])
 * example_target_group = aws.lb.TargetGroup("example",
 *     name="example",
 *     port=6081,
 *     protocol="GENEVE",
 *     vpc_id=example_aws_vpc["id"],
 *     health_check={
 *         "port": "80",
 *         "protocol": "HTTP",
 *     })
 * example_listener = aws.lb.Listener("example",
 *     load_balancer_arn=example.id,
 *     default_actions=[{
 *         "target_group_arn": example_target_group.id,
 *         "type": "forward",
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.LB.LoadBalancer("example", new()
 *     {
 *         LoadBalancerType = "gateway",
 *         Name = "example",
 *         SubnetMappings = new[]
 *         {
 *             new Aws.LB.Inputs.LoadBalancerSubnetMappingArgs
 *             {
 *                 SubnetId = exampleAwsSubnet.Id,
 *             },
 *         },
 *     });
 *     var exampleTargetGroup = new Aws.LB.TargetGroup("example", new()
 *     {
 *         Name = "example",
 *         Port = 6081,
 *         Protocol = "GENEVE",
 *         VpcId = exampleAwsVpc.Id,
 *         HealthCheck = new Aws.LB.Inputs.TargetGroupHealthCheckArgs
 *         {
 *             Port = "80",
 *             Protocol = "HTTP",
 *         },
 *     });
 *     var exampleListener = new Aws.LB.Listener("example", new()
 *     {
 *         LoadBalancerArn = example.Id,
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 TargetGroupArn = exampleTargetGroup.Id,
 *                 Type = "forward",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := lb.NewLoadBalancer(ctx, "example", &lb.LoadBalancerArgs{
 * 			LoadBalancerType: pulumi.String("gateway"),
 * 			Name:             pulumi.String("example"),
 * 			SubnetMappings: lb.LoadBalancerSubnetMappingArray{
 * 				&lb.LoadBalancerSubnetMappingArgs{
 * 					SubnetId: pulumi.Any(exampleAwsSubnet.Id),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleTargetGroup, err := lb.NewTargetGroup(ctx, "example", &lb.TargetGroupArgs{
 * 			Name:     pulumi.String("example"),
 * 			Port:     pulumi.Int(6081),
 * 			Protocol: pulumi.String("GENEVE"),
 * 			VpcId:    pulumi.Any(exampleAwsVpc.Id),
 * 			HealthCheck: &lb.TargetGroupHealthCheckArgs{
 * 				Port:     pulumi.String("80"),
 * 				Protocol: pulumi.String("HTTP"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListener(ctx, "example", &lb.ListenerArgs{
 * 			LoadBalancerArn: example.ID(),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					TargetGroupArn: exampleTargetGroup.ID(),
 * 					Type:           pulumi.String("forward"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.LoadBalancerArgs;
 * import com.pulumi.aws.lb.inputs.LoadBalancerSubnetMappingArgs;
 * import com.pulumi.aws.lb.TargetGroup;
 * import com.pulumi.aws.lb.TargetGroupArgs;
 * import com.pulumi.aws.lb.inputs.TargetGroupHealthCheckArgs;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new LoadBalancer("example", LoadBalancerArgs.builder()
 *             .loadBalancerType("gateway")
 *             .name("example")
 *             .subnetMappings(LoadBalancerSubnetMappingArgs.builder()
 *                 .subnetId(exampleAwsSubnet.id())
 *                 .build())
 *             .build());
 *         var exampleTargetGroup = new TargetGroup("exampleTargetGroup", TargetGroupArgs.builder()
 *             .name("example")
 *             .port(6081)
 *             .protocol("GENEVE")
 *             .vpcId(exampleAwsVpc.id())
 *             .healthCheck(TargetGroupHealthCheckArgs.builder()
 *                 .port(80)
 *                 .protocol("HTTP")
 *                 .build())
 *             .build());
 *         var exampleListener = new Listener("exampleListener", ListenerArgs.builder()
 *             .loadBalancerArn(example.id())
 *             .defaultActions(ListenerDefaultActionArgs.builder()
 *                 .targetGroupArn(exampleTargetGroup.id())
 *                 .type("forward")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:lb:LoadBalancer
 *     properties:
 *       loadBalancerType: gateway
 *       name: example
 *       subnetMappings:
 *         - subnetId: ${exampleAwsSubnet.id}
 *   exampleTargetGroup:
 *     type: aws:lb:TargetGroup
 *     name: example
 *     properties:
 *       name: example
 *       port: 6081
 *       protocol: GENEVE
 *       vpcId: ${exampleAwsVpc.id}
 *       healthCheck:
 *         port: 80
 *         protocol: HTTP
 *   exampleListener:
 *     type: aws:lb:Listener
 *     name: example
 *     properties:
 *       loadBalancerArn: ${example.id}
 *       defaultActions:
 *         - targetGroupArn: ${exampleTargetGroup.id}
 *           type: forward
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Mutual TLS Authentication
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.lb.LoadBalancer("example", {loadBalancerType: "application"});
 * const exampleTargetGroup = new aws.lb.TargetGroup("example", {});
 * const exampleListener = new aws.lb.Listener("example", {
 *     loadBalancerArn: example.id,
 *     defaultActions: [{
 *         targetGroupArn: exampleTargetGroup.id,
 *         type: "forward",
 *     }],
 *     mutualAuthentication: {
 *         mode: "verify",
 *         trustStoreArn: "...",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.lb.LoadBalancer("example", load_balancer_type="application")
 * example_target_group = aws.lb.TargetGroup("example")
 * example_listener = aws.lb.Listener("example",
 *     load_balancer_arn=example.id,
 *     default_actions=[{
 *         "target_group_arn": example_target_group.id,
 *         "type": "forward",
 *     }],
 *     mutual_authentication={
 *         "mode": "verify",
 *         "trust_store_arn": "...",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.LB.LoadBalancer("example", new()
 *     {
 *         LoadBalancerType = "application",
 *     });
 *     var exampleTargetGroup = new Aws.LB.TargetGroup("example");
 *     var exampleListener = new Aws.LB.Listener("example", new()
 *     {
 *         LoadBalancerArn = example.Id,
 *         DefaultActions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerDefaultActionArgs
 *             {
 *                 TargetGroupArn = exampleTargetGroup.Id,
 *                 Type = "forward",
 *             },
 *         },
 *         MutualAuthentication = new Aws.LB.Inputs.ListenerMutualAuthenticationArgs
 *         {
 *             Mode = "verify",
 *             TrustStoreArn = "...",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := lb.NewLoadBalancer(ctx, "example", &lb.LoadBalancerArgs{
 * 			LoadBalancerType: pulumi.String("application"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleTargetGroup, err := lb.NewTargetGroup(ctx, "example", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListener(ctx, "example", &lb.ListenerArgs{
 * 			LoadBalancerArn: example.ID(),
 * 			DefaultActions: lb.ListenerDefaultActionArray{
 * 				&lb.ListenerDefaultActionArgs{
 * 					TargetGroupArn: exampleTargetGroup.ID(),
 * 					Type:           pulumi.String("forward"),
 * 				},
 * 			},
 * 			MutualAuthentication: &lb.ListenerMutualAuthenticationArgs{
 * 				Mode:          pulumi.String("verify"),
 * 				TrustStoreArn: pulumi.String("..."),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.LoadBalancerArgs;
 * import com.pulumi.aws.lb.TargetGroup;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerArgs;
 * import com.pulumi.aws.lb.inputs.ListenerDefaultActionArgs;
 * import com.pulumi.aws.lb.inputs.ListenerMutualAuthenticationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new LoadBalancer("example", LoadBalancerArgs.builder()
 *             .loadBalancerType("application")
 *             .build());
 *         var exampleTargetGroup = new TargetGroup("exampleTargetGroup");
 *         var exampleListener = new Listener("exampleListener", ListenerArgs.builder()
 *             .loadBalancerArn(example.id())
 *             .defaultActions(ListenerDefaultActionArgs.builder()
 *                 .targetGroupArn(exampleTargetGroup.id())
 *                 .type("forward")
 *                 .build())
 *             .mutualAuthentication(ListenerMutualAuthenticationArgs.builder()
 *                 .mode("verify")
 *                 .trustStoreArn("...")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:lb:LoadBalancer
 *     properties:
 *       loadBalancerType: application
 *   exampleTargetGroup:
 *     type: aws:lb:TargetGroup
 *     name: example
 *   exampleListener:
 *     type: aws:lb:Listener
 *     name: example
 *     properties:
 *       loadBalancerArn: ${example.id}
 *       defaultActions:
 *         - targetGroupArn: ${exampleTargetGroup.id}
 *           type: forward
 *       mutualAuthentication:
 *         mode: verify
 *         trustStoreArn: '...'
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import listeners using their ARN. For example:
 * ```sh
 * $ pulumi import aws:lb/listener:Listener front_end arn:aws:elasticloadbalancing:us-west-2:187416307283:listener/app/front-end-alb/8e4497da625e2d8a/9ab28ade35828f96
 * ```
 * @property alpnPolicy Name of the Application-Layer Protocol Negotiation (ALPN) policy. Can be set if `protocol` is `TLS`. Valid values are `HTTP1Only`, `HTTP2Only`, `HTTP2Optional`, `HTTP2Preferred`, and `None`.
 * @property certificateArn ARN of the default SSL server certificate. Exactly one certificate is required if the protocol is HTTPS. For adding additional SSL certificates, see the `aws.lb.ListenerCertificate` resource.
 * @property defaultActions Configuration block for default actions. Detailed below.
 * @property loadBalancerArn ARN of the load balancer.
 * The following arguments are optional:
 * @property mutualAuthentication The mutual authentication configuration information. Detailed below.
 * @property port Port on which the load balancer is listening. Not valid for Gateway Load Balancers.
 * @property protocol Protocol for connections from clients to the load balancer. For Application Load Balancers, valid values are `HTTP` and `HTTPS`, with a default of `HTTP`. For Network Load Balancers, valid values are `TCP`, `TLS`, `UDP`, and `TCP_UDP`. Not valid to use `UDP` or `TCP_UDP` if dual-stack mode is enabled. Not valid for Gateway Load Balancers.
 * @property sslPolicy Name of the SSL Policy for the listener. Required if `protocol` is `HTTPS` or `TLS`.
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * > **NOTE::** Please note that listeners that are attached to Application Load Balancers must use either `HTTP` or `HTTPS` protocols while listeners that are attached to Network Load Balancers must use the `TCP` protocol.
 */
public data class ListenerArgs(
    public val alpnPolicy: Output<String>? = null,
    public val certificateArn: Output<String>? = null,
    public val defaultActions: Output<List<ListenerDefaultActionArgs>>? = null,
    public val loadBalancerArn: Output<String>? = null,
    public val mutualAuthentication: Output<ListenerMutualAuthenticationArgs>? = null,
    public val port: Output<Int>? = null,
    public val protocol: Output<String>? = null,
    public val sslPolicy: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.lb.ListenerArgs> {
    override fun toJava(): com.pulumi.aws.lb.ListenerArgs = com.pulumi.aws.lb.ListenerArgs.builder()
        .alpnPolicy(alpnPolicy?.applyValue({ args0 -> args0 }))
        .certificateArn(certificateArn?.applyValue({ args0 -> args0 }))
        .defaultActions(
            defaultActions?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        )
        .loadBalancerArn(loadBalancerArn?.applyValue({ args0 -> args0 }))
        .mutualAuthentication(
            mutualAuthentication?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .port(port?.applyValue({ args0 -> args0 }))
        .protocol(protocol?.applyValue({ args0 -> args0 }))
        .sslPolicy(sslPolicy?.applyValue({ args0 -> args0 }))
        .tags(
            tags?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }),
        ).build()
}

/**
 * Builder for [ListenerArgs].
 */
@PulumiTagMarker
public class ListenerArgsBuilder internal constructor() {
    private var alpnPolicy: Output<String>? = null

    private var certificateArn: Output<String>? = null

    private var defaultActions: Output<List<ListenerDefaultActionArgs>>? = null

    private var loadBalancerArn: Output<String>? = null

    private var mutualAuthentication: Output<ListenerMutualAuthenticationArgs>? = null

    private var port: Output<Int>? = null

    private var protocol: Output<String>? = null

    private var sslPolicy: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value Name of the Application-Layer Protocol Negotiation (ALPN) policy. Can be set if `protocol` is `TLS`. Valid values are `HTTP1Only`, `HTTP2Only`, `HTTP2Optional`, `HTTP2Preferred`, and `None`.
     */
    @JvmName("pppvclqfkpxvkrth")
    public suspend fun alpnPolicy(`value`: Output<String>) {
        this.alpnPolicy = value
    }

    /**
     * @param value ARN of the default SSL server certificate. Exactly one certificate is required if the protocol is HTTPS. For adding additional SSL certificates, see the `aws.lb.ListenerCertificate` resource.
     */
    @JvmName("caiokfqwteklcfdp")
    public suspend fun certificateArn(`value`: Output<String>) {
        this.certificateArn = value
    }

    /**
     * @param value Configuration block for default actions. Detailed below.
     */
    @JvmName("jpnimfpnsbphyebm")
    public suspend fun defaultActions(`value`: Output<List<ListenerDefaultActionArgs>>) {
        this.defaultActions = value
    }

    @JvmName("bgxqcukufhtrrypc")
    public suspend fun defaultActions(vararg values: Output<ListenerDefaultActionArgs>) {
        this.defaultActions = Output.all(values.asList())
    }

    /**
     * @param values Configuration block for default actions. Detailed below.
     */
    @JvmName("ykxwvveueeobreqw")
    public suspend fun defaultActions(values: List<Output<ListenerDefaultActionArgs>>) {
        this.defaultActions = Output.all(values)
    }

    /**
     * @param value ARN of the load balancer.
     * The following arguments are optional:
     */
    @JvmName("tycuwndqftqvqvhh")
    public suspend fun loadBalancerArn(`value`: Output<String>) {
        this.loadBalancerArn = value
    }

    /**
     * @param value The mutual authentication configuration information. Detailed below.
     */
    @JvmName("ytsjkvetvgpticug")
    public suspend fun mutualAuthentication(`value`: Output<ListenerMutualAuthenticationArgs>) {
        this.mutualAuthentication = value
    }

    /**
     * @param value Port on which the load balancer is listening. Not valid for Gateway Load Balancers.
     */
    @JvmName("qauppuyroalxqdmx")
    public suspend fun port(`value`: Output<Int>) {
        this.port = value
    }

    /**
     * @param value Protocol for connections from clients to the load balancer. For Application Load Balancers, valid values are `HTTP` and `HTTPS`, with a default of `HTTP`. For Network Load Balancers, valid values are `TCP`, `TLS`, `UDP`, and `TCP_UDP`. Not valid to use `UDP` or `TCP_UDP` if dual-stack mode is enabled. Not valid for Gateway Load Balancers.
     */
    @JvmName("vqfevqkvgbtrnjnl")
    public suspend fun protocol(`value`: Output<String>) {
        this.protocol = value
    }

    /**
     * @param value Name of the SSL Policy for the listener. Required if `protocol` is `HTTPS` or `TLS`.
     */
    @JvmName("qibxxjhvpmjeubgg")
    public suspend fun sslPolicy(`value`: Output<String>) {
        this.sslPolicy = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     * > **NOTE::** Please note that listeners that are attached to Application Load Balancers must use either `HTTP` or `HTTPS` protocols while listeners that are attached to Network Load Balancers must use the `TCP` protocol.
     */
    @JvmName("drhvoygqrgllmcyr")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Name of the Application-Layer Protocol Negotiation (ALPN) policy. Can be set if `protocol` is `TLS`. Valid values are `HTTP1Only`, `HTTP2Only`, `HTTP2Optional`, `HTTP2Preferred`, and `None`.
     */
    @JvmName("kkdmvvsjfmlutnnk")
    public suspend fun alpnPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.alpnPolicy = mapped
    }

    /**
     * @param value ARN of the default SSL server certificate. Exactly one certificate is required if the protocol is HTTPS. For adding additional SSL certificates, see the `aws.lb.ListenerCertificate` resource.
     */
    @JvmName("nptvbwpupaipfjnw")
    public suspend fun certificateArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.certificateArn = mapped
    }

    /**
     * @param value Configuration block for default actions. Detailed below.
     */
    @JvmName("clgkqgtcfpswrpyn")
    public suspend fun defaultActions(`value`: List<ListenerDefaultActionArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultActions = mapped
    }

    /**
     * @param argument Configuration block for default actions. Detailed below.
     */
    @JvmName("yhtspsnbikdhsqjo")
    public suspend fun defaultActions(argument: List<suspend ListenerDefaultActionArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            ListenerDefaultActionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.defaultActions = mapped
    }

    /**
     * @param argument Configuration block for default actions. Detailed below.
     */
    @JvmName("jyvoydiijmeicmba")
    public suspend fun defaultActions(vararg argument: suspend ListenerDefaultActionArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            ListenerDefaultActionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.defaultActions = mapped
    }

    /**
     * @param argument Configuration block for default actions. Detailed below.
     */
    @JvmName("pbrsuliabhmxqpnr")
    public suspend fun defaultActions(argument: suspend ListenerDefaultActionArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(ListenerDefaultActionArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.defaultActions = mapped
    }

    /**
     * @param values Configuration block for default actions. Detailed below.
     */
    @JvmName("btdlcpjhmjobcwmk")
    public suspend fun defaultActions(vararg values: ListenerDefaultActionArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.defaultActions = mapped
    }

    /**
     * @param value ARN of the load balancer.
     * The following arguments are optional:
     */
    @JvmName("mjhxrdvmascjafmk")
    public suspend fun loadBalancerArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.loadBalancerArn = mapped
    }

    /**
     * @param value The mutual authentication configuration information. Detailed below.
     */
    @JvmName("cwfhqadgcyuhpqji")
    public suspend fun mutualAuthentication(`value`: ListenerMutualAuthenticationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.mutualAuthentication = mapped
    }

    /**
     * @param argument The mutual authentication configuration information. Detailed below.
     */
    @JvmName("lxdnqklvtgolcmjh")
    public suspend fun mutualAuthentication(argument: suspend ListenerMutualAuthenticationArgsBuilder.() -> Unit) {
        val toBeMapped = ListenerMutualAuthenticationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.mutualAuthentication = mapped
    }

    /**
     * @param value Port on which the load balancer is listening. Not valid for Gateway Load Balancers.
     */
    @JvmName("oifekppytuviqxqm")
    public suspend fun port(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value Protocol for connections from clients to the load balancer. For Application Load Balancers, valid values are `HTTP` and `HTTPS`, with a default of `HTTP`. For Network Load Balancers, valid values are `TCP`, `TLS`, `UDP`, and `TCP_UDP`. Not valid to use `UDP` or `TCP_UDP` if dual-stack mode is enabled. Not valid for Gateway Load Balancers.
     */
    @JvmName("mxdeofnsfcvmhbaf")
    public suspend fun protocol(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.protocol = mapped
    }

    /**
     * @param value Name of the SSL Policy for the listener. Required if `protocol` is `HTTPS` or `TLS`.
     */
    @JvmName("cwkuohosqgakmldk")
    public suspend fun sslPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sslPolicy = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     * > **NOTE::** Please note that listeners that are attached to Application Load Balancers must use either `HTTP` or `HTTPS` protocols while listeners that are attached to Network Load Balancers must use the `TCP` protocol.
     */
    @JvmName("mkvshcwodavixfnf")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     * > **NOTE::** Please note that listeners that are attached to Application Load Balancers must use either `HTTP` or `HTTPS` protocols while listeners that are attached to Network Load Balancers must use the `TCP` protocol.
     */
    @JvmName("knmhpidaybcpbdxl")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): ListenerArgs = ListenerArgs(
        alpnPolicy = alpnPolicy,
        certificateArn = certificateArn,
        defaultActions = defaultActions,
        loadBalancerArn = loadBalancerArn,
        mutualAuthentication = mutualAuthentication,
        port = port,
        protocol = protocol,
        sslPolicy = sslPolicy,
        tags = tags,
    )
}
