@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.lb.kotlin

import com.pulumi.aws.lb.ListenerRuleArgs.builder
import com.pulumi.aws.lb.kotlin.inputs.ListenerRuleActionArgs
import com.pulumi.aws.lb.kotlin.inputs.ListenerRuleActionArgsBuilder
import com.pulumi.aws.lb.kotlin.inputs.ListenerRuleConditionArgs
import com.pulumi.aws.lb.kotlin.inputs.ListenerRuleConditionArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a Load Balancer Listener Rule resource.
 * > **Note:** `aws.alb.ListenerRule` is known as `aws.lb.ListenerRule`. The functionality is identical.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const frontEnd = new aws.lb.LoadBalancer("front_end", {});
 * const frontEndListener = new aws.lb.Listener("front_end", {});
 * const static = new aws.lb.ListenerRule("static", {
 *     listenerArn: frontEndListener.arn,
 *     priority: 100,
 *     actions: [{
 *         type: "forward",
 *         targetGroupArn: staticAwsLbTargetGroup.arn,
 *     }],
 *     conditions: [
 *         {
 *             pathPattern: {
 *                 values: ["/static/*"],
 *             },
 *         },
 *         {
 *             hostHeader: {
 *                 values: ["example&#46;com"],
 *             },
 *         },
 *     ],
 * });
 * // Forward action
 * const hostBasedWeightedRouting = new aws.lb.ListenerRule("host_based_weighted_routing", {
 *     listenerArn: frontEndListener.arn,
 *     priority: 99,
 *     actions: [{
 *         type: "forward",
 *         targetGroupArn: staticAwsLbTargetGroup.arn,
 *     }],
 *     conditions: [{
 *         hostHeader: {
 *             values: ["my-service&#46;*&#46;mycompany&#46;io"],
 *         },
 *     }],
 * });
 * // Weighted Forward action
 * const hostBasedRouting = new aws.lb.ListenerRule("host_based_routing", {
 *     listenerArn: frontEndListener.arn,
 *     priority: 99,
 *     actions: [{
 *         type: "forward",
 *         forward: {
 *             targetGroups: [
 *                 {
 *                     arn: main.arn,
 *                     weight: 80,
 *                 },
 *                 {
 *                     arn: canary.arn,
 *                     weight: 20,
 *                 },
 *             ],
 *             stickiness: {
 *                 enabled: true,
 *                 duration: 600,
 *             },
 *         },
 *     }],
 *     conditions: [{
 *         hostHeader: {
 *             values: ["my-service&#46;*&#46;mycompany&#46;io"],
 *         },
 *     }],
 * });
 * // Redirect action
 * const redirectHttpToHttps = new aws.lb.ListenerRule("redirect_http_to_https", {
 *     listenerArn: frontEndListener.arn,
 *     actions: [{
 *         type: "redirect",
 *         redirect: {
 *             port: "443",
 *             protocol: "HTTPS",
 *             statusCode: "HTTP_301",
 *         },
 *     }],
 *     conditions: [{
 *         httpHeader: {
 *             httpHeaderName: "X-Forwarded-For",
 *             values: ["192&#46;168&#46;1&#46;*"],
 *         },
 *     }],
 * });
 * // Fixed-response action
 * const healthCheck = new aws.lb.ListenerRule("health_check", {
 *     listenerArn: frontEndListener.arn,
 *     actions: [{
 *         type: "fixed-response",
 *         fixedResponse: {
 *             contentType: "text/plain",
 *             messageBody: "HEALTHY",
 *             statusCode: "200",
 *         },
 *     }],
 *     conditions: [{
 *         queryStrings: [
 *             {
 *                 key: "health",
 *                 value: "check",
 *             },
 *             {
 *                 value: "bar",
 *             },
 *         ],
 *     }],
 * });
 * // Authenticate-cognito Action
 * const pool = new aws.cognito.UserPool("pool", {});
 * const client = new aws.cognito.UserPoolClient("client", {});
 * const domain = new aws.cognito.UserPoolDomain("domain", {});
 * const admin = new aws.lb.ListenerRule("admin", {
 *     listenerArn: frontEndListener.arn,
 *     actions: [
 *         {
 *             type: "authenticate-cognito",
 *             authenticateCognito: {
 *                 userPoolArn: pool.arn,
 *                 userPoolClientId: client.id,
 *                 userPoolDomain: domain.domain,
 *             },
 *         },
 *         {
 *             type: "forward",
 *             targetGroupArn: staticAwsLbTargetGroup.arn,
 *         },
 *     ],
 * });
 * // Authenticate-oidc Action
 * const oidc = new aws.lb.ListenerRule("oidc", {
 *     listenerArn: frontEndListener.arn,
 *     actions: [
 *         {
 *             type: "authenticate-oidc",
 *             authenticateOidc: {
 *                 authorizationEndpoint: "https://example.com/authorization_endpoint",
 *                 clientId: "client_id",
 *                 clientSecret: "client_secret",
 *                 issuer: "https://example.com",
 *                 tokenEndpoint: "https://example.com/token_endpoint",
 *                 userInfoEndpoint: "https://example.com/user_info_endpoint",
 *             },
 *         },
 *         {
 *             type: "forward",
 *             targetGroupArn: staticAwsLbTargetGroup.arn,
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * front_end = aws.lb.LoadBalancer("front_end")
 * front_end_listener = aws.lb.Listener("front_end")
 * static = aws.lb.ListenerRule("static",
 *     listener_arn=front_end_listener.arn,
 *     priority=100,
 *     actions=[{
 *         "type": "forward",
 *         "target_group_arn": static_aws_lb_target_group["arn"],
 *     }],
 *     conditions=[
 *         {
 *             "path_pattern": {
 *                 "values": ["/static/*"],
 *             },
 *         },
 *         {
 *             "host_header": {
 *                 "values": ["example&#46;com"],
 *             },
 *         },
 *     ])
 * # Forward action
 * host_based_weighted_routing = aws.lb.ListenerRule("host_based_weighted_routing",
 *     listener_arn=front_end_listener.arn,
 *     priority=99,
 *     actions=[{
 *         "type": "forward",
 *         "target_group_arn": static_aws_lb_target_group["arn"],
 *     }],
 *     conditions=[{
 *         "host_header": {
 *             "values": ["my-service&#46;*&#46;mycompany&#46;io"],
 *         },
 *     }])
 * # Weighted Forward action
 * host_based_routing = aws.lb.ListenerRule("host_based_routing",
 *     listener_arn=front_end_listener.arn,
 *     priority=99,
 *     actions=[{
 *         "type": "forward",
 *         "forward": {
 *             "target_groups": [
 *                 {
 *                     "arn": main["arn"],
 *                     "weight": 80,
 *                 },
 *                 {
 *                     "arn": canary["arn"],
 *                     "weight": 20,
 *                 },
 *             ],
 *             "stickiness": {
 *                 "enabled": True,
 *                 "duration": 600,
 *             },
 *         },
 *     }],
 *     conditions=[{
 *         "host_header": {
 *             "values": ["my-service&#46;*&#46;mycompany&#46;io"],
 *         },
 *     }])
 * # Redirect action
 * redirect_http_to_https = aws.lb.ListenerRule("redirect_http_to_https",
 *     listener_arn=front_end_listener.arn,
 *     actions=[{
 *         "type": "redirect",
 *         "redirect": {
 *             "port": "443",
 *             "protocol": "HTTPS",
 *             "status_code": "HTTP_301",
 *         },
 *     }],
 *     conditions=[{
 *         "http_header": {
 *             "http_header_name": "X-Forwarded-For",
 *             "values": ["192&#46;168&#46;1&#46;*"],
 *         },
 *     }])
 * # Fixed-response action
 * health_check = aws.lb.ListenerRule("health_check",
 *     listener_arn=front_end_listener.arn,
 *     actions=[{
 *         "type": "fixed-response",
 *         "fixed_response": {
 *             "content_type": "text/plain",
 *             "message_body": "HEALTHY",
 *             "status_code": "200",
 *         },
 *     }],
 *     conditions=[{
 *         "query_strings": [
 *             {
 *                 "key": "health",
 *                 "value": "check",
 *             },
 *             {
 *                 "value": "bar",
 *             },
 *         ],
 *     }])
 * # Authenticate-cognito Action
 * pool = aws.cognito.UserPool("pool")
 * client = aws.cognito.UserPoolClient("client")
 * domain = aws.cognito.UserPoolDomain("domain")
 * admin = aws.lb.ListenerRule("admin",
 *     listener_arn=front_end_listener.arn,
 *     actions=[
 *         {
 *             "type": "authenticate-cognito",
 *             "authenticate_cognito": {
 *                 "user_pool_arn": pool.arn,
 *                 "user_pool_client_id": client.id,
 *                 "user_pool_domain": domain.domain,
 *             },
 *         },
 *         {
 *             "type": "forward",
 *             "target_group_arn": static_aws_lb_target_group["arn"],
 *         },
 *     ])
 * # Authenticate-oidc Action
 * oidc = aws.lb.ListenerRule("oidc",
 *     listener_arn=front_end_listener.arn,
 *     actions=[
 *         {
 *             "type": "authenticate-oidc",
 *             "authenticate_oidc": {
 *                 "authorization_endpoint": "https://example.com/authorization_endpoint",
 *                 "client_id": "client_id",
 *                 "client_secret": "client_secret",
 *                 "issuer": "https://example.com",
 *                 "token_endpoint": "https://example.com/token_endpoint",
 *                 "user_info_endpoint": "https://example.com/user_info_endpoint",
 *             },
 *         },
 *         {
 *             "type": "forward",
 *             "target_group_arn": static_aws_lb_target_group["arn"],
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var frontEnd = new Aws.LB.LoadBalancer("front_end");
 *     var frontEndListener = new Aws.LB.Listener("front_end");
 *     var @static = new Aws.LB.ListenerRule("static", new()
 *     {
 *         ListenerArn = frontEndListener.Arn,
 *         Priority = 100,
 *         Actions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = staticAwsLbTargetGroup.Arn,
 *             },
 *         },
 *         Conditions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleConditionArgs
 *             {
 *                 PathPattern = new Aws.LB.Inputs.ListenerRuleConditionPathPatternArgs
 *                 {
 *                     Values = new[]
 *                     {
 *                         "/static/*",
 *                     },
 *                 },
 *             },
 *             new Aws.LB.Inputs.ListenerRuleConditionArgs
 *             {
 *                 HostHeader = new Aws.LB.Inputs.ListenerRuleConditionHostHeaderArgs
 *                 {
 *                     Values = new[]
 *                     {
 *                         "example.com",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     // Forward action
 *     var hostBasedWeightedRouting = new Aws.LB.ListenerRule("host_based_weighted_routing", new()
 *     {
 *         ListenerArn = frontEndListener.Arn,
 *         Priority = 99,
 *         Actions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = staticAwsLbTargetGroup.Arn,
 *             },
 *         },
 *         Conditions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleConditionArgs
 *             {
 *                 HostHeader = new Aws.LB.Inputs.ListenerRuleConditionHostHeaderArgs
 *                 {
 *                     Values = new[]
 *                     {
 *                         "my-service.*.mycompany.io",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     // Weighted Forward action
 *     var hostBasedRouting = new Aws.LB.ListenerRule("host_based_routing", new()
 *     {
 *         ListenerArn = frontEndListener.Arn,
 *         Priority = 99,
 *         Actions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "forward",
 *                 Forward = new Aws.LB.Inputs.ListenerRuleActionForwardArgs
 *                 {
 *                     TargetGroups = new[]
 *                     {
 *                         new Aws.LB.Inputs.ListenerRuleActionForwardTargetGroupArgs
 *                         {
 *                             Arn = main.Arn,
 *                             Weight = 80,
 *                         },
 *                         new Aws.LB.Inputs.ListenerRuleActionForwardTargetGroupArgs
 *                         {
 *                             Arn = canary.Arn,
 *                             Weight = 20,
 *                         },
 *                     },
 *                     Stickiness = new Aws.LB.Inputs.ListenerRuleActionForwardStickinessArgs
 *                     {
 *                         Enabled = true,
 *                         Duration = 600,
 *                     },
 *                 },
 *             },
 *         },
 *         Conditions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleConditionArgs
 *             {
 *                 HostHeader = new Aws.LB.Inputs.ListenerRuleConditionHostHeaderArgs
 *                 {
 *                     Values = new[]
 *                     {
 *                         "my-service.*.mycompany.io",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     // Redirect action
 *     var redirectHttpToHttps = new Aws.LB.ListenerRule("redirect_http_to_https", new()
 *     {
 *         ListenerArn = frontEndListener.Arn,
 *         Actions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "redirect",
 *                 Redirect = new Aws.LB.Inputs.ListenerRuleActionRedirectArgs
 *                 {
 *                     Port = "443",
 *                     Protocol = "HTTPS",
 *                     StatusCode = "HTTP_301",
 *                 },
 *             },
 *         },
 *         Conditions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleConditionArgs
 *             {
 *                 HttpHeader = new Aws.LB.Inputs.ListenerRuleConditionHttpHeaderArgs
 *                 {
 *                     HttpHeaderName = "X-Forwarded-For",
 *                     Values = new[]
 *                     {
 *                         "192.168.1.*",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     // Fixed-response action
 *     var healthCheck = new Aws.LB.ListenerRule("health_check", new()
 *     {
 *         ListenerArn = frontEndListener.Arn,
 *         Actions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "fixed-response",
 *                 FixedResponse = new Aws.LB.Inputs.ListenerRuleActionFixedResponseArgs
 *                 {
 *                     ContentType = "text/plain",
 *                     MessageBody = "HEALTHY",
 *                     StatusCode = "200",
 *                 },
 *             },
 *         },
 *         Conditions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleConditionArgs
 *             {
 *                 QueryStrings = new[]
 *                 {
 *                     new Aws.LB.Inputs.ListenerRuleConditionQueryStringArgs
 *                     {
 *                         Key = "health",
 *                         Value = "check",
 *                     },
 *                     new Aws.LB.Inputs.ListenerRuleConditionQueryStringArgs
 *                     {
 *                         Value = "bar",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     // Authenticate-cognito Action
 *     var pool = new Aws.Cognito.UserPool("pool");
 *     var client = new Aws.Cognito.UserPoolClient("client");
 *     var domain = new Aws.Cognito.UserPoolDomain("domain");
 *     var admin = new Aws.LB.ListenerRule("admin", new()
 *     {
 *         ListenerArn = frontEndListener.Arn,
 *         Actions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "authenticate-cognito",
 *                 AuthenticateCognito = new Aws.LB.Inputs.ListenerRuleActionAuthenticateCognitoArgs
 *                 {
 *                     UserPoolArn = pool.Arn,
 *                     UserPoolClientId = client.Id,
 *                     UserPoolDomain = domain.Domain,
 *                 },
 *             },
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = staticAwsLbTargetGroup.Arn,
 *             },
 *         },
 *     });
 *     // Authenticate-oidc Action
 *     var oidc = new Aws.LB.ListenerRule("oidc", new()
 *     {
 *         ListenerArn = frontEndListener.Arn,
 *         Actions = new[]
 *         {
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "authenticate-oidc",
 *                 AuthenticateOidc = new Aws.LB.Inputs.ListenerRuleActionAuthenticateOidcArgs
 *                 {
 *                     AuthorizationEndpoint = "https://example.com/authorization_endpoint",
 *                     ClientId = "client_id",
 *                     ClientSecret = "client_secret",
 *                     Issuer = "https://example.com",
 *                     TokenEndpoint = "https://example.com/token_endpoint",
 *                     UserInfoEndpoint = "https://example.com/user_info_endpoint",
 *                 },
 *             },
 *             new Aws.LB.Inputs.ListenerRuleActionArgs
 *             {
 *                 Type = "forward",
 *                 TargetGroupArn = staticAwsLbTargetGroup.Arn,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cognito"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := lb.NewLoadBalancer(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		frontEndListener, err := lb.NewListener(ctx, "front_end", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListenerRule(ctx, "static", &lb.ListenerRuleArgs{
 * 			ListenerArn: frontEndListener.Arn,
 * 			Priority:    pulumi.Int(100),
 * 			Actions: lb.ListenerRuleActionArray{
 * 				&lb.ListenerRuleActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: pulumi.Any(staticAwsLbTargetGroup.Arn),
 * 				},
 * 			},
 * 			Conditions: lb.ListenerRuleConditionArray{
 * 				&lb.ListenerRuleConditionArgs{
 * 					PathPattern: &lb.ListenerRuleConditionPathPatternArgs{
 * 						Values: pulumi.StringArray{
 * 							pulumi.String("/static/*"),
 * 						},
 * 					},
 * 				},
 * 				&lb.ListenerRuleConditionArgs{
 * 					HostHeader: &lb.ListenerRuleConditionHostHeaderArgs{
 * 						Values: pulumi.StringArray{
 * 							pulumi.String("example.com"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Forward action
 * 		_, err = lb.NewListenerRule(ctx, "host_based_weighted_routing", &lb.ListenerRuleArgs{
 * 			ListenerArn: frontEndListener.Arn,
 * 			Priority:    pulumi.Int(99),
 * 			Actions: lb.ListenerRuleActionArray{
 * 				&lb.ListenerRuleActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: pulumi.Any(staticAwsLbTargetGroup.Arn),
 * 				},
 * 			},
 * 			Conditions: lb.ListenerRuleConditionArray{
 * 				&lb.ListenerRuleConditionArgs{
 * 					HostHeader: &lb.ListenerRuleConditionHostHeaderArgs{
 * 						Values: pulumi.StringArray{
 * 							pulumi.String("my-service.*.mycompany.io"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Weighted Forward action
 * 		_, err = lb.NewListenerRule(ctx, "host_based_routing", &lb.ListenerRuleArgs{
 * 			ListenerArn: frontEndListener.Arn,
 * 			Priority:    pulumi.Int(99),
 * 			Actions: lb.ListenerRuleActionArray{
 * 				&lb.ListenerRuleActionArgs{
 * 					Type: pulumi.String("forward"),
 * 					Forward: &lb.ListenerRuleActionForwardArgs{
 * 						TargetGroups: lb.ListenerRuleActionForwardTargetGroupArray{
 * 							&lb.ListenerRuleActionForwardTargetGroupArgs{
 * 								Arn:    pulumi.Any(main.Arn),
 * 								Weight: pulumi.Int(80),
 * 							},
 * 							&lb.ListenerRuleActionForwardTargetGroupArgs{
 * 								Arn:    pulumi.Any(canary.Arn),
 * 								Weight: pulumi.Int(20),
 * 							},
 * 						},
 * 						Stickiness: &lb.ListenerRuleActionForwardStickinessArgs{
 * 							Enabled:  pulumi.Bool(true),
 * 							Duration: pulumi.Int(600),
 * 						},
 * 					},
 * 				},
 * 			},
 * 			Conditions: lb.ListenerRuleConditionArray{
 * 				&lb.ListenerRuleConditionArgs{
 * 					HostHeader: &lb.ListenerRuleConditionHostHeaderArgs{
 * 						Values: pulumi.StringArray{
 * 							pulumi.String("my-service.*.mycompany.io"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Redirect action
 * 		_, err = lb.NewListenerRule(ctx, "redirect_http_to_https", &lb.ListenerRuleArgs{
 * 			ListenerArn: frontEndListener.Arn,
 * 			Actions: lb.ListenerRuleActionArray{
 * 				&lb.ListenerRuleActionArgs{
 * 					Type: pulumi.String("redirect"),
 * 					Redirect: &lb.ListenerRuleActionRedirectArgs{
 * 						Port:       pulumi.String("443"),
 * 						Protocol:   pulumi.String("HTTPS"),
 * 						StatusCode: pulumi.String("HTTP_301"),
 * 					},
 * 				},
 * 			},
 * 			Conditions: lb.ListenerRuleConditionArray{
 * 				&lb.ListenerRuleConditionArgs{
 * 					HttpHeader: &lb.ListenerRuleConditionHttpHeaderArgs{
 * 						HttpHeaderName: pulumi.String("X-Forwarded-For"),
 * 						Values: pulumi.StringArray{
 * 							pulumi.String("192.168.1.*"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Fixed-response action
 * 		_, err = lb.NewListenerRule(ctx, "health_check", &lb.ListenerRuleArgs{
 * 			ListenerArn: frontEndListener.Arn,
 * 			Actions: lb.ListenerRuleActionArray{
 * 				&lb.ListenerRuleActionArgs{
 * 					Type: pulumi.String("fixed-response"),
 * 					FixedResponse: &lb.ListenerRuleActionFixedResponseArgs{
 * 						ContentType: pulumi.String("text/plain"),
 * 						MessageBody: pulumi.String("HEALTHY"),
 * 						StatusCode:  pulumi.String("200"),
 * 					},
 * 				},
 * 			},
 * 			Conditions: lb.ListenerRuleConditionArray{
 * 				&lb.ListenerRuleConditionArgs{
 * 					QueryStrings: lb.ListenerRuleConditionQueryStringArray{
 * 						&lb.ListenerRuleConditionQueryStringArgs{
 * 							Key:   pulumi.String("health"),
 * 							Value: pulumi.String("check"),
 * 						},
 * 						&lb.ListenerRuleConditionQueryStringArgs{
 * 							Value: pulumi.String("bar"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Authenticate-cognito Action
 * 		pool, err := cognito.NewUserPool(ctx, "pool", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		client, err := cognito.NewUserPoolClient(ctx, "client", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		domain, err := cognito.NewUserPoolDomain(ctx, "domain", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewListenerRule(ctx, "admin", &lb.ListenerRuleArgs{
 * 			ListenerArn: frontEndListener.Arn,
 * 			Actions: lb.ListenerRuleActionArray{
 * 				&lb.ListenerRuleActionArgs{
 * 					Type: pulumi.String("authenticate-cognito"),
 * 					AuthenticateCognito: &lb.ListenerRuleActionAuthenticateCognitoArgs{
 * 						UserPoolArn:      pool.Arn,
 * 						UserPoolClientId: client.ID(),
 * 						UserPoolDomain:   domain.Domain,
 * 					},
 * 				},
 * 				&lb.ListenerRuleActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: pulumi.Any(staticAwsLbTargetGroup.Arn),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Authenticate-oidc Action
 * 		_, err = lb.NewListenerRule(ctx, "oidc", &lb.ListenerRuleArgs{
 * 			ListenerArn: frontEndListener.Arn,
 * 			Actions: lb.ListenerRuleActionArray{
 * 				&lb.ListenerRuleActionArgs{
 * 					Type: pulumi.String("authenticate-oidc"),
 * 					AuthenticateOidc: &lb.ListenerRuleActionAuthenticateOidcArgs{
 * 						AuthorizationEndpoint: pulumi.String("https://example.com/authorization_endpoint"),
 * 						ClientId:              pulumi.String("client_id"),
 * 						ClientSecret:          pulumi.String("client_secret"),
 * 						Issuer:                pulumi.String("https://example.com"),
 * 						TokenEndpoint:         pulumi.String("https://example.com/token_endpoint"),
 * 						UserInfoEndpoint:      pulumi.String("https://example.com/user_info_endpoint"),
 * 					},
 * 				},
 * 				&lb.ListenerRuleActionArgs{
 * 					Type:           pulumi.String("forward"),
 * 					TargetGroupArn: pulumi.Any(staticAwsLbTargetGroup.Arn),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.Listener;
 * import com.pulumi.aws.lb.ListenerRule;
 * import com.pulumi.aws.lb.ListenerRuleArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleActionArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleConditionArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleConditionPathPatternArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleConditionHostHeaderArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleActionForwardArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleActionForwardStickinessArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleActionRedirectArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleConditionHttpHeaderArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleActionFixedResponseArgs;
 * import com.pulumi.aws.cognito.UserPool;
 * import com.pulumi.aws.cognito.UserPoolClient;
 * import com.pulumi.aws.cognito.UserPoolDomain;
 * import com.pulumi.aws.lb.inputs.ListenerRuleActionAuthenticateCognitoArgs;
 * import com.pulumi.aws.lb.inputs.ListenerRuleActionAuthenticateOidcArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var frontEnd = new LoadBalancer("frontEnd");
 *         var frontEndListener = new Listener("frontEndListener");
 *         var static_ = new ListenerRule("static", ListenerRuleArgs.builder()
 *             .listenerArn(frontEndListener.arn())
 *             .priority(100)
 *             .actions(ListenerRuleActionArgs.builder()
 *                 .type("forward")
 *                 .targetGroupArn(staticAwsLbTargetGroup.arn())
 *                 .build())
 *             .conditions(
 *                 ListenerRuleConditionArgs.builder()
 *                     .pathPattern(ListenerRuleConditionPathPatternArgs.builder()
 *                         .values("/static/*")
 *                         .build())
 *                     .build(),
 *                 ListenerRuleConditionArgs.builder()
 *                     .hostHeader(ListenerRuleConditionHostHeaderArgs.builder()
 *                         .values("example.com")
 *                         .build())
 *                     .build())
 *             .build());
 *         // Forward action
 *         var hostBasedWeightedRouting = new ListenerRule("hostBasedWeightedRouting", ListenerRuleArgs.builder()
 *             .listenerArn(frontEndListener.arn())
 *             .priority(99)
 *             .actions(ListenerRuleActionArgs.builder()
 *                 .type("forward")
 *                 .targetGroupArn(staticAwsLbTargetGroup.arn())
 *                 .build())
 *             .conditions(ListenerRuleConditionArgs.builder()
 *                 .hostHeader(ListenerRuleConditionHostHeaderArgs.builder()
 *                     .values("my-service.*.mycompany.io")
 *                     .build())
 *                 .build())
 *             .build());
 *         // Weighted Forward action
 *         var hostBasedRouting = new ListenerRule("hostBasedRouting", ListenerRuleArgs.builder()
 *             .listenerArn(frontEndListener.arn())
 *             .priority(99)
 *             .actions(ListenerRuleActionArgs.builder()
 *                 .type("forward")
 *                 .forward(ListenerRuleActionForwardArgs.builder()
 *                     .targetGroups(
 *                         ListenerRuleActionForwardTargetGroupArgs.builder()
 *                             .arn(main.arn())
 *                             .weight(80)
 *                             .build(),
 *                         ListenerRuleActionForwardTargetGroupArgs.builder()
 *                             .arn(canary.arn())
 *                             .weight(20)
 *                             .build())
 *                     .stickiness(ListenerRuleActionForwardStickinessArgs.builder()
 *                         .enabled(true)
 *                         .duration(600)
 *                         .build())
 *                     .build())
 *                 .build())
 *             .conditions(ListenerRuleConditionArgs.builder()
 *                 .hostHeader(ListenerRuleConditionHostHeaderArgs.builder()
 *                     .values("my-service.*.mycompany.io")
 *                     .build())
 *                 .build())
 *             .build());
 *         // Redirect action
 *         var redirectHttpToHttps = new ListenerRule("redirectHttpToHttps", ListenerRuleArgs.builder()
 *             .listenerArn(frontEndListener.arn())
 *             .actions(ListenerRuleActionArgs.builder()
 *                 .type("redirect")
 *                 .redirect(ListenerRuleActionRedirectArgs.builder()
 *                     .port("443")
 *                     .protocol("HTTPS")
 *                     .statusCode("HTTP_301")
 *                     .build())
 *                 .build())
 *             .conditions(ListenerRuleConditionArgs.builder()
 *                 .httpHeader(ListenerRuleConditionHttpHeaderArgs.builder()
 *                     .httpHeaderName("X-Forwarded-For")
 *                     .values("192.168.1.*")
 *                     .build())
 *                 .build())
 *             .build());
 *         // Fixed-response action
 *         var healthCheck = new ListenerRule("healthCheck", ListenerRuleArgs.builder()
 *             .listenerArn(frontEndListener.arn())
 *             .actions(ListenerRuleActionArgs.builder()
 *                 .type("fixed-response")
 *                 .fixedResponse(ListenerRuleActionFixedResponseArgs.builder()
 *                     .contentType("text/plain")
 *                     .messageBody("HEALTHY")
 *                     .statusCode("200")
 *                     .build())
 *                 .build())
 *             .conditions(ListenerRuleConditionArgs.builder()
 *                 .queryStrings(
 *                     ListenerRuleConditionQueryStringArgs.builder()
 *                         .key("health")
 *                         .value("check")
 *                         .build(),
 *                     ListenerRuleConditionQueryStringArgs.builder()
 *                         .value("bar")
 *                         .build())
 *                 .build())
 *             .build());
 *         // Authenticate-cognito Action
 *         var pool = new UserPool("pool");
 *         var client = new UserPoolClient("client");
 *         var domain = new UserPoolDomain("domain");
 *         var admin = new ListenerRule("admin", ListenerRuleArgs.builder()
 *             .listenerArn(frontEndListener.arn())
 *             .actions(
 *                 ListenerRuleActionArgs.builder()
 *                     .type("authenticate-cognito")
 *                     .authenticateCognito(ListenerRuleActionAuthenticateCognitoArgs.builder()
 *                         .userPoolArn(pool.arn())
 *                         .userPoolClientId(client.id())
 *                         .userPoolDomain(domain.domain())
 *                         .build())
 *                     .build(),
 *                 ListenerRuleActionArgs.builder()
 *                     .type("forward")
 *                     .targetGroupArn(staticAwsLbTargetGroup.arn())
 *                     .build())
 *             .build());
 *         // Authenticate-oidc Action
 *         var oidc = new ListenerRule("oidc", ListenerRuleArgs.builder()
 *             .listenerArn(frontEndListener.arn())
 *             .actions(
 *                 ListenerRuleActionArgs.builder()
 *                     .type("authenticate-oidc")
 *                     .authenticateOidc(ListenerRuleActionAuthenticateOidcArgs.builder()
 *                         .authorizationEndpoint("https://example.com/authorization_endpoint")
 *                         .clientId("client_id")
 *                         .clientSecret("client_secret")
 *                         .issuer("https://example.com")
 *                         .tokenEndpoint("https://example.com/token_endpoint")
 *                         .userInfoEndpoint("https://example.com/user_info_endpoint")
 *                         .build())
 *                     .build(),
 *                 ListenerRuleActionArgs.builder()
 *                     .type("forward")
 *                     .targetGroupArn(staticAwsLbTargetGroup.arn())
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   frontEnd:
 *     type: aws:lb:LoadBalancer
 *     name: front_end
 *   frontEndListener:
 *     type: aws:lb:Listener
 *     name: front_end
 *   static:
 *     type: aws:lb:ListenerRule
 *     properties:
 *       listenerArn: ${frontEndListener.arn}
 *       priority: 100
 *       actions:
 *         - type: forward
 *           targetGroupArn: ${staticAwsLbTargetGroup.arn}
 *       conditions:
 *         - pathPattern:
 *             values:
 *               - /static/*
 *         - hostHeader:
 *             values:
 *               - example.com
 *   # Forward action
 *   hostBasedWeightedRouting:
 *     type: aws:lb:ListenerRule
 *     name: host_based_weighted_routing
 *     properties:
 *       listenerArn: ${frontEndListener.arn}
 *       priority: 99
 *       actions:
 *         - type: forward
 *           targetGroupArn: ${staticAwsLbTargetGroup.arn}
 *       conditions:
 *         - hostHeader:
 *             values:
 *               - my-service.*.mycompany.io
 *   # Weighted Forward action
 *   hostBasedRouting:
 *     type: aws:lb:ListenerRule
 *     name: host_based_routing
 *     properties:
 *       listenerArn: ${frontEndListener.arn}
 *       priority: 99
 *       actions:
 *         - type: forward
 *           forward:
 *             targetGroups:
 *               - arn: ${main.arn}
 *                 weight: 80
 *               - arn: ${canary.arn}
 *                 weight: 20
 *             stickiness:
 *               enabled: true
 *               duration: 600
 *       conditions:
 *         - hostHeader:
 *             values:
 *               - my-service.*.mycompany.io
 *   # Redirect action
 *   redirectHttpToHttps:
 *     type: aws:lb:ListenerRule
 *     name: redirect_http_to_https
 *     properties:
 *       listenerArn: ${frontEndListener.arn}
 *       actions:
 *         - type: redirect
 *           redirect:
 *             port: '443'
 *             protocol: HTTPS
 *             statusCode: HTTP_301
 *       conditions:
 *         - httpHeader:
 *             httpHeaderName: X-Forwarded-For
 *             values:
 *               - 192.168.1.*
 *   # Fixed-response action
 *   healthCheck:
 *     type: aws:lb:ListenerRule
 *     name: health_check
 *     properties:
 *       listenerArn: ${frontEndListener.arn}
 *       actions:
 *         - type: fixed-response
 *           fixedResponse:
 *             contentType: text/plain
 *             messageBody: HEALTHY
 *             statusCode: '200'
 *       conditions:
 *         - queryStrings:
 *             - key: health
 *               value: check
 *             - value: bar
 *   # Authenticate-cognito Action
 *   pool:
 *     type: aws:cognito:UserPool
 *   client:
 *     type: aws:cognito:UserPoolClient
 *   domain:
 *     type: aws:cognito:UserPoolDomain
 *   admin:
 *     type: aws:lb:ListenerRule
 *     properties:
 *       listenerArn: ${frontEndListener.arn}
 *       actions:
 *         - type: authenticate-cognito
 *           authenticateCognito:
 *             userPoolArn: ${pool.arn}
 *             userPoolClientId: ${client.id}
 *             userPoolDomain: ${domain.domain}
 *         - type: forward
 *           targetGroupArn: ${staticAwsLbTargetGroup.arn}
 *   # Authenticate-oidc Action
 *   oidc:
 *     type: aws:lb:ListenerRule
 *     properties:
 *       listenerArn: ${frontEndListener.arn}
 *       actions:
 *         - type: authenticate-oidc
 *           authenticateOidc:
 *             authorizationEndpoint: https://example.com/authorization_endpoint
 *             clientId: client_id
 *             clientSecret: client_secret
 *             issuer: https://example.com
 *             tokenEndpoint: https://example.com/token_endpoint
 *             userInfoEndpoint: https://example.com/user_info_endpoint
 *         - type: forward
 *           targetGroupArn: ${staticAwsLbTargetGroup.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import rules using their ARN. For example:
 * ```sh
 * $ pulumi import aws:lb/listenerRule:ListenerRule front_end arn:aws:elasticloadbalancing:us-west-2:187416307283:listener-rule/app/test/8e4497da625e2d8a/9ab28ade35828f96/67b3d2d36dd7c26b
 * ```
 * @property actions An Action block. Action blocks are documented below.
 * @property conditions A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
 * @property listenerArn The ARN of the listener to which to attach the rule.
 * @property priority The priority for the rule between `1` and `50000`. Leaving it unset will automatically set the rule with next available priority after currently existing highest rule. A listener can't have multiple rules with the same priority.
 * @property tags A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * */*/*/*/*/*/
 */
public data class ListenerRuleArgs(
    public val actions: Output<List<ListenerRuleActionArgs>>? = null,
    public val conditions: Output<List<ListenerRuleConditionArgs>>? = null,
    public val listenerArn: Output<String>? = null,
    public val priority: Output<Int>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.lb.ListenerRuleArgs> {
    override fun toJava(): com.pulumi.aws.lb.ListenerRuleArgs =
        com.pulumi.aws.lb.ListenerRuleArgs.builder()
            .actions(
                actions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .conditions(
                conditions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .listenerArn(listenerArn?.applyValue({ args0 -> args0 }))
            .priority(priority?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [ListenerRuleArgs].
 */
@PulumiTagMarker
public class ListenerRuleArgsBuilder internal constructor() {
    private var actions: Output<List<ListenerRuleActionArgs>>? = null

    private var conditions: Output<List<ListenerRuleConditionArgs>>? = null

    private var listenerArn: Output<String>? = null

    private var priority: Output<Int>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value An Action block. Action blocks are documented below.
     */
    @JvmName("vrltehhojuuqgvpf")
    public suspend fun actions(`value`: Output<List<ListenerRuleActionArgs>>) {
        this.actions = value
    }

    @JvmName("erharyvxcjpjmivg")
    public suspend fun actions(vararg values: Output<ListenerRuleActionArgs>) {
        this.actions = Output.all(values.asList())
    }

    /**
     * @param values An Action block. Action blocks are documented below.
     */
    @JvmName("ureykbjelvokfxsv")
    public suspend fun actions(values: List<Output<ListenerRuleActionArgs>>) {
        this.actions = Output.all(values)
    }

    /**
     * @param value A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
     */
    @JvmName("fdvlljbteokjmnat")
    public suspend fun conditions(`value`: Output<List<ListenerRuleConditionArgs>>) {
        this.conditions = value
    }

    @JvmName("xjjgdvjgpfqnfupl")
    public suspend fun conditions(vararg values: Output<ListenerRuleConditionArgs>) {
        this.conditions = Output.all(values.asList())
    }

    /**
     * @param values A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
     */
    @JvmName("igavbtrgfefnodht")
    public suspend fun conditions(values: List<Output<ListenerRuleConditionArgs>>) {
        this.conditions = Output.all(values)
    }

    /**
     * @param value The ARN of the listener to which to attach the rule.
     */
    @JvmName("ucudebhmdkbtqxeu")
    public suspend fun listenerArn(`value`: Output<String>) {
        this.listenerArn = value
    }

    /**
     * @param value The priority for the rule between `1` and `50000`. Leaving it unset will automatically set the rule with next available priority after currently existing highest rule. A listener can't have multiple rules with the same priority.
     */
    @JvmName("mppgddxcpupvcmcl")
    public suspend fun priority(`value`: Output<Int>) {
        this.priority = value
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("sfbmkbvjeqqgtuhj")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value An Action block. Action blocks are documented below.
     */
    @JvmName("xwoaolugnqdnybvf")
    public suspend fun actions(`value`: List<ListenerRuleActionArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.actions = mapped
    }

    /**
     * @param argument An Action block. Action blocks are documented below.
     */
    @JvmName("mbefnqmpiwpyqgwq")
    public suspend fun actions(argument: List<suspend ListenerRuleActionArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            ListenerRuleActionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.actions = mapped
    }

    /**
     * @param argument An Action block. Action blocks are documented below.
     */
    @JvmName("egyypsnwwrdhedko")
    public suspend fun actions(vararg argument: suspend ListenerRuleActionArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            ListenerRuleActionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.actions = mapped
    }

    /**
     * @param argument An Action block. Action blocks are documented below.
     */
    @JvmName("vvrxjvplmbntykeq")
    public suspend fun actions(argument: suspend ListenerRuleActionArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(ListenerRuleActionArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.actions = mapped
    }

    /**
     * @param values An Action block. Action blocks are documented below.
     */
    @JvmName("iwtvqbvgahorycce")
    public suspend fun actions(vararg values: ListenerRuleActionArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.actions = mapped
    }

    /**
     * @param value A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
     */
    @JvmName("ehwikibldqtqlwwq")
    public suspend fun conditions(`value`: List<ListenerRuleConditionArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.conditions = mapped
    }

    /**
     * @param argument A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
     */
    @JvmName("mdxldsrwhgikgsmj")
    public suspend fun conditions(argument: List<suspend ListenerRuleConditionArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            ListenerRuleConditionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.conditions = mapped
    }

    /**
     * @param argument A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
     */
    @JvmName("thiecyaxlooorxqg")
    public suspend fun conditions(vararg argument: suspend ListenerRuleConditionArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            ListenerRuleConditionArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.conditions = mapped
    }

    /**
     * @param argument A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
     */
    @JvmName("apxwfusmsmvmtkbk")
    public suspend fun conditions(argument: suspend ListenerRuleConditionArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(ListenerRuleConditionArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.conditions = mapped
    }

    /**
     * @param values A Condition block. Multiple condition blocks of different types can be set and all must be satisfied for the rule to match. Condition blocks are documented below.
     */
    @JvmName("xfkfoydjofutkdvk")
    public suspend fun conditions(vararg values: ListenerRuleConditionArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.conditions = mapped
    }

    /**
     * @param value The ARN of the listener to which to attach the rule.
     */
    @JvmName("kubnplshvrfxcivb")
    public suspend fun listenerArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.listenerArn = mapped
    }

    /**
     * @param value The priority for the rule between `1` and `50000`. Leaving it unset will automatically set the rule with next available priority after currently existing highest rule. A listener can't have multiple rules with the same priority.
     */
    @JvmName("xvtcdhodjdpftyln")
    public suspend fun priority(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.priority = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("wjpjhohyjblgtkuj")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("fesajubquywmmyyt")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): ListenerRuleArgs = ListenerRuleArgs(
        actions = actions,
        conditions = conditions,
        listenerArn = listenerArn,
        priority = priority,
        tags = tags,
    )
}
