@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.lb.kotlin

import com.pulumi.aws.lb.LoadBalancerArgs.builder
import com.pulumi.aws.lb.kotlin.inputs.LoadBalancerAccessLogsArgs
import com.pulumi.aws.lb.kotlin.inputs.LoadBalancerAccessLogsArgsBuilder
import com.pulumi.aws.lb.kotlin.inputs.LoadBalancerConnectionLogsArgs
import com.pulumi.aws.lb.kotlin.inputs.LoadBalancerConnectionLogsArgsBuilder
import com.pulumi.aws.lb.kotlin.inputs.LoadBalancerSubnetMappingArgs
import com.pulumi.aws.lb.kotlin.inputs.LoadBalancerSubnetMappingArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a Load Balancer resource.
 * > **Note:** `aws.alb.LoadBalancer` is known as `aws.lb.LoadBalancer`. The functionality is identical.
 * ## Example Usage
 * ### Application Load Balancer
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const test = new aws.lb.LoadBalancer("test", {
 *     name: "test-lb-tf",
 *     internal: false,
 *     loadBalancerType: "application",
 *     securityGroups: [lbSg&#46;id],
 *     subnets: .map(subnet => (subnet.id)),
 *     enableDeletionProtection: true,
 *     accessLogs: {
 *         bucket: lbLogs.id,
 *         prefix: "test-lb",
 *         enabled: true,
 *     },
 *     tags: {
 *         Environment: "production",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test = aws.lb.LoadBalancer("test",
 *     name="test-lb-tf",
 *     internal=False,
 *     load_balancer_type="application",
 *     security_groups=[lb_sg["id"]],
 *     subnets=[subnet["id"] for subnet in public],
 *     enable_deletion_protection=True,
 *     access_logs={
 *         "bucket": lb_logs["id"],
 *         "prefix": "test-lb",
 *         "enabled": True,
 *     },
 *     tags={
 *         "Environment": "production",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var test = new Aws.LB.LoadBalancer("test", new()
 *     {
 *         Name = "test-lb-tf",
 *         Internal = false,
 *         LoadBalancerType = "application",
 *         SecurityGroups = new[]
 *         {
 *             lbSg.Id,
 *         },
 *         Subnets = .Select(subnet =>
 *         {
 *             return subnet.Id;
 *         }).ToList(),
 *         EnableDeletionProtection = true,
 *         AccessLogs = new Aws.LB.Inputs.LoadBalancerAccessLogsArgs
 *         {
 *             Bucket = lbLogs.Id,
 *             Prefix = "test-lb",
 *             Enabled = true,
 *         },
 *         Tags =
 *         {
 *             { "Environment", "production" },
 *         },
 *     });
 * });
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Network Load Balancer
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const test = new aws.lb.LoadBalancer("test", {
 *     name: "test-lb-tf",
 *     internal: false,
 *     loadBalancerType: "network",
 *     subnets: .map(subnet => (subnet.id)),
 *     enableDeletionProtection: true,
 *     tags: {
 *         Environment: "production",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test = aws.lb.LoadBalancer("test",
 *     name="test-lb-tf",
 *     internal=False,
 *     load_balancer_type="network",
 *     subnets=[subnet["id"] for subnet in public],
 *     enable_deletion_protection=True,
 *     tags={
 *         "Environment": "production",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var test = new Aws.LB.LoadBalancer("test", new()
 *     {
 *         Name = "test-lb-tf",
 *         Internal = false,
 *         LoadBalancerType = "network",
 *         Subnets = .Select(subnet =>
 *         {
 *             return subnet.Id;
 *         }).ToList(),
 *         EnableDeletionProtection = true,
 *         Tags =
 *         {
 *             { "Environment", "production" },
 *         },
 *     });
 * });
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Specifying Elastic IPs
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.lb.LoadBalancer("example", {
 *     name: "example",
 *     loadBalancerType: "network",
 *     subnetMappings: [
 *         {
 *             subnetId: example1AwsSubnet.id,
 *             allocationId: example1.id,
 *         },
 *         {
 *             subnetId: example2AwsSubnet.id,
 *             allocationId: example2.id,
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.lb.LoadBalancer("example",
 *     name="example",
 *     load_balancer_type="network",
 *     subnet_mappings=[
 *         {
 *             "subnet_id": example1_aws_subnet["id"],
 *             "allocation_id": example1["id"],
 *         },
 *         {
 *             "subnet_id": example2_aws_subnet["id"],
 *             "allocation_id": example2["id"],
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.LB.LoadBalancer("example", new()
 *     {
 *         Name = "example",
 *         LoadBalancerType = "network",
 *         SubnetMappings = new[]
 *         {
 *             new Aws.LB.Inputs.LoadBalancerSubnetMappingArgs
 *             {
 *                 SubnetId = example1AwsSubnet.Id,
 *                 AllocationId = example1.Id,
 *             },
 *             new Aws.LB.Inputs.LoadBalancerSubnetMappingArgs
 *             {
 *                 SubnetId = example2AwsSubnet.Id,
 *                 AllocationId = example2.Id,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := lb.NewLoadBalancer(ctx, "example", &lb.LoadBalancerArgs{
 * 			Name:             pulumi.String("example"),
 * 			LoadBalancerType: pulumi.String("network"),
 * 			SubnetMappings: lb.LoadBalancerSubnetMappingArray{
 * 				&lb.LoadBalancerSubnetMappingArgs{
 * 					SubnetId:     pulumi.Any(example1AwsSubnet.Id),
 * 					AllocationId: pulumi.Any(example1.Id),
 * 				},
 * 				&lb.LoadBalancerSubnetMappingArgs{
 * 					SubnetId:     pulumi.Any(example2AwsSubnet.Id),
 * 					AllocationId: pulumi.Any(example2.Id),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.LoadBalancerArgs;
 * import com.pulumi.aws.lb.inputs.LoadBalancerSubnetMappingArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new LoadBalancer("example", LoadBalancerArgs.builder()
 *             .name("example")
 *             .loadBalancerType("network")
 *             .subnetMappings(
 *                 LoadBalancerSubnetMappingArgs.builder()
 *                     .subnetId(example1AwsSubnet.id())
 *                     .allocationId(example1.id())
 *                     .build(),
 *                 LoadBalancerSubnetMappingArgs.builder()
 *                     .subnetId(example2AwsSubnet.id())
 *                     .allocationId(example2.id())
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:lb:LoadBalancer
 *     properties:
 *       name: example
 *       loadBalancerType: network
 *       subnetMappings:
 *         - subnetId: ${example1AwsSubnet.id}
 *           allocationId: ${example1.id}
 *         - subnetId: ${example2AwsSubnet.id}
 *           allocationId: ${example2.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Specifying private IP addresses for an internal-facing load balancer
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.lb.LoadBalancer("example", {
 *     name: "example",
 *     loadBalancerType: "network",
 *     subnetMappings: [
 *         {
 *             subnetId: example1.id,
 *             privateIpv4Address: "10.0.1.15",
 *         },
 *         {
 *             subnetId: example2.id,
 *             privateIpv4Address: "10.0.2.15",
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.lb.LoadBalancer("example",
 *     name="example",
 *     load_balancer_type="network",
 *     subnet_mappings=[
 *         {
 *             "subnet_id": example1["id"],
 *             "private_ipv4_address": "10.0.1.15",
 *         },
 *         {
 *             "subnet_id": example2["id"],
 *             "private_ipv4_address": "10.0.2.15",
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.LB.LoadBalancer("example", new()
 *     {
 *         Name = "example",
 *         LoadBalancerType = "network",
 *         SubnetMappings = new[]
 *         {
 *             new Aws.LB.Inputs.LoadBalancerSubnetMappingArgs
 *             {
 *                 SubnetId = example1.Id,
 *                 PrivateIpv4Address = "10.0.1.15",
 *             },
 *             new Aws.LB.Inputs.LoadBalancerSubnetMappingArgs
 *             {
 *                 SubnetId = example2.Id,
 *                 PrivateIpv4Address = "10.0.2.15",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := lb.NewLoadBalancer(ctx, "example", &lb.LoadBalancerArgs{
 * 			Name:             pulumi.String("example"),
 * 			LoadBalancerType: pulumi.String("network"),
 * 			SubnetMappings: lb.LoadBalancerSubnetMappingArray{
 * 				&lb.LoadBalancerSubnetMappingArgs{
 * 					SubnetId:           pulumi.Any(example1.Id),
 * 					PrivateIpv4Address: pulumi.String("10.0.1.15"),
 * 				},
 * 				&lb.LoadBalancerSubnetMappingArgs{
 * 					SubnetId:           pulumi.Any(example2.Id),
 * 					PrivateIpv4Address: pulumi.String("10.0.2.15"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.LoadBalancer;
 * import com.pulumi.aws.lb.LoadBalancerArgs;
 * import com.pulumi.aws.lb.inputs.LoadBalancerSubnetMappingArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new LoadBalancer("example", LoadBalancerArgs.builder()
 *             .name("example")
 *             .loadBalancerType("network")
 *             .subnetMappings(
 *                 LoadBalancerSubnetMappingArgs.builder()
 *                     .subnetId(example1.id())
 *                     .privateIpv4Address("10.0.1.15")
 *                     .build(),
 *                 LoadBalancerSubnetMappingArgs.builder()
 *                     .subnetId(example2.id())
 *                     .privateIpv4Address("10.0.2.15")
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:lb:LoadBalancer
 *     properties:
 *       name: example
 *       loadBalancerType: network
 *       subnetMappings:
 *         - subnetId: ${example1.id}
 *           privateIpv4Address: 10.0.1.15
 *         - subnetId: ${example2.id}
 *           privateIpv4Address: 10.0.2.15
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import LBs using their ARN. For example:
 * ```sh
 * $ pulumi import aws:lb/loadBalancer:LoadBalancer bar arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/app/my-load-balancer/50dc6c495c0c9188
 * ```
 * @property accessLogs Access Logs block. See below.
 * @property clientKeepAlive Client keep alive value in seconds. The valid range is 60-604800 seconds. The default is 3600 seconds.
 * @property connectionLogs Connection Logs block. See below. Only valid for Load Balancers of type `application`.
 * @property customerOwnedIpv4Pool ID of the customer owned ipv4 pool to use for this load balancer.
 * @property desyncMitigationMode How the load balancer handles requests that might pose a security risk to an application due to HTTP desync. Valid values are `monitor`, `defensive` (default), `strictest`.
 * @property dnsRecordClientRoutingPolicy How traffic is distributed among the load balancer Availability Zones. Possible values are `any_availability_zone` (default), `availability_zone_affinity`, or `partial_availability_zone_affinity`. See   [Availability Zone DNS affinity](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#zonal-dns-affinity) for additional details. Only valid for `network` type load balancers.
 * @property dropInvalidHeaderFields Whether HTTP headers with header fields that are not valid are removed by the load balancer (true) or routed to targets (false). The default is false. Elastic Load Balancing requires that message header names contain only alphanumeric characters and hyphens. Only valid for Load Balancers of type `application`.
 * @property enableCrossZoneLoadBalancing If true, cross-zone load balancing of the load balancer will be enabled. For `network` and `gateway` type load balancers, this feature is disabled by default (`false`). For `application` load balancer this feature is always enabled (`true`) and cannot be disabled. Defaults to `false`.
 * @property enableDeletionProtection If true, deletion of the load balancer will be disabled via the AWS API. This will prevent this provider from deleting the load balancer. Defaults to `false`.
 * @property enableHttp2 Whether HTTP/2 is enabled in `application` load balancers. Defaults to `true`.
 * @property enableTlsVersionAndCipherSuiteHeaders Whether the two headers (`x-amzn-tls-version` and `x-amzn-tls-cipher-suite`), which contain information about the negotiated TLS version and cipher suite, are added to the client request before sending it to the target. Only valid for Load Balancers of type `application`. Defaults to `false`
 * @property enableWafFailOpen Whether to allow a WAF-enabled load balancer to route requests to targets if it is unable to forward the request to AWS WAF. Defaults to `false`.
 * @property enableXffClientPort Whether the X-Forwarded-For header should preserve the source port that the client used to connect to the load balancer in `application` load balancers. Defaults to `false`.
 * @property enforceSecurityGroupInboundRulesOnPrivateLinkTraffic Whether inbound security group rules are enforced for traffic originating from a PrivateLink. Only valid for Load Balancers of type `network`. The possible values are `on` and `off`.
 * @property idleTimeout Time in seconds that the connection is allowed to be idle. Only valid for Load Balancers of type `application`. Default: 60.
 * @property internal If true, the LB will be internal. Defaults to `false`.
 * @property ipAddressType Type of IP addresses used by the subnets for your load balancer. The possible values depend upon the load balancer type: `ipv4` (all load balancer types), `dualstack` (all load balancer types), and `dualstack-without-public-ipv4` (type `application` only).
 * @property loadBalancerType Type of load balancer to create. Possible values are `application`, `gateway`, or `network`. The default value is `application`.
 * @property name Name of the LB. This name must be unique within your AWS account, can have a maximum of 32 characters, must contain only alphanumeric characters or hyphens, and must not begin or end with a hyphen. If not specified, this provider will autogenerate a name beginning with `tf-lb`.
 * @property namePrefix Creates a unique name beginning with the specified prefix. Conflicts with `name`.
 * @property preserveHostHeader Whether the Application Load Balancer should preserve the Host header in the HTTP request and send it to the target without any change. Defaults to `false`.
 * @property securityGroups List of security group IDs to assign to the LB. Only valid for Load Balancers of type `application` or `network`. For load balancers of type `network` security groups cannot be added if none are currently present, and cannot all be removed once added. If either of these conditions are met, this will force a recreation of the resource.
 * @property subnetMappings Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
 * @property subnets List of subnet IDs to attach to the LB. For Load Balancers of type `network` subnets can only be added (see [Availability Zones](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#availability-zones)), deleting a subnet for load balancers of type `network` will force a recreation of the resource.
 * @property tags Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property xffHeaderProcessingMode Determines how the load balancer modifies the `X-Forwarded-For` header in the HTTP request before sending the request to the target. The possible values are `append`, `preserve`, and `remove`. Only valid for Load Balancers of type `application`. The default is `append`.
 */
public data class LoadBalancerArgs(
    public val accessLogs: Output<LoadBalancerAccessLogsArgs>? = null,
    public val clientKeepAlive: Output<Int>? = null,
    public val connectionLogs: Output<LoadBalancerConnectionLogsArgs>? = null,
    public val customerOwnedIpv4Pool: Output<String>? = null,
    public val desyncMitigationMode: Output<String>? = null,
    public val dnsRecordClientRoutingPolicy: Output<String>? = null,
    public val dropInvalidHeaderFields: Output<Boolean>? = null,
    public val enableCrossZoneLoadBalancing: Output<Boolean>? = null,
    public val enableDeletionProtection: Output<Boolean>? = null,
    public val enableHttp2: Output<Boolean>? = null,
    public val enableTlsVersionAndCipherSuiteHeaders: Output<Boolean>? = null,
    public val enableWafFailOpen: Output<Boolean>? = null,
    public val enableXffClientPort: Output<Boolean>? = null,
    public val enforceSecurityGroupInboundRulesOnPrivateLinkTraffic: Output<String>? = null,
    public val idleTimeout: Output<Int>? = null,
    public val `internal`: Output<Boolean>? = null,
    public val ipAddressType: Output<String>? = null,
    public val loadBalancerType: Output<String>? = null,
    public val name: Output<String>? = null,
    public val namePrefix: Output<String>? = null,
    public val preserveHostHeader: Output<Boolean>? = null,
    public val securityGroups: Output<List<String>>? = null,
    public val subnetMappings: Output<List<LoadBalancerSubnetMappingArgs>>? = null,
    public val subnets: Output<List<String>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val xffHeaderProcessingMode: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.lb.LoadBalancerArgs> {
    override fun toJava(): com.pulumi.aws.lb.LoadBalancerArgs =
        com.pulumi.aws.lb.LoadBalancerArgs.builder()
            .accessLogs(accessLogs?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .clientKeepAlive(clientKeepAlive?.applyValue({ args0 -> args0 }))
            .connectionLogs(connectionLogs?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .customerOwnedIpv4Pool(customerOwnedIpv4Pool?.applyValue({ args0 -> args0 }))
            .desyncMitigationMode(desyncMitigationMode?.applyValue({ args0 -> args0 }))
            .dnsRecordClientRoutingPolicy(dnsRecordClientRoutingPolicy?.applyValue({ args0 -> args0 }))
            .dropInvalidHeaderFields(dropInvalidHeaderFields?.applyValue({ args0 -> args0 }))
            .enableCrossZoneLoadBalancing(enableCrossZoneLoadBalancing?.applyValue({ args0 -> args0 }))
            .enableDeletionProtection(enableDeletionProtection?.applyValue({ args0 -> args0 }))
            .enableHttp2(enableHttp2?.applyValue({ args0 -> args0 }))
            .enableTlsVersionAndCipherSuiteHeaders(
                enableTlsVersionAndCipherSuiteHeaders?.applyValue({ args0 ->
                    args0
                }),
            )
            .enableWafFailOpen(enableWafFailOpen?.applyValue({ args0 -> args0 }))
            .enableXffClientPort(enableXffClientPort?.applyValue({ args0 -> args0 }))
            .enforceSecurityGroupInboundRulesOnPrivateLinkTraffic(
                enforceSecurityGroupInboundRulesOnPrivateLinkTraffic?.applyValue({ args0 ->
                    args0
                }),
            )
            .idleTimeout(idleTimeout?.applyValue({ args0 -> args0 }))
            .`internal`(`internal`?.applyValue({ args0 -> args0 }))
            .ipAddressType(ipAddressType?.applyValue({ args0 -> args0 }))
            .loadBalancerType(loadBalancerType?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .namePrefix(namePrefix?.applyValue({ args0 -> args0 }))
            .preserveHostHeader(preserveHostHeader?.applyValue({ args0 -> args0 }))
            .securityGroups(securityGroups?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .subnetMappings(
                subnetMappings?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.let({ args0 ->
                            args0.toJava()
                        })
                    })
                }),
            )
            .subnets(subnets?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .xffHeaderProcessingMode(xffHeaderProcessingMode?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [LoadBalancerArgs].
 */
@PulumiTagMarker
public class LoadBalancerArgsBuilder internal constructor() {
    private var accessLogs: Output<LoadBalancerAccessLogsArgs>? = null

    private var clientKeepAlive: Output<Int>? = null

    private var connectionLogs: Output<LoadBalancerConnectionLogsArgs>? = null

    private var customerOwnedIpv4Pool: Output<String>? = null

    private var desyncMitigationMode: Output<String>? = null

    private var dnsRecordClientRoutingPolicy: Output<String>? = null

    private var dropInvalidHeaderFields: Output<Boolean>? = null

    private var enableCrossZoneLoadBalancing: Output<Boolean>? = null

    private var enableDeletionProtection: Output<Boolean>? = null

    private var enableHttp2: Output<Boolean>? = null

    private var enableTlsVersionAndCipherSuiteHeaders: Output<Boolean>? = null

    private var enableWafFailOpen: Output<Boolean>? = null

    private var enableXffClientPort: Output<Boolean>? = null

    private var enforceSecurityGroupInboundRulesOnPrivateLinkTraffic: Output<String>? = null

    private var idleTimeout: Output<Int>? = null

    private var `internal`: Output<Boolean>? = null

    private var ipAddressType: Output<String>? = null

    private var loadBalancerType: Output<String>? = null

    private var name: Output<String>? = null

    private var namePrefix: Output<String>? = null

    private var preserveHostHeader: Output<Boolean>? = null

    private var securityGroups: Output<List<String>>? = null

    private var subnetMappings: Output<List<LoadBalancerSubnetMappingArgs>>? = null

    private var subnets: Output<List<String>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var xffHeaderProcessingMode: Output<String>? = null

    /**
     * @param value Access Logs block. See below.
     */
    @JvmName("qawoashjhmqorjjf")
    public suspend fun accessLogs(`value`: Output<LoadBalancerAccessLogsArgs>) {
        this.accessLogs = value
    }

    /**
     * @param value Client keep alive value in seconds. The valid range is 60-604800 seconds. The default is 3600 seconds.
     */
    @JvmName("lhikqqwbnrvrwcco")
    public suspend fun clientKeepAlive(`value`: Output<Int>) {
        this.clientKeepAlive = value
    }

    /**
     * @param value Connection Logs block. See below. Only valid for Load Balancers of type `application`.
     */
    @JvmName("rnpxxpsxihknoajr")
    public suspend fun connectionLogs(`value`: Output<LoadBalancerConnectionLogsArgs>) {
        this.connectionLogs = value
    }

    /**
     * @param value ID of the customer owned ipv4 pool to use for this load balancer.
     */
    @JvmName("qvyncklddcqmabol")
    public suspend fun customerOwnedIpv4Pool(`value`: Output<String>) {
        this.customerOwnedIpv4Pool = value
    }

    /**
     * @param value How the load balancer handles requests that might pose a security risk to an application due to HTTP desync. Valid values are `monitor`, `defensive` (default), `strictest`.
     */
    @JvmName("xhochkeaslnigqjm")
    public suspend fun desyncMitigationMode(`value`: Output<String>) {
        this.desyncMitigationMode = value
    }

    /**
     * @param value How traffic is distributed among the load balancer Availability Zones. Possible values are `any_availability_zone` (default), `availability_zone_affinity`, or `partial_availability_zone_affinity`. See   [Availability Zone DNS affinity](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#zonal-dns-affinity) for additional details. Only valid for `network` type load balancers.
     */
    @JvmName("qvbkwsygqhhauqti")
    public suspend fun dnsRecordClientRoutingPolicy(`value`: Output<String>) {
        this.dnsRecordClientRoutingPolicy = value
    }

    /**
     * @param value Whether HTTP headers with header fields that are not valid are removed by the load balancer (true) or routed to targets (false). The default is false. Elastic Load Balancing requires that message header names contain only alphanumeric characters and hyphens. Only valid for Load Balancers of type `application`.
     */
    @JvmName("hgiqvsehmdqkigtu")
    public suspend fun dropInvalidHeaderFields(`value`: Output<Boolean>) {
        this.dropInvalidHeaderFields = value
    }

    /**
     * @param value If true, cross-zone load balancing of the load balancer will be enabled. For `network` and `gateway` type load balancers, this feature is disabled by default (`false`). For `application` load balancer this feature is always enabled (`true`) and cannot be disabled. Defaults to `false`.
     */
    @JvmName("dryvoelawfsfjtnk")
    public suspend fun enableCrossZoneLoadBalancing(`value`: Output<Boolean>) {
        this.enableCrossZoneLoadBalancing = value
    }

    /**
     * @param value If true, deletion of the load balancer will be disabled via the AWS API. This will prevent this provider from deleting the load balancer. Defaults to `false`.
     */
    @JvmName("ybdrhspokohbywjf")
    public suspend fun enableDeletionProtection(`value`: Output<Boolean>) {
        this.enableDeletionProtection = value
    }

    /**
     * @param value Whether HTTP/2 is enabled in `application` load balancers. Defaults to `true`.
     */
    @JvmName("fkybvbtrujqebwtu")
    public suspend fun enableHttp2(`value`: Output<Boolean>) {
        this.enableHttp2 = value
    }

    /**
     * @param value Whether the two headers (`x-amzn-tls-version` and `x-amzn-tls-cipher-suite`), which contain information about the negotiated TLS version and cipher suite, are added to the client request before sending it to the target. Only valid for Load Balancers of type `application`. Defaults to `false`
     */
    @JvmName("lvjgmpxroiiageog")
    public suspend fun enableTlsVersionAndCipherSuiteHeaders(`value`: Output<Boolean>) {
        this.enableTlsVersionAndCipherSuiteHeaders = value
    }

    /**
     * @param value Whether to allow a WAF-enabled load balancer to route requests to targets if it is unable to forward the request to AWS WAF. Defaults to `false`.
     */
    @JvmName("prfkoebbummebtmo")
    public suspend fun enableWafFailOpen(`value`: Output<Boolean>) {
        this.enableWafFailOpen = value
    }

    /**
     * @param value Whether the X-Forwarded-For header should preserve the source port that the client used to connect to the load balancer in `application` load balancers. Defaults to `false`.
     */
    @JvmName("whoddwequviwtlug")
    public suspend fun enableXffClientPort(`value`: Output<Boolean>) {
        this.enableXffClientPort = value
    }

    /**
     * @param value Whether inbound security group rules are enforced for traffic originating from a PrivateLink. Only valid for Load Balancers of type `network`. The possible values are `on` and `off`.
     */
    @JvmName("nqfnauacmmhfjflh")
    public suspend fun enforceSecurityGroupInboundRulesOnPrivateLinkTraffic(`value`: Output<String>) {
        this.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic = value
    }

    /**
     * @param value Time in seconds that the connection is allowed to be idle. Only valid for Load Balancers of type `application`. Default: 60.
     */
    @JvmName("veifnundgekrrxlp")
    public suspend fun idleTimeout(`value`: Output<Int>) {
        this.idleTimeout = value
    }

    /**
     * @param value If true, the LB will be internal. Defaults to `false`.
     */
    @JvmName("vieulempmdggqykr")
    public suspend fun `internal`(`value`: Output<Boolean>) {
        this.`internal` = value
    }

    /**
     * @param value Type of IP addresses used by the subnets for your load balancer. The possible values depend upon the load balancer type: `ipv4` (all load balancer types), `dualstack` (all load balancer types), and `dualstack-without-public-ipv4` (type `application` only).
     */
    @JvmName("nigufbovmiubiqey")
    public suspend fun ipAddressType(`value`: Output<String>) {
        this.ipAddressType = value
    }

    /**
     * @param value Type of load balancer to create. Possible values are `application`, `gateway`, or `network`. The default value is `application`.
     */
    @JvmName("vlgoxkadurhsdcsk")
    public suspend fun loadBalancerType(`value`: Output<String>) {
        this.loadBalancerType = value
    }

    /**
     * @param value Name of the LB. This name must be unique within your AWS account, can have a maximum of 32 characters, must contain only alphanumeric characters or hyphens, and must not begin or end with a hyphen. If not specified, this provider will autogenerate a name beginning with `tf-lb`.
     */
    @JvmName("cmbffwxbalrgeyww")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`.
     */
    @JvmName("hrwlwsfxpexqarkw")
    public suspend fun namePrefix(`value`: Output<String>) {
        this.namePrefix = value
    }

    /**
     * @param value Whether the Application Load Balancer should preserve the Host header in the HTTP request and send it to the target without any change. Defaults to `false`.
     */
    @JvmName("uxcfkbowdpjjeuri")
    public suspend fun preserveHostHeader(`value`: Output<Boolean>) {
        this.preserveHostHeader = value
    }

    /**
     * @param value List of security group IDs to assign to the LB. Only valid for Load Balancers of type `application` or `network`. For load balancers of type `network` security groups cannot be added if none are currently present, and cannot all be removed once added. If either of these conditions are met, this will force a recreation of the resource.
     */
    @JvmName("quhwlyrjoslntghe")
    public suspend fun securityGroups(`value`: Output<List<String>>) {
        this.securityGroups = value
    }

    @JvmName("iefuyubghidjjbqw")
    public suspend fun securityGroups(vararg values: Output<String>) {
        this.securityGroups = Output.all(values.asList())
    }

    /**
     * @param values List of security group IDs to assign to the LB. Only valid for Load Balancers of type `application` or `network`. For load balancers of type `network` security groups cannot be added if none are currently present, and cannot all be removed once added. If either of these conditions are met, this will force a recreation of the resource.
     */
    @JvmName("xbpdulusbgnltdoy")
    public suspend fun securityGroups(values: List<Output<String>>) {
        this.securityGroups = Output.all(values)
    }

    /**
     * @param value Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
     */
    @JvmName("nebllsncmvcdjart")
    public suspend fun subnetMappings(`value`: Output<List<LoadBalancerSubnetMappingArgs>>) {
        this.subnetMappings = value
    }

    @JvmName("jvdftpnugmsnanmp")
    public suspend fun subnetMappings(vararg values: Output<LoadBalancerSubnetMappingArgs>) {
        this.subnetMappings = Output.all(values.asList())
    }

    /**
     * @param values Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
     */
    @JvmName("dwvyoepvosvuquea")
    public suspend fun subnetMappings(values: List<Output<LoadBalancerSubnetMappingArgs>>) {
        this.subnetMappings = Output.all(values)
    }

    /**
     * @param value List of subnet IDs to attach to the LB. For Load Balancers of type `network` subnets can only be added (see [Availability Zones](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#availability-zones)), deleting a subnet for load balancers of type `network` will force a recreation of the resource.
     */
    @JvmName("oljufmcicsevvvsq")
    public suspend fun subnets(`value`: Output<List<String>>) {
        this.subnets = value
    }

    @JvmName("tuptwecrrkrinwhh")
    public suspend fun subnets(vararg values: Output<String>) {
        this.subnets = Output.all(values.asList())
    }

    /**
     * @param values List of subnet IDs to attach to the LB. For Load Balancers of type `network` subnets can only be added (see [Availability Zones](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#availability-zones)), deleting a subnet for load balancers of type `network` will force a recreation of the resource.
     */
    @JvmName("httgitsgvbdchgnb")
    public suspend fun subnets(values: List<Output<String>>) {
        this.subnets = Output.all(values)
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ccfkdaqacahvrnfy")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Determines how the load balancer modifies the `X-Forwarded-For` header in the HTTP request before sending the request to the target. The possible values are `append`, `preserve`, and `remove`. Only valid for Load Balancers of type `application`. The default is `append`.
     */
    @JvmName("bowbicfcmlatptov")
    public suspend fun xffHeaderProcessingMode(`value`: Output<String>) {
        this.xffHeaderProcessingMode = value
    }

    /**
     * @param value Access Logs block. See below.
     */
    @JvmName("joeejmcjyngfcbnc")
    public suspend fun accessLogs(`value`: LoadBalancerAccessLogsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessLogs = mapped
    }

    /**
     * @param argument Access Logs block. See below.
     */
    @JvmName("gixkikwrtoiwmedi")
    public suspend fun accessLogs(argument: suspend LoadBalancerAccessLogsArgsBuilder.() -> Unit) {
        val toBeMapped = LoadBalancerAccessLogsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.accessLogs = mapped
    }

    /**
     * @param value Client keep alive value in seconds. The valid range is 60-604800 seconds. The default is 3600 seconds.
     */
    @JvmName("xfsimkvppfukhqqd")
    public suspend fun clientKeepAlive(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clientKeepAlive = mapped
    }

    /**
     * @param value Connection Logs block. See below. Only valid for Load Balancers of type `application`.
     */
    @JvmName("hqjcrarjcxmcxctq")
    public suspend fun connectionLogs(`value`: LoadBalancerConnectionLogsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.connectionLogs = mapped
    }

    /**
     * @param argument Connection Logs block. See below. Only valid for Load Balancers of type `application`.
     */
    @JvmName("hivsrycolwypnhev")
    public suspend fun connectionLogs(argument: suspend LoadBalancerConnectionLogsArgsBuilder.() -> Unit) {
        val toBeMapped = LoadBalancerConnectionLogsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.connectionLogs = mapped
    }

    /**
     * @param value ID of the customer owned ipv4 pool to use for this load balancer.
     */
    @JvmName("atroptqnbrfpcqhq")
    public suspend fun customerOwnedIpv4Pool(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerOwnedIpv4Pool = mapped
    }

    /**
     * @param value How the load balancer handles requests that might pose a security risk to an application due to HTTP desync. Valid values are `monitor`, `defensive` (default), `strictest`.
     */
    @JvmName("mynjjryatvvworhy")
    public suspend fun desyncMitigationMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.desyncMitigationMode = mapped
    }

    /**
     * @param value How traffic is distributed among the load balancer Availability Zones. Possible values are `any_availability_zone` (default), `availability_zone_affinity`, or `partial_availability_zone_affinity`. See   [Availability Zone DNS affinity](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#zonal-dns-affinity) for additional details. Only valid for `network` type load balancers.
     */
    @JvmName("hwtnphidccwkvddd")
    public suspend fun dnsRecordClientRoutingPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dnsRecordClientRoutingPolicy = mapped
    }

    /**
     * @param value Whether HTTP headers with header fields that are not valid are removed by the load balancer (true) or routed to targets (false). The default is false. Elastic Load Balancing requires that message header names contain only alphanumeric characters and hyphens. Only valid for Load Balancers of type `application`.
     */
    @JvmName("cyjwlivpnddmrpic")
    public suspend fun dropInvalidHeaderFields(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dropInvalidHeaderFields = mapped
    }

    /**
     * @param value If true, cross-zone load balancing of the load balancer will be enabled. For `network` and `gateway` type load balancers, this feature is disabled by default (`false`). For `application` load balancer this feature is always enabled (`true`) and cannot be disabled. Defaults to `false`.
     */
    @JvmName("emcineafmdgeikny")
    public suspend fun enableCrossZoneLoadBalancing(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableCrossZoneLoadBalancing = mapped
    }

    /**
     * @param value If true, deletion of the load balancer will be disabled via the AWS API. This will prevent this provider from deleting the load balancer. Defaults to `false`.
     */
    @JvmName("hxflskiivtlxuivh")
    public suspend fun enableDeletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableDeletionProtection = mapped
    }

    /**
     * @param value Whether HTTP/2 is enabled in `application` load balancers. Defaults to `true`.
     */
    @JvmName("wjnyuyygrwexwscq")
    public suspend fun enableHttp2(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableHttp2 = mapped
    }

    /**
     * @param value Whether the two headers (`x-amzn-tls-version` and `x-amzn-tls-cipher-suite`), which contain information about the negotiated TLS version and cipher suite, are added to the client request before sending it to the target. Only valid for Load Balancers of type `application`. Defaults to `false`
     */
    @JvmName("bchhgonmbegvgvyk")
    public suspend fun enableTlsVersionAndCipherSuiteHeaders(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableTlsVersionAndCipherSuiteHeaders = mapped
    }

    /**
     * @param value Whether to allow a WAF-enabled load balancer to route requests to targets if it is unable to forward the request to AWS WAF. Defaults to `false`.
     */
    @JvmName("jlohcuqmklsnohni")
    public suspend fun enableWafFailOpen(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableWafFailOpen = mapped
    }

    /**
     * @param value Whether the X-Forwarded-For header should preserve the source port that the client used to connect to the load balancer in `application` load balancers. Defaults to `false`.
     */
    @JvmName("vtgyaksqfgkrhxbd")
    public suspend fun enableXffClientPort(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableXffClientPort = mapped
    }

    /**
     * @param value Whether inbound security group rules are enforced for traffic originating from a PrivateLink. Only valid for Load Balancers of type `network`. The possible values are `on` and `off`.
     */
    @JvmName("bvhaalbsbybxsbxy")
    public suspend fun enforceSecurityGroupInboundRulesOnPrivateLinkTraffic(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic = mapped
    }

    /**
     * @param value Time in seconds that the connection is allowed to be idle. Only valid for Load Balancers of type `application`. Default: 60.
     */
    @JvmName("lyxxcjsghhyksxww")
    public suspend fun idleTimeout(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.idleTimeout = mapped
    }

    /**
     * @param value If true, the LB will be internal. Defaults to `false`.
     */
    @JvmName("ylifuqfqnbiaekqy")
    public suspend fun `internal`(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.`internal` = mapped
    }

    /**
     * @param value Type of IP addresses used by the subnets for your load balancer. The possible values depend upon the load balancer type: `ipv4` (all load balancer types), `dualstack` (all load balancer types), and `dualstack-without-public-ipv4` (type `application` only).
     */
    @JvmName("qqfpsgvqnlgyratj")
    public suspend fun ipAddressType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipAddressType = mapped
    }

    /**
     * @param value Type of load balancer to create. Possible values are `application`, `gateway`, or `network`. The default value is `application`.
     */
    @JvmName("rhnnnrgtcxjsqeap")
    public suspend fun loadBalancerType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.loadBalancerType = mapped
    }

    /**
     * @param value Name of the LB. This name must be unique within your AWS account, can have a maximum of 32 characters, must contain only alphanumeric characters or hyphens, and must not begin or end with a hyphen. If not specified, this provider will autogenerate a name beginning with `tf-lb`.
     */
    @JvmName("beleqgidlakmersw")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`.
     */
    @JvmName("nesmoemhivgvcxhq")
    public suspend fun namePrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namePrefix = mapped
    }

    /**
     * @param value Whether the Application Load Balancer should preserve the Host header in the HTTP request and send it to the target without any change. Defaults to `false`.
     */
    @JvmName("jjdnavvobliuovqr")
    public suspend fun preserveHostHeader(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preserveHostHeader = mapped
    }

    /**
     * @param value List of security group IDs to assign to the LB. Only valid for Load Balancers of type `application` or `network`. For load balancers of type `network` security groups cannot be added if none are currently present, and cannot all be removed once added. If either of these conditions are met, this will force a recreation of the resource.
     */
    @JvmName("qkwrlhhnlxmuunby")
    public suspend fun securityGroups(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroups = mapped
    }

    /**
     * @param values List of security group IDs to assign to the LB. Only valid for Load Balancers of type `application` or `network`. For load balancers of type `network` security groups cannot be added if none are currently present, and cannot all be removed once added. If either of these conditions are met, this will force a recreation of the resource.
     */
    @JvmName("pofpfjtcfhqsifce")
    public suspend fun securityGroups(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroups = mapped
    }

    /**
     * @param value Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
     */
    @JvmName("iuygrofricxxyaqf")
    public suspend fun subnetMappings(`value`: List<LoadBalancerSubnetMappingArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetMappings = mapped
    }

    /**
     * @param argument Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
     */
    @JvmName("hobbnlubtdpuwcrd")
    public suspend fun subnetMappings(argument: List<suspend LoadBalancerSubnetMappingArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            LoadBalancerSubnetMappingArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.subnetMappings = mapped
    }

    /**
     * @param argument Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
     */
    @JvmName("tfvktvpsjcxpeugm")
    public suspend fun subnetMappings(vararg argument: suspend LoadBalancerSubnetMappingArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            LoadBalancerSubnetMappingArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.subnetMappings = mapped
    }

    /**
     * @param argument Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
     */
    @JvmName("lifdybvrwlyedbgd")
    public suspend fun subnetMappings(argument: suspend LoadBalancerSubnetMappingArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(
            LoadBalancerSubnetMappingArgsBuilder().applySuspend {
                argument()
            }.build(),
        )
        val mapped = of(toBeMapped)
        this.subnetMappings = mapped
    }

    /**
     * @param values Subnet mapping block. See below. For Load Balancers of type `network` subnet mappings can only be added.
     */
    @JvmName("kebrkgnmfptloyae")
    public suspend fun subnetMappings(vararg values: LoadBalancerSubnetMappingArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.subnetMappings = mapped
    }

    /**
     * @param value List of subnet IDs to attach to the LB. For Load Balancers of type `network` subnets can only be added (see [Availability Zones](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#availability-zones)), deleting a subnet for load balancers of type `network` will force a recreation of the resource.
     */
    @JvmName("mgoaolwbeevfugqb")
    public suspend fun subnets(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnets = mapped
    }

    /**
     * @param values List of subnet IDs to attach to the LB. For Load Balancers of type `network` subnets can only be added (see [Availability Zones](https://docs.aws.amazon.com/elasticloadbalancing/latest/network/network-load-balancers.html#availability-zones)), deleting a subnet for load balancers of type `network` will force a recreation of the resource.
     */
    @JvmName("gcywnckqwowngwru")
    public suspend fun subnets(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.subnets = mapped
    }

    /**
     * @param value Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("tvlcjkrlmceubsoe")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("tgqdjxxqfdkbrjuk")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Determines how the load balancer modifies the `X-Forwarded-For` header in the HTTP request before sending the request to the target. The possible values are `append`, `preserve`, and `remove`. Only valid for Load Balancers of type `application`. The default is `append`.
     */
    @JvmName("iwlkvdwrocjrnvsr")
    public suspend fun xffHeaderProcessingMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.xffHeaderProcessingMode = mapped
    }

    internal fun build(): LoadBalancerArgs = LoadBalancerArgs(
        accessLogs = accessLogs,
        clientKeepAlive = clientKeepAlive,
        connectionLogs = connectionLogs,
        customerOwnedIpv4Pool = customerOwnedIpv4Pool,
        desyncMitigationMode = desyncMitigationMode,
        dnsRecordClientRoutingPolicy = dnsRecordClientRoutingPolicy,
        dropInvalidHeaderFields = dropInvalidHeaderFields,
        enableCrossZoneLoadBalancing = enableCrossZoneLoadBalancing,
        enableDeletionProtection = enableDeletionProtection,
        enableHttp2 = enableHttp2,
        enableTlsVersionAndCipherSuiteHeaders = enableTlsVersionAndCipherSuiteHeaders,
        enableWafFailOpen = enableWafFailOpen,
        enableXffClientPort = enableXffClientPort,
        enforceSecurityGroupInboundRulesOnPrivateLinkTraffic = enforceSecurityGroupInboundRulesOnPrivateLinkTraffic,
        idleTimeout = idleTimeout,
        `internal` = `internal`,
        ipAddressType = ipAddressType,
        loadBalancerType = loadBalancerType,
        name = name,
        namePrefix = namePrefix,
        preserveHostHeader = preserveHostHeader,
        securityGroups = securityGroups,
        subnetMappings = subnetMappings,
        subnets = subnets,
        tags = tags,
        xffHeaderProcessingMode = xffHeaderProcessingMode,
    )
}
