@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.lb.kotlin

import com.pulumi.aws.lb.TrustStoreRevocationArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a ELBv2 Trust Store Revocation for use with Application Load Balancer Listener resources.
 * ## Example Usage
 * ### Trust Store With Revocations
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const test = new aws.lb.TrustStore("test", {
 *     name: "tf-example-lb-ts",
 *     caCertificatesBundleS3Bucket: "...",
 *     caCertificatesBundleS3Key: "...",
 * });
 * const testTrustStoreRevocation = new aws.lb.TrustStoreRevocation("test", {
 *     trustStoreArn: test.arn,
 *     revocationsS3Bucket: "...",
 *     revocationsS3Key: "...",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test = aws.lb.TrustStore("test",
 *     name="tf-example-lb-ts",
 *     ca_certificates_bundle_s3_bucket="...",
 *     ca_certificates_bundle_s3_key="...")
 * test_trust_store_revocation = aws.lb.TrustStoreRevocation("test",
 *     trust_store_arn=test.arn,
 *     revocations_s3_bucket="...",
 *     revocations_s3_key="...")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var test = new Aws.LB.TrustStore("test", new()
 *     {
 *         Name = "tf-example-lb-ts",
 *         CaCertificatesBundleS3Bucket = "...",
 *         CaCertificatesBundleS3Key = "...",
 *     });
 *     var testTrustStoreRevocation = new Aws.LB.TrustStoreRevocation("test", new()
 *     {
 *         TrustStoreArn = test.Arn,
 *         RevocationsS3Bucket = "...",
 *         RevocationsS3Key = "...",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lb"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		test, err := lb.NewTrustStore(ctx, "test", &lb.TrustStoreArgs{
 * 			Name:                         pulumi.String("tf-example-lb-ts"),
 * 			CaCertificatesBundleS3Bucket: pulumi.String("..."),
 * 			CaCertificatesBundleS3Key:    pulumi.String("..."),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lb.NewTrustStoreRevocation(ctx, "test", &lb.TrustStoreRevocationArgs{
 * 			TrustStoreArn:       test.Arn,
 * 			RevocationsS3Bucket: pulumi.String("..."),
 * 			RevocationsS3Key:    pulumi.String("..."),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.lb.TrustStore;
 * import com.pulumi.aws.lb.TrustStoreArgs;
 * import com.pulumi.aws.lb.TrustStoreRevocation;
 * import com.pulumi.aws.lb.TrustStoreRevocationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new TrustStore("test", TrustStoreArgs.builder()
 *             .name("tf-example-lb-ts")
 *             .caCertificatesBundleS3Bucket("...")
 *             .caCertificatesBundleS3Key("...")
 *             .build());
 *         var testTrustStoreRevocation = new TrustStoreRevocation("testTrustStoreRevocation", TrustStoreRevocationArgs.builder()
 *             .trustStoreArn(test.arn())
 *             .revocationsS3Bucket("...")
 *             .revocationsS3Key("...")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   test:
 *     type: aws:lb:TrustStore
 *     properties:
 *       name: tf-example-lb-ts
 *       caCertificatesBundleS3Bucket: '...'
 *       caCertificatesBundleS3Key: '...'
 *   testTrustStoreRevocation:
 *     type: aws:lb:TrustStoreRevocation
 *     name: test
 *     properties:
 *       trustStoreArn: ${test.arn}
 *       revocationsS3Bucket: '...'
 *       revocationsS3Key: '...'
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Trust Store Revocations using their ARN. For example:
 * ```sh
 * $ pulumi import aws:lb/trustStoreRevocation:TrustStoreRevocation example arn:aws:elasticloadbalancing:us-west-2:187416307283:truststore/my-trust-store/20cfe21448b66314,6
 * ```
 * @property revocationsS3Bucket S3 Bucket name holding the client certificate CA bundle.
 * @property revocationsS3Key S3 object key holding the client certificate CA bundle.
 * @property revocationsS3ObjectVersion Version Id of CA bundle S3 bucket object, if versioned, defaults to latest if omitted.
 * @property trustStoreArn Trust Store ARN.
 */
public data class TrustStoreRevocationArgs(
    public val revocationsS3Bucket: Output<String>? = null,
    public val revocationsS3Key: Output<String>? = null,
    public val revocationsS3ObjectVersion: Output<String>? = null,
    public val trustStoreArn: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.lb.TrustStoreRevocationArgs> {
    override fun toJava(): com.pulumi.aws.lb.TrustStoreRevocationArgs =
        com.pulumi.aws.lb.TrustStoreRevocationArgs.builder()
            .revocationsS3Bucket(revocationsS3Bucket?.applyValue({ args0 -> args0 }))
            .revocationsS3Key(revocationsS3Key?.applyValue({ args0 -> args0 }))
            .revocationsS3ObjectVersion(revocationsS3ObjectVersion?.applyValue({ args0 -> args0 }))
            .trustStoreArn(trustStoreArn?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [TrustStoreRevocationArgs].
 */
@PulumiTagMarker
public class TrustStoreRevocationArgsBuilder internal constructor() {
    private var revocationsS3Bucket: Output<String>? = null

    private var revocationsS3Key: Output<String>? = null

    private var revocationsS3ObjectVersion: Output<String>? = null

    private var trustStoreArn: Output<String>? = null

    /**
     * @param value S3 Bucket name holding the client certificate CA bundle.
     */
    @JvmName("bqvusrwpeykwuhko")
    public suspend fun revocationsS3Bucket(`value`: Output<String>) {
        this.revocationsS3Bucket = value
    }

    /**
     * @param value S3 object key holding the client certificate CA bundle.
     */
    @JvmName("kbtnlunmqniyyhyg")
    public suspend fun revocationsS3Key(`value`: Output<String>) {
        this.revocationsS3Key = value
    }

    /**
     * @param value Version Id of CA bundle S3 bucket object, if versioned, defaults to latest if omitted.
     */
    @JvmName("etoyxvvifkbqvnfq")
    public suspend fun revocationsS3ObjectVersion(`value`: Output<String>) {
        this.revocationsS3ObjectVersion = value
    }

    /**
     * @param value Trust Store ARN.
     */
    @JvmName("ofrvkmjekxcfhtts")
    public suspend fun trustStoreArn(`value`: Output<String>) {
        this.trustStoreArn = value
    }

    /**
     * @param value S3 Bucket name holding the client certificate CA bundle.
     */
    @JvmName("twwjksonxkexaaci")
    public suspend fun revocationsS3Bucket(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.revocationsS3Bucket = mapped
    }

    /**
     * @param value S3 object key holding the client certificate CA bundle.
     */
    @JvmName("qvevjqqcohtsfoab")
    public suspend fun revocationsS3Key(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.revocationsS3Key = mapped
    }

    /**
     * @param value Version Id of CA bundle S3 bucket object, if versioned, defaults to latest if omitted.
     */
    @JvmName("pgfccdbqlddmynyt")
    public suspend fun revocationsS3ObjectVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.revocationsS3ObjectVersion = mapped
    }

    /**
     * @param value Trust Store ARN.
     */
    @JvmName("vstiyvrvufsulhmt")
    public suspend fun trustStoreArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.trustStoreArn = mapped
    }

    internal fun build(): TrustStoreRevocationArgs = TrustStoreRevocationArgs(
        revocationsS3Bucket = revocationsS3Bucket,
        revocationsS3Key = revocationsS3Key,
        revocationsS3ObjectVersion = revocationsS3ObjectVersion,
        trustStoreArn = trustStoreArn,
    )
}
