@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.lightsail.kotlin

import com.pulumi.aws.lightsail.LbAttachmentArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Attaches a Lightsail Instance to a Lightsail Load Balancer.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const available = aws.getAvailabilityZones({
 *     state: "available",
 *     filters: [{
 *         name: "opt-in-status",
 *         values: ["opt-in-not-required"],
 *     }],
 * });
 * const test = new aws.lightsail.Lb("test", {
 *     name: "test-load-balancer",
 *     healthCheckPath: "/",
 *     instancePort: 80,
 *     tags: {
 *         foo: "bar",
 *     },
 * });
 * const testInstance = new aws.lightsail.Instance("test", {
 *     name: "test-instance",
 *     availabilityZone: available.then(available => available.names?.[0]),
 *     blueprintId: "amazon_linux_2",
 *     bundleId: "nano_3_0",
 * });
 * const testLbAttachment = new aws.lightsail.LbAttachment("test", {
 *     lbName: test.name,
 *     instanceName: testInstance.name,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * available = aws.get_availability_zones(state="available",
 *     filters=[{
 *         "name": "opt-in-status",
 *         "values": ["opt-in-not-required"],
 *     }])
 * test = aws.lightsail.Lb("test",
 *     name="test-load-balancer",
 *     health_check_path="/",
 *     instance_port=80,
 *     tags={
 *         "foo": "bar",
 *     })
 * test_instance = aws.lightsail.Instance("test",
 *     name="test-instance",
 *     availability_zone=available.names[0],
 *     blueprint_id="amazon_linux_2",
 *     bundle_id="nano_3_0")
 * test_lb_attachment = aws.lightsail.LbAttachment("test",
 *     lb_name=test.name,
 *     instance_name=test_instance.name)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var available = Aws.GetAvailabilityZones.Invoke(new()
 *     {
 *         State = "available",
 *         Filters = new[]
 *         {
 *             new Aws.Inputs.GetAvailabilityZonesFilterInputArgs
 *             {
 *                 Name = "opt-in-status",
 *                 Values = new[]
 *                 {
 *                     "opt-in-not-required",
 *                 },
 *             },
 *         },
 *     });
 *     var test = new Aws.LightSail.Lb("test", new()
 *     {
 *         Name = "test-load-balancer",
 *         HealthCheckPath = "/",
 *         InstancePort = 80,
 *         Tags =
 *         {
 *             { "foo", "bar" },
 *         },
 *     });
 *     var testInstance = new Aws.LightSail.Instance("test", new()
 *     {
 *         Name = "test-instance",
 *         AvailabilityZone = available.Apply(getAvailabilityZonesResult => getAvailabilityZonesResult.Names[0]),
 *         BlueprintId = "amazon_linux_2",
 *         BundleId = "nano_3_0",
 *     });
 *     var testLbAttachment = new Aws.LightSail.LbAttachment("test", new()
 *     {
 *         LbName = test.Name,
 *         InstanceName = testInstance.Name,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/lightsail"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		available, err := aws.GetAvailabilityZones(ctx, &aws.GetAvailabilityZonesArgs{
 * 			State: pulumi.StringRef("available"),
 * 			Filters: []aws.GetAvailabilityZonesFilter{
 * 				{
 * 					Name: "opt-in-status",
 * 					Values: []string{
 * 						"opt-in-not-required",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		test, err := lightsail.NewLb(ctx, "test", &lightsail.LbArgs{
 * 			Name:            pulumi.String("test-load-balancer"),
 * 			HealthCheckPath: pulumi.String("/"),
 * 			InstancePort:    pulumi.Int(80),
 * 			Tags: pulumi.StringMap{
 * 				"foo": pulumi.String("bar"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		testInstance, err := lightsail.NewInstance(ctx, "test", &lightsail.InstanceArgs{
 * 			Name:             pulumi.String("test-instance"),
 * 			AvailabilityZone: pulumi.String(available.Names[0]),
 * 			BlueprintId:      pulumi.String("amazon_linux_2"),
 * 			BundleId:         pulumi.String("nano_3_0"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = lightsail.NewLbAttachment(ctx, "test", &lightsail.LbAttachmentArgs{
 * 			LbName:       test.Name,
 * 			InstanceName: testInstance.Name,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetAvailabilityZonesArgs;
 * import com.pulumi.aws.lightsail.Lb;
 * import com.pulumi.aws.lightsail.LbArgs;
 * import com.pulumi.aws.lightsail.Instance;
 * import com.pulumi.aws.lightsail.InstanceArgs;
 * import com.pulumi.aws.lightsail.LbAttachment;
 * import com.pulumi.aws.lightsail.LbAttachmentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var available = AwsFunctions.getAvailabilityZones(GetAvailabilityZonesArgs.builder()
 *             .state("available")
 *             .filters(GetAvailabilityZonesFilterArgs.builder()
 *                 .name("opt-in-status")
 *                 .values("opt-in-not-required")
 *                 .build())
 *             .build());
 *         var test = new Lb("test", LbArgs.builder()
 *             .name("test-load-balancer")
 *             .healthCheckPath("/")
 *             .instancePort("80")
 *             .tags(Map.of("foo", "bar"))
 *             .build());
 *         var testInstance = new Instance("testInstance", InstanceArgs.builder()
 *             .name("test-instance")
 *             .availabilityZone(available.applyValue(getAvailabilityZonesResult -> getAvailabilityZonesResult.names()[0]))
 *             .blueprintId("amazon_linux_2")
 *             .bundleId("nano_3_0")
 *             .build());
 *         var testLbAttachment = new LbAttachment("testLbAttachment", LbAttachmentArgs.builder()
 *             .lbName(test.name())
 *             .instanceName(testInstance.name())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   test:
 *     type: aws:lightsail:Lb
 *     properties:
 *       name: test-load-balancer
 *       healthCheckPath: /
 *       instancePort: '80'
 *       tags:
 *         foo: bar
 *   testInstance:
 *     type: aws:lightsail:Instance
 *     name: test
 *     properties:
 *       name: test-instance
 *       availabilityZone: ${available.names[0]}
 *       blueprintId: amazon_linux_2
 *       bundleId: nano_3_0
 *   testLbAttachment:
 *     type: aws:lightsail:LbAttachment
 *     name: test
 *     properties:
 *       lbName: ${test.name}
 *       instanceName: ${testInstance.name}
 * variables:
 *   available:
 *     fn::invoke:
 *       Function: aws:getAvailabilityZones
 *       Arguments:
 *         state: available
 *         filters:
 *           - name: opt-in-status
 *             values:
 *               - opt-in-not-required
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import `aws_lightsail_lb_attachment` using the name attribute. For example:
 * ```sh
 * $ pulumi import aws:lightsail/lbAttachment:LbAttachment test example-load-balancer,example-instance
 * ```
 * @property instanceName The name of the instance to attach to the load balancer.
 * @property lbName The name of the Lightsail load balancer.
 */
public data class LbAttachmentArgs(
    public val instanceName: Output<String>? = null,
    public val lbName: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.lightsail.LbAttachmentArgs> {
    override fun toJava(): com.pulumi.aws.lightsail.LbAttachmentArgs =
        com.pulumi.aws.lightsail.LbAttachmentArgs.builder()
            .instanceName(instanceName?.applyValue({ args0 -> args0 }))
            .lbName(lbName?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [LbAttachmentArgs].
 */
@PulumiTagMarker
public class LbAttachmentArgsBuilder internal constructor() {
    private var instanceName: Output<String>? = null

    private var lbName: Output<String>? = null

    /**
     * @param value The name of the instance to attach to the load balancer.
     */
    @JvmName("vamjejmfocvafwmf")
    public suspend fun instanceName(`value`: Output<String>) {
        this.instanceName = value
    }

    /**
     * @param value The name of the Lightsail load balancer.
     */
    @JvmName("tvnkvejlframybdk")
    public suspend fun lbName(`value`: Output<String>) {
        this.lbName = value
    }

    /**
     * @param value The name of the instance to attach to the load balancer.
     */
    @JvmName("crxvuceuvdqbohks")
    public suspend fun instanceName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceName = mapped
    }

    /**
     * @param value The name of the Lightsail load balancer.
     */
    @JvmName("kxnfhgldvgdixepp")
    public suspend fun lbName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.lbName = mapped
    }

    internal fun build(): LbAttachmentArgs = LbAttachmentArgs(
        instanceName = instanceName,
        lbName = lbName,
    )
}
