@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.mq.kotlin

import com.pulumi.aws.mq.BrokerArgs.builder
import com.pulumi.aws.mq.kotlin.inputs.BrokerConfigurationArgs
import com.pulumi.aws.mq.kotlin.inputs.BrokerConfigurationArgsBuilder
import com.pulumi.aws.mq.kotlin.inputs.BrokerEncryptionOptionsArgs
import com.pulumi.aws.mq.kotlin.inputs.BrokerEncryptionOptionsArgsBuilder
import com.pulumi.aws.mq.kotlin.inputs.BrokerLdapServerMetadataArgs
import com.pulumi.aws.mq.kotlin.inputs.BrokerLdapServerMetadataArgsBuilder
import com.pulumi.aws.mq.kotlin.inputs.BrokerLogsArgs
import com.pulumi.aws.mq.kotlin.inputs.BrokerLogsArgsBuilder
import com.pulumi.aws.mq.kotlin.inputs.BrokerMaintenanceWindowStartTimeArgs
import com.pulumi.aws.mq.kotlin.inputs.BrokerMaintenanceWindowStartTimeArgsBuilder
import com.pulumi.aws.mq.kotlin.inputs.BrokerUserArgs
import com.pulumi.aws.mq.kotlin.inputs.BrokerUserArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an Amazon MQ broker resource. This resources also manages users for the broker.
 * > For more information on Amazon MQ, see [Amazon MQ documentation](https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/welcome.html).
 * > **NOTE:** Amazon MQ currently places limits on **RabbitMQ** brokers. For example, a RabbitMQ broker cannot have: instances with an associated IP address of an ENI attached to the broker, an associated LDAP server to authenticate and authorize broker connections, storage type `EFS`, or audit logging. Although this resource allows you to create RabbitMQ users, RabbitMQ users cannot have console access or groups. Also, Amazon MQ does not return information about RabbitMQ users so drift detection is not possible.
 * > **NOTE:** Changes to an MQ Broker can occur when you change a parameter, such as `configuration` or `user`, and are reflected in the next maintenance window. Because of this, the provider may report a difference in its planning phase because a modification has not yet taken place. You can use the `apply_immediately` flag to instruct the service to apply the change immediately (see documentation below). Using `apply_immediately` can result in a brief downtime as the broker reboots.
 * ## Example Usage
 * ### Basic Example
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.mq.Broker("example", {
 *     brokerName: "example",
 *     configuration: {
 *         id: test.id,
 *         revision: test.latestRevision,
 *     },
 *     engineType: "ActiveMQ",
 *     engineVersion: "5.17.6",
 *     hostInstanceType: "mq.t2.micro",
 *     securityGroups: [testAwsSecurityGroup&#46;id],
 *     users: [{
 *         username: "ExampleUser",
 *         password: "MindTheGap",
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.mq.Broker("example",
 *     broker_name="example",
 *     configuration={
 *         "id": test["id"],
 *         "revision": test["latestRevision"],
 *     },
 *     engine_type="ActiveMQ",
 *     engine_version="5.17.6",
 *     host_instance_type="mq.t2.micro",
 *     security_groups=[test_aws_security_group["id"]],
 *     users=[{
 *         "username": "ExampleUser",
 *         "password": "MindTheGap",
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Mq.Broker("example", new()
 *     {
 *         BrokerName = "example",
 *         Configuration = new Aws.Mq.Inputs.BrokerConfigurationArgs
 *         {
 *             Id = test.Id,
 *             Revision = test.LatestRevision,
 *         },
 *         EngineType = "ActiveMQ",
 *         EngineVersion = "5.17.6",
 *         HostInstanceType = "mq.t2.micro",
 *         SecurityGroups = new[]
 *         {
 *             testAwsSecurityGroup.Id,
 *         },
 *         Users = new[]
 *         {
 *             new Aws.Mq.Inputs.BrokerUserArgs
 *             {
 *                 Username = "ExampleUser",
 *                 Password = "MindTheGap",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/mq"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := mq.NewBroker(ctx, "example", &mq.BrokerArgs{
 * 			BrokerName: pulumi.String("example"),
 * 			Configuration: &mq.BrokerConfigurationArgs{
 * 				Id:       pulumi.Any(test.Id),
 * 				Revision: pulumi.Any(test.LatestRevision),
 * 			},
 * 			EngineType:       pulumi.String("ActiveMQ"),
 * 			EngineVersion:    pulumi.String("5.17.6"),
 * 			HostInstanceType: pulumi.String("mq.t2.micro"),
 * 			SecurityGroups: pulumi.StringArray{
 * 				testAwsSecurityGroup.Id,
 * 			},
 * 			Users: mq.BrokerUserArray{
 * 				&mq.BrokerUserArgs{
 * 					Username: pulumi.String("ExampleUser"),
 * 					Password: pulumi.String("MindTheGap"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.mq.Broker;
 * import com.pulumi.aws.mq.BrokerArgs;
 * import com.pulumi.aws.mq.inputs.BrokerConfigurationArgs;
 * import com.pulumi.aws.mq.inputs.BrokerUserArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Broker("example", BrokerArgs.builder()
 *             .brokerName("example")
 *             .configuration(BrokerConfigurationArgs.builder()
 *                 .id(test.id())
 *                 .revision(test.latestRevision())
 *                 .build())
 *             .engineType("ActiveMQ")
 *             .engineVersion("5.17.6")
 *             .hostInstanceType("mq.t2.micro")
 *             .securityGroups(testAwsSecurityGroup.id())
 *             .users(BrokerUserArgs.builder()
 *                 .username("ExampleUser")
 *                 .password("MindTheGap")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:mq:Broker
 *     properties:
 *       brokerName: example
 *       configuration:
 *         id: ${test.id}
 *         revision: ${test.latestRevision}
 *       engineType: ActiveMQ
 *       engineVersion: 5.17.6
 *       hostInstanceType: mq.t2.micro
 *       securityGroups:
 *         - ${testAwsSecurityGroup.id}
 *       users:
 *         - username: ExampleUser
 *           password: MindTheGap
 * ```
 * <!--End PulumiCodeChooser -->
 * ### High-throughput Optimized Example
 * This example shows the use of EBS storage for high-throughput optimized performance.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.mq.Broker("example", {
 *     brokerName: "example",
 *     configuration: {
 *         id: test.id,
 *         revision: test.latestRevision,
 *     },
 *     engineType: "ActiveMQ",
 *     engineVersion: "5.17.6",
 *     storageType: "ebs",
 *     hostInstanceType: "mq.m5.large",
 *     securityGroups: [testAwsSecurityGroup&#46;id],
 *     users: [{
 *         username: "ExampleUser",
 *         password: "MindTheGap",
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.mq.Broker("example",
 *     broker_name="example",
 *     configuration={
 *         "id": test["id"],
 *         "revision": test["latestRevision"],
 *     },
 *     engine_type="ActiveMQ",
 *     engine_version="5.17.6",
 *     storage_type="ebs",
 *     host_instance_type="mq.m5.large",
 *     security_groups=[test_aws_security_group["id"]],
 *     users=[{
 *         "username": "ExampleUser",
 *         "password": "MindTheGap",
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Mq.Broker("example", new()
 *     {
 *         BrokerName = "example",
 *         Configuration = new Aws.Mq.Inputs.BrokerConfigurationArgs
 *         {
 *             Id = test.Id,
 *             Revision = test.LatestRevision,
 *         },
 *         EngineType = "ActiveMQ",
 *         EngineVersion = "5.17.6",
 *         StorageType = "ebs",
 *         HostInstanceType = "mq.m5.large",
 *         SecurityGroups = new[]
 *         {
 *             testAwsSecurityGroup.Id,
 *         },
 *         Users = new[]
 *         {
 *             new Aws.Mq.Inputs.BrokerUserArgs
 *             {
 *                 Username = "ExampleUser",
 *                 Password = "MindTheGap",
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/mq"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := mq.NewBroker(ctx, "example", &mq.BrokerArgs{
 * 			BrokerName: pulumi.String("example"),
 * 			Configuration: &mq.BrokerConfigurationArgs{
 * 				Id:       pulumi.Any(test.Id),
 * 				Revision: pulumi.Any(test.LatestRevision),
 * 			},
 * 			EngineType:       pulumi.String("ActiveMQ"),
 * 			EngineVersion:    pulumi.String("5.17.6"),
 * 			StorageType:      pulumi.String("ebs"),
 * 			HostInstanceType: pulumi.String("mq.m5.large"),
 * 			SecurityGroups: pulumi.StringArray{
 * 				testAwsSecurityGroup.Id,
 * 			},
 * 			Users: mq.BrokerUserArray{
 * 				&mq.BrokerUserArgs{
 * 					Username: pulumi.String("ExampleUser"),
 * 					Password: pulumi.String("MindTheGap"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.mq.Broker;
 * import com.pulumi.aws.mq.BrokerArgs;
 * import com.pulumi.aws.mq.inputs.BrokerConfigurationArgs;
 * import com.pulumi.aws.mq.inputs.BrokerUserArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Broker("example", BrokerArgs.builder()
 *             .brokerName("example")
 *             .configuration(BrokerConfigurationArgs.builder()
 *                 .id(test.id())
 *                 .revision(test.latestRevision())
 *                 .build())
 *             .engineType("ActiveMQ")
 *             .engineVersion("5.17.6")
 *             .storageType("ebs")
 *             .hostInstanceType("mq.m5.large")
 *             .securityGroups(testAwsSecurityGroup.id())
 *             .users(BrokerUserArgs.builder()
 *                 .username("ExampleUser")
 *                 .password("MindTheGap")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:mq:Broker
 *     properties:
 *       brokerName: example
 *       configuration:
 *         id: ${test.id}
 *         revision: ${test.latestRevision}
 *       engineType: ActiveMQ
 *       engineVersion: 5.17.6
 *       storageType: ebs
 *       hostInstanceType: mq.m5.large
 *       securityGroups:
 *         - ${testAwsSecurityGroup.id}
 *       users:
 *         - username: ExampleUser
 *           password: MindTheGap
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Cross-Region Data Replication
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const examplePrimary = new aws.mq.Broker("example_primary", {
 *     applyImmediately: true,
 *     brokerName: "example_primary",
 *     engineType: "ActiveMQ",
 *     engineVersion: "5.17.6",
 *     hostInstanceType: "mq.m5.large",
 *     securityGroups: [examplePrimaryAwsSecurityGroup&#46;id],
 *     deploymentMode: "ACTIVE_STANDBY_MULTI_AZ",
 *     users: [
 *         {
 *             username: "ExampleUser",
 *             password: "MindTheGap",
 *         },
 *         {
 *             username: "ExampleReplicationUser",
 *             password: "Example12345",
 *             replicationUser: true,
 *         },
 *     ],
 * });
 * const example = new aws.mq.Broker("example", {
 *     applyImmediately: true,
 *     brokerName: "example",
 *     engineType: "ActiveMQ",
 *     engineVersion: "5.17.6",
 *     hostInstanceType: "mq.m5.large",
 *     securityGroups: [exampleAwsSecurityGroup&#46;id],
 *     deploymentMode: "ACTIVE_STANDBY_MULTI_AZ",
 *     dataReplicationMode: "CRDR",
 *     dataReplicationPrimaryBrokerArn: primary.arn,
 *     users: [
 *         {
 *             username: "ExampleUser",
 *             password: "MindTheGap",
 *         },
 *         {
 *             username: "ExampleReplicationUser",
 *             password: "Example12345",
 *             replicationUser: true,
 *         },
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example_primary = aws.mq.Broker("example_primary",
 *     apply_immediately=True,
 *     broker_name="example_primary",
 *     engine_type="ActiveMQ",
 *     engine_version="5.17.6",
 *     host_instance_type="mq.m5.large",
 *     security_groups=[example_primary_aws_security_group["id"]],
 *     deployment_mode="ACTIVE_STANDBY_MULTI_AZ",
 *     users=[
 *         {
 *             "username": "ExampleUser",
 *             "password": "MindTheGap",
 *         },
 *         {
 *             "username": "ExampleReplicationUser",
 *             "password": "Example12345",
 *             "replication_user": True,
 *         },
 *     ])
 * example = aws.mq.Broker("example",
 *     apply_immediately=True,
 *     broker_name="example",
 *     engine_type="ActiveMQ",
 *     engine_version="5.17.6",
 *     host_instance_type="mq.m5.large",
 *     security_groups=[example_aws_security_group["id"]],
 *     deployment_mode="ACTIVE_STANDBY_MULTI_AZ",
 *     data_replication_mode="CRDR",
 *     data_replication_primary_broker_arn=primary["arn"],
 *     users=[
 *         {
 *             "username": "ExampleUser",
 *             "password": "MindTheGap",
 *         },
 *         {
 *             "username": "ExampleReplicationUser",
 *             "password": "Example12345",
 *             "replication_user": True,
 *         },
 *     ])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var examplePrimary = new Aws.Mq.Broker("example_primary", new()
 *     {
 *         ApplyImmediately = true,
 *         BrokerName = "example_primary",
 *         EngineType = "ActiveMQ",
 *         EngineVersion = "5.17.6",
 *         HostInstanceType = "mq.m5.large",
 *         SecurityGroups = new[]
 *         {
 *             examplePrimaryAwsSecurityGroup.Id,
 *         },
 *         DeploymentMode = "ACTIVE_STANDBY_MULTI_AZ",
 *         Users = new[]
 *         {
 *             new Aws.Mq.Inputs.BrokerUserArgs
 *             {
 *                 Username = "ExampleUser",
 *                 Password = "MindTheGap",
 *             },
 *             new Aws.Mq.Inputs.BrokerUserArgs
 *             {
 *                 Username = "ExampleReplicationUser",
 *                 Password = "Example12345",
 *                 ReplicationUser = true,
 *             },
 *         },
 *     });
 *     var example = new Aws.Mq.Broker("example", new()
 *     {
 *         ApplyImmediately = true,
 *         BrokerName = "example",
 *         EngineType = "ActiveMQ",
 *         EngineVersion = "5.17.6",
 *         HostInstanceType = "mq.m5.large",
 *         SecurityGroups = new[]
 *         {
 *             exampleAwsSecurityGroup.Id,
 *         },
 *         DeploymentMode = "ACTIVE_STANDBY_MULTI_AZ",
 *         DataReplicationMode = "CRDR",
 *         DataReplicationPrimaryBrokerArn = primary.Arn,
 *         Users = new[]
 *         {
 *             new Aws.Mq.Inputs.BrokerUserArgs
 *             {
 *                 Username = "ExampleUser",
 *                 Password = "MindTheGap",
 *             },
 *             new Aws.Mq.Inputs.BrokerUserArgs
 *             {
 *                 Username = "ExampleReplicationUser",
 *                 Password = "Example12345",
 *                 ReplicationUser = true,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/mq"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := mq.NewBroker(ctx, "example_primary", &mq.BrokerArgs{
 * 			ApplyImmediately: pulumi.Bool(true),
 * 			BrokerName:       pulumi.String("example_primary"),
 * 			EngineType:       pulumi.String("ActiveMQ"),
 * 			EngineVersion:    pulumi.String("5.17.6"),
 * 			HostInstanceType: pulumi.String("mq.m5.large"),
 * 			SecurityGroups: pulumi.StringArray{
 * 				examplePrimaryAwsSecurityGroup.Id,
 * 			},
 * 			DeploymentMode: pulumi.String("ACTIVE_STANDBY_MULTI_AZ"),
 * 			Users: mq.BrokerUserArray{
 * 				&mq.BrokerUserArgs{
 * 					Username: pulumi.String("ExampleUser"),
 * 					Password: pulumi.String("MindTheGap"),
 * 				},
 * 				&mq.BrokerUserArgs{
 * 					Username:        pulumi.String("ExampleReplicationUser"),
 * 					Password:        pulumi.String("Example12345"),
 * 					ReplicationUser: pulumi.Bool(true),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = mq.NewBroker(ctx, "example", &mq.BrokerArgs{
 * 			ApplyImmediately: pulumi.Bool(true),
 * 			BrokerName:       pulumi.String("example"),
 * 			EngineType:       pulumi.String("ActiveMQ"),
 * 			EngineVersion:    pulumi.String("5.17.6"),
 * 			HostInstanceType: pulumi.String("mq.m5.large"),
 * 			SecurityGroups: pulumi.StringArray{
 * 				exampleAwsSecurityGroup.Id,
 * 			},
 * 			DeploymentMode:                  pulumi.String("ACTIVE_STANDBY_MULTI_AZ"),
 * 			DataReplicationMode:             pulumi.String("CRDR"),
 * 			DataReplicationPrimaryBrokerArn: pulumi.Any(primary.Arn),
 * 			Users: mq.BrokerUserArray{
 * 				&mq.BrokerUserArgs{
 * 					Username: pulumi.String("ExampleUser"),
 * 					Password: pulumi.String("MindTheGap"),
 * 				},
 * 				&mq.BrokerUserArgs{
 * 					Username:        pulumi.String("ExampleReplicationUser"),
 * 					Password:        pulumi.String("Example12345"),
 * 					ReplicationUser: pulumi.Bool(true),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.mq.Broker;
 * import com.pulumi.aws.mq.BrokerArgs;
 * import com.pulumi.aws.mq.inputs.BrokerUserArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var examplePrimary = new Broker("examplePrimary", BrokerArgs.builder()
 *             .applyImmediately(true)
 *             .brokerName("example_primary")
 *             .engineType("ActiveMQ")
 *             .engineVersion("5.17.6")
 *             .hostInstanceType("mq.m5.large")
 *             .securityGroups(examplePrimaryAwsSecurityGroup.id())
 *             .deploymentMode("ACTIVE_STANDBY_MULTI_AZ")
 *             .users(
 *                 BrokerUserArgs.builder()
 *                     .username("ExampleUser")
 *                     .password("MindTheGap")
 *                     .build(),
 *                 BrokerUserArgs.builder()
 *                     .username("ExampleReplicationUser")
 *                     .password("Example12345")
 *                     .replicationUser(true)
 *                     .build())
 *             .build());
 *         var example = new Broker("example", BrokerArgs.builder()
 *             .applyImmediately(true)
 *             .brokerName("example")
 *             .engineType("ActiveMQ")
 *             .engineVersion("5.17.6")
 *             .hostInstanceType("mq.m5.large")
 *             .securityGroups(exampleAwsSecurityGroup.id())
 *             .deploymentMode("ACTIVE_STANDBY_MULTI_AZ")
 *             .dataReplicationMode("CRDR")
 *             .dataReplicationPrimaryBrokerArn(primary.arn())
 *             .users(
 *                 BrokerUserArgs.builder()
 *                     .username("ExampleUser")
 *                     .password("MindTheGap")
 *                     .build(),
 *                 BrokerUserArgs.builder()
 *                     .username("ExampleReplicationUser")
 *                     .password("Example12345")
 *                     .replicationUser(true)
 *                     .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplePrimary:
 *     type: aws:mq:Broker
 *     name: example_primary
 *     properties:
 *       applyImmediately: true
 *       brokerName: example_primary
 *       engineType: ActiveMQ
 *       engineVersion: 5.17.6
 *       hostInstanceType: mq.m5.large
 *       securityGroups:
 *         - ${examplePrimaryAwsSecurityGroup.id}
 *       deploymentMode: ACTIVE_STANDBY_MULTI_AZ
 *       users:
 *         - username: ExampleUser
 *           password: MindTheGap
 *         - username: ExampleReplicationUser
 *           password: Example12345
 *           replicationUser: true
 *   example:
 *     type: aws:mq:Broker
 *     properties:
 *       applyImmediately: true
 *       brokerName: example
 *       engineType: ActiveMQ
 *       engineVersion: 5.17.6
 *       hostInstanceType: mq.m5.large
 *       securityGroups:
 *         - ${exampleAwsSecurityGroup.id}
 *       deploymentMode: ACTIVE_STANDBY_MULTI_AZ
 *       dataReplicationMode: CRDR
 *       dataReplicationPrimaryBrokerArn: ${primary.arn}
 *       users:
 *         - username: ExampleUser
 *           password: MindTheGap
 *         - username: ExampleReplicationUser
 *           password: Example12345
 *           replicationUser: true
 * ```
 * <!--End PulumiCodeChooser -->
 * See the [AWS MQ documentation](https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/crdr-for-active-mq.html) on cross-region data replication for additional details.
 * ## Import
 * Using `pulumi import`, import MQ Brokers using their broker id. For example:
 * ```sh
 * $ pulumi import aws:mq/broker:Broker example a1b2c3d4-d5f6-7777-8888-9999aaaabbbbcccc
 * ```
 * @property applyImmediately Specifies whether any broker modifications are applied immediately, or during the next maintenance window. Default is `false`.
 * @property authenticationStrategy Authentication strategy used to secure the broker. Valid values are `simple` and `ldap`. `ldap` is not supported for `engine_type` `RabbitMQ`.
 * @property autoMinorVersionUpgrade Whether to automatically upgrade to new minor versions of brokers as Amazon MQ makes releases available.
 * @property brokerName Name of the broker.
 * @property configuration Configuration block for broker configuration. Applies to `engine_type` of `ActiveMQ` and `RabbitMQ` only. Detailed below.
 * @property dataReplicationMode Defines whether this broker is a part of a data replication pair. Valid values are `CRDR` and `NONE`.
 * @property dataReplicationPrimaryBrokerArn The Amazon Resource Name (ARN) of the primary broker that is used to replicate data from in a data replication pair, and is applied to the replica broker. Must be set when `data_replication_mode` is `CRDR`.
 * @property deploymentMode Deployment mode of the broker. Valid values are `SINGLE_INSTANCE`, `ACTIVE_STANDBY_MULTI_AZ`, and `CLUSTER_MULTI_AZ`. Default is `SINGLE_INSTANCE`.
 * @property encryptionOptions Configuration block containing encryption options. Detailed below.
 * @property engineType Type of broker engine. Valid values are `ActiveMQ` and `RabbitMQ`.
 * @property engineVersion Version of the broker engine. See the [AmazonMQ Broker Engine docs](https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/broker-engine.html) for supported versions. For example, `5.17.6`.
 * @property hostInstanceType Broker's instance type. For example, `mq.t3.micro`, `mq.m5.large`.
 * @property ldapServerMetadata Configuration block for the LDAP server used to authenticate and authorize connections to the broker. Not supported for `engine_type` `RabbitMQ`. Detailed below. (Currently, AWS may not process changes to LDAP server metadata.)
 * @property logs Configuration block for the logging configuration of the broker. Detailed below.
 * @property maintenanceWindowStartTime Configuration block for the maintenance window start time. Detailed below.
 * @property publiclyAccessible Whether to enable connections from applications outside of the VPC that hosts the broker's subnets.
 * @property securityGroups List of security group IDs assigned to the broker.
 * @property storageType Storage type of the broker. For `engine_type` `ActiveMQ`, the valid values are `efs` and `ebs`, and the AWS-default is `efs`. For `engine_type` `RabbitMQ`, only `ebs` is supported. When using `ebs`, only the `mq.m5` broker instance type family is supported.
 * @property subnetIds List of subnet IDs in which to launch the broker. A `SINGLE_INSTANCE` deployment requires one subnet. An `ACTIVE_STANDBY_MULTI_AZ` deployment requires multiple subnets.
 * @property tags Map of tags to assign to the broker. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property users Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
 * The following arguments are optional:
 */
public data class BrokerArgs(
    public val applyImmediately: Output<Boolean>? = null,
    public val authenticationStrategy: Output<String>? = null,
    public val autoMinorVersionUpgrade: Output<Boolean>? = null,
    public val brokerName: Output<String>? = null,
    public val configuration: Output<BrokerConfigurationArgs>? = null,
    public val dataReplicationMode: Output<String>? = null,
    public val dataReplicationPrimaryBrokerArn: Output<String>? = null,
    public val deploymentMode: Output<String>? = null,
    public val encryptionOptions: Output<BrokerEncryptionOptionsArgs>? = null,
    public val engineType: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val hostInstanceType: Output<String>? = null,
    public val ldapServerMetadata: Output<BrokerLdapServerMetadataArgs>? = null,
    public val logs: Output<BrokerLogsArgs>? = null,
    public val maintenanceWindowStartTime: Output<BrokerMaintenanceWindowStartTimeArgs>? = null,
    public val publiclyAccessible: Output<Boolean>? = null,
    public val securityGroups: Output<List<String>>? = null,
    public val storageType: Output<String>? = null,
    public val subnetIds: Output<List<String>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val users: Output<List<BrokerUserArgs>>? = null,
) : ConvertibleToJava<com.pulumi.aws.mq.BrokerArgs> {
    override fun toJava(): com.pulumi.aws.mq.BrokerArgs = com.pulumi.aws.mq.BrokerArgs.builder()
        .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
        .authenticationStrategy(authenticationStrategy?.applyValue({ args0 -> args0 }))
        .autoMinorVersionUpgrade(autoMinorVersionUpgrade?.applyValue({ args0 -> args0 }))
        .brokerName(brokerName?.applyValue({ args0 -> args0 }))
        .configuration(configuration?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .dataReplicationMode(dataReplicationMode?.applyValue({ args0 -> args0 }))
        .dataReplicationPrimaryBrokerArn(dataReplicationPrimaryBrokerArn?.applyValue({ args0 -> args0 }))
        .deploymentMode(deploymentMode?.applyValue({ args0 -> args0 }))
        .encryptionOptions(encryptionOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .engineType(engineType?.applyValue({ args0 -> args0 }))
        .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
        .hostInstanceType(hostInstanceType?.applyValue({ args0 -> args0 }))
        .ldapServerMetadata(
            ldapServerMetadata?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .logs(logs?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .maintenanceWindowStartTime(
            maintenanceWindowStartTime?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .publiclyAccessible(publiclyAccessible?.applyValue({ args0 -> args0 }))
        .securityGroups(securityGroups?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .storageType(storageType?.applyValue({ args0 -> args0 }))
        .subnetIds(subnetIds?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .users(
            users?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        ).build()
}

/**
 * Builder for [BrokerArgs].
 */
@PulumiTagMarker
public class BrokerArgsBuilder internal constructor() {
    private var applyImmediately: Output<Boolean>? = null

    private var authenticationStrategy: Output<String>? = null

    private var autoMinorVersionUpgrade: Output<Boolean>? = null

    private var brokerName: Output<String>? = null

    private var configuration: Output<BrokerConfigurationArgs>? = null

    private var dataReplicationMode: Output<String>? = null

    private var dataReplicationPrimaryBrokerArn: Output<String>? = null

    private var deploymentMode: Output<String>? = null

    private var encryptionOptions: Output<BrokerEncryptionOptionsArgs>? = null

    private var engineType: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var hostInstanceType: Output<String>? = null

    private var ldapServerMetadata: Output<BrokerLdapServerMetadataArgs>? = null

    private var logs: Output<BrokerLogsArgs>? = null

    private var maintenanceWindowStartTime: Output<BrokerMaintenanceWindowStartTimeArgs>? = null

    private var publiclyAccessible: Output<Boolean>? = null

    private var securityGroups: Output<List<String>>? = null

    private var storageType: Output<String>? = null

    private var subnetIds: Output<List<String>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var users: Output<List<BrokerUserArgs>>? = null

    /**
     * @param value Specifies whether any broker modifications are applied immediately, or during the next maintenance window. Default is `false`.
     */
    @JvmName("evtjmncxdhrfeehd")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value Authentication strategy used to secure the broker. Valid values are `simple` and `ldap`. `ldap` is not supported for `engine_type` `RabbitMQ`.
     */
    @JvmName("troeqpbjigdpasqb")
    public suspend fun authenticationStrategy(`value`: Output<String>) {
        this.authenticationStrategy = value
    }

    /**
     * @param value Whether to automatically upgrade to new minor versions of brokers as Amazon MQ makes releases available.
     */
    @JvmName("itaeykdvkbsbalfv")
    public suspend fun autoMinorVersionUpgrade(`value`: Output<Boolean>) {
        this.autoMinorVersionUpgrade = value
    }

    /**
     * @param value Name of the broker.
     */
    @JvmName("wcrsipkrisdyddis")
    public suspend fun brokerName(`value`: Output<String>) {
        this.brokerName = value
    }

    /**
     * @param value Configuration block for broker configuration. Applies to `engine_type` of `ActiveMQ` and `RabbitMQ` only. Detailed below.
     */
    @JvmName("xtghhchjmpmkmmee")
    public suspend fun configuration(`value`: Output<BrokerConfigurationArgs>) {
        this.configuration = value
    }

    /**
     * @param value Defines whether this broker is a part of a data replication pair. Valid values are `CRDR` and `NONE`.
     */
    @JvmName("mycdiposmmwjjsof")
    public suspend fun dataReplicationMode(`value`: Output<String>) {
        this.dataReplicationMode = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the primary broker that is used to replicate data from in a data replication pair, and is applied to the replica broker. Must be set when `data_replication_mode` is `CRDR`.
     */
    @JvmName("rhcvroxlcmwdousq")
    public suspend fun dataReplicationPrimaryBrokerArn(`value`: Output<String>) {
        this.dataReplicationPrimaryBrokerArn = value
    }

    /**
     * @param value Deployment mode of the broker. Valid values are `SINGLE_INSTANCE`, `ACTIVE_STANDBY_MULTI_AZ`, and `CLUSTER_MULTI_AZ`. Default is `SINGLE_INSTANCE`.
     */
    @JvmName("ecqaotuungkoxsax")
    public suspend fun deploymentMode(`value`: Output<String>) {
        this.deploymentMode = value
    }

    /**
     * @param value Configuration block containing encryption options. Detailed below.
     */
    @JvmName("ycclnnfyikbiatsj")
    public suspend fun encryptionOptions(`value`: Output<BrokerEncryptionOptionsArgs>) {
        this.encryptionOptions = value
    }

    /**
     * @param value Type of broker engine. Valid values are `ActiveMQ` and `RabbitMQ`.
     */
    @JvmName("uklsqkysxvvybdln")
    public suspend fun engineType(`value`: Output<String>) {
        this.engineType = value
    }

    /**
     * @param value Version of the broker engine. See the [AmazonMQ Broker Engine docs](https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/broker-engine.html) for supported versions. For example, `5.17.6`.
     */
    @JvmName("ksucdskyxctletyj")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Broker's instance type. For example, `mq.t3.micro`, `mq.m5.large`.
     */
    @JvmName("mdhopfynheryvqhm")
    public suspend fun hostInstanceType(`value`: Output<String>) {
        this.hostInstanceType = value
    }

    /**
     * @param value Configuration block for the LDAP server used to authenticate and authorize connections to the broker. Not supported for `engine_type` `RabbitMQ`. Detailed below. (Currently, AWS may not process changes to LDAP server metadata.)
     */
    @JvmName("ythmqlwksgodmnvj")
    public suspend fun ldapServerMetadata(`value`: Output<BrokerLdapServerMetadataArgs>) {
        this.ldapServerMetadata = value
    }

    /**
     * @param value Configuration block for the logging configuration of the broker. Detailed below.
     */
    @JvmName("vaowwguepyaxrsyi")
    public suspend fun logs(`value`: Output<BrokerLogsArgs>) {
        this.logs = value
    }

    /**
     * @param value Configuration block for the maintenance window start time. Detailed below.
     */
    @JvmName("nxtuwrbovfytjdmp")
    public suspend fun maintenanceWindowStartTime(`value`: Output<BrokerMaintenanceWindowStartTimeArgs>) {
        this.maintenanceWindowStartTime = value
    }

    /**
     * @param value Whether to enable connections from applications outside of the VPC that hosts the broker's subnets.
     */
    @JvmName("bfihnkepdxiykfbw")
    public suspend fun publiclyAccessible(`value`: Output<Boolean>) {
        this.publiclyAccessible = value
    }

    /**
     * @param value List of security group IDs assigned to the broker.
     */
    @JvmName("oahtxalxxyxtxmkt")
    public suspend fun securityGroups(`value`: Output<List<String>>) {
        this.securityGroups = value
    }

    @JvmName("vjrvpfeswnvdffim")
    public suspend fun securityGroups(vararg values: Output<String>) {
        this.securityGroups = Output.all(values.asList())
    }

    /**
     * @param values List of security group IDs assigned to the broker.
     */
    @JvmName("strmlonfwgdqxqdo")
    public suspend fun securityGroups(values: List<Output<String>>) {
        this.securityGroups = Output.all(values)
    }

    /**
     * @param value Storage type of the broker. For `engine_type` `ActiveMQ`, the valid values are `efs` and `ebs`, and the AWS-default is `efs`. For `engine_type` `RabbitMQ`, only `ebs` is supported. When using `ebs`, only the `mq.m5` broker instance type family is supported.
     */
    @JvmName("tocsalfvagmjrlgh")
    public suspend fun storageType(`value`: Output<String>) {
        this.storageType = value
    }

    /**
     * @param value List of subnet IDs in which to launch the broker. A `SINGLE_INSTANCE` deployment requires one subnet. An `ACTIVE_STANDBY_MULTI_AZ` deployment requires multiple subnets.
     */
    @JvmName("flmoqyejloqghemg")
    public suspend fun subnetIds(`value`: Output<List<String>>) {
        this.subnetIds = value
    }

    @JvmName("ucrlanrsbstxahji")
    public suspend fun subnetIds(vararg values: Output<String>) {
        this.subnetIds = Output.all(values.asList())
    }

    /**
     * @param values List of subnet IDs in which to launch the broker. A `SINGLE_INSTANCE` deployment requires one subnet. An `ACTIVE_STANDBY_MULTI_AZ` deployment requires multiple subnets.
     */
    @JvmName("pbhplrwvinplynin")
    public suspend fun subnetIds(values: List<Output<String>>) {
        this.subnetIds = Output.all(values)
    }

    /**
     * @param value Map of tags to assign to the broker. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("bqeyprtwrbmraaxc")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
     * The following arguments are optional:
     */
    @JvmName("trwmjvbsqsfciepl")
    public suspend fun users(`value`: Output<List<BrokerUserArgs>>) {
        this.users = value
    }

    @JvmName("tepqafuwcmukumdp")
    public suspend fun users(vararg values: Output<BrokerUserArgs>) {
        this.users = Output.all(values.asList())
    }

    /**
     * @param values Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
     * The following arguments are optional:
     */
    @JvmName("kidhggserqiubbmh")
    public suspend fun users(values: List<Output<BrokerUserArgs>>) {
        this.users = Output.all(values)
    }

    /**
     * @param value Specifies whether any broker modifications are applied immediately, or during the next maintenance window. Default is `false`.
     */
    @JvmName("hygtprcvvbnjiitv")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value Authentication strategy used to secure the broker. Valid values are `simple` and `ldap`. `ldap` is not supported for `engine_type` `RabbitMQ`.
     */
    @JvmName("rgejqpigrxisjnah")
    public suspend fun authenticationStrategy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.authenticationStrategy = mapped
    }

    /**
     * @param value Whether to automatically upgrade to new minor versions of brokers as Amazon MQ makes releases available.
     */
    @JvmName("xttxwdbdtyyghakb")
    public suspend fun autoMinorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoMinorVersionUpgrade = mapped
    }

    /**
     * @param value Name of the broker.
     */
    @JvmName("qexrniniytjvgson")
    public suspend fun brokerName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.brokerName = mapped
    }

    /**
     * @param value Configuration block for broker configuration. Applies to `engine_type` of `ActiveMQ` and `RabbitMQ` only. Detailed below.
     */
    @JvmName("qtmnssprofhewjtc")
    public suspend fun configuration(`value`: BrokerConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.configuration = mapped
    }

    /**
     * @param argument Configuration block for broker configuration. Applies to `engine_type` of `ActiveMQ` and `RabbitMQ` only. Detailed below.
     */
    @JvmName("fstpniswbegfksrm")
    public suspend fun configuration(argument: suspend BrokerConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = BrokerConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.configuration = mapped
    }

    /**
     * @param value Defines whether this broker is a part of a data replication pair. Valid values are `CRDR` and `NONE`.
     */
    @JvmName("dgliodsmxxyuxgsv")
    public suspend fun dataReplicationMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dataReplicationMode = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the primary broker that is used to replicate data from in a data replication pair, and is applied to the replica broker. Must be set when `data_replication_mode` is `CRDR`.
     */
    @JvmName("hdithqkddduuaifk")
    public suspend fun dataReplicationPrimaryBrokerArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dataReplicationPrimaryBrokerArn = mapped
    }

    /**
     * @param value Deployment mode of the broker. Valid values are `SINGLE_INSTANCE`, `ACTIVE_STANDBY_MULTI_AZ`, and `CLUSTER_MULTI_AZ`. Default is `SINGLE_INSTANCE`.
     */
    @JvmName("ljuwnujelguhjxlk")
    public suspend fun deploymentMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deploymentMode = mapped
    }

    /**
     * @param value Configuration block containing encryption options. Detailed below.
     */
    @JvmName("vrfemuufreuouyjf")
    public suspend fun encryptionOptions(`value`: BrokerEncryptionOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.encryptionOptions = mapped
    }

    /**
     * @param argument Configuration block containing encryption options. Detailed below.
     */
    @JvmName("ubtscpgnckqngjjf")
    public suspend fun encryptionOptions(argument: suspend BrokerEncryptionOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = BrokerEncryptionOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.encryptionOptions = mapped
    }

    /**
     * @param value Type of broker engine. Valid values are `ActiveMQ` and `RabbitMQ`.
     */
    @JvmName("kslrxkdxjcdyybcw")
    public suspend fun engineType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineType = mapped
    }

    /**
     * @param value Version of the broker engine. See the [AmazonMQ Broker Engine docs](https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/broker-engine.html) for supported versions. For example, `5.17.6`.
     */
    @JvmName("lmgvhbbuovkdfvbq")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Broker's instance type. For example, `mq.t3.micro`, `mq.m5.large`.
     */
    @JvmName("jngdpeokkbiftsph")
    public suspend fun hostInstanceType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.hostInstanceType = mapped
    }

    /**
     * @param value Configuration block for the LDAP server used to authenticate and authorize connections to the broker. Not supported for `engine_type` `RabbitMQ`. Detailed below. (Currently, AWS may not process changes to LDAP server metadata.)
     */
    @JvmName("gvprfuogswcmdofu")
    public suspend fun ldapServerMetadata(`value`: BrokerLdapServerMetadataArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ldapServerMetadata = mapped
    }

    /**
     * @param argument Configuration block for the LDAP server used to authenticate and authorize connections to the broker. Not supported for `engine_type` `RabbitMQ`. Detailed below. (Currently, AWS may not process changes to LDAP server metadata.)
     */
    @JvmName("vohinxrybwohqvbb")
    public suspend fun ldapServerMetadata(argument: suspend BrokerLdapServerMetadataArgsBuilder.() -> Unit) {
        val toBeMapped = BrokerLdapServerMetadataArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.ldapServerMetadata = mapped
    }

    /**
     * @param value Configuration block for the logging configuration of the broker. Detailed below.
     */
    @JvmName("wivivdmrsyhakcxt")
    public suspend fun logs(`value`: BrokerLogsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logs = mapped
    }

    /**
     * @param argument Configuration block for the logging configuration of the broker. Detailed below.
     */
    @JvmName("dfsppofuhhrqxota")
    public suspend fun logs(argument: suspend BrokerLogsArgsBuilder.() -> Unit) {
        val toBeMapped = BrokerLogsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.logs = mapped
    }

    /**
     * @param value Configuration block for the maintenance window start time. Detailed below.
     */
    @JvmName("lkkuvkseusjhfksn")
    public suspend fun maintenanceWindowStartTime(`value`: BrokerMaintenanceWindowStartTimeArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintenanceWindowStartTime = mapped
    }

    /**
     * @param argument Configuration block for the maintenance window start time. Detailed below.
     */
    @JvmName("hyxrmrnmytqlakax")
    public suspend fun maintenanceWindowStartTime(argument: suspend BrokerMaintenanceWindowStartTimeArgsBuilder.() -> Unit) {
        val toBeMapped = BrokerMaintenanceWindowStartTimeArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.maintenanceWindowStartTime = mapped
    }

    /**
     * @param value Whether to enable connections from applications outside of the VPC that hosts the broker's subnets.
     */
    @JvmName("ptobenaxgtdojvso")
    public suspend fun publiclyAccessible(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.publiclyAccessible = mapped
    }

    /**
     * @param value List of security group IDs assigned to the broker.
     */
    @JvmName("uxixgykmpdtichuj")
    public suspend fun securityGroups(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityGroups = mapped
    }

    /**
     * @param values List of security group IDs assigned to the broker.
     */
    @JvmName("ojeplqpgmyraxelq")
    public suspend fun securityGroups(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.securityGroups = mapped
    }

    /**
     * @param value Storage type of the broker. For `engine_type` `ActiveMQ`, the valid values are `efs` and `ebs`, and the AWS-default is `efs`. For `engine_type` `RabbitMQ`, only `ebs` is supported. When using `ebs`, only the `mq.m5` broker instance type family is supported.
     */
    @JvmName("gwdniodxxsxcuwqw")
    public suspend fun storageType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value List of subnet IDs in which to launch the broker. A `SINGLE_INSTANCE` deployment requires one subnet. An `ACTIVE_STANDBY_MULTI_AZ` deployment requires multiple subnets.
     */
    @JvmName("comyowbdruebumsk")
    public suspend fun subnetIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subnetIds = mapped
    }

    /**
     * @param values List of subnet IDs in which to launch the broker. A `SINGLE_INSTANCE` deployment requires one subnet. An `ACTIVE_STANDBY_MULTI_AZ` deployment requires multiple subnets.
     */
    @JvmName("qjptuyscdyqttcqc")
    public suspend fun subnetIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.subnetIds = mapped
    }

    /**
     * @param value Map of tags to assign to the broker. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("dorlvinuclwuumxt")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the broker. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("qxapehuimkjvlbxf")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
     * The following arguments are optional:
     */
    @JvmName("jqdcwcnrruqqcbpc")
    public suspend fun users(`value`: List<BrokerUserArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.users = mapped
    }

    /**
     * @param argument Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
     * The following arguments are optional:
     */
    @JvmName("oaccxhnuvfviprbh")
    public suspend fun users(argument: List<suspend BrokerUserArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map { BrokerUserArgsBuilder().applySuspend { it() }.build() }
        val mapped = of(toBeMapped)
        this.users = mapped
    }

    /**
     * @param argument Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
     * The following arguments are optional:
     */
    @JvmName("lyjmpomvkyhvnuqb")
    public suspend fun users(vararg argument: suspend BrokerUserArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map { BrokerUserArgsBuilder().applySuspend { it() }.build() }
        val mapped = of(toBeMapped)
        this.users = mapped
    }

    /**
     * @param argument Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
     * The following arguments are optional:
     */
    @JvmName("trdficbetdiawgbx")
    public suspend fun users(argument: suspend BrokerUserArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(BrokerUserArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.users = mapped
    }

    /**
     * @param values Configuration block for broker users. For `engine_type` of `RabbitMQ`, Amazon MQ does not return broker users preventing this resource from making user updates and drift detection. Detailed below.
     * The following arguments are optional:
     */
    @JvmName("amxvuakbvixypnyw")
    public suspend fun users(vararg values: BrokerUserArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.users = mapped
    }

    internal fun build(): BrokerArgs = BrokerArgs(
        applyImmediately = applyImmediately,
        authenticationStrategy = authenticationStrategy,
        autoMinorVersionUpgrade = autoMinorVersionUpgrade,
        brokerName = brokerName,
        configuration = configuration,
        dataReplicationMode = dataReplicationMode,
        dataReplicationPrimaryBrokerArn = dataReplicationPrimaryBrokerArn,
        deploymentMode = deploymentMode,
        encryptionOptions = encryptionOptions,
        engineType = engineType,
        engineVersion = engineVersion,
        hostInstanceType = hostInstanceType,
        ldapServerMetadata = ldapServerMetadata,
        logs = logs,
        maintenanceWindowStartTime = maintenanceWindowStartTime,
        publiclyAccessible = publiclyAccessible,
        securityGroups = securityGroups,
        storageType = storageType,
        subnetIds = subnetIds,
        tags = tags,
        users = users,
    )
}
