@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.mwaa.kotlin

import com.pulumi.aws.mwaa.EnvironmentArgs.builder
import com.pulumi.aws.mwaa.kotlin.inputs.EnvironmentLoggingConfigurationArgs
import com.pulumi.aws.mwaa.kotlin.inputs.EnvironmentLoggingConfigurationArgsBuilder
import com.pulumi.aws.mwaa.kotlin.inputs.EnvironmentNetworkConfigurationArgs
import com.pulumi.aws.mwaa.kotlin.inputs.EnvironmentNetworkConfigurationArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Creates a MWAA Environment resource.
 * ## Example Usage
 * A MWAA Environment requires an IAM role (`aws.iam.Role`), two subnets in the private zone (`aws.ec2.Subnet`) and a versioned S3 bucket (`aws.s3.BucketV2`).
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.mwaa.Environment("example", {
 *     dagS3Path: "dags/",
 *     executionRoleArn: exampleAwsIamRole.arn,
 *     name: "example",
 *     networkConfiguration: {
 *         securityGroupIds: [exampleAwsSecurityGroup&#46;id],
 *         subnetIds: _private.map(__item => __item.id),
 *     },
 *     sourceBucketArn: exampleAwsS3Bucket.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.mwaa.Environment("example",
 *     dag_s3_path="dags/",
 *     execution_role_arn=example_aws_iam_role["arn"],
 *     name="example",
 *     network_configuration={
 *         "security_group_ids": [example_aws_security_group["id"]],
 *         "subnet_ids": [__item["id"] for __item in private],
 *     },
 *     source_bucket_arn=example_aws_s3_bucket["arn"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Mwaa.Environment("example", new()
 *     {
 *         DagS3Path = "dags/",
 *         ExecutionRoleArn = exampleAwsIamRole.Arn,
 *         Name = "example",
 *         NetworkConfiguration = new Aws.Mwaa.Inputs.EnvironmentNetworkConfigurationArgs
 *         {
 *             SecurityGroupIds = new[]
 *             {
 *                 exampleAwsSecurityGroup.Id,
 *             },
 *             SubnetIds = @private.Select(__item => __item.Id).ToList(),
 *         },
 *         SourceBucketArn = exampleAwsS3Bucket.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/mwaa"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * _, err := mwaa.NewEnvironment(ctx, "example", &mwaa.EnvironmentArgs{
 * DagS3Path: pulumi.String("dags/"),
 * ExecutionRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * Name: pulumi.String("example"),
 * NetworkConfiguration: &mwaa.EnvironmentNetworkConfigurationArgs{
 * SecurityGroupIds: pulumi.StringArray{
 * exampleAwsSecurityGroup.Id,
 * },
 * SubnetIds: []pulumi.String(%!v(PANIC=Format method: fatal: A failure has occurred: unlowered splat expression @ example.pp:6,24-37)),
 * },
 * SourceBucketArn: pulumi.Any(exampleAwsS3Bucket.Arn),
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.mwaa.Environment;
 * import com.pulumi.aws.mwaa.EnvironmentArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentNetworkConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Environment("example", EnvironmentArgs.builder()
 *             .dagS3Path("dags/")
 *             .executionRoleArn(exampleAwsIamRole.arn())
 *             .name("example")
 *             .networkConfiguration(EnvironmentNetworkConfigurationArgs.builder()
 *                 .securityGroupIds(exampleAwsSecurityGroup.id())
 *                 .subnetIds(private_.stream().map(element -> element.id()).collect(toList()))
 *                 .build())
 *             .sourceBucketArn(exampleAwsS3Bucket.arn())
 *             .build());
 *     }
 * }
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Example with Airflow configuration options
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.mwaa.Environment("example", {
 *     airflowConfigurationOptions: {
 *         "core.default_task_retries": "16",
 *         "core.parallelism": "1",
 *     },
 *     dagS3Path: "dags/",
 *     executionRoleArn: exampleAwsIamRole.arn,
 *     name: "example",
 *     networkConfiguration: {
 *         securityGroupIds: [exampleAwsSecurityGroup&#46;id],
 *         subnetIds: _private.map(__item => __item.id),
 *     },
 *     sourceBucketArn: exampleAwsS3Bucket.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.mwaa.Environment("example",
 *     airflow_configuration_options={
 *         "core.default_task_retries": "16",
 *         "core.parallelism": "1",
 *     },
 *     dag_s3_path="dags/",
 *     execution_role_arn=example_aws_iam_role["arn"],
 *     name="example",
 *     network_configuration={
 *         "security_group_ids": [example_aws_security_group["id"]],
 *         "subnet_ids": [__item["id"] for __item in private],
 *     },
 *     source_bucket_arn=example_aws_s3_bucket["arn"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Mwaa.Environment("example", new()
 *     {
 *         AirflowConfigurationOptions =
 *         {
 *             { "core.default_task_retries", "16" },
 *             { "core.parallelism", "1" },
 *         },
 *         DagS3Path = "dags/",
 *         ExecutionRoleArn = exampleAwsIamRole.Arn,
 *         Name = "example",
 *         NetworkConfiguration = new Aws.Mwaa.Inputs.EnvironmentNetworkConfigurationArgs
 *         {
 *             SecurityGroupIds = new[]
 *             {
 *                 exampleAwsSecurityGroup.Id,
 *             },
 *             SubnetIds = @private.Select(__item => __item.Id).ToList(),
 *         },
 *         SourceBucketArn = exampleAwsS3Bucket.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/mwaa"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * _, err := mwaa.NewEnvironment(ctx, "example", &mwaa.EnvironmentArgs{
 * AirflowConfigurationOptions: pulumi.StringMap{
 * "core.default_task_retries": pulumi.String("16"),
 * "core.parallelism": pulumi.String("1"),
 * },
 * DagS3Path: pulumi.String("dags/"),
 * ExecutionRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * Name: pulumi.String("example"),
 * NetworkConfiguration: &mwaa.EnvironmentNetworkConfigurationArgs{
 * SecurityGroupIds: pulumi.StringArray{
 * exampleAwsSecurityGroup.Id,
 * },
 * SubnetIds: []pulumi.String(%!v(PANIC=Format method: fatal: A failure has occurred: unlowered splat expression @ example.pp:10,24-37)),
 * },
 * SourceBucketArn: pulumi.Any(exampleAwsS3Bucket.Arn),
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.mwaa.Environment;
 * import com.pulumi.aws.mwaa.EnvironmentArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentNetworkConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Environment("example", EnvironmentArgs.builder()
 *             .airflowConfigurationOptions(Map.ofEntries(
 *                 Map.entry("core.default_task_retries", 16),
 *                 Map.entry("core.parallelism", 1)
 *             ))
 *             .dagS3Path("dags/")
 *             .executionRoleArn(exampleAwsIamRole.arn())
 *             .name("example")
 *             .networkConfiguration(EnvironmentNetworkConfigurationArgs.builder()
 *                 .securityGroupIds(exampleAwsSecurityGroup.id())
 *                 .subnetIds(private_.stream().map(element -> element.id()).collect(toList()))
 *                 .build())
 *             .sourceBucketArn(exampleAwsS3Bucket.arn())
 *             .build());
 *     }
 * }
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Example with logging configurations
 * Note that Airflow task logs are enabled by default with the `INFO` log level.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.mwaa.Environment("example", {
 *     dagS3Path: "dags/",
 *     executionRoleArn: exampleAwsIamRole.arn,
 *     loggingConfiguration: {
 *         dagProcessingLogs: {
 *             enabled: true,
 *             logLevel: "DEBUG",
 *         },
 *         schedulerLogs: {
 *             enabled: true,
 *             logLevel: "INFO",
 *         },
 *         taskLogs: {
 *             enabled: true,
 *             logLevel: "WARNING",
 *         },
 *         webserverLogs: {
 *             enabled: true,
 *             logLevel: "ERROR",
 *         },
 *         workerLogs: {
 *             enabled: true,
 *             logLevel: "CRITICAL",
 *         },
 *     },
 *     name: "example",
 *     networkConfiguration: {
 *         securityGroupIds: [exampleAwsSecurityGroup&#46;id],
 *         subnetIds: _private.map(__item => __item.id),
 *     },
 *     sourceBucketArn: exampleAwsS3Bucket.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.mwaa.Environment("example",
 *     dag_s3_path="dags/",
 *     execution_role_arn=example_aws_iam_role["arn"],
 *     logging_configuration={
 *         "dag_processing_logs": {
 *             "enabled": True,
 *             "log_level": "DEBUG",
 *         },
 *         "scheduler_logs": {
 *             "enabled": True,
 *             "log_level": "INFO",
 *         },
 *         "task_logs": {
 *             "enabled": True,
 *             "log_level": "WARNING",
 *         },
 *         "webserver_logs": {
 *             "enabled": True,
 *             "log_level": "ERROR",
 *         },
 *         "worker_logs": {
 *             "enabled": True,
 *             "log_level": "CRITICAL",
 *         },
 *     },
 *     name="example",
 *     network_configuration={
 *         "security_group_ids": [example_aws_security_group["id"]],
 *         "subnet_ids": [__item["id"] for __item in private],
 *     },
 *     source_bucket_arn=example_aws_s3_bucket["arn"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Mwaa.Environment("example", new()
 *     {
 *         DagS3Path = "dags/",
 *         ExecutionRoleArn = exampleAwsIamRole.Arn,
 *         LoggingConfiguration = new Aws.Mwaa.Inputs.EnvironmentLoggingConfigurationArgs
 *         {
 *             DagProcessingLogs = new Aws.Mwaa.Inputs.EnvironmentLoggingConfigurationDagProcessingLogsArgs
 *             {
 *                 Enabled = true,
 *                 LogLevel = "DEBUG",
 *             },
 *             SchedulerLogs = new Aws.Mwaa.Inputs.EnvironmentLoggingConfigurationSchedulerLogsArgs
 *             {
 *                 Enabled = true,
 *                 LogLevel = "INFO",
 *             },
 *             TaskLogs = new Aws.Mwaa.Inputs.EnvironmentLoggingConfigurationTaskLogsArgs
 *             {
 *                 Enabled = true,
 *                 LogLevel = "WARNING",
 *             },
 *             WebserverLogs = new Aws.Mwaa.Inputs.EnvironmentLoggingConfigurationWebserverLogsArgs
 *             {
 *                 Enabled = true,
 *                 LogLevel = "ERROR",
 *             },
 *             WorkerLogs = new Aws.Mwaa.Inputs.EnvironmentLoggingConfigurationWorkerLogsArgs
 *             {
 *                 Enabled = true,
 *                 LogLevel = "CRITICAL",
 *             },
 *         },
 *         Name = "example",
 *         NetworkConfiguration = new Aws.Mwaa.Inputs.EnvironmentNetworkConfigurationArgs
 *         {
 *             SecurityGroupIds = new[]
 *             {
 *                 exampleAwsSecurityGroup.Id,
 *             },
 *             SubnetIds = @private.Select(__item => __item.Id).ToList(),
 *         },
 *         SourceBucketArn = exampleAwsS3Bucket.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/mwaa"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * _, err := mwaa.NewEnvironment(ctx, "example", &mwaa.EnvironmentArgs{
 * DagS3Path: pulumi.String("dags/"),
 * ExecutionRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * LoggingConfiguration: &mwaa.EnvironmentLoggingConfigurationArgs{
 * DagProcessingLogs: &mwaa.EnvironmentLoggingConfigurationDagProcessingLogsArgs{
 * Enabled: pulumi.Bool(true),
 * LogLevel: pulumi.String("DEBUG"),
 * },
 * SchedulerLogs: &mwaa.EnvironmentLoggingConfigurationSchedulerLogsArgs{
 * Enabled: pulumi.Bool(true),
 * LogLevel: pulumi.String("INFO"),
 * },
 * TaskLogs: &mwaa.EnvironmentLoggingConfigurationTaskLogsArgs{
 * Enabled: pulumi.Bool(true),
 * LogLevel: pulumi.String("WARNING"),
 * },
 * WebserverLogs: &mwaa.EnvironmentLoggingConfigurationWebserverLogsArgs{
 * Enabled: pulumi.Bool(true),
 * LogLevel: pulumi.String("ERROR"),
 * },
 * WorkerLogs: &mwaa.EnvironmentLoggingConfigurationWorkerLogsArgs{
 * Enabled: pulumi.Bool(true),
 * LogLevel: pulumi.String("CRITICAL"),
 * },
 * },
 * Name: pulumi.String("example"),
 * NetworkConfiguration: &mwaa.EnvironmentNetworkConfigurationArgs{
 * SecurityGroupIds: pulumi.StringArray{
 * exampleAwsSecurityGroup.Id,
 * },
 * SubnetIds: []pulumi.String(%!v(PANIC=Format method: fatal: A failure has occurred: unlowered splat expression @ example.pp:28,24-37)),
 * },
 * SourceBucketArn: pulumi.Any(exampleAwsS3Bucket.Arn),
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.mwaa.Environment;
 * import com.pulumi.aws.mwaa.EnvironmentArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentLoggingConfigurationArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentLoggingConfigurationDagProcessingLogsArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentLoggingConfigurationSchedulerLogsArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentLoggingConfigurationTaskLogsArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentLoggingConfigurationWebserverLogsArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentLoggingConfigurationWorkerLogsArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentNetworkConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Environment("example", EnvironmentArgs.builder()
 *             .dagS3Path("dags/")
 *             .executionRoleArn(exampleAwsIamRole.arn())
 *             .loggingConfiguration(EnvironmentLoggingConfigurationArgs.builder()
 *                 .dagProcessingLogs(EnvironmentLoggingConfigurationDagProcessingLogsArgs.builder()
 *                     .enabled(true)
 *                     .logLevel("DEBUG")
 *                     .build())
 *                 .schedulerLogs(EnvironmentLoggingConfigurationSchedulerLogsArgs.builder()
 *                     .enabled(true)
 *                     .logLevel("INFO")
 *                     .build())
 *                 .taskLogs(EnvironmentLoggingConfigurationTaskLogsArgs.builder()
 *                     .enabled(true)
 *                     .logLevel("WARNING")
 *                     .build())
 *                 .webserverLogs(EnvironmentLoggingConfigurationWebserverLogsArgs.builder()
 *                     .enabled(true)
 *                     .logLevel("ERROR")
 *                     .build())
 *                 .workerLogs(EnvironmentLoggingConfigurationWorkerLogsArgs.builder()
 *                     .enabled(true)
 *                     .logLevel("CRITICAL")
 *                     .build())
 *                 .build())
 *             .name("example")
 *             .networkConfiguration(EnvironmentNetworkConfigurationArgs.builder()
 *                 .securityGroupIds(exampleAwsSecurityGroup.id())
 *                 .subnetIds(private_.stream().map(element -> element.id()).collect(toList()))
 *                 .build())
 *             .sourceBucketArn(exampleAwsS3Bucket.arn())
 *             .build());
 *     }
 * }
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Example with tags
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.mwaa.Environment("example", {
 *     dagS3Path: "dags/",
 *     executionRoleArn: exampleAwsIamRole.arn,
 *     name: "example",
 *     networkConfiguration: {
 *         securityGroupIds: [exampleAwsSecurityGroup&#46;id],
 *         subnetIds: _private.map(__item => __item.id),
 *     },
 *     sourceBucketArn: exampleAwsS3Bucket.arn,
 *     tags: {
 *         Name: "example",
 *         Environment: "production",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.mwaa.Environment("example",
 *     dag_s3_path="dags/",
 *     execution_role_arn=example_aws_iam_role["arn"],
 *     name="example",
 *     network_configuration={
 *         "security_group_ids": [example_aws_security_group["id"]],
 *         "subnet_ids": [__item["id"] for __item in private],
 *     },
 *     source_bucket_arn=example_aws_s3_bucket["arn"],
 *     tags={
 *         "Name": "example",
 *         "Environment": "production",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Mwaa.Environment("example", new()
 *     {
 *         DagS3Path = "dags/",
 *         ExecutionRoleArn = exampleAwsIamRole.Arn,
 *         Name = "example",
 *         NetworkConfiguration = new Aws.Mwaa.Inputs.EnvironmentNetworkConfigurationArgs
 *         {
 *             SecurityGroupIds = new[]
 *             {
 *                 exampleAwsSecurityGroup.Id,
 *             },
 *             SubnetIds = @private.Select(__item => __item.Id).ToList(),
 *         },
 *         SourceBucketArn = exampleAwsS3Bucket.Arn,
 *         Tags =
 *         {
 *             { "Name", "example" },
 *             { "Environment", "production" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/mwaa"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * _, err := mwaa.NewEnvironment(ctx, "example", &mwaa.EnvironmentArgs{
 * DagS3Path: pulumi.String("dags/"),
 * ExecutionRoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * Name: pulumi.String("example"),
 * NetworkConfiguration: &mwaa.EnvironmentNetworkConfigurationArgs{
 * SecurityGroupIds: pulumi.StringArray{
 * exampleAwsSecurityGroup.Id,
 * },
 * SubnetIds: []pulumi.String(%!v(PANIC=Format method: fatal: A failure has occurred: unlowered splat expression @ example.pp:6,24-37)),
 * },
 * SourceBucketArn: pulumi.Any(exampleAwsS3Bucket.Arn),
 * Tags: pulumi.StringMap{
 * "Name": pulumi.String("example"),
 * "Environment": pulumi.String("production"),
 * },
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.mwaa.Environment;
 * import com.pulumi.aws.mwaa.EnvironmentArgs;
 * import com.pulumi.aws.mwaa.inputs.EnvironmentNetworkConfigurationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Environment("example", EnvironmentArgs.builder()
 *             .dagS3Path("dags/")
 *             .executionRoleArn(exampleAwsIamRole.arn())
 *             .name("example")
 *             .networkConfiguration(EnvironmentNetworkConfigurationArgs.builder()
 *                 .securityGroupIds(exampleAwsSecurityGroup.id())
 *                 .subnetIds(private_.stream().map(element -> element.id()).collect(toList()))
 *                 .build())
 *             .sourceBucketArn(exampleAwsS3Bucket.arn())
 *             .tags(Map.ofEntries(
 *                 Map.entry("Name", "example"),
 *                 Map.entry("Environment", "production")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import MWAA Environment using `Name`. For example:
 * ```sh
 * $ pulumi import aws:mwaa/environment:Environment example MyAirflowEnvironment
 * ```
 * @property airflowConfigurationOptions The `airflow_configuration_options` parameter specifies airflow override options. Check the [Official documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html#configuring-env-variables-reference) for all possible configuration options.
 * @property airflowVersion Airflow version of your environment, will be set by default to the latest version that MWAA supports.
 * @property dagS3Path The relative path to the DAG folder on your Amazon S3 storage bucket. For example, dags. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
 * @property endpointManagement Defines whether the VPC endpoints configured for the environment are created and managed by the customer or by AWS. If set to `SERVICE`, Amazon MWAA will create and manage the required VPC endpoints in your VPC. If set to `CUSTOMER`, you must create, and manage, the VPC endpoints for your VPC. Defaults to `SERVICE` if not set.
 * @property environmentClass Environment class for the cluster. Possible options are `mw1.small`, `mw1.medium`, `mw1.large`. Will be set by default to `mw1.small`. Please check the [AWS Pricing](https://aws.amazon.com/de/managed-workflows-for-apache-airflow/pricing/) for more information about the environment classes.
 * @property executionRoleArn The Amazon Resource Name (ARN) of the task execution role that the Amazon MWAA and its environment can assume. Check the [official AWS documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-create-role.html) for the detailed role specification.
 * @property kmsKey The Amazon Resource Name (ARN) of your KMS key that you want to use for encryption. Will be set to the ARN of the managed KMS key `aws/airflow` by default. Please check the [Official Documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/custom-keys-certs.html) for more information.
 * @property loggingConfiguration The Apache Airflow logs you want to send to Amazon CloudWatch Logs. See `logging_configuration` Block for details.
 * @property maxWebservers The maximum number of web servers that you want to run in your environment. Value need to be between `2` and `5`. Will be `2` by default.
 * @property maxWorkers The maximum number of workers that can be automatically scaled up. Value need to be between `1` and `25`. Will be `10` by default.
 * @property minWebservers The minimum number of web servers that you want to run in your environment. Value need to be between `2` and `5`. Will be `2` by default.
 * @property minWorkers The minimum number of workers that you want to run in your environment. Will be `1` by default.
 * @property name The name of the Apache Airflow Environment
 * @property networkConfiguration Specifies the network configuration for your Apache Airflow Environment. This includes two private subnets as well as security groups for the Airflow environment. Each subnet requires internet connection, otherwise the deployment will fail. See `network_configuration` Block for details.
 * @property pluginsS3ObjectVersion The plugins.zip file version you want to use.
 * @property pluginsS3Path The relative path to the plugins.zip file on your Amazon S3 storage bucket. For example, plugins.zip. If a relative path is provided in the request, then plugins_s3_object_version is required. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
 * @property requirementsS3ObjectVersion The requirements.txt file version you want to use.
 * @property requirementsS3Path The relative path to the requirements.txt file on your Amazon S3 storage bucket. For example, requirements.txt. If a relative path is provided in the request, then requirements_s3_object_version is required. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
 * @property schedulers The number of schedulers that you want to run in your environment. v2.0.2 and above accepts `2` - `5`, default `2`. v1.10.12 accepts `1`.
 * @property sourceBucketArn The Amazon Resource Name (ARN) of your Amazon S3 storage bucket. For example, arn:aws:s3:::airflow-mybucketname.
 * @property startupScriptS3ObjectVersion The version of the startup shell script you want to use. You must specify the version ID that Amazon S3 assigns to the file every time you update the script.
 * @property startupScriptS3Path The relative path to the script hosted in your bucket. The script runs as your environment starts before starting the Apache Airflow process. Use this script to install dependencies, modify configuration options, and set environment variables. See [Using a startup script](https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html). Supported for environment versions 2.x and later.
 * @property tags A map of resource tags to associate with the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property webserverAccessMode Specifies whether the webserver should be accessible over the internet or via your specified VPC. Possible options: `PRIVATE_ONLY` (default) and `PUBLIC_ONLY`.
 * @property weeklyMaintenanceWindowStart Specifies the start date for the weekly maintenance window.
 */
public data class EnvironmentArgs(
    public val airflowConfigurationOptions: Output<Map<String, String>>? = null,
    public val airflowVersion: Output<String>? = null,
    public val dagS3Path: Output<String>? = null,
    public val endpointManagement: Output<String>? = null,
    public val environmentClass: Output<String>? = null,
    public val executionRoleArn: Output<String>? = null,
    public val kmsKey: Output<String>? = null,
    public val loggingConfiguration: Output<EnvironmentLoggingConfigurationArgs>? = null,
    public val maxWebservers: Output<Int>? = null,
    public val maxWorkers: Output<Int>? = null,
    public val minWebservers: Output<Int>? = null,
    public val minWorkers: Output<Int>? = null,
    public val name: Output<String>? = null,
    public val networkConfiguration: Output<EnvironmentNetworkConfigurationArgs>? = null,
    public val pluginsS3ObjectVersion: Output<String>? = null,
    public val pluginsS3Path: Output<String>? = null,
    public val requirementsS3ObjectVersion: Output<String>? = null,
    public val requirementsS3Path: Output<String>? = null,
    public val schedulers: Output<Int>? = null,
    public val sourceBucketArn: Output<String>? = null,
    public val startupScriptS3ObjectVersion: Output<String>? = null,
    public val startupScriptS3Path: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val webserverAccessMode: Output<String>? = null,
    public val weeklyMaintenanceWindowStart: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.mwaa.EnvironmentArgs> {
    override fun toJava(): com.pulumi.aws.mwaa.EnvironmentArgs =
        com.pulumi.aws.mwaa.EnvironmentArgs.builder()
            .airflowConfigurationOptions(
                airflowConfigurationOptions?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .airflowVersion(airflowVersion?.applyValue({ args0 -> args0 }))
            .dagS3Path(dagS3Path?.applyValue({ args0 -> args0 }))
            .endpointManagement(endpointManagement?.applyValue({ args0 -> args0 }))
            .environmentClass(environmentClass?.applyValue({ args0 -> args0 }))
            .executionRoleArn(executionRoleArn?.applyValue({ args0 -> args0 }))
            .kmsKey(kmsKey?.applyValue({ args0 -> args0 }))
            .loggingConfiguration(
                loggingConfiguration?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .maxWebservers(maxWebservers?.applyValue({ args0 -> args0 }))
            .maxWorkers(maxWorkers?.applyValue({ args0 -> args0 }))
            .minWebservers(minWebservers?.applyValue({ args0 -> args0 }))
            .minWorkers(minWorkers?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .networkConfiguration(
                networkConfiguration?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .pluginsS3ObjectVersion(pluginsS3ObjectVersion?.applyValue({ args0 -> args0 }))
            .pluginsS3Path(pluginsS3Path?.applyValue({ args0 -> args0 }))
            .requirementsS3ObjectVersion(requirementsS3ObjectVersion?.applyValue({ args0 -> args0 }))
            .requirementsS3Path(requirementsS3Path?.applyValue({ args0 -> args0 }))
            .schedulers(schedulers?.applyValue({ args0 -> args0 }))
            .sourceBucketArn(sourceBucketArn?.applyValue({ args0 -> args0 }))
            .startupScriptS3ObjectVersion(startupScriptS3ObjectVersion?.applyValue({ args0 -> args0 }))
            .startupScriptS3Path(startupScriptS3Path?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .webserverAccessMode(webserverAccessMode?.applyValue({ args0 -> args0 }))
            .weeklyMaintenanceWindowStart(weeklyMaintenanceWindowStart?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [EnvironmentArgs].
 */
@PulumiTagMarker
public class EnvironmentArgsBuilder internal constructor() {
    private var airflowConfigurationOptions: Output<Map<String, String>>? = null

    private var airflowVersion: Output<String>? = null

    private var dagS3Path: Output<String>? = null

    private var endpointManagement: Output<String>? = null

    private var environmentClass: Output<String>? = null

    private var executionRoleArn: Output<String>? = null

    private var kmsKey: Output<String>? = null

    private var loggingConfiguration: Output<EnvironmentLoggingConfigurationArgs>? = null

    private var maxWebservers: Output<Int>? = null

    private var maxWorkers: Output<Int>? = null

    private var minWebservers: Output<Int>? = null

    private var minWorkers: Output<Int>? = null

    private var name: Output<String>? = null

    private var networkConfiguration: Output<EnvironmentNetworkConfigurationArgs>? = null

    private var pluginsS3ObjectVersion: Output<String>? = null

    private var pluginsS3Path: Output<String>? = null

    private var requirementsS3ObjectVersion: Output<String>? = null

    private var requirementsS3Path: Output<String>? = null

    private var schedulers: Output<Int>? = null

    private var sourceBucketArn: Output<String>? = null

    private var startupScriptS3ObjectVersion: Output<String>? = null

    private var startupScriptS3Path: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var webserverAccessMode: Output<String>? = null

    private var weeklyMaintenanceWindowStart: Output<String>? = null

    /**
     * @param value The `airflow_configuration_options` parameter specifies airflow override options. Check the [Official documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html#configuring-env-variables-reference) for all possible configuration options.
     */
    @JvmName("umjcqbycmvlvmhxl")
    public suspend fun airflowConfigurationOptions(`value`: Output<Map<String, String>>) {
        this.airflowConfigurationOptions = value
    }

    /**
     * @param value Airflow version of your environment, will be set by default to the latest version that MWAA supports.
     */
    @JvmName("oawambjhxintunya")
    public suspend fun airflowVersion(`value`: Output<String>) {
        this.airflowVersion = value
    }

    /**
     * @param value The relative path to the DAG folder on your Amazon S3 storage bucket. For example, dags. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
     */
    @JvmName("veykibsjyepekqgs")
    public suspend fun dagS3Path(`value`: Output<String>) {
        this.dagS3Path = value
    }

    /**
     * @param value Defines whether the VPC endpoints configured for the environment are created and managed by the customer or by AWS. If set to `SERVICE`, Amazon MWAA will create and manage the required VPC endpoints in your VPC. If set to `CUSTOMER`, you must create, and manage, the VPC endpoints for your VPC. Defaults to `SERVICE` if not set.
     */
    @JvmName("ulrewdadmbaivtvw")
    public suspend fun endpointManagement(`value`: Output<String>) {
        this.endpointManagement = value
    }

    /**
     * @param value Environment class for the cluster. Possible options are `mw1.small`, `mw1.medium`, `mw1.large`. Will be set by default to `mw1.small`. Please check the [AWS Pricing](https://aws.amazon.com/de/managed-workflows-for-apache-airflow/pricing/) for more information about the environment classes.
     */
    @JvmName("dixsqpptlsfbrvuf")
    public suspend fun environmentClass(`value`: Output<String>) {
        this.environmentClass = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the task execution role that the Amazon MWAA and its environment can assume. Check the [official AWS documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-create-role.html) for the detailed role specification.
     */
    @JvmName("pnhvllxylwusrltu")
    public suspend fun executionRoleArn(`value`: Output<String>) {
        this.executionRoleArn = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of your KMS key that you want to use for encryption. Will be set to the ARN of the managed KMS key `aws/airflow` by default. Please check the [Official Documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/custom-keys-certs.html) for more information.
     */
    @JvmName("khwdasiynlrcihpi")
    public suspend fun kmsKey(`value`: Output<String>) {
        this.kmsKey = value
    }

    /**
     * @param value The Apache Airflow logs you want to send to Amazon CloudWatch Logs. See `logging_configuration` Block for details.
     */
    @JvmName("wtehlckgjlbnuvcu")
    public suspend fun loggingConfiguration(`value`: Output<EnvironmentLoggingConfigurationArgs>) {
        this.loggingConfiguration = value
    }

    /**
     * @param value The maximum number of web servers that you want to run in your environment. Value need to be between `2` and `5`. Will be `2` by default.
     */
    @JvmName("tvvpyuvkvyeuwwks")
    public suspend fun maxWebservers(`value`: Output<Int>) {
        this.maxWebservers = value
    }

    /**
     * @param value The maximum number of workers that can be automatically scaled up. Value need to be between `1` and `25`. Will be `10` by default.
     */
    @JvmName("rxmaqxxcmidhtxlg")
    public suspend fun maxWorkers(`value`: Output<Int>) {
        this.maxWorkers = value
    }

    /**
     * @param value The minimum number of web servers that you want to run in your environment. Value need to be between `2` and `5`. Will be `2` by default.
     */
    @JvmName("tofdkxfsbaspwpnk")
    public suspend fun minWebservers(`value`: Output<Int>) {
        this.minWebservers = value
    }

    /**
     * @param value The minimum number of workers that you want to run in your environment. Will be `1` by default.
     */
    @JvmName("ogmopuoqlflqpyqc")
    public suspend fun minWorkers(`value`: Output<Int>) {
        this.minWorkers = value
    }

    /**
     * @param value The name of the Apache Airflow Environment
     */
    @JvmName("vydfldmiydosmtek")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Specifies the network configuration for your Apache Airflow Environment. This includes two private subnets as well as security groups for the Airflow environment. Each subnet requires internet connection, otherwise the deployment will fail. See `network_configuration` Block for details.
     */
    @JvmName("uqipiajsnyoplhoo")
    public suspend fun networkConfiguration(`value`: Output<EnvironmentNetworkConfigurationArgs>) {
        this.networkConfiguration = value
    }

    /**
     * @param value The plugins.zip file version you want to use.
     */
    @JvmName("aoctjqsrrioeirdl")
    public suspend fun pluginsS3ObjectVersion(`value`: Output<String>) {
        this.pluginsS3ObjectVersion = value
    }

    /**
     * @param value The relative path to the plugins.zip file on your Amazon S3 storage bucket. For example, plugins.zip. If a relative path is provided in the request, then plugins_s3_object_version is required. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
     */
    @JvmName("sbkxmqjwilmywitm")
    public suspend fun pluginsS3Path(`value`: Output<String>) {
        this.pluginsS3Path = value
    }

    /**
     * @param value The requirements.txt file version you want to use.
     */
    @JvmName("jdvdfjrhfvjnlogn")
    public suspend fun requirementsS3ObjectVersion(`value`: Output<String>) {
        this.requirementsS3ObjectVersion = value
    }

    /**
     * @param value The relative path to the requirements.txt file on your Amazon S3 storage bucket. For example, requirements.txt. If a relative path is provided in the request, then requirements_s3_object_version is required. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
     */
    @JvmName("afbefbnubrcahqqp")
    public suspend fun requirementsS3Path(`value`: Output<String>) {
        this.requirementsS3Path = value
    }

    /**
     * @param value The number of schedulers that you want to run in your environment. v2.0.2 and above accepts `2` - `5`, default `2`. v1.10.12 accepts `1`.
     */
    @JvmName("lowqoarxhlrttlll")
    public suspend fun schedulers(`value`: Output<Int>) {
        this.schedulers = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of your Amazon S3 storage bucket. For example, arn:aws:s3:::airflow-mybucketname.
     */
    @JvmName("dwhcrxuwbfxqwsad")
    public suspend fun sourceBucketArn(`value`: Output<String>) {
        this.sourceBucketArn = value
    }

    /**
     * @param value The version of the startup shell script you want to use. You must specify the version ID that Amazon S3 assigns to the file every time you update the script.
     */
    @JvmName("dmiakavpotpogtbt")
    public suspend fun startupScriptS3ObjectVersion(`value`: Output<String>) {
        this.startupScriptS3ObjectVersion = value
    }

    /**
     * @param value The relative path to the script hosted in your bucket. The script runs as your environment starts before starting the Apache Airflow process. Use this script to install dependencies, modify configuration options, and set environment variables. See [Using a startup script](https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html). Supported for environment versions 2.x and later.
     */
    @JvmName("xotyquoqpdqhkrnv")
    public suspend fun startupScriptS3Path(`value`: Output<String>) {
        this.startupScriptS3Path = value
    }

    /**
     * @param value A map of resource tags to associate with the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("uhgqiaeqecgvsglr")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Specifies whether the webserver should be accessible over the internet or via your specified VPC. Possible options: `PRIVATE_ONLY` (default) and `PUBLIC_ONLY`.
     */
    @JvmName("dexvofswoattgjcy")
    public suspend fun webserverAccessMode(`value`: Output<String>) {
        this.webserverAccessMode = value
    }

    /**
     * @param value Specifies the start date for the weekly maintenance window.
     */
    @JvmName("incjqyuylsmxofdb")
    public suspend fun weeklyMaintenanceWindowStart(`value`: Output<String>) {
        this.weeklyMaintenanceWindowStart = value
    }

    /**
     * @param value The `airflow_configuration_options` parameter specifies airflow override options. Check the [Official documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html#configuring-env-variables-reference) for all possible configuration options.
     */
    @JvmName("oojxorfftrlpdeyd")
    public suspend fun airflowConfigurationOptions(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.airflowConfigurationOptions = mapped
    }

    /**
     * @param values The `airflow_configuration_options` parameter specifies airflow override options. Check the [Official documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html#configuring-env-variables-reference) for all possible configuration options.
     */
    @JvmName("ygttkhqanfjbofro")
    public fun airflowConfigurationOptions(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.airflowConfigurationOptions = mapped
    }

    /**
     * @param value Airflow version of your environment, will be set by default to the latest version that MWAA supports.
     */
    @JvmName("tkkmysmewelstleg")
    public suspend fun airflowVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.airflowVersion = mapped
    }

    /**
     * @param value The relative path to the DAG folder on your Amazon S3 storage bucket. For example, dags. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
     */
    @JvmName("mbkweffybwuonvjb")
    public suspend fun dagS3Path(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dagS3Path = mapped
    }

    /**
     * @param value Defines whether the VPC endpoints configured for the environment are created and managed by the customer or by AWS. If set to `SERVICE`, Amazon MWAA will create and manage the required VPC endpoints in your VPC. If set to `CUSTOMER`, you must create, and manage, the VPC endpoints for your VPC. Defaults to `SERVICE` if not set.
     */
    @JvmName("vrbarkwmodkahuvg")
    public suspend fun endpointManagement(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.endpointManagement = mapped
    }

    /**
     * @param value Environment class for the cluster. Possible options are `mw1.small`, `mw1.medium`, `mw1.large`. Will be set by default to `mw1.small`. Please check the [AWS Pricing](https://aws.amazon.com/de/managed-workflows-for-apache-airflow/pricing/) for more information about the environment classes.
     */
    @JvmName("fbaqpokfvhlqguqy")
    public suspend fun environmentClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.environmentClass = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the task execution role that the Amazon MWAA and its environment can assume. Check the [official AWS documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-create-role.html) for the detailed role specification.
     */
    @JvmName("qyhwuqslskyurbyd")
    public suspend fun executionRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.executionRoleArn = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of your KMS key that you want to use for encryption. Will be set to the ARN of the managed KMS key `aws/airflow` by default. Please check the [Official Documentation](https://docs.aws.amazon.com/mwaa/latest/userguide/custom-keys-certs.html) for more information.
     */
    @JvmName("wlteaumkhvuuagpb")
    public suspend fun kmsKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKey = mapped
    }

    /**
     * @param value The Apache Airflow logs you want to send to Amazon CloudWatch Logs. See `logging_configuration` Block for details.
     */
    @JvmName("oanwvbcgienuswef")
    public suspend fun loggingConfiguration(`value`: EnvironmentLoggingConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.loggingConfiguration = mapped
    }

    /**
     * @param argument The Apache Airflow logs you want to send to Amazon CloudWatch Logs. See `logging_configuration` Block for details.
     */
    @JvmName("hefbgdbjwryyqsli")
    public suspend fun loggingConfiguration(argument: suspend EnvironmentLoggingConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = EnvironmentLoggingConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.loggingConfiguration = mapped
    }

    /**
     * @param value The maximum number of web servers that you want to run in your environment. Value need to be between `2` and `5`. Will be `2` by default.
     */
    @JvmName("vgpapclubqyehcvm")
    public suspend fun maxWebservers(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxWebservers = mapped
    }

    /**
     * @param value The maximum number of workers that can be automatically scaled up. Value need to be between `1` and `25`. Will be `10` by default.
     */
    @JvmName("cwmifmvssghuvbib")
    public suspend fun maxWorkers(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxWorkers = mapped
    }

    /**
     * @param value The minimum number of web servers that you want to run in your environment. Value need to be between `2` and `5`. Will be `2` by default.
     */
    @JvmName("raolkvrtoytclgvr")
    public suspend fun minWebservers(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.minWebservers = mapped
    }

    /**
     * @param value The minimum number of workers that you want to run in your environment. Will be `1` by default.
     */
    @JvmName("anawrlcmvlmlxxwf")
    public suspend fun minWorkers(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.minWorkers = mapped
    }

    /**
     * @param value The name of the Apache Airflow Environment
     */
    @JvmName("nufdfljpyjrmnhpo")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Specifies the network configuration for your Apache Airflow Environment. This includes two private subnets as well as security groups for the Airflow environment. Each subnet requires internet connection, otherwise the deployment will fail. See `network_configuration` Block for details.
     */
    @JvmName("fhwepqfcltyskmqx")
    public suspend fun networkConfiguration(`value`: EnvironmentNetworkConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkConfiguration = mapped
    }

    /**
     * @param argument Specifies the network configuration for your Apache Airflow Environment. This includes two private subnets as well as security groups for the Airflow environment. Each subnet requires internet connection, otherwise the deployment will fail. See `network_configuration` Block for details.
     */
    @JvmName("twictegqudatgdsm")
    public suspend fun networkConfiguration(argument: suspend EnvironmentNetworkConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = EnvironmentNetworkConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.networkConfiguration = mapped
    }

    /**
     * @param value The plugins.zip file version you want to use.
     */
    @JvmName("jtuuaxpvinnrilja")
    public suspend fun pluginsS3ObjectVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pluginsS3ObjectVersion = mapped
    }

    /**
     * @param value The relative path to the plugins.zip file on your Amazon S3 storage bucket. For example, plugins.zip. If a relative path is provided in the request, then plugins_s3_object_version is required. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
     */
    @JvmName("wwscqbfkfvimvuvs")
    public suspend fun pluginsS3Path(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pluginsS3Path = mapped
    }

    /**
     * @param value The requirements.txt file version you want to use.
     */
    @JvmName("dfigenkkmpdploue")
    public suspend fun requirementsS3ObjectVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requirementsS3ObjectVersion = mapped
    }

    /**
     * @param value The relative path to the requirements.txt file on your Amazon S3 storage bucket. For example, requirements.txt. If a relative path is provided in the request, then requirements_s3_object_version is required. For more information, see [Importing DAGs on Amazon MWAA](https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html).
     */
    @JvmName("vgciedcmpjlyntxq")
    public suspend fun requirementsS3Path(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requirementsS3Path = mapped
    }

    /**
     * @param value The number of schedulers that you want to run in your environment. v2.0.2 and above accepts `2` - `5`, default `2`. v1.10.12 accepts `1`.
     */
    @JvmName("ahpyllankmsornnk")
    public suspend fun schedulers(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.schedulers = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of your Amazon S3 storage bucket. For example, arn:aws:s3:::airflow-mybucketname.
     */
    @JvmName("ledcqtiltmydeupn")
    public suspend fun sourceBucketArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceBucketArn = mapped
    }

    /**
     * @param value The version of the startup shell script you want to use. You must specify the version ID that Amazon S3 assigns to the file every time you update the script.
     */
    @JvmName("wfejmokvawhopkef")
    public suspend fun startupScriptS3ObjectVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.startupScriptS3ObjectVersion = mapped
    }

    /**
     * @param value The relative path to the script hosted in your bucket. The script runs as your environment starts before starting the Apache Airflow process. Use this script to install dependencies, modify configuration options, and set environment variables. See [Using a startup script](https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html). Supported for environment versions 2.x and later.
     */
    @JvmName("xlaqesslqmeimxfu")
    public suspend fun startupScriptS3Path(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.startupScriptS3Path = mapped
    }

    /**
     * @param value A map of resource tags to associate with the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("umbgfdtttdvxtvhl")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of resource tags to associate with the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("irftghocjyvjfvig")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Specifies whether the webserver should be accessible over the internet or via your specified VPC. Possible options: `PRIVATE_ONLY` (default) and `PUBLIC_ONLY`.
     */
    @JvmName("ukyekrokcvhjiupb")
    public suspend fun webserverAccessMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.webserverAccessMode = mapped
    }

    /**
     * @param value Specifies the start date for the weekly maintenance window.
     */
    @JvmName("fnnmyswjutyyauwq")
    public suspend fun weeklyMaintenanceWindowStart(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.weeklyMaintenanceWindowStart = mapped
    }

    internal fun build(): EnvironmentArgs = EnvironmentArgs(
        airflowConfigurationOptions = airflowConfigurationOptions,
        airflowVersion = airflowVersion,
        dagS3Path = dagS3Path,
        endpointManagement = endpointManagement,
        environmentClass = environmentClass,
        executionRoleArn = executionRoleArn,
        kmsKey = kmsKey,
        loggingConfiguration = loggingConfiguration,
        maxWebservers = maxWebservers,
        maxWorkers = maxWorkers,
        minWebservers = minWebservers,
        minWorkers = minWorkers,
        name = name,
        networkConfiguration = networkConfiguration,
        pluginsS3ObjectVersion = pluginsS3ObjectVersion,
        pluginsS3Path = pluginsS3Path,
        requirementsS3ObjectVersion = requirementsS3ObjectVersion,
        requirementsS3Path = requirementsS3Path,
        schedulers = schedulers,
        sourceBucketArn = sourceBucketArn,
        startupScriptS3ObjectVersion = startupScriptS3ObjectVersion,
        startupScriptS3Path = startupScriptS3Path,
        tags = tags,
        webserverAccessMode = webserverAccessMode,
        weeklyMaintenanceWindowStart = weeklyMaintenanceWindowStart,
    )
}
