@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.opensearch.kotlin

import com.pulumi.aws.opensearch.InboundConnectionAccepterArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Manages an [AWS Opensearch Inbound Connection Accepter](https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AcceptInboundConnection.html). If connecting domains from different AWS accounts, ensure that the accepter is configured to use the AWS account where the _remote_ opensearch domain exists.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const current = aws.getCallerIdentity({});
 * const currentGetRegion = aws.getRegion({});
 * const foo = new aws.opensearch.OutboundConnection("foo", {
 *     connectionAlias: "outbound_connection",
 *     localDomainInfo: {
 *         ownerId: current.then(current => current.accountId),
 *         region: currentGetRegion.then(currentGetRegion => currentGetRegion.name),
 *         domainName: localDomain.domainName,
 *     },
 *     remoteDomainInfo: {
 *         ownerId: current.then(current => current.accountId),
 *         region: currentGetRegion.then(currentGetRegion => currentGetRegion.name),
 *         domainName: remoteDomain.domainName,
 *     },
 * });
 * const fooInboundConnectionAccepter = new aws.opensearch.InboundConnectionAccepter("foo", {connectionId: foo.id});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * current = aws.get_caller_identity()
 * current_get_region = aws.get_region()
 * foo = aws.opensearch.OutboundConnection("foo",
 *     connection_alias="outbound_connection",
 *     local_domain_info={
 *         "owner_id": current.account_id,
 *         "region": current_get_region.name,
 *         "domain_name": local_domain["domainName"],
 *     },
 *     remote_domain_info={
 *         "owner_id": current.account_id,
 *         "region": current_get_region.name,
 *         "domain_name": remote_domain["domainName"],
 *     })
 * foo_inbound_connection_accepter = aws.opensearch.InboundConnectionAccepter("foo", connection_id=foo.id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var current = Aws.GetCallerIdentity.Invoke();
 *     var currentGetRegion = Aws.GetRegion.Invoke();
 *     var foo = new Aws.OpenSearch.OutboundConnection("foo", new()
 *     {
 *         ConnectionAlias = "outbound_connection",
 *         LocalDomainInfo = new Aws.OpenSearch.Inputs.OutboundConnectionLocalDomainInfoArgs
 *         {
 *             OwnerId = current.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId),
 *             Region = currentGetRegion.Apply(getRegionResult => getRegionResult.Name),
 *             DomainName = localDomain.DomainName,
 *         },
 *         RemoteDomainInfo = new Aws.OpenSearch.Inputs.OutboundConnectionRemoteDomainInfoArgs
 *         {
 *             OwnerId = current.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId),
 *             Region = currentGetRegion.Apply(getRegionResult => getRegionResult.Name),
 *             DomainName = remoteDomain.DomainName,
 *         },
 *     });
 *     var fooInboundConnectionAccepter = new Aws.OpenSearch.InboundConnectionAccepter("foo", new()
 *     {
 *         ConnectionId = foo.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/opensearch"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		current, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		currentGetRegion, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		foo, err := opensearch.NewOutboundConnection(ctx, "foo", &opensearch.OutboundConnectionArgs{
 * 			ConnectionAlias: pulumi.String("outbound_connection"),
 * 			LocalDomainInfo: &opensearch.OutboundConnectionLocalDomainInfoArgs{
 * 				OwnerId:    pulumi.String(current.AccountId),
 * 				Region:     pulumi.String(currentGetRegion.Name),
 * 				DomainName: pulumi.Any(localDomain.DomainName),
 * 			},
 * 			RemoteDomainInfo: &opensearch.OutboundConnectionRemoteDomainInfoArgs{
 * 				OwnerId:    pulumi.String(current.AccountId),
 * 				Region:     pulumi.String(currentGetRegion.Name),
 * 				DomainName: pulumi.Any(remoteDomain.DomainName),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = opensearch.NewInboundConnectionAccepter(ctx, "foo", &opensearch.InboundConnectionAccepterArgs{
 * 			ConnectionId: foo.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.opensearch.OutboundConnection;
 * import com.pulumi.aws.opensearch.OutboundConnectionArgs;
 * import com.pulumi.aws.opensearch.inputs.OutboundConnectionLocalDomainInfoArgs;
 * import com.pulumi.aws.opensearch.inputs.OutboundConnectionRemoteDomainInfoArgs;
 * import com.pulumi.aws.opensearch.InboundConnectionAccepter;
 * import com.pulumi.aws.opensearch.InboundConnectionAccepterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var current = AwsFunctions.getCallerIdentity();
 *         final var currentGetRegion = AwsFunctions.getRegion();
 *         var foo = new OutboundConnection("foo", OutboundConnectionArgs.builder()
 *             .connectionAlias("outbound_connection")
 *             .localDomainInfo(OutboundConnectionLocalDomainInfoArgs.builder()
 *                 .ownerId(current.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()))
 *                 .region(currentGetRegion.applyValue(getRegionResult -> getRegionResult.name()))
 *                 .domainName(localDomain.domainName())
 *                 .build())
 *             .remoteDomainInfo(OutboundConnectionRemoteDomainInfoArgs.builder()
 *                 .ownerId(current.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()))
 *                 .region(currentGetRegion.applyValue(getRegionResult -> getRegionResult.name()))
 *                 .domainName(remoteDomain.domainName())
 *                 .build())
 *             .build());
 *         var fooInboundConnectionAccepter = new InboundConnectionAccepter("fooInboundConnectionAccepter", InboundConnectionAccepterArgs.builder()
 *             .connectionId(foo.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   foo:
 *     type: aws:opensearch:OutboundConnection
 *     properties:
 *       connectionAlias: outbound_connection
 *       localDomainInfo:
 *         ownerId: ${current.accountId}
 *         region: ${currentGetRegion.name}
 *         domainName: ${localDomain.domainName}
 *       remoteDomainInfo:
 *         ownerId: ${current.accountId}
 *         region: ${currentGetRegion.name}
 *         domainName: ${remoteDomain.domainName}
 *   fooInboundConnectionAccepter:
 *     type: aws:opensearch:InboundConnectionAccepter
 *     name: foo
 *     properties:
 *       connectionId: ${foo.id}
 * variables:
 *   current:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 *   currentGetRegion:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import AWS Opensearch Inbound Connection Accepters using the Inbound Connection ID. For example:
 * ```sh
 * $ pulumi import aws:opensearch/inboundConnectionAccepter:InboundConnectionAccepter foo connection-id
 * ```
 * @property connectionId Specifies the ID of the connection to accept.
 */
public data class InboundConnectionAccepterArgs(
    public val connectionId: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.opensearch.InboundConnectionAccepterArgs> {
    override fun toJava(): com.pulumi.aws.opensearch.InboundConnectionAccepterArgs =
        com.pulumi.aws.opensearch.InboundConnectionAccepterArgs.builder()
            .connectionId(connectionId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [InboundConnectionAccepterArgs].
 */
@PulumiTagMarker
public class InboundConnectionAccepterArgsBuilder internal constructor() {
    private var connectionId: Output<String>? = null

    /**
     * @param value Specifies the ID of the connection to accept.
     */
    @JvmName("knqfkemuhsaakcjg")
    public suspend fun connectionId(`value`: Output<String>) {
        this.connectionId = value
    }

    /**
     * @param value Specifies the ID of the connection to accept.
     */
    @JvmName("ekgeymprgslqewbe")
    public suspend fun connectionId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.connectionId = mapped
    }

    internal fun build(): InboundConnectionAccepterArgs = InboundConnectionAccepterArgs(
        connectionId = connectionId,
    )
}
