@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.pipes.kotlin

import com.pulumi.aws.pipes.PipeArgs.builder
import com.pulumi.aws.pipes.kotlin.inputs.PipeEnrichmentParametersArgs
import com.pulumi.aws.pipes.kotlin.inputs.PipeEnrichmentParametersArgsBuilder
import com.pulumi.aws.pipes.kotlin.inputs.PipeLogConfigurationArgs
import com.pulumi.aws.pipes.kotlin.inputs.PipeLogConfigurationArgsBuilder
import com.pulumi.aws.pipes.kotlin.inputs.PipeSourceParametersArgs
import com.pulumi.aws.pipes.kotlin.inputs.PipeSourceParametersArgsBuilder
import com.pulumi.aws.pipes.kotlin.inputs.PipeTargetParametersArgs
import com.pulumi.aws.pipes.kotlin.inputs.PipeTargetParametersArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Resource for managing an AWS EventBridge Pipes Pipe.
 * You can find out more about EventBridge Pipes in the [User Guide](https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html).
 * EventBridge Pipes are very configurable, and may require IAM permissions to work correctly. More information on the configuration options and IAM permissions can be found in the [User Guide](https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html).
 * > **Note:** EventBridge was formerly known as CloudWatch Events. The functionality is identical.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const main = aws.getCallerIdentity({});
 * const example = new aws.iam.Role("example", {assumeRolePolicy: JSON.stringify({
 *     Version: "2012-10-17",
 *     Statement: {
 *         Effect: "Allow",
 *         Action: "sts:AssumeRole",
 *         Principal: {
 *             Service: "pipes.amazonaws.com",
 *         },
 *         Condition: {
 *             StringEquals: {
 *                 "aws:SourceAccount": main.then(main => main.accountId),
 *             },
 *         },
 *     },
 * })});
 * const sourceQueue = new aws.sqs.Queue("source", {});
 * const source = new aws.iam.RolePolicy("source", {
 *     role: example.id,
 *     policy: pulumi.jsonStringify({
 *         Version: "2012-10-17",
 *         Statement: [{
 *             Effect: "Allow",
 *             Action: [
 *                 "sqs:DeleteMessage",
 *                 "sqs:GetQueueAttributes",
 *                 "sqs:ReceiveMessage",
 *             ],
 *             Resource: [sourceQueue&#46;arn],
 *         }],
 *     }),
 * });
 * const targetQueue = new aws.sqs.Queue("target", {});
 * const target = new aws.iam.RolePolicy("target", {
 *     role: example.id,
 *     policy: pulumi.jsonStringify({
 *         Version: "2012-10-17",
 *         Statement: [{
 *             Effect: "Allow",
 *             Action: ["sqs:SendMessage"],
 *             Resource: [targetQueue&#46;arn],
 *         }],
 *     }),
 * });
 * const examplePipe = new aws.pipes.Pipe("example", {
 *     name: "example-pipe",
 *     roleArn: example.arn,
 *     source: sourceQueue.arn,
 *     target: targetQueue.arn,
 * }, {
 *     dependsOn: [
 *         source,
 *         target,
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * main = aws.get_caller_identity()
 * example = aws.iam.Role("example", assume_role_policy=json.dumps({
 *     "Version": "2012-10-17",
 *     "Statement": {
 *         "Effect": "Allow",
 *         "Action": "sts:AssumeRole",
 *         "Principal": {
 *             "Service": "pipes.amazonaws.com",
 *         },
 *         "Condition": {
 *             "StringEquals": {
 *                 "aws:SourceAccount": main.account_id,
 *             },
 *         },
 *     },
 * }))
 * source_queue = aws.sqs.Queue("source")
 * source = aws.iam.RolePolicy("source",
 *     role=example.id,
 *     policy=pulumi.Output.json_dumps({
 *         "Version": "2012-10-17",
 *         "Statement": [{
 *             "Effect": "Allow",
 *             "Action": [
 *                 "sqs:DeleteMessage",
 *                 "sqs:GetQueueAttributes",
 *                 "sqs:ReceiveMessage",
 *             ],
 *             "Resource": [source_queue&#46;arn],
 *         }],
 *     }))
 * target_queue = aws.sqs.Queue("target")
 * target = aws.iam.RolePolicy("target",
 *     role=example.id,
 *     policy=pulumi.Output.json_dumps({
 *         "Version": "2012-10-17",
 *         "Statement": [{
 *             "Effect": "Allow",
 *             "Action": ["sqs:SendMessage"],
 *             "Resource": [target_queue&#46;arn],
 *         }],
 *     }))
 * example_pipe = aws.pipes.Pipe("example",
 *     name="example-pipe",
 *     role_arn=example.arn,
 *     source=source_queue.arn,
 *     target=target_queue.arn,
 *     opts = pulumi.ResourceOptions(depends_on=[
 *             source,
 *             target,
 *         ]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var main = Aws.GetCallerIdentity.Invoke();
 *     var example = new Aws.Iam.Role("example", new()
 *     {
 *         AssumeRolePolicy = JsonSerializer.Serialize(new Dictionary<string, object?>
 *         {
 *             ["Version"] = "2012-10-17",
 *             ["Statement"] = new Dictionary<string, object?>
 *             {
 *                 ["Effect"] = "Allow",
 *                 ["Action"] = "sts:AssumeRole",
 *                 ["Principal"] = new Dictionary<string, object?>
 *                 {
 *                     ["Service"] = "pipes.amazonaws.com",
 *                 },
 *                 ["Condition"] = new Dictionary<string, object?>
 *                 {
 *                     ["StringEquals"] = new Dictionary<string, object?>
 *                     {
 *                         ["aws:SourceAccount"] = main.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId),
 *                     },
 *                 },
 *             },
 *         }),
 *     });
 *     var sourceQueue = new Aws.Sqs.Queue("source");
 *     var source = new Aws.Iam.RolePolicy("source", new()
 *     {
 *         Role = example.Id,
 *         Policy = Output.JsonSerialize(Output.Create(new Dictionary<string, object?>
 *         {
 *             ["Version"] = "2012-10-17",
 *             ["Statement"] = new[]
 *             {
 *                 new Dictionary<string, object?>
 *                 {
 *                     ["Effect"] = "Allow",
 *                     ["Action"] = new[]
 *                     {
 *                         "sqs:DeleteMessage",
 *                         "sqs:GetQueueAttributes",
 *                         "sqs:ReceiveMessage",
 *                     },
 *                     ["Resource"] = new[]
 *                     {
 *                         sourceQueue.Arn,
 *                     },
 *                 },
 *             },
 *         })),
 *     });
 *     var targetQueue = new Aws.Sqs.Queue("target");
 *     var target = new Aws.Iam.RolePolicy("target", new()
 *     {
 *         Role = example.Id,
 *         Policy = Output.JsonSerialize(Output.Create(new Dictionary<string, object?>
 *         {
 *             ["Version"] = "2012-10-17",
 *             ["Statement"] = new[]
 *             {
 *                 new Dictionary<string, object?>
 *                 {
 *                     ["Effect"] = "Allow",
 *                     ["Action"] = new[]
 *                     {
 *                         "sqs:SendMessage",
 *                     },
 *                     ["Resource"] = new[]
 *                     {
 *                         targetQueue.Arn,
 *                     },
 *                 },
 *             },
 *         })),
 *     });
 *     var examplePipe = new Aws.Pipes.Pipe("example", new()
 *     {
 *         Name = "example-pipe",
 *         RoleArn = example.Arn,
 *         Source = sourceQueue.Arn,
 *         Target = targetQueue.Arn,
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             source,
 *             target,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/pipes"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		main, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 			"Version": "2012-10-17",
 * 			"Statement": map[string]interface{}{
 * 				"Effect": "Allow",
 * 				"Action": "sts:AssumeRole",
 * 				"Principal": map[string]interface{}{
 * 					"Service": "pipes.amazonaws.com",
 * 				},
 * 				"Condition": map[string]interface{}{
 * 					"StringEquals": map[string]interface{}{
 * 						"aws:SourceAccount": main.AccountId,
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		json0 := string(tmpJSON0)
 * 		example, err := iam.NewRole(ctx, "example", &iam.RoleArgs{
 * 			AssumeRolePolicy: pulumi.String(json0),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		sourceQueue, err := sqs.NewQueue(ctx, "source", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		source, err := iam.NewRolePolicy(ctx, "source", &iam.RolePolicyArgs{
 * 			Role: example.ID(),
 * 			Policy: sourceQueue.Arn.ApplyT(func(arn string) (pulumi.String, error) {
 * 				var _zero pulumi.String
 * 				tmpJSON1, err := json.Marshal(map[string]interface{}{
 * 					"Version": "2012-10-17",
 * 					"Statement": []map[string]interface{}{
 * 						map[string]interface{}{
 * 							"Effect": "Allow",
 * 							"Action": []string{
 * 								"sqs:DeleteMessage",
 * 								"sqs:GetQueueAttributes",
 * 								"sqs:ReceiveMessage",
 * 							},
 * 							"Resource": []string{
 * 								arn,
 * 							},
 * 						},
 * 					},
 * 				})
 * 				if err != nil {
 * 					return _zero, err
 * 				}
 * 				json1 := string(tmpJSON1)
 * 				return pulumi.String(json1), nil
 * 			}).(pulumi.StringOutput),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		targetQueue, err := sqs.NewQueue(ctx, "target", nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		target, err := iam.NewRolePolicy(ctx, "target", &iam.RolePolicyArgs{
 * 			Role: example.ID(),
 * 			Policy: targetQueue.Arn.ApplyT(func(arn string) (pulumi.String, error) {
 * 				var _zero pulumi.String
 * 				tmpJSON2, err := json.Marshal(map[string]interface{}{
 * 					"Version": "2012-10-17",
 * 					"Statement": []map[string]interface{}{
 * 						map[string]interface{}{
 * 							"Effect": "Allow",
 * 							"Action": []string{
 * 								"sqs:SendMessage",
 * 							},
 * 							"Resource": []string{
 * 								arn,
 * 							},
 * 						},
 * 					},
 * 				})
 * 				if err != nil {
 * 					return _zero, err
 * 				}
 * 				json2 := string(tmpJSON2)
 * 				return pulumi.String(json2), nil
 * 			}).(pulumi.StringOutput),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = pipes.NewPipe(ctx, "example", &pipes.PipeArgs{
 * 			Name:    pulumi.String("example-pipe"),
 * 			RoleArn: example.Arn,
 * 			Source:  sourceQueue.Arn,
 * 			Target:  targetQueue.Arn,
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			source,
 * 			target,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.iam.RolePolicy;
 * import com.pulumi.aws.iam.RolePolicyArgs;
 * import com.pulumi.aws.pipes.Pipe;
 * import com.pulumi.aws.pipes.PipeArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var main = AwsFunctions.getCallerIdentity();
 *         var example = new Role("example", RoleArgs.builder()
 *             .assumeRolePolicy(serializeJson(
 *                 jsonObject(
 *                     jsonProperty("Version", "2012-10-17"),
 *                     jsonProperty("Statement", jsonObject(
 *                         jsonProperty("Effect", "Allow"),
 *                         jsonProperty("Action", "sts:AssumeRole"),
 *                         jsonProperty("Principal", jsonObject(
 *                             jsonProperty("Service", "pipes.amazonaws.com")
 *                         )),
 *                         jsonProperty("Condition", jsonObject(
 *                             jsonProperty("StringEquals", jsonObject(
 *                                 jsonProperty("aws:SourceAccount", main.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()))
 *                             ))
 *                         ))
 *                     ))
 *                 )))
 *             .build());
 *         var sourceQueue = new Queue("sourceQueue");
 *         var source = new RolePolicy("source", RolePolicyArgs.builder()
 *             .role(example.id())
 *             .policy(sourceQueue.arn().applyValue(arn -> serializeJson(
 *                 jsonObject(
 *                     jsonProperty("Version", "2012-10-17"),
 *                     jsonProperty("Statement", jsonArray(jsonObject(
 *                         jsonProperty("Effect", "Allow"),
 *                         jsonProperty("Action", jsonArray(
 *                             "sqs:DeleteMessage",
 *                             "sqs:GetQueueAttributes",
 *                             "sqs:ReceiveMessage"
 *                         )),
 *                         jsonProperty("Resource", jsonArray(arn))
 *                     )))
 *                 ))))
 *             .build());
 *         var targetQueue = new Queue("targetQueue");
 *         var target = new RolePolicy("target", RolePolicyArgs.builder()
 *             .role(example.id())
 *             .policy(targetQueue.arn().applyValue(arn -> serializeJson(
 *                 jsonObject(
 *                     jsonProperty("Version", "2012-10-17"),
 *                     jsonProperty("Statement", jsonArray(jsonObject(
 *                         jsonProperty("Effect", "Allow"),
 *                         jsonProperty("Action", jsonArray("sqs:SendMessage")),
 *                         jsonProperty("Resource", jsonArray(arn))
 *                     )))
 *                 ))))
 *             .build());
 *         var examplePipe = new Pipe("examplePipe", PipeArgs.builder()
 *             .name("example-pipe")
 *             .roleArn(example.arn())
 *             .source(sourceQueue.arn())
 *             .target(targetQueue.arn())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(
 *                     source,
 *                     target)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:iam:Role
 *     properties:
 *       assumeRolePolicy:
 *         fn::toJSON:
 *           Version: 2012-10-17
 *           Statement:
 *             Effect: Allow
 *             Action: sts:AssumeRole
 *             Principal:
 *               Service: pipes.amazonaws.com
 *             Condition:
 *               StringEquals:
 *                 aws:SourceAccount: ${main.accountId}
 *   source:
 *     type: aws:iam:RolePolicy
 *     properties:
 *       role: ${example.id}
 *       policy:
 *         fn::toJSON:
 *           Version: 2012-10-17
 *           Statement:
 *             - Effect: Allow
 *               Action:
 *                 - sqs:DeleteMessage
 *                 - sqs:GetQueueAttributes
 *                 - sqs:ReceiveMessage
 *               Resource:
 *                 - ${sourceQueue.arn}
 *   sourceQueue:
 *     type: aws:sqs:Queue
 *     name: source
 *   target:
 *     type: aws:iam:RolePolicy
 *     properties:
 *       role: ${example.id}
 *       policy:
 *         fn::toJSON:
 *           Version: 2012-10-17
 *           Statement:
 *             - Effect: Allow
 *               Action:
 *                 - sqs:SendMessage
 *               Resource:
 *                 - ${targetQueue.arn}
 *   targetQueue:
 *     type: aws:sqs:Queue
 *     name: target
 *   examplePipe:
 *     type: aws:pipes:Pipe
 *     name: example
 *     properties:
 *       name: example-pipe
 *       roleArn: ${example.arn}
 *       source: ${sourceQueue.arn}
 *       target: ${targetQueue.arn}
 *     options:
 *       dependson:
 *         - ${source}
 *         - ${target}
 * variables:
 *   main:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Enrichment Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.pipes.Pipe("example", {
 *     name: "example-pipe",
 *     roleArn: exampleAwsIamRole.arn,
 *     source: source.arn,
 *     target: target.arn,
 *     enrichment: exampleAwsCloudwatchEventApiDestination.arn,
 *     enrichmentParameters: {
 *         httpParameters: {
 *             pathParameterValues: "example-path-param",
 *             headerParameters: {
 *                 "example-header": "example-value",
 *                 "second-example-header": "second-example-value",
 *             },
 *             queryStringParameters: {
 *                 "example-query-string": "example-value",
 *                 "second-example-query-string": "second-example-value",
 *             },
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.pipes.Pipe("example",
 *     name="example-pipe",
 *     role_arn=example_aws_iam_role["arn"],
 *     source=source["arn"],
 *     target=target["arn"],
 *     enrichment=example_aws_cloudwatch_event_api_destination["arn"],
 *     enrichment_parameters={
 *         "http_parameters": {
 *             "path_parameter_values": "example-path-param",
 *             "header_parameters": {
 *                 "example_header": "example-value",
 *                 "second_example_header": "second-example-value",
 *             },
 *             "query_string_parameters": {
 *                 "example_query_string": "example-value",
 *                 "second_example_query_string": "second-example-value",
 *             },
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Pipes.Pipe("example", new()
 *     {
 *         Name = "example-pipe",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Source = source.Arn,
 *         Target = target.Arn,
 *         Enrichment = exampleAwsCloudwatchEventApiDestination.Arn,
 *         EnrichmentParameters = new Aws.Pipes.Inputs.PipeEnrichmentParametersArgs
 *         {
 *             HttpParameters = new Aws.Pipes.Inputs.PipeEnrichmentParametersHttpParametersArgs
 *             {
 *                 PathParameterValues = "example-path-param",
 *                 HeaderParameters =
 *                 {
 *                     { "example-header", "example-value" },
 *                     { "second-example-header", "second-example-value" },
 *                 },
 *                 QueryStringParameters =
 *                 {
 *                     { "example-query-string", "example-value" },
 *                     { "second-example-query-string", "second-example-value" },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/pipes"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := pipes.NewPipe(ctx, "example", &pipes.PipeArgs{
 * 			Name:       pulumi.String("example-pipe"),
 * 			RoleArn:    pulumi.Any(exampleAwsIamRole.Arn),
 * 			Source:     pulumi.Any(source.Arn),
 * 			Target:     pulumi.Any(target.Arn),
 * 			Enrichment: pulumi.Any(exampleAwsCloudwatchEventApiDestination.Arn),
 * 			EnrichmentParameters: &pipes.PipeEnrichmentParametersArgs{
 * 				HttpParameters: &pipes.PipeEnrichmentParametersHttpParametersArgs{
 * 					PathParameterValues: pulumi.String("example-path-param"),
 * 					HeaderParameters: pulumi.StringMap{
 * 						"example-header":        pulumi.String("example-value"),
 * 						"second-example-header": pulumi.String("second-example-value"),
 * 					},
 * 					QueryStringParameters: pulumi.StringMap{
 * 						"example-query-string":        pulumi.String("example-value"),
 * 						"second-example-query-string": pulumi.String("second-example-value"),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.pipes.Pipe;
 * import com.pulumi.aws.pipes.PipeArgs;
 * import com.pulumi.aws.pipes.inputs.PipeEnrichmentParametersArgs;
 * import com.pulumi.aws.pipes.inputs.PipeEnrichmentParametersHttpParametersArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Pipe("example", PipeArgs.builder()
 *             .name("example-pipe")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .source(source.arn())
 *             .target(target.arn())
 *             .enrichment(exampleAwsCloudwatchEventApiDestination.arn())
 *             .enrichmentParameters(PipeEnrichmentParametersArgs.builder()
 *                 .httpParameters(PipeEnrichmentParametersHttpParametersArgs.builder()
 *                     .pathParameterValues("example-path-param")
 *                     .headerParameters(Map.ofEntries(
 *                         Map.entry("example-header", "example-value"),
 *                         Map.entry("second-example-header", "second-example-value")
 *                     ))
 *                     .queryStringParameters(Map.ofEntries(
 *                         Map.entry("example-query-string", "example-value"),
 *                         Map.entry("second-example-query-string", "second-example-value")
 *                     ))
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:pipes:Pipe
 *     properties:
 *       name: example-pipe
 *       roleArn: ${exampleAwsIamRole.arn}
 *       source: ${source.arn}
 *       target: ${target.arn}
 *       enrichment: ${exampleAwsCloudwatchEventApiDestination.arn}
 *       enrichmentParameters:
 *         httpParameters:
 *           pathParameterValues: example-path-param
 *           headerParameters:
 *             example-header: example-value
 *             second-example-header: second-example-value
 *           queryStringParameters:
 *             example-query-string: example-value
 *             second-example-query-string: second-example-value
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Filter Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.pipes.Pipe("example", {
 *     name: "example-pipe",
 *     roleArn: exampleAwsIamRole.arn,
 *     source: source.arn,
 *     target: target.arn,
 *     sourceParameters: {
 *         filterCriteria: {
 *             filters: [{
 *                 pattern: JSON.stringify({
 *                     source: ["event-source"],
 *                 }),
 *             }],
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * example = aws.pipes.Pipe("example",
 *     name="example-pipe",
 *     role_arn=example_aws_iam_role["arn"],
 *     source=source["arn"],
 *     target=target["arn"],
 *     source_parameters={
 *         "filter_criteria": {
 *             "filters": [{
 *                 "pattern": json.dumps({
 *                     "source": ["event-source"],
 *                 }),
 *             }],
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Pipes.Pipe("example", new()
 *     {
 *         Name = "example-pipe",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Source = source.Arn,
 *         Target = target.Arn,
 *         SourceParameters = new Aws.Pipes.Inputs.PipeSourceParametersArgs
 *         {
 *             FilterCriteria = new Aws.Pipes.Inputs.PipeSourceParametersFilterCriteriaArgs
 *             {
 *                 Filters = new[]
 *                 {
 *                     new Aws.Pipes.Inputs.PipeSourceParametersFilterCriteriaFilterArgs
 *                     {
 *                         Pattern = JsonSerializer.Serialize(new Dictionary<string, object?>
 *                         {
 *                             ["source"] = new[]
 *                             {
 *                                 "event-source",
 *                             },
 *                         }),
 *                     },
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/pipes"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 			"source": []string{
 * 				"event-source",
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		json0 := string(tmpJSON0)
 * 		_, err = pipes.NewPipe(ctx, "example", &pipes.PipeArgs{
 * 			Name:    pulumi.String("example-pipe"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Source:  pulumi.Any(source.Arn),
 * 			Target:  pulumi.Any(target.Arn),
 * 			SourceParameters: &pipes.PipeSourceParametersArgs{
 * 				FilterCriteria: &pipes.PipeSourceParametersFilterCriteriaArgs{
 * 					Filters: pipes.PipeSourceParametersFilterCriteriaFilterArray{
 * 						&pipes.PipeSourceParametersFilterCriteriaFilterArgs{
 * 							Pattern: pulumi.String(json0),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.pipes.Pipe;
 * import com.pulumi.aws.pipes.PipeArgs;
 * import com.pulumi.aws.pipes.inputs.PipeSourceParametersArgs;
 * import com.pulumi.aws.pipes.inputs.PipeSourceParametersFilterCriteriaArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Pipe("example", PipeArgs.builder()
 *             .name("example-pipe")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .source(source.arn())
 *             .target(target.arn())
 *             .sourceParameters(PipeSourceParametersArgs.builder()
 *                 .filterCriteria(PipeSourceParametersFilterCriteriaArgs.builder()
 *                     .filters(PipeSourceParametersFilterCriteriaFilterArgs.builder()
 *                         .pattern(serializeJson(
 *                             jsonObject(
 *                                 jsonProperty("source", jsonArray("event-source"))
 *                             )))
 *                         .build())
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:pipes:Pipe
 *     properties:
 *       name: example-pipe
 *       roleArn: ${exampleAwsIamRole.arn}
 *       source: ${source.arn}
 *       target: ${target.arn}
 *       sourceParameters:
 *         filterCriteria:
 *           filters:
 *             - pattern:
 *                 fn::toJSON:
 *                   source:
 *                     - event-source
 * ```
 * <!--End PulumiCodeChooser -->
 * ### CloudWatch Logs Logging Configuration Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.cloudwatch.LogGroup("example", {name: "example-pipe-target"});
 * const examplePipe = new aws.pipes.Pipe("example", {
 *     name: "example-pipe",
 *     roleArn: exampleAwsIamRole.arn,
 *     source: sourceAwsSqsQueue.arn,
 *     target: targetAwsSqsQueue.arn,
 *     logConfiguration: {
 *         includeExecutionDatas: ["ALL"],
 *         level: "INFO",
 *         cloudwatchLogsLogDestination: {
 *             logGroupArn: targetAwsCloudwatchLogGroup.arn,
 *         },
 *     },
 * }, {
 *     dependsOn: [
 *         source,
 *         target,
 *     ],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.cloudwatch.LogGroup("example", name="example-pipe-target")
 * example_pipe = aws.pipes.Pipe("example",
 *     name="example-pipe",
 *     role_arn=example_aws_iam_role["arn"],
 *     source=source_aws_sqs_queue["arn"],
 *     target=target_aws_sqs_queue["arn"],
 *     log_configuration={
 *         "include_execution_datas": ["ALL"],
 *         "level": "INFO",
 *         "cloudwatch_logs_log_destination": {
 *             "log_group_arn": target_aws_cloudwatch_log_group["arn"],
 *         },
 *     },
 *     opts = pulumi.ResourceOptions(depends_on=[
 *             source,
 *             target,
 *         ]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.CloudWatch.LogGroup("example", new()
 *     {
 *         Name = "example-pipe-target",
 *     });
 *     var examplePipe = new Aws.Pipes.Pipe("example", new()
 *     {
 *         Name = "example-pipe",
 *         RoleArn = exampleAwsIamRole.Arn,
 *         Source = sourceAwsSqsQueue.Arn,
 *         Target = targetAwsSqsQueue.Arn,
 *         LogConfiguration = new Aws.Pipes.Inputs.PipeLogConfigurationArgs
 *         {
 *             IncludeExecutionDatas = new[]
 *             {
 *                 "ALL",
 *             },
 *             Level = "INFO",
 *             CloudwatchLogsLogDestination = new Aws.Pipes.Inputs.PipeLogConfigurationCloudwatchLogsLogDestinationArgs
 *             {
 *                 LogGroupArn = targetAwsCloudwatchLogGroup.Arn,
 *             },
 *         },
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             source,
 *             target,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/pipes"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := cloudwatch.NewLogGroup(ctx, "example", &cloudwatch.LogGroupArgs{
 * 			Name: pulumi.String("example-pipe-target"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = pipes.NewPipe(ctx, "example", &pipes.PipeArgs{
 * 			Name:    pulumi.String("example-pipe"),
 * 			RoleArn: pulumi.Any(exampleAwsIamRole.Arn),
 * 			Source:  pulumi.Any(sourceAwsSqsQueue.Arn),
 * 			Target:  pulumi.Any(targetAwsSqsQueue.Arn),
 * 			LogConfiguration: &pipes.PipeLogConfigurationArgs{
 * 				IncludeExecutionDatas: pulumi.StringArray{
 * 					pulumi.String("ALL"),
 * 				},
 * 				Level: pulumi.String("INFO"),
 * 				CloudwatchLogsLogDestination: &pipes.PipeLogConfigurationCloudwatchLogsLogDestinationArgs{
 * 					LogGroupArn: pulumi.Any(targetAwsCloudwatchLogGroup.Arn),
 * 				},
 * 			},
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			source,
 * 			target,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.cloudwatch.LogGroupArgs;
 * import com.pulumi.aws.pipes.Pipe;
 * import com.pulumi.aws.pipes.PipeArgs;
 * import com.pulumi.aws.pipes.inputs.PipeLogConfigurationArgs;
 * import com.pulumi.aws.pipes.inputs.PipeLogConfigurationCloudwatchLogsLogDestinationArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new LogGroup("example", LogGroupArgs.builder()
 *             .name("example-pipe-target")
 *             .build());
 *         var examplePipe = new Pipe("examplePipe", PipeArgs.builder()
 *             .name("example-pipe")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .source(sourceAwsSqsQueue.arn())
 *             .target(targetAwsSqsQueue.arn())
 *             .logConfiguration(PipeLogConfigurationArgs.builder()
 *                 .includeExecutionDatas("ALL")
 *                 .level("INFO")
 *                 .cloudwatchLogsLogDestination(PipeLogConfigurationCloudwatchLogsLogDestinationArgs.builder()
 *                     .logGroupArn(targetAwsCloudwatchLogGroup.arn())
 *                     .build())
 *                 .build())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(
 *                     source,
 *                     target)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:cloudwatch:LogGroup
 *     properties:
 *       name: example-pipe-target
 *   examplePipe:
 *     type: aws:pipes:Pipe
 *     name: example
 *     properties:
 *       name: example-pipe
 *       roleArn: ${exampleAwsIamRole.arn}
 *       source: ${sourceAwsSqsQueue.arn}
 *       target: ${targetAwsSqsQueue.arn}
 *       logConfiguration:
 *         includeExecutionDatas:
 *           - ALL
 *         level: INFO
 *         cloudwatchLogsLogDestination:
 *           logGroupArn: ${targetAwsCloudwatchLogGroup.arn}
 *     options:
 *       dependson:
 *         - ${source}
 *         - ${target}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### SQS Source and Target Configuration Usage
 * <!--Start PulumiCodeChooser -->
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.pipes.Pipe;
 * import com.pulumi.aws.pipes.PipeArgs;
 * import com.pulumi.aws.pipes.inputs.PipeSourceParametersArgs;
 * import com.pulumi.aws.pipes.inputs.PipeSourceParametersSqsQueueParametersArgs;
 * import com.pulumi.aws.pipes.inputs.PipeTargetParametersArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Pipe("example", PipeArgs.builder()
 *             .name("example-pipe")
 *             .roleArn(exampleAwsIamRole.arn())
 *             .source(source.arn())
 *             .target(target.arn())
 *             .sourceParameters(PipeSourceParametersArgs.builder()
 *                 .sqsQueueParameters(PipeSourceParametersSqsQueueParametersArgs.builder()
 *                     .batchSize(1)
 *                     .maximumBatchingWindowInSeconds(2)
 *                     .build())
 *                 .build())
 *             .targetParameters(PipeTargetParametersArgs.builder()
 *                 .sqsQueue(%!v(PANIC=Format method: runtime error: invalid memory address or nil pointer dereference))
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:pipes:Pipe
 *     properties:
 *       name: example-pipe
 *       roleArn: ${exampleAwsIamRole.arn}
 *       source: ${source.arn}
 *       target: ${target.arn}
 *       sourceParameters:
 *         sqsQueueParameters:
 *           batchSize: 1
 *           maximumBatchingWindowInSeconds: 2
 *       targetParameters:
 *         sqsQueue:
 *           - messageDeduplicationId: example-dedupe
 *             messageGroupId: example-group
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import pipes using the `name`. For example:
 * ```sh
 * $ pulumi import aws:pipes/pipe:Pipe example my-pipe
 * ```
 * @property description A description of the pipe. At most 512 characters.
 * @property desiredState The state the pipe should be in. One of: `RUNNING`, `STOPPED`.
 * @property enrichment Enrichment resource of the pipe (typically an ARN). Read more about enrichment in the [User Guide](https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html#pipes-enrichment).
 * @property enrichmentParameters Parameters to configure enrichment for your pipe. Detailed below.
 * @property logConfiguration Logging configuration settings for the pipe. Detailed below.
 * @property name Name of the pipe. If omitted, the provider will assign a random, unique name. Conflicts with `name_prefix`.
 * @property namePrefix Creates a unique name beginning with the specified prefix. Conflicts with `name`.
 * @property roleArn ARN of the role that allows the pipe to send data to the target.
 * @property source Source resource of the pipe. This field typically requires an ARN (Amazon Resource Name). However, when using a self-managed Kafka cluster, you should use a different format. Instead of an ARN, use 'smk://' followed by the bootstrap server's address.
 * @property sourceParameters Parameters to configure a source for the pipe. Detailed below.
 * @property tags Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property target Target resource of the pipe (typically an ARN).
 * The following arguments are optional:
 * @property targetParameters Parameters to configure a target for your pipe. Detailed below.
 */
public data class PipeArgs(
    public val description: Output<String>? = null,
    public val desiredState: Output<String>? = null,
    public val enrichment: Output<String>? = null,
    public val enrichmentParameters: Output<PipeEnrichmentParametersArgs>? = null,
    public val logConfiguration: Output<PipeLogConfigurationArgs>? = null,
    public val name: Output<String>? = null,
    public val namePrefix: Output<String>? = null,
    public val roleArn: Output<String>? = null,
    public val source: Output<String>? = null,
    public val sourceParameters: Output<PipeSourceParametersArgs>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val target: Output<String>? = null,
    public val targetParameters: Output<PipeTargetParametersArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.pipes.PipeArgs> {
    override fun toJava(): com.pulumi.aws.pipes.PipeArgs = com.pulumi.aws.pipes.PipeArgs.builder()
        .description(description?.applyValue({ args0 -> args0 }))
        .desiredState(desiredState?.applyValue({ args0 -> args0 }))
        .enrichment(enrichment?.applyValue({ args0 -> args0 }))
        .enrichmentParameters(
            enrichmentParameters?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .logConfiguration(logConfiguration?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .name(name?.applyValue({ args0 -> args0 }))
        .namePrefix(namePrefix?.applyValue({ args0 -> args0 }))
        .roleArn(roleArn?.applyValue({ args0 -> args0 }))
        .source(source?.applyValue({ args0 -> args0 }))
        .sourceParameters(sourceParameters?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .target(target?.applyValue({ args0 -> args0 }))
        .targetParameters(
            targetParameters?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        ).build()
}

/**
 * Builder for [PipeArgs].
 */
@PulumiTagMarker
public class PipeArgsBuilder internal constructor() {
    private var description: Output<String>? = null

    private var desiredState: Output<String>? = null

    private var enrichment: Output<String>? = null

    private var enrichmentParameters: Output<PipeEnrichmentParametersArgs>? = null

    private var logConfiguration: Output<PipeLogConfigurationArgs>? = null

    private var name: Output<String>? = null

    private var namePrefix: Output<String>? = null

    private var roleArn: Output<String>? = null

    private var source: Output<String>? = null

    private var sourceParameters: Output<PipeSourceParametersArgs>? = null

    private var tags: Output<Map<String, String>>? = null

    private var target: Output<String>? = null

    private var targetParameters: Output<PipeTargetParametersArgs>? = null

    /**
     * @param value A description of the pipe. At most 512 characters.
     */
    @JvmName("xwtigaiujkvqqmkc")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The state the pipe should be in. One of: `RUNNING`, `STOPPED`.
     */
    @JvmName("diyyeujmkcmeollv")
    public suspend fun desiredState(`value`: Output<String>) {
        this.desiredState = value
    }

    /**
     * @param value Enrichment resource of the pipe (typically an ARN). Read more about enrichment in the [User Guide](https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html#pipes-enrichment).
     */
    @JvmName("uvveyeywnmrsbhca")
    public suspend fun enrichment(`value`: Output<String>) {
        this.enrichment = value
    }

    /**
     * @param value Parameters to configure enrichment for your pipe. Detailed below.
     */
    @JvmName("iesuposdmteodlrt")
    public suspend fun enrichmentParameters(`value`: Output<PipeEnrichmentParametersArgs>) {
        this.enrichmentParameters = value
    }

    /**
     * @param value Logging configuration settings for the pipe. Detailed below.
     */
    @JvmName("dlwluejkpqwpknji")
    public suspend fun logConfiguration(`value`: Output<PipeLogConfigurationArgs>) {
        this.logConfiguration = value
    }

    /**
     * @param value Name of the pipe. If omitted, the provider will assign a random, unique name. Conflicts with `name_prefix`.
     */
    @JvmName("mkxanhjrnktuhjye")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`.
     */
    @JvmName("dtnkorimqvgvwppt")
    public suspend fun namePrefix(`value`: Output<String>) {
        this.namePrefix = value
    }

    /**
     * @param value ARN of the role that allows the pipe to send data to the target.
     */
    @JvmName("iqwknrllghlpidpi")
    public suspend fun roleArn(`value`: Output<String>) {
        this.roleArn = value
    }

    /**
     * @param value Source resource of the pipe. This field typically requires an ARN (Amazon Resource Name). However, when using a self-managed Kafka cluster, you should use a different format. Instead of an ARN, use 'smk://' followed by the bootstrap server's address.
     */
    @JvmName("slcolcoduvflwbyw")
    public suspend fun source(`value`: Output<String>) {
        this.source = value
    }

    /**
     * @param value Parameters to configure a source for the pipe. Detailed below.
     */
    @JvmName("iniujubndqrttwul")
    public suspend fun sourceParameters(`value`: Output<PipeSourceParametersArgs>) {
        this.sourceParameters = value
    }

    /**
     * @param value Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ukrcmafanxadmsub")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Target resource of the pipe (typically an ARN).
     * The following arguments are optional:
     */
    @JvmName("kmcnypdqrglkgphs")
    public suspend fun target(`value`: Output<String>) {
        this.target = value
    }

    /**
     * @param value Parameters to configure a target for your pipe. Detailed below.
     */
    @JvmName("qiqrxroffsjslyct")
    public suspend fun targetParameters(`value`: Output<PipeTargetParametersArgs>) {
        this.targetParameters = value
    }

    /**
     * @param value A description of the pipe. At most 512 characters.
     */
    @JvmName("oqosxixaagxfmwnq")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The state the pipe should be in. One of: `RUNNING`, `STOPPED`.
     */
    @JvmName("fgqkvvntdsiddxni")
    public suspend fun desiredState(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.desiredState = mapped
    }

    /**
     * @param value Enrichment resource of the pipe (typically an ARN). Read more about enrichment in the [User Guide](https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html#pipes-enrichment).
     */
    @JvmName("pankvwvracpeaalm")
    public suspend fun enrichment(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enrichment = mapped
    }

    /**
     * @param value Parameters to configure enrichment for your pipe. Detailed below.
     */
    @JvmName("wwkrengvgupwltpv")
    public suspend fun enrichmentParameters(`value`: PipeEnrichmentParametersArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enrichmentParameters = mapped
    }

    /**
     * @param argument Parameters to configure enrichment for your pipe. Detailed below.
     */
    @JvmName("tnmvmfihuwtgcptt")
    public suspend fun enrichmentParameters(argument: suspend PipeEnrichmentParametersArgsBuilder.() -> Unit) {
        val toBeMapped = PipeEnrichmentParametersArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.enrichmentParameters = mapped
    }

    /**
     * @param value Logging configuration settings for the pipe. Detailed below.
     */
    @JvmName("roteyvtyowjnyfwu")
    public suspend fun logConfiguration(`value`: PipeLogConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.logConfiguration = mapped
    }

    /**
     * @param argument Logging configuration settings for the pipe. Detailed below.
     */
    @JvmName("mcgfarapvymnrwof")
    public suspend fun logConfiguration(argument: suspend PipeLogConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = PipeLogConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.logConfiguration = mapped
    }

    /**
     * @param value Name of the pipe. If omitted, the provider will assign a random, unique name. Conflicts with `name_prefix`.
     */
    @JvmName("kyeddmkbooimwoah")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Creates a unique name beginning with the specified prefix. Conflicts with `name`.
     */
    @JvmName("nvnonvulokjofijm")
    public suspend fun namePrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namePrefix = mapped
    }

    /**
     * @param value ARN of the role that allows the pipe to send data to the target.
     */
    @JvmName("bfampefplcisccbd")
    public suspend fun roleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.roleArn = mapped
    }

    /**
     * @param value Source resource of the pipe. This field typically requires an ARN (Amazon Resource Name). However, when using a self-managed Kafka cluster, you should use a different format. Instead of an ARN, use 'smk://' followed by the bootstrap server's address.
     */
    @JvmName("djjmcwridublilyx")
    public suspend fun source(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.source = mapped
    }

    /**
     * @param value Parameters to configure a source for the pipe. Detailed below.
     */
    @JvmName("hovvcjutethjeebq")
    public suspend fun sourceParameters(`value`: PipeSourceParametersArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceParameters = mapped
    }

    /**
     * @param argument Parameters to configure a source for the pipe. Detailed below.
     */
    @JvmName("wqiduytnvlmiykld")
    public suspend fun sourceParameters(argument: suspend PipeSourceParametersArgsBuilder.() -> Unit) {
        val toBeMapped = PipeSourceParametersArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.sourceParameters = mapped
    }

    /**
     * @param value Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jrbiwxsmgatupmsq")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value mapping of resource tags. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("pgtplcfmserrerbs")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Target resource of the pipe (typically an ARN).
     * The following arguments are optional:
     */
    @JvmName("jwhwlbhvporxjbaa")
    public suspend fun target(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.target = mapped
    }

    /**
     * @param value Parameters to configure a target for your pipe. Detailed below.
     */
    @JvmName("pohbtadinallxcdc")
    public suspend fun targetParameters(`value`: PipeTargetParametersArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.targetParameters = mapped
    }

    /**
     * @param argument Parameters to configure a target for your pipe. Detailed below.
     */
    @JvmName("hyeniqpwfweultbk")
    public suspend fun targetParameters(argument: suspend PipeTargetParametersArgsBuilder.() -> Unit) {
        val toBeMapped = PipeTargetParametersArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.targetParameters = mapped
    }

    internal fun build(): PipeArgs = PipeArgs(
        description = description,
        desiredState = desiredState,
        enrichment = enrichment,
        enrichmentParameters = enrichmentParameters,
        logConfiguration = logConfiguration,
        name = name,
        namePrefix = namePrefix,
        roleArn = roleArn,
        source = source,
        sourceParameters = sourceParameters,
        tags = tags,
        target = target,
        targetParameters = targetParameters,
    )
}
