@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.rds.kotlin

import com.pulumi.aws.rds.ClusterArgs.builder
import com.pulumi.aws.rds.kotlin.enums.EngineMode
import com.pulumi.aws.rds.kotlin.enums.EngineType
import com.pulumi.aws.rds.kotlin.inputs.ClusterRestoreToPointInTimeArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterRestoreToPointInTimeArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.ClusterS3ImportArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterS3ImportArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.ClusterScalingConfigurationArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterScalingConfigurationArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.ClusterServerlessv2ScalingConfigurationArgs
import com.pulumi.aws.rds.kotlin.inputs.ClusterServerlessv2ScalingConfigurationArgsBuilder
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages a [RDS Aurora Cluster](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/AuroraUserGuide/CHAP_Aurora&#46;html) or a [RDS Multi-AZ DB Cluster](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html). To manage cluster instances that inherit configuration from the cluster (when not running the cluster in `serverless` engine mode), see the `aws.rds.ClusterInstance` resource. To manage non-Aurora DB instances (e.g., MySQL, PostgreSQL, SQL Server, etc.), see the `aws.rds.Instance` resource.
 * For information on the difference between the available Aurora MySQL engines see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html) in the Amazon RDS User Guide.
 * Changes to an RDS Cluster can occur when you manually change a parameter, such as `port`, and are reflected in the next maintenance window. Because of this, this provider may report a difference in its planning phase because a modification has not yet taken place. You can use the `apply_immediately` flag to instruct the service to apply the change immediately (see documentation below).
 * > **Note:** Multi-AZ DB clusters are supported only for the MySQL and PostgreSQL DB engines.
 * > **Note:** `ca_certificate_identifier` is only supported for Multi-AZ DB clusters.
 * > **Note:** using `apply_immediately` can result in a brief downtime as the server reboots. See the AWS Docs on [RDS Maintenance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html) for more information.
 * > **Note:** All arguments including the username and password will be stored in the raw state as plain-text.
 * > **NOTE on RDS Clusters and RDS Cluster Role Associations:** Pulumi provides both a standalone RDS Cluster Role Association - (an association between an RDS Cluster and a single IAM Role) and an RDS Cluster resource with `iam_roles` attributes. Use one resource or the other to associate IAM Roles and RDS Clusters. Not doing so will cause a conflict of associations and will result in the association being overwritten.
 * ## Example Usage
 * ### Aurora MySQL 2.x (MySQL 5.7)
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _default = new aws.rds.Cluster("default", {
 *     clusterIdentifier: "aurora-cluster-demo",
 *     engine: aws.rds.EngineType.AuroraMysql,
 *     engineVersion: "5.7.mysql_aurora.2.03.2",
 *     availabilityZones: [
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     databaseName: "mydb",
 *     masterUsername: "foo",
 *     masterPassword: "must_be_eight_characters",
 *     backupRetentionPeriod: 5,
 *     preferredBackupWindow: "07:00-09:00",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * default = aws.rds.Cluster("default",
 *     cluster_identifier="aurora-cluster-demo",
 *     engine=aws.rds.EngineType.AURORA_MYSQL,
 *     engine_version="5.7.mysql_aurora.2.03.2",
 *     availability_zones=[
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     database_name="mydb",
 *     master_username="foo",
 *     master_password="must_be_eight_characters",
 *     backup_retention_period=5,
 *     preferred_backup_window="07:00-09:00")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @default = new Aws.Rds.Cluster("default", new()
 *     {
 *         ClusterIdentifier = "aurora-cluster-demo",
 *         Engine = Aws.Rds.EngineType.AuroraMysql,
 *         EngineVersion = "5.7.mysql_aurora.2.03.2",
 *         AvailabilityZones = new[]
 *         {
 *             "us-west-2a",
 *             "us-west-2b",
 *             "us-west-2c",
 *         },
 *         DatabaseName = "mydb",
 *         MasterUsername = "foo",
 *         MasterPassword = "must_be_eight_characters",
 *         BackupRetentionPeriod = 5,
 *         PreferredBackupWindow = "07:00-09:00",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewCluster(ctx, "default", &rds.ClusterArgs{
 * 			ClusterIdentifier: pulumi.String("aurora-cluster-demo"),
 * 			Engine:            pulumi.String(rds.EngineTypeAuroraMysql),
 * 			EngineVersion:     pulumi.String("5.7.mysql_aurora.2.03.2"),
 * 			AvailabilityZones: pulumi.StringArray{
 * 				pulumi.String("us-west-2a"),
 * 				pulumi.String("us-west-2b"),
 * 				pulumi.String("us-west-2c"),
 * 			},
 * 			DatabaseName:          pulumi.String("mydb"),
 * 			MasterUsername:        pulumi.String("foo"),
 * 			MasterPassword:        pulumi.String("must_be_eight_characters"),
 * 			BackupRetentionPeriod: pulumi.Int(5),
 * 			PreferredBackupWindow: pulumi.String("07:00-09:00"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Cluster("default", ClusterArgs.builder()
 *             .clusterIdentifier("aurora-cluster-demo")
 *             .engine("aurora-mysql")
 *             .engineVersion("5.7.mysql_aurora.2.03.2")
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .databaseName("mydb")
 *             .masterUsername("foo")
 *             .masterPassword("must_be_eight_characters")
 *             .backupRetentionPeriod(5)
 *             .preferredBackupWindow("07:00-09:00")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   default:
 *     type: aws:rds:Cluster
 *     properties:
 *       clusterIdentifier: aurora-cluster-demo
 *       engine: aurora-mysql
 *       engineVersion: 5.7.mysql_aurora.2.03.2
 *       availabilityZones:
 *         - us-west-2a
 *         - us-west-2b
 *         - us-west-2c
 *       databaseName: mydb
 *       masterUsername: foo
 *       masterPassword: must_be_eight_characters
 *       backupRetentionPeriod: 5
 *       preferredBackupWindow: 07:00-09:00
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Aurora MySQL 1.x (MySQL 5.6)
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _default = new aws.rds.Cluster("default", {
 *     clusterIdentifier: "aurora-cluster-demo",
 *     availabilityZones: [
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     databaseName: "mydb",
 *     masterUsername: "foo",
 *     masterPassword: "must_be_eight_characters",
 *     backupRetentionPeriod: 5,
 *     preferredBackupWindow: "07:00-09:00",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * default = aws.rds.Cluster("default",
 *     cluster_identifier="aurora-cluster-demo",
 *     availability_zones=[
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     database_name="mydb",
 *     master_username="foo",
 *     master_password="must_be_eight_characters",
 *     backup_retention_period=5,
 *     preferred_backup_window="07:00-09:00")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @default = new Aws.Rds.Cluster("default", new()
 *     {
 *         ClusterIdentifier = "aurora-cluster-demo",
 *         AvailabilityZones = new[]
 *         {
 *             "us-west-2a",
 *             "us-west-2b",
 *             "us-west-2c",
 *         },
 *         DatabaseName = "mydb",
 *         MasterUsername = "foo",
 *         MasterPassword = "must_be_eight_characters",
 *         BackupRetentionPeriod = 5,
 *         PreferredBackupWindow = "07:00-09:00",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewCluster(ctx, "default", &rds.ClusterArgs{
 * 			ClusterIdentifier: pulumi.String("aurora-cluster-demo"),
 * 			AvailabilityZones: pulumi.StringArray{
 * 				pulumi.String("us-west-2a"),
 * 				pulumi.String("us-west-2b"),
 * 				pulumi.String("us-west-2c"),
 * 			},
 * 			DatabaseName:          pulumi.String("mydb"),
 * 			MasterUsername:        pulumi.String("foo"),
 * 			MasterPassword:        pulumi.String("must_be_eight_characters"),
 * 			BackupRetentionPeriod: pulumi.Int(5),
 * 			PreferredBackupWindow: pulumi.String("07:00-09:00"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Cluster("default", ClusterArgs.builder()
 *             .clusterIdentifier("aurora-cluster-demo")
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .databaseName("mydb")
 *             .masterUsername("foo")
 *             .masterPassword("must_be_eight_characters")
 *             .backupRetentionPeriod(5)
 *             .preferredBackupWindow("07:00-09:00")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   default:
 *     type: aws:rds:Cluster
 *     properties:
 *       clusterIdentifier: aurora-cluster-demo
 *       availabilityZones:
 *         - us-west-2a
 *         - us-west-2b
 *         - us-west-2c
 *       databaseName: mydb
 *       masterUsername: foo
 *       masterPassword: must_be_eight_characters
 *       backupRetentionPeriod: 5
 *       preferredBackupWindow: 07:00-09:00
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Aurora with PostgreSQL engine
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const postgresql = new aws.rds.Cluster("postgresql", {
 *     clusterIdentifier: "aurora-cluster-demo",
 *     engine: aws.rds.EngineType.AuroraPostgresql,
 *     availabilityZones: [
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     databaseName: "mydb",
 *     masterUsername: "foo",
 *     masterPassword: "must_be_eight_characters",
 *     backupRetentionPeriod: 5,
 *     preferredBackupWindow: "07:00-09:00",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * postgresql = aws.rds.Cluster("postgresql",
 *     cluster_identifier="aurora-cluster-demo",
 *     engine=aws.rds.EngineType.AURORA_POSTGRESQL,
 *     availability_zones=[
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     database_name="mydb",
 *     master_username="foo",
 *     master_password="must_be_eight_characters",
 *     backup_retention_period=5,
 *     preferred_backup_window="07:00-09:00")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var postgresql = new Aws.Rds.Cluster("postgresql", new()
 *     {
 *         ClusterIdentifier = "aurora-cluster-demo",
 *         Engine = Aws.Rds.EngineType.AuroraPostgresql,
 *         AvailabilityZones = new[]
 *         {
 *             "us-west-2a",
 *             "us-west-2b",
 *             "us-west-2c",
 *         },
 *         DatabaseName = "mydb",
 *         MasterUsername = "foo",
 *         MasterPassword = "must_be_eight_characters",
 *         BackupRetentionPeriod = 5,
 *         PreferredBackupWindow = "07:00-09:00",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewCluster(ctx, "postgresql", &rds.ClusterArgs{
 * 			ClusterIdentifier: pulumi.String("aurora-cluster-demo"),
 * 			Engine:            pulumi.String(rds.EngineTypeAuroraPostgresql),
 * 			AvailabilityZones: pulumi.StringArray{
 * 				pulumi.String("us-west-2a"),
 * 				pulumi.String("us-west-2b"),
 * 				pulumi.String("us-west-2c"),
 * 			},
 * 			DatabaseName:          pulumi.String("mydb"),
 * 			MasterUsername:        pulumi.String("foo"),
 * 			MasterPassword:        pulumi.String("must_be_eight_characters"),
 * 			BackupRetentionPeriod: pulumi.Int(5),
 * 			PreferredBackupWindow: pulumi.String("07:00-09:00"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var postgresql = new Cluster("postgresql", ClusterArgs.builder()
 *             .clusterIdentifier("aurora-cluster-demo")
 *             .engine("aurora-postgresql")
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .databaseName("mydb")
 *             .masterUsername("foo")
 *             .masterPassword("must_be_eight_characters")
 *             .backupRetentionPeriod(5)
 *             .preferredBackupWindow("07:00-09:00")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   postgresql:
 *     type: aws:rds:Cluster
 *     properties:
 *       clusterIdentifier: aurora-cluster-demo
 *       engine: aurora-postgresql
 *       availabilityZones:
 *         - us-west-2a
 *         - us-west-2b
 *         - us-west-2c
 *       databaseName: mydb
 *       masterUsername: foo
 *       masterPassword: must_be_eight_characters
 *       backupRetentionPeriod: 5
 *       preferredBackupWindow: 07:00-09:00
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Multi-AZ Cluster
 * > More information about RDS Multi-AZ Clusters can be found in the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html).
 * To create a Multi-AZ RDS cluster, you must additionally specify the `engine`, `storage_type`, `allocated_storage`, `iops` and `db_cluster_instance_class` attributes.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.rds.Cluster("example", {
 *     clusterIdentifier: "example",
 *     availabilityZones: [
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     engine: "mysql",
 *     dbClusterInstanceClass: "db.r6gd.xlarge",
 *     storageType: "io1",
 *     allocatedStorage: 100,
 *     iops: 1000,
 *     masterUsername: "test",
 *     masterPassword: "mustbeeightcharaters",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.rds.Cluster("example",
 *     cluster_identifier="example",
 *     availability_zones=[
 *         "us-west-2a",
 *         "us-west-2b",
 *         "us-west-2c",
 *     ],
 *     engine="mysql",
 *     db_cluster_instance_class="db.r6gd.xlarge",
 *     storage_type="io1",
 *     allocated_storage=100,
 *     iops=1000,
 *     master_username="test",
 *     master_password="mustbeeightcharaters")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Rds.Cluster("example", new()
 *     {
 *         ClusterIdentifier = "example",
 *         AvailabilityZones = new[]
 *         {
 *             "us-west-2a",
 *             "us-west-2b",
 *             "us-west-2c",
 *         },
 *         Engine = "mysql",
 *         DbClusterInstanceClass = "db.r6gd.xlarge",
 *         StorageType = "io1",
 *         AllocatedStorage = 100,
 *         Iops = 1000,
 *         MasterUsername = "test",
 *         MasterPassword = "mustbeeightcharaters",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewCluster(ctx, "example", &rds.ClusterArgs{
 * 			ClusterIdentifier: pulumi.String("example"),
 * 			AvailabilityZones: pulumi.StringArray{
 * 				pulumi.String("us-west-2a"),
 * 				pulumi.String("us-west-2b"),
 * 				pulumi.String("us-west-2c"),
 * 			},
 * 			Engine:                 pulumi.String("mysql"),
 * 			DbClusterInstanceClass: pulumi.String("db.r6gd.xlarge"),
 * 			StorageType:            pulumi.String("io1"),
 * 			AllocatedStorage:       pulumi.Int(100),
 * 			Iops:                   pulumi.Int(1000),
 * 			MasterUsername:         pulumi.String("test"),
 * 			MasterPassword:         pulumi.String("mustbeeightcharaters"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .clusterIdentifier("example")
 *             .availabilityZones(
 *                 "us-west-2a",
 *                 "us-west-2b",
 *                 "us-west-2c")
 *             .engine("mysql")
 *             .dbClusterInstanceClass("db.r6gd.xlarge")
 *             .storageType("io1")
 *             .allocatedStorage(100)
 *             .iops(1000)
 *             .masterUsername("test")
 *             .masterPassword("mustbeeightcharaters")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:rds:Cluster
 *     properties:
 *       clusterIdentifier: example
 *       availabilityZones:
 *         - us-west-2a
 *         - us-west-2b
 *         - us-west-2c
 *       engine: mysql
 *       dbClusterInstanceClass: db.r6gd.xlarge
 *       storageType: io1
 *       allocatedStorage: 100
 *       iops: 1000
 *       masterUsername: test
 *       masterPassword: mustbeeightcharaters
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Serverless v2 Cluster
 * > More information about RDS Serverless v2 Clusters can be found in the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.html).
 * > **Note:** Unlike Serverless v1, in Serverless v2 the `storage_encrypted` value is set to `false` by default.
 * This is because Serverless v1 uses the `serverless` `engine_mode`, but Serverless v2 uses the `provisioned` `engine_mode`.
 * To create a Serverless v2 RDS cluster, you must additionally specify the `engine_mode` and `serverlessv2_scaling_configuration` attributes. An `aws.rds.ClusterInstance` resource must also be added to the cluster with the `instance_class` attribute specified.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.rds.Cluster("example", {
 *     clusterIdentifier: "example",
 *     engine: aws.rds.EngineType.AuroraPostgresql,
 *     engineMode: aws.rds.EngineMode.Provisioned,
 *     engineVersion: "13.6",
 *     databaseName: "test",
 *     masterUsername: "test",
 *     masterPassword: "must_be_eight_characters",
 *     storageEncrypted: true,
 *     serverlessv2ScalingConfiguration: {
 *         maxCapacity: 1,
 *         minCapacity: 0.5,
 *     },
 * });
 * const exampleClusterInstance = new aws.rds.ClusterInstance("example", {
 *     clusterIdentifier: example.id,
 *     instanceClass: "db.serverless",
 *     engine: example.engine,
 *     engineVersion: example.engineVersion,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.rds.Cluster("example",
 *     cluster_identifier="example",
 *     engine=aws.rds.EngineType.AURORA_POSTGRESQL,
 *     engine_mode=aws.rds.EngineMode.PROVISIONED,
 *     engine_version="13.6",
 *     database_name="test",
 *     master_username="test",
 *     master_password="must_be_eight_characters",
 *     storage_encrypted=True,
 *     serverlessv2_scaling_configuration={
 *         "max_capacity": 1,
 *         "min_capacity": 0.5,
 *     })
 * example_cluster_instance = aws.rds.ClusterInstance("example",
 *     cluster_identifier=example.id,
 *     instance_class="db.serverless",
 *     engine=example.engine,
 *     engine_version=example.engine_version)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Rds.Cluster("example", new()
 *     {
 *         ClusterIdentifier = "example",
 *         Engine = Aws.Rds.EngineType.AuroraPostgresql,
 *         EngineMode = Aws.Rds.EngineMode.Provisioned,
 *         EngineVersion = "13.6",
 *         DatabaseName = "test",
 *         MasterUsername = "test",
 *         MasterPassword = "must_be_eight_characters",
 *         StorageEncrypted = true,
 *         Serverlessv2ScalingConfiguration = new Aws.Rds.Inputs.ClusterServerlessv2ScalingConfigurationArgs
 *         {
 *             MaxCapacity = 1,
 *             MinCapacity = 0.5,
 *         },
 *     });
 *     var exampleClusterInstance = new Aws.Rds.ClusterInstance("example", new()
 *     {
 *         ClusterIdentifier = example.Id,
 *         InstanceClass = "db.serverless",
 *         Engine = example.Engine,
 *         EngineVersion = example.EngineVersion,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := rds.NewCluster(ctx, "example", &rds.ClusterArgs{
 * 			ClusterIdentifier: pulumi.String("example"),
 * 			Engine:            pulumi.String(rds.EngineTypeAuroraPostgresql),
 * 			EngineMode:        pulumi.String(rds.EngineModeProvisioned),
 * 			EngineVersion:     pulumi.String("13.6"),
 * 			DatabaseName:      pulumi.String("test"),
 * 			MasterUsername:    pulumi.String("test"),
 * 			MasterPassword:    pulumi.String("must_be_eight_characters"),
 * 			StorageEncrypted:  pulumi.Bool(true),
 * 			Serverlessv2ScalingConfiguration: &rds.ClusterServerlessv2ScalingConfigurationArgs{
 * 				MaxCapacity: pulumi.Float64(1),
 * 				MinCapacity: pulumi.Float64(0.5),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewClusterInstance(ctx, "example", &rds.ClusterInstanceArgs{
 * 			ClusterIdentifier: example.ID(),
 * 			InstanceClass:     pulumi.String("db.serverless"),
 * 			Engine:            example.Engine,
 * 			EngineVersion:     example.EngineVersion,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.inputs.ClusterServerlessv2ScalingConfigurationArgs;
 * import com.pulumi.aws.rds.ClusterInstance;
 * import com.pulumi.aws.rds.ClusterInstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Cluster("example", ClusterArgs.builder()
 *             .clusterIdentifier("example")
 *             .engine("aurora-postgresql")
 *             .engineMode("provisioned")
 *             .engineVersion("13.6")
 *             .databaseName("test")
 *             .masterUsername("test")
 *             .masterPassword("must_be_eight_characters")
 *             .storageEncrypted(true)
 *             .serverlessv2ScalingConfiguration(ClusterServerlessv2ScalingConfigurationArgs.builder()
 *                 .maxCapacity(1)
 *                 .minCapacity(0.5)
 *                 .build())
 *             .build());
 *         var exampleClusterInstance = new ClusterInstance("exampleClusterInstance", ClusterInstanceArgs.builder()
 *             .clusterIdentifier(example.id())
 *             .instanceClass("db.serverless")
 *             .engine(example.engine())
 *             .engineVersion(example.engineVersion())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:rds:Cluster
 *     properties:
 *       clusterIdentifier: example
 *       engine: aurora-postgresql
 *       engineMode: provisioned
 *       engineVersion: '13.6'
 *       databaseName: test
 *       masterUsername: test
 *       masterPassword: must_be_eight_characters
 *       storageEncrypted: true
 *       serverlessv2ScalingConfiguration:
 *         maxCapacity: 1
 *         minCapacity: 0.5
 *   exampleClusterInstance:
 *     type: aws:rds:ClusterInstance
 *     name: example
 *     properties:
 *       clusterIdentifier: ${example.id}
 *       instanceClass: db.serverless
 *       engine: ${example.engine}
 *       engineVersion: ${example.engineVersion}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS/Aurora Managed Master Passwords via Secrets Manager, default KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws&#46;amazon&#46;com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `manage_master_user_password` attribute to enable managing the master password with Secrets Manager. You can also update an existing cluster to use Secrets Manager by specify the `manage_master_user_password` attribute and removing the `master_password` attribute (removal is required).
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const test = new aws.rds.Cluster("test", {
 *     clusterIdentifier: "example",
 *     databaseName: "test",
 *     manageMasterUserPassword: true,
 *     masterUsername: "test",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test = aws.rds.Cluster("test",
 *     cluster_identifier="example",
 *     database_name="test",
 *     manage_master_user_password=True,
 *     master_username="test")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var test = new Aws.Rds.Cluster("test", new()
 *     {
 *         ClusterIdentifier = "example",
 *         DatabaseName = "test",
 *         ManageMasterUserPassword = true,
 *         MasterUsername = "test",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewCluster(ctx, "test", &rds.ClusterArgs{
 * 			ClusterIdentifier:        pulumi.String("example"),
 * 			DatabaseName:             pulumi.String("test"),
 * 			ManageMasterUserPassword: pulumi.Bool(true),
 * 			MasterUsername:           pulumi.String("test"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new Cluster("test", ClusterArgs.builder()
 *             .clusterIdentifier("example")
 *             .databaseName("test")
 *             .manageMasterUserPassword(true)
 *             .masterUsername("test")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   test:
 *     type: aws:rds:Cluster
 *     properties:
 *       clusterIdentifier: example
 *       databaseName: test
 *       manageMasterUserPassword: true
 *       masterUsername: test
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS/Aurora Managed Master Passwords via Secrets Manager, specific KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws&#46;amazon&#46;com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `master_user_secret_kms_key_id` attribute to specify a specific KMS Key.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kms.Key("example", {description: "Example KMS Key"});
 * const test = new aws.rds.Cluster("test", {
 *     clusterIdentifier: "example",
 *     databaseName: "test",
 *     manageMasterUserPassword: true,
 *     masterUsername: "test",
 *     masterUserSecretKmsKeyId: example.keyId,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kms.Key("example", description="Example KMS Key")
 * test = aws.rds.Cluster("test",
 *     cluster_identifier="example",
 *     database_name="test",
 *     manage_master_user_password=True,
 *     master_username="test",
 *     master_user_secret_kms_key_id=example.key_id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kms.Key("example", new()
 *     {
 *         Description = "Example KMS Key",
 *     });
 *     var test = new Aws.Rds.Cluster("test", new()
 *     {
 *         ClusterIdentifier = "example",
 *         DatabaseName = "test",
 *         ManageMasterUserPassword = true,
 *         MasterUsername = "test",
 *         MasterUserSecretKmsKeyId = example.KeyId,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := kms.NewKey(ctx, "example", &kms.KeyArgs{
 * 			Description: pulumi.String("Example KMS Key"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewCluster(ctx, "test", &rds.ClusterArgs{
 * 			ClusterIdentifier:        pulumi.String("example"),
 * 			DatabaseName:             pulumi.String("test"),
 * 			ManageMasterUserPassword: pulumi.Bool(true),
 * 			MasterUsername:           pulumi.String("test"),
 * 			MasterUserSecretKmsKeyId: example.KeyId,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Key("example", KeyArgs.builder()
 *             .description("Example KMS Key")
 *             .build());
 *         var test = new Cluster("test", ClusterArgs.builder()
 *             .clusterIdentifier("example")
 *             .databaseName("test")
 *             .manageMasterUserPassword(true)
 *             .masterUsername("test")
 *             .masterUserSecretKmsKeyId(example.keyId())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kms:Key
 *     properties:
 *       description: Example KMS Key
 *   test:
 *     type: aws:rds:Cluster
 *     properties:
 *       clusterIdentifier: example
 *       databaseName: test
 *       manageMasterUserPassword: true
 *       masterUsername: test
 *       masterUserSecretKmsKeyId: ${example.keyId}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Global Cluster Restored From Snapshot
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = aws.rds.getClusterSnapshot({
 *     dbClusterIdentifier: "example-original-cluster",
 *     mostRecent: true,
 * });
 * const exampleCluster = new aws.rds.Cluster("example", {
 *     engine: aws.rds.EngineType.Aurora,
 *     engineVersion: "5.6.mysql_aurora.1.22.4",
 *     clusterIdentifier: "example",
 *     snapshotIdentifier: example.then(example => example.id),
 * });
 * const exampleGlobalCluster = new aws.rds.GlobalCluster("example", {
 *     globalClusterIdentifier: "example",
 *     sourceDbClusterIdentifier: exampleCluster.arn,
 *     forceDestroy: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.rds.get_cluster_snapshot(db_cluster_identifier="example-original-cluster",
 *     most_recent=True)
 * example_cluster = aws.rds.Cluster("example",
 *     engine=aws.rds.EngineType.AURORA,
 *     engine_version="5.6.mysql_aurora.1.22.4",
 *     cluster_identifier="example",
 *     snapshot_identifier=example.id)
 * example_global_cluster = aws.rds.GlobalCluster("example",
 *     global_cluster_identifier="example",
 *     source_db_cluster_identifier=example_cluster.arn,
 *     force_destroy=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = Aws.Rds.GetClusterSnapshot.Invoke(new()
 *     {
 *         DbClusterIdentifier = "example-original-cluster",
 *         MostRecent = true,
 *     });
 *     var exampleCluster = new Aws.Rds.Cluster("example", new()
 *     {
 *         Engine = Aws.Rds.EngineType.Aurora,
 *         EngineVersion = "5.6.mysql_aurora.1.22.4",
 *         ClusterIdentifier = "example",
 *         SnapshotIdentifier = example.Apply(getClusterSnapshotResult => getClusterSnapshotResult.Id),
 *     });
 *     var exampleGlobalCluster = new Aws.Rds.GlobalCluster("example", new()
 *     {
 *         GlobalClusterIdentifier = "example",
 *         SourceDbClusterIdentifier = exampleCluster.Arn,
 *         ForceDestroy = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := rds.LookupClusterSnapshot(ctx, &rds.LookupClusterSnapshotArgs{
 * 			DbClusterIdentifier: pulumi.StringRef("example-original-cluster"),
 * 			MostRecent:          pulumi.BoolRef(true),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleCluster, err := rds.NewCluster(ctx, "example", &rds.ClusterArgs{
 * 			Engine:             pulumi.String(rds.EngineTypeAurora),
 * 			EngineVersion:      pulumi.String("5.6.mysql_aurora.1.22.4"),
 * 			ClusterIdentifier:  pulumi.String("example"),
 * 			SnapshotIdentifier: pulumi.String(example.Id),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewGlobalCluster(ctx, "example", &rds.GlobalClusterArgs{
 * 			GlobalClusterIdentifier:   pulumi.String("example"),
 * 			SourceDbClusterIdentifier: exampleCluster.Arn,
 * 			ForceDestroy:              pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.RdsFunctions;
 * import com.pulumi.aws.rds.inputs.GetClusterSnapshotArgs;
 * import com.pulumi.aws.rds.Cluster;
 * import com.pulumi.aws.rds.ClusterArgs;
 * import com.pulumi.aws.rds.GlobalCluster;
 * import com.pulumi.aws.rds.GlobalClusterArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = RdsFunctions.getClusterSnapshot(GetClusterSnapshotArgs.builder()
 *             .dbClusterIdentifier("example-original-cluster")
 *             .mostRecent(true)
 *             .build());
 *         var exampleCluster = new Cluster("exampleCluster", ClusterArgs.builder()
 *             .engine("aurora")
 *             .engineVersion("5.6.mysql_aurora.1.22.4")
 *             .clusterIdentifier("example")
 *             .snapshotIdentifier(example.applyValue(getClusterSnapshotResult -> getClusterSnapshotResult.id()))
 *             .build());
 *         var exampleGlobalCluster = new GlobalCluster("exampleGlobalCluster", GlobalClusterArgs.builder()
 *             .globalClusterIdentifier("example")
 *             .sourceDbClusterIdentifier(exampleCluster.arn())
 *             .forceDestroy(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleCluster:
 *     type: aws:rds:Cluster
 *     name: example
 *     properties:
 *       engine: aurora
 *       engineVersion: 5.6.mysql_aurora.1.22.4
 *       clusterIdentifier: example
 *       snapshotIdentifier: ${example.id}
 *   exampleGlobalCluster:
 *     type: aws:rds:GlobalCluster
 *     name: example
 *     properties:
 *       globalClusterIdentifier: example
 *       sourceDbClusterIdentifier: ${exampleCluster.arn}
 *       forceDestroy: true
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: aws:rds:getClusterSnapshot
 *       Arguments:
 *         dbClusterIdentifier: example-original-cluster
 *         mostRecent: true
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import RDS Clusters using the `cluster_identifier`. For example:
 * ```sh
 * $ pulumi import aws:rds/cluster:Cluster aurora_cluster aurora-prod-cluster
 * ```
 * @property allocatedStorage The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.
 * @property allowMajorVersionUpgrade Enable to allow major engine version upgrades when changing engine versions. Defaults to `false`.
 * @property applyImmediately Specifies whether any cluster modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon RDS Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
 * @property availabilityZones List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
 * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
 * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
 * A maximum of 3 AZs can be configured.
 * @property backtrackWindow Target backtrack window, in seconds. Only available for `aurora` and `aurora-mysql` engines currently. To disable backtracking, set this value to `0`. Defaults to `0`. Must be between `0` and `259200` (72 hours)
 * @property backupRetentionPeriod Days to retain backups for. Default `1`
 * @property caCertificateIdentifier The CA certificate identifier to use for the DB cluster's server certificate.
 * @property clusterIdentifier The cluster identifier. If omitted, this provider will assign a random, unique identifier.
 * @property clusterIdentifierPrefix Creates a unique cluster identifier beginning with the specified prefix. Conflicts with `cluster_identifier`.
 * @property clusterMembers List of RDS Instances that are a part of this cluster
 * @property copyTagsToSnapshot Copy all Cluster `tags` to snapshots. Default is `false`.
 * @property databaseName Name for an automatically created database on cluster creation. There are different naming restrictions per database engine: [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints)
 * @property dbClusterInstanceClass The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example `db.m6g.xlarge`. Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes and availability for your engine, see [DB instance class](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) in the Amazon RDS User Guide.
 * @property dbClusterParameterGroupName A cluster parameter group to associate with the cluster.
 * @property dbInstanceParameterGroupName Instance parameter group to associate with all instances of the DB cluster. The `db_instance_parameter_group_name` parameter is only valid in combination with the `allow_major_version_upgrade` parameter.
 * @property dbSubnetGroupName DB subnet group to associate with this DB cluster.
 * **NOTE:** This must match the `db_subnet_group_name` specified on every `aws.rds.ClusterInstance` in the cluster.
 * @property dbSystemId For use with RDS Custom.
 * @property deleteAutomatedBackups Specifies whether to remove automated backups immediately after the DB cluster is deleted. Default is `true`.
 * @property deletionProtection If the DB cluster should have deletion protection enabled.
 * The database can't be deleted when this value is set to `true`.
 * The default is `false`.
 * @property domain The ID of the Directory Service Active Directory domain to create the cluster in.
 * @property domainIamRoleName The name of the IAM role to be used when making API calls to the Directory Service.
 * @property enableGlobalWriteForwarding Whether cluster should forward writes to an associated global cluster. Applied to secondary clusters to enable them to forward writes to an `aws.rds.GlobalCluster`'s primary cluster. See the [User Guide for Aurora](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-write-forwarding.html) for more information.
 * @property enableHttpEndpoint Enable HTTP endpoint (data API). Only valid for some combinations of `engine_mode`, `engine` and `engine_version` and only available in some regions. See the [Region and version availability](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html#data-api.regions) section of the documentation. This option also does not work with any of these options specified: `snapshot_identifier`, `replication_source_identifier`, `s3_import`.
 * @property enableLocalWriteForwarding Whether read replicas can forward write operations to the writer DB instance in the DB cluster. By default, write operations aren't allowed on reader DB instances.. See the [User Guide for Aurora](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-write-forwarding.html) for more information. **NOTE:** Local write forwarding requires Aurora MySQL version 3.04 or higher.
 * @property enabledCloudwatchLogsExports Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
 * @property engine Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
 * @property engineLifecycleSupport The life cycle type for this DB instance. This setting is valid for cluster types Aurora DB clusters and Multi-AZ DB clusters. Valid values are `open-source-rds-extended-support`, `open-source-rds-extended-support-disabled`. Default value is `open-source-rds-extended-support`. [Using Amazon RDS Extended Support]: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/extended-support.html
 * @property engineMode Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html) for limitations when using `serverless`.
 * @property engineVersion Database engine version. Updating this argument results in an outage. See the [Aurora MySQL](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL&#46;Updates&#46;html) and [Aurora Postgres](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.html) documentation for your configured engine to determine this value, or by running `aws rds describe-db-engine-versions`. For example with Aurora MySQL 2, a potential value for this argument is `5.7.mysql_aurora.2.03.2`. The value can contain a partial version where supported by the API. The actual engine version used is returned in the attribute `engine_version_actual`, , see Attribute Reference below.
 * @property finalSnapshotIdentifier Name of your final DB snapshot when this DB cluster is deleted. If omitted, no final snapshot will be made.
 * @property globalClusterIdentifier Global cluster identifier specified on `aws.rds.GlobalCluster`.
 * @property iamDatabaseAuthenticationEnabled Specifies whether or not mappings of AWS Identity and Access Management (IAM) accounts to database accounts is enabled. Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html) for availability and limitations.
 * @property iamRoles List of ARNs for the IAM roles to associate to the RDS Cluster.
 * @property iops Amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid Iops values, see [Amazon RDS Provisioned IOPS storage to improve performance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS) in the Amazon RDS User Guide. (This setting is required to create a Multi-AZ DB cluster). Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
 * @property kmsKeyId ARN for the KMS encryption key. When specifying `kms_key_id`, `storage_encrypted` needs to be set to true.
 * @property manageMasterUserPassword Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `master_password` is provided.
 * @property masterPassword Password for the master DB user. Note that this may show up in logs, and it will be stored in the state file. Please refer to the [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). Cannot be set if `manage_master_user_password` is set to `true`.
 * @property masterUserSecretKmsKeyId Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
 * @property masterUsername Username for the master DB user. Please refer to the [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). This argument does not support in-place updates and cannot be changed during a restore from snapshot.
 * @property networkType Network type of the cluster. Valid values: `IPV4`, `DUAL`.
 * @property performanceInsightsEnabled Valid only for Non-Aurora Multi-AZ DB Clusters. Enables Performance Insights for the RDS Cluster
 * @property performanceInsightsKmsKeyId Valid only for Non-Aurora Multi-AZ DB Clusters. Specifies the KMS Key ID to encrypt Performance Insights data. If not specified, the default RDS KMS key will be used (`aws/rds`).
 * @property performanceInsightsRetentionPeriod Valid only for Non-Aurora Multi-AZ DB Clusters. Specifies the amount of time to retain performance insights data for. Defaults to 7 days if Performance Insights are enabled. Valid values are `7`, `month * 31` (where month is a number of months from 1-23), and `731`. See [here](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.Overview.cost.html) for more information on retention periods.
 * @property port Port on which the DB accepts connections.
 * @property preferredBackupWindow Daily time range during which automated backups are created if automated backups are enabled using the BackupRetentionPeriod parameter.Time in UTC. Default: A 30-minute window selected at random from an 8-hour block of time per region, e.g. `04:00-09:00`.
 * @property preferredMaintenanceWindow Weekly time range during which system maintenance can occur, in (UTC) e.g., `wed:04:00-wed:04:30`
 * @property replicationSourceIdentifier ARN of a source DB cluster or DB instance if this DB cluster is to be created as a Read Replica. If DB Cluster is part of a Global Cluster, use the `lifecycle` configuration block `ignore_changes` argument to prevent this provider from showing differences for this argument instead of configuring this value.
 * @property restoreToPointInTime Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-pitr.html). More details below.
 * @property s3Import
 * @property scalingConfiguration Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
 * @property serverlessv2ScalingConfiguration Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
 * @property skipFinalSnapshot Determines whether a final DB snapshot is created before the DB cluster is deleted. If true is specified, no DB snapshot is created. If false is specified, a DB snapshot is created before the DB cluster is deleted, using the value from `final_snapshot_identifier`. Default is `false`.
 * @property snapshotIdentifier Specifies whether or not to create this cluster from a snapshot. You can use either the name or ARN when specifying a DB cluster snapshot, or the ARN when specifying a DB snapshot. Conflicts with `global_cluster_identifier`. Clusters cannot be restored from snapshot **and** joined to an existing global cluster in a single operation. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-getting-started.html#aurora-global-database.use-snapshot) or the Global Cluster Restored From Snapshot example for instructions on building a global cluster starting with a snapshot.
 * @property sourceRegion The source region for an encrypted replica DB cluster.
 * @property storageEncrypted Specifies whether the DB cluster is encrypted. The default is `false` for `provisioned` `engine_mode` and `true` for `serverless` `engine_mode`. When restoring an unencrypted `snapshot_identifier`, the `kms_key_id` argument must be provided to encrypt the restored cluster. The provider will only perform drift detection if a configuration value is provided.
 * @property storageType (Forces new for Multi-AZ DB clusters) Specifies the storage type to be associated with the DB cluster. For Aurora DB clusters, `storage_type` modifications can be done in-place. For Multi-AZ DB Clusters, the `iops` argument must also be set. Valid values are: `""`, `aurora-iopt1` (Aurora DB Clusters); `io1`, `io2` (Multi-AZ DB Clusters). Default: `""` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters).
 * @property tags A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcSecurityGroupIds List of VPC security groups to associate with the Cluster
 */
public data class ClusterArgs(
    public val allocatedStorage: Output<Int>? = null,
    public val allowMajorVersionUpgrade: Output<Boolean>? = null,
    public val applyImmediately: Output<Boolean>? = null,
    public val availabilityZones: Output<List<String>>? = null,
    public val backtrackWindow: Output<Int>? = null,
    public val backupRetentionPeriod: Output<Int>? = null,
    public val caCertificateIdentifier: Output<String>? = null,
    public val clusterIdentifier: Output<String>? = null,
    public val clusterIdentifierPrefix: Output<String>? = null,
    public val clusterMembers: Output<List<String>>? = null,
    public val copyTagsToSnapshot: Output<Boolean>? = null,
    public val databaseName: Output<String>? = null,
    public val dbClusterInstanceClass: Output<String>? = null,
    public val dbClusterParameterGroupName: Output<String>? = null,
    public val dbInstanceParameterGroupName: Output<String>? = null,
    public val dbSubnetGroupName: Output<String>? = null,
    public val dbSystemId: Output<String>? = null,
    public val deleteAutomatedBackups: Output<Boolean>? = null,
    public val deletionProtection: Output<Boolean>? = null,
    public val domain: Output<String>? = null,
    public val domainIamRoleName: Output<String>? = null,
    public val enableGlobalWriteForwarding: Output<Boolean>? = null,
    public val enableHttpEndpoint: Output<Boolean>? = null,
    public val enableLocalWriteForwarding: Output<Boolean>? = null,
    public val enabledCloudwatchLogsExports: Output<List<String>>? = null,
    public val engine: Output<Either<String, EngineType>>? = null,
    public val engineLifecycleSupport: Output<String>? = null,
    public val engineMode: Output<Either<String, EngineMode>>? = null,
    public val engineVersion: Output<String>? = null,
    public val finalSnapshotIdentifier: Output<String>? = null,
    public val globalClusterIdentifier: Output<String>? = null,
    public val iamDatabaseAuthenticationEnabled: Output<Boolean>? = null,
    public val iamRoles: Output<List<String>>? = null,
    public val iops: Output<Int>? = null,
    public val kmsKeyId: Output<String>? = null,
    public val manageMasterUserPassword: Output<Boolean>? = null,
    public val masterPassword: Output<String>? = null,
    public val masterUserSecretKmsKeyId: Output<String>? = null,
    public val masterUsername: Output<String>? = null,
    public val networkType: Output<String>? = null,
    public val performanceInsightsEnabled: Output<Boolean>? = null,
    public val performanceInsightsKmsKeyId: Output<String>? = null,
    public val performanceInsightsRetentionPeriod: Output<Int>? = null,
    public val port: Output<Int>? = null,
    public val preferredBackupWindow: Output<String>? = null,
    public val preferredMaintenanceWindow: Output<String>? = null,
    public val replicationSourceIdentifier: Output<String>? = null,
    public val restoreToPointInTime: Output<ClusterRestoreToPointInTimeArgs>? = null,
    public val s3Import: Output<ClusterS3ImportArgs>? = null,
    public val scalingConfiguration: Output<ClusterScalingConfigurationArgs>? = null,
    public val serverlessv2ScalingConfiguration: Output<ClusterServerlessv2ScalingConfigurationArgs>? =
        null,
    public val skipFinalSnapshot: Output<Boolean>? = null,
    public val snapshotIdentifier: Output<String>? = null,
    public val sourceRegion: Output<String>? = null,
    public val storageEncrypted: Output<Boolean>? = null,
    public val storageType: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcSecurityGroupIds: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.rds.ClusterArgs> {
    override fun toJava(): com.pulumi.aws.rds.ClusterArgs = com.pulumi.aws.rds.ClusterArgs.builder()
        .allocatedStorage(allocatedStorage?.applyValue({ args0 -> args0 }))
        .allowMajorVersionUpgrade(allowMajorVersionUpgrade?.applyValue({ args0 -> args0 }))
        .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
        .availabilityZones(availabilityZones?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .backtrackWindow(backtrackWindow?.applyValue({ args0 -> args0 }))
        .backupRetentionPeriod(backupRetentionPeriod?.applyValue({ args0 -> args0 }))
        .caCertificateIdentifier(caCertificateIdentifier?.applyValue({ args0 -> args0 }))
        .clusterIdentifier(clusterIdentifier?.applyValue({ args0 -> args0 }))
        .clusterIdentifierPrefix(clusterIdentifierPrefix?.applyValue({ args0 -> args0 }))
        .clusterMembers(clusterMembers?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .copyTagsToSnapshot(copyTagsToSnapshot?.applyValue({ args0 -> args0 }))
        .databaseName(databaseName?.applyValue({ args0 -> args0 }))
        .dbClusterInstanceClass(dbClusterInstanceClass?.applyValue({ args0 -> args0 }))
        .dbClusterParameterGroupName(dbClusterParameterGroupName?.applyValue({ args0 -> args0 }))
        .dbInstanceParameterGroupName(dbInstanceParameterGroupName?.applyValue({ args0 -> args0 }))
        .dbSubnetGroupName(dbSubnetGroupName?.applyValue({ args0 -> args0 }))
        .dbSystemId(dbSystemId?.applyValue({ args0 -> args0 }))
        .deleteAutomatedBackups(deleteAutomatedBackups?.applyValue({ args0 -> args0 }))
        .deletionProtection(deletionProtection?.applyValue({ args0 -> args0 }))
        .domain(domain?.applyValue({ args0 -> args0 }))
        .domainIamRoleName(domainIamRoleName?.applyValue({ args0 -> args0 }))
        .enableGlobalWriteForwarding(enableGlobalWriteForwarding?.applyValue({ args0 -> args0 }))
        .enableHttpEndpoint(enableHttpEndpoint?.applyValue({ args0 -> args0 }))
        .enableLocalWriteForwarding(enableLocalWriteForwarding?.applyValue({ args0 -> args0 }))
        .enabledCloudwatchLogsExports(
            enabledCloudwatchLogsExports?.applyValue({ args0 ->
                args0.map({ args0 -> args0 })
            }),
        )
        .engine(
            engine?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .engineLifecycleSupport(engineLifecycleSupport?.applyValue({ args0 -> args0 }))
        .engineMode(
            engineMode?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
        .finalSnapshotIdentifier(finalSnapshotIdentifier?.applyValue({ args0 -> args0 }))
        .globalClusterIdentifier(globalClusterIdentifier?.applyValue({ args0 -> args0 }))
        .iamDatabaseAuthenticationEnabled(iamDatabaseAuthenticationEnabled?.applyValue({ args0 -> args0 }))
        .iamRoles(iamRoles?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .iops(iops?.applyValue({ args0 -> args0 }))
        .kmsKeyId(kmsKeyId?.applyValue({ args0 -> args0 }))
        .manageMasterUserPassword(manageMasterUserPassword?.applyValue({ args0 -> args0 }))
        .masterPassword(masterPassword?.applyValue({ args0 -> args0 }))
        .masterUserSecretKmsKeyId(masterUserSecretKmsKeyId?.applyValue({ args0 -> args0 }))
        .masterUsername(masterUsername?.applyValue({ args0 -> args0 }))
        .networkType(networkType?.applyValue({ args0 -> args0 }))
        .performanceInsightsEnabled(performanceInsightsEnabled?.applyValue({ args0 -> args0 }))
        .performanceInsightsKmsKeyId(performanceInsightsKmsKeyId?.applyValue({ args0 -> args0 }))
        .performanceInsightsRetentionPeriod(
            performanceInsightsRetentionPeriod?.applyValue({ args0 ->
                args0
            }),
        )
        .port(port?.applyValue({ args0 -> args0 }))
        .preferredBackupWindow(preferredBackupWindow?.applyValue({ args0 -> args0 }))
        .preferredMaintenanceWindow(preferredMaintenanceWindow?.applyValue({ args0 -> args0 }))
        .replicationSourceIdentifier(replicationSourceIdentifier?.applyValue({ args0 -> args0 }))
        .restoreToPointInTime(
            restoreToPointInTime?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .s3Import(s3Import?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .scalingConfiguration(
            scalingConfiguration?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .serverlessv2ScalingConfiguration(
            serverlessv2ScalingConfiguration?.applyValue({ args0 ->
                args0.let({ args0 -> args0.toJava() })
            }),
        )
        .skipFinalSnapshot(skipFinalSnapshot?.applyValue({ args0 -> args0 }))
        .snapshotIdentifier(snapshotIdentifier?.applyValue({ args0 -> args0 }))
        .sourceRegion(sourceRegion?.applyValue({ args0 -> args0 }))
        .storageEncrypted(storageEncrypted?.applyValue({ args0 -> args0 }))
        .storageType(storageType?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .vpcSecurityGroupIds(
            vpcSecurityGroupIds?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        ).build()
}

/**
 * Builder for [ClusterArgs].
 */
@PulumiTagMarker
public class ClusterArgsBuilder internal constructor() {
    private var allocatedStorage: Output<Int>? = null

    private var allowMajorVersionUpgrade: Output<Boolean>? = null

    private var applyImmediately: Output<Boolean>? = null

    private var availabilityZones: Output<List<String>>? = null

    private var backtrackWindow: Output<Int>? = null

    private var backupRetentionPeriod: Output<Int>? = null

    private var caCertificateIdentifier: Output<String>? = null

    private var clusterIdentifier: Output<String>? = null

    private var clusterIdentifierPrefix: Output<String>? = null

    private var clusterMembers: Output<List<String>>? = null

    private var copyTagsToSnapshot: Output<Boolean>? = null

    private var databaseName: Output<String>? = null

    private var dbClusterInstanceClass: Output<String>? = null

    private var dbClusterParameterGroupName: Output<String>? = null

    private var dbInstanceParameterGroupName: Output<String>? = null

    private var dbSubnetGroupName: Output<String>? = null

    private var dbSystemId: Output<String>? = null

    private var deleteAutomatedBackups: Output<Boolean>? = null

    private var deletionProtection: Output<Boolean>? = null

    private var domain: Output<String>? = null

    private var domainIamRoleName: Output<String>? = null

    private var enableGlobalWriteForwarding: Output<Boolean>? = null

    private var enableHttpEndpoint: Output<Boolean>? = null

    private var enableLocalWriteForwarding: Output<Boolean>? = null

    private var enabledCloudwatchLogsExports: Output<List<String>>? = null

    private var engine: Output<Either<String, EngineType>>? = null

    private var engineLifecycleSupport: Output<String>? = null

    private var engineMode: Output<Either<String, EngineMode>>? = null

    private var engineVersion: Output<String>? = null

    private var finalSnapshotIdentifier: Output<String>? = null

    private var globalClusterIdentifier: Output<String>? = null

    private var iamDatabaseAuthenticationEnabled: Output<Boolean>? = null

    private var iamRoles: Output<List<String>>? = null

    private var iops: Output<Int>? = null

    private var kmsKeyId: Output<String>? = null

    private var manageMasterUserPassword: Output<Boolean>? = null

    private var masterPassword: Output<String>? = null

    private var masterUserSecretKmsKeyId: Output<String>? = null

    private var masterUsername: Output<String>? = null

    private var networkType: Output<String>? = null

    private var performanceInsightsEnabled: Output<Boolean>? = null

    private var performanceInsightsKmsKeyId: Output<String>? = null

    private var performanceInsightsRetentionPeriod: Output<Int>? = null

    private var port: Output<Int>? = null

    private var preferredBackupWindow: Output<String>? = null

    private var preferredMaintenanceWindow: Output<String>? = null

    private var replicationSourceIdentifier: Output<String>? = null

    private var restoreToPointInTime: Output<ClusterRestoreToPointInTimeArgs>? = null

    private var s3Import: Output<ClusterS3ImportArgs>? = null

    private var scalingConfiguration: Output<ClusterScalingConfigurationArgs>? = null

    private var serverlessv2ScalingConfiguration: Output<ClusterServerlessv2ScalingConfigurationArgs>? =
        null

    private var skipFinalSnapshot: Output<Boolean>? = null

    private var snapshotIdentifier: Output<String>? = null

    private var sourceRegion: Output<String>? = null

    private var storageEncrypted: Output<Boolean>? = null

    private var storageType: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcSecurityGroupIds: Output<List<String>>? = null

    /**
     * @param value The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.
     */
    @JvmName("neyatcfijqiunilk")
    public suspend fun allocatedStorage(`value`: Output<Int>) {
        this.allocatedStorage = value
    }

    /**
     * @param value Enable to allow major engine version upgrades when changing engine versions. Defaults to `false`.
     */
    @JvmName("bpmivlbtwjcvkurl")
    public suspend fun allowMajorVersionUpgrade(`value`: Output<Boolean>) {
        this.allowMajorVersionUpgrade = value
    }

    /**
     * @param value Specifies whether any cluster modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon RDS Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("fhnygpwkncdvkjmr")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("gredudciluhukfto")
    public suspend fun availabilityZones(`value`: Output<List<String>>) {
        this.availabilityZones = value
    }

    @JvmName("cbphtruorddgkgef")
    public suspend fun availabilityZones(vararg values: Output<String>) {
        this.availabilityZones = Output.all(values.asList())
    }

    /**
     * @param values List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("tbyubxhebtoltlov")
    public suspend fun availabilityZones(values: List<Output<String>>) {
        this.availabilityZones = Output.all(values)
    }

    /**
     * @param value Target backtrack window, in seconds. Only available for `aurora` and `aurora-mysql` engines currently. To disable backtracking, set this value to `0`. Defaults to `0`. Must be between `0` and `259200` (72 hours)
     */
    @JvmName("baomiejtctlurpjw")
    public suspend fun backtrackWindow(`value`: Output<Int>) {
        this.backtrackWindow = value
    }

    /**
     * @param value Days to retain backups for. Default `1`
     */
    @JvmName("jlooitgtaoiqpnlk")
    public suspend fun backupRetentionPeriod(`value`: Output<Int>) {
        this.backupRetentionPeriod = value
    }

    /**
     * @param value The CA certificate identifier to use for the DB cluster's server certificate.
     */
    @JvmName("ymtjvnwkclajrmia")
    public suspend fun caCertificateIdentifier(`value`: Output<String>) {
        this.caCertificateIdentifier = value
    }

    /**
     * @param value The cluster identifier. If omitted, this provider will assign a random, unique identifier.
     */
    @JvmName("xmofyftlhhgthrjm")
    public suspend fun clusterIdentifier(`value`: Output<String>) {
        this.clusterIdentifier = value
    }

    /**
     * @param value Creates a unique cluster identifier beginning with the specified prefix. Conflicts with `cluster_identifier`.
     */
    @JvmName("ydqqurpfsdekfreq")
    public suspend fun clusterIdentifierPrefix(`value`: Output<String>) {
        this.clusterIdentifierPrefix = value
    }

    /**
     * @param value List of RDS Instances that are a part of this cluster
     */
    @JvmName("ustyrpgyrqboheoo")
    public suspend fun clusterMembers(`value`: Output<List<String>>) {
        this.clusterMembers = value
    }

    @JvmName("ifkedthhblpmrnqa")
    public suspend fun clusterMembers(vararg values: Output<String>) {
        this.clusterMembers = Output.all(values.asList())
    }

    /**
     * @param values List of RDS Instances that are a part of this cluster
     */
    @JvmName("brvpjgdvqhsbnykh")
    public suspend fun clusterMembers(values: List<Output<String>>) {
        this.clusterMembers = Output.all(values)
    }

    /**
     * @param value Copy all Cluster `tags` to snapshots. Default is `false`.
     */
    @JvmName("upxhewpkohfalyxl")
    public suspend fun copyTagsToSnapshot(`value`: Output<Boolean>) {
        this.copyTagsToSnapshot = value
    }

    /**
     * @param value Name for an automatically created database on cluster creation. There are different naming restrictions per database engine: [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints)
     */
    @JvmName("welbylogqbqwvqku")
    public suspend fun databaseName(`value`: Output<String>) {
        this.databaseName = value
    }

    /**
     * @param value The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example `db.m6g.xlarge`. Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes and availability for your engine, see [DB instance class](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) in the Amazon RDS User Guide.
     */
    @JvmName("mnbcxvanyopwnkox")
    public suspend fun dbClusterInstanceClass(`value`: Output<String>) {
        this.dbClusterInstanceClass = value
    }

    /**
     * @param value A cluster parameter group to associate with the cluster.
     */
    @JvmName("ivghcsbsiqkqawae")
    public suspend fun dbClusterParameterGroupName(`value`: Output<String>) {
        this.dbClusterParameterGroupName = value
    }

    /**
     * @param value Instance parameter group to associate with all instances of the DB cluster. The `db_instance_parameter_group_name` parameter is only valid in combination with the `allow_major_version_upgrade` parameter.
     */
    @JvmName("vcdtldxfedjdjcsa")
    public suspend fun dbInstanceParameterGroupName(`value`: Output<String>) {
        this.dbInstanceParameterGroupName = value
    }

    /**
     * @param value DB subnet group to associate with this DB cluster.
     * **NOTE:** This must match the `db_subnet_group_name` specified on every `aws.rds.ClusterInstance` in the cluster.
     */
    @JvmName("qjkovecsshbprjrb")
    public suspend fun dbSubnetGroupName(`value`: Output<String>) {
        this.dbSubnetGroupName = value
    }

    /**
     * @param value For use with RDS Custom.
     */
    @JvmName("cmogjedthwilfmto")
    public suspend fun dbSystemId(`value`: Output<String>) {
        this.dbSystemId = value
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB cluster is deleted. Default is `true`.
     */
    @JvmName("cckuasqicprsldvu")
    public suspend fun deleteAutomatedBackups(`value`: Output<Boolean>) {
        this.deleteAutomatedBackups = value
    }

    /**
     * @param value If the DB cluster should have deletion protection enabled.
     * The database can't be deleted when this value is set to `true`.
     * The default is `false`.
     */
    @JvmName("yhrtekggcwjduwjl")
    public suspend fun deletionProtection(`value`: Output<Boolean>) {
        this.deletionProtection = value
    }

    /**
     * @param value The ID of the Directory Service Active Directory domain to create the cluster in.
     */
    @JvmName("xwintxciliuafthe")
    public suspend fun domain(`value`: Output<String>) {
        this.domain = value
    }

    /**
     * @param value The name of the IAM role to be used when making API calls to the Directory Service.
     */
    @JvmName("qkjikdsywojgwmvx")
    public suspend fun domainIamRoleName(`value`: Output<String>) {
        this.domainIamRoleName = value
    }

    /**
     * @param value Whether cluster should forward writes to an associated global cluster. Applied to secondary clusters to enable them to forward writes to an `aws.rds.GlobalCluster`'s primary cluster. See the [User Guide for Aurora](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-write-forwarding.html) for more information.
     */
    @JvmName("phxpeplmsacimuep")
    public suspend fun enableGlobalWriteForwarding(`value`: Output<Boolean>) {
        this.enableGlobalWriteForwarding = value
    }

    /**
     * @param value Enable HTTP endpoint (data API). Only valid for some combinations of `engine_mode`, `engine` and `engine_version` and only available in some regions. See the [Region and version availability](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html#data-api.regions) section of the documentation. This option also does not work with any of these options specified: `snapshot_identifier`, `replication_source_identifier`, `s3_import`.
     */
    @JvmName("xlkqbegwpbxhawlr")
    public suspend fun enableHttpEndpoint(`value`: Output<Boolean>) {
        this.enableHttpEndpoint = value
    }

    /**
     * @param value Whether read replicas can forward write operations to the writer DB instance in the DB cluster. By default, write operations aren't allowed on reader DB instances.. See the [User Guide for Aurora](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-write-forwarding.html) for more information. **NOTE:** Local write forwarding requires Aurora MySQL version 3.04 or higher.
     */
    @JvmName("tnltwuuqtdyamiws")
    public suspend fun enableLocalWriteForwarding(`value`: Output<Boolean>) {
        this.enableLocalWriteForwarding = value
    }

    /**
     * @param value Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("qjqpyemurrdaoqtp")
    public suspend fun enabledCloudwatchLogsExports(`value`: Output<List<String>>) {
        this.enabledCloudwatchLogsExports = value
    }

    @JvmName("ccsbthievyfwjnvt")
    public suspend fun enabledCloudwatchLogsExports(vararg values: Output<String>) {
        this.enabledCloudwatchLogsExports = Output.all(values.asList())
    }

    /**
     * @param values Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("uahilnxcayjtmlps")
    public suspend fun enabledCloudwatchLogsExports(values: List<Output<String>>) {
        this.enabledCloudwatchLogsExports = Output.all(values)
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("ibwpqjouignvmkdx")
    public suspend fun engine(`value`: Output<Either<String, EngineType>>) {
        this.engine = value
    }

    /**
     * @param value The life cycle type for this DB instance. This setting is valid for cluster types Aurora DB clusters and Multi-AZ DB clusters. Valid values are `open-source-rds-extended-support`, `open-source-rds-extended-support-disabled`. Default value is `open-source-rds-extended-support`. [Using Amazon RDS Extended Support]: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/extended-support.html
     */
    @JvmName("qonkvxjacvhflfku")
    public suspend fun engineLifecycleSupport(`value`: Output<String>) {
        this.engineLifecycleSupport = value
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("wwccocrivkwqmgcu")
    public suspend fun engineMode(`value`: Output<Either<String, EngineMode>>) {
        this.engineMode = value
    }

    /**
     * @param value Database engine version. Updating this argument results in an outage. See the [Aurora MySQL](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL&#46;Updates&#46;html) and [Aurora Postgres](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.html) documentation for your configured engine to determine this value, or by running `aws rds describe-db-engine-versions`. For example with Aurora MySQL 2, a potential value for this argument is `5.7.mysql_aurora.2.03.2`. The value can contain a partial version where supported by the API. The actual engine version used is returned in the attribute `engine_version_actual`, , see Attribute Reference below.
     */
    @JvmName("jclhkhcyjiijyeys")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value Name of your final DB snapshot when this DB cluster is deleted. If omitted, no final snapshot will be made.
     */
    @JvmName("qrfhvdrujskqppif")
    public suspend fun finalSnapshotIdentifier(`value`: Output<String>) {
        this.finalSnapshotIdentifier = value
    }

    /**
     * @param value Global cluster identifier specified on `aws.rds.GlobalCluster`.
     */
    @JvmName("tbbwpqcdwxfafoxe")
    public suspend fun globalClusterIdentifier(`value`: Output<String>) {
        this.globalClusterIdentifier = value
    }

    /**
     * @param value Specifies whether or not mappings of AWS Identity and Access Management (IAM) accounts to database accounts is enabled. Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html) for availability and limitations.
     */
    @JvmName("lhfkwegurivgsbft")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Output<Boolean>) {
        this.iamDatabaseAuthenticationEnabled = value
    }

    /**
     * @param value List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("uhnmmwuvlmblfmcf")
    public suspend fun iamRoles(`value`: Output<List<String>>) {
        this.iamRoles = value
    }

    @JvmName("nhoqgkxdllrgujif")
    public suspend fun iamRoles(vararg values: Output<String>) {
        this.iamRoles = Output.all(values.asList())
    }

    /**
     * @param values List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("nkeyxexeiemlkvxi")
    public suspend fun iamRoles(values: List<Output<String>>) {
        this.iamRoles = Output.all(values)
    }

    /**
     * @param value Amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid Iops values, see [Amazon RDS Provisioned IOPS storage to improve performance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS) in the Amazon RDS User Guide. (This setting is required to create a Multi-AZ DB cluster). Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
     */
    @JvmName("qxdmplsxuxwatdms")
    public suspend fun iops(`value`: Output<Int>) {
        this.iops = value
    }

    /**
     * @param value ARN for the KMS encryption key. When specifying `kms_key_id`, `storage_encrypted` needs to be set to true.
     */
    @JvmName("vdhcnwbbufreacub")
    public suspend fun kmsKeyId(`value`: Output<String>) {
        this.kmsKeyId = value
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `master_password` is provided.
     */
    @JvmName("utqewadlumudvocy")
    public suspend fun manageMasterUserPassword(`value`: Output<Boolean>) {
        this.manageMasterUserPassword = value
    }

    /**
     * @param value Password for the master DB user. Note that this may show up in logs, and it will be stored in the state file. Please refer to the [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("gmgveigqpjdboupg")
    public suspend fun masterPassword(`value`: Output<String>) {
        this.masterPassword = value
    }

    /**
     * @param value Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("hsfmjteierspndbc")
    public suspend fun masterUserSecretKmsKeyId(`value`: Output<String>) {
        this.masterUserSecretKmsKeyId = value
    }

    /**
     * @param value Username for the master DB user. Please refer to the [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). This argument does not support in-place updates and cannot be changed during a restore from snapshot.
     */
    @JvmName("blcvmykjvtiohacn")
    public suspend fun masterUsername(`value`: Output<String>) {
        this.masterUsername = value
    }

    /**
     * @param value Network type of the cluster. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("sfcrhyihvxpvmbxd")
    public suspend fun networkType(`value`: Output<String>) {
        this.networkType = value
    }

    /**
     * @param value Valid only for Non-Aurora Multi-AZ DB Clusters. Enables Performance Insights for the RDS Cluster
     */
    @JvmName("tosbgmlnjmaogepy")
    public suspend fun performanceInsightsEnabled(`value`: Output<Boolean>) {
        this.performanceInsightsEnabled = value
    }

    /**
     * @param value Valid only for Non-Aurora Multi-AZ DB Clusters. Specifies the KMS Key ID to encrypt Performance Insights data. If not specified, the default RDS KMS key will be used (`aws/rds`).
     */
    @JvmName("btslheojvbsjlhlo")
    public suspend fun performanceInsightsKmsKeyId(`value`: Output<String>) {
        this.performanceInsightsKmsKeyId = value
    }

    /**
     * @param value Valid only for Non-Aurora Multi-AZ DB Clusters. Specifies the amount of time to retain performance insights data for. Defaults to 7 days if Performance Insights are enabled. Valid values are `7`, `month * 31` (where month is a number of months from 1-23), and `731`. See [here](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.Overview.cost.html) for more information on retention periods.
     */
    @JvmName("atapylvtossmctkr")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Output<Int>) {
        this.performanceInsightsRetentionPeriod = value
    }

    /**
     * @param value Port on which the DB accepts connections.
     */
    @JvmName("tpledkfdvpotaveb")
    public suspend fun port(`value`: Output<Int>) {
        this.port = value
    }

    /**
     * @param value Daily time range during which automated backups are created if automated backups are enabled using the BackupRetentionPeriod parameter.Time in UTC. Default: A 30-minute window selected at random from an 8-hour block of time per region, e.g. `04:00-09:00`.
     */
    @JvmName("vpfqwhtddhcixbvy")
    public suspend fun preferredBackupWindow(`value`: Output<String>) {
        this.preferredBackupWindow = value
    }

    /**
     * @param value Weekly time range during which system maintenance can occur, in (UTC) e.g., `wed:04:00-wed:04:30`
     */
    @JvmName("ufiecsdrkvjfveok")
    public suspend fun preferredMaintenanceWindow(`value`: Output<String>) {
        this.preferredMaintenanceWindow = value
    }

    /**
     * @param value ARN of a source DB cluster or DB instance if this DB cluster is to be created as a Read Replica. If DB Cluster is part of a Global Cluster, use the `lifecycle` configuration block `ignore_changes` argument to prevent this provider from showing differences for this argument instead of configuring this value.
     */
    @JvmName("vaaprpsysmjgkqsd")
    public suspend fun replicationSourceIdentifier(`value`: Output<String>) {
        this.replicationSourceIdentifier = value
    }

    /**
     * @param value Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-pitr.html). More details below.
     */
    @JvmName("instyfnwfhehqkyh")
    public suspend fun restoreToPointInTime(`value`: Output<ClusterRestoreToPointInTimeArgs>) {
        this.restoreToPointInTime = value
    }

    /**
     * @param value
     */
    @JvmName("libtkvkysyplqovl")
    public suspend fun s3Import(`value`: Output<ClusterS3ImportArgs>) {
        this.s3Import = value
    }

    /**
     * @param value Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
     */
    @JvmName("qnhrrlhtqspibbqr")
    public suspend fun scalingConfiguration(`value`: Output<ClusterScalingConfigurationArgs>) {
        this.scalingConfiguration = value
    }

    /**
     * @param value Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
     */
    @JvmName("bfpkviufvwdukvtl")
    public suspend fun serverlessv2ScalingConfiguration(`value`: Output<ClusterServerlessv2ScalingConfigurationArgs>) {
        this.serverlessv2ScalingConfiguration = value
    }

    /**
     * @param value Determines whether a final DB snapshot is created before the DB cluster is deleted. If true is specified, no DB snapshot is created. If false is specified, a DB snapshot is created before the DB cluster is deleted, using the value from `final_snapshot_identifier`. Default is `false`.
     */
    @JvmName("prsxyjfjcdqrabul")
    public suspend fun skipFinalSnapshot(`value`: Output<Boolean>) {
        this.skipFinalSnapshot = value
    }

    /**
     * @param value Specifies whether or not to create this cluster from a snapshot. You can use either the name or ARN when specifying a DB cluster snapshot, or the ARN when specifying a DB snapshot. Conflicts with `global_cluster_identifier`. Clusters cannot be restored from snapshot **and** joined to an existing global cluster in a single operation. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-getting-started.html#aurora-global-database.use-snapshot) or the Global Cluster Restored From Snapshot example for instructions on building a global cluster starting with a snapshot.
     */
    @JvmName("pvgjicayvrfeajas")
    public suspend fun snapshotIdentifier(`value`: Output<String>) {
        this.snapshotIdentifier = value
    }

    /**
     * @param value The source region for an encrypted replica DB cluster.
     */
    @JvmName("dvyjhowycsnamtwd")
    public suspend fun sourceRegion(`value`: Output<String>) {
        this.sourceRegion = value
    }

    /**
     * @param value Specifies whether the DB cluster is encrypted. The default is `false` for `provisioned` `engine_mode` and `true` for `serverless` `engine_mode`. When restoring an unencrypted `snapshot_identifier`, the `kms_key_id` argument must be provided to encrypt the restored cluster. The provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("wqythhxtoksbohnp")
    public suspend fun storageEncrypted(`value`: Output<Boolean>) {
        this.storageEncrypted = value
    }

    /**
     * @param value (Forces new for Multi-AZ DB clusters) Specifies the storage type to be associated with the DB cluster. For Aurora DB clusters, `storage_type` modifications can be done in-place. For Multi-AZ DB Clusters, the `iops` argument must also be set. Valid values are: `""`, `aurora-iopt1` (Aurora DB Clusters); `io1`, `io2` (Multi-AZ DB Clusters). Default: `""` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters).
     */
    @JvmName("cjxiirktifctdxbl")
    public suspend fun storageType(`value`: Output<String>) {
        this.storageType = value
    }

    /**
     * @param value A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("rhtgpwhxpxjphajv")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value List of VPC security groups to associate with the Cluster
     */
    @JvmName("kjewruewrieirinc")
    public suspend fun vpcSecurityGroupIds(`value`: Output<List<String>>) {
        this.vpcSecurityGroupIds = value
    }

    @JvmName("hhagowygdphgcfcf")
    public suspend fun vpcSecurityGroupIds(vararg values: Output<String>) {
        this.vpcSecurityGroupIds = Output.all(values.asList())
    }

    /**
     * @param values List of VPC security groups to associate with the Cluster
     */
    @JvmName("waypnjktteuaccdg")
    public suspend fun vpcSecurityGroupIds(values: List<Output<String>>) {
        this.vpcSecurityGroupIds = Output.all(values)
    }

    /**
     * @param value The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.
     */
    @JvmName("fgvpylhvtarrhupu")
    public suspend fun allocatedStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocatedStorage = mapped
    }

    /**
     * @param value Enable to allow major engine version upgrades when changing engine versions. Defaults to `false`.
     */
    @JvmName("veqatgtydbqvrfxd")
    public suspend fun allowMajorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowMajorVersionUpgrade = mapped
    }

    /**
     * @param value Specifies whether any cluster modifications are applied immediately, or during the next maintenance window. Default is `false`. See [Amazon RDS Documentation for more information&#46;](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("ykihnyjobxykpcje")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("mpamckuqyviylprx")
    public suspend fun availabilityZones(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZones = mapped
    }

    /**
     * @param values List of EC2 Availability Zones for the DB cluster storage where DB cluster instances can be created.
     * RDS automatically assigns 3 AZs if less than 3 AZs are configured, which will show as a difference requiring resource recreation next pulumi up.
     * We recommend specifying 3 AZs or using the `lifecycle` configuration block `ignore_changes` argument if necessary.
     * A maximum of 3 AZs can be configured.
     */
    @JvmName("yxltopjgkkvyypmn")
    public suspend fun availabilityZones(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.availabilityZones = mapped
    }

    /**
     * @param value Target backtrack window, in seconds. Only available for `aurora` and `aurora-mysql` engines currently. To disable backtracking, set this value to `0`. Defaults to `0`. Must be between `0` and `259200` (72 hours)
     */
    @JvmName("ojpsefbjvidlwymb")
    public suspend fun backtrackWindow(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backtrackWindow = mapped
    }

    /**
     * @param value Days to retain backups for. Default `1`
     */
    @JvmName("xhidbefkuwtulooc")
    public suspend fun backupRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupRetentionPeriod = mapped
    }

    /**
     * @param value The CA certificate identifier to use for the DB cluster's server certificate.
     */
    @JvmName("aorcbmiiragubmky")
    public suspend fun caCertificateIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.caCertificateIdentifier = mapped
    }

    /**
     * @param value The cluster identifier. If omitted, this provider will assign a random, unique identifier.
     */
    @JvmName("hyusrbgrdhpkaofa")
    public suspend fun clusterIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterIdentifier = mapped
    }

    /**
     * @param value Creates a unique cluster identifier beginning with the specified prefix. Conflicts with `cluster_identifier`.
     */
    @JvmName("obwsklwtsoysddlw")
    public suspend fun clusterIdentifierPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterIdentifierPrefix = mapped
    }

    /**
     * @param value List of RDS Instances that are a part of this cluster
     */
    @JvmName("ffqnjcwpgmhcdlvl")
    public suspend fun clusterMembers(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.clusterMembers = mapped
    }

    /**
     * @param values List of RDS Instances that are a part of this cluster
     */
    @JvmName("ydvexrytpfceivwk")
    public suspend fun clusterMembers(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.clusterMembers = mapped
    }

    /**
     * @param value Copy all Cluster `tags` to snapshots. Default is `false`.
     */
    @JvmName("sneirdvdkuhjqmib")
    public suspend fun copyTagsToSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyTagsToSnapshot = mapped
    }

    /**
     * @param value Name for an automatically created database on cluster creation. There are different naming restrictions per database engine: [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints)
     */
    @JvmName("vktbqdgcbthwtobf")
    public suspend fun databaseName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseName = mapped
    }

    /**
     * @param value The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example `db.m6g.xlarge`. Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes and availability for your engine, see [DB instance class](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) in the Amazon RDS User Guide.
     */
    @JvmName("sulrbbgwamsgkiyn")
    public suspend fun dbClusterInstanceClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbClusterInstanceClass = mapped
    }

    /**
     * @param value A cluster parameter group to associate with the cluster.
     */
    @JvmName("iqnahslwwkfmgqpo")
    public suspend fun dbClusterParameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbClusterParameterGroupName = mapped
    }

    /**
     * @param value Instance parameter group to associate with all instances of the DB cluster. The `db_instance_parameter_group_name` parameter is only valid in combination with the `allow_major_version_upgrade` parameter.
     */
    @JvmName("uhapklrxbrqpleyu")
    public suspend fun dbInstanceParameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbInstanceParameterGroupName = mapped
    }

    /**
     * @param value DB subnet group to associate with this DB cluster.
     * **NOTE:** This must match the `db_subnet_group_name` specified on every `aws.rds.ClusterInstance` in the cluster.
     */
    @JvmName("naidesdnwtvqqbir")
    public suspend fun dbSubnetGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbSubnetGroupName = mapped
    }

    /**
     * @param value For use with RDS Custom.
     */
    @JvmName("mrjscewatyqnotpu")
    public suspend fun dbSystemId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbSystemId = mapped
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB cluster is deleted. Default is `true`.
     */
    @JvmName("qaevjcsvntbgvkps")
    public suspend fun deleteAutomatedBackups(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deleteAutomatedBackups = mapped
    }

    /**
     * @param value If the DB cluster should have deletion protection enabled.
     * The database can't be deleted when this value is set to `true`.
     * The default is `false`.
     */
    @JvmName("npjcbvkhttiydxcc")
    public suspend fun deletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtection = mapped
    }

    /**
     * @param value The ID of the Directory Service Active Directory domain to create the cluster in.
     */
    @JvmName("opoheucdlqastgta")
    public suspend fun domain(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domain = mapped
    }

    /**
     * @param value The name of the IAM role to be used when making API calls to the Directory Service.
     */
    @JvmName("cvktgsnnufosbhmm")
    public suspend fun domainIamRoleName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainIamRoleName = mapped
    }

    /**
     * @param value Whether cluster should forward writes to an associated global cluster. Applied to secondary clusters to enable them to forward writes to an `aws.rds.GlobalCluster`'s primary cluster. See the [User Guide for Aurora](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-write-forwarding.html) for more information.
     */
    @JvmName("whtximuryxatwawr")
    public suspend fun enableGlobalWriteForwarding(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableGlobalWriteForwarding = mapped
    }

    /**
     * @param value Enable HTTP endpoint (data API). Only valid for some combinations of `engine_mode`, `engine` and `engine_version` and only available in some regions. See the [Region and version availability](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html#data-api.regions) section of the documentation. This option also does not work with any of these options specified: `snapshot_identifier`, `replication_source_identifier`, `s3_import`.
     */
    @JvmName("helgtkkjyrnmybhl")
    public suspend fun enableHttpEndpoint(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableHttpEndpoint = mapped
    }

    /**
     * @param value Whether read replicas can forward write operations to the writer DB instance in the DB cluster. By default, write operations aren't allowed on reader DB instances.. See the [User Guide for Aurora](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-write-forwarding.html) for more information. **NOTE:** Local write forwarding requires Aurora MySQL version 3.04 or higher.
     */
    @JvmName("nbsshkikcxrrxkyx")
    public suspend fun enableLocalWriteForwarding(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enableLocalWriteForwarding = mapped
    }

    /**
     * @param value Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("fglxwlyxpooqqfhx")
    public suspend fun enabledCloudwatchLogsExports(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param values Set of log types to export to cloudwatch. If omitted, no logs will be exported. The following log types are supported: `audit`, `error`, `general`, `slowquery`, `postgresql` (PostgreSQL).
     */
    @JvmName("gwtetpflwocgfccw")
    public suspend fun enabledCloudwatchLogsExports(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("splrscyojeeuwavk")
    public suspend fun engine(`value`: Either<String, EngineType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("giqbnvbnufjgidaj")
    public fun engine(`value`: String) {
        val toBeMapped = Either.ofLeft<String, EngineType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value Name of the database engine to be used for this DB cluster. Valid Values: `aurora-mysql`, `aurora-postgresql`, `mysql`, `postgres`. (Note that `mysql` and `postgres` are Multi-AZ RDS clusters).
     */
    @JvmName("afchpffivwxcdnyd")
    public fun engine(`value`: EngineType) {
        val toBeMapped = Either.ofRight<String, EngineType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value The life cycle type for this DB instance. This setting is valid for cluster types Aurora DB clusters and Multi-AZ DB clusters. Valid values are `open-source-rds-extended-support`, `open-source-rds-extended-support-disabled`. Default value is `open-source-rds-extended-support`. [Using Amazon RDS Extended Support]: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/extended-support.html
     */
    @JvmName("ryyikkmadbguurmw")
    public suspend fun engineLifecycleSupport(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineLifecycleSupport = mapped
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("fivkxaapxqutdvqm")
    public suspend fun engineMode(`value`: Either<String, EngineMode>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineMode = mapped
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("gylmbvgippxfcxhm")
    public fun engineMode(`value`: String) {
        val toBeMapped = Either.ofLeft<String, EngineMode>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engineMode = mapped
    }

    /**
     * @param value Database engine mode. Valid values: `global` (only valid for Aurora MySQL 1.21 and earlier), `parallelquery`, `provisioned`, `serverless`. Defaults to: `provisioned`. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html) for limitations when using `serverless`.
     */
    @JvmName("ctnpgulxkwqxmdhf")
    public fun engineMode(`value`: EngineMode) {
        val toBeMapped = Either.ofRight<String, EngineMode>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.engineMode = mapped
    }

    /**
     * @param value Database engine version. Updating this argument results in an outage. See the [Aurora MySQL](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL&#46;Updates&#46;html) and [Aurora Postgres](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.html) documentation for your configured engine to determine this value, or by running `aws rds describe-db-engine-versions`. For example with Aurora MySQL 2, a potential value for this argument is `5.7.mysql_aurora.2.03.2`. The value can contain a partial version where supported by the API. The actual engine version used is returned in the attribute `engine_version_actual`, , see Attribute Reference below.
     */
    @JvmName("yysybgxtrdrulicl")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value Name of your final DB snapshot when this DB cluster is deleted. If omitted, no final snapshot will be made.
     */
    @JvmName("vcajykjpnjvxgyhr")
    public suspend fun finalSnapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.finalSnapshotIdentifier = mapped
    }

    /**
     * @param value Global cluster identifier specified on `aws.rds.GlobalCluster`.
     */
    @JvmName("wnvvgwbrcyyqeosw")
    public suspend fun globalClusterIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.globalClusterIdentifier = mapped
    }

    /**
     * @param value Specifies whether or not mappings of AWS Identity and Access Management (IAM) accounts to database accounts is enabled. Please see [AWS Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html) for availability and limitations.
     */
    @JvmName("vuplexdcelrrbgkf")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamDatabaseAuthenticationEnabled = mapped
    }

    /**
     * @param value List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("kwmlhvbsuegsdcqc")
    public suspend fun iamRoles(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamRoles = mapped
    }

    /**
     * @param values List of ARNs for the IAM roles to associate to the RDS Cluster.
     */
    @JvmName("ugrsdcjpaidityqj")
    public suspend fun iamRoles(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.iamRoles = mapped
    }

    /**
     * @param value Amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid Iops values, see [Amazon RDS Provisioned IOPS storage to improve performance](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS) in the Amazon RDS User Guide. (This setting is required to create a Multi-AZ DB cluster). Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
     */
    @JvmName("lulgdsdryyxtigxm")
    public suspend fun iops(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iops = mapped
    }

    /**
     * @param value ARN for the KMS encryption key. When specifying `kms_key_id`, `storage_encrypted` needs to be set to true.
     */
    @JvmName("ipuauyrvflgnmuom")
    public suspend fun kmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKeyId = mapped
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `master_password` is provided.
     */
    @JvmName("rosjciiqyjwwjnbb")
    public suspend fun manageMasterUserPassword(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.manageMasterUserPassword = mapped
    }

    /**
     * @param value Password for the master DB user. Note that this may show up in logs, and it will be stored in the state file. Please refer to the [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("ssbddwyinwvdriod")
    public suspend fun masterPassword(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterPassword = mapped
    }

    /**
     * @param value Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("woagqpdewsesqlcs")
    public suspend fun masterUserSecretKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterUserSecretKmsKeyId = mapped
    }

    /**
     * @param value Username for the master DB user. Please refer to the [RDS Naming Constraints](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints). This argument does not support in-place updates and cannot be changed during a restore from snapshot.
     */
    @JvmName("yrdwqumbvbleqnnd")
    public suspend fun masterUsername(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterUsername = mapped
    }

    /**
     * @param value Network type of the cluster. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("ssyobglnhpwqgohu")
    public suspend fun networkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkType = mapped
    }

    /**
     * @param value Valid only for Non-Aurora Multi-AZ DB Clusters. Enables Performance Insights for the RDS Cluster
     */
    @JvmName("kknxiqamqlwdkmfu")
    public suspend fun performanceInsightsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsEnabled = mapped
    }

    /**
     * @param value Valid only for Non-Aurora Multi-AZ DB Clusters. Specifies the KMS Key ID to encrypt Performance Insights data. If not specified, the default RDS KMS key will be used (`aws/rds`).
     */
    @JvmName("nnmqlwifubnimvpg")
    public suspend fun performanceInsightsKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsKmsKeyId = mapped
    }

    /**
     * @param value Valid only for Non-Aurora Multi-AZ DB Clusters. Specifies the amount of time to retain performance insights data for. Defaults to 7 days if Performance Insights are enabled. Valid values are `7`, `month * 31` (where month is a number of months from 1-23), and `731`. See [here](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.Overview.cost.html) for more information on retention periods.
     */
    @JvmName("jekejkiggqivdkdy")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsRetentionPeriod = mapped
    }

    /**
     * @param value Port on which the DB accepts connections.
     */
    @JvmName("ejctrkiwgimuquwc")
    public suspend fun port(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value Daily time range during which automated backups are created if automated backups are enabled using the BackupRetentionPeriod parameter.Time in UTC. Default: A 30-minute window selected at random from an 8-hour block of time per region, e.g. `04:00-09:00`.
     */
    @JvmName("ogbqvyvjcyttlvif")
    public suspend fun preferredBackupWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredBackupWindow = mapped
    }

    /**
     * @param value Weekly time range during which system maintenance can occur, in (UTC) e.g., `wed:04:00-wed:04:30`
     */
    @JvmName("tsxqqhlwokqxpxee")
    public suspend fun preferredMaintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preferredMaintenanceWindow = mapped
    }

    /**
     * @param value ARN of a source DB cluster or DB instance if this DB cluster is to be created as a Read Replica. If DB Cluster is part of a Global Cluster, use the `lifecycle` configuration block `ignore_changes` argument to prevent this provider from showing differences for this argument instead of configuring this value.
     */
    @JvmName("evfkpqokobqhveuw")
    public suspend fun replicationSourceIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicationSourceIdentifier = mapped
    }

    /**
     * @param value Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-pitr.html). More details below.
     */
    @JvmName("ykstvtysygjngbto")
    public suspend fun restoreToPointInTime(`value`: ClusterRestoreToPointInTimeArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restoreToPointInTime = mapped
    }

    /**
     * @param argument Nested attribute for [point in time restore](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-pitr.html). More details below.
     */
    @JvmName("kmpombxmoucalxpr")
    public suspend fun restoreToPointInTime(argument: suspend ClusterRestoreToPointInTimeArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterRestoreToPointInTimeArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.restoreToPointInTime = mapped
    }

    /**
     * @param value
     */
    @JvmName("yvrafgqwjhgtfofp")
    public suspend fun s3Import(`value`: ClusterS3ImportArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.s3Import = mapped
    }

    /**
     * @param argument
     */
    @JvmName("eojjrxqosejlerst")
    public suspend fun s3Import(argument: suspend ClusterS3ImportArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterS3ImportArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.s3Import = mapped
    }

    /**
     * @param value Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
     */
    @JvmName("oqyrdbnujgdrlubo")
    public suspend fun scalingConfiguration(`value`: ClusterScalingConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.scalingConfiguration = mapped
    }

    /**
     * @param argument Nested attribute with scaling properties. Only valid when `engine_mode` is set to `serverless`. More details below.
     */
    @JvmName("eeyrxnhwibnypgng")
    public suspend fun scalingConfiguration(argument: suspend ClusterScalingConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterScalingConfigurationArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.scalingConfiguration = mapped
    }

    /**
     * @param value Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
     */
    @JvmName("vjaivusrnyfxbhen")
    public suspend fun serverlessv2ScalingConfiguration(`value`: ClusterServerlessv2ScalingConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverlessv2ScalingConfiguration = mapped
    }

    /**
     * @param argument Nested attribute with scaling properties for ServerlessV2. Only valid when `engine_mode` is set to `provisioned`. More details below.
     */
    @JvmName("atfpgrepxnrvorek")
    public suspend fun serverlessv2ScalingConfiguration(argument: suspend ClusterServerlessv2ScalingConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = ClusterServerlessv2ScalingConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.serverlessv2ScalingConfiguration = mapped
    }

    /**
     * @param value Determines whether a final DB snapshot is created before the DB cluster is deleted. If true is specified, no DB snapshot is created. If false is specified, a DB snapshot is created before the DB cluster is deleted, using the value from `final_snapshot_identifier`. Default is `false`.
     */
    @JvmName("mpojuxsrijexcyyd")
    public suspend fun skipFinalSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.skipFinalSnapshot = mapped
    }

    /**
     * @param value Specifies whether or not to create this cluster from a snapshot. You can use either the name or ARN when specifying a DB cluster snapshot, or the ARN when specifying a DB snapshot. Conflicts with `global_cluster_identifier`. Clusters cannot be restored from snapshot **and** joined to an existing global cluster in a single operation. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database-getting-started.html#aurora-global-database.use-snapshot) or the Global Cluster Restored From Snapshot example for instructions on building a global cluster starting with a snapshot.
     */
    @JvmName("tbswnsothafowdkl")
    public suspend fun snapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotIdentifier = mapped
    }

    /**
     * @param value The source region for an encrypted replica DB cluster.
     */
    @JvmName("ekfjlwwuaxntaxuo")
    public suspend fun sourceRegion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceRegion = mapped
    }

    /**
     * @param value Specifies whether the DB cluster is encrypted. The default is `false` for `provisioned` `engine_mode` and `true` for `serverless` `engine_mode`. When restoring an unencrypted `snapshot_identifier`, the `kms_key_id` argument must be provided to encrypt the restored cluster. The provider will only perform drift detection if a configuration value is provided.
     */
    @JvmName("habxgvixpglpeehc")
    public suspend fun storageEncrypted(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageEncrypted = mapped
    }

    /**
     * @param value (Forces new for Multi-AZ DB clusters) Specifies the storage type to be associated with the DB cluster. For Aurora DB clusters, `storage_type` modifications can be done in-place. For Multi-AZ DB Clusters, the `iops` argument must also be set. Valid values are: `""`, `aurora-iopt1` (Aurora DB Clusters); `io1`, `io2` (Multi-AZ DB Clusters). Default: `""` (Aurora DB Clusters); `io1` (Multi-AZ DB Clusters).
     */
    @JvmName("mcmevdjmotjwlefi")
    public suspend fun storageType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("pheqvpelteovvmjj")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the DB cluster. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("rmlxooxklmynyhpd")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value List of VPC security groups to associate with the Cluster
     */
    @JvmName("iggtsaaqacsrwjlg")
    public suspend fun vpcSecurityGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    /**
     * @param values List of VPC security groups to associate with the Cluster
     */
    @JvmName("teoscxweqnvjmvej")
    public suspend fun vpcSecurityGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    internal fun build(): ClusterArgs = ClusterArgs(
        allocatedStorage = allocatedStorage,
        allowMajorVersionUpgrade = allowMajorVersionUpgrade,
        applyImmediately = applyImmediately,
        availabilityZones = availabilityZones,
        backtrackWindow = backtrackWindow,
        backupRetentionPeriod = backupRetentionPeriod,
        caCertificateIdentifier = caCertificateIdentifier,
        clusterIdentifier = clusterIdentifier,
        clusterIdentifierPrefix = clusterIdentifierPrefix,
        clusterMembers = clusterMembers,
        copyTagsToSnapshot = copyTagsToSnapshot,
        databaseName = databaseName,
        dbClusterInstanceClass = dbClusterInstanceClass,
        dbClusterParameterGroupName = dbClusterParameterGroupName,
        dbInstanceParameterGroupName = dbInstanceParameterGroupName,
        dbSubnetGroupName = dbSubnetGroupName,
        dbSystemId = dbSystemId,
        deleteAutomatedBackups = deleteAutomatedBackups,
        deletionProtection = deletionProtection,
        domain = domain,
        domainIamRoleName = domainIamRoleName,
        enableGlobalWriteForwarding = enableGlobalWriteForwarding,
        enableHttpEndpoint = enableHttpEndpoint,
        enableLocalWriteForwarding = enableLocalWriteForwarding,
        enabledCloudwatchLogsExports = enabledCloudwatchLogsExports,
        engine = engine,
        engineLifecycleSupport = engineLifecycleSupport,
        engineMode = engineMode,
        engineVersion = engineVersion,
        finalSnapshotIdentifier = finalSnapshotIdentifier,
        globalClusterIdentifier = globalClusterIdentifier,
        iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled,
        iamRoles = iamRoles,
        iops = iops,
        kmsKeyId = kmsKeyId,
        manageMasterUserPassword = manageMasterUserPassword,
        masterPassword = masterPassword,
        masterUserSecretKmsKeyId = masterUserSecretKmsKeyId,
        masterUsername = masterUsername,
        networkType = networkType,
        performanceInsightsEnabled = performanceInsightsEnabled,
        performanceInsightsKmsKeyId = performanceInsightsKmsKeyId,
        performanceInsightsRetentionPeriod = performanceInsightsRetentionPeriod,
        port = port,
        preferredBackupWindow = preferredBackupWindow,
        preferredMaintenanceWindow = preferredMaintenanceWindow,
        replicationSourceIdentifier = replicationSourceIdentifier,
        restoreToPointInTime = restoreToPointInTime,
        s3Import = s3Import,
        scalingConfiguration = scalingConfiguration,
        serverlessv2ScalingConfiguration = serverlessv2ScalingConfiguration,
        skipFinalSnapshot = skipFinalSnapshot,
        snapshotIdentifier = snapshotIdentifier,
        sourceRegion = sourceRegion,
        storageEncrypted = storageEncrypted,
        storageType = storageType,
        tags = tags,
        vpcSecurityGroupIds = vpcSecurityGroupIds,
    )
}
