@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.rds.kotlin

import com.pulumi.aws.rds.CustomDbEngineVersionArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an custom engine version (CEV) resource for Amazon RDS Custom. For additional information, see [Working with CEVs for RDS Custom for Oracle](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/UserGuide/custom-cev&#46;html) and [Working with CEVs for RDS Custom for SQL Server](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/UserGuide/custom-cev-sqlserver&#46;html) in the the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Welcome.html).
 * ## Example Usage
 * ### RDS Custom for Oracle Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kms.Key("example", {description: "KMS symmetric key for RDS Custom for Oracle"});
 * const exampleCustomDbEngineVersion = new aws.rds.CustomDbEngineVersion("example", {
 *     databaseInstallationFilesS3BucketName: "DOC-EXAMPLE-BUCKET",
 *     databaseInstallationFilesS3Prefix: "1915_GI/",
 *     engine: "custom-oracle-ee-cdb",
 *     engineVersion: "19.cdb_cev1",
 *     kmsKeyId: example.arn,
 *     manifest: `  {
 * \x09"databaseInstallationFileNames":["V982063-01&#46;zip"]
 *   }
 * `,
 *     tags: {
 *         Name: "example",
 *         Key: "value",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kms.Key("example", description="KMS symmetric key for RDS Custom for Oracle")
 * example_custom_db_engine_version = aws.rds.CustomDbEngineVersion("example",
 *     database_installation_files_s3_bucket_name="DOC-EXAMPLE-BUCKET",
 *     database_installation_files_s3_prefix="1915_GI/",
 *     engine="custom-oracle-ee-cdb",
 *     engine_version="19.cdb_cev1",
 *     kms_key_id=example.arn,
 *     manifest="""  {
 * \x09"databaseInstallationFileNames":["V982063-01&#46;zip"]
 *   }
 * """,
 *     tags={
 *         "Name": "example",
 *         "Key": "value",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kms.Key("example", new()
 *     {
 *         Description = "KMS symmetric key for RDS Custom for Oracle",
 *     });
 *     var exampleCustomDbEngineVersion = new Aws.Rds.CustomDbEngineVersion("example", new()
 *     {
 *         DatabaseInstallationFilesS3BucketName = "DOC-EXAMPLE-BUCKET",
 *         DatabaseInstallationFilesS3Prefix = "1915_GI/",
 *         Engine = "custom-oracle-ee-cdb",
 *         EngineVersion = "19.cdb_cev1",
 *         KmsKeyId = example.Arn,
 *         Manifest = @"  {
 * 	""databaseInstallationFileNames"":[""V982063-01&#46;zip""]
 *   }
 * ",
 *         Tags =
 *         {
 *             { "Name", "example" },
 *             { "Key", "value" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := kms.NewKey(ctx, "example", &kms.KeyArgs{
 * 			Description: pulumi.String("KMS symmetric key for RDS Custom for Oracle"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewCustomDbEngineVersion(ctx, "example", &rds.CustomDbEngineVersionArgs{
 * 			DatabaseInstallationFilesS3BucketName: pulumi.String("DOC-EXAMPLE-BUCKET"),
 * 			DatabaseInstallationFilesS3Prefix:     pulumi.String("1915_GI/"),
 * 			Engine:                                pulumi.String("custom-oracle-ee-cdb"),
 * 			EngineVersion:                         pulumi.String("19.cdb_cev1"),
 * 			KmsKeyId:                              example.Arn,
 * 			Manifest:                              pulumi.String("  {\n	\"databaseInstallationFileNames\":[\"V982063-01&#46;zip\"]\n  }\n"),
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("example"),
 * 				"Key":  pulumi.String("value"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.rds.CustomDbEngineVersion;
 * import com.pulumi.aws.rds.CustomDbEngineVersionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Key("example", KeyArgs.builder()
 *             .description("KMS symmetric key for RDS Custom for Oracle")
 *             .build());
 *         var exampleCustomDbEngineVersion = new CustomDbEngineVersion("exampleCustomDbEngineVersion", CustomDbEngineVersionArgs.builder()
 *             .databaseInstallationFilesS3BucketName("DOC-EXAMPLE-BUCKET")
 *             .databaseInstallationFilesS3Prefix("1915_GI/")
 *             .engine("custom-oracle-ee-cdb")
 *             .engineVersion("19.cdb_cev1")
 *             .kmsKeyId(example.arn())
 *             .manifest("""
 *   {
 * 	"databaseInstallationFileNames":["V982063-01&#46;zip"]
 *   }
 *             """)
 *             .tags(Map.ofEntries(
 *                 Map.entry("Name", "example"),
 *                 Map.entry("Key", "value")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kms:Key
 *     properties:
 *       description: KMS symmetric key for RDS Custom for Oracle
 *   exampleCustomDbEngineVersion:
 *     type: aws:rds:CustomDbEngineVersion
 *     name: example
 *     properties:
 *       databaseInstallationFilesS3BucketName: DOC-EXAMPLE-BUCKET
 *       databaseInstallationFilesS3Prefix: 1915_GI/
 *       engine: custom-oracle-ee-cdb
 *       engineVersion: 19.cdb_cev1
 *       kmsKeyId: ${example.arn}
 *       manifest: |2
 *           {
 *         	"databaseInstallationFileNames":["V982063-01&#46;zip"]
 *           }
 *       tags:
 *         Name: example
 *         Key: value
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Custom for Oracle External Manifest Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * const example = new aws.kms.Key("example", {description: "KMS symmetric key for RDS Custom for Oracle"});
 * const exampleCustomDbEngineVersion = new aws.rds.CustomDbEngineVersion("example", {
 *     databaseInstallationFilesS3BucketName: "DOC-EXAMPLE-BUCKET",
 *     databaseInstallationFilesS3Prefix: "1915_GI/",
 *     engine: "custom-oracle-ee-cdb",
 *     engineVersion: "19.cdb_cev1",
 *     kmsKeyId: example.arn,
 *     filename: "manifest_1915_GI.json",
 *     manifestHash: std.filebase64sha256({
 *         input: json,
 *     }).then(invoke => invoke.result),
 *     tags: {
 *         Name: "example",
 *         Key: "value",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * example = aws.kms.Key("example", description="KMS symmetric key for RDS Custom for Oracle")
 * example_custom_db_engine_version = aws.rds.CustomDbEngineVersion("example",
 *     database_installation_files_s3_bucket_name="DOC-EXAMPLE-BUCKET",
 *     database_installation_files_s3_prefix="1915_GI/",
 *     engine="custom-oracle-ee-cdb",
 *     engine_version="19.cdb_cev1",
 *     kms_key_id=example.arn,
 *     filename="manifest_1915_GI.json",
 *     manifest_hash=std.filebase64sha256(input=json).result,
 *     tags={
 *         "Name": "example",
 *         "Key": "value",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kms.Key("example", new()
 *     {
 *         Description = "KMS symmetric key for RDS Custom for Oracle",
 *     });
 *     var exampleCustomDbEngineVersion = new Aws.Rds.CustomDbEngineVersion("example", new()
 *     {
 *         DatabaseInstallationFilesS3BucketName = "DOC-EXAMPLE-BUCKET",
 *         DatabaseInstallationFilesS3Prefix = "1915_GI/",
 *         Engine = "custom-oracle-ee-cdb",
 *         EngineVersion = "19.cdb_cev1",
 *         KmsKeyId = example.Arn,
 *         Filename = "manifest_1915_GI.json",
 *         ManifestHash = Std.Filebase64sha256.Invoke(new()
 *         {
 *             Input = json,
 *         }).Apply(invoke => invoke.Result),
 *         Tags =
 *         {
 *             { "Name", "example" },
 *             { "Key", "value" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := kms.NewKey(ctx, "example", &kms.KeyArgs{
 * 			Description: pulumi.String("KMS symmetric key for RDS Custom for Oracle"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		invokeFilebase64sha256, err := std.Filebase64sha256(ctx, &std.Filebase64sha256Args{
 * 			Input: json,
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewCustomDbEngineVersion(ctx, "example", &rds.CustomDbEngineVersionArgs{
 * 			DatabaseInstallationFilesS3BucketName: pulumi.String("DOC-EXAMPLE-BUCKET"),
 * 			DatabaseInstallationFilesS3Prefix:     pulumi.String("1915_GI/"),
 * 			Engine:                                pulumi.String("custom-oracle-ee-cdb"),
 * 			EngineVersion:                         pulumi.String("19.cdb_cev1"),
 * 			KmsKeyId:                              example.Arn,
 * 			Filename:                              pulumi.String("manifest_1915_GI.json"),
 * 			ManifestHash:                          pulumi.String(invokeFilebase64sha256.Result),
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("example"),
 * 				"Key":  pulumi.String("value"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.rds.CustomDbEngineVersion;
 * import com.pulumi.aws.rds.CustomDbEngineVersionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Key("example", KeyArgs.builder()
 *             .description("KMS symmetric key for RDS Custom for Oracle")
 *             .build());
 *         var exampleCustomDbEngineVersion = new CustomDbEngineVersion("exampleCustomDbEngineVersion", CustomDbEngineVersionArgs.builder()
 *             .databaseInstallationFilesS3BucketName("DOC-EXAMPLE-BUCKET")
 *             .databaseInstallationFilesS3Prefix("1915_GI/")
 *             .engine("custom-oracle-ee-cdb")
 *             .engineVersion("19.cdb_cev1")
 *             .kmsKeyId(example.arn())
 *             .filename("manifest_1915_GI.json")
 *             .manifestHash(StdFunctions.filebase64sha256(Filebase64sha256Args.builder()
 *                 .input(json)
 *                 .build()).result())
 *             .tags(Map.ofEntries(
 *                 Map.entry("Name", "example"),
 *                 Map.entry("Key", "value")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kms:Key
 *     properties:
 *       description: KMS symmetric key for RDS Custom for Oracle
 *   exampleCustomDbEngineVersion:
 *     type: aws:rds:CustomDbEngineVersion
 *     name: example
 *     properties:
 *       databaseInstallationFilesS3BucketName: DOC-EXAMPLE-BUCKET
 *       databaseInstallationFilesS3Prefix: 1915_GI/
 *       engine: custom-oracle-ee-cdb
 *       engineVersion: 19.cdb_cev1
 *       kmsKeyId: ${example.arn}
 *       filename: manifest_1915_GI.json
 *       manifestHash:
 *         fn::invoke:
 *           Function: std:filebase64sha256
 *           Arguments:
 *             input: ${json}
 *           Return: result
 *       tags:
 *         Name: example
 *         Key: value
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Custom for SQL Server Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * // CEV creation requires an AMI owned by the operator
 * const test = new aws.rds.CustomDbEngineVersion("test", {
 *     engine: "custom-sqlserver-se",
 *     engineVersion: "15.00.4249.2.cev-1",
 *     sourceImageId: "ami-0aa12345678a12ab1",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * # CEV creation requires an AMI owned by the operator
 * test = aws.rds.CustomDbEngineVersion("test",
 *     engine="custom-sqlserver-se",
 *     engine_version="15.00.4249.2.cev-1",
 *     source_image_id="ami-0aa12345678a12ab1")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     // CEV creation requires an AMI owned by the operator
 *     var test = new Aws.Rds.CustomDbEngineVersion("test", new()
 *     {
 *         Engine = "custom-sqlserver-se",
 *         EngineVersion = "15.00.4249.2.cev-1",
 *         SourceImageId = "ami-0aa12345678a12ab1",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// CEV creation requires an AMI owned by the operator
 * 		_, err := rds.NewCustomDbEngineVersion(ctx, "test", &rds.CustomDbEngineVersionArgs{
 * 			Engine:        pulumi.String("custom-sqlserver-se"),
 * 			EngineVersion: pulumi.String("15.00.4249.2.cev-1"),
 * 			SourceImageId: pulumi.String("ami-0aa12345678a12ab1"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.CustomDbEngineVersion;
 * import com.pulumi.aws.rds.CustomDbEngineVersionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // CEV creation requires an AMI owned by the operator
 *         var test = new CustomDbEngineVersion("test", CustomDbEngineVersionArgs.builder()
 *             .engine("custom-sqlserver-se")
 *             .engineVersion("15.00.4249.2.cev-1")
 *             .sourceImageId("ami-0aa12345678a12ab1")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # CEV creation requires an AMI owned by the operator
 *   test:
 *     type: aws:rds:CustomDbEngineVersion
 *     properties:
 *       engine: custom-sqlserver-se
 *       engineVersion: 15.00.4249.2.cev-1
 *       sourceImageId: ami-0aa12345678a12ab1
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Custom for SQL Server Usage with AMI from another region
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.ec2.AmiCopy("example", {
 *     name: "sqlserver-se-2019-15.00.4249.2",
 *     description: "A copy of ami-xxxxxxxx",
 *     sourceAmiId: "ami-xxxxxxxx",
 *     sourceAmiRegion: "us-east-1",
 * });
 * // CEV creation requires an AMI owned by the operator
 * const test = new aws.rds.CustomDbEngineVersion("test", {
 *     engine: "custom-sqlserver-se",
 *     engineVersion: "15.00.4249.2.cev-1",
 *     sourceImageId: example.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ec2.AmiCopy("example",
 *     name="sqlserver-se-2019-15.00.4249.2",
 *     description="A copy of ami-xxxxxxxx",
 *     source_ami_id="ami-xxxxxxxx",
 *     source_ami_region="us-east-1")
 * # CEV creation requires an AMI owned by the operator
 * test = aws.rds.CustomDbEngineVersion("test",
 *     engine="custom-sqlserver-se",
 *     engine_version="15.00.4249.2.cev-1",
 *     source_image_id=example.id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Ec2.AmiCopy("example", new()
 *     {
 *         Name = "sqlserver-se-2019-15.00.4249.2",
 *         Description = "A copy of ami-xxxxxxxx",
 *         SourceAmiId = "ami-xxxxxxxx",
 *         SourceAmiRegion = "us-east-1",
 *     });
 *     // CEV creation requires an AMI owned by the operator
 *     var test = new Aws.Rds.CustomDbEngineVersion("test", new()
 *     {
 *         Engine = "custom-sqlserver-se",
 *         EngineVersion = "15.00.4249.2.cev-1",
 *         SourceImageId = example.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := ec2.NewAmiCopy(ctx, "example", &ec2.AmiCopyArgs{
 * 			Name:            pulumi.String("sqlserver-se-2019-15.00.4249.2"),
 * 			Description:     pulumi.String("A copy of ami-xxxxxxxx"),
 * 			SourceAmiId:     pulumi.String("ami-xxxxxxxx"),
 * 			SourceAmiRegion: pulumi.String("us-east-1"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// CEV creation requires an AMI owned by the operator
 * 		_, err = rds.NewCustomDbEngineVersion(ctx, "test", &rds.CustomDbEngineVersionArgs{
 * 			Engine:        pulumi.String("custom-sqlserver-se"),
 * 			EngineVersion: pulumi.String("15.00.4249.2.cev-1"),
 * 			SourceImageId: example.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ec2.AmiCopy;
 * import com.pulumi.aws.ec2.AmiCopyArgs;
 * import com.pulumi.aws.rds.CustomDbEngineVersion;
 * import com.pulumi.aws.rds.CustomDbEngineVersionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new AmiCopy("example", AmiCopyArgs.builder()
 *             .name("sqlserver-se-2019-15.00.4249.2")
 *             .description("A copy of ami-xxxxxxxx")
 *             .sourceAmiId("ami-xxxxxxxx")
 *             .sourceAmiRegion("us-east-1")
 *             .build());
 *         // CEV creation requires an AMI owned by the operator
 *         var test = new CustomDbEngineVersion("test", CustomDbEngineVersionArgs.builder()
 *             .engine("custom-sqlserver-se")
 *             .engineVersion("15.00.4249.2.cev-1")
 *             .sourceImageId(example.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:AmiCopy
 *     properties:
 *       name: sqlserver-se-2019-15.00.4249.2
 *       description: A copy of ami-xxxxxxxx
 *       sourceAmiId: ami-xxxxxxxx
 *       sourceAmiRegion: us-east-1
 *   # CEV creation requires an AMI owned by the operator
 *   test:
 *     type: aws:rds:CustomDbEngineVersion
 *     properties:
 *       engine: custom-sqlserver-se
 *       engineVersion: 15.00.4249.2.cev-1
 *       sourceImageId: ${example.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import custom engine versions for Amazon RDS custom using the `engine` and `engine_version` separated by a colon (`:`). For example:
 * ```sh
 * $ pulumi import aws:rds/customDbEngineVersion:CustomDbEngineVersion example custom-oracle-ee-cdb:19.cdb_cev1
 * ```
 * @property databaseInstallationFilesS3BucketName The name of the Amazon S3 bucket that contains the database installation files.
 * @property databaseInstallationFilesS3Prefix The prefix for the Amazon S3 bucket that contains the database installation files.
 * @property description The description of the CEV.
 * @property engine The name of the database engine. Valid values are `custom-oracle*`, `custom-sqlserver*`.
 * @property engineVersion The version of the database engine.
 * @property filename The name of the manifest file within the local filesystem. Conflicts with `manifest`.
 * @property kmsKeyId The ARN of the AWS KMS key that is used to encrypt the database installation files. Required for RDS Custom for Oracle.
 * @property manifest The manifest file, in JSON format, that contains the list of database installation files. Conflicts with `filename`.
 * @property manifestHash Used to trigger updates. Must be set to a base64-encoded SHA256 hash of the manifest source specified with `filename`. The usual way to set this is filebase64sha256("manifest.json") where "manifest.json" is the local filename of the manifest source.
 * @property sourceImageId The ID of the AMI to create the CEV from. Required for RDS Custom for SQL Server. For RDS Custom for Oracle, you can specify an AMI ID that was used in a different Oracle CEV.
 * @property status The status of the CEV. Valid values are `available`, `inactive`, `inactive-except-restore`.
 * @property tags A mapping of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class CustomDbEngineVersionArgs(
    public val databaseInstallationFilesS3BucketName: Output<String>? = null,
    public val databaseInstallationFilesS3Prefix: Output<String>? = null,
    public val description: Output<String>? = null,
    public val engine: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val filename: Output<String>? = null,
    public val kmsKeyId: Output<String>? = null,
    public val manifest: Output<String>? = null,
    public val manifestHash: Output<String>? = null,
    public val sourceImageId: Output<String>? = null,
    public val status: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.rds.CustomDbEngineVersionArgs> {
    override fun toJava(): com.pulumi.aws.rds.CustomDbEngineVersionArgs =
        com.pulumi.aws.rds.CustomDbEngineVersionArgs.builder()
            .databaseInstallationFilesS3BucketName(
                databaseInstallationFilesS3BucketName?.applyValue({ args0 ->
                    args0
                }),
            )
            .databaseInstallationFilesS3Prefix(
                databaseInstallationFilesS3Prefix?.applyValue({ args0 ->
                    args0
                }),
            )
            .description(description?.applyValue({ args0 -> args0 }))
            .engine(engine?.applyValue({ args0 -> args0 }))
            .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
            .filename(filename?.applyValue({ args0 -> args0 }))
            .kmsKeyId(kmsKeyId?.applyValue({ args0 -> args0 }))
            .manifest(manifest?.applyValue({ args0 -> args0 }))
            .manifestHash(manifestHash?.applyValue({ args0 -> args0 }))
            .sourceImageId(sourceImageId?.applyValue({ args0 -> args0 }))
            .status(status?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [CustomDbEngineVersionArgs].
 */
@PulumiTagMarker
public class CustomDbEngineVersionArgsBuilder internal constructor() {
    private var databaseInstallationFilesS3BucketName: Output<String>? = null

    private var databaseInstallationFilesS3Prefix: Output<String>? = null

    private var description: Output<String>? = null

    private var engine: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var filename: Output<String>? = null

    private var kmsKeyId: Output<String>? = null

    private var manifest: Output<String>? = null

    private var manifestHash: Output<String>? = null

    private var sourceImageId: Output<String>? = null

    private var status: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value The name of the Amazon S3 bucket that contains the database installation files.
     */
    @JvmName("wnwvkjonixgwmqfe")
    public suspend fun databaseInstallationFilesS3BucketName(`value`: Output<String>) {
        this.databaseInstallationFilesS3BucketName = value
    }

    /**
     * @param value The prefix for the Amazon S3 bucket that contains the database installation files.
     */
    @JvmName("ogmojwtebnnwysfq")
    public suspend fun databaseInstallationFilesS3Prefix(`value`: Output<String>) {
        this.databaseInstallationFilesS3Prefix = value
    }

    /**
     * @param value The description of the CEV.
     */
    @JvmName("qhsmfjxeaiicbprv")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The name of the database engine. Valid values are `custom-oracle*`, `custom-sqlserver*`.
     */
    @JvmName("pwppdgnvefahcxuu")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value The version of the database engine.
     */
    @JvmName("iqvhtgpscvhsgtry")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value The name of the manifest file within the local filesystem. Conflicts with `manifest`.
     */
    @JvmName("slsocmtkrwrwkhkh")
    public suspend fun filename(`value`: Output<String>) {
        this.filename = value
    }

    /**
     * @param value The ARN of the AWS KMS key that is used to encrypt the database installation files. Required for RDS Custom for Oracle.
     */
    @JvmName("lrrwnheipmhiwuvp")
    public suspend fun kmsKeyId(`value`: Output<String>) {
        this.kmsKeyId = value
    }

    /**
     * @param value The manifest file, in JSON format, that contains the list of database installation files. Conflicts with `filename`.
     */
    @JvmName("yloabbpnimtbggvt")
    public suspend fun manifest(`value`: Output<String>) {
        this.manifest = value
    }

    /**
     * @param value Used to trigger updates. Must be set to a base64-encoded SHA256 hash of the manifest source specified with `filename`. The usual way to set this is filebase64sha256("manifest.json") where "manifest.json" is the local filename of the manifest source.
     */
    @JvmName("bhqfnwsgsnkhwvpi")
    public suspend fun manifestHash(`value`: Output<String>) {
        this.manifestHash = value
    }

    /**
     * @param value The ID of the AMI to create the CEV from. Required for RDS Custom for SQL Server. For RDS Custom for Oracle, you can specify an AMI ID that was used in a different Oracle CEV.
     */
    @JvmName("qjpiuapqpichmgfi")
    public suspend fun sourceImageId(`value`: Output<String>) {
        this.sourceImageId = value
    }

    /**
     * @param value The status of the CEV. Valid values are `available`, `inactive`, `inactive-except-restore`.
     */
    @JvmName("naguexmyleygmvom")
    public suspend fun status(`value`: Output<String>) {
        this.status = value
    }

    /**
     * @param value A mapping of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("vbuesaueegbymfkf")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value The name of the Amazon S3 bucket that contains the database installation files.
     */
    @JvmName("ihfawducipstxcin")
    public suspend fun databaseInstallationFilesS3BucketName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseInstallationFilesS3BucketName = mapped
    }

    /**
     * @param value The prefix for the Amazon S3 bucket that contains the database installation files.
     */
    @JvmName("hoomgbrenbrhidcp")
    public suspend fun databaseInstallationFilesS3Prefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseInstallationFilesS3Prefix = mapped
    }

    /**
     * @param value The description of the CEV.
     */
    @JvmName("htllwteblijpghhw")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The name of the database engine. Valid values are `custom-oracle*`, `custom-sqlserver*`.
     */
    @JvmName("wrishyeiqqpwyxrl")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value The version of the database engine.
     */
    @JvmName("swjanbuprpdtyivu")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value The name of the manifest file within the local filesystem. Conflicts with `manifest`.
     */
    @JvmName("yqolsxhdrelmcsnm")
    public suspend fun filename(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.filename = mapped
    }

    /**
     * @param value The ARN of the AWS KMS key that is used to encrypt the database installation files. Required for RDS Custom for Oracle.
     */
    @JvmName("rdqotdbbcqnkctfq")
    public suspend fun kmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKeyId = mapped
    }

    /**
     * @param value The manifest file, in JSON format, that contains the list of database installation files. Conflicts with `filename`.
     */
    @JvmName("eyuwghurrljcgrdc")
    public suspend fun manifest(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.manifest = mapped
    }

    /**
     * @param value Used to trigger updates. Must be set to a base64-encoded SHA256 hash of the manifest source specified with `filename`. The usual way to set this is filebase64sha256("manifest.json") where "manifest.json" is the local filename of the manifest source.
     */
    @JvmName("umvbgclgxyncieuj")
    public suspend fun manifestHash(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.manifestHash = mapped
    }

    /**
     * @param value The ID of the AMI to create the CEV from. Required for RDS Custom for SQL Server. For RDS Custom for Oracle, you can specify an AMI ID that was used in a different Oracle CEV.
     */
    @JvmName("mciqjkavoxupvauq")
    public suspend fun sourceImageId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceImageId = mapped
    }

    /**
     * @param value The status of the CEV. Valid values are `available`, `inactive`, `inactive-except-restore`.
     */
    @JvmName("ahepphqvpydcsrej")
    public suspend fun status(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.status = mapped
    }

    /**
     * @param value A mapping of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("bklqliclotsynspt")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A mapping of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("kwmvmtjhfnkjqgev")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): CustomDbEngineVersionArgs = CustomDbEngineVersionArgs(
        databaseInstallationFilesS3BucketName = databaseInstallationFilesS3BucketName,
        databaseInstallationFilesS3Prefix = databaseInstallationFilesS3Prefix,
        description = description,
        engine = engine,
        engineVersion = engineVersion,
        filename = filename,
        kmsKeyId = kmsKeyId,
        manifest = manifest,
        manifestHash = manifestHash,
        sourceImageId = sourceImageId,
        status = status,
        tags = tags,
    )
}
