@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.rds.kotlin

import com.pulumi.aws.rds.InstanceArgs.builder
import com.pulumi.aws.rds.kotlin.enums.InstanceType
import com.pulumi.aws.rds.kotlin.enums.StorageType
import com.pulumi.aws.rds.kotlin.inputs.InstanceBlueGreenUpdateArgs
import com.pulumi.aws.rds.kotlin.inputs.InstanceBlueGreenUpdateArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.InstanceRestoreToPointInTimeArgs
import com.pulumi.aws.rds.kotlin.inputs.InstanceRestoreToPointInTimeArgsBuilder
import com.pulumi.aws.rds.kotlin.inputs.InstanceS3ImportArgs
import com.pulumi.aws.rds.kotlin.inputs.InstanceS3ImportArgsBuilder
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides an RDS instance resource.  A DB instance is an isolated database
 * environment in the cloud.  A DB instance can contain multiple user-created
 * databases.
 * Changes to a DB instance can occur when you manually change a parameter, such as
 * `allocated_storage`, and are reflected in the next maintenance window. Because
 * of this, this provider may report a difference in its planning phase because a
 * modification has not yet taken place. You can use the `apply_immediately` flag
 * to instruct the service to apply the change immediately (see documentation
 * below).
 * When upgrading the major version of an engine, `allow_major_version_upgrade` must be set to `true`.
 * > **Note:** using `apply_immediately` can result in a brief downtime as the server reboots.
 * See the AWS Docs on [RDS Instance Maintenance][instance-maintenance] for more information.
 * > **Note:** All arguments including the username and password will be stored in the raw state as plain-text.
 * Read more about sensitive data instate.
 * ## RDS Instance Class Types
 * Amazon RDS supports instance classes for the following use cases: General-purpose, Memory-optimized, Burstable Performance, and Optimized-reads.
 * For more information please read the AWS RDS documentation about [DB Instance Class Types](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html)
 * ## Low-Downtime Updates
 * By default, RDS applies updates to DB Instances in-place, which can lead to service interruptions.
 * Low-downtime updates minimize service interruptions by performing the updates with an [RDS Blue/Green deployment][blue-green] and switching over the instances when complete.
 * Low-downtime updates are only available for DB Instances using MySQL, MariaDB and PostgreSQL,
 * as other engines are not supported by RDS Blue/Green deployments.
 * They cannot be used with DB Instances with replicas.
 * Backups must be enabled to use low-downtime updates.
 * Enable low-downtime updates by setting `blue_green_update.enabled` to `true`.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _default = new aws.rds.Instance("default", {
 *     allocatedStorage: 10,
 *     dbName: "mydb",
 *     engine: "mysql",
 *     engineVersion: "8.0",
 *     instanceClass: aws.rds.InstanceType.T3_Micro,
 *     username: "foo",
 *     password: "foobarbaz",
 *     parameterGroupName: "default.mysql8.0",
 *     skipFinalSnapshot: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * default = aws.rds.Instance("default",
 *     allocated_storage=10,
 *     db_name="mydb",
 *     engine="mysql",
 *     engine_version="8.0",
 *     instance_class=aws.rds.InstanceType.T3_MICRO,
 *     username="foo",
 *     password="foobarbaz",
 *     parameter_group_name="default.mysql8.0",
 *     skip_final_snapshot=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @default = new Aws.Rds.Instance("default", new()
 *     {
 *         AllocatedStorage = 10,
 *         DbName = "mydb",
 *         Engine = "mysql",
 *         EngineVersion = "8.0",
 *         InstanceClass = Aws.Rds.InstanceType.T3_Micro,
 *         Username = "foo",
 *         Password = "foobarbaz",
 *         ParameterGroupName = "default.mysql8.0",
 *         SkipFinalSnapshot = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewInstance(ctx, "default", &rds.InstanceArgs{
 * 			AllocatedStorage:   pulumi.Int(10),
 * 			DbName:             pulumi.String("mydb"),
 * 			Engine:             pulumi.String("mysql"),
 * 			EngineVersion:      pulumi.String("8.0"),
 * 			InstanceClass:      pulumi.String(rds.InstanceType_T3_Micro),
 * 			Username:           pulumi.String("foo"),
 * 			Password:           pulumi.String("foobarbaz"),
 * 			ParameterGroupName: pulumi.String("default.mysql8.0"),
 * 			SkipFinalSnapshot:  pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(10)
 *             .dbName("mydb")
 *             .engine("mysql")
 *             .engineVersion("8.0")
 *             .instanceClass("db.t3.micro")
 *             .username("foo")
 *             .password("foobarbaz")
 *             .parameterGroupName("default.mysql8.0")
 *             .skipFinalSnapshot(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   default:
 *     type: aws:rds:Instance
 *     properties:
 *       allocatedStorage: 10
 *       dbName: mydb
 *       engine: mysql
 *       engineVersion: '8.0'
 *       instanceClass: db.t3.micro
 *       username: foo
 *       password: foobarbaz
 *       parameterGroupName: default.mysql8.0
 *       skipFinalSnapshot: true
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Custom for Oracle Usage with Replica
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * // Lookup the available instance classes for the custom engine for the region being operated in
 * const custom-oracle = aws.rds.getOrderableDbInstance({
 *     engine: "custom-oracle-ee",
 *     engineVersion: "19.c.ee.002",
 *     licenseModel: "bring-your-own-license",
 *     storageType: "gp3",
 *     preferredInstanceClasses: [
 *         "db.r5.xlarge",
 *         "db.r5.2xlarge",
 *         "db.r5.4xlarge",
 *     ],
 * });
 * // The RDS instance resource requires an ARN. Look up the ARN of the KMS key associated with the CEV.
 * const byId = aws.kms.getKey({
 *     keyId: "example-ef278353ceba4a5a97de6784565b9f78",
 * });
 * const _default = new aws.rds.Instance("default", {
 *     allocatedStorage: 50,
 *     autoMinorVersionUpgrade: false,
 *     customIamInstanceProfile: "AWSRDSCustomInstanceProfile",
 *     backupRetentionPeriod: 7,
 *     dbSubnetGroupName: dbSubnetGroupName,
 *     engine: custom_oracle.then(custom_oracle => custom_oracle.engine),
 *     engineVersion: custom_oracle.then(custom_oracle => custom_oracle.engineVersion),
 *     identifier: "ee-instance-demo",
 *     instanceClass: custom_oracle.then(custom_oracle => custom_oracle.instanceClass).apply((x) => aws.rds.InstanceType[x]),
 *     kmsKeyId: byId.then(byId => byId.arn),
 *     licenseModel: custom_oracle.then(custom_oracle => custom_oracle.licenseModel),
 *     multiAz: false,
 *     password: "avoid-plaintext-passwords",
 *     username: "test",
 *     storageEncrypted: true,
 * });
 * const test_replica = new aws.rds.Instance("test-replica", {
 *     replicateSourceDb: _default.identifier,
 *     replicaMode: "mounted",
 *     autoMinorVersionUpgrade: false,
 *     customIamInstanceProfile: "AWSRDSCustomInstanceProfile",
 *     backupRetentionPeriod: 7,
 *     identifier: "ee-instance-replica",
 *     instanceClass: custom_oracle.then(custom_oracle => custom_oracle.instanceClass).apply((x) => aws.rds.InstanceType[x]),
 *     kmsKeyId: byId.then(byId => byId.arn),
 *     multiAz: false,
 *     skipFinalSnapshot: true,
 *     storageEncrypted: true,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * # Lookup the available instance classes for the custom engine for the region being operated in
 * custom_oracle = aws.rds.get_orderable_db_instance(engine="custom-oracle-ee",
 *     engine_version="19.c.ee.002",
 *     license_model="bring-your-own-license",
 *     storage_type="gp3",
 *     preferred_instance_classes=[
 *         "db.r5.xlarge",
 *         "db.r5.2xlarge",
 *         "db.r5.4xlarge",
 *     ])
 * # The RDS instance resource requires an ARN. Look up the ARN of the KMS key associated with the CEV.
 * by_id = aws.kms.get_key(key_id="example-ef278353ceba4a5a97de6784565b9f78")
 * default = aws.rds.Instance("default",
 *     allocated_storage=50,
 *     auto_minor_version_upgrade=False,
 *     custom_iam_instance_profile="AWSRDSCustomInstanceProfile",
 *     backup_retention_period=7,
 *     db_subnet_group_name=db_subnet_group_name,
 *     engine=custom_oracle.engine,
 *     engine_version=custom_oracle.engine_version,
 *     identifier="ee-instance-demo",
 *     instance_class=custom_oracle.instance_class.apply(lambda x: aws.rds.InstanceType(x)),
 *     kms_key_id=by_id.arn,
 *     license_model=custom_oracle.license_model,
 *     multi_az=False,
 *     password="avoid-plaintext-passwords",
 *     username="test",
 *     storage_encrypted=True)
 * test_replica = aws.rds.Instance("test-replica",
 *     replicate_source_db=default.identifier,
 *     replica_mode="mounted",
 *     auto_minor_version_upgrade=False,
 *     custom_iam_instance_profile="AWSRDSCustomInstanceProfile",
 *     backup_retention_period=7,
 *     identifier="ee-instance-replica",
 *     instance_class=custom_oracle.instance_class.apply(lambda x: aws.rds.InstanceType(x)),
 *     kms_key_id=by_id.arn,
 *     multi_az=False,
 *     skip_final_snapshot=True,
 *     storage_encrypted=True)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Lookup the available instance classes for the custom engine for the region being operated in
 *     var custom_oracle = Aws.Rds.GetOrderableDbInstance.Invoke(new()
 *     {
 *         Engine = "custom-oracle-ee",
 *         EngineVersion = "19.c.ee.002",
 *         LicenseModel = "bring-your-own-license",
 *         StorageType = "gp3",
 *         PreferredInstanceClasses = new[]
 *         {
 *             "db.r5.xlarge",
 *             "db.r5.2xlarge",
 *             "db.r5.4xlarge",
 *         },
 *     });
 *     // The RDS instance resource requires an ARN. Look up the ARN of the KMS key associated with the CEV.
 *     var byId = Aws.Kms.GetKey.Invoke(new()
 *     {
 *         KeyId = "example-ef278353ceba4a5a97de6784565b9f78",
 *     });
 *     var @default = new Aws.Rds.Instance("default", new()
 *     {
 *         AllocatedStorage = 50,
 *         AutoMinorVersionUpgrade = false,
 *         CustomIamInstanceProfile = "AWSRDSCustomInstanceProfile",
 *         BackupRetentionPeriod = 7,
 *         DbSubnetGroupName = dbSubnetGroupName,
 *         Engine = custom_oracle.Apply(custom_oracle => custom_oracle.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.Engine)),
 *         EngineVersion = custom_oracle.Apply(custom_oracle => custom_oracle.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.EngineVersion)),
 *         Identifier = "ee-instance-demo",
 *         InstanceClass = custom_oracle.Apply(custom_oracle => custom_oracle.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.InstanceClass)).Apply(System.Enum.Parse<Aws.Rds.InstanceType>),
 *         KmsKeyId = byId.Apply(getKeyResult => getKeyResult.Arn),
 *         LicenseModel = custom_oracle.Apply(custom_oracle => custom_oracle.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.LicenseModel)),
 *         MultiAz = false,
 *         Password = "avoid-plaintext-passwords",
 *         Username = "test",
 *         StorageEncrypted = true,
 *     });
 *     var test_replica = new Aws.Rds.Instance("test-replica", new()
 *     {
 *         ReplicateSourceDb = @default.Identifier,
 *         ReplicaMode = "mounted",
 *         AutoMinorVersionUpgrade = false,
 *         CustomIamInstanceProfile = "AWSRDSCustomInstanceProfile",
 *         BackupRetentionPeriod = 7,
 *         Identifier = "ee-instance-replica",
 *         InstanceClass = custom_oracle.Apply(custom_oracle => custom_oracle.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.InstanceClass)).Apply(System.Enum.Parse<Aws.Rds.InstanceType>),
 *         KmsKeyId = byId.Apply(getKeyResult => getKeyResult.Arn),
 *         MultiAz = false,
 *         SkipFinalSnapshot = true,
 *         StorageEncrypted = true,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Lookup the available instance classes for the custom engine for the region being operated in
 * 		custom_oracle, err := rds.GetOrderableDbInstance(ctx, &rds.GetOrderableDbInstanceArgs{
 * 			Engine:        "custom-oracle-ee",
 * 			EngineVersion: pulumi.StringRef("19.c.ee.002"),
 * 			LicenseModel:  pulumi.StringRef("bring-your-own-license"),
 * 			StorageType:   pulumi.StringRef("gp3"),
 * 			PreferredInstanceClasses: []string{
 * 				"db.r5.xlarge",
 * 				"db.r5.2xlarge",
 * 				"db.r5.4xlarge",
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// The RDS instance resource requires an ARN. Look up the ARN of the KMS key associated with the CEV.
 * 		byId, err := kms.LookupKey(ctx, &kms.LookupKeyArgs{
 * 			KeyId: "example-ef278353ceba4a5a97de6784565b9f78",
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewInstance(ctx, "default", &rds.InstanceArgs{
 * 			AllocatedStorage:         pulumi.Int(50),
 * 			AutoMinorVersionUpgrade:  pulumi.Bool(false),
 * 			CustomIamInstanceProfile: pulumi.String("AWSRDSCustomInstanceProfile"),
 * 			BackupRetentionPeriod:    pulumi.Int(7),
 * 			DbSubnetGroupName:        pulumi.Any(dbSubnetGroupName),
 * 			Engine:                   pulumi.String(custom_oracle.Engine),
 * 			EngineVersion:            pulumi.String(custom_oracle.EngineVersion),
 * 			Identifier:               pulumi.String("ee-instance-demo"),
 * 			InstanceClass:            custom_oracle.InstanceClass.ApplyT(func(x *string) rds.InstanceType { return rds.InstanceType(*x) }).(rds.InstanceTypeOutput),
 * 			KmsKeyId:                 pulumi.String(byId.Arn),
 * 			LicenseModel:             pulumi.String(custom_oracle.LicenseModel),
 * 			MultiAz:                  pulumi.Bool(false),
 * 			Password:                 pulumi.String("avoid-plaintext-passwords"),
 * 			Username:                 pulumi.String("test"),
 * 			StorageEncrypted:         pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewInstance(ctx, "test-replica", &rds.InstanceArgs{
 * 			ReplicateSourceDb:        _default.Identifier,
 * 			ReplicaMode:              pulumi.String("mounted"),
 * 			AutoMinorVersionUpgrade:  pulumi.Bool(false),
 * 			CustomIamInstanceProfile: pulumi.String("AWSRDSCustomInstanceProfile"),
 * 			BackupRetentionPeriod:    pulumi.Int(7),
 * 			Identifier:               pulumi.String("ee-instance-replica"),
 * 			InstanceClass:            custom_oracle.InstanceClass.ApplyT(func(x *string) rds.InstanceType { return rds.InstanceType(*x) }).(rds.InstanceTypeOutput),
 * 			KmsKeyId:                 pulumi.String(byId.Arn),
 * 			MultiAz:                  pulumi.Bool(false),
 * 			SkipFinalSnapshot:        pulumi.Bool(true),
 * 			StorageEncrypted:         pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.RdsFunctions;
 * import com.pulumi.aws.rds.inputs.GetOrderableDbInstanceArgs;
 * import com.pulumi.aws.kms.KmsFunctions;
 * import com.pulumi.aws.kms.inputs.GetKeyArgs;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Lookup the available instance classes for the custom engine for the region being operated in
 *         final var custom-oracle = RdsFunctions.getOrderableDbInstance(GetOrderableDbInstanceArgs.builder()
 *             .engine("custom-oracle-ee")
 *             .engineVersion("19.c.ee.002")
 *             .licenseModel("bring-your-own-license")
 *             .storageType("gp3")
 *             .preferredInstanceClasses(
 *                 "db.r5.xlarge",
 *                 "db.r5.2xlarge",
 *                 "db.r5.4xlarge")
 *             .build());
 *         // The RDS instance resource requires an ARN. Look up the ARN of the KMS key associated with the CEV.
 *         final var byId = KmsFunctions.getKey(GetKeyArgs.builder()
 *             .keyId("example-ef278353ceba4a5a97de6784565b9f78")
 *             .build());
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(50)
 *             .autoMinorVersionUpgrade(false)
 *             .customIamInstanceProfile("AWSRDSCustomInstanceProfile")
 *             .backupRetentionPeriod(7)
 *             .dbSubnetGroupName(dbSubnetGroupName)
 *             .engine(custom_oracle.engine())
 *             .engineVersion(custom_oracle.engineVersion())
 *             .identifier("ee-instance-demo")
 *             .instanceClass(custom_oracle.instanceClass())
 *             .kmsKeyId(byId.applyValue(getKeyResult -> getKeyResult.arn()))
 *             .licenseModel(custom_oracle.licenseModel())
 *             .multiAz(false)
 *             .password("avoid-plaintext-passwords")
 *             .username("test")
 *             .storageEncrypted(true)
 *             .build());
 *         var test_replica = new Instance("test-replica", InstanceArgs.builder()
 *             .replicateSourceDb(default_.identifier())
 *             .replicaMode("mounted")
 *             .autoMinorVersionUpgrade(false)
 *             .customIamInstanceProfile("AWSRDSCustomInstanceProfile")
 *             .backupRetentionPeriod(7)
 *             .identifier("ee-instance-replica")
 *             .instanceClass(custom_oracle.instanceClass())
 *             .kmsKeyId(byId.applyValue(getKeyResult -> getKeyResult.arn()))
 *             .multiAz(false)
 *             .skipFinalSnapshot(true)
 *             .storageEncrypted(true)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   default:
 *     type: aws:rds:Instance
 *     properties:
 *       allocatedStorage: 50
 *       autoMinorVersionUpgrade: false # Custom for Oracle does not support minor version upgrades
 *       customIamInstanceProfile: AWSRDSCustomInstanceProfile
 *       backupRetentionPeriod: 7
 *       dbSubnetGroupName: ${dbSubnetGroupName}
 *       engine: ${["custom-oracle"].engine}
 *       engineVersion: ${["custom-oracle"].engineVersion}
 *       identifier: ee-instance-demo
 *       instanceClass: ${["custom-oracle"].instanceClass}
 *       kmsKeyId: ${byId.arn}
 *       licenseModel: ${["custom-oracle"].licenseModel}
 *       multiAz: false # Custom for Oracle does not support multi-az
 *       password: avoid-plaintext-passwords
 *       username: test
 *       storageEncrypted: true
 *   test-replica:
 *     type: aws:rds:Instance
 *     properties:
 *       replicateSourceDb: ${default.identifier}
 *       replicaMode: mounted
 *       autoMinorVersionUpgrade: false
 *       customIamInstanceProfile: AWSRDSCustomInstanceProfile
 *       backupRetentionPeriod: 7
 *       identifier: ee-instance-replica
 *       instanceClass: ${["custom-oracle"].instanceClass}
 *       kmsKeyId: ${byId.arn}
 *       multiAz: false # Custom for Oracle does not support multi-az
 *       skipFinalSnapshot: true
 *       storageEncrypted: true
 * variables:
 *   # Lookup the available instance classes for the custom engine for the region being operated in
 *   custom-oracle:
 *     fn::invoke:
 *       Function: aws:rds:getOrderableDbInstance
 *       Arguments:
 *         engine: custom-oracle-ee
 *         engineVersion: 19.c.ee.002
 *         licenseModel: bring-your-own-license
 *         storageType: gp3
 *         preferredInstanceClasses:
 *           - db.r5.xlarge
 *           - db.r5.2xlarge
 *           - db.r5.4xlarge
 *   # The RDS instance resource requires an ARN. Look up the ARN of the KMS key associated with the CEV.
 *   byId:
 *     fn::invoke:
 *       Function: aws:kms:getKey
 *       Arguments:
 *         keyId: example-ef278353ceba4a5a97de6784565b9f78
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Custom for SQL Server
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * // Lookup the available instance classes for the custom engine for the region being operated in
 * const custom-sqlserver = aws.rds.getOrderableDbInstance({
 *     engine: "custom-sqlserver-se",
 *     engineVersion: "15.00.4249.2.v1",
 *     storageType: "gp3",
 *     preferredInstanceClasses: [
 *         "db.r5.xlarge",
 *         "db.r5.2xlarge",
 *         "db.r5.4xlarge",
 *     ],
 * });
 * // The RDS instance resource requires an ARN. Look up the ARN of the KMS key.
 * const byId = aws.kms.getKey({
 *     keyId: "example-ef278353ceba4a5a97de6784565b9f78",
 * });
 * const example = new aws.rds.Instance("example", {
 *     allocatedStorage: 500,
 *     autoMinorVersionUpgrade: false,
 *     customIamInstanceProfile: "AWSRDSCustomSQLServerInstanceProfile",
 *     backupRetentionPeriod: 7,
 *     dbSubnetGroupName: dbSubnetGroupName,
 *     engine: custom_sqlserver.then(custom_sqlserver => custom_sqlserver.engine),
 *     engineVersion: custom_sqlserver.then(custom_sqlserver => custom_sqlserver.engineVersion),
 *     identifier: "sql-instance-demo",
 *     instanceClass: custom_sqlserver.then(custom_sqlserver => custom_sqlserver.instanceClass).apply((x) => aws.rds.InstanceType[x]),
 *     kmsKeyId: byId.then(byId => byId.arn),
 *     multiAz: false,
 *     password: "avoid-plaintext-passwords",
 *     storageEncrypted: true,
 *     username: "test",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * # Lookup the available instance classes for the custom engine for the region being operated in
 * custom_sqlserver = aws.rds.get_orderable_db_instance(engine="custom-sqlserver-se",
 *     engine_version="15.00.4249.2.v1",
 *     storage_type="gp3",
 *     preferred_instance_classes=[
 *         "db.r5.xlarge",
 *         "db.r5.2xlarge",
 *         "db.r5.4xlarge",
 *     ])
 * # The RDS instance resource requires an ARN. Look up the ARN of the KMS key.
 * by_id = aws.kms.get_key(key_id="example-ef278353ceba4a5a97de6784565b9f78")
 * example = aws.rds.Instance("example",
 *     allocated_storage=500,
 *     auto_minor_version_upgrade=False,
 *     custom_iam_instance_profile="AWSRDSCustomSQLServerInstanceProfile",
 *     backup_retention_period=7,
 *     db_subnet_group_name=db_subnet_group_name,
 *     engine=custom_sqlserver.engine,
 *     engine_version=custom_sqlserver.engine_version,
 *     identifier="sql-instance-demo",
 *     instance_class=custom_sqlserver.instance_class.apply(lambda x: aws.rds.InstanceType(x)),
 *     kms_key_id=by_id.arn,
 *     multi_az=False,
 *     password="avoid-plaintext-passwords",
 *     storage_encrypted=True,
 *     username="test")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Lookup the available instance classes for the custom engine for the region being operated in
 *     var custom_sqlserver = Aws.Rds.GetOrderableDbInstance.Invoke(new()
 *     {
 *         Engine = "custom-sqlserver-se",
 *         EngineVersion = "15.00.4249.2.v1",
 *         StorageType = "gp3",
 *         PreferredInstanceClasses = new[]
 *         {
 *             "db.r5.xlarge",
 *             "db.r5.2xlarge",
 *             "db.r5.4xlarge",
 *         },
 *     });
 *     // The RDS instance resource requires an ARN. Look up the ARN of the KMS key.
 *     var byId = Aws.Kms.GetKey.Invoke(new()
 *     {
 *         KeyId = "example-ef278353ceba4a5a97de6784565b9f78",
 *     });
 *     var example = new Aws.Rds.Instance("example", new()
 *     {
 *         AllocatedStorage = 500,
 *         AutoMinorVersionUpgrade = false,
 *         CustomIamInstanceProfile = "AWSRDSCustomSQLServerInstanceProfile",
 *         BackupRetentionPeriod = 7,
 *         DbSubnetGroupName = dbSubnetGroupName,
 *         Engine = custom_sqlserver.Apply(custom_sqlserver => custom_sqlserver.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.Engine)),
 *         EngineVersion = custom_sqlserver.Apply(custom_sqlserver => custom_sqlserver.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.EngineVersion)),
 *         Identifier = "sql-instance-demo",
 *         InstanceClass = custom_sqlserver.Apply(custom_sqlserver => custom_sqlserver.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.InstanceClass)).Apply(System.Enum.Parse<Aws.Rds.InstanceType>),
 *         KmsKeyId = byId.Apply(getKeyResult => getKeyResult.Arn),
 *         MultiAz = false,
 *         Password = "avoid-plaintext-passwords",
 *         StorageEncrypted = true,
 *         Username = "test",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Lookup the available instance classes for the custom engine for the region being operated in
 * 		custom_sqlserver, err := rds.GetOrderableDbInstance(ctx, &rds.GetOrderableDbInstanceArgs{
 * 			Engine:        "custom-sqlserver-se",
 * 			EngineVersion: pulumi.StringRef("15.00.4249.2.v1"),
 * 			StorageType:   pulumi.StringRef("gp3"),
 * 			PreferredInstanceClasses: []string{
 * 				"db.r5.xlarge",
 * 				"db.r5.2xlarge",
 * 				"db.r5.4xlarge",
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// The RDS instance resource requires an ARN. Look up the ARN of the KMS key.
 * 		byId, err := kms.LookupKey(ctx, &kms.LookupKeyArgs{
 * 			KeyId: "example-ef278353ceba4a5a97de6784565b9f78",
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewInstance(ctx, "example", &rds.InstanceArgs{
 * 			AllocatedStorage:         pulumi.Int(500),
 * 			AutoMinorVersionUpgrade:  pulumi.Bool(false),
 * 			CustomIamInstanceProfile: pulumi.String("AWSRDSCustomSQLServerInstanceProfile"),
 * 			BackupRetentionPeriod:    pulumi.Int(7),
 * 			DbSubnetGroupName:        pulumi.Any(dbSubnetGroupName),
 * 			Engine:                   pulumi.String(custom_sqlserver.Engine),
 * 			EngineVersion:            pulumi.String(custom_sqlserver.EngineVersion),
 * 			Identifier:               pulumi.String("sql-instance-demo"),
 * 			InstanceClass:            custom_sqlserver.InstanceClass.ApplyT(func(x *string) rds.InstanceType { return rds.InstanceType(*x) }).(rds.InstanceTypeOutput),
 * 			KmsKeyId:                 pulumi.String(byId.Arn),
 * 			MultiAz:                  pulumi.Bool(false),
 * 			Password:                 pulumi.String("avoid-plaintext-passwords"),
 * 			StorageEncrypted:         pulumi.Bool(true),
 * 			Username:                 pulumi.String("test"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.RdsFunctions;
 * import com.pulumi.aws.rds.inputs.GetOrderableDbInstanceArgs;
 * import com.pulumi.aws.kms.KmsFunctions;
 * import com.pulumi.aws.kms.inputs.GetKeyArgs;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Lookup the available instance classes for the custom engine for the region being operated in
 *         final var custom-sqlserver = RdsFunctions.getOrderableDbInstance(GetOrderableDbInstanceArgs.builder()
 *             .engine("custom-sqlserver-se")
 *             .engineVersion("15.00.4249.2.v1")
 *             .storageType("gp3")
 *             .preferredInstanceClasses(
 *                 "db.r5.xlarge",
 *                 "db.r5.2xlarge",
 *                 "db.r5.4xlarge")
 *             .build());
 *         // The RDS instance resource requires an ARN. Look up the ARN of the KMS key.
 *         final var byId = KmsFunctions.getKey(GetKeyArgs.builder()
 *             .keyId("example-ef278353ceba4a5a97de6784565b9f78")
 *             .build());
 *         var example = new Instance("example", InstanceArgs.builder()
 *             .allocatedStorage(500)
 *             .autoMinorVersionUpgrade(false)
 *             .customIamInstanceProfile("AWSRDSCustomSQLServerInstanceProfile")
 *             .backupRetentionPeriod(7)
 *             .dbSubnetGroupName(dbSubnetGroupName)
 *             .engine(custom_sqlserver.engine())
 *             .engineVersion(custom_sqlserver.engineVersion())
 *             .identifier("sql-instance-demo")
 *             .instanceClass(custom_sqlserver.instanceClass())
 *             .kmsKeyId(byId.applyValue(getKeyResult -> getKeyResult.arn()))
 *             .multiAz(false)
 *             .password("avoid-plaintext-passwords")
 *             .storageEncrypted(true)
 *             .username("test")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:rds:Instance
 *     properties:
 *       allocatedStorage: 500
 *       autoMinorVersionUpgrade: false # Custom for SQL Server does not support minor version upgrades
 *       customIamInstanceProfile: AWSRDSCustomSQLServerInstanceProfile
 *       backupRetentionPeriod: 7
 *       dbSubnetGroupName: ${dbSubnetGroupName}
 *       engine: ${["custom-sqlserver"].engine}
 *       engineVersion: ${["custom-sqlserver"].engineVersion}
 *       identifier: sql-instance-demo
 *       instanceClass: ${["custom-sqlserver"].instanceClass}
 *       kmsKeyId: ${byId.arn}
 *       multiAz: false # Custom for SQL Server does support multi-az
 *       password: avoid-plaintext-passwords
 *       storageEncrypted: true
 *       username: test
 * variables:
 *   # Lookup the available instance classes for the custom engine for the region being operated in
 *   custom-sqlserver:
 *     fn::invoke:
 *       Function: aws:rds:getOrderableDbInstance
 *       Arguments:
 *         engine: custom-sqlserver-se
 *         engineVersion: 15.00.4249.2.v1
 *         storageType: gp3
 *         preferredInstanceClasses:
 *           - db.r5.xlarge
 *           - db.r5.2xlarge
 *           - db.r5.4xlarge
 *   # The RDS instance resource requires an ARN. Look up the ARN of the KMS key.
 *   byId:
 *     fn::invoke:
 *       Function: aws:kms:getKey
 *       Arguments:
 *         keyId: example-ef278353ceba4a5a97de6784565b9f78
 * ```
 * <!--End PulumiCodeChooser -->
 * ### RDS Db2 Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * // Lookup the default version for the engine. Db2 Standard Edition is `db2-se`, Db2 Advanced Edition is `db2-ae`.
 * const default = aws.rds.getEngineVersion({
 *     engine: "db2-se",
 * });
 * // Lookup the available instance classes for the engine in the region being operated in
 * const example = Promise.all([_default, _default])&#46;then(([_default, _default1]) => aws.rds.getOrderableDbInstance({
 *     engine: _default.engine,
 *     engineVersion: _default1.version,
 *     licenseModel: "bring-your-own-license",
 *     storageType: "gp3",
 *     preferredInstanceClasses: [
 *         "db.t3.small",
 *         "db.r6i.large",
 *         "db.m6i.large",
 *     ],
 * }));
 * // The RDS Db2 instance resource requires licensing information. Create a new parameter group using the default paramater group as a source, and set license information.
 * const exampleParameterGroup = new aws.rds.ParameterGroup("example", {
 *     name: "db-db2-params",
 *     family: _default.then(_default => _default.parameterGroupFamily),
 *     parameters: [
 *         {
 *             applyMethod: "immediate",
 *             name: "rds.ibm_customer_id",
 *             value: "0",
 *         },
 *         {
 *             applyMethod: "immediate",
 *             name: "rds.ibm_site_id",
 *             value: "0",
 *         },
 *     ],
 * });
 * // Create the RDS Db2 instance, use the data sources defined to set attributes
 * const exampleInstance = new aws.rds.Instance("example", {
 *     allocatedStorage: 100,
 *     backupRetentionPeriod: 7,
 *     dbName: "test",
 *     engine: example.then(example => example.engine),
 *     engineVersion: example.then(example => example.engineVersion),
 *     identifier: "db2-instance-demo",
 *     instanceClass: example.then(example => example.instanceClass).apply((x) => aws.rds.InstanceType[x]),
 *     parameterGroupName: exampleParameterGroup.name,
 *     password: "avoid-plaintext-passwords",
 *     username: "test",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * # Lookup the default version for the engine. Db2 Standard Edition is `db2-se`, Db2 Advanced Edition is `db2-ae`.
 * default = aws.rds.get_engine_version(engine="db2-se")
 * # Lookup the available instance classes for the engine in the region being operated in
 * example = aws.rds.get_orderable_db_instance(engine=default.engine,
 *     engine_version=default.version,
 *     license_model="bring-your-own-license",
 *     storage_type="gp3",
 *     preferred_instance_classes=[
 *         "db.t3.small",
 *         "db.r6i.large",
 *         "db.m6i.large",
 *     ])
 * # The RDS Db2 instance resource requires licensing information. Create a new parameter group using the default paramater group as a source, and set license information.
 * example_parameter_group = aws.rds.ParameterGroup("example",
 *     name="db-db2-params",
 *     family=default.parameter_group_family,
 *     parameters=[
 *         {
 *             "apply_method": "immediate",
 *             "name": "rds.ibm_customer_id",
 *             "value": "0",
 *         },
 *         {
 *             "apply_method": "immediate",
 *             "name": "rds.ibm_site_id",
 *             "value": "0",
 *         },
 *     ])
 * # Create the RDS Db2 instance, use the data sources defined to set attributes
 * example_instance = aws.rds.Instance("example",
 *     allocated_storage=100,
 *     backup_retention_period=7,
 *     db_name="test",
 *     engine=example.engine,
 *     engine_version=example.engine_version,
 *     identifier="db2-instance-demo",
 *     instance_class=example.instance_class.apply(lambda x: aws.rds.InstanceType(x)),
 *     parameter_group_name=example_parameter_group.name,
 *     password="avoid-plaintext-passwords",
 *     username="test")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Lookup the default version for the engine. Db2 Standard Edition is `db2-se`, Db2 Advanced Edition is `db2-ae`.
 *     var @default = Aws.Rds.GetEngineVersion.Invoke(new()
 *     {
 *         Engine = "db2-se",
 *     });
 *     // Lookup the available instance classes for the engine in the region being operated in
 *     var example = Aws.Rds.GetOrderableDbInstance.Invoke(new()
 *     {
 *         Engine = @default.Apply(getEngineVersionResult => getEngineVersionResult.Engine),
 *         EngineVersion = @default.Apply(getEngineVersionResult => getEngineVersionResult.Version),
 *         LicenseModel = "bring-your-own-license",
 *         StorageType = "gp3",
 *         PreferredInstanceClasses = new[]
 *         {
 *             "db.t3.small",
 *             "db.r6i.large",
 *             "db.m6i.large",
 *         },
 *     });
 *     // The RDS Db2 instance resource requires licensing information. Create a new parameter group using the default paramater group as a source, and set license information.
 *     var exampleParameterGroup = new Aws.Rds.ParameterGroup("example", new()
 *     {
 *         Name = "db-db2-params",
 *         Family = @default.Apply(@default => @default.Apply(getEngineVersionResult => getEngineVersionResult.ParameterGroupFamily)),
 *         Parameters = new[]
 *         {
 *             new Aws.Rds.Inputs.ParameterGroupParameterArgs
 *             {
 *                 ApplyMethod = "immediate",
 *                 Name = "rds.ibm_customer_id",
 *                 Value = "0",
 *             },
 *             new Aws.Rds.Inputs.ParameterGroupParameterArgs
 *             {
 *                 ApplyMethod = "immediate",
 *                 Name = "rds.ibm_site_id",
 *                 Value = "0",
 *             },
 *         },
 *     });
 *     // Create the RDS Db2 instance, use the data sources defined to set attributes
 *     var exampleInstance = new Aws.Rds.Instance("example", new()
 *     {
 *         AllocatedStorage = 100,
 *         BackupRetentionPeriod = 7,
 *         DbName = "test",
 *         Engine = example.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.Engine),
 *         EngineVersion = example.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.EngineVersion),
 *         Identifier = "db2-instance-demo",
 *         InstanceClass = example.Apply(getOrderableDbInstanceResult => getOrderableDbInstanceResult.InstanceClass).Apply(System.Enum.Parse<Aws.Rds.InstanceType>),
 *         ParameterGroupName = exampleParameterGroup.Name,
 *         Password = "avoid-plaintext-passwords",
 *         Username = "test",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Lookup the default version for the engine. Db2 Standard Edition is `db2-se`, Db2 Advanced Edition is `db2-ae`.
 * 		_default, err := rds.GetEngineVersion(ctx, &rds.GetEngineVersionArgs{
 * 			Engine: "db2-se",
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Lookup the available instance classes for the engine in the region being operated in
 * 		example, err := rds.GetOrderableDbInstance(ctx, &rds.GetOrderableDbInstanceArgs{
 * 			Engine:        _default.Engine,
 * 			EngineVersion: pulumi.StringRef(_default.Version),
 * 			LicenseModel:  pulumi.StringRef("bring-your-own-license"),
 * 			StorageType:   pulumi.StringRef("gp3"),
 * 			PreferredInstanceClasses: []string{
 * 				"db.t3.small",
 * 				"db.r6i.large",
 * 				"db.m6i.large",
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// The RDS Db2 instance resource requires licensing information. Create a new parameter group using the default paramater group as a source, and set license information.
 * 		exampleParameterGroup, err := rds.NewParameterGroup(ctx, "example", &rds.ParameterGroupArgs{
 * 			Name:   pulumi.String("db-db2-params"),
 * 			Family: pulumi.String(_default.ParameterGroupFamily),
 * 			Parameters: rds.ParameterGroupParameterArray{
 * 				&rds.ParameterGroupParameterArgs{
 * 					ApplyMethod: pulumi.String("immediate"),
 * 					Name:        pulumi.String("rds.ibm_customer_id"),
 * 					Value:       pulumi.String("0"),
 * 				},
 * 				&rds.ParameterGroupParameterArgs{
 * 					ApplyMethod: pulumi.String("immediate"),
 * 					Name:        pulumi.String("rds.ibm_site_id"),
 * 					Value:       pulumi.String("0"),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Create the RDS Db2 instance, use the data sources defined to set attributes
 * 		_, err = rds.NewInstance(ctx, "example", &rds.InstanceArgs{
 * 			AllocatedStorage:      pulumi.Int(100),
 * 			BackupRetentionPeriod: pulumi.Int(7),
 * 			DbName:                pulumi.String("test"),
 * 			Engine:                pulumi.String(example.Engine),
 * 			EngineVersion:         pulumi.String(example.EngineVersion),
 * 			Identifier:            pulumi.String("db2-instance-demo"),
 * 			InstanceClass:         example.InstanceClass.ApplyT(func(x *string) rds.InstanceType { return rds.InstanceType(*x) }).(rds.InstanceTypeOutput),
 * 			ParameterGroupName:    exampleParameterGroup.Name,
 * 			Password:              pulumi.String("avoid-plaintext-passwords"),
 * 			Username:              pulumi.String("test"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.RdsFunctions;
 * import com.pulumi.aws.rds.inputs.GetEngineVersionArgs;
 * import com.pulumi.aws.rds.inputs.GetOrderableDbInstanceArgs;
 * import com.pulumi.aws.rds.ParameterGroup;
 * import com.pulumi.aws.rds.ParameterGroupArgs;
 * import com.pulumi.aws.rds.inputs.ParameterGroupParameterArgs;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Lookup the default version for the engine. Db2 Standard Edition is `db2-se`, Db2 Advanced Edition is `db2-ae`.
 *         final var default = RdsFunctions.getEngineVersion(GetEngineVersionArgs.builder()
 *             .engine("db2-se")
 *             .build());
 *         // Lookup the available instance classes for the engine in the region being operated in
 *         final var example = RdsFunctions.getOrderableDbInstance(GetOrderableDbInstanceArgs.builder()
 *             .engine(default_.engine())
 *             .engineVersion(default_.version())
 *             .licenseModel("bring-your-own-license")
 *             .storageType("gp3")
 *             .preferredInstanceClasses(
 *                 "db.t3.small",
 *                 "db.r6i.large",
 *                 "db.m6i.large")
 *             .build());
 *         // The RDS Db2 instance resource requires licensing information. Create a new parameter group using the default paramater group as a source, and set license information.
 *         var exampleParameterGroup = new ParameterGroup("exampleParameterGroup", ParameterGroupArgs.builder()
 *             .name("db-db2-params")
 *             .family(default_.parameterGroupFamily())
 *             .parameters(
 *                 ParameterGroupParameterArgs.builder()
 *                     .applyMethod("immediate")
 *                     .name("rds.ibm_customer_id")
 *                     .value(0)
 *                     .build(),
 *                 ParameterGroupParameterArgs.builder()
 *                     .applyMethod("immediate")
 *                     .name("rds.ibm_site_id")
 *                     .value(0)
 *                     .build())
 *             .build());
 *         // Create the RDS Db2 instance, use the data sources defined to set attributes
 *         var exampleInstance = new Instance("exampleInstance", InstanceArgs.builder()
 *             .allocatedStorage(100)
 *             .backupRetentionPeriod(7)
 *             .dbName("test")
 *             .engine(example.applyValue(getOrderableDbInstanceResult -> getOrderableDbInstanceResult.engine()))
 *             .engineVersion(example.applyValue(getOrderableDbInstanceResult -> getOrderableDbInstanceResult.engineVersion()))
 *             .identifier("db2-instance-demo")
 *             .instanceClass(example.applyValue(getOrderableDbInstanceResult -> getOrderableDbInstanceResult.instanceClass()))
 *             .parameterGroupName(exampleParameterGroup.name())
 *             .password("avoid-plaintext-passwords")
 *             .username("test")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # The RDS Db2 instance resource requires licensing information. Create a new parameter group using the default paramater group as a source, and set license information.
 *   exampleParameterGroup:
 *     type: aws:rds:ParameterGroup
 *     name: example
 *     properties:
 *       name: db-db2-params
 *       family: ${default.parameterGroupFamily}
 *       parameters:
 *         - applyMethod: immediate
 *           name: rds.ibm_customer_id
 *           value: 0
 *         - applyMethod: immediate
 *           name: rds.ibm_site_id
 *           value: 0
 *   # Create the RDS Db2 instance, use the data sources defined to set attributes
 *   exampleInstance:
 *     type: aws:rds:Instance
 *     name: example
 *     properties:
 *       allocatedStorage: 100
 *       backupRetentionPeriod: 7
 *       dbName: test
 *       engine: ${example.engine}
 *       engineVersion: ${example.engineVersion}
 *       identifier: db2-instance-demo
 *       instanceClass: ${example.instanceClass}
 *       parameterGroupName: ${exampleParameterGroup.name}
 *       password: avoid-plaintext-passwords
 *       username: test
 * variables:
 *   # Lookup the default version for the engine. Db2 Standard Edition is `db2-se`, Db2 Advanced Edition is `db2-ae`.
 *   default:
 *     fn::invoke:
 *       Function: aws:rds:getEngineVersion
 *       Arguments:
 *         engine: db2-se
 *   # Lookup the available instance classes for the engine in the region being operated in
 *   example:
 *     fn::invoke:
 *       Function: aws:rds:getOrderableDbInstance
 *       Arguments:
 *         engine: ${default.engine}
 *         engineVersion: ${default.version}
 *         licenseModel: bring-your-own-license
 *         storageType: gp3
 *         preferredInstanceClasses:
 *           - db.t3.small
 *           - db.r6i.large
 *           - db.m6i.large
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Storage Autoscaling
 * To enable Storage Autoscaling with instances that support the feature, define the `max_allocated_storage` argument higher than the `allocated_storage` argument. This provider will automatically hide differences with the `allocated_storage` argument value if autoscaling occurs.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.rds.Instance("example", {
 *     allocatedStorage: 50,
 *     maxAllocatedStorage: 100,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.rds.Instance("example",
 *     allocated_storage=50,
 *     max_allocated_storage=100)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Rds.Instance("example", new()
 *     {
 *         AllocatedStorage = 50,
 *         MaxAllocatedStorage = 100,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewInstance(ctx, "example", &rds.InstanceArgs{
 * 			AllocatedStorage:    pulumi.Int(50),
 * 			MaxAllocatedStorage: pulumi.Int(100),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Instance("example", InstanceArgs.builder()
 *             .allocatedStorage(50)
 *             .maxAllocatedStorage(100)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:rds:Instance
 *     properties:
 *       allocatedStorage: 50
 *       maxAllocatedStorage: 100
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Managed Master Passwords via Secrets Manager, default KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws&#46;amazon&#46;com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `manage_master_user_password` attribute to enable managing the master password with Secrets Manager. You can also update an existing cluster to use Secrets Manager by specify the `manage_master_user_password` attribute and removing the `password` attribute (removal is required).
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _default = new aws.rds.Instance("default", {
 *     allocatedStorage: 10,
 *     dbName: "mydb",
 *     engine: "mysql",
 *     engineVersion: "8.0",
 *     instanceClass: aws.rds.InstanceType.T3_Micro,
 *     manageMasterUserPassword: true,
 *     username: "foo",
 *     parameterGroupName: "default.mysql8.0",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * default = aws.rds.Instance("default",
 *     allocated_storage=10,
 *     db_name="mydb",
 *     engine="mysql",
 *     engine_version="8.0",
 *     instance_class=aws.rds.InstanceType.T3_MICRO,
 *     manage_master_user_password=True,
 *     username="foo",
 *     parameter_group_name="default.mysql8.0")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @default = new Aws.Rds.Instance("default", new()
 *     {
 *         AllocatedStorage = 10,
 *         DbName = "mydb",
 *         Engine = "mysql",
 *         EngineVersion = "8.0",
 *         InstanceClass = Aws.Rds.InstanceType.T3_Micro,
 *         ManageMasterUserPassword = true,
 *         Username = "foo",
 *         ParameterGroupName = "default.mysql8.0",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := rds.NewInstance(ctx, "default", &rds.InstanceArgs{
 * 			AllocatedStorage:         pulumi.Int(10),
 * 			DbName:                   pulumi.String("mydb"),
 * 			Engine:                   pulumi.String("mysql"),
 * 			EngineVersion:            pulumi.String("8.0"),
 * 			InstanceClass:            pulumi.String(rds.InstanceType_T3_Micro),
 * 			ManageMasterUserPassword: pulumi.Bool(true),
 * 			Username:                 pulumi.String("foo"),
 * 			ParameterGroupName:       pulumi.String("default.mysql8.0"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(10)
 *             .dbName("mydb")
 *             .engine("mysql")
 *             .engineVersion("8.0")
 *             .instanceClass("db.t3.micro")
 *             .manageMasterUserPassword(true)
 *             .username("foo")
 *             .parameterGroupName("default.mysql8.0")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   default:
 *     type: aws:rds:Instance
 *     properties:
 *       allocatedStorage: 10
 *       dbName: mydb
 *       engine: mysql
 *       engineVersion: '8.0'
 *       instanceClass: db.t3.micro
 *       manageMasterUserPassword: true
 *       username: foo
 *       parameterGroupName: default.mysql8.0
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Managed Master Passwords via Secrets Manager, specific KMS Key
 * > More information about RDS/Aurora Aurora integrates with Secrets Manager to manage master user passwords for your DB clusters can be found in the [RDS User Guide](https://aws&#46;amazon&#46;com/about-aws/whats-new/2022/12/amazon-rds-integration-aws-secrets-manager/) and [Aurora User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html).
 * You can specify the `master_user_secret_kms_key_id` attribute to specify a specific KMS Key.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.kms.Key("example", {description: "Example KMS Key"});
 * const _default = new aws.rds.Instance("default", {
 *     allocatedStorage: 10,
 *     dbName: "mydb",
 *     engine: "mysql",
 *     engineVersion: "8.0",
 *     instanceClass: aws.rds.InstanceType.T3_Micro,
 *     manageMasterUserPassword: true,
 *     masterUserSecretKmsKeyId: example.keyId,
 *     username: "foo",
 *     parameterGroupName: "default.mysql8.0",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.kms.Key("example", description="Example KMS Key")
 * default = aws.rds.Instance("default",
 *     allocated_storage=10,
 *     db_name="mydb",
 *     engine="mysql",
 *     engine_version="8.0",
 *     instance_class=aws.rds.InstanceType.T3_MICRO,
 *     manage_master_user_password=True,
 *     master_user_secret_kms_key_id=example.key_id,
 *     username="foo",
 *     parameter_group_name="default.mysql8.0")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Kms.Key("example", new()
 *     {
 *         Description = "Example KMS Key",
 *     });
 *     var @default = new Aws.Rds.Instance("default", new()
 *     {
 *         AllocatedStorage = 10,
 *         DbName = "mydb",
 *         Engine = "mysql",
 *         EngineVersion = "8.0",
 *         InstanceClass = Aws.Rds.InstanceType.T3_Micro,
 *         ManageMasterUserPassword = true,
 *         MasterUserSecretKmsKeyId = example.KeyId,
 *         Username = "foo",
 *         ParameterGroupName = "default.mysql8.0",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/rds"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := kms.NewKey(ctx, "example", &kms.KeyArgs{
 * 			Description: pulumi.String("Example KMS Key"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = rds.NewInstance(ctx, "default", &rds.InstanceArgs{
 * 			AllocatedStorage:         pulumi.Int(10),
 * 			DbName:                   pulumi.String("mydb"),
 * 			Engine:                   pulumi.String("mysql"),
 * 			EngineVersion:            pulumi.String("8.0"),
 * 			InstanceClass:            pulumi.String(rds.InstanceType_T3_Micro),
 * 			ManageMasterUserPassword: pulumi.Bool(true),
 * 			MasterUserSecretKmsKeyId: example.KeyId,
 * 			Username:                 pulumi.String("foo"),
 * 			ParameterGroupName:       pulumi.String("default.mysql8.0"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.rds.Instance;
 * import com.pulumi.aws.rds.InstanceArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Key("example", KeyArgs.builder()
 *             .description("Example KMS Key")
 *             .build());
 *         var default_ = new Instance("default", InstanceArgs.builder()
 *             .allocatedStorage(10)
 *             .dbName("mydb")
 *             .engine("mysql")
 *             .engineVersion("8.0")
 *             .instanceClass("db.t3.micro")
 *             .manageMasterUserPassword(true)
 *             .masterUserSecretKmsKeyId(example.keyId())
 *             .username("foo")
 *             .parameterGroupName("default.mysql8.0")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:kms:Key
 *     properties:
 *       description: Example KMS Key
 *   default:
 *     type: aws:rds:Instance
 *     properties:
 *       allocatedStorage: 10
 *       dbName: mydb
 *       engine: mysql
 *       engineVersion: '8.0'
 *       instanceClass: db.t3.micro
 *       manageMasterUserPassword: true
 *       masterUserSecretKmsKeyId: ${example.keyId}
 *       username: foo
 *       parameterGroupName: default.mysql8.0
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import DB Instances using the `identifier`. For example:
 * ```sh
 * $ pulumi import aws:rds/instance:Instance default mydb-rds-instance
 * ```
 * @property allocatedStorage The allocated storage in gibibytes. If `max_allocated_storage` is configured, this argument represents the initial storage allocation and differences from the configuration will be ignored automatically when Storage Autoscaling occurs. If `replicate_source_db` is set, the value is ignored during the creation of the instance.
 * @property allowMajorVersionUpgrade Indicates that major version
 * upgrades are allowed. Changing this parameter does not result in an outage and
 * the change is asynchronously applied as soon as possible.
 * @property applyImmediately Specifies whether any database modifications
 * are applied immediately, or during the next maintenance window. Default is
 * `false`. See [Amazon RDS Documentation for more
 * information.](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
 * @property autoMinorVersionUpgrade Indicates that minor engine upgrades
 * will be applied automatically to the DB instance during the maintenance window.
 * Defaults to true.
 * @property availabilityZone The AZ for the RDS instance.
 * @property backupRetentionPeriod The days to retain backups for.
 * Must be between `0` and `35`.
 * Default is `0`.
 * Must be greater than `0` if the database is used as a source for a [Read Replica][instance-replication],
 * uses low-downtime updates,
 * or will use [RDS Blue/Green deployments][blue-green].
 * @property backupTarget Specifies where automated backups and manual snapshots are stored. Possible values are `region` (default) and `outposts`. See [Working with Amazon RDS on AWS Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html) for more information.
 * @property backupWindow The daily time range (in UTC) during which automated backups are created if they are enabled.
 * Example: "09:46-10:16". Must not overlap with `maintenance_window`.
 * @property blueGreenUpdate Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
 * See `blue_green_update` below.
 * @property caCertIdentifier The identifier of the CA certificate for the DB instance.
 * @property characterSetName The character set name to use for DB encoding in Oracle and Microsoft SQL instances (collation).
 * This can't be changed.
 * See [Oracle Character Sets Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html) or
 * [Server-Level Collation for Microsoft SQL Server](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.CommonDBATasks.Collation.html) for more information.
 * Cannot be set  with `replicate_source_db`, `restore_to_point_in_time`, `s3_import`, or `snapshot_identifier`.
 * @property copyTagsToSnapshot Copy all Instance `tags` to snapshots. Default is `false`.
 * @property customIamInstanceProfile The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
 * @property customerOwnedIpEnabled Indicates whether to enable a customer-owned IP address (CoIP) for an RDS on Outposts DB instance. See [CoIP for RDS on Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html#rds-on-outposts.coip) for more information.
 * > **NOTE:** Removing the `replicate_source_db` attribute from an existing RDS
 * Replicate database managed by the provider will promote the database to a fully
 * standalone database.
 * @property dbName The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Note that this does not apply for Oracle or SQL Server engines. See the [AWS documentation](https://awscli.amazonaws.com/v2/documentation/api/latest/reference/rds/create-db-instance.html) for more details on what applies for those engines. If you are providing an Oracle db name, it needs to be in all upper case. Cannot be specified for a replica.
 * @property dbSubnetGroupName Name of DB subnet group. DB instance will
 * be created in the VPC associated with the DB subnet group. If unspecified, will
 * be created in the `default` VPC, or in EC2 Classic, if available. When working
 * with read replicas, it should be specified only if the source database
 * specifies an instance in another AWS Region. See [DBSubnetGroupName in API
 * action CreateDBInstanceReadReplica](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html)
 * for additional read replica constraints.
 * @property dedicatedLogVolume Use a dedicated log volume (DLV) for the DB instance. Requires Provisioned IOPS. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.dlv) for more details.
 * @property deleteAutomatedBackups Specifies whether to remove automated backups immediately after the DB instance is deleted. Default is `true`.
 * @property deletionProtection If the DB instance should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
 * @property domain The ID of the Directory Service Active Directory domain to create the instance in. Conflicts with `domain_fqdn`, `domain_ou`, `domain_auth_secret_arn` and a `domain_dns_ips`.
 * @property domainAuthSecretArn The ARN for the Secrets Manager secret with the self managed Active Directory credentials for the user joining the domain. Conflicts with `domain` and `domain_iam_role_name`.
 * @property domainDnsIps The IPv4 DNS IP addresses of your primary and secondary self managed Active Directory domain controllers. Two IP addresses must be provided. If there isn't a secondary domain controller, use the IP address of the primary domain controller for both entries in the list. Conflicts with `domain` and `domain_iam_role_name`.
 * @property domainFqdn The fully qualified domain name (FQDN) of the self managed Active Directory domain. Conflicts with `domain` and `domain_iam_role_name`.
 * @property domainIamRoleName The name of the IAM role to be used when making API calls to the Directory Service. Conflicts with `domain_fqdn`, `domain_ou`, `domain_auth_secret_arn` and a `domain_dns_ips`.
 * @property domainOu The self managed Active Directory organizational unit for your DB instance to join. Conflicts with `domain` and `domain_iam_role_name`.
 * @property enabledCloudwatchLogsExports Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. For supported values, see the EnableCloudwatchLogsExports.member.N parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html).
 * @property engine The database engine to use. For supported values, see the Engine parameter in [API action CreateDBInstance](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/APIReference/API_CreateDBInstance&#46;html)&#46; Note that for Amazon Aurora instances the engine must match the DB cluster's engine'&#46; For information on the difference between the available Aurora MySQL engines see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html) in the Amazon RDS User Guide.
 * @property engineLifecycleSupport The life cycle type for this DB instance. This setting applies only to RDS for MySQL and RDS for PostgreSQL. Valid values are `open-source-rds-extended-support`, `open-source-rds-extended-support-disabled`. Default value is `open-source-rds-extended-support`. [Using Amazon RDS Extended Support]: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/extended-support.html
 * @property engineVersion The engine version to use. If `auto_minor_version_upgrade` is enabled, you can provide a prefix of the version such as `8.0` (for `8.0.36`). The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. For supported values, see the EngineVersion parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html). Note that for Amazon Aurora instances the engine version must match the DB cluster's engine version'.
 * @property finalSnapshotIdentifier The name of your final DB snapshot
 * when this DB instance is deleted. Must be provided if `skip_final_snapshot` is
 * set to `false`. The value must begin with a letter, only contain alphanumeric characters and hyphens, and not end with a hyphen or contain two consecutive hyphens. Must not be provided when deleting a read replica.
 * @property iamDatabaseAuthenticationEnabled Specifies whether mappings of AWS Identity and Access Management (IAM) accounts to database
 * accounts is enabled.
 * @property identifier The name of the RDS instance, if omitted, this provider will assign a random, unique identifier. Required if `restore_to_point_in_time` is specified.
 * @property identifierPrefix Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
 * @property instanceClass The instance type of the RDS instance.
 * @property iops The amount of provisioned IOPS. Setting this implies a
 * storage_type of "io1". Can only be set when `storage_type` is `"io1"` or `"gp3"`.
 * Cannot be specified for gp3 storage if the `allocated_storage` value is below a per-`engine` threshold.
 * See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
 * @property kmsKeyId The ARN for the KMS encryption key. If creating an
 * encrypted replica, set this to the destination KMS ARN.
 * @property licenseModel License model information for this DB instance. Valid values for this field are as follows:
 * * RDS for MariaDB: `general-public-license`
 * * RDS for Microsoft SQL Server: `license-included`
 * * RDS for MySQL: `general-public-license`
 * * RDS for Oracle: `bring-your-own-license | license-included`
 * * RDS for PostgreSQL: `postgresql-license`
 * @property maintenanceWindow The window to perform maintenance in.
 * Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00". See [RDS
 * Maintenance Window
 * docs](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow)
 * for more information.
 * @property manageMasterUserPassword Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `password` is provided.
 * @property masterUserSecretKmsKeyId The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
 * @property maxAllocatedStorage When configured, the upper limit to which Amazon RDS can automatically scale the storage of the DB instance. Configuring this will automatically ignore differences to `allocated_storage`. Must be greater than or equal to `allocated_storage` or `0` to disable Storage Autoscaling.
 * @property monitoringInterval The interval, in seconds, between points
 * when Enhanced Monitoring metrics are collected for the DB instance. To disable
 * collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid
 * Values: 0, 1, 5, 10, 15, 30, 60.
 * @property monitoringRoleArn The ARN for the IAM role that permits RDS
 * to send enhanced monitoring metrics to CloudWatch Logs. You can find more
 * information on the [AWS
 * Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html)
 * what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
 * @property multiAz Specifies if the RDS instance is multi-AZ
 * @property name
 * @property ncharCharacterSetName The national character set is used in the NCHAR, NVARCHAR2, and NCLOB data types for Oracle instances. This can't be changed. See [Oracle Character Sets
 * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html).
 * @property networkType The network type of the DB instance. Valid values: `IPV4`, `DUAL`.
 * @property optionGroupName Name of the DB option group to associate.
 * @property parameterGroupName Name of the DB parameter group to associate.
 * @property password (Required unless `manage_master_user_password` is set to true or unless a `snapshot_identifier` or `replicate_source_db`
 * is provided or `manage_master_user_password` is set.) Password for the master DB user. Note that this may show up in
 * logs, and it will be stored in the state file. Cannot be set if `manage_master_user_password` is set to `true`.
 * @property performanceInsightsEnabled Specifies whether Performance Insights are enabled. Defaults to false.
 * @property performanceInsightsKmsKeyId The ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true. Once KMS key is set, it can never be changed.
 * @property performanceInsightsRetentionPeriod Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
 * @property port The port on which the DB accepts connections.
 * @property publiclyAccessible Bool to control if instance is publicly
 * accessible. Default is `false`.
 * @property replicaMode Specifies whether the replica is in either `mounted` or `open-read-only` mode. This attribute
 * is only supported by Oracle instances. Oracle replicas operate in `open-read-only` mode unless otherwise specified. See [Working with Oracle Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html) for more information.
 * @property replicateSourceDb Specifies that this resource is a Replicate
 * database, and to use this value as the source database. This correlates to the
 * `identifier` of another Amazon RDS Database to replicate (if replicating within
 * a single region) or ARN of the Amazon RDS Database to replicate (if replicating
 * cross-region). Note that if you are
 * creating a cross-region replica of an encrypted database you will also need to
 * specify a `kms_key_id`. See [DB Instance Replication][instance-replication] and [Working with
 * PostgreSQL and MySQL Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_ReadRepl.html)
 * for more information on using Replication.
 * @property restoreToPointInTime A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
 * @property s3Import Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
 * @property skipFinalSnapshot Determines whether a final DB snapshot is
 * created before the DB instance is deleted. If true is specified, no DBSnapshot
 * is created. If false is specified, a DB snapshot is created before the DB
 * instance is deleted, using the value from `final_snapshot_identifier`. Default
 * is `false`.
 * @property snapshotIdentifier Specifies whether or not to create this
 * database from a snapshot. This correlates to the snapshot ID you'd find in the
 * RDS console, e.g: rds:production-2015-06-26-06-05.
 * @property storageEncrypted Specifies whether the DB instance is
 * encrypted. Note that if you are creating a cross-region read replica this field
 * is ignored and you should instead declare `kms_key_id` with a valid ARN. The
 * default is `false` if not specified.
 * @property storageThroughput The storage throughput value for the DB instance. Can only be set when `storage_type` is `"gp3"`. Cannot be specified if the `allocated_storage` value is below a per-`engine` threshold. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
 * @property storageType One of "standard" (magnetic), "gp2" (general
 * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
 * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
 * "gp2" if not.
 * @property tags A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property timezone Time zone of the DB instance. `timezone` is currently
 * only supported by Microsoft SQL Server. The `timezone` can only be set on
 * creation. See [MSSQL User
 * Guide](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone)
 * for more information.
 * @property upgradeStorageConfig Whether to upgrade the storage file system configuration on the read replica. Can only be set with `replicate_source_db`.
 * @property username (Required unless a `snapshot_identifier` or `replicate_source_db`
 * is provided) Username for the master DB user. Cannot be specified for a replica.
 * @property vpcSecurityGroupIds List of VPC security groups to
 * associate.
 */
public data class InstanceArgs(
    public val allocatedStorage: Output<Int>? = null,
    public val allowMajorVersionUpgrade: Output<Boolean>? = null,
    public val applyImmediately: Output<Boolean>? = null,
    public val autoMinorVersionUpgrade: Output<Boolean>? = null,
    public val availabilityZone: Output<String>? = null,
    public val backupRetentionPeriod: Output<Int>? = null,
    public val backupTarget: Output<String>? = null,
    public val backupWindow: Output<String>? = null,
    public val blueGreenUpdate: Output<InstanceBlueGreenUpdateArgs>? = null,
    public val caCertIdentifier: Output<String>? = null,
    public val characterSetName: Output<String>? = null,
    public val copyTagsToSnapshot: Output<Boolean>? = null,
    public val customIamInstanceProfile: Output<String>? = null,
    public val customerOwnedIpEnabled: Output<Boolean>? = null,
    public val dbName: Output<String>? = null,
    public val dbSubnetGroupName: Output<String>? = null,
    public val dedicatedLogVolume: Output<Boolean>? = null,
    public val deleteAutomatedBackups: Output<Boolean>? = null,
    public val deletionProtection: Output<Boolean>? = null,
    public val domain: Output<String>? = null,
    public val domainAuthSecretArn: Output<String>? = null,
    public val domainDnsIps: Output<List<String>>? = null,
    public val domainFqdn: Output<String>? = null,
    public val domainIamRoleName: Output<String>? = null,
    public val domainOu: Output<String>? = null,
    public val enabledCloudwatchLogsExports: Output<List<String>>? = null,
    public val engine: Output<String>? = null,
    public val engineLifecycleSupport: Output<String>? = null,
    public val engineVersion: Output<String>? = null,
    public val finalSnapshotIdentifier: Output<String>? = null,
    public val iamDatabaseAuthenticationEnabled: Output<Boolean>? = null,
    public val identifier: Output<String>? = null,
    public val identifierPrefix: Output<String>? = null,
    public val instanceClass: Output<Either<String, InstanceType>>? = null,
    public val iops: Output<Int>? = null,
    public val kmsKeyId: Output<String>? = null,
    public val licenseModel: Output<String>? = null,
    public val maintenanceWindow: Output<String>? = null,
    public val manageMasterUserPassword: Output<Boolean>? = null,
    public val masterUserSecretKmsKeyId: Output<String>? = null,
    public val maxAllocatedStorage: Output<Int>? = null,
    public val monitoringInterval: Output<Int>? = null,
    public val monitoringRoleArn: Output<String>? = null,
    public val multiAz: Output<Boolean>? = null,
    @Deprecated(
        message = """
  This property has been deprecated. Please use 'dbName' instead.
  """,
    )
    public val name: Output<String>? = null,
    public val ncharCharacterSetName: Output<String>? = null,
    public val networkType: Output<String>? = null,
    public val optionGroupName: Output<String>? = null,
    public val parameterGroupName: Output<String>? = null,
    public val password: Output<String>? = null,
    public val performanceInsightsEnabled: Output<Boolean>? = null,
    public val performanceInsightsKmsKeyId: Output<String>? = null,
    public val performanceInsightsRetentionPeriod: Output<Int>? = null,
    public val port: Output<Int>? = null,
    public val publiclyAccessible: Output<Boolean>? = null,
    public val replicaMode: Output<String>? = null,
    public val replicateSourceDb: Output<String>? = null,
    public val restoreToPointInTime: Output<InstanceRestoreToPointInTimeArgs>? = null,
    public val s3Import: Output<InstanceS3ImportArgs>? = null,
    public val skipFinalSnapshot: Output<Boolean>? = null,
    public val snapshotIdentifier: Output<String>? = null,
    public val storageEncrypted: Output<Boolean>? = null,
    public val storageThroughput: Output<Int>? = null,
    public val storageType: Output<Either<String, StorageType>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val timezone: Output<String>? = null,
    public val upgradeStorageConfig: Output<Boolean>? = null,
    public val username: Output<String>? = null,
    public val vpcSecurityGroupIds: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.rds.InstanceArgs> {
    override fun toJava(): com.pulumi.aws.rds.InstanceArgs = com.pulumi.aws.rds.InstanceArgs.builder()
        .allocatedStorage(allocatedStorage?.applyValue({ args0 -> args0 }))
        .allowMajorVersionUpgrade(allowMajorVersionUpgrade?.applyValue({ args0 -> args0 }))
        .applyImmediately(applyImmediately?.applyValue({ args0 -> args0 }))
        .autoMinorVersionUpgrade(autoMinorVersionUpgrade?.applyValue({ args0 -> args0 }))
        .availabilityZone(availabilityZone?.applyValue({ args0 -> args0 }))
        .backupRetentionPeriod(backupRetentionPeriod?.applyValue({ args0 -> args0 }))
        .backupTarget(backupTarget?.applyValue({ args0 -> args0 }))
        .backupWindow(backupWindow?.applyValue({ args0 -> args0 }))
        .blueGreenUpdate(blueGreenUpdate?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .caCertIdentifier(caCertIdentifier?.applyValue({ args0 -> args0 }))
        .characterSetName(characterSetName?.applyValue({ args0 -> args0 }))
        .copyTagsToSnapshot(copyTagsToSnapshot?.applyValue({ args0 -> args0 }))
        .customIamInstanceProfile(customIamInstanceProfile?.applyValue({ args0 -> args0 }))
        .customerOwnedIpEnabled(customerOwnedIpEnabled?.applyValue({ args0 -> args0 }))
        .dbName(dbName?.applyValue({ args0 -> args0 }))
        .dbSubnetGroupName(dbSubnetGroupName?.applyValue({ args0 -> args0 }))
        .dedicatedLogVolume(dedicatedLogVolume?.applyValue({ args0 -> args0 }))
        .deleteAutomatedBackups(deleteAutomatedBackups?.applyValue({ args0 -> args0 }))
        .deletionProtection(deletionProtection?.applyValue({ args0 -> args0 }))
        .domain(domain?.applyValue({ args0 -> args0 }))
        .domainAuthSecretArn(domainAuthSecretArn?.applyValue({ args0 -> args0 }))
        .domainDnsIps(domainDnsIps?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .domainFqdn(domainFqdn?.applyValue({ args0 -> args0 }))
        .domainIamRoleName(domainIamRoleName?.applyValue({ args0 -> args0 }))
        .domainOu(domainOu?.applyValue({ args0 -> args0 }))
        .enabledCloudwatchLogsExports(
            enabledCloudwatchLogsExports?.applyValue({ args0 ->
                args0.map({ args0 -> args0 })
            }),
        )
        .engine(engine?.applyValue({ args0 -> args0 }))
        .engineLifecycleSupport(engineLifecycleSupport?.applyValue({ args0 -> args0 }))
        .engineVersion(engineVersion?.applyValue({ args0 -> args0 }))
        .finalSnapshotIdentifier(finalSnapshotIdentifier?.applyValue({ args0 -> args0 }))
        .iamDatabaseAuthenticationEnabled(iamDatabaseAuthenticationEnabled?.applyValue({ args0 -> args0 }))
        .identifier(identifier?.applyValue({ args0 -> args0 }))
        .identifierPrefix(identifierPrefix?.applyValue({ args0 -> args0 }))
        .instanceClass(
            instanceClass?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .iops(iops?.applyValue({ args0 -> args0 }))
        .kmsKeyId(kmsKeyId?.applyValue({ args0 -> args0 }))
        .licenseModel(licenseModel?.applyValue({ args0 -> args0 }))
        .maintenanceWindow(maintenanceWindow?.applyValue({ args0 -> args0 }))
        .manageMasterUserPassword(manageMasterUserPassword?.applyValue({ args0 -> args0 }))
        .masterUserSecretKmsKeyId(masterUserSecretKmsKeyId?.applyValue({ args0 -> args0 }))
        .maxAllocatedStorage(maxAllocatedStorage?.applyValue({ args0 -> args0 }))
        .monitoringInterval(monitoringInterval?.applyValue({ args0 -> args0 }))
        .monitoringRoleArn(monitoringRoleArn?.applyValue({ args0 -> args0 }))
        .multiAz(multiAz?.applyValue({ args0 -> args0 }))
        .name(name?.applyValue({ args0 -> args0 }))
        .ncharCharacterSetName(ncharCharacterSetName?.applyValue({ args0 -> args0 }))
        .networkType(networkType?.applyValue({ args0 -> args0 }))
        .optionGroupName(optionGroupName?.applyValue({ args0 -> args0 }))
        .parameterGroupName(parameterGroupName?.applyValue({ args0 -> args0 }))
        .password(password?.applyValue({ args0 -> args0 }))
        .performanceInsightsEnabled(performanceInsightsEnabled?.applyValue({ args0 -> args0 }))
        .performanceInsightsKmsKeyId(performanceInsightsKmsKeyId?.applyValue({ args0 -> args0 }))
        .performanceInsightsRetentionPeriod(
            performanceInsightsRetentionPeriod?.applyValue({ args0 ->
                args0
            }),
        )
        .port(port?.applyValue({ args0 -> args0 }))
        .publiclyAccessible(publiclyAccessible?.applyValue({ args0 -> args0 }))
        .replicaMode(replicaMode?.applyValue({ args0 -> args0 }))
        .replicateSourceDb(replicateSourceDb?.applyValue({ args0 -> args0 }))
        .restoreToPointInTime(
            restoreToPointInTime?.applyValue({ args0 ->
                args0.let({ args0 ->
                    args0.toJava()
                })
            }),
        )
        .s3Import(s3Import?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .skipFinalSnapshot(skipFinalSnapshot?.applyValue({ args0 -> args0 }))
        .snapshotIdentifier(snapshotIdentifier?.applyValue({ args0 -> args0 }))
        .storageEncrypted(storageEncrypted?.applyValue({ args0 -> args0 }))
        .storageThroughput(storageThroughput?.applyValue({ args0 -> args0 }))
        .storageType(
            storageType?.applyValue({ args0 ->
                args0.transform({ args0 -> args0 }, { args0 ->
                    args0.let({ args0 -> args0.toJava() })
                })
            }),
        )
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .timezone(timezone?.applyValue({ args0 -> args0 }))
        .upgradeStorageConfig(upgradeStorageConfig?.applyValue({ args0 -> args0 }))
        .username(username?.applyValue({ args0 -> args0 }))
        .vpcSecurityGroupIds(
            vpcSecurityGroupIds?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0
                })
            }),
        ).build()
}

/**
 * Builder for [InstanceArgs].
 */
@PulumiTagMarker
public class InstanceArgsBuilder internal constructor() {
    private var allocatedStorage: Output<Int>? = null

    private var allowMajorVersionUpgrade: Output<Boolean>? = null

    private var applyImmediately: Output<Boolean>? = null

    private var autoMinorVersionUpgrade: Output<Boolean>? = null

    private var availabilityZone: Output<String>? = null

    private var backupRetentionPeriod: Output<Int>? = null

    private var backupTarget: Output<String>? = null

    private var backupWindow: Output<String>? = null

    private var blueGreenUpdate: Output<InstanceBlueGreenUpdateArgs>? = null

    private var caCertIdentifier: Output<String>? = null

    private var characterSetName: Output<String>? = null

    private var copyTagsToSnapshot: Output<Boolean>? = null

    private var customIamInstanceProfile: Output<String>? = null

    private var customerOwnedIpEnabled: Output<Boolean>? = null

    private var dbName: Output<String>? = null

    private var dbSubnetGroupName: Output<String>? = null

    private var dedicatedLogVolume: Output<Boolean>? = null

    private var deleteAutomatedBackups: Output<Boolean>? = null

    private var deletionProtection: Output<Boolean>? = null

    private var domain: Output<String>? = null

    private var domainAuthSecretArn: Output<String>? = null

    private var domainDnsIps: Output<List<String>>? = null

    private var domainFqdn: Output<String>? = null

    private var domainIamRoleName: Output<String>? = null

    private var domainOu: Output<String>? = null

    private var enabledCloudwatchLogsExports: Output<List<String>>? = null

    private var engine: Output<String>? = null

    private var engineLifecycleSupport: Output<String>? = null

    private var engineVersion: Output<String>? = null

    private var finalSnapshotIdentifier: Output<String>? = null

    private var iamDatabaseAuthenticationEnabled: Output<Boolean>? = null

    private var identifier: Output<String>? = null

    private var identifierPrefix: Output<String>? = null

    private var instanceClass: Output<Either<String, InstanceType>>? = null

    private var iops: Output<Int>? = null

    private var kmsKeyId: Output<String>? = null

    private var licenseModel: Output<String>? = null

    private var maintenanceWindow: Output<String>? = null

    private var manageMasterUserPassword: Output<Boolean>? = null

    private var masterUserSecretKmsKeyId: Output<String>? = null

    private var maxAllocatedStorage: Output<Int>? = null

    private var monitoringInterval: Output<Int>? = null

    private var monitoringRoleArn: Output<String>? = null

    private var multiAz: Output<Boolean>? = null

    private var name: Output<String>? = null

    private var ncharCharacterSetName: Output<String>? = null

    private var networkType: Output<String>? = null

    private var optionGroupName: Output<String>? = null

    private var parameterGroupName: Output<String>? = null

    private var password: Output<String>? = null

    private var performanceInsightsEnabled: Output<Boolean>? = null

    private var performanceInsightsKmsKeyId: Output<String>? = null

    private var performanceInsightsRetentionPeriod: Output<Int>? = null

    private var port: Output<Int>? = null

    private var publiclyAccessible: Output<Boolean>? = null

    private var replicaMode: Output<String>? = null

    private var replicateSourceDb: Output<String>? = null

    private var restoreToPointInTime: Output<InstanceRestoreToPointInTimeArgs>? = null

    private var s3Import: Output<InstanceS3ImportArgs>? = null

    private var skipFinalSnapshot: Output<Boolean>? = null

    private var snapshotIdentifier: Output<String>? = null

    private var storageEncrypted: Output<Boolean>? = null

    private var storageThroughput: Output<Int>? = null

    private var storageType: Output<Either<String, StorageType>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var timezone: Output<String>? = null

    private var upgradeStorageConfig: Output<Boolean>? = null

    private var username: Output<String>? = null

    private var vpcSecurityGroupIds: Output<List<String>>? = null

    /**
     * @param value The allocated storage in gibibytes. If `max_allocated_storage` is configured, this argument represents the initial storage allocation and differences from the configuration will be ignored automatically when Storage Autoscaling occurs. If `replicate_source_db` is set, the value is ignored during the creation of the instance.
     */
    @JvmName("xmaeffhmdinrhcbd")
    public suspend fun allocatedStorage(`value`: Output<Int>) {
        this.allocatedStorage = value
    }

    /**
     * @param value Indicates that major version
     * upgrades are allowed. Changing this parameter does not result in an outage and
     * the change is asynchronously applied as soon as possible.
     */
    @JvmName("mjoiicfxnoianvac")
    public suspend fun allowMajorVersionUpgrade(`value`: Output<Boolean>) {
        this.allowMajorVersionUpgrade = value
    }

    /**
     * @param value Specifies whether any database modifications
     * are applied immediately, or during the next maintenance window. Default is
     * `false`. See [Amazon RDS Documentation for more
   * information.](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("lnskrmikcrigqsgr")
    public suspend fun applyImmediately(`value`: Output<Boolean>) {
        this.applyImmediately = value
    }

    /**
     * @param value Indicates that minor engine upgrades
     * will be applied automatically to the DB instance during the maintenance window.
     * Defaults to true.
     */
    @JvmName("ifehinvmqcgpojod")
    public suspend fun autoMinorVersionUpgrade(`value`: Output<Boolean>) {
        this.autoMinorVersionUpgrade = value
    }

    /**
     * @param value The AZ for the RDS instance.
     */
    @JvmName("leqlfpaheywajgok")
    public suspend fun availabilityZone(`value`: Output<String>) {
        this.availabilityZone = value
    }

    /**
     * @param value The days to retain backups for.
     * Must be between `0` and `35`.
     * Default is `0`.
     * Must be greater than `0` if the database is used as a source for a [Read Replica][instance-replication],
     * uses low-downtime updates,
     * or will use [RDS Blue/Green deployments][blue-green].
     */
    @JvmName("hvkslguafdgitxgw")
    public suspend fun backupRetentionPeriod(`value`: Output<Int>) {
        this.backupRetentionPeriod = value
    }

    /**
     * @param value Specifies where automated backups and manual snapshots are stored. Possible values are `region` (default) and `outposts`. See [Working with Amazon RDS on AWS Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html) for more information.
     */
    @JvmName("calsbqltjgykfapu")
    public suspend fun backupTarget(`value`: Output<String>) {
        this.backupTarget = value
    }

    /**
     * @param value The daily time range (in UTC) during which automated backups are created if they are enabled.
     * Example: "09:46-10:16". Must not overlap with `maintenance_window`.
     */
    @JvmName("iujsmlghrxayttlf")
    public suspend fun backupWindow(`value`: Output<String>) {
        this.backupWindow = value
    }

    /**
     * @param value Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
     * See `blue_green_update` below.
     */
    @JvmName("aybbycgodupnxavq")
    public suspend fun blueGreenUpdate(`value`: Output<InstanceBlueGreenUpdateArgs>) {
        this.blueGreenUpdate = value
    }

    /**
     * @param value The identifier of the CA certificate for the DB instance.
     */
    @JvmName("sgkuwaoyymvvtcql")
    public suspend fun caCertIdentifier(`value`: Output<String>) {
        this.caCertIdentifier = value
    }

    /**
     * @param value The character set name to use for DB encoding in Oracle and Microsoft SQL instances (collation).
     * This can't be changed.
     * See [Oracle Character Sets Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html) or
     * [Server-Level Collation for Microsoft SQL Server](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.CommonDBATasks.Collation.html) for more information.
     * Cannot be set  with `replicate_source_db`, `restore_to_point_in_time`, `s3_import`, or `snapshot_identifier`.
     */
    @JvmName("kqjikluinrlikkbp")
    public suspend fun characterSetName(`value`: Output<String>) {
        this.characterSetName = value
    }

    /**
     * @param value Copy all Instance `tags` to snapshots. Default is `false`.
     */
    @JvmName("nhhmxlpxnabtsatw")
    public suspend fun copyTagsToSnapshot(`value`: Output<Boolean>) {
        this.copyTagsToSnapshot = value
    }

    /**
     * @param value The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
     */
    @JvmName("jcpimtbmvhlkfmtc")
    public suspend fun customIamInstanceProfile(`value`: Output<String>) {
        this.customIamInstanceProfile = value
    }

    /**
     * @param value Indicates whether to enable a customer-owned IP address (CoIP) for an RDS on Outposts DB instance. See [CoIP for RDS on Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html#rds-on-outposts.coip) for more information.
     * > **NOTE:** Removing the `replicate_source_db` attribute from an existing RDS
     * Replicate database managed by the provider will promote the database to a fully
     * standalone database.
     */
    @JvmName("mpgxrcgphunueamh")
    public suspend fun customerOwnedIpEnabled(`value`: Output<Boolean>) {
        this.customerOwnedIpEnabled = value
    }

    /**
     * @param value The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Note that this does not apply for Oracle or SQL Server engines. See the [AWS documentation](https://awscli.amazonaws.com/v2/documentation/api/latest/reference/rds/create-db-instance.html) for more details on what applies for those engines. If you are providing an Oracle db name, it needs to be in all upper case. Cannot be specified for a replica.
     */
    @JvmName("kutaphiiypimcuwk")
    public suspend fun dbName(`value`: Output<String>) {
        this.dbName = value
    }

    /**
     * @param value Name of DB subnet group. DB instance will
     * be created in the VPC associated with the DB subnet group. If unspecified, will
     * be created in the `default` VPC, or in EC2 Classic, if available. When working
     * with read replicas, it should be specified only if the source database
     * specifies an instance in another AWS Region. See [DBSubnetGroupName in API
   * action CreateDBInstanceReadReplica](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html)
     * for additional read replica constraints.
     */
    @JvmName("bqmdhangcyohwtxd")
    public suspend fun dbSubnetGroupName(`value`: Output<String>) {
        this.dbSubnetGroupName = value
    }

    /**
     * @param value Use a dedicated log volume (DLV) for the DB instance. Requires Provisioned IOPS. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.dlv) for more details.
     */
    @JvmName("bxjrxxgisxepwgyy")
    public suspend fun dedicatedLogVolume(`value`: Output<Boolean>) {
        this.dedicatedLogVolume = value
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB instance is deleted. Default is `true`.
     */
    @JvmName("xkdxmfpndyxtuoli")
    public suspend fun deleteAutomatedBackups(`value`: Output<Boolean>) {
        this.deleteAutomatedBackups = value
    }

    /**
     * @param value If the DB instance should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
     */
    @JvmName("awlpwrdcgnlioluu")
    public suspend fun deletionProtection(`value`: Output<Boolean>) {
        this.deletionProtection = value
    }

    /**
     * @param value The ID of the Directory Service Active Directory domain to create the instance in. Conflicts with `domain_fqdn`, `domain_ou`, `domain_auth_secret_arn` and a `domain_dns_ips`.
     */
    @JvmName("bndniknkywhealok")
    public suspend fun domain(`value`: Output<String>) {
        this.domain = value
    }

    /**
     * @param value The ARN for the Secrets Manager secret with the self managed Active Directory credentials for the user joining the domain. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("yekjqhckgeglakuc")
    public suspend fun domainAuthSecretArn(`value`: Output<String>) {
        this.domainAuthSecretArn = value
    }

    /**
     * @param value The IPv4 DNS IP addresses of your primary and secondary self managed Active Directory domain controllers. Two IP addresses must be provided. If there isn't a secondary domain controller, use the IP address of the primary domain controller for both entries in the list. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("sxhniirwhxbbnwov")
    public suspend fun domainDnsIps(`value`: Output<List<String>>) {
        this.domainDnsIps = value
    }

    @JvmName("ulfmwjsywjjekigm")
    public suspend fun domainDnsIps(vararg values: Output<String>) {
        this.domainDnsIps = Output.all(values.asList())
    }

    /**
     * @param values The IPv4 DNS IP addresses of your primary and secondary self managed Active Directory domain controllers. Two IP addresses must be provided. If there isn't a secondary domain controller, use the IP address of the primary domain controller for both entries in the list. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("rpfxuomwapgukoyk")
    public suspend fun domainDnsIps(values: List<Output<String>>) {
        this.domainDnsIps = Output.all(values)
    }

    /**
     * @param value The fully qualified domain name (FQDN) of the self managed Active Directory domain. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("qmdikdaabpedpvyp")
    public suspend fun domainFqdn(`value`: Output<String>) {
        this.domainFqdn = value
    }

    /**
     * @param value The name of the IAM role to be used when making API calls to the Directory Service. Conflicts with `domain_fqdn`, `domain_ou`, `domain_auth_secret_arn` and a `domain_dns_ips`.
     */
    @JvmName("giewsiipfffnjudj")
    public suspend fun domainIamRoleName(`value`: Output<String>) {
        this.domainIamRoleName = value
    }

    /**
     * @param value The self managed Active Directory organizational unit for your DB instance to join. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("gbhfiqehnxtxieua")
    public suspend fun domainOu(`value`: Output<String>) {
        this.domainOu = value
    }

    /**
     * @param value Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. For supported values, see the EnableCloudwatchLogsExports.member.N parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html).
     */
    @JvmName("lvnmyesbfrtcdoty")
    public suspend fun enabledCloudwatchLogsExports(`value`: Output<List<String>>) {
        this.enabledCloudwatchLogsExports = value
    }

    @JvmName("pctvlfjsaxrwcsef")
    public suspend fun enabledCloudwatchLogsExports(vararg values: Output<String>) {
        this.enabledCloudwatchLogsExports = Output.all(values.asList())
    }

    /**
     * @param values Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. For supported values, see the EnableCloudwatchLogsExports.member.N parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html).
     */
    @JvmName("pvgefoftbwangxak")
    public suspend fun enabledCloudwatchLogsExports(values: List<Output<String>>) {
        this.enabledCloudwatchLogsExports = Output.all(values)
    }

    /**
     * @param value The database engine to use. For supported values, see the Engine parameter in [API action CreateDBInstance](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/APIReference/API_CreateDBInstance&#46;html)&#46; Note that for Amazon Aurora instances the engine must match the DB cluster's engine'&#46; For information on the difference between the available Aurora MySQL engines see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html) in the Amazon RDS User Guide.
     */
    @JvmName("ckjttfhyhmtvixnc")
    public suspend fun engine(`value`: Output<String>) {
        this.engine = value
    }

    /**
     * @param value The life cycle type for this DB instance. This setting applies only to RDS for MySQL and RDS for PostgreSQL. Valid values are `open-source-rds-extended-support`, `open-source-rds-extended-support-disabled`. Default value is `open-source-rds-extended-support`. [Using Amazon RDS Extended Support]: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/extended-support.html
     */
    @JvmName("uanjqnoyftpynrlx")
    public suspend fun engineLifecycleSupport(`value`: Output<String>) {
        this.engineLifecycleSupport = value
    }

    /**
     * @param value The engine version to use. If `auto_minor_version_upgrade` is enabled, you can provide a prefix of the version such as `8.0` (for `8.0.36`). The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. For supported values, see the EngineVersion parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html). Note that for Amazon Aurora instances the engine version must match the DB cluster's engine version'.
     */
    @JvmName("jebyvvrjahuewhpn")
    public suspend fun engineVersion(`value`: Output<String>) {
        this.engineVersion = value
    }

    /**
     * @param value The name of your final DB snapshot
     * when this DB instance is deleted. Must be provided if `skip_final_snapshot` is
     * set to `false`. The value must begin with a letter, only contain alphanumeric characters and hyphens, and not end with a hyphen or contain two consecutive hyphens. Must not be provided when deleting a read replica.
     */
    @JvmName("fxfcpraoitlbtokl")
    public suspend fun finalSnapshotIdentifier(`value`: Output<String>) {
        this.finalSnapshotIdentifier = value
    }

    /**
     * @param value Specifies whether mappings of AWS Identity and Access Management (IAM) accounts to database
     * accounts is enabled.
     */
    @JvmName("wfskgvnxswojqiry")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Output<Boolean>) {
        this.iamDatabaseAuthenticationEnabled = value
    }

    /**
     * @param value The name of the RDS instance, if omitted, this provider will assign a random, unique identifier. Required if `restore_to_point_in_time` is specified.
     */
    @JvmName("qjemggcnyfvyfsxw")
    public suspend fun identifier(`value`: Output<String>) {
        this.identifier = value
    }

    /**
     * @param value Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
     */
    @JvmName("qdneoryvbxrnkbes")
    public suspend fun identifierPrefix(`value`: Output<String>) {
        this.identifierPrefix = value
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("ssicomefsarmpjmp")
    public suspend fun instanceClass(`value`: Output<Either<String, InstanceType>>) {
        this.instanceClass = value
    }

    /**
     * @param value The amount of provisioned IOPS. Setting this implies a
     * storage_type of "io1". Can only be set when `storage_type` is `"io1"` or `"gp3"`.
     * Cannot be specified for gp3 storage if the `allocated_storage` value is below a per-`engine` threshold.
     * See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("urjgnxvfxoyervwa")
    public suspend fun iops(`value`: Output<Int>) {
        this.iops = value
    }

    /**
     * @param value The ARN for the KMS encryption key. If creating an
     * encrypted replica, set this to the destination KMS ARN.
     */
    @JvmName("elmhjyehvakxmswg")
    public suspend fun kmsKeyId(`value`: Output<String>) {
        this.kmsKeyId = value
    }

    /**
     * @param value License model information for this DB instance. Valid values for this field are as follows:
     * * RDS for MariaDB: `general-public-license`
     * * RDS for Microsoft SQL Server: `license-included`
     * * RDS for MySQL: `general-public-license`
     * * RDS for Oracle: `bring-your-own-license | license-included`
     * * RDS for PostgreSQL: `postgresql-license`
     */
    @JvmName("psocwqytcwlmldsb")
    public suspend fun licenseModel(`value`: Output<String>) {
        this.licenseModel = value
    }

    /**
     * @param value The window to perform maintenance in.
     * Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00". See [RDS
   * Maintenance Window
   * docs](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow)
     * for more information.
     */
    @JvmName("xnvkbkgfbugutnya")
    public suspend fun maintenanceWindow(`value`: Output<String>) {
        this.maintenanceWindow = value
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `password` is provided.
     */
    @JvmName("wclwqyrciuuudiwy")
    public suspend fun manageMasterUserPassword(`value`: Output<Boolean>) {
        this.manageMasterUserPassword = value
    }

    /**
     * @param value The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("kgcyvaqoonordmng")
    public suspend fun masterUserSecretKmsKeyId(`value`: Output<String>) {
        this.masterUserSecretKmsKeyId = value
    }

    /**
     * @param value When configured, the upper limit to which Amazon RDS can automatically scale the storage of the DB instance. Configuring this will automatically ignore differences to `allocated_storage`. Must be greater than or equal to `allocated_storage` or `0` to disable Storage Autoscaling.
     */
    @JvmName("mvnswxgnsmtckmuq")
    public suspend fun maxAllocatedStorage(`value`: Output<Int>) {
        this.maxAllocatedStorage = value
    }

    /**
     * @param value The interval, in seconds, between points
     * when Enhanced Monitoring metrics are collected for the DB instance. To disable
     * collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid
     * Values: 0, 1, 5, 10, 15, 30, 60.
     */
    @JvmName("epgfhpcjtcprmsda")
    public suspend fun monitoringInterval(`value`: Output<Int>) {
        this.monitoringInterval = value
    }

    /**
     * @param value The ARN for the IAM role that permits RDS
     * to send enhanced monitoring metrics to CloudWatch Logs. You can find more
     * information on the [AWS
   * Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html)
     * what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
     */
    @JvmName("hwjneghnnalttmcs")
    public suspend fun monitoringRoleArn(`value`: Output<String>) {
        this.monitoringRoleArn = value
    }

    /**
     * @param value Specifies if the RDS instance is multi-AZ
     */
    @JvmName("dpgoswxcukhhvgux")
    public suspend fun multiAz(`value`: Output<Boolean>) {
        this.multiAz = value
    }

    /**
     * @param value
     */
    @Deprecated(
        message = """
  This property has been deprecated. Please use 'dbName' instead.
  """,
    )
    @JvmName("ddowujifmhljttwi")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The national character set is used in the NCHAR, NVARCHAR2, and NCLOB data types for Oracle instances. This can't be changed. See [Oracle Character Sets
   * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html).
     */
    @JvmName("qxrsgoalblqamdoe")
    public suspend fun ncharCharacterSetName(`value`: Output<String>) {
        this.ncharCharacterSetName = value
    }

    /**
     * @param value The network type of the DB instance. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("ptccuhejiddmqeqh")
    public suspend fun networkType(`value`: Output<String>) {
        this.networkType = value
    }

    /**
     * @param value Name of the DB option group to associate.
     */
    @JvmName("yvsrwaainkydwogn")
    public suspend fun optionGroupName(`value`: Output<String>) {
        this.optionGroupName = value
    }

    /**
     * @param value Name of the DB parameter group to associate.
     */
    @JvmName("bupwdhuadvugmnaq")
    public suspend fun parameterGroupName(`value`: Output<String>) {
        this.parameterGroupName = value
    }

    /**
     * @param value (Required unless `manage_master_user_password` is set to true or unless a `snapshot_identifier` or `replicate_source_db`
     * is provided or `manage_master_user_password` is set.) Password for the master DB user. Note that this may show up in
     * logs, and it will be stored in the state file. Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("lcqjecsbxgisiuuh")
    public suspend fun password(`value`: Output<String>) {
        this.password = value
    }

    /**
     * @param value Specifies whether Performance Insights are enabled. Defaults to false.
     */
    @JvmName("jtvpksnmjdqrueuy")
    public suspend fun performanceInsightsEnabled(`value`: Output<Boolean>) {
        this.performanceInsightsEnabled = value
    }

    /**
     * @param value The ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true. Once KMS key is set, it can never be changed.
     */
    @JvmName("vsfkfnhsiijnjkdo")
    public suspend fun performanceInsightsKmsKeyId(`value`: Output<String>) {
        this.performanceInsightsKmsKeyId = value
    }

    /**
     * @param value Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
     */
    @JvmName("juvwijayhqtybnva")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Output<Int>) {
        this.performanceInsightsRetentionPeriod = value
    }

    /**
     * @param value The port on which the DB accepts connections.
     */
    @JvmName("tlbfcpuuipmhxmqa")
    public suspend fun port(`value`: Output<Int>) {
        this.port = value
    }

    /**
     * @param value Bool to control if instance is publicly
     * accessible. Default is `false`.
     */
    @JvmName("pijerkydcmwrapxq")
    public suspend fun publiclyAccessible(`value`: Output<Boolean>) {
        this.publiclyAccessible = value
    }

    /**
     * @param value Specifies whether the replica is in either `mounted` or `open-read-only` mode. This attribute
     * is only supported by Oracle instances. Oracle replicas operate in `open-read-only` mode unless otherwise specified. See [Working with Oracle Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html) for more information.
     */
    @JvmName("gcisogimbnebntav")
    public suspend fun replicaMode(`value`: Output<String>) {
        this.replicaMode = value
    }

    /**
     * @param value Specifies that this resource is a Replicate
     * database, and to use this value as the source database. This correlates to the
     * `identifier` of another Amazon RDS Database to replicate (if replicating within
     * a single region) or ARN of the Amazon RDS Database to replicate (if replicating
     * cross-region). Note that if you are
     * creating a cross-region replica of an encrypted database you will also need to
     * specify a `kms_key_id`. See [DB Instance Replication][instance-replication] and [Working with
   * PostgreSQL and MySQL Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_ReadRepl.html)
     * for more information on using Replication.
     */
    @JvmName("colrtgpupmcsvlpr")
    public suspend fun replicateSourceDb(`value`: Output<String>) {
        this.replicateSourceDb = value
    }

    /**
     * @param value A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
     */
    @JvmName("hvhpcxveygytbkib")
    public suspend fun restoreToPointInTime(`value`: Output<InstanceRestoreToPointInTimeArgs>) {
        this.restoreToPointInTime = value
    }

    /**
     * @param value Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
     */
    @JvmName("fqiemktbarafecsa")
    public suspend fun s3Import(`value`: Output<InstanceS3ImportArgs>) {
        this.s3Import = value
    }

    /**
     * @param value Determines whether a final DB snapshot is
     * created before the DB instance is deleted. If true is specified, no DBSnapshot
     * is created. If false is specified, a DB snapshot is created before the DB
     * instance is deleted, using the value from `final_snapshot_identifier`. Default
     * is `false`.
     */
    @JvmName("sreewibieahnravy")
    public suspend fun skipFinalSnapshot(`value`: Output<Boolean>) {
        this.skipFinalSnapshot = value
    }

    /**
     * @param value Specifies whether or not to create this
     * database from a snapshot. This correlates to the snapshot ID you'd find in the
     * RDS console, e.g: rds:production-2015-06-26-06-05.
     */
    @JvmName("scanpfasfqinkvqf")
    public suspend fun snapshotIdentifier(`value`: Output<String>) {
        this.snapshotIdentifier = value
    }

    /**
     * @param value Specifies whether the DB instance is
     * encrypted. Note that if you are creating a cross-region read replica this field
     * is ignored and you should instead declare `kms_key_id` with a valid ARN. The
     * default is `false` if not specified.
     */
    @JvmName("fiojtodhblgqbjrr")
    public suspend fun storageEncrypted(`value`: Output<Boolean>) {
        this.storageEncrypted = value
    }

    /**
     * @param value The storage throughput value for the DB instance. Can only be set when `storage_type` is `"gp3"`. Cannot be specified if the `allocated_storage` value is below a per-`engine` threshold. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("uvphbfityiualwqb")
    public suspend fun storageThroughput(`value`: Output<Int>) {
        this.storageThroughput = value
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("fmqvoqwfsvmwjgmo")
    public suspend fun storageType(`value`: Output<Either<String, StorageType>>) {
        this.storageType = value
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("knkpixoyuipqvbnr")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Time zone of the DB instance. `timezone` is currently
     * only supported by Microsoft SQL Server. The `timezone` can only be set on
     * creation. See [MSSQL User
   * Guide](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone)
     * for more information.
     */
    @JvmName("photyqvcfpilivok")
    public suspend fun timezone(`value`: Output<String>) {
        this.timezone = value
    }

    /**
     * @param value Whether to upgrade the storage file system configuration on the read replica. Can only be set with `replicate_source_db`.
     */
    @JvmName("ytkeoutwjfogofmg")
    public suspend fun upgradeStorageConfig(`value`: Output<Boolean>) {
        this.upgradeStorageConfig = value
    }

    /**
     * @param value (Required unless a `snapshot_identifier` or `replicate_source_db`
     * is provided) Username for the master DB user. Cannot be specified for a replica.
     */
    @JvmName("cnidjevospjoqhjj")
    public suspend fun username(`value`: Output<String>) {
        this.username = value
    }

    /**
     * @param value List of VPC security groups to
     * associate.
     */
    @JvmName("mudpvywcwpclktlh")
    public suspend fun vpcSecurityGroupIds(`value`: Output<List<String>>) {
        this.vpcSecurityGroupIds = value
    }

    @JvmName("gxsxaisiwxlubmuu")
    public suspend fun vpcSecurityGroupIds(vararg values: Output<String>) {
        this.vpcSecurityGroupIds = Output.all(values.asList())
    }

    /**
     * @param values List of VPC security groups to
     * associate.
     */
    @JvmName("kgooashmoqephhpf")
    public suspend fun vpcSecurityGroupIds(values: List<Output<String>>) {
        this.vpcSecurityGroupIds = Output.all(values)
    }

    /**
     * @param value The allocated storage in gibibytes. If `max_allocated_storage` is configured, this argument represents the initial storage allocation and differences from the configuration will be ignored automatically when Storage Autoscaling occurs. If `replicate_source_db` is set, the value is ignored during the creation of the instance.
     */
    @JvmName("ceuackvvafivtbfx")
    public suspend fun allocatedStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allocatedStorage = mapped
    }

    /**
     * @param value Indicates that major version
     * upgrades are allowed. Changing this parameter does not result in an outage and
     * the change is asynchronously applied as soon as possible.
     */
    @JvmName("jqjebuokfanyjegv")
    public suspend fun allowMajorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.allowMajorVersionUpgrade = mapped
    }

    /**
     * @param value Specifies whether any database modifications
     * are applied immediately, or during the next maintenance window. Default is
     * `false`. See [Amazon RDS Documentation for more
   * information.](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
     */
    @JvmName("nsogpmaovjsrgdax")
    public suspend fun applyImmediately(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.applyImmediately = mapped
    }

    /**
     * @param value Indicates that minor engine upgrades
     * will be applied automatically to the DB instance during the maintenance window.
     * Defaults to true.
     */
    @JvmName("kjhurgoxtixsonuv")
    public suspend fun autoMinorVersionUpgrade(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoMinorVersionUpgrade = mapped
    }

    /**
     * @param value The AZ for the RDS instance.
     */
    @JvmName("doogneepdiuvkmdq")
    public suspend fun availabilityZone(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.availabilityZone = mapped
    }

    /**
     * @param value The days to retain backups for.
     * Must be between `0` and `35`.
     * Default is `0`.
     * Must be greater than `0` if the database is used as a source for a [Read Replica][instance-replication],
     * uses low-downtime updates,
     * or will use [RDS Blue/Green deployments][blue-green].
     */
    @JvmName("wabkudxhpaolrnth")
    public suspend fun backupRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupRetentionPeriod = mapped
    }

    /**
     * @param value Specifies where automated backups and manual snapshots are stored. Possible values are `region` (default) and `outposts`. See [Working with Amazon RDS on AWS Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html) for more information.
     */
    @JvmName("nfngetkuymukqiag")
    public suspend fun backupTarget(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupTarget = mapped
    }

    /**
     * @param value The daily time range (in UTC) during which automated backups are created if they are enabled.
     * Example: "09:46-10:16". Must not overlap with `maintenance_window`.
     */
    @JvmName("lodirwlmeirtncgp")
    public suspend fun backupWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.backupWindow = mapped
    }

    /**
     * @param value Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
     * See `blue_green_update` below.
     */
    @JvmName("bpkxrxrwrfbrivoa")
    public suspend fun blueGreenUpdate(`value`: InstanceBlueGreenUpdateArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.blueGreenUpdate = mapped
    }

    /**
     * @param argument Enables low-downtime updates using [RDS Blue/Green deployments][blue-green].
     * See `blue_green_update` below.
     */
    @JvmName("pgswofijrngmpyuq")
    public suspend fun blueGreenUpdate(argument: suspend InstanceBlueGreenUpdateArgsBuilder.() -> Unit) {
        val toBeMapped = InstanceBlueGreenUpdateArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.blueGreenUpdate = mapped
    }

    /**
     * @param value The identifier of the CA certificate for the DB instance.
     */
    @JvmName("xysmktjhtjoxbghy")
    public suspend fun caCertIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.caCertIdentifier = mapped
    }

    /**
     * @param value The character set name to use for DB encoding in Oracle and Microsoft SQL instances (collation).
     * This can't be changed.
     * See [Oracle Character Sets Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html) or
     * [Server-Level Collation for Microsoft SQL Server](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.CommonDBATasks.Collation.html) for more information.
     * Cannot be set  with `replicate_source_db`, `restore_to_point_in_time`, `s3_import`, or `snapshot_identifier`.
     */
    @JvmName("tfcvwpbjuhodnofs")
    public suspend fun characterSetName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.characterSetName = mapped
    }

    /**
     * @param value Copy all Instance `tags` to snapshots. Default is `false`.
     */
    @JvmName("wfdnpnsistrordtq")
    public suspend fun copyTagsToSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyTagsToSnapshot = mapped
    }

    /**
     * @param value The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.
     */
    @JvmName("jepqoruxrvhhprfw")
    public suspend fun customIamInstanceProfile(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customIamInstanceProfile = mapped
    }

    /**
     * @param value Indicates whether to enable a customer-owned IP address (CoIP) for an RDS on Outposts DB instance. See [CoIP for RDS on Outposts](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-on-outposts.html#rds-on-outposts.coip) for more information.
     * > **NOTE:** Removing the `replicate_source_db` attribute from an existing RDS
     * Replicate database managed by the provider will promote the database to a fully
     * standalone database.
     */
    @JvmName("qewgviejgbtupcvl")
    public suspend fun customerOwnedIpEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerOwnedIpEnabled = mapped
    }

    /**
     * @param value The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Note that this does not apply for Oracle or SQL Server engines. See the [AWS documentation](https://awscli.amazonaws.com/v2/documentation/api/latest/reference/rds/create-db-instance.html) for more details on what applies for those engines. If you are providing an Oracle db name, it needs to be in all upper case. Cannot be specified for a replica.
     */
    @JvmName("ddfhkmnslaiwmisv")
    public suspend fun dbName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbName = mapped
    }

    /**
     * @param value Name of DB subnet group. DB instance will
     * be created in the VPC associated with the DB subnet group. If unspecified, will
     * be created in the `default` VPC, or in EC2 Classic, if available. When working
     * with read replicas, it should be specified only if the source database
     * specifies an instance in another AWS Region. See [DBSubnetGroupName in API
   * action CreateDBInstanceReadReplica](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html)
     * for additional read replica constraints.
     */
    @JvmName("knfmmmptmeitykyr")
    public suspend fun dbSubnetGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dbSubnetGroupName = mapped
    }

    /**
     * @param value Use a dedicated log volume (DLV) for the DB instance. Requires Provisioned IOPS. See the [AWS documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.dlv) for more details.
     */
    @JvmName("bsyjyfssaulxsocu")
    public suspend fun dedicatedLogVolume(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dedicatedLogVolume = mapped
    }

    /**
     * @param value Specifies whether to remove automated backups immediately after the DB instance is deleted. Default is `true`.
     */
    @JvmName("nkxasotjeeyliwbo")
    public suspend fun deleteAutomatedBackups(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deleteAutomatedBackups = mapped
    }

    /**
     * @param value If the DB instance should have deletion protection enabled. The database can't be deleted when this value is set to `true`. The default is `false`.
     */
    @JvmName("hyucuegpieodwmmh")
    public suspend fun deletionProtection(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deletionProtection = mapped
    }

    /**
     * @param value The ID of the Directory Service Active Directory domain to create the instance in. Conflicts with `domain_fqdn`, `domain_ou`, `domain_auth_secret_arn` and a `domain_dns_ips`.
     */
    @JvmName("ausybswowpmiaokw")
    public suspend fun domain(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domain = mapped
    }

    /**
     * @param value The ARN for the Secrets Manager secret with the self managed Active Directory credentials for the user joining the domain. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("ybbowccalddrbobk")
    public suspend fun domainAuthSecretArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainAuthSecretArn = mapped
    }

    /**
     * @param value The IPv4 DNS IP addresses of your primary and secondary self managed Active Directory domain controllers. Two IP addresses must be provided. If there isn't a secondary domain controller, use the IP address of the primary domain controller for both entries in the list. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("mgnnxxsyxgptgkry")
    public suspend fun domainDnsIps(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainDnsIps = mapped
    }

    /**
     * @param values The IPv4 DNS IP addresses of your primary and secondary self managed Active Directory domain controllers. Two IP addresses must be provided. If there isn't a secondary domain controller, use the IP address of the primary domain controller for both entries in the list. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("mohjytbufdelcywp")
    public suspend fun domainDnsIps(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.domainDnsIps = mapped
    }

    /**
     * @param value The fully qualified domain name (FQDN) of the self managed Active Directory domain. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("agkhiqmdjamiaqqk")
    public suspend fun domainFqdn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainFqdn = mapped
    }

    /**
     * @param value The name of the IAM role to be used when making API calls to the Directory Service. Conflicts with `domain_fqdn`, `domain_ou`, `domain_auth_secret_arn` and a `domain_dns_ips`.
     */
    @JvmName("siijaxiuvtfoqmiq")
    public suspend fun domainIamRoleName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainIamRoleName = mapped
    }

    /**
     * @param value The self managed Active Directory organizational unit for your DB instance to join. Conflicts with `domain` and `domain_iam_role_name`.
     */
    @JvmName("rmhnuuvcadckqheh")
    public suspend fun domainOu(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domainOu = mapped
    }

    /**
     * @param value Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. For supported values, see the EnableCloudwatchLogsExports.member.N parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html).
     */
    @JvmName("rmvaewgbltlwnkqy")
    public suspend fun enabledCloudwatchLogsExports(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param values Set of log types to enable for exporting to CloudWatch logs. If omitted, no logs will be exported. For supported values, see the EnableCloudwatchLogsExports.member.N parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html).
     */
    @JvmName("uqmvhlcbpfwvntdq")
    public suspend fun enabledCloudwatchLogsExports(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.enabledCloudwatchLogsExports = mapped
    }

    /**
     * @param value The database engine to use. For supported values, see the Engine parameter in [API action CreateDBInstance](https://docs&#46;aws&#46;amazon&#46;com/AmazonRDS/latest/APIReference/API_CreateDBInstance&#46;html)&#46; Note that for Amazon Aurora instances the engine must match the DB cluster's engine'&#46; For information on the difference between the available Aurora MySQL engines see [Comparison between Aurora MySQL 1 and Aurora MySQL 2](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/AuroraMySQL.Updates.20180206.html) in the Amazon RDS User Guide.
     */
    @JvmName("begqfdwcjdufjurn")
    public suspend fun engine(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engine = mapped
    }

    /**
     * @param value The life cycle type for this DB instance. This setting applies only to RDS for MySQL and RDS for PostgreSQL. Valid values are `open-source-rds-extended-support`, `open-source-rds-extended-support-disabled`. Default value is `open-source-rds-extended-support`. [Using Amazon RDS Extended Support]: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/extended-support.html
     */
    @JvmName("lyjequpyhhwedcyp")
    public suspend fun engineLifecycleSupport(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineLifecycleSupport = mapped
    }

    /**
     * @param value The engine version to use. If `auto_minor_version_upgrade` is enabled, you can provide a prefix of the version such as `8.0` (for `8.0.36`). The actual engine version used is returned in the attribute `engine_version_actual`, see Attribute Reference below. For supported values, see the EngineVersion parameter in [API action CreateDBInstance](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html). Note that for Amazon Aurora instances the engine version must match the DB cluster's engine version'.
     */
    @JvmName("sjquovgarvfnvlvq")
    public suspend fun engineVersion(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.engineVersion = mapped
    }

    /**
     * @param value The name of your final DB snapshot
     * when this DB instance is deleted. Must be provided if `skip_final_snapshot` is
     * set to `false`. The value must begin with a letter, only contain alphanumeric characters and hyphens, and not end with a hyphen or contain two consecutive hyphens. Must not be provided when deleting a read replica.
     */
    @JvmName("dwaqyedbgsssjwxr")
    public suspend fun finalSnapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.finalSnapshotIdentifier = mapped
    }

    /**
     * @param value Specifies whether mappings of AWS Identity and Access Management (IAM) accounts to database
     * accounts is enabled.
     */
    @JvmName("piommoflddqolfyj")
    public suspend fun iamDatabaseAuthenticationEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iamDatabaseAuthenticationEnabled = mapped
    }

    /**
     * @param value The name of the RDS instance, if omitted, this provider will assign a random, unique identifier. Required if `restore_to_point_in_time` is specified.
     */
    @JvmName("ugvktgtkyrormwgq")
    public suspend fun identifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identifier = mapped
    }

    /**
     * @param value Creates a unique identifier beginning with the specified prefix. Conflicts with `identifier`.
     */
    @JvmName("tfdgmiiiqtvhwrnp")
    public suspend fun identifierPrefix(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identifierPrefix = mapped
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("mcwgrtqdivecgsly")
    public suspend fun instanceClass(`value`: Either<String, InstanceType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("nkgypotpqbxbvtyk")
    public fun instanceClass(`value`: String) {
        val toBeMapped = Either.ofLeft<String, InstanceType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value The instance type of the RDS instance.
     */
    @JvmName("iqogphqdvrvujmaf")
    public fun instanceClass(`value`: InstanceType) {
        val toBeMapped = Either.ofRight<String, InstanceType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.instanceClass = mapped
    }

    /**
     * @param value The amount of provisioned IOPS. Setting this implies a
     * storage_type of "io1". Can only be set when `storage_type` is `"io1"` or `"gp3"`.
     * Cannot be specified for gp3 storage if the `allocated_storage` value is below a per-`engine` threshold.
     * See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("nnfbgmmtbctfcwvk")
    public suspend fun iops(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.iops = mapped
    }

    /**
     * @param value The ARN for the KMS encryption key. If creating an
     * encrypted replica, set this to the destination KMS ARN.
     */
    @JvmName("woxdbinqqmkchohh")
    public suspend fun kmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKeyId = mapped
    }

    /**
     * @param value License model information for this DB instance. Valid values for this field are as follows:
     * * RDS for MariaDB: `general-public-license`
     * * RDS for Microsoft SQL Server: `license-included`
     * * RDS for MySQL: `general-public-license`
     * * RDS for Oracle: `bring-your-own-license | license-included`
     * * RDS for PostgreSQL: `postgresql-license`
     */
    @JvmName("tlvxbmejfunpcjbs")
    public suspend fun licenseModel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.licenseModel = mapped
    }

    /**
     * @param value The window to perform maintenance in.
     * Syntax: "ddd:hh24:mi-ddd:hh24:mi". Eg: "Mon:00:00-Mon:03:00". See [RDS
   * Maintenance Window
   * docs](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow)
     * for more information.
     */
    @JvmName("awruhjfveflarbgg")
    public suspend fun maintenanceWindow(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintenanceWindow = mapped
    }

    /**
     * @param value Set to true to allow RDS to manage the master user password in Secrets Manager. Cannot be set if `password` is provided.
     */
    @JvmName("thtcdaadjekmqxqh")
    public suspend fun manageMasterUserPassword(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.manageMasterUserPassword = mapped
    }

    /**
     * @param value The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. To use a KMS key in a different Amazon Web Services account, specify the key ARN or alias ARN. If not specified, the default KMS key for your Amazon Web Services account is used.
     */
    @JvmName("nnbsfcxrcxdjabgp")
    public suspend fun masterUserSecretKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.masterUserSecretKmsKeyId = mapped
    }

    /**
     * @param value When configured, the upper limit to which Amazon RDS can automatically scale the storage of the DB instance. Configuring this will automatically ignore differences to `allocated_storage`. Must be greater than or equal to `allocated_storage` or `0` to disable Storage Autoscaling.
     */
    @JvmName("ktwhnngashgccytv")
    public suspend fun maxAllocatedStorage(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxAllocatedStorage = mapped
    }

    /**
     * @param value The interval, in seconds, between points
     * when Enhanced Monitoring metrics are collected for the DB instance. To disable
     * collecting Enhanced Monitoring metrics, specify 0. The default is 0. Valid
     * Values: 0, 1, 5, 10, 15, 30, 60.
     */
    @JvmName("qlfxoqceyopfwnug")
    public suspend fun monitoringInterval(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.monitoringInterval = mapped
    }

    /**
     * @param value The ARN for the IAM role that permits RDS
     * to send enhanced monitoring metrics to CloudWatch Logs. You can find more
     * information on the [AWS
   * Documentation](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.html)
     * what IAM permissions are needed to allow Enhanced Monitoring for RDS Instances.
     */
    @JvmName("wsrkwhlygcxtmfoj")
    public suspend fun monitoringRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.monitoringRoleArn = mapped
    }

    /**
     * @param value Specifies if the RDS instance is multi-AZ
     */
    @JvmName("yabhpxmyrckpgynl")
    public suspend fun multiAz(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.multiAz = mapped
    }

    /**
     * @param value
     */
    @Deprecated(
        message = """
  This property has been deprecated. Please use 'dbName' instead.
  """,
    )
    @JvmName("btkhskkfuklsdwyk")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The national character set is used in the NCHAR, NVARCHAR2, and NCLOB data types for Oracle instances. This can't be changed. See [Oracle Character Sets
   * Supported in Amazon RDS](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.OracleCharacterSets.html).
     */
    @JvmName("sbafopghroiaandg")
    public suspend fun ncharCharacterSetName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ncharCharacterSetName = mapped
    }

    /**
     * @param value The network type of the DB instance. Valid values: `IPV4`, `DUAL`.
     */
    @JvmName("xabevrkwmhvgprfi")
    public suspend fun networkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.networkType = mapped
    }

    /**
     * @param value Name of the DB option group to associate.
     */
    @JvmName("spjcjhtqytnvwrdh")
    public suspend fun optionGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.optionGroupName = mapped
    }

    /**
     * @param value Name of the DB parameter group to associate.
     */
    @JvmName("njhawoucnegaqdtt")
    public suspend fun parameterGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parameterGroupName = mapped
    }

    /**
     * @param value (Required unless `manage_master_user_password` is set to true or unless a `snapshot_identifier` or `replicate_source_db`
     * is provided or `manage_master_user_password` is set.) Password for the master DB user. Note that this may show up in
     * logs, and it will be stored in the state file. Cannot be set if `manage_master_user_password` is set to `true`.
     */
    @JvmName("ogxuwxhrxcqbycgl")
    public suspend fun password(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.password = mapped
    }

    /**
     * @param value Specifies whether Performance Insights are enabled. Defaults to false.
     */
    @JvmName("wdmxdekhpdjgncql")
    public suspend fun performanceInsightsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsEnabled = mapped
    }

    /**
     * @param value The ARN for the KMS key to encrypt Performance Insights data. When specifying `performance_insights_kms_key_id`, `performance_insights_enabled` needs to be set to true. Once KMS key is set, it can never be changed.
     */
    @JvmName("fkwplmiiqiobfwnw")
    public suspend fun performanceInsightsKmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsKmsKeyId = mapped
    }

    /**
     * @param value Amount of time in days to retain Performance Insights data. Valid values are `7`, `731` (2 years) or a multiple of `31`. When specifying `performance_insights_retention_period`, `performance_insights_enabled` needs to be set to true. Defaults to '7'.
     */
    @JvmName("ewepimexvwhycbel")
    public suspend fun performanceInsightsRetentionPeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.performanceInsightsRetentionPeriod = mapped
    }

    /**
     * @param value The port on which the DB accepts connections.
     */
    @JvmName("tfhixjhcsytjyluf")
    public suspend fun port(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.port = mapped
    }

    /**
     * @param value Bool to control if instance is publicly
     * accessible. Default is `false`.
     */
    @JvmName("cpoerursggygwccj")
    public suspend fun publiclyAccessible(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.publiclyAccessible = mapped
    }

    /**
     * @param value Specifies whether the replica is in either `mounted` or `open-read-only` mode. This attribute
     * is only supported by Oracle instances. Oracle replicas operate in `open-read-only` mode unless otherwise specified. See [Working with Oracle Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html) for more information.
     */
    @JvmName("hyuudewqpxnnsiet")
    public suspend fun replicaMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicaMode = mapped
    }

    /**
     * @param value Specifies that this resource is a Replicate
     * database, and to use this value as the source database. This correlates to the
     * `identifier` of another Amazon RDS Database to replicate (if replicating within
     * a single region) or ARN of the Amazon RDS Database to replicate (if replicating
     * cross-region). Note that if you are
     * creating a cross-region replica of an encrypted database you will also need to
     * specify a `kms_key_id`. See [DB Instance Replication][instance-replication] and [Working with
   * PostgreSQL and MySQL Read Replicas](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_ReadRepl.html)
     * for more information on using Replication.
     */
    @JvmName("yfdwyutpeaysynlo")
    public suspend fun replicateSourceDb(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replicateSourceDb = mapped
    }

    /**
     * @param value A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
     */
    @JvmName("fogxwjvghrbsffrg")
    public suspend fun restoreToPointInTime(`value`: InstanceRestoreToPointInTimeArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restoreToPointInTime = mapped
    }

    /**
     * @param argument A configuration block for restoring a DB instance to an arbitrary point in time. Requires the `identifier` argument to be set with the name of the new DB instance to be created. See Restore To Point In Time below for details.
     */
    @JvmName("jfdeyxhkkmffmmbe")
    public suspend fun restoreToPointInTime(argument: suspend InstanceRestoreToPointInTimeArgsBuilder.() -> Unit) {
        val toBeMapped = InstanceRestoreToPointInTimeArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.restoreToPointInTime = mapped
    }

    /**
     * @param value Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
     */
    @JvmName("akaoseemuioswclo")
    public suspend fun s3Import(`value`: InstanceS3ImportArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.s3Import = mapped
    }

    /**
     * @param argument Restore from a Percona Xtrabackup in S3.  See [Importing Data into an Amazon RDS MySQL DB Instance](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/MySQL.Procedural.Importing.html)
     */
    @JvmName("ghcmnxyqdiylncyu")
    public suspend fun s3Import(argument: suspend InstanceS3ImportArgsBuilder.() -> Unit) {
        val toBeMapped = InstanceS3ImportArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.s3Import = mapped
    }

    /**
     * @param value Determines whether a final DB snapshot is
     * created before the DB instance is deleted. If true is specified, no DBSnapshot
     * is created. If false is specified, a DB snapshot is created before the DB
     * instance is deleted, using the value from `final_snapshot_identifier`. Default
     * is `false`.
     */
    @JvmName("xyrvbnwpgxjfmwda")
    public suspend fun skipFinalSnapshot(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.skipFinalSnapshot = mapped
    }

    /**
     * @param value Specifies whether or not to create this
     * database from a snapshot. This correlates to the snapshot ID you'd find in the
     * RDS console, e.g: rds:production-2015-06-26-06-05.
     */
    @JvmName("mxsppmlwkqochlbu")
    public suspend fun snapshotIdentifier(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.snapshotIdentifier = mapped
    }

    /**
     * @param value Specifies whether the DB instance is
     * encrypted. Note that if you are creating a cross-region read replica this field
     * is ignored and you should instead declare `kms_key_id` with a valid ARN. The
     * default is `false` if not specified.
     */
    @JvmName("nvtptgmaocvtneov")
    public suspend fun storageEncrypted(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageEncrypted = mapped
    }

    /**
     * @param value The storage throughput value for the DB instance. Can only be set when `storage_type` is `"gp3"`. Cannot be specified if the `allocated_storage` value is below a per-`engine` threshold. See the [RDS User Guide](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#gp3-storage) for details.
     */
    @JvmName("nqedpuqdqtwbdsme")
    public suspend fun storageThroughput(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageThroughput = mapped
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("dffpjfviyojyxuak")
    public suspend fun storageType(`value`: Either<String, StorageType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("uqgsslrfgrgtoqrq")
    public fun storageType(`value`: String) {
        val toBeMapped = Either.ofLeft<String, StorageType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value One of "standard" (magnetic), "gp2" (general
     * purpose SSD), "gp3" (general purpose SSD that needs `iops` independently)
     * or "io1" (provisioned IOPS SSD). The default is "io1" if `iops` is specified,
     * "gp2" if not.
     */
    @JvmName("xqcmlbeykbqwucsc")
    public fun storageType(`value`: StorageType) {
        val toBeMapped = Either.ofRight<String, StorageType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.storageType = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cgfigncknsnntwmd")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("oisgdlfopclfgrwg")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Time zone of the DB instance. `timezone` is currently
     * only supported by Microsoft SQL Server. The `timezone` can only be set on
     * creation. See [MSSQL User
   * Guide](http://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone)
     * for more information.
     */
    @JvmName("qffefdmdqnjasqpy")
    public suspend fun timezone(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.timezone = mapped
    }

    /**
     * @param value Whether to upgrade the storage file system configuration on the read replica. Can only be set with `replicate_source_db`.
     */
    @JvmName("eqyokirvwqptddta")
    public suspend fun upgradeStorageConfig(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.upgradeStorageConfig = mapped
    }

    /**
     * @param value (Required unless a `snapshot_identifier` or `replicate_source_db`
     * is provided) Username for the master DB user. Cannot be specified for a replica.
     */
    @JvmName("dongjjrrthlofmfl")
    public suspend fun username(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.username = mapped
    }

    /**
     * @param value List of VPC security groups to
     * associate.
     */
    @JvmName("icxjuckxxfqnkqcr")
    public suspend fun vpcSecurityGroupIds(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    /**
     * @param values List of VPC security groups to
     * associate.
     */
    @JvmName("ettvudmcsutlkslf")
    public suspend fun vpcSecurityGroupIds(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.vpcSecurityGroupIds = mapped
    }

    internal fun build(): InstanceArgs = InstanceArgs(
        allocatedStorage = allocatedStorage,
        allowMajorVersionUpgrade = allowMajorVersionUpgrade,
        applyImmediately = applyImmediately,
        autoMinorVersionUpgrade = autoMinorVersionUpgrade,
        availabilityZone = availabilityZone,
        backupRetentionPeriod = backupRetentionPeriod,
        backupTarget = backupTarget,
        backupWindow = backupWindow,
        blueGreenUpdate = blueGreenUpdate,
        caCertIdentifier = caCertIdentifier,
        characterSetName = characterSetName,
        copyTagsToSnapshot = copyTagsToSnapshot,
        customIamInstanceProfile = customIamInstanceProfile,
        customerOwnedIpEnabled = customerOwnedIpEnabled,
        dbName = dbName,
        dbSubnetGroupName = dbSubnetGroupName,
        dedicatedLogVolume = dedicatedLogVolume,
        deleteAutomatedBackups = deleteAutomatedBackups,
        deletionProtection = deletionProtection,
        domain = domain,
        domainAuthSecretArn = domainAuthSecretArn,
        domainDnsIps = domainDnsIps,
        domainFqdn = domainFqdn,
        domainIamRoleName = domainIamRoleName,
        domainOu = domainOu,
        enabledCloudwatchLogsExports = enabledCloudwatchLogsExports,
        engine = engine,
        engineLifecycleSupport = engineLifecycleSupport,
        engineVersion = engineVersion,
        finalSnapshotIdentifier = finalSnapshotIdentifier,
        iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled,
        identifier = identifier,
        identifierPrefix = identifierPrefix,
        instanceClass = instanceClass,
        iops = iops,
        kmsKeyId = kmsKeyId,
        licenseModel = licenseModel,
        maintenanceWindow = maintenanceWindow,
        manageMasterUserPassword = manageMasterUserPassword,
        masterUserSecretKmsKeyId = masterUserSecretKmsKeyId,
        maxAllocatedStorage = maxAllocatedStorage,
        monitoringInterval = monitoringInterval,
        monitoringRoleArn = monitoringRoleArn,
        multiAz = multiAz,
        name = name,
        ncharCharacterSetName = ncharCharacterSetName,
        networkType = networkType,
        optionGroupName = optionGroupName,
        parameterGroupName = parameterGroupName,
        password = password,
        performanceInsightsEnabled = performanceInsightsEnabled,
        performanceInsightsKmsKeyId = performanceInsightsKmsKeyId,
        performanceInsightsRetentionPeriod = performanceInsightsRetentionPeriod,
        port = port,
        publiclyAccessible = publiclyAccessible,
        replicaMode = replicaMode,
        replicateSourceDb = replicateSourceDb,
        restoreToPointInTime = restoreToPointInTime,
        s3Import = s3Import,
        skipFinalSnapshot = skipFinalSnapshot,
        snapshotIdentifier = snapshotIdentifier,
        storageEncrypted = storageEncrypted,
        storageThroughput = storageThroughput,
        storageType = storageType,
        tags = tags,
        timezone = timezone,
        upgradeStorageConfig = upgradeStorageConfig,
        username = username,
        vpcSecurityGroupIds = vpcSecurityGroupIds,
    )
}
